# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['magiclink', 'magiclink.migrations']

package_data = \
{'': ['*'], 'magiclink': ['templates/magiclink/*']}

install_requires = \
['Django>=2.1']

setup_kwargs = {
    'name': 'django-magiclink',
    'version': '0.5.0',
    'description': 'Passwordless Authentication for Django with Magic Links',
    'long_description': '# Django MagicLink\n\n\nPasswordless Authentication for Django with Magic Links.\n\nThis package was create a fully tested with a focus on [ease of setup](#steps-to-impliment), [security](#security) and testing (coverage is currently at 100%). The idea is to use sane defaults to quickly create secure single-use token authentication for Django.\n\n\n## Install\n\n```bash\npip install django-magiclink\n```\n\n\n## Setup\n\nThe setup of the app is simple but has a few steps and a few templates that need overriding.\n\n1. [Install the app](#install)\n1. [Configure the app](#configuration) adding urls and settings. There are also a number of [additional configuration settings](#configuration-settings)\n1. [Set up the login page](#login-page) by overriding the login page template\n1. [Override the login sent page HTML](#login-sent-page)\n1. [Set up the magic link email](#magic-link-email) (optional) by setting the email logo and colours. It\'s also possible to override the email templates\n1. [Create a signup page](#signup-page) (optional) depending on your settings configuration\n\n\n### Basic login flow\n\n1. The user signs up via the sign up page (This can be skipped if `MAGICLINK_REQUIRE_SIGNUP = False`)\n1. They enter their email on the login page to request a magic link\n1. A magic link is sent to users email address\n1. The user is redirected to a login sent page\n1. The user clicks on the magic link in their email\n1. The user is logged in and redirected\n\n\n*If you want to create a different passwordless login flow see the [Manual usage](#manual-usage) section*\n\n\n#### Configuration\n\nAdd to the `urlpatterns` in `urls.py`:\n```python\nurlpatterns = [\n    ...\n    path(\'auth/\', include(\'magiclink.urls\', namespace=\'magiclink\')),\n    ...\n]\n```\n\nAdd `magiclink` to your `INSTALLED_APPS`:\n```python\nINSTALLED_APPS = (\n    ...\n    \'magiclink\',\n    ...\n)\n```\n\n```python\nAUTHENTICATION_BACKENDS = (\n    \'magiclink.backends.MagicLinkBackend\',\n    ...\n    \'django.contrib.auth.backends.ModelBackend\',\n)\n```\n*Note: MagicLinkBackend should be placed at the top of AUTHENTICATION_BACKENDS* to ensure it is used\n\n\nAdd the following settings to your `settings.py` (you will need to replace the template names in the below steps):\n```python\n# Set Djangos login URL to the magiclink login page\nLOGIN_URL = \'magiclink:login\'\n\nMAGICLINK_LOGIN_TEMPLATE_NAME = \'magiclink/login.html\'\nMAGICLINK_LOGIN_SENT_TEMPLATE_NAME = \'magiclink/login_sent.html\'\n\n# Optional:\n# If this setting is set to False a user account will be created the first\n# time a user requests a login link.\nMAGICLINK_REQUIRE_SIGNUP = True\nMAGICLINK_SIGNUP_TEMPLATE_NAME = \'magiclink/signup.html\'\n```\n\nSee [additional configuration settings](#configuration-settings) for all of the different available settings.\n\n\n#### Login page\n\nEach login page will need different HTML so you need to set the `MAGICLINK_LOGIN_TEMPLATE_NAME` setting to a template of your own. When overriding this template please ensure the following code is included:\n\n```html\n<form action="{% url \'magiclink:login\' %}{% if request.GET.next %}?next={{ request.GET.next }}{% endif %}" method="post">\n    {% csrf_token %}\n    {{ login_form }}\n    <button type=\'submit\'>Send login email</button>\n</form>\n```\n\nSee the login docs if you want to create your own login view\n\n\n#### Login sent page\n\nAfter the user has requested a magic link, they will be redirected to a success page. The HTML for this page can be overridden using the setting `MAGICLINK_LOGIN_SENT_TEMPLATE_NAME`. It is advised you return a simple message telling the user to check their email:\n\n```html\n<h1>Check your email</h1>\n<p>We have sent you a magic link to your email address</p>\n<p>Please click the link to be logged in automatically</p>\n```\n\n#### Magic link email\n\nThe login email which includes the magic link needs to be configured. By default, a simple HTML template is used which can be adapted to your own branding using the `MAGICLINK_EMAIL_STYLES` setting, or you can override the template (see below)\n\nThis `MAGICLINK_EMAIL_STYLES` setting should be a dict with the following key values:\n\n```python\nMAGICLINK_EMAIL_STYLES = {\n    \'logo_url\': \'https://example.com/logo.png\',\n    \'background-colour\': \'#ffffff\',\n    \'main-text-color\': \'#000000\',\n    \'button-background-color\': \'#0078be\',\n    \'button-text-color\': \'#ffffff\',\n}\n```\n*Note: The logo URL must be a full URL. For email client support you should use either a jpeg or png.*\n\nIf this email template is not to your liking you can override the email templates (one for text and one for html). To do so you need to override the `MAGICLINK_EMAIL_TEMPLATE_NAME_TEXT` and `MAGICLINK_EMAIL_TEMPLATE_NAME_HTML` settings.  If you override these templates the following context variables are available:\n\n* `{{ subject }}` - The subject of the email "Your login magic link"\n* `{{ magiclink }}` - The magic link URL\n* `{{ user }}` - The full user object\n* `{{ expiry }}` - Datetime for when the magiclink expires\n* `{{ ip_address }}` - The IP address of the person who requested the magic link\n* `{{ created }}` - Datetime of when the magic link was created\n* `{{ same_ip }}` - The value of MAGICLINK_REQUIRE_SAME_IP\n* `{{ same_browser }}` - The value of MAGICLINK_REQUIRE_SAME_BROWSER\n* `{{ token_uses }}` - The value of MAGICLINK_TOKEN_USES\n\n\n#### Signup page\n\nIf you want users to have to signup before being able to log in you will want to override the signup page template. This is needed when `MAGICLINK_REQUIRE_SIGNUP = True`. On successful signup the user will be sent a login email.\n\nWhen overriding this template please ensure the following content is included:\n\n```html\n<form action="{% url \'magiclink:signup\' %}" method="post">\n    {% csrf_token %}\n    {{ SignupForm }}\n    <button type=\'submit\'>Signup</button>\n</form>\n<p>Already have an account? <a href=\'{% url \'magiclink:login\' %}\'>Log in here</a></p>\n```\n\nThere are actually several forms avalible in the context on this page depending on what information you want to collect.\n* **SignupFormEmailOnly** - Only includes an `email` field\n* **SignupForm** - Includes `name` and `email` fields\n* **SignupFormWithUsername** - Includes `Username` and `email` fields\n* **SignupFormFull** - Includes `username`, `name` and `email` fields\n\n\nLike the login for the sign up flow can be overridden if you require more information from the user on signup. See the login/setup docs for more details\nToDo: Include docs on how to use post_save signal to send Welcome email?\n\n\n#### Configuration settings\n\nBelow are the different settings that can be overridden. To do so place the setting into you `settings.py`.\n\n*Note: Each of the url / redirect settings can either be a URL or url name*\n\n```python\n\n# Override the login page template. See \'Login page\' in the Setup section\nMAGICLINK_LOGIN_TEMPLATE_NAME = \'myapp/login.html\'\n\n# Override the login page template. See \'Login sent page\' in the Setup section\nMAGICLINK_LOGIN_SENT_TEMPLATE_NAME = \'myapp/login_sent.html\'\n\n# If this setting is set to False a user account will be created the first time\n# a user requests a login link.\nMAGICLINK_REQUIRE_SIGNUP = True\n# Override the login page template. See \'Login sent page\' in the Setup section\nMAGICLINK_SIGNUP_TEMPLATE_NAME = \'myapp/signup.html\'\n\n# Set Djangos login redirect URL to be used once the user opens the magic link\n# This will be used whenever a ?next parameter is not set on login\nLOGIN_REDIRECT_URL = \'/accounts/profile/\'\n\n# If a new user is created via the signup page use this setting to send them to\n# a different url than LOGIN_REDIRECT_URL when clicking the magic link\n# This will fall back to LOGIN_REDIRECT_URL\nMAGICLINK_SIGNUP_LOGIN_REDIRECT = \'/welcome\'\n\n# Change the url a user is redirect to after requesting a magic link\nMAGICLINK_LOGIN_SENT_REDIRECT = \'magiclink:login_sent\'\n\n# Ensure the branding of the login email is correct. This setting is not needed\n# if you override the `login_email.html` template\nMAGICLINK_EMAIL_STYLES = {\n    \'logo_url\': \'\',\n    \'background-colour\': \'#ffffff\',\n    \'main-text-color\': \'#000000\',\n    \'button-background-color\': \'#0078be\',\n    \'button-text-color\': \'#ffffff\',\n}\n\n# If you want to use your own email templates you can override the text and\n# html templates used with:\nMAGICLINK_EMAIL_TEMPLATE_NAME_TEXT = \'myapp/login_email.text\'\nMAGICLINK_EMAIL_TEMPLATE_NAME_HTML = \'myapp/login_email.html\'\n\n# How long a magic link is valid for before returning an error\nMAGICLINK_AUTH_TIMEOUT = 300  # In second - Default is 5 minutes\n\n# Email address is not case sensitive. If this setting is set to True all\n# emails addresses will be set to lowercase before any checks are run against it\nMAGICLINK_IGNORE_EMAIL_CASE = True\n\n# Allow superusers to login via a magic link\nMAGICLINK_ALLOW_SUPERUSER_LOGIN = True\n\n# Allow staff users to login via a magic link\nMAGICLINK_ALLOW_STAFF_LOGIN = True\n\n# Override the default magic link length\n# Warning: Overriding this setting has security implications, shorter tokens\n# are much more susceptible to brute force attacks*\nMAGICLINK_TOKEN_LENGTH = 50\n\n# Require the user email to be included in the verification link\n# Warning: If this is set to false tokens are more vulnerable to brute force\nMAGICLINK_VERIFY_INCLUDE_EMAIL = True\n\n# Ensure the user who clicked magic link used the same browser as the\n# initial login request.\n# Note: This can cause issues on devices where the default browser is\n# different from the browser being used by the user such as on iOS)*\nMAGICLINK_REQUIRE_SAME_BROWSER = True\n\n# Ensure the user who clicked magic link has the same IP address as the\n# initial login request.\nMAGICLINK_REQUIRE_SAME_IP = True\n\n# The number of times a login token can be used before being disabled\nMAGICLINK_TOKEN_USES = 1\n\n# How often a user can request a new login token (basic rate limiting).\nMAGICLINK_LOGIN_REQUEST_TIME_LIMIT = 30  # In seconds\n\n# Disable all other tokens for a user when a new token is requested\nMAGICLINK_ONE_TOKEN_PER_USER = True\n```\n\n\n## Security\n\nUsing magic links can be dangerous as poorly implemented login links can be brute-forced and emails can be forwarded by accident. There are several security measures used to mitigate these risks:\n\n* The one-time password issued will be valid for 5 minutes before it expires\n* The user\'s email is specified alongside login tokens to stop URLs being brute-forced\n* Each login token will be at least 20 digits?\n* The initial request and its response must take place from the same IP address\n* The initial request and its response must take place in the same browser\n* Each one-time link can only be used once\n* Only the last one-time link issued will be accepted. Once the latest one is issued, any others are invalidated.\n\n*Note: Each of the above settings can be overridden*\n\n\n## Manual usage\n\ndjango-magiclink uses a model to help create, send and validate magic links. A `create_magiclink` helper function can be used easily create a MagicLink using the correct settings:\n\n```python\nfrom magiclink.helpers import create_magiclink\n\nmagiclink = create_magiclink(email, request, redirect_url=\'\')\n\n# Generates the full magic link url and send it in a email\nmagiclink.send(request)\n\n# If you want to build the magic link from the model but don\'t want to send\n# an email you can you can use `MagicLink.generate_url()`\nmagic_link_url = magiclink.generate_url(request)\n```\n',
    'author': 'Matt Pye',
    'author_email': 'pyematt@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pyepye/django-magiclink',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
