"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('Map State', () => {
    test('State Machine With Map State', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const map = new stepfunctions.Map(stack, 'Map State', {
            maxConcurrency: 1,
            itemsPath: stepfunctions.Data.stringAt('$.inputForMap'),
            parameters: {
                foo: 'foo',
                bar: stepfunctions.Data.stringAt('$.bar'),
            },
        });
        map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
        // THEN
        expect(render(map)).toStrictEqual({
            StartAt: 'Map State',
            States: {
                'Map State': {
                    Type: 'Map',
                    End: true,
                    Parameters: { foo: 'foo', bar: '$.bar' },
                    Iterator: {
                        StartAt: 'Pass State',
                        States: {
                            'Pass State': {
                                Type: 'Pass',
                                End: true,
                            },
                        },
                    },
                    ItemsPath: '$.inputForMap',
                    MaxConcurrency: 1,
                },
            },
        });
    }),
        test('synth is successful', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: 1,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap'),
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            app.synth();
        }),
        test('fails in synthesis if iterator is missing', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: 1,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap'),
                });
                return map;
            });
            expect(() => app.synth()).toThrow(/Map state must have a non-empty iterator/);
        }),
        test('fails in synthesis when maxConcurrency is a float', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: 1.2,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap'),
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            expect(() => app.synth()).toThrow(/maxConcurrency has to be a positive integer/);
        }),
        test('fails in synthesis when maxConcurrency is a negative integer', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: -1,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap'),
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            expect(() => app.synth()).toThrow(/maxConcurrency has to be a positive integer/);
        }),
        test('fails in synthesis when maxConcurrency is too big to be an integer', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: Number.MAX_VALUE,
                    itemsPath: stepfunctions.Data.stringAt('$.inputForMap'),
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            expect(() => app.synth()).toThrow(/maxConcurrency has to be a positive integer/);
        }),
        test('isPositiveInteger is false with negative number', () => {
            expect(stepfunctions.isPositiveInteger(-1)).toEqual(false);
        }),
        test('isPositiveInteger is false with decimal number', () => {
            expect(stepfunctions.isPositiveInteger(1.2)).toEqual(false);
        }),
        test('isPositiveInteger is false with a value greater than safe integer', () => {
            const valueToTest = Number.MAX_SAFE_INTEGER + 1;
            expect(stepfunctions.isPositiveInteger(valueToTest)).toEqual(false);
        }),
        test('isPositiveInteger is true with 0', () => {
            expect(stepfunctions.isPositiveInteger(0)).toEqual(true);
        }),
        test('isPositiveInteger is true with 10', () => {
            expect(stepfunctions.isPositiveInteger(10)).toEqual(true);
        }),
        test('isPositiveInteger is true with max integer value', () => {
            expect(stepfunctions.isPositiveInteger(Number.MAX_SAFE_INTEGER)).toEqual(true);
        });
});
function render(sm) {
    return new cdk.Stack().resolve(new stepfunctions.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
function createAppWithMap(mapFactory) {
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'my-stack');
    const map = mapFactory(stack);
    new stepfunctions.StateGraph(map, 'Test Graph');
    return app;
}
//# sourceMappingURL=data:application/json;base64,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