<a name="aea.aea_builder"></a>
# aea.aea`_`builder

This module contains utilities for building an AEA.


<a name="aea.aea_builder.AEABuilder"></a>
## AEABuilder Objects

```python
class AEABuilder()
```

This class helps to build an AEA.

It follows the fluent interface. Every method of the builder
returns the instance of the builder itself.

Note: the method 'build()' is guaranteed of being
re-entrant with respect to the 'add_component(path)'
method. That is, you can invoke the building method
many times against the same builder instance, and the
returned agent instance will not share the
components with other agents, e.g.:

builder = AEABuilder()
builder.add_component(...)
...

# first call
my_aea_1 = builder.build()

# following agents will have different components.
my_aea_2 = builder.build()  # all good

However, if you manually loaded some of the components and added
them with the method 'add_component_instance()', then calling build
more than one time is prevented:

builder = AEABuilder()
builder.add_component_instance(...)
...  # other initialization code

# first call
my_aea_1 = builder.build()

# second call to `build()` would raise a Value Error.
# call reset
builder.reset()

# re-add the component and private keys
builder.add_component_instance(...)
... # add private keys

# second call
my_aea_2 = builder.builder()

<a name="aea.aea_builder.AEABuilder.__init__"></a>
#### `__`init`__`

```python
 | __init__(with_default_packages: bool = True)
```

Initialize the builder.

**Arguments**:

- `with_default_packages`: add the default packages.

<a name="aea.aea_builder.AEABuilder.reset"></a>
#### reset

```python
 | reset(is_full_reset: bool = False) -> None
```

Reset the builder.

A full reset causes a reset of all data on the builder. A partial reset
only resets:
- name,
- private keys, and
- component instances

**Arguments**:

- `is_full_reset`: whether it is a full reset or not.

**Returns**:

None

<a name="aea.aea_builder.AEABuilder.set_timeout"></a>
#### set`_`timeout

```python
 | set_timeout(timeout: Optional[float]) -> "AEABuilder"
```

Set agent loop idle timeout in seconds.

**Arguments**:

- `timeout`: timeout in seconds

**Returns**:

self

<a name="aea.aea_builder.AEABuilder.set_execution_timeout"></a>
#### set`_`execution`_`timeout

```python
 | set_execution_timeout(execution_timeout: Optional[float]) -> "AEABuilder"
```

Set agent execution timeout in seconds.

**Arguments**:

- `execution_timeout`: execution_timeout in seconds

**Returns**:

self

<a name="aea.aea_builder.AEABuilder.set_max_reactions"></a>
#### set`_`max`_`reactions

```python
 | set_max_reactions(max_reactions: Optional[int]) -> "AEABuilder"
```

Set agent max reaction in one react.

**Arguments**:

- `max_reactions`: int

**Returns**:

self

<a name="aea.aea_builder.AEABuilder.set_decision_maker_handler"></a>
#### set`_`decision`_`maker`_`handler

```python
 | set_decision_maker_handler(decision_maker_handler_dotted_path: str, file_path: Path) -> "AEABuilder"
```

Set decision maker handler class.

**Arguments**:

- `decision_maker_handler_dotted_path`: the dotted path to the decision maker handler
- `file_path`: the file path to the file which contains the decision maker handler

**Returns**:

self

<a name="aea.aea_builder.AEABuilder.set_skill_exception_policy"></a>
#### set`_`skill`_`exception`_`policy

```python
 | set_skill_exception_policy(skill_exception_policy: Optional[ExceptionPolicyEnum]) -> "AEABuilder"
```

Set skill exception policy.

**Arguments**:

- `skill_exception_policy`: the policy

**Returns**:

self

<a name="aea.aea_builder.AEABuilder.set_default_routing"></a>
#### set`_`default`_`routing

```python
 | set_default_routing(default_routing: Dict[PublicId, PublicId]) -> "AEABuilder"
```

Set default routing.

This is a map from public ids (protocols) to public ids (connections).

**Arguments**:

- `default_routing`: the default routing mapping

**Returns**:

self

<a name="aea.aea_builder.AEABuilder.set_loop_mode"></a>
#### set`_`loop`_`mode

```python
 | set_loop_mode(loop_mode: Optional[str]) -> "AEABuilder"
```

Set the loop mode.

**Arguments**:

- `loop_mode`: the agent loop mode

**Returns**:

self

<a name="aea.aea_builder.AEABuilder.set_runtime_mode"></a>
#### set`_`runtime`_`mode

```python
 | set_runtime_mode(runtime_mode: Optional[str]) -> "AEABuilder"
```

Set the runtime mode.

**Arguments**:

- `runtime_mode`: the agent runtime mode

**Returns**:

self

<a name="aea.aea_builder.AEABuilder.set_search_service_address"></a>
#### set`_`search`_`service`_`address

```python
 | set_search_service_address(search_service_address: str) -> "AEABuilder"
```

Set the search service address.

**Arguments**:

- `search_service_address`: the search service address

**Returns**:

self

<a name="aea.aea_builder.AEABuilder.set_name"></a>
#### set`_`name

```python
 | set_name(name: str) -> "AEABuilder"
```

Set the name of the agent.

**Arguments**:

- `name`: the name of the agent.

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.set_default_connection"></a>
#### set`_`default`_`connection

```python
 | set_default_connection(public_id: PublicId) -> "AEABuilder"
```

Set the default connection.

**Arguments**:

- `public_id`: the public id of the default connection package.

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.add_private_key"></a>
#### add`_`private`_`key

```python
 | add_private_key(identifier: str, private_key_path: Optional[PathLike] = None, is_connection: bool = False) -> "AEABuilder"
```

Add a private key path.

**Arguments**:

- `identifier`: the identifier for that private key path.
- `private_key_path`: an (optional) path to the private key file.
If None, the key will be created at build time.
- `is_connection`: if the pair is for the connection cryptos

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.remove_private_key"></a>
#### remove`_`private`_`key

```python
 | remove_private_key(identifier: str, is_connection: bool = False) -> "AEABuilder"
```

Remove a private key path by identifier, if present.

**Arguments**:

- `identifier`: the identifier of the private key.
- `is_connection`: if the pair is for the connection cryptos

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.private_key_paths"></a>
#### private`_`key`_`paths

```python
 | @property
 | private_key_paths() -> Dict[str, Optional[str]]
```

Get the private key paths.

<a name="aea.aea_builder.AEABuilder.connection_private_key_paths"></a>
#### connection`_`private`_`key`_`paths

```python
 | @property
 | connection_private_key_paths() -> Dict[str, Optional[str]]
```

Get the connection private key paths.

<a name="aea.aea_builder.AEABuilder.set_default_ledger"></a>
#### set`_`default`_`ledger

```python
 | set_default_ledger(identifier: str) -> "AEABuilder"
```

Set a default ledger API to use.

**Arguments**:

- `identifier`: the identifier of the ledger api

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.add_component"></a>
#### add`_`component

```python
 | add_component(component_type: ComponentType, directory: PathLike, skip_consistency_check: bool = False) -> "AEABuilder"
```

Add a component, given its type and the directory.

**Arguments**:

- `component_type`: the component type.
- `directory`: the directory path.
- `skip_consistency_check`: if True, the consistency check are skipped.

**Raises**:

- `AEAException`: if a component is already registered with the same component id.
| or if there's a missing dependency.

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.add_component_instance"></a>
#### add`_`component`_`instance

```python
 | add_component_instance(component: Component) -> "AEABuilder"
```

Add already initialized component object to resources or connections.

Please, pay attention, all dependencies have to be already loaded.

Notice also that this will make the call to 'build()' non re-entrant.
You will have to `reset()` the builder before calling `build()` again.

:params component: Component instance already initialized.

<a name="aea.aea_builder.AEABuilder.set_context_namespace"></a>
#### set`_`context`_`namespace

```python
 | set_context_namespace(context_namespace: Dict[str, Any]) -> "AEABuilder"
```

Set the context namespace.

<a name="aea.aea_builder.AEABuilder.remove_component"></a>
#### remove`_`component

```python
 | remove_component(component_id: ComponentId) -> "AEABuilder"
```

Remove a component.

**Arguments**:

- `component_id`: the public id of the component.

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.add_protocol"></a>
#### add`_`protocol

```python
 | add_protocol(directory: PathLike) -> "AEABuilder"
```

Add a protocol to the agent.

**Arguments**:

- `directory`: the path to the protocol directory

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.remove_protocol"></a>
#### remove`_`protocol

```python
 | remove_protocol(public_id: PublicId) -> "AEABuilder"
```

Remove protocol.

**Arguments**:

- `public_id`: the public id of the protocol

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.add_connection"></a>
#### add`_`connection

```python
 | add_connection(directory: PathLike) -> "AEABuilder"
```

Add a connection to the agent.

**Arguments**:

- `directory`: the path to the connection directory

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.remove_connection"></a>
#### remove`_`connection

```python
 | remove_connection(public_id: PublicId) -> "AEABuilder"
```

Remove a connection.

**Arguments**:

- `public_id`: the public id of the connection

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.add_skill"></a>
#### add`_`skill

```python
 | add_skill(directory: PathLike) -> "AEABuilder"
```

Add a skill to the agent.

**Arguments**:

- `directory`: the path to the skill directory

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.remove_skill"></a>
#### remove`_`skill

```python
 | remove_skill(public_id: PublicId) -> "AEABuilder"
```

Remove protocol.

**Arguments**:

- `public_id`: the public id of the skill

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.add_contract"></a>
#### add`_`contract

```python
 | add_contract(directory: PathLike) -> "AEABuilder"
```

Add a contract to the agent.

**Arguments**:

- `directory`: the path to the contract directory

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.remove_contract"></a>
#### remove`_`contract

```python
 | remove_contract(public_id: PublicId) -> "AEABuilder"
```

Remove protocol.

**Arguments**:

- `public_id`: the public id of the contract

**Returns**:

the AEABuilder

<a name="aea.aea_builder.AEABuilder.build"></a>
#### build

```python
 | build(connection_ids: Optional[Collection[PublicId]] = None) -> AEA
```

Build the AEA.

This method is re-entrant only if the components have been
added through the method 'add_component'. If some of them
have been loaded with 'add_component_instance', it
can be called only once, and further calls are only possible
after a call to 'reset' and re-loading of the components added
via 'add_component_instance' and the private keys.

**Arguments**:

- `connection_ids`: select only these connections to run the AEA.

**Returns**:

the AEA object.

**Raises**:

- `ValueError`: if we cannot

<a name="aea.aea_builder.AEABuilder.set_from_configuration"></a>
#### set`_`from`_`configuration

```python
 | set_from_configuration(agent_configuration: AgentConfig, aea_project_path: Path, skip_consistency_check: bool = False) -> None
```

Set builder variables from AgentConfig.

:params agent_configuration: AgentConfig to get values from.
:params aea_project_path: PathLike root directory of the agent project.

**Arguments**:

- `skip_consistency_check`: if True, the consistency check are skipped.

**Returns**:

None

<a name="aea.aea_builder.AEABuilder.from_aea_project"></a>
#### from`_`aea`_`project

```python
 | @classmethod
 | from_aea_project(cls, aea_project_path: PathLike, skip_consistency_check: bool = False) -> "AEABuilder"
```

Construct the builder from an AEA project.

- load agent configuration file
- set name and default configurations
- load private keys
- load ledger API configurations
- set default ledger
- load every component

**Arguments**:

- `aea_project_path`: path to the AEA project.
- `skip_consistency_check`: if True, the consistency check are skipped.

**Returns**:

an AEABuilder.

