"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const path = require("path");
const asg = require("@aws-cdk/aws-autoscaling");
const aws_autoscaling_hooktargets_1 = require("@aws-cdk/aws-autoscaling-hooktargets");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
/**
 * @experimental
 */
class GitlabRunnerAutoscaling extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        const token = props.gitlabToken;
        const tags = (_a = props.tags) !== null && _a !== void 0 ? _a : ['gitlab', 'awscdk', 'runner'];
        const gitlabUrl = (_b = props.gitlabUrl) !== null && _b !== void 0 ? _b : 'https://gitlab.com/';
        const instanceType = (_c = props.instanceType) !== null && _c !== void 0 ? _c : 't3.micro';
        const userData = ec2.UserData.forLinux();
        userData.addCommands('yum update -y ', 'sleep 15 && yum install docker git -y && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock', 'systemctl restart docker && systemctl enable docker', `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register gitlab/gitlab-runner:alpine register --non-interactive --url ${gitlabUrl} --registration-token ${token} \
      --executor docker --docker-image "alpine:latest" \
      --description "A Runner on EC2 Instance (${instanceType})" --tag-list "${tags.join(',')}" --docker-privileged ${this.dockerVolumesList(props === null || props === void 0 ? void 0 : props.dockerVolumes)} `, 'sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner gitlab/gitlab-runner:alpine', 'usermod -aG docker ssm-user');
        this.instanceRole = (_d = props.instanceRole) !== null && _d !== void 0 ? _d : new iam.Role(this, 'GitlabRunnerInstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            description: 'For EC2 Instance (Gitlab Runner) Role',
        });
        this.instanceRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        this.vpc = (_e = props.vpc) !== null && _e !== void 0 ? _e : new ec2.Vpc(this, 'VPC');
        this.securityGroup = new ec2.SecurityGroup(this, 'GitlabRunnerSecurityGroup', {
            vpc: this.vpc,
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.instanceRole.roleName],
        });
        const lt = new ec2.CfnLaunchTemplate(this, 'GitlabRunnerLaunchTemplate', {
            launchTemplateData: {
                imageId: ec2.MachineImage.latestAmazonLinux({
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                }).getImage(this).imageId,
                instanceType: instanceType,
                instanceMarketOptions: {
                    marketType: props.spotInstance ? 'spot' : undefined,
                    spotOptions: props.spotInstance ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
                blockDeviceMappings: [
                    {
                        deviceName: '/dev/xvda',
                        ebs: {
                            volumeSize: (_f = props.ebsSize) !== null && _f !== void 0 ? _f : 60,
                        },
                    },
                ],
                iamInstanceProfile: {
                    arn: instanceProfile.attrArn,
                },
                securityGroupIds: this.securityGroup.connections.securityGroups.map((m) => m.securityGroupId),
            },
        });
        this.autoscalingGroup = new asg.AutoScalingGroup(this, 'GitlabRunnerAutoscalingGroup', {
            instanceType: new ec2.InstanceType(instanceType),
            autoScalingGroupName: `Gitlab Runners (${instanceType})`,
            vpc: this.vpc,
            vpcSubnets: props.vpcSubnet,
            machineImage: ec2.MachineImage.latestAmazonLinux({
                generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            minCapacity: props.minCapacity,
            maxCapacity: props.maxCapacity,
            desiredCapacity: props.desiredCapacity,
        });
        const cfnAsg = this.autoscalingGroup.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        this.autoscalingGroup.node.tryRemoveChild('LaunchConfig');
        const unregisterPolicy = new iam.Policy(this, 'GitlabRunnerUnregisterPolicy', {
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: [
                        'ssm:SendCommand',
                        'autoscaling:DescribeAutoScalingGroups',
                        'logs:CreateLogGroup',
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                }),
            ],
        });
        const unregisterRole = new iam.Role(this, 'GitlabRunnerUnregisterRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            description: 'For Gitlab Runner Unregistering Function Role',
        });
        unregisterRole.attachInlinePolicy(unregisterPolicy);
        // Lambda function to unregiser runners on terminating instance
        const unregisterFunction = new lambda.Function(this, 'GitlabRunnerUnregisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets')),
            handler: 'autoscaling_events.unregister',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHook', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_TERMINATING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(unregisterFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        // Lambda function to unregiser runners on destroying stack
        const unregisterCustomResource = new lambda.Function(this, 'GitlabRunnerUnregisterCustomResource', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets')),
            handler: 'autoscaling_events.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const unregisterProvider = new cr.Provider(this, 'GitlabRunnerUnregisterProvider', {
            onEventHandler: unregisterCustomResource,
        });
        const customResource = new cdk.CustomResource(this, 'GitlabRunnerCustomResource', {
            serviceToken: unregisterProvider.serviceToken,
            properties: {
                AutoScalingGroupNames: [this.autoscalingGroup.autoScalingGroupName],
            },
        });
        customResource.node.addDependency(unregisterProvider);
        new cdk.CfnOutput(this, 'GitlabRunnerAutoScalingGroupArn', {
            value: this.autoscalingGroup.autoScalingGroupArn,
        });
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2l0bGFiLXJ1bm5lci1hdXRvc2NhbGluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9naXRsYWItcnVubmVyLWF1dG9zY2FsaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDZCQUE2QjtBQUM3QixnREFBZ0Q7QUFDaEQsc0ZBQW9FO0FBQ3BFLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsOENBQThDO0FBQzlDLDBDQUEwQztBQUMxQyxxQ0FBcUM7QUFDckMsZ0RBQWdEOzs7O0FBNEtoRCxNQUFhLHVCQUF3QixTQUFRLEdBQUcsQ0FBQyxTQUFTOzs7O0lBc0J4RCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQW1DOztRQUMvRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLE1BQU0sS0FBSyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDaEMsTUFBTSxJQUFJLFNBQUcsS0FBSyxDQUFDLElBQUksbUNBQUksQ0FBQyxRQUFRLEVBQUUsUUFBUSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzFELE1BQU0sU0FBUyxTQUFHLEtBQUssQ0FBQyxTQUFTLG1DQUFJLHFCQUFxQixDQUFDO1FBQzNELE1BQU0sWUFBWSxTQUFHLEtBQUssQ0FBQyxZQUFZLG1DQUFJLFVBQVUsQ0FBQztRQUN0RCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBRXpDLFFBQVEsQ0FBQyxXQUFXLENBQ2xCLGdCQUFnQixFQUNoQixrSUFBa0ksRUFDbEkscURBQXFELEVBQ3JEO21HQUM2RixTQUFTLHlCQUF5QixLQUFLOztpREFFekYsWUFBWSxrQkFBa0IsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLGFBQUwsS0FBSyx1QkFBTCxLQUFLLENBQUUsYUFBYSxDQUFDLEdBQUcsRUFDL0osNkxBQTZMLEVBQzdMLDZCQUE2QixDQUM5QixDQUFDO1FBRUYsSUFBSSxDQUFDLFlBQVksU0FDZixLQUFLLENBQUMsWUFBWSxtQ0FDbEIsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSwwQkFBMEIsRUFBRTtZQUM3QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7WUFDeEQsV0FBVyxFQUFFLHVDQUF1QztTQUNyRCxDQUFDLENBQUM7UUFDTCxJQUFJLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUNoQyxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDhCQUE4QixDQUFDLENBQzNFLENBQUM7UUFFRixJQUFJLENBQUMsR0FBRyxTQUFHLEtBQUssQ0FBQyxHQUFHLG1DQUFJLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFakQsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLDJCQUEyQixFQUFFO1lBQzVFLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztTQUNkLENBQUMsQ0FBQztRQUNILE1BQU0sZUFBZSxHQUFHLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUMxRSxLQUFLLEVBQUUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQztTQUNwQyxDQUFDLENBQUM7UUFDSCxNQUFNLEVBQUUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsNEJBQTRCLEVBQUU7WUFDdkUsa0JBQWtCLEVBQUU7Z0JBQ2xCLE9BQU8sRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDO29CQUMxQyxVQUFVLEVBQUUsR0FBRyxDQUFDLHFCQUFxQixDQUFDLGNBQWM7aUJBQ3JELENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTztnQkFDekIsWUFBWSxFQUFFLFlBQVk7Z0JBQzFCLHFCQUFxQixFQUFFO29CQUNyQixVQUFVLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxTQUFTO29CQUNuRCxXQUFXLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUM7d0JBQ2hDLGdCQUFnQixFQUFFLFVBQVU7cUJBQzdCLENBQUMsQ0FBQyxDQUFDLFNBQVM7aUJBQ2Q7Z0JBQ0QsUUFBUSxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztnQkFDMUMsbUJBQW1CLEVBQUU7b0JBQ25CO3dCQUNFLFVBQVUsRUFBRSxXQUFXO3dCQUN2QixHQUFHLEVBQUU7NEJBQ0gsVUFBVSxRQUFFLEtBQUssQ0FBQyxPQUFPLG1DQUFJLEVBQUU7eUJBQ2hDO3FCQUNGO2lCQUNGO2dCQUNELGtCQUFrQixFQUFFO29CQUNsQixHQUFHLEVBQUUsZUFBZSxDQUFDLE9BQU87aUJBQzdCO2dCQUNELGdCQUFnQixFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQ2pFLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUN6QjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSw4QkFBOEIsRUFBRTtZQUNyRixZQUFZLEVBQUUsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLFlBQVksQ0FBQztZQUNoRCxvQkFBb0IsRUFBRSxtQkFBbUIsWUFBWSxHQUFHO1lBQ3hELEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNiLFVBQVUsRUFBRSxLQUFLLENBQUMsU0FBUztZQUMzQixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQztnQkFDL0MsVUFBVSxFQUFFLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxjQUFjO2FBQ3JELENBQUM7WUFDRixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZTtTQUN2QyxDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQTRCLENBQUM7UUFDekYsTUFBTSxDQUFDLDJCQUEyQixDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDOUQsTUFBTSxDQUFDLG1CQUFtQixDQUFDLGdCQUFnQixFQUFFO1lBQzNDLGdCQUFnQixFQUFFLEVBQUUsQ0FBQyxHQUFHO1lBQ3hCLE9BQU8sRUFBRSxFQUFFLENBQUMsdUJBQXVCO1NBQ3BDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRTFELE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSw4QkFBOEIsRUFBRTtZQUM1RSxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO29CQUN4QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7b0JBQ2hCLE9BQU8sRUFBRTt3QkFDUCxpQkFBaUI7d0JBQ2pCLHVDQUF1Qzt3QkFDdkMscUJBQXFCO3dCQUNyQixzQkFBc0I7d0JBQ3RCLG1CQUFtQjtxQkFDcEI7aUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSw0QkFBNEIsRUFBRTtZQUN0RSxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7WUFDM0QsV0FBVyxFQUFFLCtDQUErQztTQUM3RCxDQUFDLENBQUM7UUFDSCxjQUFjLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUVwRCwrREFBK0Q7UUFDL0QsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGdDQUFnQyxFQUFFO1lBQ3JGLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxXQUFXLENBQUMsQ0FBQztZQUM5RCxPQUFPLEVBQUUsK0JBQStCO1lBQ3hDLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVU7WUFDbEMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUNqQyxJQUFJLEVBQUUsY0FBYztZQUNwQixZQUFZLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxPQUFPO1NBQ3pDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxnQkFBZ0IsQ0FBQywyQkFBMkIsRUFBRTtZQUNsRSxtQkFBbUIsRUFBRSxHQUFHLENBQUMsbUJBQW1CLENBQUMsb0JBQW9CO1lBQ2pFLGtCQUFrQixFQUFFLElBQUksMENBQVksQ0FBQyxrQkFBa0IsQ0FBQztZQUN4RCxhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxRQUFRO1lBQ3pDLGdCQUFnQixFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztTQUMzQyxDQUFDLENBQUM7UUFFSCwyREFBMkQ7UUFDM0QsTUFBTSx3QkFBd0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLHNDQUFzQyxFQUFFO1lBQ2pHLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxXQUFXLENBQUMsQ0FBQztZQUM5RCxPQUFPLEVBQUUsNkJBQTZCO1lBQ3RDLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVU7WUFDbEMsSUFBSSxFQUFFLGNBQWM7WUFDcEIsWUFBWSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTztTQUN6QyxDQUFDLENBQUM7UUFFSCxNQUFNLGtCQUFrQixHQUFHLElBQUksRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsZ0NBQWdDLEVBQUU7WUFDakYsY0FBYyxFQUFFLHdCQUF3QjtTQUN6QyxDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLDRCQUE0QixFQUFFO1lBQ2hGLFlBQVksRUFBRSxrQkFBa0IsQ0FBQyxZQUFZO1lBQzdDLFVBQVUsRUFBRTtnQkFDVixxQkFBcUIsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxvQkFBb0IsQ0FBQzthQUNwRTtTQUNGLENBQUMsQ0FBQztRQUNILGNBQWMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFFdEQsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxpQ0FBaUMsRUFBRTtZQUN6RCxLQUFLLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQjtTQUNqRCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBQ08saUJBQWlCLENBQUMsWUFBeUM7UUFDakUsSUFBSSxVQUFVLEdBQVcsOERBQThELENBQUM7UUFDeEYsSUFBSSxZQUFZLEVBQUU7WUFDaEIsSUFBSSxRQUFRLEdBQWEsRUFBRSxDQUFDO1lBQzVCLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQ3ZCLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxJQUFJLENBQUMsQ0FBQyxhQUFhLEdBQUcsQ0FBQyxDQUFDO1lBQ3RELENBQUMsQ0FBQyxDQUFDO1lBQ0gsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDbkIsVUFBVSxHQUFHLEdBQUcsVUFBVSxxQkFBcUIsQ0FBQyxFQUFFLENBQUM7WUFDckQsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUNELE9BQU8sVUFBVSxDQUFDO0lBQ3BCLENBQUM7Q0FDRjtBQTVMRCwwREE0TEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgYXNnIGZyb20gJ0Bhd3MtY2RrL2F3cy1hdXRvc2NhbGluZyc7XG5pbXBvcnQgeyBGdW5jdGlvbkhvb2sgfSBmcm9tICdAYXdzLWNkay9hd3MtYXV0b3NjYWxpbmctaG9va3RhcmdldHMnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBjciBmcm9tICdAYXdzLWNkay9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IERvY2tlclZvbHVtZXMgfSBmcm9tICcuL2dpdGxhYi1ydW5uZXItaW50ZXJmYWNlcyc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmdQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGdpdGxhYlRva2VuOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbnN0YW5jZVJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzcG90SW5zdGFuY2U/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWluQ2FwYWNpdHk/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWF4Q2FwYWNpdHk/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVzaXJlZENhcGFjaXR5PzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRhZ3M/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGdpdGxhYlVybD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZWJzU2l6ZT86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldD86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkb2NrZXJWb2x1bWVzPzogRG9ja2VyVm9sdW1lc1tdO1xufVxuXG5leHBvcnQgY2xhc3MgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmcgZXh0ZW5kcyBjZGsuQ29uc3RydWN0IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGluc3RhbmNlUm9sZTogaWFtLklSb2xlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGF1dG9zY2FsaW5nR3JvdXA6IGFzZy5BdXRvU2NhbGluZ0dyb3VwO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA6IGVjMi5JU2VjdXJpdHlHcm91cDtcblxuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogR2l0bGFiUnVubmVyQXV0b3NjYWxpbmdQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgY29uc3QgdG9rZW4gPSBwcm9wcy5naXRsYWJUb2tlbjtcbiAgICBjb25zdCB0YWdzID0gcHJvcHMudGFncyA/PyBbJ2dpdGxhYicsICdhd3NjZGsnLCAncnVubmVyJ107XG4gICAgY29uc3QgZ2l0bGFiVXJsID0gcHJvcHMuZ2l0bGFiVXJsID8/ICdodHRwczovL2dpdGxhYi5jb20vJztcbiAgICBjb25zdCBpbnN0YW5jZVR5cGUgPSBwcm9wcy5pbnN0YW5jZVR5cGUgPz8gJ3QzLm1pY3JvJztcbiAgICBjb25zdCB1c2VyRGF0YSA9IGVjMi5Vc2VyRGF0YS5mb3JMaW51eCgpO1xuXG4gICAgdXNlckRhdGEuYWRkQ29tbWFuZHMoXG4gICAgICAneXVtIHVwZGF0ZSAteSAnLFxuICAgICAgJ3NsZWVwIDE1ICYmIHl1bSBpbnN0YWxsIGRvY2tlciBnaXQgLXkgJiYgc3lzdGVtY3RsIHN0YXJ0IGRvY2tlciAmJiB1c2VybW9kIC1hRyBkb2NrZXIgZWMyLXVzZXIgJiYgY2htb2QgNzc3IC92YXIvcnVuL2RvY2tlci5zb2NrJyxcbiAgICAgICdzeXN0ZW1jdGwgcmVzdGFydCBkb2NrZXIgJiYgc3lzdGVtY3RsIGVuYWJsZSBkb2NrZXInLFxuICAgICAgYGRvY2tlciBydW4gLWQgLXYgL2hvbWUvZWMyLXVzZXIvLmdpdGxhYi1ydW5uZXI6L2V0Yy9naXRsYWItcnVubmVyIC12IC92YXIvcnVuL2RvY2tlci5zb2NrOi92YXIvcnVuL2RvY2tlci5zb2NrIFxcXG4gICAgICAtLW5hbWUgZ2l0bGFiLXJ1bm5lci1yZWdpc3RlciBnaXRsYWIvZ2l0bGFiLXJ1bm5lcjphbHBpbmUgcmVnaXN0ZXIgLS1ub24taW50ZXJhY3RpdmUgLS11cmwgJHtnaXRsYWJVcmx9IC0tcmVnaXN0cmF0aW9uLXRva2VuICR7dG9rZW59IFxcXG4gICAgICAtLWV4ZWN1dG9yIGRvY2tlciAtLWRvY2tlci1pbWFnZSBcImFscGluZTpsYXRlc3RcIiBcXFxuICAgICAgLS1kZXNjcmlwdGlvbiBcIkEgUnVubmVyIG9uIEVDMiBJbnN0YW5jZSAoJHtpbnN0YW5jZVR5cGV9KVwiIC0tdGFnLWxpc3QgXCIke3RhZ3Muam9pbignLCcpfVwiIC0tZG9ja2VyLXByaXZpbGVnZWQgJHt0aGlzLmRvY2tlclZvbHVtZXNMaXN0KHByb3BzPy5kb2NrZXJWb2x1bWVzKX0gYCxcbiAgICAgICdzbGVlcCAyICYmIGRvY2tlciBydW4gLS1yZXN0YXJ0IGFsd2F5cyAtZCAtdiAvaG9tZS9lYzItdXNlci8uZ2l0bGFiLXJ1bm5lcjovZXRjL2dpdGxhYi1ydW5uZXIgLXYgL3Zhci9ydW4vZG9ja2VyLnNvY2s6L3Zhci9ydW4vZG9ja2VyLnNvY2sgLS1uYW1lIGdpdGxhYi1ydW5uZXIgZ2l0bGFiL2dpdGxhYi1ydW5uZXI6YWxwaW5lJyxcbiAgICAgICd1c2VybW9kIC1hRyBkb2NrZXIgc3NtLXVzZXInLFxuICAgICk7XG5cbiAgICB0aGlzLmluc3RhbmNlUm9sZSA9XG4gICAgICBwcm9wcy5pbnN0YW5jZVJvbGUgPz9cbiAgICAgIG5ldyBpYW0uUm9sZSh0aGlzLCAnR2l0bGFiUnVubmVySW5zdGFuY2VSb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgZGVzY3JpcHRpb246ICdGb3IgRUMyIEluc3RhbmNlIChHaXRsYWIgUnVubmVyKSBSb2xlJyxcbiAgICAgIH0pO1xuICAgIHRoaXMuaW5zdGFuY2VSb2xlLmFkZE1hbmFnZWRQb2xpY3koXG4gICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblNTTU1hbmFnZWRJbnN0YW5jZUNvcmUnKSxcbiAgICApO1xuXG4gICAgdGhpcy52cGMgPSBwcm9wcy52cGMgPz8gbmV3IGVjMi5WcGModGhpcywgJ1ZQQycpO1xuXG4gICAgdGhpcy5zZWN1cml0eUdyb3VwID0gbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdHaXRsYWJSdW5uZXJTZWN1cml0eUdyb3VwJywge1xuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICB9KTtcbiAgICBjb25zdCBpbnN0YW5jZVByb2ZpbGUgPSBuZXcgaWFtLkNmbkluc3RhbmNlUHJvZmlsZSh0aGlzLCAnSW5zdGFuY2VQcm9maWxlJywge1xuICAgICAgcm9sZXM6IFt0aGlzLmluc3RhbmNlUm9sZS5yb2xlTmFtZV0sXG4gICAgfSk7XG4gICAgY29uc3QgbHQgPSBuZXcgZWMyLkNmbkxhdW5jaFRlbXBsYXRlKHRoaXMsICdHaXRsYWJSdW5uZXJMYXVuY2hUZW1wbGF0ZScsIHtcbiAgICAgIGxhdW5jaFRlbXBsYXRlRGF0YToge1xuICAgICAgICBpbWFnZUlkOiBlYzIuTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHtcbiAgICAgICAgICBnZW5lcmF0aW9uOiBlYzIuQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yLFxuICAgICAgICB9KS5nZXRJbWFnZSh0aGlzKS5pbWFnZUlkLFxuICAgICAgICBpbnN0YW5jZVR5cGU6IGluc3RhbmNlVHlwZSxcbiAgICAgICAgaW5zdGFuY2VNYXJrZXRPcHRpb25zOiB7XG4gICAgICAgICAgbWFya2V0VHlwZTogcHJvcHMuc3BvdEluc3RhbmNlID8gJ3Nwb3QnIDogdW5kZWZpbmVkLFxuICAgICAgICAgIHNwb3RPcHRpb25zOiBwcm9wcy5zcG90SW5zdGFuY2UgPyB7XG4gICAgICAgICAgICBzcG90SW5zdGFuY2VUeXBlOiAnb25lLXRpbWUnLFxuICAgICAgICAgIH0gOiB1bmRlZmluZWQsXG4gICAgICAgIH0sXG4gICAgICAgIHVzZXJEYXRhOiBjZGsuRm4uYmFzZTY0KHVzZXJEYXRhLnJlbmRlcigpKSxcbiAgICAgICAgYmxvY2tEZXZpY2VNYXBwaW5nczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGRldmljZU5hbWU6ICcvZGV2L3h2ZGEnLFxuICAgICAgICAgICAgZWJzOiB7XG4gICAgICAgICAgICAgIHZvbHVtZVNpemU6IHByb3BzLmVic1NpemUgPz8gNjAsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIGlhbUluc3RhbmNlUHJvZmlsZToge1xuICAgICAgICAgIGFybjogaW5zdGFuY2VQcm9maWxlLmF0dHJBcm4sXG4gICAgICAgIH0sXG4gICAgICAgIHNlY3VyaXR5R3JvdXBJZHM6IHRoaXMuc2VjdXJpdHlHcm91cC5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwcy5tYXAoXG4gICAgICAgICAgKG0pID0+IG0uc2VjdXJpdHlHcm91cElkLFxuICAgICAgICApLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMuYXV0b3NjYWxpbmdHcm91cCA9IG5ldyBhc2cuQXV0b1NjYWxpbmdHcm91cCh0aGlzLCAnR2l0bGFiUnVubmVyQXV0b3NjYWxpbmdHcm91cCcsIHtcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IGVjMi5JbnN0YW5jZVR5cGUoaW5zdGFuY2VUeXBlKSxcbiAgICAgIGF1dG9TY2FsaW5nR3JvdXBOYW1lOiBgR2l0bGFiIFJ1bm5lcnMgKCR7aW5zdGFuY2VUeXBlfSlgLFxuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgIHZwY1N1Ym5ldHM6IHByb3BzLnZwY1N1Ym5ldCxcbiAgICAgIG1hY2hpbmVJbWFnZTogZWMyLk1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7XG4gICAgICAgIGdlbmVyYXRpb246IGVjMi5BbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIsXG4gICAgICB9KSxcbiAgICAgIG1pbkNhcGFjaXR5OiBwcm9wcy5taW5DYXBhY2l0eSxcbiAgICAgIG1heENhcGFjaXR5OiBwcm9wcy5tYXhDYXBhY2l0eSxcbiAgICAgIGRlc2lyZWRDYXBhY2l0eTogcHJvcHMuZGVzaXJlZENhcGFjaXR5LFxuICAgIH0pO1xuXG4gICAgY29uc3QgY2ZuQXNnID0gdGhpcy5hdXRvc2NhbGluZ0dyb3VwLm5vZGUudHJ5RmluZENoaWxkKCdBU0cnKSBhcyBhc2cuQ2ZuQXV0b1NjYWxpbmdHcm91cDtcbiAgICBjZm5Bc2cuYWRkUHJvcGVydHlEZWxldGlvbk92ZXJyaWRlKCdMYXVuY2hDb25maWd1cmF0aW9uTmFtZScpO1xuICAgIGNmbkFzZy5hZGRQcm9wZXJ0eU92ZXJyaWRlKCdMYXVuY2hUZW1wbGF0ZScsIHtcbiAgICAgIExhdW5jaFRlbXBsYXRlSWQ6IGx0LnJlZixcbiAgICAgIFZlcnNpb246IGx0LmF0dHJMYXRlc3RWZXJzaW9uTnVtYmVyLFxuICAgIH0pO1xuICAgIHRoaXMuYXV0b3NjYWxpbmdHcm91cC5ub2RlLnRyeVJlbW92ZUNoaWxkKCdMYXVuY2hDb25maWcnKTtcblxuICAgIGNvbnN0IHVucmVnaXN0ZXJQb2xpY3kgPSBuZXcgaWFtLlBvbGljeSh0aGlzLCAnR2l0bGFiUnVubmVyVW5yZWdpc3RlclBvbGljeScsIHtcbiAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICdzc206U2VuZENvbW1hbmQnLFxuICAgICAgICAgICAgJ2F1dG9zY2FsaW5nOkRlc2NyaWJlQXV0b1NjYWxpbmdHcm91cHMnLFxuICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nR3JvdXAnLFxuICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAgICdsb2dzOlB1dExvZ0V2ZW50cycsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgY29uc3QgdW5yZWdpc3RlclJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ0dpdGxhYlJ1bm5lclVucmVnaXN0ZXJSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgICBkZXNjcmlwdGlvbjogJ0ZvciBHaXRsYWIgUnVubmVyIFVucmVnaXN0ZXJpbmcgRnVuY3Rpb24gUm9sZScsXG4gICAgfSk7XG4gICAgdW5yZWdpc3RlclJvbGUuYXR0YWNoSW5saW5lUG9saWN5KHVucmVnaXN0ZXJQb2xpY3kpO1xuXG4gICAgLy8gTGFtYmRhIGZ1bmN0aW9uIHRvIHVucmVnaXNlciBydW5uZXJzIG9uIHRlcm1pbmF0aW5nIGluc3RhbmNlXG4gICAgY29uc3QgdW5yZWdpc3RlckZ1bmN0aW9uID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCAnR2l0bGFiUnVubmVyVW5yZWdpc3RlckZ1bmN0aW9uJywge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMnKSksXG4gICAgICBoYW5kbGVyOiAnYXV0b3NjYWxpbmdfZXZlbnRzLnVucmVnaXN0ZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfOCxcbiAgICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDYwKSxcbiAgICAgIHJvbGU6IHVucmVnaXN0ZXJSb2xlLFxuICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX0RBWSxcbiAgICB9KTtcblxuICAgIHRoaXMuYXV0b3NjYWxpbmdHcm91cC5hZGRMaWZlY3ljbGVIb29rKCdHaXRsYWJSdW5uZXJMaWZlQ3ljbGVIb29rJywge1xuICAgICAgbGlmZWN5Y2xlVHJhbnNpdGlvbjogYXNnLkxpZmVjeWNsZVRyYW5zaXRpb24uSU5TVEFOQ0VfVEVSTUlOQVRJTkcsXG4gICAgICBub3RpZmljYXRpb25UYXJnZXQ6IG5ldyBGdW5jdGlvbkhvb2sodW5yZWdpc3RlckZ1bmN0aW9uKSxcbiAgICAgIGRlZmF1bHRSZXN1bHQ6IGFzZy5EZWZhdWx0UmVzdWx0LkNPTlRJTlVFLFxuICAgICAgaGVhcnRiZWF0VGltZW91dDogY2RrLkR1cmF0aW9uLnNlY29uZHMoNjApLFxuICAgIH0pO1xuXG4gICAgLy8gTGFtYmRhIGZ1bmN0aW9uIHRvIHVucmVnaXNlciBydW5uZXJzIG9uIGRlc3Ryb3lpbmcgc3RhY2tcbiAgICBjb25zdCB1bnJlZ2lzdGVyQ3VzdG9tUmVzb3VyY2UgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdHaXRsYWJSdW5uZXJVbnJlZ2lzdGVyQ3VzdG9tUmVzb3VyY2UnLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL2Fzc2V0cycpKSxcbiAgICAgIGhhbmRsZXI6ICdhdXRvc2NhbGluZ19ldmVudHMub25fZXZlbnQnLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfOCxcbiAgICAgIHJvbGU6IHVucmVnaXN0ZXJSb2xlLFxuICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX0RBWSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHVucmVnaXN0ZXJQcm92aWRlciA9IG5ldyBjci5Qcm92aWRlcih0aGlzLCAnR2l0bGFiUnVubmVyVW5yZWdpc3RlclByb3ZpZGVyJywge1xuICAgICAgb25FdmVudEhhbmRsZXI6IHVucmVnaXN0ZXJDdXN0b21SZXNvdXJjZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGN1c3RvbVJlc291cmNlID0gbmV3IGNkay5DdXN0b21SZXNvdXJjZSh0aGlzLCAnR2l0bGFiUnVubmVyQ3VzdG9tUmVzb3VyY2UnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IHVucmVnaXN0ZXJQcm92aWRlci5zZXJ2aWNlVG9rZW4sXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIEF1dG9TY2FsaW5nR3JvdXBOYW1lczogW3RoaXMuYXV0b3NjYWxpbmdHcm91cC5hdXRvU2NhbGluZ0dyb3VwTmFtZV0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGN1c3RvbVJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeSh1bnJlZ2lzdGVyUHJvdmlkZXIpO1xuXG4gICAgbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ0dpdGxhYlJ1bm5lckF1dG9TY2FsaW5nR3JvdXBBcm4nLCB7XG4gICAgICB2YWx1ZTogdGhpcy5hdXRvc2NhbGluZ0dyb3VwLmF1dG9TY2FsaW5nR3JvdXBBcm4sXG4gICAgfSk7XG4gIH1cbiAgcHJpdmF0ZSBkb2NrZXJWb2x1bWVzTGlzdChkb2NrZXJWb2x1bWU6IERvY2tlclZvbHVtZXNbXSB8IHVuZGVmaW5lZCk6IHN0cmluZyB7XG4gICAgbGV0IHRlbXBTdHJpbmcgOnN0cmluZyA9ICctLWRvY2tlci12b2x1bWVzIFwiL3Zhci9ydW4vZG9ja2VyLnNvY2s6L3Zhci9ydW4vZG9ja2VyLnNvY2tcIic7XG4gICAgaWYgKGRvY2tlclZvbHVtZSkge1xuICAgICAgbGV0IHRlbXBMaXN0IDpzdHJpbmdbXSA9IFtdO1xuICAgICAgZG9ja2VyVm9sdW1lLmZvckVhY2goZSA9PiB7XG4gICAgICAgIHRlbXBMaXN0LnB1c2goYFwiJHtlLmhvc3RQYXRofToke2UuY29udGFpbmVyUGF0aH1cImApO1xuICAgICAgfSk7XG4gICAgICB0ZW1wTGlzdC5mb3JFYWNoKGUgPT4ge1xuICAgICAgICB0ZW1wU3RyaW5nID0gYCR7dGVtcFN0cmluZ30gLS1kb2NrZXItdm9sdW1lcyAke2V9YDtcbiAgICAgIH0pO1xuICAgIH1cbiAgICByZXR1cm4gdGVtcFN0cmluZztcbiAgfVxufVxuIl19