import numpy as np
import pandas as pd

def standard_index(array: np.ndarray, 
                   time_index: pd.DatetimeIndex,
                   variable: str = 'var',
                   unit: str = 'unit',
                   distribution: str = 'gamma',
                   rolling: bool = True,
                   deseasonalize: bool = True,
                   window: str = '360D', 
                   agg_method: str = 'sum',
                   plot_distributions: bool = False,
                   plot_out_dir: str = 'png/',
                   plot_sxi_ts: bool = False,
                   plot_hist_bins: int = 30,
                   plot_lw: float = 2.0) -> np.ndarray:

    from my_.files.handy import create_dirs
    from my_.figures.single import square, square_top_cax
    from my_.figures.save import save_png
    from my_.plot.basic import hist, plot
    from my_.series.time import index
    from my_.plot.init_ax import init_dist, init_ts_2
    from my_.math.stats import distribution_fit, distribution_pdf, distribution_cdf
    from my_.plot.legend import color_legend

    if isinstance(array, list): array = array[0]

    if (distribution == 'gamma') and deseasonalize: 
                       
        print('\nDeseasonalize and gamma distribution')
        print('are not compatible...\n')

        raise NotImplementedError

    valid_start = time_index.iloc[0] + pd.Timedelta(window)

    time_index_valid = time_index.loc[valid_start:]

    series = pd.Series(array,
                       index = time_index)

    if deseasonalize:

        series_mean_year = series.groupby(series.index.dayofyear, 
                                          group_keys = False) \
                                          .mean().values
        
        mean_year_sub = lambda x: x - series_mean_year

        series_deseasonalized = series.groupby(series.index.year, 
                                               group_keys = False) \
                                               .apply(mean_year_sub)

        series = series_deseasonalized

    series_roll = series.rolling(window).agg(agg_method) if rolling else series

    series_roll_valid = series_roll.loc[valid_start:]

    dummy_out = np.array([np.nan] * len(series_roll_valid)) 

    if ((distribution == 'gamma') and 
        (np.any(series_roll_valid < 0) or 
         np.all(series_roll_valid == 0))): return dummy_out

    if np.all(np.isnan(series_roll_valid)): return dummy_out

    if np.all(series_roll_valid == series_roll_valid.iloc[0]): return dummy_out

    if distribution == 'gaussian_kde':

        from my_.math.stats import gauss_kde_pdf, gauss_kde_cdf

        pdf_xs, pdf_ys = gauss_kde_pdf(series_roll_valid)

        cdf_xs, cdf_ys = gauss_kde_cdf(series_roll_valid)

    else:

        parameter = distribution_fit(series_roll_valid, 
                                     distribution = distribution)

        pdf_xs, pdf_ys = distribution_pdf(distribution = distribution,
                                          parameter = parameter)
    
        cdf_xs, cdf_ys = distribution_cdf(distribution = distribution,
                                          parameter = parameter)
    
    pdf_normal_xs, pdf_normal_ys = distribution_pdf()

    cdf_normal_xs, cdf_normal_ys = distribution_cdf()

    series_roll_cdf = np.interp(series_roll_valid, 
                                cdf_xs, 
                                cdf_ys)

    series_roll_sxi = np.interp(series_roll_cdf, 
                                cdf_normal_ys, 
                                cdf_normal_xs)

    str_deseasonalize = 'deseasonalized' if deseasonalize else ''

    file_out = '_'.join([f'{variable}',
                         f'{distribution}',
                         f'{window}',
                         f'{agg_method}',
                         f'{str_deseasonalize}'])

    if plot_distributions: 

        create_dirs(plot_out_dir)
        
        fig, ax = square(4, 4)
        init_dist(ax,pdf_xs, pdf_ys, xlabel = f'{window} {agg_method} {variable} [{unit}]', ylabel = 'Probability density')
        plot(ax, xs = pdf_xs, ys = pdf_ys, lw = plot_lw, zorder = 2)
        hist(ax, series_roll, bins = plot_hist_bins, zorder = 1)
        save_png(fig, f'{plot_out_dir}/PDF_{file_out}.png')

        fig, ax = square(4, 4)
        init_dist(ax, cdf_xs, cdf_ys, xlabel = f'{window} {agg_method} {variable} [{unit}]', ylabel = 'Cummulative probability density')
        plot(ax, xs = cdf_xs, ys = cdf_ys, lw = plot_lw, zorder = 2)
        save_png(fig, f'{plot_out_dir}/CDF_{file_out}.png')
    
        fig, ax = square(4, 4)
        init_dist(ax, pdf_normal_xs, pdf_normal_ys, xlabel = 'SXI', ylabel = 'Probability density')
        plot(ax, xs = pdf_normal_xs, ys = pdf_normal_ys, lw = plot_lw, zorder = 2)
        save_png(fig, f'{plot_out_dir}/pdf_norm.png')    

        fig, ax = square(4, 4)
        init_dist(ax, cdf_normal_xs, cdf_normal_ys, xlabel = 'SXI', ylabel = 'Cummulative probability density')
        plot(ax, xs = cdf_normal_xs, ys = cdf_normal_ys, lw = plot_lw, zorder = 2)
        save_png(fig, f'{plot_out_dir}/cdf_norm.png')

    if plot_sxi_ts:
        
        create_dirs(plot_out_dir)

        fig, ax, cax = square_top_cax(fy = 4)
        init_ts_2(ax, time_index, array, xlabel = 'Time', ylabel = f'{variable} [{unit}]')
        plot(ax, xs = time_index, ys = array, colors = 'k', lw = plot_lw, zorder = 2)
        color_legend(cax, {variable: 0}, ['k', 'firebrick'])        
        save_png(fig, f'{plot_out_dir}/TS_{file_out}.png')

        if deseasonalize:

            fig, ax, cax = square_top_cax(fy = 4)
            init_ts_2(ax, time_index, series_deseasonalized, xlabel = 'Time', ylabel = f'{variable} [{unit}]')
            plot(ax, xs = time_index, ys = series_deseasonalized, colors = 'k', lw = plot_lw, zorder = 2)
            color_legend(cax, {variable: 0}, ['k', 'firebrick'])        
            save_png(fig, f'{plot_out_dir}/TS_ds_{file_out}.png')


        fig, ax, cax = square_top_cax(fy = 4)
        init_ts_2(ax, time_index, series_roll_sxi, xlabel = 'Time', ylabel = 'SXI [-]')
        plot(ax, xs = time_index_valid, ys = series_roll_sxi, lw = plot_lw, colors = 'firebrick', zorder = 2)
        color_legend(cax, {f'{variable} Standardized Index': 1}, ['k', 'firebrick'])        
        save_png(fig, f'{plot_out_dir}/SXI_{file_out}.png')

    return series_roll_sxi


if __name__ == '__main__':

    from my_.data import CLM5
    from my_.data import COSMOREA6
    from my_.data.templates import gridded_data
    
    variables = ['GPP']
    year_start = 2012
    year_end = 2020
    i_lat = 812
    i_lon = 712
    dist = 'gaussian_kde'
    window = '90D'
    deseasonalize = True
    #source = CLM5.BGC_EU3_8daily
    #COSMOREA6.EUCORDEX_3km_8daily#CLM5.BGC_EU3_8daily

    #data = gridded_data(**source)
    #arrays = data.get_values(variables,
    #                         y0 = year_start,
    #                         y1 = year_end) 
    GPP = np.array([3.69392 , 2.81603 , -0.711945 , 5.31109 , 1.63796 , 1.61217 , 1.13784 , 2.61726 , 3.81802 , 3.75727 , 3.53871 , 4.90933 , 4.95198 , 5.15394 , 5.81373 , 6.59562 , 7.39076 , 5.96018 , 5.84207 , 8.17073 , 4.25254 , 11.8514 , 5.189 , 6.45591 , 11.1777 , 10.649 , 13.3179 , 13.6022 , 10.8048 , 10.0913 , 12.3326 , 10.2265 , 6.44889 , 10.6127 , 8.01606 , 3.83121 , 7.17195 , 8.22656 , 5.31142 , 1.08123 , 3.87013 , 3.85611 , 2.93418 , 1.75488 , 3.79543 , 3.43744 , 2.18498 , 4.66021 , 3.05669 , 1.08878 , 4.57127 , 2.29017 , 2.10223 , 3.33314 , 1.01059 , 2.94224 , 5.62244 , 2.23341 , 1.47518 , 1.28729 , 1.65421 , 0.705452 , 2.28164 , 3.26798 , 1.00872 , 1.93867 , 5.68966 , 6.77373 , 7.01369 , 9.26536 , 8.98096 , 6.11828 , 6.99489 , 8.24915 , 7.14276 , 12.2939 , 11.4696 , 13.3629 , 13.6696 , 5.76542 , 7.45386 , 11.1641 , 10.6523 , 10.6293 , 11.5228 , 6.87018 , 9.52754 , 6.27285 , 5.75253 , 7.56154 , 7.69553 , 5.81597 , 5.218 , 4.23297 , 3.68273 , 3.24878 , 2.06546 , 3.07259 , 6.29606 , 4.66286 , 1.7841 , 1.81283 , 1.50838 , 2.26134 , 2.3725 , 1.88479 , 2.56402 , 5.67429 , 2.76765 , 2.24736 , 3.01273 , 4.10501 , 4.79657 , 7.47451 , 6.82342 , 5.23117 , 5.77341 , 6.02948 , 6.75167 , 6.99849 , 7.176 , 6.40459 , 8.24213 , 7.7571 , 9.38048 , 8.72395 , 8.74608 , 7.48141 , 8.97312 , 10.679 , 11.6474 , 10.641 , 10.719 , 9.9704 , 12.8094 , 10.3549 , 8.493 , 8.0811 , 7.44968 , 6.05204 , 5.32178 , 5.83105 , 9.14843 , 6.70523 , 5.83784 , 4.75379 , 5.50003 , 4.32342 , 4.06783 , 2.62912 , 2.15707 , 2.1123 , 2.09961 , 2.18101 , 1.38207 , 1.47927 , 1.83053 , 1.92745 , 1.6823 , 2.49798 , 1.97598 , 1.46845 , 1.31364 , 4.18613 , 3.48576 , 3.86642 , 4.41096 , 5.67686 , 7.29826 , 4.27638 , 6.9028 , 4.70823 , 8.07038 , 8.75423 , 10.2016 , 10.3258 , 7.93786 , 10.3758 , 7.54497 , 9.02877 , 9.07724 , 9.39096 , 6.0511 , 10.5823 , 11.7956 , 9.10696 , 7.69095 , 7.78951 , 8.73198 , 10.9554 , 10.1727 , 7.82416 , 7.14657 , 8.65542 , 5.86756 , 4.88329 , 4.76977 , 4.40676 , 3.73391 , 4.10779 , 3.69552 , 2.87059 , 2.81794 , 1.69139 , 2.97492 , 1.87399 , 1.87469 , 1.9772 , 2.30148 , 4.01164 , 4.32008 , 2.48735 , 1.5826 , 2.82315 , 2.87679 , 0.726501 , 1.3191 , 2.62533 , 2.23275 , 2.19172 , 3.87582 , 5.64592 , 6.63958 , 5.76289 , 5.88085 , 7.48857 , 8.32065 , 7.65032 , 7.8872 , 6.78035 , 8.92647 , 10.7393 , 8.86924 , 10.4517 , 12.5354 , 11.8095 , 12.4452 , 10.1069 , 9.58676 , 8.13205 , 7.794 , 7.01386 , 6.9878 , 9.26246 , 9.55166 , 5.60958 , 7.29722 , 5.59416 , 3.2339 , 4.43701 , 3.61503 , 4.36936 , 2.50411 , 2.46923 , 1.7858 , 1.39141 , 1.90898 , 1.45567 , 4.10472 , 2.39512 , 2.38005 , 3.85837 , 2.64256 , 1.51704 , -2.71732 , -2.54315 , -0.0439852 , 0.860822 , -0.528003 , 0.703209 , 5.15883 , 4.84814 , 6.31404 , 5.64727 , 7.5693 , 5.53527 , 5.41182 , 4.3904 , 5.64007 , 10.5019 , 9.72898 , 12.4043 , 11.1756 , 13.833 , 14.4086 , 11.4994 , 8.13884 , 9.39842 , 9.54336 , 9.90243 , 12.2423 , 9.59476 , 9.27729 , 13.5499 , 9.61472 , 8.03031 , 6.42434 , 9.15741 , 7.96015 , 4.35666 , 5.59064 , 5.73554 , 5.1395 , 3.48166 , 2.37688 , 2.00354 , 1.68526 , 1.24217 , 1.72121 , 4.97257 , 1.80005 , 3.23266 , 2.30911 , 1.0862 , 2.09405 , 0.945545 , 2.23345 , 3.89135 , 2.44449 , 1.8441 , 2.14409 , 3.19063 , 3.38742 , 3.70106 , 6.06198 , 7.13757 , 7.53588 , 7.69605 , 9.2713 , 9.3912 , 10.6897 , 8.62345 , 8.45775 , 6.76954 , 8.77311 , 12.6419 , 13.7649 , 8.65949 , 4.96855 , 9.88274 , 10.7816 , 8.52397 , 7.0806 , 8.49315 , 11.6308 , 9.15058 , 5.79337 , 8.02852 , 8.21708 , 5.13775 , 6.24126 , 4.90074 , 4.81529 , 8.38216 , 3.28 , -0.649416 , 2.89692 , 2.6186 , 1.4912 , 1.97951 , 2.02692 , 2.047 , 2.74335 , 2.76316 , 2.7496 , 2.76045 , 1.51606 , 1.3713 , 1.40364 , 1.78647 , 3.62673 , 4.97067 , 4.53579 , 3.72472 , 4.18809 , 5.62835 , 5.78237 , 5.19177 , 4.27765 , 5.34824 , 7.29499 , 5.31547 , 5.01371 , 7.01406 , 9.61004 , 8.48142 , 9.08221 , 12.4083 , 12.0879 , 8.66339 , 10.9162 , 8.60315 , 11.6632 , 11.1992 , 13.2009 , 11.2388 , 7.84129 , 14.0148 , 15.8934 , 7.21839 , 9.14887 , 5.43867 , 4.0466 , 3.43914 , 4.95229 , 3.05272 , 4.22759 , 3.67972 , 4.11851 , 4.5187 , 5.21754 , 3.77962 , 3.52648 , 3.71744 , 4.08843 , 2.60183 , 2.15458 , 3.608 , 3.50811 , 3.1214 , 1.97335 , 2.18224 , 2.93627 , 4.47726 , 3.97541 , 3.13996 , 4.05777 , 3.48138 , 2.67514 , 4.46868 , 3.98812 , 4.70149 , 7.15628 , 6.14138 , 4.47065 , 6.55504 , 8.27552 , 5.61507 , 7.34043 , 12.6918 , 8.47443 , 8.53085 , 12.3456 , 8.87913 , 7.21102 , 10.163 , 7.67473 , 4.98275 , 7.01894 , 8.08451 , 6.34932 , 7.93176 , 8.16545 , 5.25159 , 4.87478 , 3.86021 , 3.82768 , 2.13892 , 3.93117 , 3.56833 , 3.30271 , 2.05404 , 2.53636 , 2.8823 , 1.76973 , 1.33278 , 1.68519 , 1.32393])
    P = np.array([13.829 , 1.171 , 3.914 , 0 , 0.157 , 0.029 , 3.257 , 1.243 , 0.129 , 1.229 , 0.1 , 0.186 , 0.086 , 0.357 , 2.757 , 2.329 , 4.2 , 1.9 , 3.514 , 0.729 , 1.9 , 1.014 , 3.557 , 4.043 , 2.1 , 3.843 , 0.743 , 6.743 , 1.914 , 3.757 , 0.529 , 3.829 , 0.014 , 0.886 , 3.186 , 0 , 0.914 , 0.071 , 3.014 , 10 , 3.357 , 1.486 , 2.443 , 4.829 , 1.857 , 0.057 , 1.271 , 0.886 , 1.1 , 5.629 , 4.229 , 7.544 , 1.686 , 0.286 , 0 , 0.5 , 8.743 , 0.5 , 2.557 , 0.057 , 1.957 , 1.6 , 1.286 , 0.971 , 0.229 , 0.1 , 2.286 , 0 , 1.329 , 0.029 , 2.743 , 2.271 , 2.757 , 3.014 , 0.586 , 1.514 , 4.286 , 2.743 , 1.329 , 0 , 0.143 , 1.6 , 2.671 , 1.014 , 0.829 , 1.214 , 0.014 , 6.3 , 3.414 , 0 , 0.014 , 0.943 , 3.6 , 1.743 , 4.786 , 6.329 , 7.357 , 0.857 , 1.586 , 0.986 , 1.029 , 0.243 , 1.586 , 3.562 , 5.43 , 1.6 , 1.983 , 4.046 , 0.969 , 2.536 , 3.659 , 1.449 , 1.717 , 0.081 , 0.671 , 1.941 , 0.107 , 0.623 , 0.111 , 1.753 , 1.366 , 7.106 , 5.77 , 0.041 , 2.206 , 0.056 , 3.664 , 1.31 , 0.187 , 4.223 , 7.26 , 7.839 , 5.254 , 6.219 , 11.009 , 7.987 , 3.999 , 7.377 , 2.686 , 0.053 , 0.086 , 7.737 , 1.129 , 1.776 , 3.133 , 4.703 , 2.973 , 4.779 , 0.491 , 2.25 , 0.226 , 0.607 , 1.801 , 9.714 , 8.873 , 4.346 , 3.829 , 9.261 , 1.499 , 4.076 , 3.667 , 0.461 , 0.263 , 3.546 , 4.501 , 0.16 , 0.319 , 1.737 , 10.933 , 2.911 , 0.217 , 0 , 2.829 , 5.176 , 0.111 , 1.033 , 0.494 , 1.56 , 1.024 , 0.37 , 7.986 , 0.266 , 3.631 , 2.64 , 2.974 , 6.551 , 1.38 , 0.967 , 2.503 , 2.076 , 7.513 , 1.973 , 9.551 , 4.289 , 0 , 0 , 0 , 0 , 0 , 0.064 , 0.424 , 8.743 , 7.167 , 13.591 , 1.899 , 5.823 , 2.967 , 1.496 , 3.837 , 6.924 , 1.301 , 1.589 , 9.171 , 5.651 , 1.911 , 9.669 , 2.627 , 3.733 , 0 , 0.84 , 8.616 , 1.537 , 1.446 , 1.003 , 4.857 , 1.243 , 1.101 , 0.373 , 2.599 , 13.439 , 7.609 , 5.1 , 5.094 , 1.11 , 1.257 , 2.281 , 7.441 , 7.413 , 7.081 , 1.229 , 0.287 , 0.546 , 0.549 , 1.707 , 0 , 0.417 , 0.39 , 2.119 , 1.409 , 4.871 , 2.193 , 0.607 , 3.136 , 4.734 , 4.337 , 0.124 , 0.67 , 0.889 , 0.844 , 1.407 , 2.051 , 5.657 , 0.31 , 0.13 , 1.94 , 0.86 , 0.661 , 8.016 , 5.114 , 7.624 , 3.024 , 2.404 , 0.7 , 0.113 , 0.429 , 2.009 , 0.27 , 3.071 , 0.404 , 2.911 , 0 , 2.039 , 2.93 , 0.111 , 1.636 , 3.217 , 0.559 , 4.751 , 2.681 , 7.614 , 1.903 , 5.673 , 4.989 , 1.909 , 1.756 , 6.696 , 7.739 , 0.561 , 2.539 , 4.221 , 2.264 , 0.604 , 4.074 , 1.877 , 4.049 , 2.211 , 3.933 , 5.937 , 2.911 , 9.309 , 3.261 , 6.13 , 8.471 , 0.223 , 8.62 , 2.937 , 6.076 , 2.17 , 1.069 , 0 , 0.86 , 2.489 , 4.923 , 0.676 , 4.533 , 1.236 , 0.659 , 0 , 3.884 , 1.869 , 1.273 , 4.234 , 5.356 , 8.607 , 2.867 , 1.636 , 0.231 , 0.044 , 0.079 , 0.809 , 1.834 , 1.806 , 0 , 2.591 , 2.263 , 1.68 , 1.094 , 3.594 , 0.077 , 7.807 , 0.286 , 1.769 , 0.014 , 0.01 , 1.81 , 4.693 , 3.689 , 1.274 , 1.397 , 4.501 , 10.951 , 2.306 , 10.099 , 1.658 , 0.744 , 7.954 , 2.146 , 5.709 , 1.801 , 7.72 , 0.214 , 0.273 , 3.543 , 9.924 , 13.894 , 0.466 , 0 , 1.871 , 0.939 , 0.16 , 4.293 , 2.526 , 4.294 , 1.176 , 1.346 , 0.286 , 5.159 , 1.85 , 3.061 , 0.023 , 0.087 , 0.46 , 0.789 , 4.193 , 0.684 , 7.429 , 4.251 , 0 , 0.086 , 3.597 , 0.683 , 2.204 , 5.599 , 6.461 , 4.179 , 8.203 , 1.406 , 4.916 , 2.367 , 6.377 , 0.243 , 5.824 , 5.369 , 5.701 , 4.939 , 1.748 , 2.563 , 4.594 , 1.513 , 3.499 , 10.204 , 0 , 0.011 , 9.397 , 2.434 , 0 , 1.051 , 1.231 , 0.823 , 0.08 , 0.011 , 0 , 0.431 , 4.354 , 1.596 , 0 , 0.543 , 0 , 2.776 , 2.256 , 2.814 , 1.886 , 3.119 , 2.257 , 0.763 , 0.936 , 0.744 , 4.047 , 9.809 , 0.657 , 3.569 , 0.176 , 0.163 , 0 , 9.413 , 3.487 , 5.883 , 0.741 , 2.281 , 5.87 , 0.137 , 2.299 , 1.627 , 3.973 , 1.354 , 2.084 , 5.163 , 5.793 , ])
    time = pd.to_datetime([20120101, 20120108, 20120115, 20120122, 20120129, 20120205, 20120212, 20120219, 20120226, 20120304, 20120311, 20120318, 20120325, 20120401, 20120408, 20120415, 20120422, 20120429, 20120506, 20120513, 20120520, 20120527, 20120603, 20120610, 20120617, 20120624, 20120701, 20120708, 20120715, 20120722, 20120729, 20120805, 20120812, 20120819, 20120826, 20120902, 20120909, 20120916, 20120923, 20120930, 20121007, 20121014, 20121021, 20121028, 20121104, 20121111, 20121118, 20121125, 20121202, 20121209, 20121216, 20121223, 20130101, 20130108, 20130115, 20130122, 20130129, 20130205, 20130212, 20130219, 20130226, 20130305, 20130312, 20130319, 20130326, 20130402, 20130409, 20130416, 20130423, 20130430, 20130507, 20130514, 20130521, 20130528, 20130604, 20130611, 20130618, 20130625, 20130702, 20130709, 20130716, 20130723, 20130730, 20130806, 20130813, 20130820, 20130827, 20130903, 20130910, 20130917, 20130924, 20131001, 20131008, 20131015, 20131022, 20131029, 20131105, 20131112, 20131119, 20131126, 20131203, 20131210, 20131217, 20131224, 20140101, 20140108, 20140115, 20140122, 20140129, 20140205, 20140212, 20140219, 20140226, 20140305, 20140312, 20140319, 20140326, 20140402, 20140409, 20140416, 20140423, 20140430, 20140507, 20140514, 20140521, 20140528, 20140604, 20140611, 20140618, 20140625, 20140702, 20140709, 20140716, 20140723, 20140730, 20140806, 20140813, 20140820, 20140827, 20140903, 20140910, 20140917, 20140924, 20141001, 20141008, 20141015, 20141022, 20141029, 20141105, 20141112, 20141119, 20141126, 20141203, 20141210, 20141217, 20141224, 20150101, 20150108, 20150115, 20150122, 20150129, 20150205, 20150212, 20150219, 20150226, 20150305, 20150312, 20150319, 20150326, 20150402, 20150409, 20150416, 20150423, 20150430, 20150507, 20150514, 20150521, 20150528, 20150604, 20150611, 20150618, 20150625, 20150702, 20150709, 20150716, 20150723, 20150730, 20150806, 20150813, 20150820, 20150827, 20150903, 20150910, 20150917, 20150924, 20151001, 20151008, 20151015, 20151022, 20151029, 20151105, 20151112, 20151119, 20151126, 20151203, 20151210, 20151217, 20151224, 20160101, 20160108, 20160115, 20160122, 20160129, 20160205, 20160212, 20160219, 20160226, 20160304, 20160311, 20160318, 20160325, 20160401, 20160408, 20160415, 20160422, 20160429, 20160506, 20160513, 20160520, 20160527, 20160603, 20160610, 20160617, 20160624, 20160701, 20160708, 20160715, 20160722, 20160729, 20160805, 20160812, 20160819, 20160826, 20160902, 20160909, 20160916, 20160923, 20160930, 20161007, 20161014, 20161021, 20161028, 20161104, 20161111, 20161118, 20161125, 20161202, 20161209, 20161216, 20161223, 20170101, 20170108, 20170115, 20170122, 20170129, 20170205, 20170212, 20170219, 20170226, 20170305, 20170312, 20170319, 20170326, 20170402, 20170409, 20170416, 20170423, 20170430, 20170507, 20170514, 20170521, 20170528, 20170604, 20170611, 20170618, 20170625, 20170702, 20170709, 20170716, 20170723, 20170730, 20170806, 20170813, 20170820, 20170827, 20170903, 20170910, 20170917, 20170924, 20171001, 20171008, 20171015, 20171022, 20171029, 20171105, 20171112, 20171119, 20171126, 20171203, 20171210, 20171217, 20171224, 20180101, 20180108, 20180115, 20180122, 20180129, 20180205, 20180212, 20180219, 20180226, 20180305, 20180312, 20180319, 20180326, 20180402, 20180409, 20180416, 20180423, 20180430, 20180507, 20180514, 20180521, 20180528, 20180604, 20180611, 20180618, 20180625, 20180702, 20180709, 20180716, 20180723, 20180730, 20180806, 20180813, 20180820, 20180827, 20180903, 20180910, 20180917, 20180924, 20181001, 20181008, 20181015, 20181022, 20181029, 20181105, 20181112, 20181119, 20181126, 20181203, 20181210, 20181217, 20181224, 20190101, 20190108, 20190115, 20190122, 20190129, 20190205, 20190212, 20190219, 20190226, 20190305, 20190312, 20190319, 20190326, 20190402, 20190409, 20190416, 20190423, 20190430, 20190507, 20190514, 20190521, 20190528, 20190604, 20190611, 20190618, 20190625, 20190702, 20190709, 20190716, 20190723, 20190730, 20190806, 20190813, 20190820, 20190827, 20190903, 20190910, 20190917, 20190924, 20191001, 20191008, 20191015, 20191022, 20191029, 20191105, 20191112, 20191119, 20191126, 20191203, 20191210, 20191217, 20191224, 20200101, 20200108, 20200115, 20200122, 20200129, 20200205, 20200212, 20200219, 20200226, 20200304, 20200311, 20200318, 20200325, 20200401, 20200408, 20200415, 20200422, 20200429, 20200506, 20200513, 20200520, 20200527, 20200603, 20200610, 20200617, 20200624, 20200701, 20200708, 20200715, 20200722, 20200729, 20200805, 20200812, 20200819, 20200826, 20200902, 20200909, 20200916, 20200923, 20200930, 20201007, 20201014, 20201021, 20201028, 20201104, 20201111, 20201118, 20201125, 20201202, 20201209, 20201216, 20201223], format = '%Y%m%d').to_series()
    
    #ICOS =  np.array([4.306, 1.121, 0.365, 2.25, 1.977, 3.23, 3.087, 1.787, 0.937, 4.303, 1.66, 4.668, 1.877, 1.571, 1.3, 0.87, 2.429, 2.143, 1.097, 0.894, 2.27, 2.545, 3.937, 1.574, 2.96, 2.513, 0.684, 0.903, 3.414, 2.2, 7.002, 6.463, 2.101, 2.874, 1.869, 5.771, 4.698, 1.691, 3.674, 1.998, 1.297, 2.352, 3.642, 2.405, 4.497, 0, 6.21, 3.562, 4.185, 5.209, 3.558, 2.227, 3.029, 5.496, 4.419, 1.922, 0.6, 2.394, 2.996, 0.959, 2.079, 3.183, 3.629, 0.77, 1.757, 1.592, 4.076, 3.491, 4.101, 2.85, 3.664, 5.27, 5.448, 1.405, 2.847, 1.925, 2.664, 2.926, 1.022, 1.723, 2.745, 1.686, 1.777, 6.622, 3.959, 2.355, 6.195, 1.695, 2.162, 2.362, 1.28, 2.765, 2.839, 4.575, 4.575, 4.09, 3.056, 4.992, 0.701, 0.704, 0.903, 2.271, 1.597, 4.185, 2.351, 3.628, 1.693, 4.288])

    standard_index(P, #arrays[variables[0]][:,i_lat, i_lon], 
                   time_index = time,
                   variable = 'P',
                   unit = 'gC/day',#source['variable_units'][variables[0]],
                   distribution = dist,
                   deseasonalize = deseasonalize,
                   agg_method = 'sum',
                   plot_distributions = True, 
                   plot_sxi_ts = True,
                   window = window,)