from __future__ import with_statement
with_statement # Code checker

import base64
import time
import sys
import tempfile
import subprocess
import select
import signal
import os
import shutil
from datetime import datetime
from optparse import OptionParser

import httplib2
import simplejson as json
import bzrlib.branch
import bzrlib.export

build_timeout = 120 # The timeout for when a process can be considered to hang
timeout_message = '''
Build terminated by Wheeljack, reason: TIMEOUT EXCEEDED. 

The build took more than %s seconds without printing to stdout or
stderr. Wheeljack assumes this means that process is stuck and terminates it.

Process output:
===============
''' % build_timeout

command_error_message = '''
Build terminated abnormally. The build command failed:

%s
'''

repository_error_message = '''
Error trying to interact with the source repository. Please make sure the
setting is correct. Error output:

%s
'''

def log_info(message, *args):
    print message % args

def log_error(message, *args):
    print >> sys.stderr, message % args

class BuildError(Exception):

    def __init__(self, cmd, revision, start, output):
        self.cmd = cmd
        self.revision = revision
        self.start = start
        self.output = output

    def __str__(self):
        return 'Command: %s failed. Output from executiion:\n\n%s' % (
            self.cmd, self.output)


class BzrExport(object):
    """A context manager which can be used to create a bzr export.

    This context manager creates an export and returns it's path from
    __enter__. The export is created within a temporary directory. Regardless
    of errors this directory is cleared upon __exit__.
    """

    def __init__(self, repository):
        self.repository = repository

    def __enter__(self):
        """Return the path used for the export."""
        self.build_container = tempfile.mkdtemp()
        build_dir = os.path.join(self.build_container, 'build')

        branch, subdir = bzrlib.branch.Branch.open_containing(self.repository)
        tree = branch.basis_tree()
        bzrlib.export.export(tree, build_dir)
        return build_dir

    def __exit__(self, type, value, traceback):
        """Remove the export directory."""
        shutil.rmtree(self.build_container)

class Report(object):
    """Report's contain the information from a build process"""

    def __init__(self, project, revision, start, end, output, success=False):
        self.project = project
        self.revision = revision
        self.start = start
        self.end = end
        self.output = output
        self.success = success


class Project(object):
    """A project encapsulates the build requirements for a specific project.

    Project's are able to check the state of the version control system to
    indicate if they need to be built. The can also start a build process.
    """

    def __init__(self, url, name, last_revision, build_cmd, repository):
        self.url = url
        self.name = name
        self.last_revision = last_revision
        self.build_cmd = build_cmd
        self.repository = repository

    def revision(self):
        """Return the current revision of this project."""
        try:
            branch, subdir = bzrlib.branch.Branch.open_containing(
                self.repository)
        except bzrlib.errors.BzrError, e:
            raise BuildError(self.build_cmd, 'checkout-problem', 
                             datetime.now(), 
                             repository_error_message % str(e))
        return str(branch.revno())

    def is_updated(self):
        """Check if the project has been updated since the last build."""
        return self.last_revision != self.revision()

    def build(self, env=None):
        """Build the project.

        This returns the output from the execution as a string in case of
        success. Problems with the build are raised as a `BuildError`.
        """
        start = datetime.now()
        export = BzrExport(self.repository)

        with export as build_dir:
            try:
                process = subprocess.Popen(self.build_cmd.split(' '),
                                           cwd=build_dir,
                                           env=env,
                                           stdout=subprocess.PIPE,
                                           stderr=subprocess.STDOUT)
            except OSError, e:
                raise BuildError(self.build_cmd, self.revision(), start,
                                 command_error_message % e.strerror)
            # Build the project in a way that it won't hang the builder in case
            # of problems.
            output = []
            while process.poll() is None:
                r, w, x = select.select([process.stdout], [], [],
                                        build_timeout)
                if len(r) == 0: # The timeout was triggered
                    os.kill(process.pid, signal.SIGKILL)
                    raise BuildError(self.build_cmd, start, self.revision(),
                                     timeout_message + ''.join(output))
                else:
                    output.append(r[0].read(1))

            output.append(process.stdout.read())

        return Report(
            self, self.revision(), start, datetime.now(), ''.join(output),
            success=process.returncode==0)


class Server(object):

    def __init__(self, url, user, passwd):
        self.url = url
        self.user = user
        self.passwd = passwd

    def request(self, url, **kwargs):
        """Do a HTTP request against the server.

        This returns a tuple of response and content."""
        auth = 'basic %s' % (base64.b64encode('%s:%s' % (self.user,
                                                         self.passwd)))
        http = httplib2.Http()
        return http.request(url, headers={'authorization': auth}, **kwargs)

    def load_projects(self):
        """Return an iterable of project objects."""
        response, content = self.request(self.url)
        if response.status != 200:
            log_error('Error connecting to the server: %s', content)
            return []

        projects = []
        for project in json.loads(content)['projects']:
            r, c = self.request(project['href'])
            projects.append(
                Project(**dict([(str(key), value) for (key, value) in
                                json.loads(c).iteritems()])))
        return projects

    def submit_build_report(self, report):
        """Submit the build report to the server."""
        data = {'start': report.start.isoformat(),
                'end': report.end.isoformat(),
                'output': report.output,
                'revision': report.revision,
                'success': report.success}
        self.request(report.project.url, method='PUT', body=json.dumps(data))

def build_projects(server):
    for project in server.load_projects():
        log_info('Building project: %s', project.name)
        log_info('\tChecking version: %s', project.repository)
        try:
            if not project.is_updated():
                log_info('\tProject is up-to-date, skipping build')
                continue
            log_info('\tStarting build for project')
            report = project.build()
        except BuildError, e:
            report = Report(project, e.revision, e.start,
                            datetime.now(), e.output)

        server.submit_build_report(report)


def main():
    from bzrlib.plugin import load_plugins
    load_plugins()

    parser = OptionParser(usage='usage: %prog [options] username password')
    parser.add_option("-l", "--url", dest="url",
                       default='http://localhost:8000/api/',
                       help="The URL for the server API [default: %default]")
    parser.add_option(
        "-c", "--continuous",
        action="store_true", dest="continuous", default=False,
        help="After building all projects pause and then run again.")
    parser.add_option(
        "-s", "--sleep", dest="sleep", metavar='MINUTES',
        default=60, type="int",
        help="Sleep the specified amount of time in between build runs.")

    options, args = parser.parse_args()

    if len(args) != 2:
        parser.print_help()
        sys.exit(1)

    log_info('Starting builds')
    server = Server(options.url, *args)

    if options.continuous:
        while True:
            build_projects(server)
            time.sleep(options.sleep * 60) # Let the system relax for a bit
    else:
        build_projects(server)


if __name__ == '__main__':
    main()
