"""
This module will handle the exceptions for this package
"""

from .metrics import metrics


def error_handler(reference, hypothesis):
    """
    This function provides the overall wrapper to handle exceptions within this package

    Parameters
    ----------
    reference : str, list or numpy array
        The ground truth transcription of a recorded speech or the expected output of a live speech.
    hypothesis : str, list or numpy array
        The text generated by a speech-to-text algorithm/system which will be compared to the reference text.

    Raises
    ------
    ValueError
        if the two input parameters do not contain the same amount of elements.
    AttributeError
        if input text is not a string, list or np.ndarray data type.

    Returns
    -------
    np.ndarray
        This function will return a ragged array containing the Word Error Rate, Levenshtein distance, the number of
        words in the reference sequence, insertions count, deletions count, substitutions count, a list of inserted
        words, a list of deleted words and a list of substituted words.
    """
    try:
        word_error_rate_breakdown = metrics(reference, hypothesis)
    except ValueError as exc:
        raise ValueError("The Reference and Hypothesis input parameters must have the same number of elements.") from\
            exc
    except AttributeError as exc:
        raise AttributeError("All text should be in a string format. Please check your input does not include any "
                             "Numeric data types.") from exc
    return word_error_rate_breakdown
