"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const hello = require("./hello-k8s");
const util_1 = require("./util");
class EksClusterStack extends util_1.TestStack {
    constructor(scope, id) {
        super(scope, id);
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal()
        });
        const cluster = new eks.Cluster(this, 'Cluster', {
            mastersRole,
            defaultCapacity: 0,
        });
        // fargate profile for resources in the "default" namespace
        cluster.addFargateProfile('default', {
            selectors: [{ namespace: 'default' }]
        });
        // add some capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        cluster.addCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            minCapacity: 3,
        });
        // spot instances (up to 10)
        cluster.addCapacity('spot', {
            spotPrice: '0.1094',
            instanceType: new ec2.InstanceType('t3.large'),
            maxCapacity: 10,
            bootstrapOptions: {
                kubeletExtraArgs: '--node-labels foo=bar,goo=far',
                awsApiRetryAttempts: 5
            }
        });
        // apply a kubernetes manifest
        cluster.addResource('HelloApp', ...hello.resources);
        // add two Helm charts to the cluster. This will be the Kubernetes dashboard and the Nginx Ingress Controller
        cluster.addChart('dashboard', { chart: 'kubernetes-dashboard', repository: 'https://kubernetes-charts.storage.googleapis.com' });
        cluster.addChart('nginx-ingress', { chart: 'nginx-ingress', repository: 'https://helm.nginx.com/stable', namespace: 'kube-system' });
        new core_1.CfnOutput(this, 'ClusterEndpoint', { value: cluster.clusterEndpoint });
        new core_1.CfnOutput(this, 'ClusterArn', { value: cluster.clusterArn });
        new core_1.CfnOutput(this, 'ClusterCertificateAuthorityData', { value: cluster.clusterCertificateAuthorityData });
        new core_1.CfnOutput(this, 'ClusterName', { value: cluster.clusterName });
    }
}
const app = new core_1.App();
// since the EKS optimized AMI is hard-coded here based on the region,
// we need to actually pass in a specific region.
new EksClusterStack(app, 'aws-cdk-eks-cluster-test');
app.synth();
//# sourceMappingURL=data:application/json;base64,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