"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateHealthChecks = void 0;
const cdk = require("@aws-cdk/core");
/**
 * Validates health check properties, throws an error if they are misconfigured.
 *
 * @param healthCheck Healthcheck property from a Virtual Node or Virtual Gateway
 */
function validateHealthChecks(healthCheck) {
    Object.keys(healthCheck)
        .filter((key) => HEALTH_CHECK_PROPERTY_THRESHOLDS[key] &&
        typeof healthCheck[key] === 'number' &&
        !cdk.Token.isUnresolved(healthCheck[key])).map((key) => {
        const [min, max] = HEALTH_CHECK_PROPERTY_THRESHOLDS[key];
        const value = healthCheck[key];
        if (value < min) {
            throw new Error(`The value of '${key}' is below the minimum threshold (expected >=${min}, got ${value})`);
        }
        if (value > max) {
            throw new Error(`The value of '${key}' is above the maximum threshold (expected <=${max}, got ${value})`);
        }
    });
}
exports.validateHealthChecks = validateHealthChecks;
/**
 * Minimum and maximum thresholds for HeathCheck numeric properties
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_HealthCheckPolicy.html
 */
const HEALTH_CHECK_PROPERTY_THRESHOLDS = {
    healthyThreshold: [2, 10],
    intervalMillis: [5000, 300000],
    port: [1, 65535],
    timeoutMillis: [2000, 60000],
    unhealthyThreshold: [2, 10],
};
//# sourceMappingURL=data:application/json;base64,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