"""Base Bidimensional Fuzzy Entropy function."""
import numpy as np    

def FuzzEn2D(Mat, m=None, tau=1, r=None, Fx='default', Logx=np.exp(1), Lock=True):
    """FuzzEn2D  estimates the bidimensional fuzzy entropy of a data matrix.
 
    .. code-block:: python
    
        Fuzz2D = FuzzEn2D(Mat) 
        
    Returns the bidimensional fuzzy entropy estimate (``Fuzz2D``) estimated for 
    the data matrix (``Mat``) using the default parameters: time delay = 1,
    fuzzy function (``Fx``) = ``'default'``, fuzzy function parameters (``r``) = (0.2, 2,
    logarithm = natural, matrix template size = [floor(H/10) floor(W/10)] 
    (where H and W represent the height (rows) and width (columns) of the data matrix ``'Mat'``) 
    ** The minimum number of rows and columns of Mat must be > 10.
 
    .. code-block:: python
    
        Fuzz2D = FuzzEn2D(Mat, keyword = value, ...)

    Returns the bidimensional fuzzy entropy (``Fuzz2D``) estimates for the data
    matrix (``Mat``) using the specified 'keyword' arguments:
        :m:     - Template submatrix dimensions, an integer (for sub-matrix with same height and width) or a two-element vector of integers [height, width] with a minimum value > 1.  (default: [floor(H/10) floor(W/10)])
        :tau:   - Time Delay, a positive integer  (default: 1)
        :Fx:    - Fuzzy funtion name, one of the following:
            {``'sigmoid'``, ``'modsampen'``, ``'default'``, ``'gudermannian'``, ``'linear'``}
        :r:     - Fuzzy function parameters, a 1 element scalar or a 2 element vector of positive values.
                The ``r`` parameters for each fuzzy function are defined as follows:
               
                  - sigmoid:      
                      r(1) = divisor of the exponential argument
                      r(2) = value subtracted from argument (pre-division)
                  - modsampen:    
                      r(1) = divisor of the exponential argument
                      r(2) = value subtracted from argument (pre-division)
                  - default:  
                      r(1) = divisor of the exponential argument
                      r(2) = argument exponent (pre-division)
                  - gudermannian:   
                      r  = a scalar whose value is the numerator of  argument to gudermannian function:
                      GD(x) = atan(tanh(r/x)). GD(x) is normalised to have a maximum value of 1.
                  - linear:        
                      r  = an integer value. When r = 0, the argument of the exponential function is 
                      normalised between [0 1]. When r = 1, the minimuum value of the exponential argument is set to 0.    
                      
        :Logx:  - Logarithm base, a positive scalar    (default: natural)
        :Lock:  - By default, ``FuzzEn2D`` only permits matrices with a maximum  size of 128 x 128 to prevent RAM overload. 
                  e.g. For ``Mat`` = [200 x 200], ``m`` = 3, and ``tau`` = 1, ``FuzzEn2D`` 
                  creates a vector of 753049836 elements. To enable matrices greater than [128 x 128] elements, set ``Lock = False`` (default: True)
                  
                  ``CAUTION: unlocking the permitted matrix size may cause memory``
                  ``errors that could lead your Python IDE to crash.``
    
    :See also:
        ``SampEn2D``, ``DistEn2D``, ``FuzzEn``, ``XFuzzEn``, ``XMSEn``
    
    :References:
        [1] Luiz Fernando Segato Dos Santos, et al.,
            "Multidimensional and fuzzy sample entropy (SampEnMF) for
            quantifying H&E histological images of colorectal cancer."
            Computers in biology and medicine 
            103 (2018): 148-160.
    
        [2] Mirvana Hilal and Anne Humeau-Heurtier,
            "Bidimensional fuzzy entropy: Principle analysis and biomedical
            applications."
            41st Annual International Conference of the IEEE (EMBC) Society
            2019.
    
    """  
    
    Mat = np.squeeze(Mat)
    NL,NW = Mat.shape     
    if m is None:
        m = np.array(Mat.shape)//10
    if r is None:
        r = (0.2*np.std(Mat), 2)

    assert Mat.ndim==2 and min(Mat.shape)>5 , \
    "Mat:   must be a 2D numpy array with height & width > 10"
    assert (isinstance(m,int) and m>1) or (isinstance(m, (np.ndarray,tuple))  
                                           and len(m)==2 and min(m)>1), \
    "m:     must be an integer > 1, or a 2 element tuple of integers > 1"
    assert isinstance(tau,int) and (tau > 0), "tau:   must be an integer > 0"
    assert isinstance(r,(int,float)) or ((r[0] >= 0) and len(r) ==2), "r:     must be 2 element tuple of positive values"
    assert isinstance(Logx,(int,float)) and (Logx>0), "Logx:     must be a positive value"
    assert Fx.lower() in ['default','sigmoid','modsampen','gudermannian','linear'] \
            and isinstance(Fx,str), "Fx:    must be one of the following strings - \
            'default', 'sigmoid', 'modsampen', 'gudermannian', 'linear'" 
    assert isinstance(Lock,bool) and ((Lock==True and max(Mat.shape)<=128) or Lock==False), \
    "Lock:      To prevent memory storage errors, matrix width & length must \
    have <= 128 elements. To estimate FuzzEn2D for the current matrix (%d x %d) \
    change Lock to False. \
    CAUTION: unlocking the permitted matrix size may cause memory \
    errors that could lead your Python IDE to crash.."%(NW,NL)
    
    if isinstance(m,int):
        mL = int(m); mW = int(m)        
    else:
        mL = int(m[0]); mW = int(m[1])   
    
    if isinstance(r,tuple) and Fx.lower()=='linear':
        r = 0
        print('Multiple values for r entered. Default value (0) used.') 
    elif isinstance(r,tuple) and Fx.lower()=='gudermannian':
        r = r[0];
        print('Multiple values for r entered. First value used.')     
        
    Fun = globals()[Fx.lower()]
    NL = NL - mL*tau
    NW = NW - mW*tau
    X1 = np.zeros((NL*NW,mL,mW))
    X2 = np.zeros((NL*NW,mL+1,mW+1))
    p = 0
    for k in range(NL):        
        for n in range(NW):
            Temp2 = Mat[k:(mL+1)*tau+k:tau,n:(mW+1)*tau+n:tau]
            Temp1 = Temp2[:-1,:-1]
            X1[p,:,:] = Temp1 - np.mean(Temp1)
            X2[p,:,:] = Temp2 - np.mean(Temp2)
            p += 1            
    if p != NL*NW:
        print('Warning: Potential error with submatrix division.')        
    Ny = int(p*(p-1)/2)
    if Ny > 300000000:
        print('Warning: Number of pairwise distance calculations is ' + str(Ny))
    
    Y1 = np.zeros(p-1)
    Y2 = np.zeros(p-1)
    for k in range(p-1):
        Temp1 = Fun(np.max(abs(X1[k+1:,:,:] - X1[k,:,:]),axis=(1,2)),r)
        Y1[k] = np.sum(Temp1)        
        Temp2 = Fun(np.max(abs(X2[k+1:,:,:] - X2[k,:,:]),axis=(1,2)),r)
        Y2[k] = np.sum(Temp2) 
        
    Fuzz2D = -np.log(sum(Y2)/sum(Y1))/np.log(Logx)
    return Fuzz2D


def sigmoid(x,r):
    assert isinstance(r,tuple), 'When Fx = "Sigmoid", r must be a two-element tuple.'
    y = 1/(1 + np.exp((x-r[1])/r[0]))
    return y  
def default(x,r):   
    assert isinstance(r,tuple), 'When Fx = "Default", r must be a two-element tuple.'
    y = np.exp(-(x**r[1])/r[0])
    return y     
def modsampen(x,r):
    assert isinstance(r,tuple), 'When Fx = "Modsampen", r must be a two-element tuple.'
    y = 1/(1 + np.exp((x-r[1])/r[0]));
    return y    
def gudermannian(x,r):
    if r <= 0:
        raise Exception('When Fx = "Gudermannian", r must be a scalar > 0.')
    y = np.arctan(np.tanh(r/x))    
    y = y/np.max(y)    
    return y    
def linear(x,r):    
    if r == 0 and x.shape[0]>1:    
        y = np.exp(-(x - np.min(x))/np.ptp(x))
    elif r == 1:
        y = np.exp(-(x - np.min(x)))
    elif r == 0 and x.shape[0]==1:   
        y = 0;
    else:
        print(r)
        raise Exception('When Fx = "Linear", r must be 0 or 1')
    return y