# -*- coding: utf-8 -*-

"""Main module."""
import threading

from pyevr import apis
from pyevr.openapi_client.api_client import ApiClient
from pyevr.openapi_client.configuration import Configuration


HOLDING_BASE_CHILDREN = (
    'ForestNotice',
    'ForestNoticeAllOf',
    'InventoryAct',
    'InventoryActAllOf',
    'ConsolidatedAct',
    'ConsolidatedActAllOf',
    'SalesContract',
    'SalesContractAllOf',
    'ForestAct',
    'ContractForTransferOfCuttingRights',
)


class ExtendedApiClient(ApiClient):
    """Extended API client generated by openapi-generator-cli."""
    DESERIALIZATION_STACKS = {}

    def deserialize_data(self, response_data, response_type):
        """Deserializes parsed json into an object.

        :param response_data: parsed json to be deserialized.
        :param response_type: class literal for
            deserialized object, or string of class name.

        :return: deserialized object.
        """
        return self._deserialize(response_data, response_type)

    def _deserialize_model(self, data, klass):
        thread_name = threading.current_thread().name
        if thread_name not in self.DESERIALIZATION_STACKS:
            self.DESERIALIZATION_STACKS[thread_name] = []
        stack = self.DESERIALIZATION_STACKS[thread_name]
        stack.append(getattr(klass, "__name__", str(klass)))

        try:
            deserialized_data = super()._deserialize_model(data, klass)
        except ValueError as e:
            if stack:
                message = "Error happened at {location}, data was {data}. Original error was {message}".format(
                    location=".".join(stack),
                    data=repr(data),
                    message=repr(e),
                )
            else:
                message = None
            stack.clear()
            if message:
                raise ValueError(message) from e
            else:
                raise e
        else:
            stack.pop()
        return deserialized_data

    def sanitize_for_serialization(self, obj):
        """Builds a JSON POST object.

        If obj is one of the subclasses of `pyevr.openapi_client.models.holding_base.HoldingBase` adds the corresponding
        type to dictionary. Otherwise returns the dictionary from `pyevr.openapi_client.api_client.ApiClient`

        :param obj: The data to serialize.
        :return: The serialized form of data.
        """
        obj_dict = super().sanitize_for_serialization(obj)

        # Add holding base type to dictionary
        class_name = obj.__class__.__name__
        if class_name in HOLDING_BASE_CHILDREN:
            if class_name.endswith('AllOf'):
                class_name = class_name[:-5]
            obj_dict['type'] = class_name

        return obj_dict


class EVRClient(object):
    """API client class for EVR.

    :param api_key: Company API key in EVR
    :param host: EVR host. Defaults to test host (optional)
    """

    def __init__(self, api_key: str, host: str = None):
        configuration = Configuration(api_key={'EVR-APIKEY': api_key})
        if host is not None:
            configuration.host = host
        self.openapi_client = ExtendedApiClient(configuration)

        self.assortments = apis.AssortmentsAPI(self.openapi_client)
        self.certificates = apis.CertificatesAPI(self.openapi_client)
        self.measurements = apis.MeasurementsAPI(self.openapi_client)
        self.measurement_units = apis.MeasurementUnitsAPI(self.openapi_client)
        self.organizations = apis.OrganizationsAPI(self.openapi_client)
        self.place_of_deliveries = apis.PlaceOfDeliveriesAPI(self.openapi_client)
        self.waybills = apis.WaybillsAPI(self.openapi_client)

    def deserialize_data(self, response_data, response_type):
        return self.openapi_client.deserialize_data(response_data, response_type)
