# Graph Embedding
### Introduction
This module provides the services and implementation for various graph embedding models.

## Getting Started
### Installation

You can install the DGLL Graph Embedding version 1.0.3 from [PyPI](https://pypi.org/project/dgllge/) as:

    pip install dgllge

## Usage and Tutorial
#### input graph
```
# import module
import ge


# Set Path to Data
data_dir = "Your Path to Data"
dataset = "File/Dataset Name"


# Load a Graph
inputGraph = ge.loadGraph(data_dir, dataset)
```

#### Configurable Parameter for Graph Embedding
```
embedDim = 2 # embedding size
numbOfWalksPerVertex = 2 # walks per vertex
walkLength = 4 # walk lenght
lr =0.025 # learning rate
windowSize = 3 # window size
```

#### Choose One of the Following Graph Embedding Models
```
# DeepWalk
rw = ge.DeepWalk(inputGraph, walkLength=walkLength, embedDim=embedDim, numbOfWalksPerVertex=numbOfWalksPerVertex, \
              windowSize=windowSize, lr = lr)
              
 ```
```
# Node2Vec
rw = ge.Node2vec(inputGraph, walkLength=walkLength, embedDim=embedDim, numbOfWalksPerVertex=numbOfWalksPerVertex, \
               windowSize=windowSize, lr=lr, p = 0.5, q = 0.8)
```
```
# Struc2Vec
rw = ge.Struc2Vec(inputGraph, walkLength=walkLength, embedDim=embedDim, numbOfWalksPerVertex=numbOfWalksPerVertex, \
              windowSize=windowSize, lr = lr)
```
              
#### Skip Gram model
```
modelSkipGram = ge.SkipGramModel(rw.totalNodes, rw.embedDim)
```
#### Want Node Embedding or Edge Embedding
```
# Learning Node Embedding
model = rw.learnNodeEmbedding(modelSkipGram)
```


```
# Learning Edge Embedding
model = rw.learnEdgeEmbedding(modelSkipGram)
```


#### Plot Embedding
```
ge.plot_2DEmbedding(rw)
```

#### Save Embedding to Disk
```
ge.saveEmbedding(data_dir, dataset, rw)
```
#### Generate  Embedding for a Specific Node or Edge
```
node1 = 35
node2 = 40

# Get Embedding for a node
emb = rw.getNodeEmbedding(node1)
print("Node Embedding", emb)

# Get Embedding for an edge
emb = rw.getEdgeEmbedding(node1, node2)
print("Edge Embedding", emb)
```

### Utilize embedding for training classification models/classifiers

```
# import modules
import ge
import numpy as np
import scipy.sparse as sp
```

#### Load Embedding
```
idsEmbeddingClsLabels = np.genfromtxt("cora.embedding", dtype=np.dtype(str))
```

#### Prepare data for training 
```
labels = idsEmbeddingClsLabels[:, -1]
embedding = sp.csr_matrix(idsEmbeddingClsLabels[:, 1:-1], dtype=np.float32)
```

```
tr = ge.TrainingClassifiers()
y = tr.labelEnocder(labels)
```
#### Prepare Train test data
```
X_train, X_test, y_train, y_test = tr.prepareTrainTestData(embedding, labels, 0.33)
```

#### Choose one of the following classifier for training a classifier

```
y_pred = tr.applyDecisionTree(X_train.toarray(), y_train, X_test.toarray())
```
```
y_pred = tr.applyLogistic(X_train.toarray(), y_train, X_test.toarray())
```

```
y_pred = tr.applyRandomForest(X_train.toarray(), y_train, X_test.toarray())
```
```
y_pred = tr.apply_GradientBoosting(X_train.toarray(), y_train, X_test.toarray())
```
```
y_pred = tr.applyMLP(X_train.toarray(), y_train, X_test.toarray())
```
#### Get Accuracy
```
print("Accuracy:", tr.accuracy(y_test, y_pred))
```
<!-- LICENSE.txt -->
## License

Distributed under the MIT License. See `LICENSE` for more information.

<p align="right">(<a href="#top">back to top</a>)</p>

<!-- ACKNOWLEDGMENTS -->
## Acknowledgments
* [Data and Knowlege Engineering Lab (DKE)](http://dke.khu.ac.kr/)
<p align="right">(<a href="#top">back to top</a>)</p>

## Links
Project's GitHub Link: [@Graph-Embedding](https://github.com/sahibzada-irfanullah/Graph-Embedding)

Project's PyPI Link: [@dgllge](https://pypi.org/project/dgllge/)

<p align="right">(<a href="#top">back to top</a>)</p>