"""Utilities based on numpy arrays."""
import os
import random
import typing as t
from unittest.mock import MagicMock

import numpy as np
import numpy.typing as npt


def seed_python_numpy(seed: int) -> None:
    """Seed all randomness generated by the Python standard library and numpy.

    Args:
        seed (int): Seed to use for the random number generation.
    """
    random.seed(seed)
    np.random.seed(seed)
    os.environ["PYTHONHASHSEED"] = str(seed)


AnyT = t.TypeVar("AnyT")


def null_object(
    cls: type[AnyT],
    property_returns: dict[str, t.Any] | None = None,
    method_returns: dict[str, t.Any] | None = None,
) -> AnyT:
    """Create a null object following the Null object pattern.

    Args:
        cls: The class of the object for which a null object will be created.
        property_returns: A dictionary mapping property names to the values that
            should be returned when the property is accessed.
        method_returns: A dictionary mapping method names to the values that should
            be returned when the method is called.

    Returns:
        A dummy object having the same spec as `cls` that does nothing for any method
        call.

    Example:
        ```python
        >>> import pytest
        ... class Foo:
        ...     def __init__(self, x: int):
        ...         self.x = x
        ...     def bar(self, y: list[int]) -> None:
        ...         y.append(self.x)
        ... actual_foo = Foo(1)
        ... l = []
        ... actual_foo.bar(l)
        ... assert l == [1]
        ... null_foo = null_object(Foo)
        ... new_l = []
        ... null_foo.bar(new_l)
        ... assert new_l == []
        ... with pytest.raises(AttributeError):
        ...     null_foo.x
        ... with pytest.raises(AttributeError):
        ...     null_foo.a_method_that_does_not_exist()

        ```

    Usage:
    ------
    Instead of the following pattern:
    ```python
    >>> foo: Foo | None = None
    ... if foo is not None:
    ...     foo.bar([])

    ```

    Do:
    ```python
    >>> foo = null_object(Foo)
    ... foo.bar([])  # This will do nothing

    ```
    """
    _property_returns = property_returns or {}
    _method_returns = method_returns or {}

    class NullObject(cls):  # type: ignore
        def __init__(self, *args: t.Any, **kwargs: t.Any) -> None:
            pass

        def __getattribute__(self, __name: str) -> t.Any:
            if __name in _property_returns:
                return _property_returns[__name]
            if __name in _method_returns:
                return MagicMock(return_value=_method_returns[__name])
            if hasattr(super(), __name):
                # If the attribute exists in the parent class, but attribute was not
                # specified in method or property returns, return it's mocked version
                return MagicMock()
            return super().__getattribute__(__name)  # type: ignore

    NullObject.__name__ = cls.__name__
    return t.cast(AnyT, NullObject())


@t.overload
def rescale(
    x: float,
    old_min: float,
    old_max: float,
    new_min: float,
    new_max: float,
) -> float:
    ...


@t.overload
def rescale(
    x: npt.NDArray[t.Any],
    old_min: float | npt.NDArray[t.Any],
    old_max: float | npt.NDArray[t.Any],
    new_min: float | npt.NDArray[t.Any],
    new_max: float | npt.NDArray[t.Any],
) -> npt.NDArray[t.Any]:
    ...


@t.overload
def rescale(
    x: float | npt.NDArray[t.Any],
    old_min: npt.NDArray[t.Any],
    old_max: float | npt.NDArray[t.Any],
    new_min: float | npt.NDArray[t.Any],
    new_max: float | npt.NDArray[t.Any],
) -> npt.NDArray[t.Any]:
    ...


@t.overload
def rescale(
    x: float | npt.NDArray[t.Any],
    old_min: float | npt.NDArray[t.Any],
    old_max: npt.NDArray[t.Any],
    new_min: float | npt.NDArray[t.Any],
    new_max: float | npt.NDArray[t.Any],
) -> npt.NDArray[t.Any]:
    ...


@t.overload
def rescale(
    x: float | npt.NDArray[t.Any],
    old_min: float | npt.NDArray[t.Any],
    old_max: float | npt.NDArray[t.Any],
    new_min: npt.NDArray[t.Any],
    new_max: float | npt.NDArray[t.Any],
) -> npt.NDArray[t.Any]:
    ...


@t.overload
def rescale(
    x: float | npt.NDArray[t.Any],
    old_min: float | npt.NDArray[t.Any],
    old_max: float | npt.NDArray[t.Any],
    new_min: float | npt.NDArray[t.Any],
    new_max: npt.NDArray[t.Any],
) -> npt.NDArray[t.Any]:
    ...


def rescale(
    x: float | npt.NDArray[t.Any],
    old_min: float | npt.NDArray[t.Any],
    old_max: float | npt.NDArray[t.Any],
    new_min: float | npt.NDArray[t.Any],
    new_max: float | npt.NDArray[t.Any],
) -> float | npt.NDArray[t.Any]:
    """Rescale a value from one range to another.

    Args:
        x: The value to rescale.
        old_min: The minimum value of the old range.
        old_max: The maximum value of the old range.
        new_min: The minimum value of the new range.
        new_max: The maximum value of the new range.

    Returns:
        The rescaled value.

    NOTE:
        The returned value is a float if and only if all the arguments are floats.

    Example:
        ```python
        >>> import pytest
        >>> rescale(0.5, 0, 1, 0, 10)
        5.0
        >>> rescale(0.5, 0, 1, 10, 0)
        5.0
        >>> with pytest.raises(ValueError):
        ...     rescale(0.5, 0, 0, 0, 10)
        >>> with pytest.raises(ValueError):
        ...     rescale(0.5, 0, 1, 0, 0)

        ```
    """
    if np.allclose(old_min, old_max):
        raise ValueError("old_max must be different from old_min")
    if np.allclose(new_min, new_max):
        raise ValueError("new_max must be different from new_min")
    old_range = old_max - old_min
    new_range = new_max - new_min
    return (((x - old_min) * new_range) / old_range) + new_min
