import { Construct } from 'constructs';
/**
 * Configuration for Cognito sending emails via Amazon SES.
 *
 * @stability stable
 */
export interface UserPoolSESOptions {
    /**
     * The verified Amazon SES email address that Cognito should use to send emails.
     *
     * The email address used must be a verified email address
     * in Amazon SES and must be configured to allow Cognito to
     * send emails.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html
     * @stability stable
     */
    readonly fromEmail: string;
    /**
     * An optional name that should be used as the sender's name along with the email.
     *
     * @default - no name
     * @stability stable
     */
    readonly fromName?: string;
    /**
     * The destination to which the receiver of the email should reploy to.
     *
     * @default - same as the fromEmail
     * @stability stable
     */
    readonly replyTo?: string;
    /**
     * The name of a configuration set in Amazon SES that should be applied to emails sent via Cognito.
     *
     * @default - no configuration set
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-emailconfiguration.html#cfn-cognito-userpool-emailconfiguration-configurationset
     * @stability stable
     */
    readonly configurationSetName?: string;
    /**
     * Required if the UserPool region is different than the SES region.
     *
     * If sending emails with a Amazon SES verified email address,
     * and the region that SES is configured is different than the
     * region in which the UserPool is deployed, you must specify that
     * region here.
     *
     * Must be 'us-east-1', 'us-west-2', or 'eu-west-1'
     *
     * @default - The same region as the Cognito UserPool
     * @stability stable
     */
    readonly sesRegion?: string;
}
/**
 * Result of binding email settings with a user pool
 */
interface UserPoolEmailConfig {
    /**
     * The name of the configuration set in SES.
     *
     * @default - none
     */
    readonly configurationSet?: string;
    /**
     * Specifies whether to use Cognito's built in email functionality
     * or SES.
     *
     * @default - Cognito built in email functionality
     */
    readonly emailSendingAccount?: string;
    /**
     * Identifies either the sender's email address or the sender's
     * name with their email address.
     *
     * If emailSendingAccount is DEVELOPER then this cannot be specified.
     *
     * @default 'no-reply@verificationemail.com'
     */
    readonly from?: string;
    /**
     * The destination to which the receiver of the email should reply to.
     *
     * @default - same as `from`
     */
    readonly replyToEmailAddress?: string;
    /**
     * The ARN of a verified email address in Amazon SES.
     *
     * required if emailSendingAccount is DEVELOPER or if
     * 'from' is provided.
     *
     * @default - none
     */
    readonly sourceArn?: string;
}
/**
 * Configure how Cognito sends emails.
 *
 * @stability stable
 */
export declare abstract class UserPoolEmail {
    /**
     * Send email using Cognito.
     *
     * @stability stable
     */
    static withCognito(replyTo?: string): UserPoolEmail;
    /**
     * Send email using SES.
     *
     * @stability stable
     */
    static withSES(options: UserPoolSESOptions): UserPoolEmail;
    /**
     * Returns the email configuration for a Cognito UserPool
     * that controls how Cognito will send emails
     * @internal
     */
    abstract _bind(scope: Construct): UserPoolEmailConfig;
}
export {};
