"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const s3 = require("aws-cdk-lib/aws-s3");
const sns = require("aws-cdk-lib/aws-sns");
const core_1 = require("@aws-solutions-constructs/core");
test('All get methods return non-null objects', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = new lib_1.S3ToSns(stack, 'test-s3-sns', {});
    expect(app.snsTopic !== null);
    expect(app.s3Bucket !== null);
    expect(app.s3LoggingBucket !== null);
    expect(app.encryptionKey !== null);
    expect(app.s3BucketInterface !== null);
});
test('construct creates default event notification', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {});
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            TopicConfigurations: [
                {
                    Events: ['s3:ObjectCreated:*'],
                    TopicArn: {
                        Ref: "tests3snsSnsTopicF02F6BD0"
                    }
                }
            ]
        }
    });
});
test('construct uses existingBucketObj property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [existingBucketObj] = defaults.buildS3Bucket(stack, {
        bucketProps: {
            bucketName: 'existing-bucket-name'
        }
    });
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        existingBucketObj
    });
    expect(stack).toHaveResourceLike("AWS::S3::Bucket", {
        BucketName: 'existing-bucket-name'
    });
});
test('construct uses existing topic and key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = defaults.buildEncryptionKey(stack, {
        description: 'existing-key-description'
    });
    const [existingTopicObj] = defaults.buildTopic(stack, {
        encryptionKey: cmk,
        topicProps: {
            topicName: 'existing-topic-name'
        }
    });
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        existingTopicObj,
        existingTopicEncryptionKey: cmk
    });
    expect(stack).toHaveResourceLike("AWS::SNS::Topic", {
        TopicName: 'existing-topic-name'
    });
    expect(stack).toHaveResourceLike("AWS::KMS::Key", {
        Description: 'existing-key-description'
    });
    // Make sure the construct did not create any other topics or keys created
    expect(stack).toCountResources('AWS::KMS::Key', 1);
    expect(stack).toCountResources('AWS::SNS::Topic', 1);
});
test('construct uses specific event types and filters', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        s3EventTypes: [s3.EventType.OBJECT_REMOVED],
        s3EventFilters: [
            {
                prefix: 'the/place',
                suffix: '*.mp3'
            }
        ]
    });
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            TopicConfigurations: [
                {
                    Events: [
                        's3:ObjectRemoved:*'
                    ],
                    Filter: {
                        Key: {
                            FilterRules: [
                                {
                                    Name: 'suffix',
                                    Value: '*.mp3'
                                },
                                {
                                    Name: 'prefix',
                                    Value: 'the/place'
                                }
                            ]
                        }
                    },
                    TopicArn: {
                        Ref: "tests3snsSnsTopicF02F6BD0"
                    }
                }
            ]
        }
    });
});
test('Topic is encrypted with imported CMK when set on encryptionKey prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = defaults.buildEncryptionKey(stack, {
        description: 'existing-key-description'
    });
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        encryptionKey: cmk
    });
    core_1.expectKmsKeyAttachedToCorrectResource(stack, 'AWS::SNS::Topic', 'existing-key-description');
});
test('Topic is encrypted with provided encryptionKeyProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        encryptionKeyProps: {
            description: 'existing-key-description'
        }
    });
    core_1.expectKmsKeyAttachedToCorrectResource(stack, 'AWS::SNS::Topic', 'existing-key-description');
});
test('Topic is encrypted with imported CMK when set on topicProps.masterKey prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = defaults.buildEncryptionKey(stack, {
        description: 'existing-key-description'
    });
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        topicProps: {
            masterKey: cmk
        }
    });
    core_1.expectKmsKeyAttachedToCorrectResource(stack, 'AWS::SNS::Topic', 'existing-key-description');
});
test('Topic is encrypted by default with Customer-managed KMS key when no other encryption properties are set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {});
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "tests3snsEncryptionKey6C553584",
                "Arn"
            ]
        }
    });
    expect(stack).toCountResources('AWS::KMS::Key', 1);
    expect(stack).toCountResources('AWS::SNS::Topic', 1);
});
test('Topic is encrypted with SQS-managed KMS Key when enable encryption flag is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        enableEncryptionWithCustomerManagedKey: false
    });
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition"
                    },
                    ":kms:",
                    {
                        Ref: "AWS::Region"
                    },
                    ":",
                    {
                        Ref: "AWS::AccountId"
                    },
                    ":alias/aws/sns"
                ]
            ]
        }
    });
    expect(stack).toCountResources('AWS::KMS::Key', 0);
    expect(stack).toCountResources('AWS::SNS::Topic', 1);
});
test('Construct does not override unencrypted topic when passed in existingTopicObj prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingTopicObj = new sns.Topic(stack, 'Topic', {
        topicName: 'existing-topic-name'
    });
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        existingTopicObj
    });
    expect(stack).toCountResources('AWS::KMS::Key', 0);
    expect(stack).toCountResources('AWS::SNS::Topic', 1);
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        TopicName: 'existing-topic-name'
    });
});
//# sourceMappingURL=data:application/json;base64,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