from typing import Any, Dict, List, Type, TypeVar

from pydantic import BaseModel, Field

from ..types import UNSET

T = TypeVar("T", bound="ModelDbt")


class ModelDbt(BaseModel):
    """Query that is based on a dbt model

    Attributes:
        compiled_sql (str): Compiled SQL in the dbt model
        database (str): Name of the database containing the generated table
        dbt_unique_id (str): Unique ID of the model assigned by dbt (usually some combination of the schema and table
            name)
        model_id (str): Model id that refer to a dbt model
        name (str): Name of the table generated by the dbt model
        raw_sql (str): Raw SQL in the dbt model
        schema (str): Name of the schema containing the generated table
    """

    compiled_sql: str = None
    database: str = None
    dbt_unique_id: str = None
    model_id: str = None
    name: str = None
    raw_sql: str = None
    schema_: str = Field(default=..., alias="schema")
    additional_properties: Dict[str, Any] = Field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        compiled_sql = self.compiled_sql
        database = self.database
        dbt_unique_id = self.dbt_unique_id
        model_id = self.model_id
        name = self.name
        raw_sql = self.raw_sql
        schema = self.schema

        field_dict: Dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "compiledSql": compiled_sql,
                "database": database,
                "dbtUniqueId": dbt_unique_id,
                "modelId": model_id,
                "name": name,
                "rawSql": raw_sql,
                "schema": schema,
            }
        )

        return field_dict

    @classmethod
    def from_dict(cls: Type[T], src_dict: Dict[str, Any]) -> T:
        if src_dict is None or src_dict is UNSET:
            return {}
        d = {k: v if v is not None else UNSET for k, v in src_dict.items()}
        compiled_sql = d.pop("compiledSql")

        database = d.pop("database")

        dbt_unique_id = d.pop("dbtUniqueId")

        model_id = d.pop("modelId")

        name = d.pop("name")

        raw_sql = d.pop("rawSql")

        schema = d.pop("schema")

        model_dbt = cls(
            compiled_sql=compiled_sql,
            database=database,
            dbt_unique_id=dbt_unique_id,
            model_id=model_id,
            name=name,
            raw_sql=raw_sql,
            schema=schema,
        )

        model_dbt.additional_properties = d
        return model_dbt

    @property
    def additional_keys(self) -> List[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties.get(key)

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
