from vdb.lib import utils as utils


def test_normalise():
    res = utils.normalise_num(100, 3)
    assert res == 100
    res = utils.normalise_num(100, 4)
    assert res == 1000
    res = utils.normalise_num(100, 2)
    assert res == 100
    res = utils.normalise_num('5.4b1', 3)
    assert res == 541
    assert utils.normalise_version_str("2.0.0", 3) == "2.0.0"
    assert utils.normalise_version_str("2.0.10", 4) == "2.0.10.0"


def test_version_compare():
    res = utils.version_compare("2.0.0", "2.0.0", "3.0.0", "2.0.0", "3.0.0")
    assert not res
    res = utils.version_compare("3.0.0", "2.0.0", "3.0.0", "2.0.0", "3.0.0")
    assert not res
    res = utils.version_compare("3.0.0", "2.0.0", "2.7.9.4")
    assert not res
    res = utils.version_compare("2.0.0", "2.0.0", "2.7.9.4")
    assert res
    res = utils.version_compare("2.7.0", "2.0.0", "2.7.9.4")
    assert res
    res = utils.version_compare("2.7.9.4", "2.0.0", "2.7.9.4")
    assert res
    res = utils.version_compare("2.7.9.5", "2.0.0", "2.7.9.4")
    assert not res
    res = utils.version_compare("1.0.0", "2.0.0", "2.7.9.4")
    assert not res
    res = utils.version_compare("4.0.0", "2.0.0", "2.7.9.4")
    assert not res
    res = utils.version_compare("3.7.9.4", "2.0.0", "*")
    assert res
    res = utils.version_compare("2.10.0", "2.1.0", "2.1.5")
    assert not res
    res = utils.version_compare("2.10.0", "2.0.0", "2.9.10")
    assert not res
    res = utils.version_compare("2.0.0", None, "2.7.9", "2.0.0")
    assert not res
    res = utils.version_compare("2.7.9", "2.0.0", None, None, "2.7.9")
    assert not res
    res = utils.version_compare("1.0", "1.0", "3.0")
    assert res
    res = utils.version_compare("1.0", "1.0.0", "3.0")
    assert res
    res = utils.version_compare("2.0", "1.0", "3.0")
    assert res
    res = utils.version_compare("2.0.0", "1.0", "3.0")
    assert res
    res = utils.version_compare("3.0", "1.0", "3.0")
    assert res
    res = utils.version_compare("3.0.0", "1.0", "3.0")
    assert res
    res = utils.version_compare("4.0", "1.0", "3.0")
    assert not res
    res = utils.version_compare("2.9.10.5", "*", "*", None, "2.8.11")
    assert not res
    res = utils.version_compare("2.9.10.5", "2.0.0", "*", None, "2.9.10.5")
    assert not res
    res = utils.version_compare("2.9.10.3", "2.0.0", "*", None, "2.9.10.5")
    assert res
    res = utils.version_compare("2.9.10.4", "2.0.0", "*", None, "2.9.10.5")
    assert res
    res = utils.version_compare("2.9.10.3", "2.9.0", "2.9.10.4", None, None)
    assert res
    res = utils.version_compare("2.9.10.4", "2.9.0", "2.9.10.4", None, None)
    assert res
    res = utils.version_compare("2.9.10.3", None, "2.9.10.4")
    assert res
    res = utils.version_compare("2.9.10.4", None, "2.9.10.5")
    assert res
    res = utils.version_compare("2.9.10.4", None, "2.9.10.1")
    assert not res
    res = utils.version_compare("2.9.10.4", None, "2.9.10")
    assert not res
    res = utils.version_compare("2.9.10.3", "0", "2.9.10.4")
    assert res
    res = utils.version_compare("2.9.10.4", "0", "2.9.10.5")
    assert res
    res = utils.version_compare("2.9.10.4", "0", "2.9.10.1")
    assert not res
    res = utils.version_compare("2.9.10.4", "0", "2.9.10")
    assert not res
    res = utils.version_compare("5.2.0.RELEASE", "5.2.0", "5.2.4")
    assert res
    res = utils.version_compare("5.2.1.FINAL", "5.2.0", "5.2.4")
    assert res
    res = utils.version_compare("5.2.5.FINAL", "5.2.0", "5.2.4")
    assert not res
    res = utils.version_compare("5.2.0.RELEASE", "5.2.0-alpha1", "5.2.4")
    assert res
    res = utils.version_compare("5.2.1.FINAL", "5.2.0-beta.1", "5.2.4")
    assert res
    res = utils.version_compare("5.2.5.FINAL", "5.2.0", "5.2.4")
    assert not res
    res = utils.version_compare("2.0.27.Final", None, "2.1.0")
    assert res
    res = utils.version_compare("2.0.27.Final", None, "2.0.29")
    assert res
    res = utils.version_compare("2.0.27.Final", None, None, None, "2.1.0")
    assert res
    res = utils.version_compare("2.0.27.Final", None, None, None, "2.0.29")
    assert res
    res = utils.version_compare("2.0.27.Final", "*", "*", None, "2.1.1")
    assert res
    res = utils.version_compare("2.0.27.Final", "*", "*", None, "2.0.29")
    assert res
    res = utils.version_compare("2.0.27.RELEASE", "*", "2.0.27", None, None)
    assert res
    res = utils.version_compare("0.6.18", "0", "0.6.19-r0", None, None)
    assert res
    res = utils.version_compare("2.15.4-r0", "0", "2.14.1-r0", None, None)
    assert not res
    res = utils.version_compare("1.8.19", "0", "1.8.20_p2-r0", None, None)
    assert res
    res = utils.version_compare("5.8.9", "0", "6.0_p20170701-r0", None, None)
    assert res
    res = utils.version_compare("1.5", "0", "1.6_rc2-r5", None, None)
    assert res
    res = utils.version_compare("0.99.1", "0", "0.99.4-r0", None, None)
    assert res
    res = utils.version_compare("0.7.0", "0", "0.7.1_alpha-r0", None, None)
    assert res
    res = utils.version_compare("0.7.0", "0", "*", None, None)
    assert res
    res = utils.version_compare("1.0.3", "0", "*", None, None)
    assert res
    res = utils.version_compare("1.0.3", "1.4.1", "*", None, None)
    assert not res
    res = utils.version_compare("12.1.0.2.0", "*", "12.1.0.2", None, None)
    assert res
    res = utils.version_compare("7.24.1", "0.0.0", None, None, "7.23.2")
    assert not res
    res = utils.version_compare("7.24.1", "8.0.0-alpha.0", None, None, "8.0.0-alpha.4")
    assert not res


def test_version_compare_go():
    res = utils.version_compare("v1.1.1", "v1.1.0", "v1.1.2")
    assert res
    res = utils.version_compare("v1.1.3", "v1.1.0", "v1.1.2")
    assert not res
    res = utils.version_compare(
        "v0.0.0-20190308221718-c2843e01d9a2", "2019-03-25", "2019-03-25"
    )
    assert not res
    res = utils.version_compare("v0.0.0-20190308221718-c2843e01d9a2", "*", "2017-03-17")
    assert not res
    res = utils.version_compare("v0.0.0-20190308221718-c2843e01d9a2", "*", "2019-03-17")
    assert res
    res = utils.version_compare(
        "v0.0.0-20180904163835-0709b304e793", "2019-03-25", "2019-03-25"
    )
    assert not res
    res = utils.version_compare(
        "v0.0.0-20180904163835-0709b304e793", "2018-09-03", "2018-09-05"
    )
    assert res
    res = utils.version_compare(
        "v0.0.0-20180904163835-0709b304e793", "2018-09-04", "2018-09-05"
    )
    assert res
    res = utils.version_compare(
        "v0.0.0-20180904163835-0709b304e793", "2018-09-05", "2018-09-05"
    )
    assert not res
    res = utils.version_compare("v0.0.0-20180904163835-0709b304e793", "*", "2017-03-17")
    assert not res
    res = utils.version_compare("v0.0.0-20180826012351-8a410e7b638d", "*", "2018-09-25")
    assert res
    res = utils.version_compare("v0.0.0-20180826012351-8a410e7b638d", "*", "2018-07-12")
    assert not res
    res = utils.version_compare("v0.0.0-20180826012351-8a410e7b638d", "*", "2018-09-27")
    assert res


def test_version_parts_compare():
    res = utils.version_compare("42.2.8", "*", "9.1.22")
    assert not res
    res = utils.version_compare("42.2.8", "*", "9.3.15")
    assert not res
    res = utils.version_compare("42.2.8", "*", "9.0")
    assert not res
    res = utils.version_compare("1.2.0", "0.7.0", "1.2.0")
    assert res
    res = utils.version_compare("2.1.8", "2.0.0", "2.0.14")
    assert not res
    res = utils.version_compare("2.1.800.0", "2.0.0", "2.2.14")
    assert res
    res = utils.version_compare("2.1.8.0", "2.0.0.800", "2.2.14.10")
    assert res
    res = utils.version_compare("82.1.8.0", "52.0.0", "96.2")
    assert res


def test_version_build_compare():
    res = utils.version_compare("6.0.1", "3.01", "5.4b2", None, None)
    assert not res
    res = utils.version_compare("6.0.1", "5.4", "5.4b2", None, None)
    assert not res
    res = utils.version_compare("6.0.1", "5.4b1", "5.4b4", None, None)
    assert not res
    res = utils.version_compare("6.0.1", "2.11", "5.8b1", None, None)
    assert not res
    res = utils.version_compare("6.0.1", "2.11", "6.8b1", None, None)
    assert res
    res = utils.version_compare("6.0.1", "2.11", "6.8.0b1", None, None)
    assert res
    res = utils.version_compare("1.2.0", "1.2.0-alpha", "1.2.0-beta")
    assert not res
    res = utils.version_compare("1.2.0", "1.1.0-alpha", "1.2.0-beta")
    assert not res
    res = utils.version_compare("1.3.0", "1.2.0-alpha", "1.3.1-beta")
    assert res
    res = utils.version_compare("1.3.0", "1.2.0", "1.3.1-beta")
    assert res
    res = utils.version_compare("1.3.0", "1.2.0-beta", "1.3.2.0")
    assert res
    res = utils.version_compare("4.3.6.RELEASE", "5.0.0.RC1", "5.0.0.RC3")
    assert not res


def test_version_build_compare1():
    sver, srest = utils.convert_to_semver("2.1.5.1-r3")
    assert sver
    res = utils.version_compare("2.1.5.1-r3", "1.2.1-r0", "2.1.5.1-r1")
    assert not res
    res = utils.version_compare("2.1.5.1-r3", "2.1.5.1-r2", "2.1.5.1-r6")
    assert res
    res = utils.version_compare("2.1.5.1-rc3", "2.1.5.1-rc2", "2.1.5.1-rc6")
    assert res
    res = utils.version_compare("7.24.1", "0.0.0", "7.23.2")
    assert not res
    res = utils.version_compare("7.24.1", "8.0.0-alpha.0", "8.0.0-alpha.4")
    assert not res


def test_version_build_diff_compare():
    res = utils.version_compare("7.0.0", "*", "*", None, "2020-04-23t00-58-49z")
    assert not res
    res = utils.version_compare("7.0.0", "*", "*", None, "2018-05-16t23-35-33z")
    assert not res
    res = utils.version_compare("7.0.0", "2018-05-16t23-35-33z", "2020-04-23t00-58-49z")
    assert not res
    res = utils.version_compare(
        "7.0.0", None, "2020-04-23t00-58-49z", "2020-04-23t00-58-49z", None
    )
    assert not res
    res = utils.version_compare(
        "7.0.0", None, "2018-05-16t23-35-33z", "2018-05-16t23-35-33z", None
    )
    assert not res
    res = utils.version_compare("7.0.0", "*", None, "2020-04-23t00-58-49z", None)
    assert not res
    res = utils.version_compare("7.0.0", "*", None, "2018-05-16t23-35-33z", None)
    assert not res


def test_version_hash_compare():
    res = utils.version_compare(
        "3.1.2", "0", None, None, "acb672b6a179567632e032f547582f30fa2f4aa7"
    )
    assert not res
    res = utils.version_compare(
        "3.1.2",
        "acb672b6a179567632e032f547582f30fa2f4aa7",
        "acb672b6a179567632e032f547582f30fa2f4aa7",
        None,
        None,
    )
    assert not res
    res = utils.version_compare(
        "8b626d45127d6f5ada7d815b83cfdc09e8cb1394",
        "8b626d45127d6f5ada7d815b83cfdc09e8cb1394",
        "8b626d45127d6f5ada7d815b83cfdc09e8cb1394",
        None,
        None,
    )
    assert res


def test_os_build_compare():
    res = utils.version_compare("2.8.4-1+squeeze4", "0", "2.8.6-1+squeeze4")
    assert res
    res = utils.version_compare("2.8.4-7+squeeze4", "0", "2.8.6-1+squeeze4")
    assert res
    res = utils.version_compare("2.6.34-1squeeze8", "0", "2.6.32-48squeeze8")
    assert not res
    res = utils.version_compare("3.0.1-8+wheezy6", "0", "3.0.4-3+wheezy6")
    assert res


def test_os_build_compare1():
    res = utils.version_compare("1:51.2.1-1~deb7u1", "0", "1:52.2.1-1~deb7u1")
    assert res
    res = utils.version_compare("2:51.2.1-1~deb7u1", "0", "1:52.2.1-1~deb7u1")
    assert res
    res = utils.version_compare("6:51.2.1-1~deb7u1", "0", "2:52.2.1-1~deb7u1")
    assert res
    res = utils.version_compare(
        "1.2.0-1.2+wheezy4+deb7u1", "0", "1.2.1-2.2+wheezy4+deb7u1"
    )
    assert res
    res = utils.version_compare("8:7.2.947-7+deb7u4", "0", "2:7.3.547-7+deb7u4")
    assert res
    res = utils.version_compare("9.04~dfsg-6.3+deb7u7", "0", "9.05~dfsg-6.3+deb7u7")
    assert res
    res = utils.version_compare("1:1.7.5.4-1+wheezy5", "0", "1:1.7.10.4-1+wheezy5")
    assert res
    res = utils.version_compare("1:1.7.11.1-1+wheezy5", "0", "1:1.7.10.4-1+wheezy5")
    assert not res
    res = utils.version_compare(
        "7u180-2.6.14-2~deb8u1", "7u179-2.6.14-2~deb8u1", "7u181-2.6.14-2~deb8u1"
    )
    assert res
    res = utils.version_compare(
        "7u182-2.8.14-2~deb8u1", "7u179-2.6.14-2~deb8u1", "7u181-2.6.14-2~deb8u1"
    )
    assert res
    res = utils.version_compare(
        "1.0.8~git20140621.1.440916e+dfsg1-13~deb8u3",
        "0",
        "1.1.0~git20140921.1.440916e+dfsg1-13~deb8u3",
    )
    assert res
    res = utils.version_compare("2:3.25-1+debu8u4", "0", "2:3.26-1+debu8u4")
    assert res
    res = utils.version_compare(
        "2.4.9", "0", "2.5.0.26054~ReleaseCandidate3.ds2-1+squeeze1"
    )
    assert res
    res = utils.version_compare("2.4.15-7.woody.2.1", "0", "2.4.17-2.woody.2.2")
    assert res
    res = utils.version_compare(
        "0.12.0+git20120207.aaa852f-1+deb7u1",
        "0",
        "0.12.1+git20120407.aaa852f-1+deb7u1",
    )
    assert res
    res = utils.version_compare("2.36.1-8+deb11u1", 0, "*", None, "2.36.1-8+deb11u1")
    assert not res
    res = utils.version_compare("1.10-4+deb11u1", 0, "*", None, "1.10-4+deb11u1")
    assert not res
    res = utils.version_compare("1.15.0", 0, "*", None, "1.15.1-2ubuntu2.2")
    assert res
    res = utils.version_compare("1:1.1.1k-7.el8_6", 0, "*", None, "1.1.1k")
    assert not res
    res = utils.version_compare("1.1.1f-1ubuntu2.8", 0, "*", None, "1.1.1-1ubuntu2.1")
    assert not res
    res = utils.version_compare("0.23.22-1.amzn2.0.1", 0, "*", None, "0.23.22")
    assert not res


def test_ubuntu_openssl():
    res = utils.version_compare("1.1.1f-1ubuntu2.8", 0, "*", None, "1.1.1.e")
    assert not res
    res = utils.version_compare(
        "1.1.1f-1ubuntu2.8", 0, "*", None, "1.1.1-1ubuntu2.1~18.04.2"
    )
    assert not res
    res = utils.version_compare(
        "1.1.1f-1ubuntu2.8", 0, "*", None, "1.1.1-1ubuntu2.1~18.04.6"
    )
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1g-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1k-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1d-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1j-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1i-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1l-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1a-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1~~pre9-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1o-1")
    assert res
    res = utils.version_compare("1.10-0ubuntu4", 0, "*", None, "1.10-0ubuntu4.1")
    assert res
    res = utils.version_compare("2.2.19-3ubuntu2.1", 0, "*", None, "*")
    assert res
    res = utils.version_compare("2.2.19-3ubuntu2.1", 0, "*", None, None)
    assert res
    res = utils.version_compare("2.34-0.1ubuntu9.1", 0, "*", None, "2.34-0.1ubuntu9.3")
    assert res
    res = utils.version_compare("3.0-1", 0, "*", None, "3.0-1ubuntu0.1")
    assert res
    # Not working
    # res = utils.version_compare("1.1.0j-1~deb9u1", 0, "*", None, "1.1.0f-3+deb9u1")
    # assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1g-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", 0, "*", None, "1.1.1k-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", "*", "*", None, "1.1.1g-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb11u3", "*", "*", None, "1.1.1k-1")
    assert not res
    res = utils.version_compare("1.1.1n-0+deb10u3", "*", "*", None, "1.1.1d-1")
    assert not res
    res = utils.version_compare("2:8.39-12build1", 0, "*", None, "8.39-9ubuntu0.1")
    assert not res
    res = utils.version_compare(
        "7.7.0+dfsg-1ubuntu1", 0, "*", None, "7.7.0+dfsg-1ubuntu1.1"
    )
    assert res


def test_ubuntu_compare2():
    # res = utils.version_compare("1.45.5-2ubuntu1", 0, "*", None, "1.45.5-1")
    # assert not res
    res = utils.version_compare("3.0-1", 0, "*", None, "3.0-1ubuntu0.1")
    assert res
    res = utils.version_compare("10.34-7", 0, "*", None, "10.34-7ubuntu0.1")
    assert res
    res = utils.version_compare(
        "2:6.2.0+dfsg-4", 0, "*", None, "2:6.2.0+dfsg-4ubuntu0.1"
    )
    assert res
    # res = utils.version_compare(
    #     "3.8.10-0ubuntu1~20.04.6", 0, "*", None, "3.8.10-0ubuntu1~20.04.1"
    # )
    # assert not res
    # res = utils.version_compare(
    #     "20210119~20.04.2", 0, "*", None, "20211016ubuntu0.20.04.1"
    # )
    # assert not res
    # res = utils.version_compare("2.1.27+dfsg-2", 0, "*", None, "2.1.27+dfsg-2ubuntu0.1")
    # assert res


def test_debian_build_compare():
    res = utils.version_compare("1:2019.10.06-1", 0, "*", None, "20070829-6+deb7u1")
    assert not res
    res = utils.version_compare("1.0.2l-2+deb9u3", 0, "*", None, "1.0.2u-1~deb9u5")
    assert res
    res = utils.version_compare("232-25+deb9u4", 0, "*", None, "232-25+deb9u8")
    assert res
    res = utils.version_compare("1.1.0f-3+deb9u2", 0, "*", None, "1.1.0h-3")
    assert res
    res = utils.version_compare("4.9.110-3+deb9u4", 0, "*", None, "4.9.240-1")
    assert res
    res = utils.version_compare("232-25+deb9u4", 0, "*", None, "232-25+deb9u8")
    assert res
    res = utils.version_compare("3.5.8-5+deb9u3", 0, "*", None, "3.5.8-5+deb9u6")
    assert res
    res = utils.version_compare("1.43.4-2", 0, "*", None, "1.46.6~rc1-1")
    assert res
    res = utils.version_compare("1.0.2l-2+deb9u3", 0, "*", None, "1.1.0c-1")
    assert res
    res = utils.version_compare("4.9.110-3+deb9u4", 0, "*", None, "4.9.161-1")
    assert res
    res = utils.version_compare("4.9.110-3+deb9u4", 0, "*", None, "4.9.240-1")
    assert res
    res = utils.version_compare("4.9.110-3+deb9u4", 0, "*", None, "4.9.110-1")
    assert not res


def test_debian_build_compare2():
    res = utils.version_compare("4.9.110-3+deb9u4", 0, "*", None, "4.9.20-1")
    assert not res
    res = utils.version_compare("1.43.4-2", 0, "*", None, "1.46.6~rc1-1")
    assert res
    res = utils.version_compare("1.0.6-8.1", 0, "*", None, "1.0.6-9.1")
    assert res
    res = utils.version_compare("1.0.2l-2+deb9u3", 0, "*", None, "1.0.2l-2+deb9u2")
    assert not res
    res = utils.version_compare("7.52.1-5+deb9u7", 0, "*", None, "7.52.1-5+deb9u3")
    assert not res
    res = utils.version_compare("1.0.2l-2+deb9u3", "*", "*", None, "1.0.2l-2+deb9u2")
    assert not res
    res = utils.version_compare("7.52.1-5+deb9u7", "*", "*", None, "7.52.1-5+deb9u3")
    assert not res
    res = utils.version_compare("7.52.1-5+deb9u7", 0, "*", None, "7.52.1-5+deb9u2")
    assert not res
    res = utils.version_compare("1.7.6-2+deb9u3", 0, "*", None, "1.7.6-2+deb9u2")
    assert not res
    res = utils.version_compare("2.24-11+deb9u3", 0, "*", None, "2.24-11+deb9u2")
    assert not res
    res = utils.version_compare("4.9.110-3+deb9u4", 0, "*", None, "4.19.37-2")
    assert res
    res = utils.version_compare("4.9.110-3+deb9u4", 0, "*", None, "4.17.3-1")
    assert res
    res = utils.version_compare("4.9.110-3+deb9u4", 0, "*", None, "4.16.16-1")
    assert res
    res = utils.version_compare("1.15-1+deb9u1", 0, "*", None, "1.2.4-5woody4")
    assert not res
    res = utils.version_compare("1.15-1+deb9u1", 0, "*", None, "1.8.3+dfsg-4squeeze5")
    assert not res
    res = utils.version_compare(
        "1.15-1+deb9u1", 0, "*", None, "1.6.dfsg.4~beta1-5lenny7"
    )
    assert not res
    res = utils.version_compare("1.15-1+deb9u1", 0, "*", None, "1.10.1+dfsg-5+deb7u6")
    assert not res
    res = utils.version_compare("2018e-0+deb9u1", 0, "*", None, "2018c-0+deb7u1")
    assert not res


def test_debian_build_compare3():
    res = utils.version_compare("20161130+nmu1+deb9u1", 0, "*", None, "20200601~deb9u2")
    assert res
    res = utils.version_compare("232-25+deb9u4", 0, "*", None, "249~rc2-1")
    assert res
    res = utils.version_compare("232-25+deb9u4", 0, "*", None, "232-25+deb9u6")
    assert res
    res = utils.version_compare("232-25+deb9u4", 0, "*", None, "232-25+deb9u8")
    assert res
    res = utils.version_compare("232-25+deb9u4", "*", "*", None, "232-25+deb9u6")
    assert res
    res = utils.version_compare("232-25+deb9u4", "*", "*", None, "232-25+deb9u8")
    assert res
    res = utils.version_compare("2.50.3-2", 0, "*", None, "2.50.3-2+deb9u1")
    assert res
    res = utils.version_compare("2.35.2-2", 0, "*", None, "2.37-3")
    assert res
    res = utils.version_compare("2.35.2-2", "*", "*", None, "2.37-3")
    assert res
    res = utils.version_compare("2.36", "*", "*", None, "2.37-3")
    assert res
    res = utils.version_compare("247.3-7", "*", "*", None, "249~rc2-1")
    assert res
    res = utils.version_compare("247.3-7", 0, "*", None, "249~rc2-1")
    assert res
    res = utils.version_compare("247.3-7", "*", "*", None, "99.99.9")
    assert res
    res = utils.version_compare("4.3.0", "2.0.20126.16343", "4.3.3", None, None)
    assert res


def test_redhat_build_compare():
    data = """
1.18.2-21.el8 1.18.2-22.el8_7
1.11.948 1.12.261
2.12.1 2.12.6.1, 2.13.2.1
2.12.1 2.12.7.1, 2.13.4.1
2.12.1 2.12.7.1, 2.13.4
2.8.6 2.8.9
25.1-android 30.0
1.25.0 1.31.0
1.25.0 1.33.3
3.16.1 3.16.3, 3.19.6, 3.20.3, 3.21.7
2.6 2.7
1.10.9 1.9.16, 1.10.11
1.10.9 1.9.16, 1.10.11
6.5.0 6.6.0
1.20 1.21
1.20 1.21
1.20 1.21
1.20 1.21
1.9 1.10.0
2.3.0 2.5.1
1.5.21 1.6.0
1.14.3 1.15.3
2.13.3 2.13.9
1.28 1.31
1.28 1.31
1.28 1.31
1.28 1.31
1.28 1.32
1.30 1.31
1.30 1.31
1.30 1.31
1.30 1.31
1.30 1.32
6.7.1 11.8.5, 12.1.0
1.0.0 4.0.1, 3.0.1
1.7.1 2.4.0
1.25.11 1.26.5
v0.0.0-20210421170649-83a5a9bb288b 0.0.0-20211202192323-5770296d904e
v0.0.0-20210421170649-83a5a9bb288b 0.0.0-20220314234659-1baeb1ce4c0b
v0.0.0-20210326060303-6b1517762897 0.0.0-20210520170846-37e1c6afe023
v0.0.0-20210326060303-6b1517762897 0.0.0-20211209124913-491a49abca63
v0.0.0-20210326060303-6b1517762897 0.0.0-20220906165146-f3363e06e74c
v0.0.0-20210326060303-6b1517762897 0.0.0-20210428140749-89ef3d95e781
v0.0.0-20210502180810-71e4cd670f79 0.0.0-20220412211240-33da011f77ad
v0.0.0-20200622213623-75b288015ac9 0.0.0-20201216223049-8b5274cf687f
v0.0.0-20200622213623-75b288015ac9 0.0.0-20211202192323-5770296d904e
v0.0.0-20200622213623-75b288015ac9 0.0.0-20220314234659-1baeb1ce4c0b
v0.0.0-20210520170846-37e1c6afe023 0.0.0-20211209124913-491a49abca63
v0.0.0-20210520170846-37e1c6afe023 0.0.0-20220906165146-f3363e06e74c
v0.0.0-20210630005230-0f9fa26af87c 0.0.0-20220412211240-33da011f77ad
v0.3.6 0.3.7
v0.3.6 0.3.8
v0.2.0 v0.3.1
v1.11.0 1.11.1
v0.0.0-20210921065528-437939a70204 0.0.0-20220412211240-33da011f77ad
v0.0.0-20211019181941-9d821ace8654 0.0.0-20220412211240-33da011f77ad
v0.0.0-20210421170649-83a5a9bb288b 0.0.0-20211202192323-5770296d904e
v0.0.0-20210421170649-83a5a9bb288b 0.0.0-20220314234659-1baeb1ce4c0b
v0.0.0-20210405180319-a5a99cb37ef4 0.0.0-20210520170846-37e1c6afe023
v0.0.0-20210405180319-a5a99cb37ef4 0.0.0-20211209124913-491a49abca63
v0.0.0-20210405180319-a5a99cb37ef4 0.0.0-20220906165146-f3363e06e74c
v0.0.0-20210405180319-a5a99cb37ef4 0.0.0-20210428140749-89ef3d95e781
v0.0.0-20210616094352-59db8d763f22 0.0.0-20220412211240-33da011f77ad
v0.3.7 0.3.8
"""
    for line in data.split("\n"):
        tmp_a = line.split(" ")
        if len(tmp_a) != 2:
            continue
        res = utils.version_compare(tmp_a[0], 0, "*", None, tmp_a[1])
        assert res
        res = utils.version_compare(tmp_a[1], 0, "*", None, tmp_a[0])
        assert not res
    res = utils.version_compare("14.20.1", 0, "*", None, "12")
    assert not res
    res = utils.version_compare("6.0", 0, "*", None, "5.4")
    assert not res
    res = utils.version_compare("3.3.2", "1.8", None, None, "41.0.2")
    assert res
    res = utils.version_compare("39.0.0", "0.8.1", None, None, "39.0.1")
    assert res
    res = utils.version_compare("39.0.0", "0", None, None, "1.5.3")
    assert not res
    res = utils.version_compare("38.0.0", "37.0.0", "*", None, "38.0.3")
    assert res
    res = utils.version_compare("3.3.0", "0.5", "*", None, "41.0.0")
    assert res
    res = utils.version_compare("41.0.0", "40.0.0", "*", None, "41.0.2")
    assert res


def test_parse_uri():
    vendor, package, version, cve_type = utils.parse_cpe(
        "cpe:2.3:o:google:android:9.0:*:*:*:*:*:*:*"
    )
    assert vendor == "google"
    assert package == "android"
    assert version == "9.0"


def test_version_len():
    assert utils.version_len("1.0.0") == 3
    assert utils.version_len("2.1.800.5") == 6
    assert utils.version_len("1.2.0-beta1") == 3
    assert utils.version_len("1.3.0.beta1") == 3


def test_fix_test():
    assert (
        utils.fix_text(
            "Unauthenticated crypto and weak IV in Magento\\Framework\\Encryption"
        )
        == "Unauthenticated crypto and weak IV in Magento Framework Encryption"
    )


def test_convert_md_references():
    assert utils.convert_md_references(
        "- [Issue #60](https://github.com/tj/node-growl/issues/60)\n- [PR #61](https://github.com/tj/node-growl/pull/61)"
    ) == [
        {"name": "Issue #60", "url": "https://github.com/tj/node-growl/issues/60"},
        {"name": "PR #61", "url": "https://github.com/tj/node-growl/pull/61"},
    ]


def test_ubuntu_versions():
    assert utils.check_epoch("1:4.14-1.1+deb7u1build0.12.04.1")
    assert utils.convert_to_semver("0.3.14-11ubuntu6")[0]
    assert utils.convert_to_semver("1.4.9a-1ubuntu0.1")[0]
    assert utils.convert_to_semver("20140405-0ubuntu1")[0]
    assert utils.convert_to_semver("1.10+dfsg~beta1-2ubuntu0.7")[0]
    assert utils.convert_to_semver("1.11.3+dfsg-3ubuntu2")[0]
    assert utils.convert_to_semver("1.0.4cvs20051004-2")[0]
    assert utils.convert_to_semver("7.0.0-0ubuntu45")[0]
    assert utils.convert_to_semver("1.5.0+1.5.1cvs20051015-1ubuntu10")[0]
    assert utils.convert_to_semver("1.2.8rel-1ubuntu3")[0]
    assert utils.convert_to_semver("3.3.8really3.3.7-0ubuntu5.2")[0]
    assert utils.convert_to_semver("1.5.0+1.5.1cvs20051015-1ubuntu10")[0]
    assert utils.convert_to_semver("0.9.2+cvs.1.0.dev.2004.07.28-4ubuntu1")[0]
    assert utils.convert_to_semver("8.1.2-0.20040524cvs-2")[0]
    assert utils.convert_to_semver("6.2.4.5.dfsg1-0.14ubuntu0.1")[0]
    assert utils.convert_to_semver("1.5.dfsg+1.5.0.13~prepatch070731-0ubuntu1")[0]
    assert utils.convert_to_semver("01.03.00.99.svn.300-3")[0]
    assert utils.check_epoch("1:1.1.1k-7.el8_6")


def test_parse_purl():
    assert utils.parse_purl(
        "pkg:maven/org.jenkins-ci.plugins/JDK_Parameter_Plugin"
    ) == {
        "type": "maven",
        "namespace": "org.jenkins-ci.plugins",
        "name": "JDK_Parameter_Plugin",
        "version": None,
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.parse_purl("pkg:golang/github.com/sjqzhang/go-fastdfs") == {
        "type": "golang",
        "namespace": "",
        "name": "github.com/sjqzhang/go-fastdfs",
        "version": None,
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.parse_purl("pkg:golang/vitess.io/vitess") == {
        "type": "golang",
        "namespace": "",
        "name": "vitess.io/vitess",
        "version": None,
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.parse_purl("pkg:golang/stdlib") == {
        "type": "golang",
        "namespace": None,
        "name": "stdlib",
        "version": None,
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.parse_purl("pkg:golang/github.com/containerd/containerd") == {
        "type": "golang",
        "namespace": "",
        "name": "github.com/containerd/containerd",
        "version": None,
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.parse_purl("pkg:gem/remote:@https:%2F%2Frubygems.org%2F") == {}


def test_purl_vers_convert():
    assert (
        utils.to_purl_vers(
            "almalinux",
            [
                {
                    "version": "0",
                    "status": "affected",
                    "versionType": "semver",
                    "lessThan": "2.3.4",
                }
            ],
        )
        == "vers:rpm/<2.3.4"
    )
    assert (
        utils.to_purl_vers(
            "almalinux",
            [
                {
                    "version": "0",
                    "status": "affected",
                    "versionType": "semver",
                    "lessThan": "5.14.21-150500.13.5.1",
                }
            ],
        )
        == "vers:rpm/<5.14.21-150500.13.5.1"
    )
    assert (
        utils.to_purl_vers(
            "gem",
            [
                {
                    "version": "0.1.1",
                    "status": "affected",
                    "versionType": "gem",
                    "lessThanOrEqual": "4.3.5",
                }
            ],
        )
        == "vers:gem/>=0.1.1|<=4.3.5"
    )
    test_tuples = [
        (
            "npm",
            [
                {
                    "version": "0.0.0",
                    "status": "affected",
                    "versionType": "npm",
                    "lessThanOrEqual": "*",
                }
            ],
            "vers:npm/*",
        ),
        (
            "npm",
            [
                {
                    "version": "0.0.0",
                    "status": "affected",
                    "versionType": "npm",
                    "lessThan": "1.1.3",
                }
            ],
            "vers:npm/>=0.0.0|<1.1.3",
        ),
        (
            "npm",
            [
                {
                    "version": "0.0.0",
                    "status": "affected",
                    "versionType": "npm",
                    "lessThan": "3.5.0",
                }
            ],
            "vers:npm/>=0.0.0|<3.5.0",
        ),
        (
            "swift",
            [
                {
                    "version": "4.0.0-rc.2.5",
                    "status": "affected",
                    "versionType": "semver",
                    "lessThan": "4.29.4",
                }
            ],
            "vers:swift/>=4.0.0-rc.2.5|<4.29.4",
        ),
        (
            "swift",
            [
                {
                    "version": "0.0.0",
                    "status": "affected",
                    "versionType": "semver",
                    "lessThan": "4.60.3",
                }
            ],
            "vers:swift/>=0.0.0|<4.60.3",
        ),
        (
            "swift",
            [
                {
                    "version": "0.0.0",
                    "status": "affected",
                    "versionType": "semver",
                    "lessThan": "4.60.3",
                }
            ],
            "vers:swift/>=0.0.0|<4.60.3",
        ),
        (
            "maven",
            [
                {
                    "version": "8.5.0",
                    "status": "affected",
                    "versionType": "maven",
                    "lessThan": "8.5.94",
                }
            ],
            "vers:maven/>=8.5.0|<8.5.94",
        ),
        (
            "maven",
            [
                {
                    "version": "11.0.0-M1",
                    "status": "affected",
                    "versionType": "maven",
                    "lessThan": "11.0.0-M12",
                }
            ],
            "vers:maven/>=11.0.0-M1|<11.0.0-M12",
        ),
        (
            "debian",
            [{"version": "88.88.8", "status": "affected", "versionType": "maven"}],
            "vers:deb/88.88.8",
        ),
        (
            "debian",
            [
                {
                    "version": "53.5",
                    "status": "affected",
                    "versionType": "maven",
                    "lessThanOrEqual": "53.5",
                }
            ],
            "vers:deb/53.5",
        ),
    ]
    for tt in test_tuples:
        assert utils.to_purl_vers(tt[0], tt[1]) == tt[2]


def test_url_to_purl():
    assert utils.url_to_purl("https://github.com/owasp-dep-scan/dep-scan") == {
        "name": "dep-scan",
        "namespace": "owasp-dep-scan",
        "qualifiers": None,
        "subpath": None,
        "type": "github",
        "version": None,
    }
    assert utils.url_to_purl(
        "https://github.com/opennms/opennms/commit/8710463077c10034fcfa06556a98fb1a1a64fd0d"
    ) == {
        "name": "opennms",
        "namespace": "opennms",
        "qualifiers": None,
        "subpath": None,
        "type": "github",
        "version": "8710463077c10034fcfa06556a98fb1a1a64fd0d",
    }
    assert utils.url_to_purl(
        "https://git.eyrie.org/?p=kerberos/remctl.git%3ba=commit%3bh=86c7e44090c988112a37589d2c7a94029eb5e641"
    ) == {
        "type": "generic",
        "namespace": "git.eyrie.org/kerberos",
        "name": "remctl",
        "version": "86c7e44090c988112a37589d2c7a94029eb5e641",
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.url_to_purl(
        "https://git.savannah.gnu.org/cgit/wget.git/commit/?id=1fc9c95ec144499e69dc8ec76dbe07799d7d82cd"
    ) == {
        "type": "generic",
        "namespace": "git.savannah.gnu.org",
        "name": "wget",
        "version": "1fc9c95ec144499e69dc8ec76dbe07799d7d82cd",
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.url_to_purl(
        "https://git.gnupg.org/cgi-bin/gitweb.cgi?p=libgcrypt.git%3ba=commit%3bh=9010d1576e278a4274ad3f4aa15776c28f6ba965"
    ) == {
        "type": "generic",
        "namespace": "git.gnupg.org",
        "name": "libgcrypt",
        "version": "9010d1576e278a4274ad3f4aa15776c28f6ba965",
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.url_to_purl(
        "https://git.openssl.org/gitweb/?p=openssl.git%3ba=commitdiff%3bh=3984ef0b72831da8b3ece4745cac4f8575b19098"
    ) == {
        "type": "generic",
        "namespace": "git.openssl.org",
        "name": "openssl",
        "version": "3984ef0b72831da8b3ece4745cac4f8575b19098",
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.url_to_purl(
        "https://github.com/rsyslog/librelp/blob/532aa362f0f7a8d037505b0a27a1df452f9bac9e/src/tcp.c#l1205"
    ) == {
        "type": "github",
        "namespace": "rsyslog",
        "name": "librelp",
        "version": "532aa362f0f7a8d037505b0a27a1df452f9bac9e",
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.url_to_purl(
        "https://github.com/lightsaml/lightsaml/releases/tag/1.3.5"
    ) == {
        "type": "github",
        "namespace": "lightsaml",
        "name": "lightsaml",
        "version": "1.3.5",
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.url_to_purl(
        "https://gitlab.labs.nic.cz/labs/bird/commit/e8bc64e308586b6502090da2775af84cd760ed0d"
    ) == {
        "type": "generic",
        "namespace": "gitlab.labs.nic.cz/labs",
        "name": "bird",
        "version": "e8bc64e308586b6502090da2775af84cd760ed0d",
        "qualifiers": None,
        "subpath": None,
    }
    assert utils.url_to_purl(
        "http://git.kernel.org/cgit/linux/kernel/git/torvalds/linux.git/commit/?id=0fa3ecd87848c9c93c2c828ef4c3a8ca36ce46c7"
    ) == {
        "type": "generic",
        "namespace": "git.kernel.org/linux",
        "name": "kernel",
        "version": "0fa3ecd87848c9c93c2c828ef4c3a8ca36ce46c7",
        "qualifiers": None,
        "subpath": None,
    }


def test_vers_compare():
    assert not utils.vers_compare('7.24.1', 'vers:npm/<7.23.2')
    assert utils.vers_compare("1.12", "vers:rpm/<=1.12.8-24.el8_8.1")
    assert not utils.vers_compare("1.13.0", "vers:rpm/<=1.12.8-24.el8_8.1")
    assert utils.vers_compare("5.5", "vers:deb/<5.6~rc2")
    assert not utils.vers_compare("5.7", "vers:deb/<5.6~rc2")
