# SPDX-License-Identifier: GPL-3.0+
#
# pylint: disable=too-many-lines,consider-using-dict-comprehension

"""
Tools for analyzing UEFI firmware using radare2
"""

from typing import Dict
from uefi_r2.uefi_types import UefiGuid

PROTOCOLS_GUIDS = [
    UefiGuid(
        "C1D7859D-5719-46C3-A298D071E30264D1", name="ACOUSTIC_SETUP_PROTOCOL_GUID"
    ),
    UefiGuid("19D96D3F-6A6A-47D2-B1957B2432DA3BE2", name="ADD_BOOT_OPTION_GUID"),
    UefiGuid("E14F04FA-8706-4353-92F29C2424746F9F", name="ADVANCED_FORM_SET_GUID"),
    UefiGuid("B2FA4764-3B6E-43D3-91DF87D15A3E5668", name="AHCI_BUS_INIT_PROTOCOL_GUID"),
    UefiGuid("B2FA5764-3B6E-43D3-91DF87D15A3E5668", name="AHCI_SMM_PROTOCOL_GUID"),
    UefiGuid(
        "CB5C54C0-230D-43DB-922C24D34F8C915C",
        name="AMICSM_PCIBUSNUM_XLAT_PROTOCOL_GUID",
    ),
    UefiGuid("C811FA38-42C8-4579-A9BB60E94EDDFB34", name="AMITSESETUP_GUID"),
    UefiGuid(
        "541D5A75-95EE-43C7-9E5D2394DC486249", name="AMITSE_ADMIN_PASSWORD_VALID_GUID"
    ),
    UefiGuid(
        "C48D651C-9D0E-4CE7-AD39EDD1AB836B30",
        name="AMITSE_AFTER_FIRST_BOOT_OPTION_GUID",
    ),
    UefiGuid(
        "1B6BC809-C986-4937-934F1EA58622FE50", name="AMITSE_BOOT_ORDER_CHANGE_GUID"
    ),
    UefiGuid(
        "58279C2D-FB19-466E-B42ECD437016DC25", name="AMITSE_DRIVER_HEALTH_CTRL_GUID"
    ),
    UefiGuid(
        "0885F288-418C-4BE1-A6AF8BAD61DA08FE", name="AMITSE_DRIVER_HEALTH_ENB_GUID"
    ),
    UefiGuid("7459A7D4-6533-4480-BBA779E25A4443C9", name="AMITSE_DRIVER_HEALTH_GUID"),
    UefiGuid(
        "3677770F-EFB2-43B2-B8AEB302E9604882", name="AMITSE_EVENT_BEFORE_BOOT_GUID"
    ),
    UefiGuid(
        "D69240C0-DD40-4F2D-98634848DA6E615F", name="AMITSE_INVALID_PASSWORD_GUID"
    ),
    UefiGuid("D84BEFF0-159A-4B60-9AB9AC5C474BD3B1", name="AMITSE_NVRAM_UPDATE_GUID"),
    UefiGuid(
        "073E7E01-2611-4E85-B896A3B6767CBA00", name="AMITSE_PASSWORD_PROMPT_ENTER_GUID"
    ),
    UefiGuid(
        "B9B038B0-E2B6-4AAB-94354165ECFED032", name="AMITSE_PASSWORD_PROMPT_EXIT_GUID"
    ),
    UefiGuid("71202EEE-5F53-40D9-AB3D9E0C26D96657", name="AMITSE_SETUP_ENTER_GUID"),
    UefiGuid(
        "AB1404CA-4801-4208-98BF30D521DAD4D3", name="AMITSE_USER_PASSWORD_VALID_GUID"
    ),
    UefiGuid("26DC4851-195F-4AE1-9A19FBF883BBB35E", name="AMI_APTIO_SIG_OWNER_GUID"),
    UefiGuid("1DB184AE-81F5-4E72-85442BAB0C2CAC5C", name="AMI_BBS_DEVICE_PATH_GUID"),
    UefiGuid(
        "1D26ADC3-B011-EE2C-217789BBAACC3392", name="AMI_BEFORE_CPU_RC_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E9008D70-2A4E-47EA-8EC472E25767E5EF", name="AMI_BIOSPPI_FLAGS_MANAGEMENT_GUID"
    ),
    UefiGuid(
        "0273146C-96C4-45A1-A7AF78E0524A0AE2", name="AMI_BOARD_INFO_PROTOCOL_GUID"
    ),
    UefiGuid("E6F4F8F7-4992-47B2-83028508745E4A23", name="AMI_BOARD_INFO_SECTION_GUID"),
    UefiGuid("9CF0F18E-7C7D-49DE-B5AABBBAD6B21007", name="AMI_CALLBACK_GUID"),
    UefiGuid("BF66FDF7-F64C-4B11-8AB7F843AA2A8BEA", name="AMI_CAPSULE_HOB_GUID"),
    UefiGuid("B167C2F5-E26A-4DFF-8E1C0807C7F02A88", name="AMI_CCID_IO_PROTOCOL_GUID"),
    UefiGuid("5FDEE00D-DA40-405A-B92ECF4A80EA8F76", name="AMI_CCID_PRESENCE_GUID"),
    UefiGuid("B098F766-B17A-4005-8309EB23F1448C15", name="AMI_CMOS_BAD_FLAG_HOB_GUID"),
    UefiGuid("D4CA32B1-B1F2-4FF3-B47566E7B8192A3B", name="AMI_CPUID_CKSUM_HOB_GUID"),
    UefiGuid(
        "AC9CF0A8-E551-4BE2-AD0AE1B564EEA273", name="AMI_CPU_INFO_2_PROTOCOL_GUID"
    ),
    UefiGuid("3EA824D1-81E3-4FF5-BD43BB9C65DF7C46", name="AMI_CSM_DRIVER_STARTED_GUID"),
    UefiGuid("2362EA9C-84E5-4DFF-83BCB5ACECB57CBB", name="AMI_CSM_THUNK_PROTOCOL_GUID"),
    UefiGuid(
        "4E7AF417-C200-400A-9D18865ADAD30896", name="AMI_DDDT_PRESENT_FLAG_HOB_GUID"
    ),
    UefiGuid(
        "AB21ACC3-BA33-EE2C-66BC125677111AB2", name="AMI_DEBUGGER_CPU_PROTOCOL_GUID"
    ),
    UefiGuid("E894B313-54CA-4BB2-8CDD851E8AC9027C", name="AMI_DEBUGPORT_HOB_GUID"),
    UefiGuid(
        "2D6447EF-3BC9-41A0-AC194D51D01B4CE6", name="AMI_DEVICE_NAME_DEVICE_PATH_GUID"
    ),
    UefiGuid(
        "5F87BA17-957D-433D-9E15C0E7C8798899",
        name="AMI_DIGITAL_SIGNATURE_PROTOCOL_GUID",
    ),
    UefiGuid("D4CA32B3-B1FE-4FF7-B07360EDBB162233", name="AMI_DIMM_SPD_DATA_HOB_GUID"),
    UefiGuid("A7E2CE72-DC32-4BC0-9E35FEB30AE5CC47", name="AMI_EARLY_BIST_PPI_GUID"),
    UefiGuid(
        "0ADFB62D-FF74-484C-8944F85C4BEA87A8", name="AMI_EFIKEYCODE_PROTOCOL_GUID"
    ),
    UefiGuid("3496A19A-2E99-41BA-833E0FDE2EBF2A55", name="AMI_FAST_BOOT_PROTOCOL_GUID"),
    UefiGuid(
        "5E794317-A07E-45DF-94BB1C997D6232CA", name="AMI_FW_RECOVERY_CAPSULE_GUID"
    ),
    UefiGuid("FC9A50C1-8B3D-40D0-99126E26D7896CBA", name="AMI_HECI_SMM_GUID"),
    UefiGuid(
        "982D8C6F-F6F6-4135-A309A4593EA56417",
        name="AMI_INTERNAL_FACTORY_TDC_TDP_HOB_GUID",
    ),
    UefiGuid("94567C6F-F7A9-4229-1330FE11CCAB3A11", name="AMI_INTERNAL_UCODE_HOB_GUID"),
    UefiGuid(
        "B27DAB38-9814-4E06-A5A265AE9A14258F", name="AMI_INT_SMM_COMM_PROTOCOL_GUID"
    ),
    UefiGuid(
        "FC8BE767-89F1-4D6E-80996F021EBC87CC", name="AMI_IRQ_DMA_MASK_VARIABLE_GUID"
    ),
    UefiGuid("BA7C46D1-9C5E-4FC8-943D1A491F23FE01", name="AMI_ISO9660_MEDIA_GUID"),
    UefiGuid("16287BA4-5B9D-4D98-919F7B7B78CB2BE0", name="AMI_LOAD_CSM_GUID"),
    UefiGuid("99E275E7-75A0-4B37-A2E6C5385E6C00CB", name="AMI_MASKED_DEVICE_PATH_GUID"),
    UefiGuid("B3DAE700-2A77-4EA4-AF793297B484BE61", name="AMI_MEASURE_PCIOPROM_GUID"),
    UefiGuid("5023B95C-DB26-429B-A648BD47664C8012", name="AMI_MEDIA_DEVICE_PATH_GUID"),
    UefiGuid(
        "85226559-0DEF-48D8-A8C9B746D6A4DF01", name="AMI_MEMORY_ERROR_REPORT_PPI_GUID"
    ),
    UefiGuid("A6351A87-2965-4718-88C70B5B5AC0B5E4", name="AMI_MRC_INFO_HOB_GUID"),
    UefiGuid("6737934B-A27E-4C05-AD5B6AB86273680B", name="AMI_NB_MRC_INFO_GUID"),
    UefiGuid("717FC150-ABD9-4614-80150B3323EAB95C", name="AMI_NVRAM_SPD_MAP_GUID"),
    UefiGuid(
        "542D6248-4198-4960-9F592384646D63B4", name="AMI_OPROM_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid(
        "5F171F5F-8385-4086-A69B1FCF06AE4A3D",
        name="AMI_OS_PPI_CONFIRMATION_OVERRIDE_GUID",
    ),
    UefiGuid("04627B9B-385E-4744-9021E66319F20394", name="AMI_PB_KEY_RSA2048_GUID"),
    UefiGuid(
        "F42A009D-977F-4F08-9440BCA5A3BED9AF", name="AMI_PCI_BUS_EXT_PROTOCOL_GUID"
    ),
    UefiGuid("F4491BA4-7672-486F-B4D799899D22DA57", name="AMI_PEIM_HOB_GUID"),
    UefiGuid("EC9C36FD-1642-4B84-91FA919C2D066FB4", name="AMI_PEIM_LOAD_HOB_GUID"),
    UefiGuid("64C96700-6B4C-480C-A3E1B8BDE8F602B2", name="AMI_PEI_AFTER_MRC_GUID"),
    UefiGuid(
        "F824CCBB-D8E0-4522-8AA865F04B463DB5", name="AMI_PEI_CPUINIT_POLICY_PPI_GUID"
    ),
    UefiGuid(
        "9F58E424-B96B-45A5-ADDCD2FE394A99D9", name="AMI_PEI_END_MEMORY_DETECT_GUID"
    ),
    UefiGuid("633194BE-1697-11E1-B5F02CB24824019B", name="AMI_PEI_END_OF_MRC_GUID"),
    UefiGuid("E813E116-C099-4D21-9C34A552D5E9A5D0", name="AMI_PEI_MRC_DEFAULT_GUID"),
    UefiGuid(
        "9CE4D938-9C87-41D0-9E5534923FAF8B4F", name="AMI_PEI_NBINIT_POLICY_PPI_GUID"
    ),
    UefiGuid(
        "1F0F049E-3A68-4C97-865ABC5EED7920E7", name="AMI_PEI_NB_CPU_ONLY_RESET_PPI_GUID"
    ),
    UefiGuid("584CC99F-4BE8-43D1-A45A933DC39479FC", name="AMI_PEI_NB_CUSTOM_PPI_GUID"),
    UefiGuid(
        "97F91E78-EA12-4EA6-B7B37B0678C28673", name="AMI_PEI_PCI_TABLE_INIT_PPI_GUID"
    ),
    UefiGuid(
        "95E8152B-1B98-4F11-8A77DB26583EBC42", name="AMI_PEI_SBINIT_POLICY_PPI_GUID"
    ),
    UefiGuid("38965BB5-8097-40F5-B7428CC14A649B64", name="AMI_PEI_SB_CUSTOM_PPI_GUID"),
    UefiGuid(
        "61187967-9A77-419D-AAEA64DD56190815",
        name="AMI_PEI_SB_OEM_PLATFORM_POLICY_OVERRIDE_PPI_GUID",
    ),
    UefiGuid(
        "584CC99F-4BE8-43D1-A45A933DC39479FC", name="AMI_PEI_SET_NB_SUBID_PPI_GUID"
    ),
    UefiGuid("4D6C0496-8DE4-4AF2-9A2E9BE5B9156AC5", name="AMI_PERF_TUNE_DATA_HOB_GUID"),
    UefiGuid(
        "181E874D-C089-4C99-8EC26D6761348220", name="AMI_PLL_OVER_VOTAGE_FLAG_HOB_GUID"
    ),
    UefiGuid(
        "8A91B1E1-56C7-4ADC-ABEB1C2CA1729EFF", name="AMI_POST_MANAGER_PROTOCOL_GUID"
    ),
    UefiGuid("DAC3CB98-2295-412E-826DFDEEA320CF31", name="AMI_RECOVERY_IMAGE_HOB_GUID"),
    UefiGuid("39E8CDA1-6A35-4CDD-A902D1A795F70380", name="AMI_RESET_TYPE_HOB_GUID"),
    UefiGuid(
        "308DD02C-092B-4123-A2AF3EF4440A6B4A", name="AMI_RESET_TYPE_VARIABLE_GUID"
    ),
    UefiGuid("D7642443-87B7-4832-96070E1EA81C1D86", name="AMI_ROM_LAYOUT_HOB_GUID"),
    UefiGuid("589BC616-BB4F-47ED-92F793393C697E25", name="AMI_SB_SMI_PROTOCOL_GUID"),
    UefiGuid("D84BEFF0-159A-4B60-9AB9AC5C474BD3B1", name="AMI_SETUP_NVRAM_UPDATE_GUID"),
    UefiGuid(
        "3D6CDB0A-5B1F-43A3-A43B441267F9DAB5", name="AMI_SMBIOS_CPU_INFO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "7D6B8734-B754-443F-B5887743843AD3F1", name="AMI_SMBIOS_MEMORY_INFO_HOB_GUID"
    ),
    UefiGuid("017CB4B7-B80C-4040-B6C8EA982BBB25B7", name="AMI_SMBUS_HOB_GUID"),
    UefiGuid("72E40094-2EE1-497A-8F334C934A9E9C0C", name="AMI_SMBUS_SMM_PROTOCOL_GUID"),
    UefiGuid(
        "91ABC830-16FC-4D9E-A1895FC8BB411402",
        name="AMI_SMM_DIGITAL_SIGNATURE_PROTOCOL_GUID",
    ),
    UefiGuid("EB5198EB-E7F5-45A1-9CCBE53364BB4992", name="AMI_SMM_INFO_PROTOCOL_GUID"),
    UefiGuid(
        "4599D26F-1A11-49B8-B91F858745CFF824",
        name="AMI_STANDARD_DEFAULTS_VARIABLE_GUID",
    ),
    UefiGuid(
        "CD541D77-6699-4B36-A31E1AA4C5D5B946", name="AMI_STATUS_CODE_CPU_BIST_DATA_GUID"
    ),
    UefiGuid(
        "7D3DCEEE-CBCE-4EA7-87096E552F1EDBDE", name="AMI_TCG_CONFIRMATION_FLAGS_GUID"
    ),
    UefiGuid(
        "A8A2093B-FEFA-43C1-8E62CE526847265E", name="AMI_TCG_EFI_OS_VARIABLE_GUID"
    ),
    UefiGuid("2325F2FC-5683-4648-97C49A520DFBE325", name="AMI_TCG_PERM_FLAGS_GUID"),
    UefiGuid(
        "890C2CEF-43C8-4209-A78DAE14AA1798B4",
        name="AMI_TCG_PLATFORM_PPI_AFTER_MEM_GUID",
    ),
    UefiGuid(
        "C1E6791D-F35B-43EF-920ABE06BA7F86A1",
        name="AMI_TCG_PLATFORM_PPI_BEFORE_MEM_GUID",
    ),
    UefiGuid("05687F4A-3CA7-4D19-9BC5E180CEA3569F", name="AMI_TCG_PLATFORM_PPI_GUID"),
    UefiGuid(
        "320BDC39-3FA0-4BA9-BF2DB33F72BA9CA1", name="AMI_TCG_PLATFORM_PROTOCOL_GUID"
    ),
    UefiGuid("A8A2093B-FEFA-43C1-8E62CE526847265E", name="AMI_TCG_RESETVAR_HOB_GUID"),
    UefiGuid("3EF7500E-CF55-474F-8E7E009E0EACECD2", name="AMI_USB_SMM_PROTOCOL_GUID"),
    UefiGuid("4EFA0DB6-26DC-4BB1-A76F14BC630C7B3C", name="AMT_FORCE_PUSH_PET_HOB_GUID"),
    UefiGuid(
        "ACC8E1E4-9F9F-4E40-A57EF99E52F34CA5", name="AMT_FORCE_PUSH_PET_POLICY_GUID"
    ),
    UefiGuid(
        "D7AC94AF-A498-45EC-BFA2A56E9534618B", name="AMT_FORCE_PUSH_PET_VARIABLE_GUID"
    ),
    UefiGuid("6046E678-24EF-4005-BA39BDA11F6D555D", name="AMT_INT16_CSM_GUID"),
    UefiGuid(
        "40B09B5A-F0EF-4627-93D527F04B754D05", name="AMT_READY_TO_BOOT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "82627ACF-D92D-416D-8A6F783CACD91223", name="AOAC_EC_WAKEUP_CUSTOM_PPI_GUID"
    ),
    UefiGuid("FB045DB2-598E-485A-BA305D7B1B1BD54D", name="AOAC_FFS_TABLE_STORAGE_GUID"),
    UefiGuid("4A3CA68B-7723-48FB-803D578CC1FEC44D", name="APTIO_FW_CAPSULE_GUID"),
    UefiGuid("EA816D2C-CEE5-4F02-99B5D3905CBBD077", name="APTIO_HII_PROTOCOL_GUID"),
    UefiGuid(
        "532B6532-6499-428D-ACB1F6F779C94DF9", name="AUTHVAR_MAILBOX_VARIABLE_GUID"
    ),
    UefiGuid(
        "DBC9FD21-FAD8-45B0-9E7827158867CC93",
        name="BDS_ALL_DRIVERS_CONNECTED_PROTOCOL_GUID",
    ),
    UefiGuid(
        "3AA83745-9454-4F7A-A7C090DBD02FAB8E", name="BDS_CONNECT_DRIVERS_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0BA8263C-A8BD-4AAD-B4026A6AF2F7E77D", name="BLOCK_DEVICE_RECOVERY_CAPSULE_GUID"
    ),
    UefiGuid("EF152FB4-7B2F-427D-BDB47E0A05826E64", name="BOOT_FLOW_VARIABLE_GUID"),
    UefiGuid("8B33FFE0-D71C-4F82-9CEBC97058C13F8E", name="BOOT_FORM_SET_GUID"),
    UefiGuid("B4909CF3-7B93-4751-9BD85BA8220B9BB2", name="BOOT_MANAGER_GUID"),
    UefiGuid("052E6EB0-F240-42C5-830945874545C6B4", name="BOOT_NOW_COUNT_GUID"),
    UefiGuid("ADFE34C8-9AE1-4F8F-BE13CF96A2CB2C5B", name="CHIPSET_FORM_SET_GUID"),
    UefiGuid("D5367802-B873-4C0F-B54431B7CCF5C555", name="CMOS_MANAGER_HOB_GUID"),
    UefiGuid(
        "2DF1E051-906D-4EFF-869D24E65378FB9E",
        name="CONSOLE_IN_DEVICES_STARTED_PROTOCOL_GUID",
    ),
    UefiGuid(
        "EF9A3971-C1A0-4A93-BD405AA165F2DC3A",
        name="CONSOLE_OUT_DEVICES_STARTED_PROTOCOL_GUID",
    ),
    UefiGuid("17088572-377F-44EF-8F4EB09FFF46A070", name="CPU_MICROCODE_FILE_GUID"),
    UefiGuid(
        "DF665292-79D7-40E2-BA51F7D494628185", name="CPU_WAKE_UP_BUFFER_VARIABLE_GUID"
    ),
    UefiGuid(
        "3A4E4376-4871-4B0E-A02FED36F2AECD00", name="CSM_VIDEO_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid("6865C455-8626-40D8-90F4A69460A4AB5A", name="DCA_HOB_GUID"),
    UefiGuid("97CA1A5B-B760-4D1F-A54BD19092032C90", name="DEBUGGER_TERMINAL_VAR_GUID"),
    UefiGuid("16271FCA-55D9-4A33-93FC5A3EB128DEB6", name="DEFINE_GUID_CRBDXEBin_GUID"),
    UefiGuid("221F1D4F-034C-4BEA-B2BBB7A9672B06D7", name="DEFINE_GUID_CRBSMIBin_GUID"),
    UefiGuid(
        "333BB2A3-4F20-4C8B-AC380672D74315F8",
        name="DEFINE_GUID_AcpiPlatformPeiBin_GUID",
    ),
    UefiGuid("6E59DF06-62D3-40B0-82B5175CF84A94E4", name="DEFINE_GUID_OEMPEIBin_GUID"),
    UefiGuid("B7D19491-E55A-470D-850885A5DFA41974", name="DEFINE_GUID_SBDXEBin_GUID"),
    UefiGuid("BFE205C9-5B17-4F8F-937589614AF8E199", name="DEFINE_GUID_OEMDXEBin_GUID"),
    UefiGuid("C1FBD624-27EA-40D1-AA4894C3DC5C7E0D", name="DEFINE_GUID_SBPEIBin_GUID"),
    UefiGuid("0D1ED2F7-E92B-4562-92DD5C82EC917EAE", name="DEFINE_GUID_CRBPEIBin_GUID"),
    UefiGuid(
        "DFD8D5CC-5AED-4820-A2B65C55E4E640EF",
        name="DEFINE_GUID_AcpiPlatformSmiBin_GUID",
    ),
    UefiGuid("79AA6086-035A-4AD9-A89AA6D5AA27F0E2", name="DEFINE_GUID_NBPEIBin_GUID"),
    UefiGuid("E4ECD0B2-E277-4F2B-BECBE4D75C9A812E", name="DEFINE_GUID_NBDXEBin_GUID"),
    UefiGuid("E23F86E1-056E-4888-B685CFCD67C179D4", name="DEFINE_GUID_SBRunBin_GUID"),
    UefiGuid("F6C73719-F34C-479C-B32F277FCBBCFE4F", name="DEL_BOOT_OPTION_GUID"),
    UefiGuid("CE673A28-800D-4B4A-83162661F9B3D9C6", name="DIMM_TS_INFO_GUID"),
    UefiGuid("5C6FA2C9-9768-45F6-8E645AECCADAB481", name="DXE_CPU_CACHE_PROTOCOL_GUID"),
    UefiGuid("E223CF65-F6CE-4122-B3AF4BD18AFF40A1", name="DXE_CPU_INFO_PROTOCOL_GUID"),
    UefiGuid(
        "BD26CDC9-A092-462A-877A5AB6ADCE4812",
        name="DXE_CPU_PLATFORM_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "00E98021-F4FE-46CC-AB2D894C373AFA01",
        name="DXE_ENHANCED_SPEEDSTEP_PROTOCOL_GUID",
    ),
    UefiGuid(
        "E0A31DBE-F20E-4E2A-9F2B9F02A27D3D69",
        name="DXE_PCH_PLATFORM_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "B2AB115E-C8B6-4036-BF31E74BD8926CCE", name="DXE_PLATFORM_AMT_POLICY_GUID"
    ),
    UefiGuid("F8BFF014-18FB-4EF9-B10CAE22738DBEED", name="DXE_PLATFORM_ME_POLICY_GUID"),
    UefiGuid("CD2333D7-6A0A-4C76-8350240ADA36A2C7", name="DXE_PLATFORM_SA_POLICY_GUID"),
    UefiGuid("6199DC36-E114-4E0D-809999A0BD80A971", name="DXE_PLATFORM_SG_POLICY_GUID"),
    UefiGuid(
        "20DAF0FC-5548-44DC-A42A60EAF0A22E47", name="DXE_PLATFORM_TDT_POLICY_GUID"
    ),
    UefiGuid(
        "43F66FFD-B288-4139-BAD5B89803E730A2", name="DXE_TXT_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid("B63BF800-F267-4F55-9217E97FB3B69846", name="DYNAMIC_PAGE_COUNT_GUID"),
    UefiGuid(
        "8D59D32B-C655-4AE9-9B15F25904992A43", name="EFI_ABSOLUTE_POINTER_PROTOCOL_GUID"
    ),
    UefiGuid("8868E871-E4F1-11D3-BC220080C73C8881", name="EFI_ACPI_20_TABLE_GUID"),
    UefiGuid("125F2DE1-FB85-440C-A54C4D99358A8D38", name="EFI_ACPI_S3_SAVE_GUID"),
    UefiGuid("EB97088E-CFDF-49C6-BE4BD906A5B20E86", name="EFI_ACPI_SDT_PROTOCOL_GUID"),
    UefiGuid("DBFF9D55-89B7-46DA-BDDF677D3DC0241D", name="EFI_ACPI_SUPPORT_GUID"),
    UefiGuid("EB9D2D30-2D88-11D3-9A160090273FC14D", name="EFI_ACPI_TABLE_GUID"),
    UefiGuid(
        "FFE06BDD-6107-46A6-7BB25A9C7EC5275C", name="EFI_ACPI_TABLE_PROTOCOL_GUID"
    ),
    UefiGuid("7E374E25-8E01-4FEE-87F2390C23C606CD", name="EFI_ACPI_TABLE_STORAGE_GUID"),
    UefiGuid("AF9FFD67-EC10-488A-9DFC6CBF5EE22C2E", name="EFI_ACPI_VARIABLE_GUID"),
    UefiGuid(
        "EBBE2D1B-1647-4BDA-AB9A7863E396D41A", name="EFI_ACTIVE_BIOS_PROTOCOL_GUID"
    ),
    UefiGuid(
        "8555FD40-140B-4F3C-905E3BF378A099FA",
        name="EFI_ACTIVE_MANAGEMENT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "67820532-7613-4DD3-9ED73D9BE3A7DA63", name="EFI_AHCI_INT13_INIT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "CC93A70B-EC27-49C5-8B3413931EFED6E2",
        name="EFI_ALERT_STANDARD_FORMAT_PROTOCOL_GUID",
    ),
    UefiGuid("F496922D-172F-4BBC-A1EB0EEB949C3486", name="EFI_ALTERNATE_FV_BLOCK_GUID"),
    UefiGuid(
        "120D28AA-6630-46F0-8157C0ADC2383BF5", name="EFI_AMI_LEGACYBOOT_PROTOCOL_GUID"
    ),
    UefiGuid("FC510EE7-FFDC-11D4-BD410080C73C8881", name="EFI_APRIORI_GUID"),
    UefiGuid("F4B427BB-BA21-4F16-BC4E43E416AB619C", name="EFI_ARP_PROTOCOL_GUID"),
    UefiGuid(
        "F44C00EE-1F2C-4A00-AA091C9F3E0800A3",
        name="EFI_ARP_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7671D9D0-53DB-4173-AA692327F21F0BC7",
        name="EFI_AUTHENTICATION_INFO_PROTOCOL_GUID",
    ),
    UefiGuid(
        "995188B1-9F96-11D4-87AE0006292E8A3B", name="EFI_AUTHORIZATION_PROTOCOL_GUID"
    ),
    UefiGuid("B979746A-8C1F-4A2B-97E478E93A71A70A", name="EFI_BDAT_ACCESS_GUID"),
    UefiGuid("665E3FF6-46CC-11D4-9A380090273FC14D", name="EFI_BDS_ARCH_PROTOCOL_GUID"),
    UefiGuid("0B64AAB0-5429-11D4-981600A0C91FADCF", name="EFI_BIS_PROTOCOL_GUID"),
    UefiGuid("964E5B21-6459-11D2-8E3900A0C969723B", name="EFI_BLOCK_IO_PROTOCOL_GUID"),
    UefiGuid(
        "470E1529-B79E-4E32-A0FE6A156D29F9B2", name="EFI_BOOT_SCRIPT_SAVE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3BC1B285-8A15-4A82-AABF4D7D13FB3265",
        name="EFI_BUS_SPECIFIC_DRIVER_OVERRIDE_PROTOCOL_GUID",
    ),
    UefiGuid("BB628AE0-CD4F-49FE-8D6063186FD1E05B", name="EFI_CACHE_INSTALL_PPI_GUID"),
    UefiGuid("7F0013A7-DC79-4B22-809911F75FDC829D", name="EFI_CACHE_SUBCLASS_GUID"),
    UefiGuid("FAC2EFAD-8511-4E34-9CAE16A257BA9488", name="EFI_CAPSULE_AMI_GUID"),
    UefiGuid(
        "5053697E-2CBC-4819-90D90580DEEE5754", name="EFI_CAPSULE_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid("3B6686BD-0D76-4030-B70EB5519E2FC5A0", name="EFI_CAPSULE_GUID"),
    UefiGuid("8B34EAC7-2690-460B-8BA5D5CF32831735", name="EFI_CAPSULE_INFO_GUID"),
    UefiGuid("711C703F-C285-4B10-A3B036ECBD3C8BE2", name="EFI_CAPSULE_VENDOR_GUID"),
    UefiGuid("3C5766E8-269C-4E34-AA14ED776E85B3B6", name="EFI_CERT_RSA2048_GUID"),
    UefiGuid("67F8444F-8743-48F1-A3281EAAB8736080", name="EFI_CERT_RSA2048_SHA1_GUID"),
    UefiGuid(
        "E2B36190-879B-4A3D-AD8DF2E7BBA32784", name="EFI_CERT_RSA2048_SHA256_GUID"
    ),
    UefiGuid("826CA512-CF10-4AC9-B187BE01496631BD", name="EFI_CERT_SHA1_GUID"),
    UefiGuid("C1C41626-504C-4092-ACA941F936934328", name="EFI_CERT_SHA256_GUID"),
    UefiGuid("4AAFD29D-68DF-49EE-8AA9347D375665A7", name="EFI_CERT_TYPE_PKCS7_GUID"),
    UefiGuid(
        "A7717414-C616-4977-9420844712A735BF", name="EFI_CERT_TYPE_RSA2048_SHA256_GUID"
    ),
    UefiGuid("CBBEE336-2682-4CD6-818B0A0D967E5A67", name="EFI_CLP_PROTOCOL_GUID"),
    UefiGuid(
        "5A6A93F4-2907-4A34-BD116CA8A0959E09", name="EFI_CMOS_DATA_HOB_INSTALLED_GUID"
    ),
    UefiGuid(
        "64C475EF-344B-492C-93ADAB9EB4395004",
        name="EFI_COMPATIBLE_MEMORY_TESTED_PROTOCOL_GUID",
    ),
    UefiGuid(
        "6A7A5CFF-E8D9-4F70-BADA75AB3025CE14", name="EFI_COMPONENT_NAME2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "107A772C-D5E1-11D4-9A460090273FC14D", name="EFI_COMPONENT_NAME_PROTOCOL_GUID"
    ),
    UefiGuid("98B8D59B-E8BA-48EE-98DDC295392F1EDB", name="EFI_CONFIG_FILE_NAME_GUID"),
    UefiGuid("D3B36F2B-D551-11D4-9A460090273FC14D", name="EFI_CONSOLE_IN_DEVICE_GUID"),
    UefiGuid("368CDA0D-CF31-4B9B-8CF6E7D1BFFF157E", name="EFI_CONSOLE_LOCK_GUID"),
    UefiGuid("D3B36F2C-D551-11D4-9A460090273FC14D", name="EFI_CONSOLE_OUT_DEVICE_GUID"),
    UefiGuid("26BACCB1-6F42-11D4-BCE70080C73C8881", name="EFI_CPU_ARCH_PROTOCOL_GUID"),
    UefiGuid("AD61F191-AE5F-4C0E-B9FAE869D288C64F", name="EFI_CPU_IO2_PROTOCOL_GUID"),
    UefiGuid("B0732526-38C8-4B40-887761C7B06AAC45", name="EFI_CPU_IO_PROTOCOL_GUID"),
    UefiGuid("F064C91F-188C-4F56-B7FD30A9B86A29F3", name="EFI_CPU_TYPE_FRU_GUID"),
    UefiGuid(
        "FC1BCDB0-7D31-49AA-936AA4600D9DD083",
        name="EFI_CRC32_GUIDED_SECTION_EXTRACTION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "9A44198E-A4A2-44E6-8A1F39BEFDAC896F",
        name="EFI_CUSTOMIZED_DECOMPRESS_PROTOCOL_GUID",
    ),
    UefiGuid("AE80D021-618E-11D4-BCD70080C73C8881", name="EFI_DATA_HUB_PROTOCOL_GUID"),
    UefiGuid("EBA4E8D2-3858-41EC-A2812647BA9660D0", name="EFI_DEBUGPORT_PROTOCOL_GUID"),
    UefiGuid(
        "BE499C92-7D4B-11D4-BCEE0080C73C8881", name="EFI_DEBUG_ASSERT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "49152E77-1ADA-4764-B7A27AFEFED95E8B", name="EFI_DEBUG_IMAGE_INFO_TABLE_GUID"
    ),
    UefiGuid("3BD930FD-F823-4948-869198E6FE36ACE2", name="EFI_DEBUG_MASK_PPI_GUID"),
    UefiGuid(
        "4C8A2451-C207-405B-969499EA13251341", name="EFI_DEBUG_MASK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "2755590C-6F3C-42FA-9EA4A3BA543CDA25", name="EFI_DEBUG_SUPPORT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "D8117CFE-94A6-11D4-9A3A0090273FC14D", name="EFI_DECOMPRESS_PROTOCOL_GUID"
    ),
    UefiGuid(
        "15853D7C-3DDF-43E0-A1CBEBF85B8F872C",
        name="EFI_DEFERRED_IMAGE_LOAD_PROTOCOL_GUID",
    ),
    UefiGuid("AF6AC311-84C3-11D2-8E3C00A0C969723B", name="EFI_DEVICE_IO_PROTOCOL_GUID"),
    UefiGuid(
        "05C99A21-C70F-4AD2-8A5F35DF3343F51E",
        name="EFI_DEVICE_PATH_FROM_TEXT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "09576E91-6D3F-11D2-8E3900A0C969723B", name="EFI_DEVICE_PATH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "8B843E20-8132-4852-90CC551A4E4A7F1C",
        name="EFI_DEVICE_PATH_TO_TEXT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "0379BE4E-D706-437D-B037EDB82FB772A4",
        name="EFI_DEVICE_PATH_UTILITIES_PROTOCOL_GUID",
    ),
    UefiGuid("8A219718-4EF5-4761-91C8C0F04BDA9E56", name="EFI_DHCP4_PROTOCOL_GUID"),
    UefiGuid(
        "9D9A39D8-BD42-4A73-A4D58EE94BE11380",
        name="EFI_DHCP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "9E498932-4ABC-45AF-A34D0247787BE7C6", name="EFI_DISK_INFO_AHCI_INTERFACE_GUID"
    ),
    UefiGuid(
        "5E948FE3-26D3-42B5-AF17610287188DEC", name="EFI_DISK_INFO_IDE_INTERFACE_GUID"
    ),
    UefiGuid("D432A67F-14DC-484B-B3BB3F0291849327", name="EFI_DISK_INFO_PROTOCOL_GUID"),
    UefiGuid(
        "08F74BAA-EA36-41D9-952121A70F8780BC", name="EFI_DISK_INFO_SCSI_INTERFACE_GUID"
    ),
    UefiGuid(
        "CB871572-C11A-47B5-B492675EAFA77727", name="EFI_DISK_INFO_USB_INTERFACE_GUID"
    ),
    UefiGuid("CE345171-BA0B-11D2-8E4F00A0C969723B", name="EFI_DISK_IO_PROTOCOL_GUID"),
    UefiGuid("67EF7A73-2594-4A5E-930AE166FABCD2C8", name="EFI_DMI_FORMAT_FRU_GUID"),
    UefiGuid(
        "18A031AB-B443-4D1A-A5C00C09261E9F71", name="EFI_DRIVER_BINDING_PROTOCOL_GUID"
    ),
    UefiGuid(
        "BFD7DC1D-24F1-40D9-82E72E09BB6B4EBE",
        name="EFI_DRIVER_CONFIGURATION2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "107A772B-D5E1-11D4-9A460090273FC14D",
        name="EFI_DRIVER_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "4D330321-025F-4AAC-90D85ED900173B63",
        name="EFI_DRIVER_DIAGNOSTICS2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "0784924F-E296-11D4-9A490090273FC14D",
        name="EFI_DRIVER_DIAGNOSTICS_PROTOCOL_GUID",
    ),
    UefiGuid(
        "B1EE129E-DA36-4181-91F804A4923766A7",
        name="EFI_DRIVER_FAMILY_OVERRIDE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "2A534210-9280-41D8-AE79CADA01A2B127", name="EFI_DRIVER_HEALTH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "5C198761-16A8-4E69-972C89D67954F81D",
        name="EFI_DRIVER_SUPPORTED_EFI_VERSION_PROTOCOL_GUID",
    ),
    UefiGuid("9851740C-22E0-440D-9090EF2D71C251C9", name="EFI_DXE_CMOS_ACCESS_GUID"),
    UefiGuid("0AE8CE5D-E448-4437-A8D7EBF5F194F731", name="EFI_DXE_IPL_PPI_GUID"),
    UefiGuid("05AD34BA-6F02-4214-952E4DA0398E2BB9", name="EFI_DXE_SERVICES_TABLE_GUID"),
    UefiGuid(
        "60FF8964-E906-41D0-AFEDF241E974E08E",
        name="EFI_DXE_SMM_READY_TO_LOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "13AC6DD1-73D0-11D4-B06B00AA00BD6DE7", name="EFI_EBC_INTERPRETER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "BD8C1056-9F36-44EC-92A8A6337F817986", name="EFI_EDID_ACTIVE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "1C0C34F6-D380-41FA-A0498AD06C1A66AA", name="EFI_EDID_DISCOVERED_PROTOCOL_GUID"
    ),
    UefiGuid(
        "48ECB431-FB72-45C0-A922F458FE040BD5", name="EFI_EDID_OVERRIDE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "62CEEF5A-1D7C-4943-9B3A95E2494C8990", name="EFI_EMUL6064KBDINPUT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "7578B307-B25B-44F9-892E209B0E3993C6", name="EFI_EMUL6064MSINPUT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "6EA0F71C-614A-437E-8F49243AD4E83268", name="EFI_EMUL6064TRAP_PROTOCOL_GUID"
    ),
    UefiGuid("2A571201-4966-47F6-8B86F31E41F32F10", name="EFI_EVENT_LEGACY_BOOT_GUID"),
    UefiGuid(
        "5AEA42B5-31E1-4515-BC31B8D5257565A6",
        name="EFI_EXTENDED_SAL_BASE_IO_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "D9E9FA06-0FE0-41C3-96FB83425A3394F8",
        name="EFI_EXTENDED_SAL_BASE_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "0EDC9494-2743-4BA5-88180AEF5213F188",
        name="EFI_EXTENDED_SAL_CACHE_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "D5E4EE5F-3E0A-453C-A725B692BB06365A",
        name="EFI_EXTENDED_SAL_ELOG_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A2271DF1-BCBB-4F1D-98A906BC172F071A",
        name="EFI_EXTENDED_SAL_FV_BLOCK_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "76B75C23-FE4F-4E17-A2AD1A653DBB494A",
        name="EFI_EXTENDED_SAL_LOCK_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "CB3FD86E-38A3-4C03-9A5C90CFA3A2AB7A",
        name="EFI_EXTENDED_SAL_MCA_LOG_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "2A591128-6CC7-42B1-8AF058933B682DBB",
        name="EFI_EXTENDED_SAL_MCA_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "697D81A2-CF18-4DC0-9E0D06113B618A3F",
        name="EFI_EXTENDED_SAL_MP_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "899AFD18-75E8-408B-A41A6E2E7ECDF454",
        name="EFI_EXTENDED_SAL_MTC_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "E1CD9D21-0FC2-438D-970304E66D961E57",
        name="EFI_EXTENDED_SAL_PAL_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A46B1A31-AD66-4905-92F62B4659DC3063",
        name="EFI_EXTENDED_SAL_PCI_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7D019990-8CE1-46F5-A7763C5198676AA0",
        name="EFI_EXTENDED_SAL_RESET_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7E97A470-EFDB-4D02-8FCE6190D27BA296",
        name="EFI_EXTENDED_SAL_RTC_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "4A153B6E-85A1-4982-98F46A8CFCA4ABA1",
        name="EFI_EXTENDED_SAL_SENSOR_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "04356799-81B7-4E08-A38DD978FA47BA42",
        name="EFI_EXTENDED_SAL_SM_COM_LAYER_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid("38802700-868A-4B4E-81D44F1BDCCFB46F", name="EFI_EXTENDED_SAL_SST_GUID"),
    UefiGuid(
        "53A58D06-AC27-4D8C-B5E9F08A80654170",
        name="EFI_EXTENDED_SAL_STALL_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "00DBD91D-55E9-420F-96395E9F8437B44F",
        name="EFI_EXTENDED_SAL_STATUS_CODE_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "4ECB6C53-C641-4370-8CB23B0E496E8378",
        name="EFI_EXTENDED_SAL_VARIABLE_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "C1A74056-260E-4871-A031E645A65B6E11",
        name="EFI_EXTENDED_SAL_VIRTUAL_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "143B7632-B81B-4CB7-ABD3B625A5B9BFFE",
        name="EFI_EXT_SCSI_PASS_THRU_PROTOCOL_GUID",
    ),
    UefiGuid(
        "3EBD9E82-2C78-4DE6-97868D4BFCB7C881",
        name="EFI_FAULT_TOLERANT_WRITE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "1BA0062E-C779-4582-8566336AE8F78F09", name="EFI_FFS_VOLUME_TOP_FILE_GUID"
    ),
    UefiGuid(
        "09576E93-6D3F-11D2-8E3900A0C969723B", name="EFI_FILE_SYSTEM_INFO_ID_GUID"
    ),
    UefiGuid(
        "DB47D7D3-FE81-11D3-9A350090273FC14D",
        name="EFI_FILE_SYSTEM_VOLUME_LABEL_INFO_ID_GUID",
    ),
    UefiGuid("36164812-A023-44E5-BD8505BF3C7700AA", name="EFI_FIND_FV_PPI_GUID"),
    UefiGuid(
        "8C8CE578-8A3D-4F1C-9935896185C32DD3", name="EFI_FIRMWARE_FILE_SYSTEM2_GUID"
    ),
    UefiGuid(
        "7A9354D9-0468-444A-81CE0BF617D890DF", name="EFI_FIRMWARE_FILE_SYSTEM_GUID"
    ),
    UefiGuid(
        "220E73B6-6BDB-4413-8405B974B108619A", name="EFI_FIRMWARE_VOLUME2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "8F644FA9-E850-4DB1-9CE20B44698E8DA4",
        name="EFI_FIRMWARE_VOLUME_BLOCK2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "8F644FA9-E850-4DB1-9CE20B44698E8DA4",
        name="EFI_FIRMWARE_VOLUME_BLOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7AA35A69-506C-444F-A7AF694BF56F71C8",
        name="EFI_FIRMWARE_VOLUME_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "389F751F-1838-4388-8390CD8154BD27F8", name="EFI_FIRMWARE_VOLUME_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B9D4C360-BCFB-4F9B-929853C136982258", name="EFI_FORM_BROWSER2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0FB7C852-ADCA-4853-8D0FFBA71B1CE11A", name="EFI_FORM_BROWSER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "F3E4543D-CF35-6CEF-35C44FE6344DFC54", name="EFI_FORM_CALLBACK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B7084E63-46B7-4D1A-8677E30B53DBF050", name="EFI_FRAMEWORK_DEVICE_PATH_GUID"
    ),
    UefiGuid("3F557189-8DAE-45AE-A0B32B99CA7AA7A0", name="EFI_FTW_LITE_PROTOCOL_GUID"),
    UefiGuid(
        "53A4C71B-B581-4170-91B38DB87A4B5C46", name="EFI_FVB_EXTENSION_PROTOCOL_GUID"
    ),
    UefiGuid(
        "309DE7F1-7F5E-4ACE-B49C531BE5AA95EF",
        name="EFI_GENERIC_MEMORY_TEST_PROTOCOL_GUID",
    ),
    UefiGuid("59D1C24F-50F1-401A-B101F33E0DAED443", name="EFI_GENERIC_VARIABLE_GUID"),
    UefiGuid(
        "074E1E48-8132-47A1-8C2C3F14AD9A66DC", name="EFI_GLOBAL_NVS_AREA_PROTOCOL_GUID"
    ),
    UefiGuid("8BE4DF61-93CA-11D2-AA0D00E098032B8C", name="EFI_GLOBAL_VARIABLE_GUID"),
    UefiGuid("B5D09084-80AD-4759-B51C27548AFB8B8D", name="EFI_GPIO_PROTOCOL_GUID"),
    UefiGuid(
        "9042A9DE-23DC-4A38-96FB7ADED080516A", name="EFI_GRAPHICS_OUTPUT_PROTOCOL_GUID"
    ),
    UefiGuid("FD44820B-F1AB-41C0-AE4E0C55556EB9BD", name="EFI_GUID"),
    UefiGuid(
        "414E6BDD-E47B-47CC-B244BB61020CF516", name="EFI_HARDWARE_ERROR_VARIABLE_GUID"
    ),
    UefiGuid(
        "2AE9D80F-3FB2-4095-B7B1E93157B946B6", name="EFI_HASH_ALGORITHM_SHA1_GUID"
    ),
    UefiGuid(
        "8DF01A06-9BD5-4BF7-B021DB4FD9CCF45B", name="EFI_HASH_ALGORITHM_SHA224_GUID"
    ),
    UefiGuid(
        "51AA59DE-FDF2-4EA3-BC63875FB7842EE9", name="EFI_HASH_ALGORITHM_SHA256_GUID"
    ),
    UefiGuid(
        "EFA96432-DE33-4DD2-AEE6328C33DF777A", name="EFI_HASH_ALGORITHM_SHA384_GUID"
    ),
    UefiGuid(
        "CAA4381E-750C-4770-B8707A23B4E42130", name="EFI_HASH_ALGORITHM_SHA512_GUID"
    ),
    UefiGuid("0AF7C79C-65B5-4319-B0AE44EC484E4AD7", name="EFI_HASH_ALGORTIHM_MD5_GUID"),
    UefiGuid("C5184932-DBA5-46DB-A5BACC0BDA9C1435", name="EFI_HASH_PROTOCOL_GUID"),
    UefiGuid(
        "42881C98-A4F3-44B0-A39DDFA18667D8CD",
        name="EFI_HASH_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "330D4706-F2A0-4E4F-A369B66FA8D54385",
        name="EFI_HII_CONFIG_ACCESS_PROTOCOL_GUID",
    ),
    UefiGuid(
        "587E72D7-CC50-4F79-8209CA291FC1A10F",
        name="EFI_HII_CONFIG_ROUTING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "EF9FC172-A1B2-4693-B3276D32FC416042", name="EFI_HII_DATABASE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "F22FC20C-8CF4-45EB-8E06AD4E50B95DD3", name="EFI_HII_DRIVER_HEALTH_FORMSET_GUID"
    ),
    UefiGuid("2A57AE75-8B7A-4C64-8656DB51DDC36F7B", name="EFI_HII_EXT_PROTOCOL_GUID"),
    UefiGuid("E9CA4775-8657-47FC-97E77ED65A084324", name="EFI_HII_FONT_PROTOCOL_GUID"),
    UefiGuid(
        "94D411B7-7669-45C3-BA3BF3A58A715681", name="EFI_HII_FRONT_PAGE_CLASS_GUID"
    ),
    UefiGuid("31A6406A-6BDF-4E46-B2A2EBAA89C40920", name="EFI_HII_IMAGE_PROTOCOL_GUID"),
    UefiGuid("EA816D2C-CEE5-4F02-99B5D3905CBBD077", name="EFI_HII_NEW_PROTOCOL_GUID"),
    UefiGuid("CD361957-AFBE-425E-A3585F5889CFFE7B", name="EFI_HII_OLD_PROTOCOL_GUID"),
    UefiGuid(
        "6A1EE763-D47A-43B4-AABEEF1DE2AB56FC", name="EFI_HII_PACKAGE_LIST_PROTOCOL_GUID"
    ),
    UefiGuid(
        "93039971-8545-4B04-B45E32EB8326040E",
        name="EFI_HII_PLATFORM_SETUP_FORMSET_GUID",
    ),
    UefiGuid("D7AD636E-B997-459B-BF3F8846897980E1", name="EFI_HII_PROTOCOL_GUID"),
    UefiGuid(
        "14982A4F-B0ED-45B8-A8115A7A9BC232DF",
        name="EFI_HII_SET_KEYBOARD_LAYOUT_EVENT_GUID",
    ),
    UefiGuid(
        "0FD96974-23AA-4CDC-B9CB98D17750322A", name="EFI_HII_STRING_PROTOCOL_GUID"
    ),
    UefiGuid(
        "337F4407-5AEE-4B83-B2A74EADCA3088CD",
        name="EFI_HII_USER_CREDENTIAL_FORMSET_GUID",
    ),
    UefiGuid("7739F24C-93D7-11D4-9A3A0090273FC14D", name="EFI_HOB_LIST_GUID"),
    UefiGuid(
        "564B33CD-C92A-4593-90BF2473E43C6322",
        name="EFI_HOB_MEMORY_ALLOC_BSP_STORE_GUID",
    ),
    UefiGuid(
        "4ED4BF27-4092-42E9-807D527B1D00C9BD", name="EFI_HOB_MEMORY_ALLOC_STACK_GUID"
    ),
    UefiGuid("F1E48287-3FE1-4535-89AB48D6C3DA2759", name="EFI_HOT_KEYS_PROTOCOL_GUID"),
    UefiGuid("BE644001-E7D4-48B1-B0968BA047BC7AE7", name="EFI_HT_BIST_HOB_GUID"),
    UefiGuid(
        "956A2ED0-A6CF-409A-B8F535F14C3E3C02",
        name="EFI_IDER_CONTROLLER_DRIVER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A1E37052-80D9-4E65-A3173E9A55C43EC9",
        name="EFI_IDE_CONTROLLER_INIT_PROTOCOL_GUID",
    ),
    UefiGuid("0F0B1735-87A0-4193-B266538C38AF48CE", name="EFI_IFR_TIANO_GUID"),
    UefiGuid(
        "D719B2CB-3D3A-4596-A3BCDAD00E67656F", name="EFI_IMAGE_SECURITY_DATABASE_GUID"
    ),
    UefiGuid(
        "EB23F55A-7863-4AC2-8D3D956535DE0375",
        name="EFI_INCOMPATIBLE_PCI_DEVICE_SUPPORT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "79E8C9C7-1152-4F00-B83114F1C4041AE0", name="EFI_INTEL_MFG_FORMAT_FRU_GUID"
    ),
    UefiGuid("D4A28A3E-DCF2-43CF-A2B7F3572A7CAB09", name="EFI_IOBASE_HOB_GUID"),
    UefiGuid(
        "3B95AA31-3793-434B-8667C8070892E05E", name="EFI_IP4_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid("41D94CD2-35B6-455A-8258D4E51334AADD", name="EFI_IP4_PROTOCOL_GUID"),
    UefiGuid(
        "C51711E7-B4BF-404A-BFB80A048EF1FFE4",
        name="EFI_IP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("89DEEF37-31D8-469C-95B60169BC866CFB", name="EFI_IP_PROTOCOL_GUID"),
    UefiGuid("64A892DC-5561-4536-92C7799BFC183355", name="EFI_ISA_ACPI_PROTOCOL_GUID"),
    UefiGuid("7EE2BD44-3DA0-11D4-9A380090273FC14D", name="EFI_ISA_IO_PROTOCOL_GUID"),
    UefiGuid(
        "59324945-EC44-4C0D-B1CD9DB139DF070C",
        name="EFI_ISCSI_INITIATOR_NAME_PROTOCOL_GUID",
    ),
    UefiGuid(
        "38321DBA-4FE0-4E17-8AEC413055EAEDC1", name="EFI_LEGACY_8259_PROTOCOL_GUID"
    ),
    UefiGuid(
        "8E008510-9BB1-457D-9F70897ABA865DB9", name="EFI_LEGACY_BIOS_EXT_PROTOCOL_GUID"
    ),
    UefiGuid("2E3044AC-879F-490F-9760BBDFAF695F50", name="EFI_LEGACY_BIOS_GUID"),
    UefiGuid(
        "783658A3-4172-4421-A299E009079C0CB4",
        name="EFI_LEGACY_BIOS_PLATFORM_PROTOCOL_GUID",
    ),
    UefiGuid(
        "DB9A1E3D-45CB-4ABB-853BE5387FDB2E2D", name="EFI_LEGACY_BIOS_PROTOCOL_GUID"
    ),
    UefiGuid(
        "4C51A7BA-7195-442D-8792BEEA6E2FF6EC",
        name="EFI_LEGACY_BIOS_THUNK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "31CE593D-108A-485D-ADB278F21F2966BE", name="EFI_LEGACY_INTERRUPT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "A062CF1F-8473-4AA3-8793600BC4FFA9A9", name="EFI_LEGACY_SREDIR_PROTOCOL_GUID"
    ),
    UefiGuid(
        "7CC7ED80-9A68-4781-80E4DA1699105AFE", name="EFI_LIGHT_ISA_IO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "BC62157E-3E33-4FEC-99202D3B36D750DF",
        name="EFI_LOADED_IMAGE_DEVICE_PATH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "5B1B31A1-9562-11D2-8E3F00A0C969723B", name="EFI_LOADED_IMAGE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "4006C0C1-FCB3-403E-996D4A6C8724E06D", name="EFI_LOAD_FILE2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "7AB33A91-ACE5-4326-B572E7EE33D39F16", name="EFI_MANAGED_NETWORK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "F36FF770-A7E1-42CF-9ED256F0F271F44C",
        name="EFI_MANAGED_NETWORK_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("80DBD530-B74C-4F11-8C03418665532831", name="EFI_MEMORY_CONFIG_DATA_GUID"),
    UefiGuid("1D7ADD6E-B2DA-4B0B-B29F49CB42F46356", name="EFI_MEMORY_PRODUCER_GUID"),
    UefiGuid(
        "1767CEED-DB82-47CD-BF2B68458A8CCFFF", name="EFI_MEMORY_SUBCLASS_DRIVER_GUID"
    ),
    UefiGuid("4E8F4EBB-64B9-4E05-9B184CFE49235097", name="EFI_MEMORY_SUBCLASS_GUID"),
    UefiGuid("D50234F4-6F4B-43E8-A0133C1E33D9B9B1", name="EFI_MEMORY_TYPE_FRU_GUID"),
    UefiGuid(
        "26BACCB2-6F42-11D4-BCE70080C73C8881", name="EFI_METRONOME_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "1BAD711C-D451-4241-B1F38537812E0C70", name="EFI_ME_BIOS_EXTENSION_SETUP_GUID"
    ),
    UefiGuid("E1A21D94-4A20-4E0E-AE09A9A21F24BB9E", name="EFI_ME_FW_SKU_VARIABLE_GUID"),
    UefiGuid(
        "11FBFDFB-10D2-43E6-B5B1B4386EDCCB9A", name="EFI_ME_RC_INFO_PROTOCOL_GUID"
    ),
    UefiGuid("86AD232B-D33A-465C-BF5F41370BA92FE2", name="EFI_MINI_SHELL_FILE_GUID"),
    UefiGuid("62512C92-63C4-4D80-82B1C1A4DC4480E5", name="EFI_MISC_PRODUCER_GUID"),
    UefiGuid(
        "F50E702C-8653-4CDE-BBCE43B4D55B34B8", name="EFI_MISC_SUBCLASS_DRIVER_GUID"
    ),
    UefiGuid("772484B2-7482-4B91-9F9AAD43F81C5881", name="EFI_MISC_SUBCLASS_GUID"),
    UefiGuid(
        "1DA97072-BDDC-4B30-99F172A0B56FFF2A",
        name="EFI_MONTONIC_COUNTER_ARCH_PROTOCOL_GUID",
    ),
    UefiGuid("EB9D2D2F-2D88-11D3-9A160090273FC14D", name="EFI_MPS_TABLE_GUID"),
    UefiGuid(
        "3FDDA605-A76E-4F46-AD2912F4531B3D08", name="EFI_MP_SERVICES_PROTOCOL_GUID"
    ),
    UefiGuid("78247C57-63DB-4708-99C2A8B4A9A61F6B", name="EFI_MTFTP4_PROTOCOL_GUID"),
    UefiGuid(
        "2FE800BE-8F01-4AA6-946BD71388E1833F",
        name="EFI_MTFTP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "FF2D54D4-5C55-4C06-8549C3627CB8B995",
        name="EFI_NB_ERROR_LOG_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid("D7BD52B0-B2DC-4F08-B467DE50D728F6BD", name="EFI_NB_MRC_INFO_GUID"),
    UefiGuid(
        "E18541CD-F755-4F73-928D643C8A79B229",
        name="EFI_NETWORK_INTERFACE_IDENTIFIER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "0DCA3D4D-12DA-4728-BF7E86CEB928D067", name="EFI_NIC_IP4_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid(
        "D8944553-C4DD-41F4-9B30E1397CFB267B", name="EFI_NIC_IP4_CONFIG_VARIABLE_GUID"
    ),
    UefiGuid(
        "68B81E51-2583-4582-95DBC5723236C4F1",
        name="EFI_NONSMMEMUL6064TRAP_PROTOCOL_GUID",
    ),
    UefiGuid(
        "170E13C0-BF1B-4218-871D2ABDC6F887BC", name="EFI_OEM_BADGING_PROTOCOL_GUID"
    ),
    UefiGuid(
        "C12A7328-F81F-11D2-BA4B00A0C93EC93B", name="EFI_PART_TYPE_EFI_SYSTEM_PART_GUID"
    ),
    UefiGuid(
        "024DEE41-33E7-11D3-9D690008C781F39F", name="EFI_PART_TYPE_LEGACY_MBR_GUID"
    ),
    UefiGuid("7644C181-FA6E-46DA-80CB04B9904062E8", name="EFI_PATH_FILE_NAME_GUID"),
    UefiGuid("CA54F443-1EF2-4DAB-9E7E6DB7B720B587", name="EFI_PCCARD_CS_PROTOCOL_GUID"),
    UefiGuid("C38E6D34-5A7F-4BF9-BE5794DD30380276", name="EFI_PCCARD_SS_PROTOCOL_GUID"),
    UefiGuid("984EB4E9-5A95-41DE-AAD053668CA513C0", name="EFI_PCH_INFO_PROTOCOL_GUID"),
    UefiGuid("271DD6F2-54CB-45E6-85858C923C1AC706", name="EFI_PCH_S3_IMAGE_GUID"),
    UefiGuid(
        "E287D20B-D897-4E1E-A5D9977763936A04", name="EFI_PCH_S3_SUPPORT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3677D529-326F-4603-A926EAACE01DCBB0", name="EFI_PCI_EXPRESS_BASE_ADDRESS_GUID"
    ),
    UefiGuid("0B280816-52E7-4E51-AA5711BD41CBEFC3", name="EFI_PCI_HOTPLUG_DEVICE_GUID"),
    UefiGuid(
        "AA0E8BC1-DABC-46B0-A84437B8169B2BEA",
        name="EFI_PCI_HOT_PLUG_INIT_PROTOCOL_GUID",
    ),
    UefiGuid("4CF5B200-68B8-4CA5-9EECB23E3F50029A", name="EFI_PCI_IO_PROTOCOL_GUID"),
    UefiGuid(
        "7462660F-1CBD-48DA-AD1191717913831C", name="EFI_PCI_OPTION_ROM_TABLE_GUID"
    ),
    UefiGuid(
        "07D75280-27D4-4D69-90D05643E238B341", name="EFI_PCI_PLATFORM_PROTOCOL_GUID"
    ),
    UefiGuid(
        "2F707EBB-4A1A-11D4-9A380090273FC14D",
        name="EFI_PCI_ROOT_BRIDGE_IO_PROTOCOL_GUID",
    ),
    UefiGuid(
        "F238F47C-1DE9-4E8B-81B9CC924E6B5BE5", name="EFI_PCMCIA_REQUEST_PROTOCOL_GUID"
    ),
    UefiGuid("E0C14753-F9BE-11D2-9A0C0090273FC14D", name="EFI_PC_ANSI_GUID"),
    UefiGuid(
        "20B0F1C2-B0D8-4C5D-AAD9F44580DFDF8B", name="EFI_PEI_AMI_KEYCODE_PPI_GUID"
    ),
    UefiGuid(
        "1B45CC0A-156A-428A-AF6249864DA0E6E6", name="EFI_PEI_APRIORI_FILE_NAME_GUID"
    ),
    UefiGuid("066785B1-EDB8-46DC-842F574404B8692F", name="EFI_PEI_CAPSULE_PPI_GUID"),
    UefiGuid("CCBF2786-CD6C-4308-BDFA625870CEBF81", name="EFI_PEI_CMOS_ACCESS_GUID"),
    UefiGuid(
        "73DDB5E1-5FB4-4751-AF1E83CF75BECBB6",
        name="EFI_PEI_CONSPLIT_AMI_KEYCODE_PPI_GUID",
    ),
    UefiGuid("D641A0F5-CB7C-4846-A3801D01B4D9E3B9", name="EFI_PEI_CORE_PRIVATE_GUID"),
    UefiGuid(
        "E6AF1F7B-FC3F-46DA-A828A3B457A44282", name="EFI_PEI_CPU_IO_PPI_INSTALLED_GUID"
    ),
    UefiGuid("1A36E4E7-FAB6-476A-8E75695A0576FDD7", name="EFI_PEI_DECOMPRESS_PPI_GUID"),
    UefiGuid(
        "0DE2CE25-446A-45A7-BFC937DA26344B37",
        name="EFI_PEI_DEVICE_RECOVERY_MODULE_PPI_GUID",
    ),
    UefiGuid(
        "605EA650-C65C-42E1-BA8091A52AB618C6", name="EFI_PEI_END_OF_PEI_PHASE_PPI_GUID"
    ),
    UefiGuid(
        "49EDB1C1-BF21-4761-BB12EB0031AABB39",
        name="EFI_PEI_FIRMWARE_VOLUME_INFO_PPI_GUID",
    ),
    UefiGuid(
        "D8117CFC-94A6-11D4-9A3A0090273FC14D",
        name="EFI_PEI_FLUSH_INSTRUCTION_CACHE_GUID",
    ),
    UefiGuid(
        "C1FCD448-6300-4458-B86428DF015364BC", name="EFI_PEI_LOADED_IMAGE_PPI_GUID"
    ),
    UefiGuid("B9E0ABFE-5979-4914-977F6DEE78C278A6", name="EFI_PEI_LOAD_FILE_GUID"),
    UefiGuid("B9E0ABFE-5979-4914-977F6DEE78C278A6", name="EFI_PEI_LOAD_FILE_PPI_GUID"),
    UefiGuid("057A449A-1FDC-4C06-BFC9F53F6A99BB92", name="EFI_PEI_PCI_CFG2_PPI_GUID"),
    UefiGuid(
        "E1F2EBA0-F7B9-4A26-8620131221642A90", name="EFI_PEI_PCI_CFG_PPI_INSTALLED_GUID"
    ),
    UefiGuid(
        "10F432DE-DEEC-4631-80CD47F65D8F80BB", name="EFI_PEI_PERFORMANCE_HOB_GUID"
    ),
    UefiGuid("D8117CFF-94A6-11D4-9A3A0090273FC14D", name="EFI_PEI_PE_COFF_LOADER_GUID"),
    UefiGuid(
        "8C72C36E-DACC-4E81-8D60FFA61D88FF54",
        name="EFI_PEI_PLATFORMPEI_EXECUTED_PPI_GUID",
    ),
    UefiGuid(
        "2AB86EF5-ECB5-4134-B5563854CA1FE1B4",
        name="EFI_PEI_READ_ONLY_VARIABLE2_PPI_GUID",
    ),
    UefiGuid(
        "3CDC90C6-13FB-4A75-9E7959E9DD78B9FA",
        name="EFI_PEI_READ_ONLY_VARIABLE_PPI_GUID",
    ),
    UefiGuid(
        "FB6D9542-612D-4F45-872F5CFF52E93DCF", name="EFI_PEI_RECOVERY_MODULE_PPI_GUID"
    ),
    UefiGuid(
        "229832D3-7A30-4B36-B827F40CB7D45436",
        name="EFI_PEI_REPORT_PROGRESS_CODE_PPI_GUID",
    ),
    UefiGuid("EF398D58-9DFD-4103-BF9478C6F4FE712F", name="EFI_PEI_RESET_PPI_GUID"),
    UefiGuid(
        "0065D394-9951-4144-82A30AFC8579C251", name="EFI_PEI_RSC_HANDLER_PPI_GUID"
    ),
    UefiGuid("6D582DBC-DB85-4514-8FCC5ADF6227B147", name="EFI_PEI_S3_RESUME2_PPI_GUID"),
    UefiGuid(
        "4F89E208-E144-4804-9EC80F894F7E36D7",
        name="EFI_PEI_SECTION_EXTRACTION_PPI_GUID",
    ),
    UefiGuid("DCD0BE23-9586-40F4-B64306522CED4EDE", name="EFI_PEI_SECURITY2_PPI_GUID"),
    UefiGuid("1F4C6F90-B06B-48D8-A201BAE5F1CD7D56", name="EFI_PEI_STALL_PPI_GUID"),
    UefiGuid(
        "DBE23AA9-A345-4B97-85B6B226F1617389",
        name="EFI_PEI_TEMPORARY_RAM_SUPPORT_PPI_GUID",
    ),
    UefiGuid(
        "D8117D02-94A6-11D4-9A3A0090273FC14D", name="EFI_PEI_TRANSFER_CONTROL_GUID"
    ),
    UefiGuid(
        "FFECFFFF-923C-14D2-9E3F22A0C969563B", name="EFI_PERFORMANCE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3FDDA605-A76E-4F46-AD2912F4531B3D08", name="EFI_PI_MP_SERVICES_PROTOCOL_GUID"
    ),
    UefiGuid(
        "6B30C738-A391-11D4-9A3B0090273FC14D",
        name="EFI_PLATFORM_DRIVER_OVERRIDE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "EC2E931B-3281-48A5-8107DF8A8BED3C5D",
        name="EFI_PLATFORM_GOP_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "D9035175-8CE2-47DE-A8B8CC98E5E2A885", name="EFI_PLATFORM_INFO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0859BA18-7DD7-4ED7-A88E109C63917BDD", name="EFI_PLATFORM_MEMTEST_PROTOCOL_GUID"
    ),
    UefiGuid(
        "345ECC0E-0CB6-4B75-BB571B129C47333E",
        name="EFI_PLATFORM_TO_DRIVER_CONFIGURATION_CLP_GUID",
    ),
    UefiGuid(
        "642CD590-8059-4C0A-A958C5EC07D23C4B",
        name="EFI_PLATFORM_TO_DRIVER_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid("0468A601-C535-46FD-A95DBBAB991B178C", name="EFI_POWER_ON_HOB_GUID"),
    UefiGuid(
        "6C50CDCB-7F46-4DCC-8DDDD9F0A3C61128",
        name="EFI_PPM_GLOBAL_NVS_AREA_PROTOCOL_GUID",
    ),
    UefiGuid("D71DB106-E32D-4225-BFF4DE6D77871761", name="EFI_PPM_INFO_PROTOCOL_GUID"),
    UefiGuid(
        "E451DCBE-96A1-4729-A5CF6B9C2CFF47FD", name="EFI_PRIMARY_CONSOLE_IN_DEVICE_GUID"
    ),
    UefiGuid(
        "62BDF38A-E3D5-492C-950C23A7F66E672E",
        name="EFI_PRIMARY_CONSOLE_OUT_DEVICE_GUID",
    ),
    UefiGuid(
        "5A68191B-9B97-4752-9946E36A5DA942B1",
        name="EFI_PRIMARY_STANDARD_ERROR_DEVICE_GUID",
    ),
    UefiGuid("DF2D868E-32FC-4CF0-8E6BFFD95D1343D0", name="EFI_PRINT_PROTOCOL_GUID"),
    UefiGuid("1BF06AEA-5BEC-4A8D-9576749B09562D30", name="EFI_PROCESSOR_PRODUCER_GUID"),
    UefiGuid("26FDEB7E-B8AF-4CCF-AA9702633CE48CA7", name="EFI_PROCESSOR_SUBCLASS_GUID"),
    UefiGuid(
        "4DF19259-DC71-4D46-BEF1357BB578C418", name="EFI_PS2_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid(
        "245DCA21-FB7B-11D3-8F0100A0C969723B",
        name="EFI_PXE_BASE_CODE_CALLBACK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "03C4E603-AC28-11D3-9A2D0090273FC14D", name="EFI_PXE_BASE_CODE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "C1544C01-92A4-4198-8A84778583C23621",
        name="EFI_PXE_DHCP4_CALLBACK_PROTOCOL_GUID",
    ),
    UefiGuid("03C4E624-AC28-11D3-9A2D0090293FC14D", name="EFI_PXE_DHCP4_PROTOCOL_GUID"),
    UefiGuid(
        "27CFAC87-46CC-11D4-9A380090273FC14D",
        name="EFI_REAL_TIME_CLOCK_ARCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "27CFAC88-46CC-11D4-9A380090273FC14D", name="EFI_RESET_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E857CAF6-C046-45DC-BE3FEE0765FBA887", name="EFI_S3_SAVE_STATE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "320AFE62-E593-49CB-A9F1D4C2F4AF014C",
        name="EFI_S3_SMM_SAVE_STATE_PROTOCOL_GUID",
    ),
    UefiGuid("EB9D2D32-2D88-11D3-9A160090273FC14D", name="EFI_SAL_SYSTEM_TABLE_GUID"),
    UefiGuid("493B5BAC-BB9E-4BF5-837920E2ACA98541", name="EFI_SA_INFO_PROTOCOL_GUID"),
    UefiGuid(
        "F281FC6E-F4C4-431C-962B2F13AE7984EC",
        name="EFI_SB_PCIE_ERROR_LOG_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid("932F47E6-2362-4002-803E3CD54B138F85", name="EFI_SCSI_IO_PROTOCOL_GUID"),
    UefiGuid(
        "A59E8FCF-BDA0-43BB-90B1D3732ECAA877", name="EFI_SCSI_PASS_THRU_PROTOCOL_GUID"
    ),
    UefiGuid(
        "448F5DA4-6DD7-4FE1-930769224192215D",
        name="EFI_SECTION_EXTRACTION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A46423E3-4617-49F1-B9FFD1BFA9115839", name="EFI_SECURITY_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "78E4D245-CD4D-4A05-A2BA4743E86CFCAB", name="EFI_SECURITY_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid(
        "6F8C2B35-FEF4-448D-8256E11B19D61077", name="EFI_SEC_PLATFORM_INFORMATION_GUID"
    ),
    UefiGuid("3BF4AF16-AB7C-4B43-898DAB26AC5DDC6C", name="EFI_SEC_SMI_FLASH_GUID"),
    UefiGuid(
        "F52C3858-5EF8-4D41-834EC39EEF8A45A3", name="EFI_SERIAL_GPIO_PROTOCOL_GUID"
    ),
    UefiGuid("BB25CF6F-F1D4-11D2-9A0C0090273FC1FD", name="EFI_SERIAL_IO_PROTOCOL_GUID"),
    UefiGuid("C57AD6B7-0515-40A8-9D21551652854E37", name="EFI_SHELL_FILE_GUID"),
    UefiGuid(
        "C723F288-52F9-4D80-B633E152F930A0DC", name="EFI_SIMPLE_AUDIO_OUT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "964E5B22-6459-11D2-8E3900A0C969723B",
        name="EFI_SIMPLE_FILE_SYSTEM_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A19832B9-AC25-11D3-9A2D0090273FC14D", name="EFI_SIMPLE_NETWORK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "DD9E7534-7762-4698-8C14F58517A625AA",
        name="EFI_SIMPLE_TEXT_INPUT_EX_PROTOCOL_GUID",
    ),
    UefiGuid(
        "387477C1-69C7-11D2-8E3900A0C969723B",
        name="EFI_SIMPLE_TEXT_INPUT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "387477C1-69C7-11D2-8E3900A0C969723B", name="EFI_SIMPLE_TEXT_IN_PROTOCOL_GUID"
    ),
    UefiGuid(
        "387477C2-69C7-11D2-8E3900A0C969723B", name="EFI_SIMPLE_TEXT_OUT_PROTOCOL_GUID"
    ),
    UefiGuid("215FDD18-BD50-4FEB-890B58CA0B4739E9", name="EFI_SIO_PROTOCOL_GUID"),
    UefiGuid(
        "0903DD14-2CA0-458A-B5EB0C0CA30D785C", name="EFI_SMBIOS_BOARD_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E380280C-4C35-4AA3-B9617AE489A2B926", name="EFI_SMBIOS_DYNAMIC_DATA_GUID"
    ),
    UefiGuid("4B3082A3-80C6-4D7E-9CD0583917265DF1", name="EFI_SMBIOS_NVRAM_DATA_GUID"),
    UefiGuid("03583FF6-CB36-4940-947EB9B39F4AFAF7", name="EFI_SMBIOS_PROTOCOL_GUID"),
    UefiGuid("DAF4BF89-CE71-4917-B522C89D32FBC59F", name="EFI_SMBIOS_STATIC_DATA_GUID"),
    UefiGuid("EB9D2D31-2D88-11D3-9A160090273FC14D", name="EFI_SMBIOS_TABLE_GUID"),
    UefiGuid("707BE83E-0BF6-40A5-BE6434C03AA0B8E2", name="EFI_SMBUS_ARP_MAP_GUID"),
    UefiGuid("E49D33ED-513D-4634-B6986F55AA751C1B", name="EFI_SMBUS_HC_PROTOCOL_GUID"),
    UefiGuid(
        "3792095A-E309-4C1E-AA0185F5655A17F1", name="EFI_SMM_ACCESS_PROTOCOL_GUID"
    ),
    UefiGuid("1390954D-DA95-4227-93287282C217DAA8", name="EFI_SMM_BASE_PROTOCOL_GUID"),
    UefiGuid(
        "E512DFE4-BF44-480D-9B7A777B0BE32775",
        name="EFI_SMM_BIOS_WRITE_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid("E5D3026A-1CA5-40F0-8FB64B1AFA3C6EAA", name="EFI_SMM_CMOS_ACCESS_GUID"),
    UefiGuid(
        "C68ED8E2-9DC6-4CBD-9D94DB65ACC5C332",
        name="EFI_SMM_COMMUNICATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "8D12E231-C667-4FD1-98F22449A7E7B2E5", name="EFI_SMM_CONTROL_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3242A9D8-CE70-4AA0-955D5E7B140DE4D2", name="EFI_SMM_CPU_IO2_PROTOCOL_GUID"
    ),
    UefiGuid("5F439A0B-45D8-4682-A4F4F0576B513441", name="EFI_SMM_CPU_IO_GUID"),
    UefiGuid("EB346B97-975F-4A9F-8B22F8E92BB3D569", name="EFI_SMM_CPU_PROTOCOL_GUID"),
    UefiGuid(
        "21F302AD-6E94-471B-84BCB14800403A1D",
        name="EFI_SMM_CPU_SAVE_STATE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "25566B03-B577-4CBF-958CED663EA24380",
        name="EFI_SMM_GPI_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "E0744B81-9513-49CD-8CEAE9245E7039DA", name="EFI_SMM_GPI_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3920405B-C897-44DA-88F34C498A6FF736",
        name="EFI_SMM_ICHN_DISPATCH_EX_PROTOCOL_GUID",
    ),
    UefiGuid(
        "C50B323E-9075-4F2A-AC8ED2596A1085CC",
        name="EFI_SMM_ICHN_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "58DC368D-7BFA-4E77-ABBC0E29418DF930",
        name="EFI_SMM_IO_TRAP_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "DB7F536B-EDE4-4714-A5C8E346EBAA201D",
        name="EFI_SMM_IO_TRAP_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "8BC1714D-FFCB-41C3-89DC6C74D06D98EA",
        name="EFI_SMM_PCI_ROOT_BRIDGE_IO_PROTOCOL_GUID",
    ),
    UefiGuid(
        "4CEC368E-8E8E-4D71-8BE1958C45FC8A53",
        name="EFI_SMM_PERIODIC_TIMER_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "9CCA03FC-4C9E-4A19-9B06ED7B479BDE55",
        name="EFI_SMM_PERIODIC_TIMER_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "1B1183FA-1823-46A7-88729C578755409D",
        name="EFI_SMM_POWER_BUTTON_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "B709EFA0-47A6-4B41-B93112ECE7A8EE56",
        name="EFI_SMM_POWER_BUTTON_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "FCDF7788-7878-11FF-CC7788CCCC7788CC", name="EFI_SMM_RC_TO_APTIO_BRIDGE_GUID"
    ),
    UefiGuid(
        "47B7FA8C-F4BD-4AF6-8200333086F0D2C8",
        name="EFI_SMM_READY_TO_LOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "2FF29FA7-5E80-4ED9-B380017D3C554FF4", name="EFI_SMM_RSC_HANDLER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "A56897A1-A77F-4600-84DB22B0A801FA9A", name="EFI_SMM_RUNTIME_PROTOCOL_GUID"
    ),
    UefiGuid(
        "395C33FE-287F-413E-A0558088C0E1D43E",
        name="EFI_SMM_RUNTIME_SERVICES_TABLE_GUID",
    ),
    UefiGuid("72E40094-2EE1-497A-8F334C934A9E9C0C", name="EFI_SMM_SMBUS_PROTOCOL_GUID"),
    UefiGuid("AFF3BE49-6D71-421C-9277A0B1508E847D", name="EFI_SMM_SPI_PROTOCOL_GUID"),
    UefiGuid(
        "7300C4A1-43F2-4017-A51BC81A7F40585B",
        name="EFI_SMM_STANDBY_BUTTON_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "78965B98-B0BF-449E-8B22D2914E498A98",
        name="EFI_SMM_STANDBY_BUTTON_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "6AFD2B77-98C1-4ACD-A6F98A9439DE0FB1", name="EFI_SMM_STATUS_CODE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "18A3C6DC-5EEA-48C8-A1C1B53389F98999", name="EFI_SMM_SW_DISPATCH2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E541B773-DD11-420C-B026DF993653F8BF", name="EFI_SMM_SW_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "456D2859-A84B-4E47-A2EE3276D886997D", name="EFI_SMM_SX_DISPATCH2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0E2D6BB1-C624-446D-9982693CD181A607", name="EFI_SMM_TCO_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid("2A82FCE6-8BB6-413E-B9EB45DFC0522DF3", name="EFI_SMM_THUNK_PROTOCOL_GUID"),
    UefiGuid(
        "EE9B8D90-C5A6-40A2-BDE252558D33CCA1",
        name="EFI_SMM_USB_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A05B6FFD-87AF-4E42-95C96228B63CF3F3", name="EFI_SMM_USB_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "ED32D533-99E6-4209-9CC02D72CDD998A7", name="EFI_SMM_VARIABLE_PROTOCOL_GUID"
    ),
    UefiGuid("EC7F4FA1-B217-42FC-A7F7020C4305D5BA", name="EFI_SMRAM_NVS_HEADER_GUID"),
    UefiGuid("5F67D40C-1D06-4E3E-8B4AD2713F46E1DC", name="EFI_SMTP_PROTOCOL_GUID"),
    UefiGuid(
        "400B4476-3081-11D6-87ED00062945C3B9",
        name="EFI_SPEAKER_INTERFACE_PROTOCOL_GUID",
    ),
    UefiGuid("01A660D9-8009-4330-BA8971B076CD5D0A", name="EFI_SPI_PROTOCOL_GUID"),
    UefiGuid("C9DCF469-A7C4-11D5-87DA00062945C3B9", name="EFI_STANDARD_CALLER_ID_GUID"),
    UefiGuid(
        "D3B36F2D-D551-11D4-9A460090273FC14D", name="EFI_STANDARD_ERROR_DEVICE_GUID"
    ),
    UefiGuid(
        "D98E3EA3-6F39-4BE4-82CE5A890CCB2C95", name="EFI_STATUS_CODE_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "DA571595-4D99-487C-827C2622677D3307",
        name="EFI_STATUS_CODE_DATA_TYPE_ASSERT_GUID",
    ),
    UefiGuid(
        "9A4E9246-D553-11D5-87E200062945C3B9",
        name="EFI_STATUS_CODE_DATA_TYPE_DEBUG_GUID",
    ),
    UefiGuid(
        "AB359CE3-99B3-AE18-C89D95D3B072E19B",
        name="EFI_STATUS_CODE_DATA_TYPE_ERROR_GUID",
    ),
    UefiGuid(
        "3BC2BD12-AD2E-11D5-87DD00062945C3B9",
        name="EFI_STATUS_CODE_DATA_TYPE_EXCEPTION_HANDLER_GUID",
    ),
    UefiGuid(
        "A356AB39-35C4-35DA-B37AF8EA9E8B36A3",
        name="EFI_STATUS_CODE_DATA_TYPE_PROGRESS_CODE_GUID",
    ),
    UefiGuid(
        "92D11080-496F-4D95-BE7E037488382B0A",
        name="EFI_STATUS_CODE_DATA_TYPE_STRING_GUID",
    ),
    UefiGuid("D083E94C-6560-42E4-B6D42DF75ADF6A2A", name="EFI_STATUS_CODE_GUID"),
    UefiGuid(
        "335984BD-E805-409A-B8F8D27ECE5FF7A6", name="EFI_STATUS_CODE_SPECIFIC_DATA_GUID"
    ),
    UefiGuid(
        "C88B0B6D-0DFC-49A7-9CB449074B4C3A78",
        name="EFI_STORAGE_SECURITY_COMMAND_PROTOCOL_GUID",
    ),
    UefiGuid(
        "1259F60D-B754-468E-A7894DB85D55E87E",
        name="EFI_SWAP_ADDRESS_RANGE_PROTOCOL_GUID",
    ),
    UefiGuid("FFF12B8D-7696-4C8B-A9852747075B4F50", name="EFI_SYSTEM_NV_DATA_FV_GUID"),
    UefiGuid("D6E5092D-C7B2-4872-AF66FDC0E6F95E78", name="EFI_SYSTEM_NV_DATA_HOB_GUID"),
    UefiGuid("AAB16018-679D-4461-BA20E70CF7866A9B", name="EFI_SYSTEM_TYPE_FRU_GUID"),
    UefiGuid("1E93E633-D65A-459E-AB8493D9EC266D18", name="EFI_TAPE_IO_PROTOCOL_GUID"),
    UefiGuid("5F7D4E0E-3D6D-42BC-A9420E91E83E3C31", name="EFI_TCG_LOG_HOB_GUID"),
    UefiGuid(
        "3CDC90C6-13FB-4A75-9E7959E9DD78B9FA",
        name="EFI_TCG_PEI_READ_ONLY_VARIABLE_PPI_GUID",
    ),
    UefiGuid(
        "8C4C9A41-BF56-4627-9E0AC8386D66115C", name="EFI_TCG_PLATFORM_PROTOCOL_GUID"
    ),
    UefiGuid(
        "8C4C9A41-BF56-4627-9E0AC8386D66115C", name="EFI_TCG_PRIVATE_INTERFACE_GUID"
    ),
    UefiGuid("F541796D-A62E-4954-A7759584F61B9CDD", name="EFI_TCG_PROTOCOL_GUID"),
    UefiGuid(
        "BBB810BB-5EF0-4E8F-B298AD74AA50EF0A", name="EFI_TCG_WAKE_EVENT_DATA_HOB_GUID"
    ),
    UefiGuid("65530BC7-A359-410F-B0105AADC7EC2B62", name="EFI_TCP4_PROTOCOL_GUID"),
    UefiGuid(
        "00720665-67EB-4A99-BAF7D3C33A1C7CC9",
        name="EFI_TCP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("02B3D5F2-AC28-11D3-9A2D0090273FC14D", name="EFI_TCP_PROTOCOL_GUID"),
    UefiGuid("5187359B-790D-425B-A593CA1CDB3CEBAD", name="EFI_TCP_SOCK_PROTOCOL_GUID"),
    UefiGuid("0BF70067-D53B-42DF-B770E92C91C61411", name="EFI_TDT_PROTOCOL_GUID"),
    UefiGuid(
        "6D3569D4-85E5-4943-AE46EE67A6E1AB5A", name="EFI_TELNET_SERVER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E84CF29C-191F-4EAE-96E1F46AECEAEA0B", name="EFI_TIANO_DECOMPRESS_PROTOCOL_GUID"
    ),
    UefiGuid(
        "26BACCB3-6F42-11D4-BCE70080C73C8881", name="EFI_TIMER_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid("9D0DA369-540B-46F8-85A02B5F2C301E15", name="EFI_TIME_VARIABLE_GUID"),
    UefiGuid(
        "DE161CFE-1E60-42A1-8CC3EE7EF0735212", name="EFI_TPM_DEVICE_PROTOCOL_GUID"
    ),
    UefiGuid("A0C6D918-8DEE-41BB-9D926753A572B652", name="EFI_TPM_STATE_INIT_HUB_GUID"),
    UefiGuid("601211DC-0B12-4C09-AE27AB0EC9692468", name="EFI_TXT_INFO_PROTOCOL_GUID"),
    UefiGuid("3AD9DF29-4501-478D-B1F87F7FE70E50F3", name="EFI_UDP4_PROTOCOL_GUID"),
    UefiGuid(
        "83F01464-99BD-45E5-B383AF6305D8E9E6",
        name="EFI_UDP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("C56FB1B2-017B-4984-A3F63F734FFA9E33", name="EFI_UDP_PROTOCOL_GUID"),
    UefiGuid("982C298B-F4FA-41CB-B83877AA688FB839", name="EFI_UGA_DRAW_PROTOCOL_GUID"),
    UefiGuid("61A4D49E-6F68-4F1B-B922A86EED0B07A2", name="EFI_UGA_IO_PROTOCOL_GUID"),
    UefiGuid(
        "A45B3A0D-2E55-4C03-AD9C27D4820B507E", name="EFI_UGA_SPLASH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "A4C751FC-23AE-4C3E-92E94964CF63F349",
        name="EFI_UNICODE_COLLATION2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A4C751FC-23AE-4C3E-92E94964CF63F349",
        name="EFI_UNICODE_COLLATION_PROTOCOL2_GUID",
    ),
    UefiGuid(
        "1D85CD7F-F43D-11D2-9A0C0090273FC14D",
        name="EFI_UNICODE_COLLATION_PROTOCOL_GUID",
    ),
    UefiGuid("3E745226-9818-45B6-A2ACD7CD0E8BA2BC", name="EFI_USB2_HC_PROTOCOL_GUID"),
    UefiGuid("2B2F68DA-0CD2-44CF-8E8BBBA20B1B5B75", name="EFI_USB_ATAPI_PROTOCOL_GUID"),
    UefiGuid("F5089266-1AA0-4953-97D8562F8A73B519", name="EFI_USB_HC_PROTOCOL_GUID"),
    UefiGuid("2B2F68D6-0CD2-44CF-8E8BBBA20B1B5B75", name="EFI_USB_IO_PROTOCOL_GUID"),
    UefiGuid(
        "5859CB76-6BEF-468A-BE2DB3DD1A27F012", name="EFI_USB_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid("2AD8E2D2-2E91-4CD1-95F5E78FE5EBE316", name="EFI_USB_PROTOCOL_GUID"),
    UefiGuid(
        "71EE5E94-65B9-45D5-821A3A4D86CFE6BE", name="EFI_USER_CREDENTIAL_PROTOCOL_GUID"
    ),
    UefiGuid(
        "85B75607-F7CE-471E-B7E42AEA5F7232EE",
        name="EFI_USER_INFO_ACCESS_SETUP_ADMIN_GUID",
    ),
    UefiGuid(
        "1DB29AE0-9DCB-43BC-8D875DA14964DDE2",
        name="EFI_USER_INFO_ACCESS_SETUP_NORMAL_GUID",
    ),
    UefiGuid(
        "BDB38125-4D63-49F4-821261CF5A190AF8",
        name="EFI_USER_INFO_ACCESS_SETUP_RESTRICTED_GUID",
    ),
    UefiGuid(
        "6FD5B00C-D426-4283-98876CF5CF1CB1FE", name="EFI_USER_MANAGER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "1E5668E2-8481-11D4-BCF10080C73C8881", name="EFI_VARIABLE_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "F088CD91-A046-11D2-8E4200A0C969723B", name="EFI_VARIABLE_STORE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "6441F818-6362-4E44-B5707DBA31DD2453",
        name="EFI_VARIABLE_WRITE_ARCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "C7735A2F-88F5-4882-AE63FAAC8C8B86B3", name="EFI_VGA_MINI_PORT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "745D377A-B988-47B2-B18FBBC80DC56698",
        name="EFI_VIRTUAL_MEMORY_ACCESS_PROTOCOL_GUID",
    ),
    UefiGuid("DFA66065-B419-11D3-9A2D0090273FC14D", name="EFI_VT_100_GUID"),
    UefiGuid("7BAEC70B-57E0-4C76-8E872F9E28088343", name="EFI_VT_100_PLUS_GUID"),
    UefiGuid("AD15A0D6-8BEC-4ACF-A073D01DE77E2D88", name="EFI_VT_UTF8_GUID"),
    UefiGuid(
        "665E3FF5-46CC-11D4-9A380090273FC14D",
        name="EFI_WATCHDOG_TIMER_ARCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "96EB4AD6-A32A-11D4-BCFD0080C73C8881",
        name="EFI_WIN_NT_BUS_DRIVER_IO_PROTOCOL_GUID",
    ),
    UefiGuid("0C95A93D-A006-11D4-BCFA0080C73C8881", name="EFI_WIN_NT_SERIAL_PORT_GUID"),
    UefiGuid(
        "58C518B1-76F3-11D4-BCEA0080C73C8881", name="EFI_WIN_NT_THUNK_PROTOCOL_GUID"
    ),
    UefiGuid("ADDEBF82-A560-46B9-A28078C6AB61AEDA", name="ERROR_MANAGER_GUID"),
    UefiGuid("A43B03DC-C18A-41B1-91C83FF9AAA25713", name="EXIT_FORM_SET_GUID"),
    UefiGuid("EC87D643-EBA4-4BB5-A1E53F3E36B20DA9", name="EXIT_PAGE_GUID"),
    UefiGuid("0FBA43B8-ACDB-4EE2-AB310FA4D5B6C3C5", name="FAST_BOOT_TSE_PROTOCOL_GUID"),
    UefiGuid("B540A530-6978-4DA7-91CB7207D764D262", name="FAST_BOOT_VARIABLE_GUID"),
    UefiGuid("2EBE0275-6458-4AF9-91EDD3F4EDB100AA", name="FID_FFS_SECTION_GUID"),
    UefiGuid("3FD1D3A2-99F7-420B-BC698BB1D492A332", name="FID_TABLE_GUID"),
    UefiGuid("2EBE0275-6458-4AF9-91EDD3F4EDB100AA", name="FID_TABLE_SECTION_GUID"),
    UefiGuid(
        "BC412D75-2729-4C3A-B1935B9A588FF66F", name="FIRMWARE_PERFORMANCE_PROTOCOL_GUID"
    ),
    UefiGuid("755B6596-6896-4BA3-B3DD1C629FD1EA88", name="FLASH_PROTOCOL_GUID"),
    UefiGuid("ECB867AB-8DF4-492D-8150A7FD1B9B5A75", name="FLASH_SMM_PROTOCOL_GUID"),
    UefiGuid("974231D5-ED4B-44D1-8870CE515CC14D68", name="FLASH_UPDATE_GUID"),
    UefiGuid("DAC2B117-B5FB-4964-A3120DCC77061B9B", name="FONT_FFS_FILE_GUID"),
    UefiGuid(
        "444C3203-F8B1-42A7-ABE92E58025BE12A", name="FPDT_PERFORMANCE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "DE28BC59-6228-41BD-BDF6A3B9ADB58DA1",
        name="FRAMEWORK_EFI_FIRMWARE_VOLUME_BLOCK_PROTOCOL_GUID",
    ),
    UefiGuid("B5C59087-FEAC-4B41-9D80790BA5AA070F", name="FW_VERSION_GUID"),
    UefiGuid(
        "DE28BC59-6228-41BD-BDF6A3B9ADB58DA1", name="FW_VOLUME_BLOCK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "AD77AE29-4C20-4FDD-85048176619B676A", name="HDD_SECURITY_END_PROTOCOL_GUID"
    ),
    UefiGuid(
        "CE6F86BB-B800-4C71-B2D13897A3BC1DAE", name="HDD_SECURITY_INIT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "9401BD4F-1A00-4990-AB56DAF0E4E348DE", name="HDD_SMART_INIT_PROTOCOL_GUID"
    ),
    UefiGuid("1FD29BE6-70D0-42A4-A6E7E5D10E6AC376", name="HDD_UNLOCKED_GUID"),
    UefiGuid("CFB33810-6E87-4284-B203A66ABE07F6E8", name="HECI_PROTOCOL_GUID"),
    UefiGuid(
        "97E409E6-4CC1-11D9-81F6000000000000", name="HII_RESOURCES_FFS_SECTION_GUID"
    ),
    UefiGuid("220AC432-1D43-49E5-A74F4C9DA67AD23B", name="HOT_PLUG_DEVICE_GUID"),
    UefiGuid(
        "8E8CBC58-834C-41E3-B8CAF00CCF5A717C", name="ICC_OVERCLOCKING_PROTOCOL_GUID"
    ),
    UefiGuid("64192DCA-D034-49D2-A6DE65A829EB4C74", name="ICC_PERSISTENT_DATA_GUID"),
    UefiGuid(
        "7B77FB8B-1E0D-4D7E-953F3980A261E077", name="ICC_VOLATILE_SETUP_DATA_GUID"
    ),
    UefiGuid(
        "8AA67071-8BEC-47AB-83C8CD0EB723D072",
        name="IDE_BUS_DRIVER_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("E159A956-3299-4EE9-917665181A4E5E9F", name="IDE_BUS_INIT_PROTOCOL_GUID"),
    UefiGuid(
        "20E28787-DF32-4BDA-B7E7CBBDA3371EF8", name="IDE_CONTROLLER_PROTOCOL_GUID"
    ),
    UefiGuid("51AA65FC-82B6-49E6-95E2E6827A8D7DB4", name="IDE_HPA_INTERFACE_GUID"),
    UefiGuid(
        "67BC3883-7E79-4BC1-A33E3AF7D17589BA", name="IDE_POWER_MGMT_INTERFACE_GUID"
    ),
    UefiGuid("F4F63529-281E-4040-A313C1D6766384BE", name="IDE_SECURITY_INTERFACE_GUID"),
    UefiGuid("69967A8C-1159-4522-AA8974CDC6E599A0", name="IDE_SECURITY_PWNV_GUID"),
    UefiGuid("5578AE16-F1C9-4E8F-B129BA07F8FCF84A", name="IDE_SETUP_PROTOCOL_GUID"),
    UefiGuid("FFBD9AD2-F1DB-4F92-A649EB9EEDEA86B5", name="IDE_SMART_INTERFACE_GUID"),
    UefiGuid(
        "60AC3A8F-4D66-4CD4-895AC3F06E6665EE", name="IFFS_ACPI_TABLE_STORAGE_GUID"
    ),
    UefiGuid(
        "A5559F06-6415-4759-8869DE15F9CD9C9B", name="IFFS_GLOBAL_NVS_AREA_PROTOCOL_GUID"
    ),
    UefiGuid("D3BFE2DE-3DAF-11DF-BA40E3A556D89593", name="IFFS_GPT_GUID"),
    UefiGuid("D231DB5B-4A9C-4092-A8C59CA0BC7D6AA6", name="IFFS_INFO_PROTOCOL_GUID"),
    UefiGuid(
        "65639144-D492-4328-A498F4B5545E4A30",
        name="IFFS_PARTITION_STATUS_PROTOCOL_GUID",
    ),
    UefiGuid("F9F0B131-F346-4F16-80DDF941072B3A7D", name="IFFS_PERSISTENT_DATA_GUID"),
    UefiGuid(
        "42BBACA3-7161-4891-AC10C75E2E4DF614", name="IFFS_PLATFORM_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid("3D0E663A-DC72-4489-87C5E49EE773A452", name="IFFS_PPI_GUID"),
    UefiGuid("CDC5DDDF-E79D-41EC-A9B06565490DB9D3", name="IGD_OPREGION_PROTOCOL_GUID"),
    UefiGuid("01AB1829-CECD-4CFA-A18CEA75D66F3E74", name="INTEL_MEBX_PROTOCOL_GUID"),
    UefiGuid("A56074DB-65FE-45F7-BD212D2BDD8E9652", name="LEGACY_DEV_ORDER_GUID"),
    UefiGuid("56EC3091-954C-11D2-8E3F00A0C969723B", name="LOAD_FILE_PROTOCOL_GUID"),
    UefiGuid("985EEE91-BCAC-4238-877857EFDC93F24E", name="MAIN_FORM_SET_GUID"),
    UefiGuid(
        "E5D0875A-F647-4E16-BE4D95024029CC44", name="MDES_STATUS_CODE_PROTOCOL_GUID"
    ),
    UefiGuid("7C81C66A-4F11-47AB-82D367C4D635AED1", name="MEBX_FILE_GUID"),
    UefiGuid(
        "E20939BE-32D4-41BE-A150897F85D49829", name="MEMORY_ONLY_RESET_CONTROL_GUID"
    ),
    UefiGuid("6F20F7C8-E5EF-4F21-8D19EDC5F0C496AE", name="MEM_INFO_PROTOCOL_GUID"),
    UefiGuid("0732BD39-D6B0-4039-B6C29654466DE525", name="MEUD_ERROR_GUID"),
    UefiGuid("FEAAA7A6-CB95-4670-B499877FA6CA6BAE", name="MEUD_FILE_GUID"),
    UefiGuid("B441DF87-8D94-4811-85F70F9A7BF89D2A", name="ME_ALERT_AT_HANDLER_GUID"),
    UefiGuid(
        "71A19494-2AB6-4E96-8581CF34254273FE", name="ME_BIOS_PAYLOAD_DATA_PROTOCOL_GUID"
    ),
    UefiGuid("78259433-7B6D-4DB3-9AE836C4C2C3A17D", name="ME_INFO_SETUP_GUID"),
    UefiGuid(
        "B8CDCED7-BDC4-4464-9A1AFF3FBDF74869", name="ME_PLATFORM_GET_RESET_TYPE_GUID"
    ),
    UefiGuid("9B6EECF6-FBDC-4DB6-9562D25F40D57661", name="MICROCODE_LOADER_PPI_GUID"),
    UefiGuid(
        "97E409E6-4CC1-11D9-81F6000000000000", name="MINI_SETUP_RESOURCE_SECTION_GUID"
    ),
    UefiGuid("2EF1BA1A-C836-4A50-BF89525FF29FF787", name="MXM30_PEI_GUID"),
    UefiGuid("4EA9D4FE-E6F6-410B-90370F98B5968B65", name="MXM3_EFI_GUID"),
    UefiGuid("6707536E-46AF-42D3-8F6C15F2F202C234", name="MXMOEM1_GUID"),
    UefiGuid("4EA9D4FE-E6F6-410B-80370F98B5968B65", name="NBCI_EFI_GUID"),
    UefiGuid("D1405D16-7AFC-4695-BB1241459D3695A2", name="NETWORK_STACK_GUID"),
    UefiGuid("C0EC00FD-C2F8-4E47-90EF9C8155285BEC", name="NVRAM_HOB_GUID"),
    UefiGuid(
        "54913A6D-F4EE-4CDB-847574062BFCECF5",
        name="NVRAM_MAILBOX_ADDRESS_VARIABLE_GUID",
    ),
    UefiGuid("595A6EDC-6D2C-474A-90823B992851DFFE", name="OEM_RECOVERY_CAPSULE_GUID"),
    UefiGuid("05CA01FC-0FC1-11DC-901100173153EBA8", name="OEM_ROM_HOLE_0_GUID"),
    UefiGuid("05CA01FD-0FC1-11DC-901100173153EBA8", name="OEM_ROM_HOLE_1_GUID"),
    UefiGuid("F4B2C007-94A1-4CD5-A710F4141FCEBCA0", name="OEM_TSE_VAR_GUID"),
    UefiGuid("5D206DD3-516A-47DC-A1BC6DA204AABE08", name="ONBOARD_RAID_GUID"),
    UefiGuid("59AF16B0-661D-4865-A38138DE68385D8D", name="OPAL_SEC_INIT_PROTOCOL_GUID"),
    UefiGuid(
        "F2A128FF-257B-456E-9DE863E7C7DCDFAC", name="OPROM_START_END_PROTOCOL_GUID"
    ),
    UefiGuid("8DB699CC-BC81-41E2-AAC6D81D5300D759", name="PARTITION_VARIABLE_GUID"),
    UefiGuid("79CE097A-91AA-41FF-B3A8534559B30DB1", name="PASSWORD_HOB_GUID"),
    UefiGuid("ED097352-9041-445A-80B6B29D509E8845", name="PCH_DMI_TC_VC_PPI_GUID"),
    UefiGuid(
        "99D5757C-D906-11E0-8D788DE44824019B", name="PCH_EFI_RAID_DRIVER_EXECUTION_GUID"
    ),
    UefiGuid("E8C7CE14-1EED-48FD-83478E554110C0D9", name="PCH_INIT_PPI_GUID"),
    UefiGuid("E6C2F70A-B604-4877-85BADEEC89E117EB", name="PCH_INIT_VARIABLE_GUID"),
    UefiGuid("8C376010-2400-4D7D-B47B9D851DF3C9D1", name="PCH_ME_UMA_PPI_GUID"),
    UefiGuid(
        "22074E71-BCCC-4517-87577995EDFD8032", name="PCH_PLATFORM_POLICY_PPI_GUID"
    ),
    UefiGuid(
        "3A3300AB-C929-487D-AB34159BC13562C0", name="PCH_RESET_CALLBACK_PROTOCOL_GUID"
    ),
    UefiGuid("DB63592C-B8CC-44C8-918C51F534598A5A", name="PCH_RESET_PROTOCOL_GUID"),
    UefiGuid(
        "BB929DA9-68F7-4035-B22CA3BB3F23DA55", name="PCH_SATA_CONTROLLER_DRIVER_GUID"
    ),
    UefiGuid("C02B0573-2B4E-4A31-A31A94567B50442C", name="PCH_USB_POLICY_PPI_GUID"),
    UefiGuid(
        "B4A1208E-4D9A-4EA2-9D6BE41A61E6C5AC", name="PEI_AMT_PLATFORM_POLICY_PPI_GUID"
    ),
    UefiGuid(
        "881807D2-98D1-4EC9-AFA07746C42F2449", name="PEI_AMT_STATUS_CODE_PPI_GUID"
    ),
    UefiGuid("D1E59F50-E8C3-4545-BF6111F002233C97", name="PEI_AP_STARTUP_FILE_GUID"),
    UefiGuid("A45E60D1-C719-44AA-B07AAA777F85906D", name="PEI_ATA_CONTROLLER_PPI_GUID"),
    UefiGuid("1B8DDEA4-DEB2-4152-91C2B73CB16CE464", name="PEI_ATA_POLICY_PPI_GUID"),
    UefiGuid("B6EC423C-21D2-490D-85C6DD5864EAA674", name="PEI_BASE_MEMORY_TEST_GUID"),
    UefiGuid("2D27C618-7DCD-41F5-BB1021166BE7E143", name="PEI_BIOS_ACM_FILE_GUID"),
    UefiGuid("695D8AA1-42EE-4C46-805C6EA6BCE799E3", name="PEI_BLOCK_IO_PPI_GUID"),
    UefiGuid(
        "FD7C4665-34BE-426B-B1F83AB753CE44B0", name="PEI_BOOT_SCRIPT_DONE_PPI_GUID"
    ),
    UefiGuid(
        "ABD42895-78CF-4872-84441B5C180BFBFF", name="PEI_BOOT_SCRIPT_EXECUTER_PPI_GUID"
    ),
    UefiGuid("C153205A-E898-4C24-8689A4B4BCC5C8A2", name="PEI_CACHE_PPI_GUID"),
    UefiGuid("3ACF33EE-D892-40F4-A2FC3854D2E1323D", name="PEI_CAPSULE_PPI_GUID"),
    UefiGuid("E6AF1F7B-FC3F-46DA-A828A3B457A44282", name="PEI_CPU_IO_PPI_GUID"),
    UefiGuid(
        "7B8EE7A1-4E35-4556-BB566797E24445C2", name="PEI_CPU_PLATFORM_POLICY_PPI_GUID"
    ),
    UefiGuid(
        "605EA650-C65C-42E1-BA8091A52AB618C6", name="PEI_END_OF_PEI_PHASE_PPI_GUID"
    ),
    UefiGuid("F34C2FA0-DE88-4270-8414961222F4521C", name="PEI_FLASH_MAP_PPI_GUID"),
    UefiGuid("EE0EA811-FBD9-4777-B95ABA4F71101F74", name="PEI_HECI_PPI_GUID"),
    UefiGuid(
        "7E13637A-C3F8-43D1-B051ED19D708EC7A",
        name="PEI_IDE_RECOVERY_NATIVE_MODE_PPI_GUID",
    ),
    UefiGuid(
        "DE8F2878-36D5-498E-BA59168C2647B335", name="PEI_IFFS_TRANSITION_START_PPI_GUID"
    ),
    UefiGuid("643B8786-B417-48D2-8F5E7819931CAED8", name="PEI_IN_MEMORY_GUID"),
    UefiGuid(
        "7AE3CEB7-2EE2-48FA-AA493510BC83CABF", name="PEI_ME_PLATFORM_POLICY_PPI_GUID"
    ),
    UefiGuid("E1F2EBA0-F7B9-4A26-8620131221642A90", name="PEI_PCI_CFG_PPI_GUID"),
    UefiGuid(
        "F894643D-C449-42D1-8EA885BDD8C65BDE",
        name="PEI_PERMANENT_MEMORY_INSTALLED_PPI_GUID",
    ),
    UefiGuid(
        "30EB2979-B0F7-4D60-B2DC1A2C96CEB1F4", name="PEI_PLATFORM_MEMORY_RANGE_PPI_GUID"
    ),
    UefiGuid(
        "9A7EF41E-C140-4BD1-B8841E11240B4CE6", name="PEI_PLATFORM_MEMORY_SIZE_PPI_GUID"
    ),
    UefiGuid(
        "3CDC90C6-13FB-4A75-9E7959E9DD78B9FA",
        name="PEI_READ_ONLY_VARIABLE_ACCESS_PPI_GUID",
    ),
    UefiGuid("EF398D58-9DFD-4103-BF9478C6F4FE712F", name="PEI_RESET_PPI_GUID"),
    UefiGuid("4426CCB2-E684-4A8A-AE4020D4B025B710", name="PEI_S3_RESUME_PPI_GUID"),
    UefiGuid("1388066E-3A57-4EFA-98F3C12F3A958A29", name="PEI_SECURITY_PPI_GUID"),
    UefiGuid("9CA93627-B65B-4324-A202C0B461764543", name="PEI_SMBUS2_PPI_GUID"),
    UefiGuid("63B6E435-32BC-49C6-81BDB7A1A0FE1A6C", name="PEI_SMBUS_POLICY_PPI_GUID"),
    UefiGuid("ABD42895-78CF-4872-84441B5C180BFBDA", name="PEI_SMBUS_PPI_GUID"),
    UefiGuid("268F33A9-CCCD-48BE-881786053AC32ED6", name="PEI_SMM_ACCESS_PPI_GUID"),
    UefiGuid("61C68702-4D7E-4F43-8DEFA74305CE74C5", name="PEI_SMM_CONTROL_PPI_GUID"),
    UefiGuid("64E42E92-E958-4E99-90E2F7FD90A76C59", name="PEI_SPI_PPI_GUID"),
    UefiGuid("1F4C6F90-B06B-48D8-A201BAE5F1CD7D56", name="PEI_STALL_PPI_GUID"),
    UefiGuid(
        "26F8AB01-D3CD-489C-984FDFDEF768395B", name="PEI_STATUS_CODE_MEMORY_PPI_GUID"
    ),
    UefiGuid("229832D3-7A30-4B36-B827F40CB7D45436", name="PEI_STATUS_CODE_PPI_GUID"),
    UefiGuid("70FFF0FF-A543-45B9-8BE31BDB90412080", name="PEI_TCG_INTERNAL_FLAGS_GUID"),
    UefiGuid("177D39D2-43B8-40C8-9AE13C5198D6941E", name="PEI_TCG_PPI_GUID"),
    UefiGuid(
        "E9DB0D58-D48D-47F6-9C6E6F40E86C7B41", name="PEI_TPM_INITIALIZED_PPI_GUID"
    ),
    UefiGuid("CA4853F4-E94B-42B4-8642CDE28A7FAC2D", name="PEI_TPM_PPI_2_GUID"),
    UefiGuid("E9DB0D58-D48D-47F6-9C6E6F40E86C7B41", name="PEI_TPM_PPI_GUID"),
    UefiGuid(
        "38CDD10B-767D-4F6E-A74467EE1DFE2FA5", name="PEI_TXT_MEMORY_UNLOCKED_PPI_GUID"
    ),
    UefiGuid("3BC1F6DE-693E-4547-A30021823CA420B2", name="PEI_USB_CONTROLLER_PPI_GUID"),
    UefiGuid("C49189F3-1D4C-4AD7-A439D013AB720931", name="PERF_TUNE_PPI_GUID"),
    UefiGuid("DD6C613A-5A77-4B4F-A61E3BDD2AE21D81", name="PERF_TUNE_PROTOCOL_GUID"),
    UefiGuid("A7C88FAE-EBEC-45ED-A7C55FA755177306", name="PERF_TUNE_WDT_PPI_GUID"),
    UefiGuid("C0557EED-9A89-4770-9626FCA051F2BA09", name="PERF_TUNE_WDT_PROTOCOL_GUID"),
    UefiGuid("A6C0E11E-929E-42B3-90CC4F778E03FF57", name="PKPUB_KEY_GUID"),
    UefiGuid(
        "9E625A27-4840-47CC-A6B51E9311CFC60E", name="PK_PUB_FFS_FILE_EFI_AUTH_VAR_GUID"
    ),
    UefiGuid("6737F69B-B8CC-45BC-9327CCF5EEF70CDE", name="PLATFORM_IDE_PROTOCOL_GUID"),
    UefiGuid("E806424F-D425-4B1A-BC265F690389A15A", name="PLATFORM_ME_HOOK_PPI_GUID"),
    UefiGuid(
        "BC52476E-F67E-4301-B262369C4878AAC2", name="PLATFORM_ME_HOOK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "299141BB-211A-48A5-92C06F9A0A3A006E",
        name="POWER_MANAGEMENT_ACPI_TABLE_STORAGE_GUID",
    ),
    UefiGuid(
        "6F27E990-7BC8-42C9-8F719980063693E9", name="PPM_PLATFORM_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid(
        "A60C7DCD-512F-4F02-B180522E015E06B7",
        name="PPM_PROCESSOR_SUPPORT_PROTOCOL_3_GUID",
    ),
    UefiGuid("3FEEC852-F14C-4E7F-97FD4C3A8C5BBECC", name="PR_KEY_FFS_FILE_RAW_GUID"),
    UefiGuid("4E0F9BD4-E338-4B26-843EBD3AD9B2837B", name="PR_KEY_GUID"),
    UefiGuid("95DFCAE5-BB28-4D6B-B1E23AF3A6BF434F", name="PTID_FFS_TABLE_STORAGE_GUID"),
    UefiGuid("80E1202E-2697-4264-9CC980762C3E5863", name="RECOVERY_FORM_SET_GUID"),
    UefiGuid("36E835BB-661D-4D37-8DE5885325DAE910", name="ROM_CACHE_ENABLE_PPI_GUID"),
    UefiGuid("DDE1BC72-D45E-4209-AB8514462D2F5074", name="ROM_IMAGE_ADDRESS_GUID"),
    UefiGuid("EE2F45D2-5BA4-441E-8A1DAA22DFA3B6C5", name="ROM_IMAGE_MEMORY_HOB_GUID"),
    UefiGuid("0DCA793A-EA96-42D8-BD7BDC7F684E38C1", name="ROM_LAYOUT_FFS_GUID"),
    UefiGuid("88A15A4F-977D-4682-B17CDA1F316C1F32", name="ROM_LAYOUT_SECTION_GUID"),
    UefiGuid("A33319B5-8EE1-45E0-8C9F809F5B0902CC", name="RSDP_PLUS_PROTOCOL_GUID"),
    UefiGuid(
        "2ACB6627-DF02-4E23-B4F96A93FA6E9DA6", name="SATA_CONTROLLER_PROTOCOL_GUID"
    ),
    UefiGuid("5F5D3502-8A4B-40CA-88A22305427A131A", name="SA_PEG_DATA_HOB_GUID"),
    UefiGuid("C4975200-64F1-4FB6-9773F6A9F89D985E", name="SA_PEG_DATA_VARIABLE_GUID"),
    UefiGuid("09EA8911-BE0D-4230-A003EDC693B48E11", name="SA_PEI_INIT_PPI_GUID"),
    UefiGuid("150CE416-EE63-46B6-8BA37322BBE04637", name="SA_PLATFORM_POLICY_PPI_GUID"),
    UefiGuid("981CEAEE-931C-4A17-B9C866C7BCFD77E1", name="SECURITY_FORM_SET_GUID"),
    UefiGuid(
        "699ADD70-8554-4993-83F6D2CDC081DD85", name="SERIAL_RECOVERY_CAPSULE_GUID"
    ),
    UefiGuid("9221315B-30BB-46B5-813E1B1BF4712BD3", name="SETUP_DEFAULTS_FFS_GUID"),
    UefiGuid("EC87D643-EBA4-4BB5-A1E53F3E36B20DA9", name="SETUP_GUID"),
    UefiGuid("5B232086-350A-42C7-A70E3497B5765D85", name="SGOEM_ACPI_SSDT_GUID"),
    UefiGuid("6A061113-FE54-4A07-A28E0A69359EB069", name="SGTPV_ACPI_SSDT_GUID"),
    UefiGuid("CACB3817-81E6-497E-87FFC8FA8F24EC28", name="SG_ACPI_TABLE_STORAGE_GUID"),
    UefiGuid("648CE07B-AE5D-4973-BD3C8C9153C05DC5", name="SG_INFO_HOB_GUID"),
    UefiGuid("AA40D00E-5B9B-4DAD-A487FC1BCBEF8170", name="SG_PEI_INIT_PPI_GUID"),
    UefiGuid("A391E822-7044-4CF4-BE5B345E44B8627C", name="SG_PLATFORM_POLICY_PPI_GUID"),
    UefiGuid("5820DE98-FC8E-4B0B-A4B90A940D162A7E", name="SIO_DEV_STATUS_VAR_GUID"),
    UefiGuid("964E5B21-6000-23D2-9E3901A0C969723B", name="SIO_IO_PROTOCOL_GUID"),
    UefiGuid("FD44820B-F1AB-41C0-AE4E0C55556EB9BD", name="SMBIOS_FLASH_DATA_FFS_GUID"),
    UefiGuid("950C3A26-E0C2-491C-B6B20374F5C73B96", name="SMM_CHILD_DISPATCHER2_GUID"),
    UefiGuid("F328E36C-23B6-4A95-854B32E19534CD75", name="SMM_COMMUNICATE_HEADER_GUID"),
    UefiGuid("FC9A50C1-8B3D-40D0-99126E26D7896CBA", name="SMM_HECI_PROTOCOL_GUID"),
    UefiGuid("5848FD2D-D6AF-474B-827595DDE70AE823", name="SMRAM_CPU_DATA_HEADER_GUID"),
    UefiGuid(
        "429501D9-E447-40F4-867B75C93A1DB54E", name="SMRAM_CPU_DATA_VARIABLE_GUID"
    ),
    UefiGuid(
        "882F2546-EF1F-4090-9F9C93845AD7841C", name="SM_BUS_CONTROLLER_IDENTIFIER_GUID"
    ),
    UefiGuid("E770BB69-BCB4-4D04-9E9723FF9456FEAC", name="SYSTEM_ACCESS_GUID"),
    UefiGuid("EB9D2D30-2D88-11D3-9A160090273FC14D", name="TCG_EFI_ACPI_TABLE_GUID"),
    UefiGuid("7739F24C-93D7-11D4-9A3A0090273FC14D", name="TCG_EFI_HOB_LIST_GUID"),
    UefiGuid("70FFF0FF-A543-45B9-8BE31BDB90412080", name="TCG_INTERNAL_FLAGS_GUID"),
    UefiGuid("6E605536-A30A-4D56-939E1C373F798D7B", name="TCG_LOCK_DOWN_VAR_GUID"),
    UefiGuid(
        "A76B4E22-B50A-401D-8B355124B0BA4104", name="TCG_PLATFORM_SETUP_PEI_POLICY_GUID"
    ),
    UefiGuid(
        "BB6CBEFF-E072-40D2-A6EBBAB75BDE87E7", name="TCG_PLATFORM_SETUP_POLICY_GUID"
    ),
    UefiGuid("F3ED95DF-828E-41C7-BCA016C41965A634", name="TCG_PPI_SYNC_FLAG_GUID"),
    UefiGuid("FA8F55E8-AB22-42DD-B9167DCE39002574", name="TDTHI_PROTOCOL_FIXED_GUID"),
    UefiGuid("3C4852D6-D47B-4F46-B05EB5EDC1AA430A", name="TDTHI_PROTOCOL_GUID"),
    UefiGuid(
        "7B77FB8B-1E0D-4D7E-953F3980A261E076", name="TDT_VOLATILE_SETUP_DATA_GUID"
    ),
    UefiGuid(
        "DBE23AA9-A345-4B97-85B6B226F1617389", name="TEMPORARY_RAM_SUPPORT_PPI_GUID"
    ),
    UefiGuid("560BF58A-1E0D-4D7E-953F2980A261E031", name="TERMINAL_VAR_GUID"),
    UefiGuid("B6460A49-0AC1-484F-AE58F16EB239DB3D", name="TPM_BOOT_FLOW_VARIABLE_GUID"),
    UefiGuid("2986883F-88E0-48D0-4B8220C26948DDAC", name="TXT_INFO_HOB_GUID"),
    UefiGuid("3D989471-CFAC-46B7-9B1C08430109402D", name="TXT_ONE_TOUCH_GUID"),
    UefiGuid(
        "FA2338AD-80DF-49D0-9396CF7145D03A76", name="TXT_ONE_TOUCH_OP_PROTOCOL_GUID"
    ),
    UefiGuid("CF31FAC5-C24E-11D2-85F300A0C93EC93B", name="UNKNOWN_DEVICE_GUID"),
    UefiGuid(
        "89E3C1DC-B5E3-4D34-AEADDD7EB2828C18", name="USB_TIMING_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid("C4CC0DE8-0687-4422-99C165351A5D5F95", name="USER_DEFAULTS_GUID"),
    UefiGuid("92C7D0BB-679E-479D-878DD4B82968578B", name="WDT_APP_PROTOCOL_GUID"),
    UefiGuid("65675786-ACCA-4B11-8AB7F843AA2A8BEA", name="WDT_HOB_GUID"),
    UefiGuid("78CE2354-CFBC-4643-AEBA07A27FA892BF", name="WDT_PERSISTENT_DATA_GUID"),
    UefiGuid("F38D1338-AF7A-4FB6-91DB1A9C2183570D", name="WDT_PPI_GUID"),
    UefiGuid("B42B8D12-2ACB-499A-A920DD5BE6CF09B1", name="WDT_PROTOCOL_GUID"),
    UefiGuid("041FD986-B016-47CE-AFC52269EDD4AD64", name="ASROCK_ACPIS4_DXE_GUID"),
    UefiGuid("04EAAAA1-29A1-11D7-883800500473D4EB", name="ASROCK_USBRT_GUID"),
    UefiGuid("0916E322-3740-31CE-AD62BD172CECCA36", name="ASROCK_RAID_SETUP_GUID"),
    UefiGuid("09CE2C2D-4D83-44B8-97D01085EB5490B8", name="ASROCK_RAID_LOADER_GUID"),
    UefiGuid("0C3798AA-D22E-4DD6-8CB43C9CFB1D86D3", name="ASROCK_SIOSLPSMI_GUID"),
    UefiGuid("0F886E75-3029-4ECF-B69422894D3F2143", name="ASROCK_PLEDDXE_GUID"),
    UefiGuid("1216BD2A-C05E-4C6D-91E7EB8E37AD3B7A", name="ASROCK_A_DEFAULT_DXE_GUID"),
    UefiGuid(
        "132EC73B-BD26-4FBF-B1938ACB50EFBD89", name="ASROCK_USER_DEF_SETUP_DXE_GUID"
    ),
    UefiGuid("1B6E9D50-12E4-4B55-81D91ABFEC59D4FC", name="ASROCK_WAKEUP_CTRL_SMM_GUID"),
    UefiGuid("1DFB7BFA-BF8E-4D11-97662FB0D0442310", name="ASROCK_AMI_AGESA_DXE_GUID"),
    UefiGuid("2167F964-7298-4A8F-9A2ABFE498D600A8", name="ASROCK_HDD_READY_SMI_GUID"),
    UefiGuid("2ADB8F5B-C7D4-4E7E-BE2A23634DF668A1", name="ASROCK_MOUSE_DRIVER_GUID"),
    UefiGuid("316B1230-0500-4592-8C09EABA0FB6B07F", name="ASROCK_IDESMM_GUID"),
    UefiGuid("3A53114D-5673-4DD9-B5B3CB72C347669B", name="ASROCK_BFGSMI_GUID"),
    UefiGuid("3D9FC54D-19E5-4AD0-B98602D687D760E5", name="ASROCK_ASRLOGODXE_GUID"),
    UefiGuid("405F8FE6-5213-4B85-B82197B77AFF795B", name="ASROCK_ASM104_X_DXE_GUID"),
    UefiGuid("4AC94F68-E651-4B38-A1D82D5DEFFA0F60", name="ASROCK_HDAUDIO_SMI_GUID"),
    UefiGuid("4B680E2D-0D63-4F62-B9307AE995B9B3A3", name="ASROCK_SM_BUS_DXE_GUID"),
    UefiGuid("4C006CD9-19BA-4617-8483609194A1ACFC", name="ASROCK_USBINT13_GUID"),
    UefiGuid("4C494E55-5849-5342-4554544552212121", name="ASROCK_SLP_SUPPORT_GUID"),
    UefiGuid("4F9147CB-972E-4F5C-B869A02182C9D93C", name="ASROCK_PATA_CONTROLLER_GUID"),
    UefiGuid("510DF6A1-B6C1-4948-AEE759F220F898BD", name="ASROCK_SATA_CONTROLLER_GUID"),
    UefiGuid("5199296F-2808-4AFA-94C499F6B06C871E", name="ASROCK_ACPIS4_SMM_GUID"),
    UefiGuid("5446C293-339B-47CD-B719585DE39408CC", name="ASROCK_POST_REPORT_GUID"),
    UefiGuid("5640497F-645E-4611-B915E682C8BD4783", name="ASROCK_CLOCK_GEN_DXE_GUID"),
    UefiGuid("580DD900-385D-11D7-883A00500473D4EB", name="ASROCK_UHCD_GUID"),
    UefiGuid("59242DD8-E7CF-4979-B60EA6067E2A185F", name="ASROCK_LEGACY_REGION_GUID"),
    UefiGuid("6298FE18-D5EF-42B7-BB0C2953283F5704", name="ASROCK_SLEEP_SMI_GUID"),
    UefiGuid(
        "6869C5B3-AC8D-4973-8B37E354DBF34ADD", name="ASROCK_CMOS_MANAGER_SMM_GUID"
    ),
    UefiGuid(
        "6950AFFF-6EE3-11DD-AD8B0800200C9A66", name="ASROCK_AMD_AGESA_DXE_DRIVER_GUID"
    ),
    UefiGuid("70E1A818-0BE1-4449-BFD49EF68C7F02A8", name="ASROCK_RE_FLASH_GUID"),
    UefiGuid(
        "71ED12D1-250B-42FB-8C1710DCFA771701", name="ASROCK_LEGACY_INTERRUPT_GUID"
    ),
    UefiGuid(
        "753630C9-FAE5-47A9-BBBF88D621CD7282", name="ASROCK_SMM_CHILD_DISPATCHER_GUID"
    ),
    UefiGuid("7687A497-FF4E-48FF-BCAA67DBF65C42D1", name="ASROCK_BFGDXE_GUID"),
    UefiGuid("77E5B618-2612-4C28-BCDFA2B14D8A3EFE", name="ASROCK_IFLASHSETUP_GUID"),
    UefiGuid("7BBC4123-5D5E-4C1C-95D130EBA48701CA", name="ASROCK_S4_SLEEPDELAY_GUID"),
    UefiGuid("81E4B34A-268C-4FCC-8E391C1D3E6F6E69", name="ASROCK_HDD_READY_DXE_GUID"),
    UefiGuid("8B0E1353-358C-4E2F-9345EEBF1DAB989B", name="ASROCK_RTLANDXE_GUID"),
    UefiGuid("8B12F29F-7135-4EAB-AFDA0512F74B65EA", name="ASROCK_AMD_SB900_DXE_GUID"),
    UefiGuid(
        "981AD9CD-8685-4AAB-AFDA0512F74B65EA", name="ASROCK_SB900_SMBUS_LIGHT_GUID"
    ),
    UefiGuid("9EF8C276-B3E6-4AD9-8FE7C74F71C47248", name="ASROCK_AAFTBL_SMI_GUID"),
    UefiGuid("A16BA302-6514-4287-BEE36223B7DE2C21", name="ASROCK_NVRAMID_GUID"),
    UefiGuid("A9B700CF-019E-4D8B-A3A788E1EA01699E", name="ASROCK_IDE_SECURITY_GUID"),
    UefiGuid("A9CE66D5-6D37-451E-A9BFF58F7A0B3CC2", name="ASROCK_ASM1061_DXE_GUID"),
    UefiGuid("AD1F487A-BA56-48DC-8EAAE8FBEA74B8F2", name="ASROCK_ASM104_X_SMI_GUID"),
    UefiGuid("B324F29B-5E97-4BF9-B4E05125C1F605D5", name="ASROCK_RTLANSMI_GUID"),
    UefiGuid("B859281C-16FA-45A8-92011C3830A973BD", name="ASROCK_GEC_UPDATE_SMI_GUID"),
    UefiGuid("BB9175E1-3E95-4B7E-87509D6AA34EB7F6", name="ASROCK_APMOFF_GUID"),
    UefiGuid("BC327DBD-B982-4F55-9F79056AD7E987C5", name="ASROCK_SMIFLASH_GUID"),
    UefiGuid("C468B382-4550-4909-AD572496141B3F4A", name="ASROCK_RAID_X64_GUID"),
    UefiGuid("C7C89169-1A42-4E5E-B34BE6830671C9A6", name="ASROCK_AMD_SB900_SMM_GUID"),
    UefiGuid("C8B64E46-9819-45AB-875D09DFDA6D8B1B", name="ASROCK_FIREWIRE_GUID"),
    UefiGuid("D57C852E-809F-45CF-A377D77BC0CB78EE", name="ASROCK_IDE_SMART_GUID"),
    UefiGuid(
        "D7E6ABC1-971B-461B-B5E43C3766267AD9", name="ASROCK_SB_INTERFACE_DXE_GUID"
    ),
    UefiGuid(
        "DF73ABDA-7A42-4E5E-B34BE6830671C9A6",
        name="ASROCK_AMD_SB900_SMM_DISPATCHER_GUID",
    ),
    UefiGuid("E1B26D32-414F-46DB-9AEE420077F35DBE", name="ASROCK_AMDCPU_DXE_GUID"),
    UefiGuid("E2A74738-8934-48F5-841299E948C8DC1B", name="ASROCK_SMBIOS_DMIEDIT_GUID"),
    UefiGuid(
        "E469AA26-9268-4EB0-A087DD4CEE37404B", name="ASROCK_SECURITY_SELECT_DXE_GUID"
    ),
    UefiGuid(
        "E706CB54-84B8-40BD-832F7FB2D5CB87B3", name="ASROCK_FILE_EXPLORER_LITE_GUID"
    ),
    UefiGuid("E92BAE14-3DFD-4C70-9FE63899F36C7846", name="ASROCK_PLEDSMM_GUID"),
    UefiGuid(
        "EA4DAEE8-A851-4A9F-ABF4B79FA2528291", name="ASROCK_CPU_SMBIOS_DRIVER_GUID"
    ),
    UefiGuid("FF11E702-8923-47CC-AD30D9E0E240DD10", name="ASROCK_AAFTBL_DXE_GUID"),
    UefiGuid("25247A74-9440-47D5-BF0AED92A4D6EBA4", name="DELL_COMPUTRACE_BINARY_GUID"),
    UefiGuid("17772369-D262-4B90-9F31BDC41F2663A5", name="DELL_MEBX_SETUP_BINARY_GUID"),
    UefiGuid("038CE287-B806-45B6-A819514DAF4B91B9", name="DELL_ERROR_LOG_CONFIG_GUID"),
    UefiGuid("041CE993-6677-4720-A665375B80FFE012", name="DELL_FP_THUNK_GUID"),
    UefiGuid("08ABE065-C359-4B95-8D59C1B58EB657B5", name="DELL_INTEL_LOM_SMM_GUID"),
    UefiGuid("08E56A30-62ED-41C6-9240B7455EE653D7", name="DELL_LEGACY_VIDEO_ROM_GUID"),
    UefiGuid("08F2C63B-08DE-4CCD-8670ACFE644A1C48", name="DELL_PCH_S3_SUPPORT_GUID"),
    UefiGuid("0914DE08-434D-4F9F-93B36FA48AEAF7E0", name="DELL_SPI_PART_MICRON_GUID"),
    UefiGuid("099FD87F-4B39-43F6-AB47F801F99209F7", name="DELL_DCP_REGISTER_GUID"),
    UefiGuid("09B98059-2D74-43C5-9709AD0A0051867D", name="DELL_AHCI_DXE_GUID"),
    UefiGuid(
        "0A7521E1-09F1-44AE-9B5F2781B78971D6", name="DELL_SIMPLE_BOOT_FLAG_SRC_GUID"
    ),
    UefiGuid("0B234DF1-0080-412B-87168A81A3A11FAA", name="DELL_EDIAGS_DXE_GUID"),
    UefiGuid("0B2E406A-ED5B-4668-BCD12B1DB01841FC", name="DELL_ACPI_OEM_CONFIG_GUID"),
    UefiGuid("0B7BF76C-721A-44FB-A75852E13CA457F1", name="DELL_SPI_PART_MXIC_GUID"),
    UefiGuid("0BA38E92-154A-4BE2-A5E0BE80571CDB71", name="DELL_DTWLAN_SMM_GUID"),
    UefiGuid("0BB12CAC-CFA9-4839-A4C21797248109D0", name="DELL_LEGACY16_ROM_GUID"),
    UefiGuid(
        "0BC22608-ECC5-47B9-A0384F0A68372E44", name="DELL_DEVICE_NAME_CONFIG_GUID"
    ),
    UefiGuid("0CD3D2FA-B21D-47B5-BA95D8C00B38F324", name="DELL_AI13_ROM_GUID"),
    UefiGuid("0D24A235-9C12-446C-8ECBEFC1F5280209", name="DELL_JPEG_DECODER_GUID"),
    UefiGuid("0D28C529-87D4-4298-8A5440F22A9FE24A", name="DELL_DA_HDD_PROTECTION_GUID"),
    UefiGuid("0D81FDC5-CB98-4B9F-B93B70A9C0663ABE", name="DELL_DCCS_SMM_DRIVER_GUID"),
    UefiGuid(
        "0DDE9636-8321-4EDF-9F140BFCA3B473F5", name="DELL_INTRUSION_DETECT_SMM_GUID"
    ),
    UefiGuid("0E7383B1-83EE-41A4-939E24C886F03AD6", name="DELL_VIRT_RTC_SMM_GUID"),
    UefiGuid("0EA3D651-49B4-47D9-BDC3E4A379F3DE76", name="DELL_SMM_HWM_DRIVER_GUID"),
    UefiGuid("0EB4FAC0-A3EC-4FED-9783A8D0FFD1AE33", name="DELL_SM_BIOS_STRUC_DB_GUID"),
    UefiGuid(
        "1181E16D-AF11-4C52-847E516DD09BD376", name="DELL_CENTURY_ROLLOVER_SMM_GUID"
    ),
    UefiGuid(
        "119F3764-A7C2-4329-B25CE6305E743049", name="DELL_SMM_SECURITY_VAULT_GUID"
    ),
    UefiGuid("11DD0F6C-209B-47CE-8A98418C9B7CC756", name="DELL_DXE_SW_SMI_GUID"),
    UefiGuid("12963E55-5826-469E-A934A3CBB3076EC5", name="DELL_SMM_SB_ACPI_GUID"),
    UefiGuid("1323C7F8-DAD5-4126-A54B7A05FBF41515", name="DELL_SMM_ACCESS_GUID"),
    UefiGuid(
        "1478454A-4584-4CCA-B0D2120ACE129DBB", name="DELL_MFG_MODE_SMM_DRIVER_GUID"
    ),
    UefiGuid("14E7D9BE-7834-477E-A942FF5F823EE249", name="DELL_SPI_PART_ATMEL_GUID"),
    UefiGuid("150EC16F-0BB6-42A8-B60579794B89D15B", name="DELL_EDIAGS_CONFIG_GUID"),
    UefiGuid("166FD043-EA13-4848-BB3C6FA295B94627", name="DELL_VARIABLE_SMM_GUID"),
    UefiGuid("16C368FE-F174-4881-92CE388699D34D95", name="DELL_SMM_GPIO_POLICY_GUID"),
    UefiGuid("16D0A23E-C09C-407D-A14AAD058FDD0CA1", name="DELL_ACPI_GUID"),
    UefiGuid("17390333-4CE6-48A8-AFB90EF55B666B74", name="DELL_RUNTIME_ACPI_DXE_GUID"),
    UefiGuid("1916F73E-C824-4F91-AC9CBF1FD28E4431", name="DELL_SERVICE_MENU_GUID"),
    UefiGuid("194EE52B-02FE-418D-AA102E61E51CB894", name="DELL_MONOTONIC_COUNTER_GUID"),
    UefiGuid("1A1D1204-E385-4943-A8925334CE3809AC", name="DELL_CSM_OEM_ROM_GUID"),
    UefiGuid(
        "1AFE6BD0-C9C5-44D4-B7BD8F5E7D0F2560", name="DELL_DIAGS_SB_CONTROL_SMM_GUID"
    ),
    UefiGuid(
        "1BE65202-9318-492D-A55108DF2BD60AEE", name="DELL_AMT_PLATFORM_POLICY_GUID"
    ),
    UefiGuid(
        "1C33C958-FEA4-435F-93EF0A9C14AE87E4", name="DELL_MFG_AUTHENTICATION_GUID"
    ),
    UefiGuid("1C733CB5-4C8F-44D9-BBF095477089EB08", name="DELL_SM_BIOS_STRUC_DA_GUID"),
    UefiGuid("1CE692F0-3317-4574-8B574800ED091F40", name="DELL_DXE_COMPUTRACE_GUID"),
    UefiGuid("1D82219F-386F-4F26-AFAC613F242C0760", name="DELL_SMM_STATUS_CODE_GUID"),
    UefiGuid(
        "1EF02CF1-BF69-49C3-8FDC5BB55D224CFA", name="DELL_POWER_BUTTON_CONFIG_GUID"
    ),
    UefiGuid("2000A832-C674-486A-81BECB4670A4F7FD", name="DELL_REBOOT_DXE_GUID"),
    UefiGuid("216A681C-9E51-415F-9ADACB9FBCD05F5D", name="DELL_FLASH_IO_DXE_GUID"),
    UefiGuid("231D3A65-9442-4507-A4E4E9D2176578F1", name="DELL_SM_BIOS_STRUC_B1_GUID"),
    UefiGuid("2374EDDF-F203-4FC0-A20E61BAD73089D6", name="DELL_IO_TRAP_GUID"),
    UefiGuid(
        "2386E231-A9CF-4F52-946A6F6B6C133B46", name="DELL_SM_BIOS_DXE_DRIVER_GUID"
    ),
    UefiGuid(
        "24CCD374-3DF6-4181-86F6E3C66920A145", name="DELL_UPDATE_MEMORY_RECORD_GUID"
    ),
    UefiGuid("25ACF158-DD61-4E64-9A4955851E9A26C7", name="DELL_BIOS_BLK_IO_SRC_GUID"),
    UefiGuid("26A2481E-4424-46A2-9943CC4039EAD8F8", name="DELL_ACPIS3_SAVE_GUID"),
    UefiGuid("26C04CF3-F5FB-4968-8D57C7FA0A932783", name="DELL_SB_SERVICES_SMM_GUID"),
    UefiGuid(
        "27287751-867A-46CD-91AF416FF6AE3DFE", name="DELL_SYSTEM_ID_CONFIG_DXE_GUID"
    ),
    UefiGuid("27587B71-37F9-4A48-B57058B6D14F6DEC", name="DELL_OROM_KB_ACCESS_GUID"),
    UefiGuid("27F3838F-BA27-4FFA-B37435BEA28A431B", name="DELL_SMSC5544_DXE_GUID"),
    UefiGuid("290EA249-6E88-423C-B0DA75CDDE7920CC", name="DELL_AMT_PET_ALERT_GUID"),
    UefiGuid("29CF55F8-B675-4F5D-8F2FB87A3ECFD063", name="DELL_CSM_VIDEO_GUID"),
    UefiGuid("2AEDA0EB-1392-4232-A4F9C57A3C2FA2D9", name="DELL_BINDINGS_SMM_GUID"),
    UefiGuid("2D6DCB78-CAE2-4459-927C64A6B7E64A75", name="DELL_EDIAGS_SMM_GUID"),
    UefiGuid("2D7522FA-67F0-4768-B6E0FD76F17F4B04", name="DELL_PBA_UPEK_DPBA_GUID"),
    UefiGuid(
        "2DF10014-CF21-4280-8C3FE539B8EE5150", name="DELL_PPM_POLICY_INIT_DXE_GUID"
    ),
    UefiGuid("2DF61403-C180-45A5-A22B1A3DB88FE9BA", name="DELL_GENERIC_SIO_DXE_GUID"),
    UefiGuid("2E3F2275-89CD-4DE4-BD84BEBFD94BCDCC", name="DELL_SM_BIOS_SMM_DA_CI_GUID"),
    UefiGuid("2EFF67E3-5987-46E8-811180FFA67AEB93", name="DELL_USB_MOUSE_GUID"),
    UefiGuid("2FB216ED-4A67-4833-9F5C248124CD9AAC", name="DELL_DXE_GPIO_POLICY_GUID"),
    UefiGuid("2FB92EFA-2EE0-4BAE-9EB67444125E1EF7", name="DELL_GSET_GUID"),
    UefiGuid("2FB92EFA-2EE0-4BAE-9EB67464125E1EF7", name="DELL_UHCI_GUID"),
    UefiGuid("2FFDAF4A-A9B5-4EF6-AA3BA89611AAB284", name="DELL_SPI_PART_PROM_JET_GUID"),
    UefiGuid("3022E512-B94A-4F12-806D7EF1177899D8", name="DELL_PCI_HOT_PLUG_GUID"),
    UefiGuid(
        "329F126F-299E-4BC5-8310179F10EB46C9", name="DELL_SMB_PROTOCOL_ACCESS_GUID"
    ),
    UefiGuid(
        "33C6406D-2F6B-41B5-870552BAFB633C09",
        name="DELL_ALERT_STANDARD_FORMAT_DXE_GUID",
    ),
    UefiGuid("36AEBB56-1F8B-4BC1-8EFE4085A5D4C40A", name="DELL_LEG_USB_MEM_SMM_GUID"),
    UefiGuid("37287C8B-DC93-45D5-9AAF3CD2E906B12B", name="DELL_ERROR_HANDLER_SMM_GUID"),
    UefiGuid(
        "37287C8B-DC93-45D5-9AAF3CD2E906FFFF", name="DELL_DEFAULT_BOOT_LIST_SMM_GUID"
    ),
    UefiGuid("386EB7BA-6344-44E8-985C3978FD47916B", name="DELL_TPMDXE_GUID"),
    UefiGuid("38871BF0-C64A-4896-B8E462D4850C7E68", name="DELL_OEM_SX_SMM_GUID"),
    UefiGuid("38871BF0-C64A-4896-B8E462D7750C7D68", name="DELL_AC_LOSS_SMM_GUID"),
    UefiGuid(
        "38EF2B8B-CF3D-47E2-97E3557115E43985", name="DELL_DIAGS_SB_CONTROL_DXE_GUID"
    ),
    UefiGuid("3962DEB5-FDF6-4829-889CD544918E84D9", name="DELL_SMM_LEGACY_REGION_GUID"),
    UefiGuid("39A7FADC-9A24-4553-8CBD9B72B24C22FD", name="DELL_PS2_EMUL_GUID"),
    UefiGuid("3A29DD19-2C84-4A70-9A01FBDDAFFB6461", name="DELL_MEMORY_S4_GUID"),
    UefiGuid("3A4044D2-F614-4124-A772207BCDBE3DB0", name="DELL_SB_SERVICES_DXE_GUID"),
    UefiGuid("3B37B724-4AF7-4A35-97B3D2FB8F4F563C", name="DELL_STOP_SHIP_CONFIG_GUID"),
    UefiGuid("3BC1C867-2274-43F8-A4BA939A5BD6D473", name="DELL_BLOCK_IO_GUID"),
    UefiGuid("3C1DE39F-D207-408A-AACC731CFB7F1DD7", name="DELL_PCI_BUS_GUID"),
    UefiGuid("3D5674D0-AE75-4723-91591D653B40402F", name="DELL_TX_TDXE_GUID"),
    UefiGuid("3DDE54E9-8D18-4EA8-9423C2E846E18E23", name="DELL_SM_BIOS_STRUC_D3_GUID"),
    UefiGuid("41846A2E-7AAA-4498-BC26B5ACD49F92E1", name="DELL_SMB_BOOT_PRIORITY_GUID"),
    UefiGuid(
        "418FAF6C-2BCF-4141-89839D33E1B67B4C", name="DELL_SPI_CONTROLLER_SMM_GUID"
    ),
    UefiGuid(
        "41F46692-9513-4D1C-8BB3B6A168182DE4",
        name="DELL_ME_PCI_PLATFORM_CONFIG_DXE_GUID",
    ),
    UefiGuid("42C32DCF-F25A-47F3-8F4CABD478D77F50", name="DELL_PBA_UPEK_DXE_GUID"),
    UefiGuid("45FDB42B-1F95-4A53-9BA41C8BFB8BCB01", name="DELL_GPE_CONFIG_GUID"),
    UefiGuid("46FCD5D8-A9B3-42E3-904A012F07BC5F5C", name="DELL_POWER_OFF_SMM_GUID"),
    UefiGuid("4772C6DA-506D-4C95-AC17AE66885F6D53", name="DELL_GPE_GUID"),
    UefiGuid("47B2138F-5A6B-4C65-91678740595D16AD", name="DELL_PASSWORD_MGR_SMM_GUID"),
    UefiGuid(
        "47E83D0C-2AF4-468D-91919CE468C18E9E", name="DELL_SMM_SB_GPIO_CONTROL_GUID"
    ),
    UefiGuid("492261E4-0659-424C-82B673274389E7A7", name="DELL_RAID_ROM_GUID"),
    UefiGuid("4A37320B-3FB3-4365-97309E89C600395D", name="DELL_SMM_DISPATCHER_GUID"),
    UefiGuid(
        "4A64F200-F99C-42F3-80C70D91E2D3EB5D", name="DELL_DXE_SIO_GPIO_CONTROL_GUID"
    ),
    UefiGuid("4B87C89B-39A9-478F-A4B6816A9EDC5087", name="DELL_VARIABLE_DXE_GUID"),
    UefiGuid("4E98A9FE-DF5A-4136-B9224FB970CBD733", name="DELL_CPU_SMM_GUID"),
    UefiGuid(
        "51D687C8-196E-4CFE-8A0C8368980EA8B6", name="DELL_CHASSIS_CONFIG_SMM_GUID"
    ),
    UefiGuid("52312CDF-D4A9-453E-A9DF9FDAF51CD622", name="DELL_SMM_DELAY_GUID"),
    UefiGuid("52A88B2A-65FD-4951-8B2506FB6598128B", name="DELL_SPI_PART_EON_GUID"),
    UefiGuid(
        "54BD6A5D-38F2-44BD-84AB108BE4F45914", name="DELL_CHASSIS_CONFIG_DXE_GUID"
    ),
    UefiGuid(
        "55A267FC-6607-4023-884E48E5D3D3F4E2", name="DELL_HIIPARSER_DXE_DRIVER_GUID"
    ),
    UefiGuid("55BB5A4B-3B3F-442C-BF090F87A67E8EE7", name="DELL_BINDINGS_DXE_GUID"),
    UefiGuid("55E76644-78A5-4A82-A9007126A5798892", name="DELL_HECI_DXE_GUID"),
    UefiGuid(
        "55F6A413-280F-41E3-93946580F29DD7D0", name="DELL_SMM_BBSMANAGER_LEGACY_GUID"
    ),
    UefiGuid("56EEE346-500C-4936-951A2DA5F0FF39BB", name="DELL_GRASSLAKE_CONFIG_GUID"),
    UefiGuid("571D1ED1-C2D9-418E-953A248EBC687048", name="DELL_IDE_SMM_GUID"),
    UefiGuid("57F435A8-1C37-4834-AD61B03D94DDCEF3", name="DELL_PASSWORD_MGR_DXE_GUID"),
    UefiGuid("5820EEB4-C135-4854-9D2AAA9EFC4475E9", name="DELL_ME_FW_DOWNGRADE_GUID"),
    UefiGuid("587E8797-C2FB-41B7-AE9BF53A6C562DF5", name="DELL_CSM_CONFIG_GUID"),
    UefiGuid("58CDC779-DC8C-491B-BAC3F86E06D1865F", name="DELL_GPE_OROM_GUID"),
    UefiGuid(
        "590915CC-8622-4D61-BE64C33EE760141F", name="DELL_SM_BIOS_SMM_DRIVER_GUID"
    ),
    UefiGuid("59287178-59B2-49CA-BC63532B12EA2C53", name="DELL_PCH_SMBUS_SMM_GUID"),
    UefiGuid("59378206-861B-4380-A3492F2F4F030C4B", name="DELL_DASH_BIOS_MANAGER_GUID"),
    UefiGuid("5AAB83E5-F027-4CA7-BFD016358CC9E453", name="DELL_WDT_DXE_GUID"),
    UefiGuid("5AE3F37E-4EAE-41AE-824035465B5E81EB", name="DELL_CORE_DXE_GUID"),
    UefiGuid("5D44BE77-5669-41D0-B6851BF3F83EFB98", name="DELL_PASSWORD_UI_GUID"),
    UefiGuid(
        "5DA9E544-DC2D-4670-A3D5985236D5DE45", name="DELL_HOT_SOS_SMM_DRIVER_GUID"
    ),
    UefiGuid("5DB37E5C-2EEB-43F8-ADA0243702D3945D", name="DELL_SIO_CONFIG_DXE_GUID"),
    UefiGuid("5E9CABA3-F2B1-497A-ADAC24F575E9CDE9", name="DELL_TCG_DXE_GUID"),
    UefiGuid("5EDA3123-0840-45F7-ADAB13A655EF3BB7", name="DELL_ACPI_CONFIG_GUID"),
    UefiGuid("60EBDAA4-1565-4D9A-99C888DFA65549A5", name="DELL_FLASH_SCREEN_DXE_GUID"),
    UefiGuid("612E0D09-5D11-4516-82EE6DD605FE265F", name="DELL_TXT_DXE_GUID"),
    UefiGuid("631B4DF7-BAEA-4C1F-A0615B6462652822", name="DELL_DIAGS_DXE_DRIVER_GUID"),
    UefiGuid("6426C814-601A-4116-9E9FBF9D6F8F254F", name="DELL_FLASH_UPDATE_DXE_GUID"),
    UefiGuid("65E9EB9D-812D-4161-99C505AC4BCCA494", name="DELL_SMM_GPE_GUID"),
    UefiGuid("67499F84-F2E5-4DD2-9E56C6E389DD6173", name="DELL_RTC_AUTO_ON_SMM_GUID"),
    UefiGuid(
        "69EF78BC-3B71-4ECC-834F3B74F9148430", name="DELL_SMM_FLASH_WRITE_PROTECT_GUID"
    ),
    UefiGuid("69F97A4E-1395-434A-A5D3FF9CEA284885", name="DELL_DIAGS_CONFIG_GUID"),
    UefiGuid("6AA1A3CC-52C4-4B03-99F7288F345C1DB3", name="DELL_AUDIO_CONFIG_DXE_GUID"),
    UefiGuid("6B287864-759C-42C4-B435A74AB694CD3B", name="DELL_SPECIAL_BOOT_STUB_GUID"),
    UefiGuid("6E8CD2B7-B636-4859-85EDC637BDCA5919", name="DELL_DA_SERVICE_TAG_GUID"),
    UefiGuid("6E97D4F4-1898-4408-A62D99CADED32FD8", name="DELL_DXE_SB_ACPI_GUID"),
    UefiGuid(
        "7122810D-CCD3-4B09-A0AB8D107645C978", name="DELL_MONOTONIC_COUNTER_SMM_GUID"
    ),
    UefiGuid("71287108-BF58-41EA-B71CB3622DEBCA9D", name="DELL_SMM_SB_GENERIC_GUID"),
    UefiGuid("750890A6-7ACF-4F4F-81BDB400C2BEA95A", name="DELL_ACPI_MODE_ENABLE_GUID"),
    UefiGuid("758880AB-4CA2-4DAA-A83ED1F9ADF2C2F1", name="DELL_SMM_SIO_EMI_GUID"),
    UefiGuid("75BE667C-48E1-452D-B0FC363138ED87FF", name="DELL_PSTATE_CONTROL_GUID"),
    UefiGuid("76A1DF7D-754C-40E1-BC723FCAD842DEF6", name="DELL_DA_EPPID_GUID"),
    UefiGuid("76BE1ABF-1E1E-321D-B7F09E0984FDDABC", name="DELL_PROGRESS_BAR_GUID"),
    UefiGuid("76E1E9EE-5FC7-4613-B96FFA797796573A", name="DELL_PBA_MGR_SMM_GUID"),
    UefiGuid("773CB08B-511A-4BD5-85AD41D4F4B64A52", name="DELL_AMT_SETUP_GUID"),
    UefiGuid(
        "7808F5C2-9CB0-4609-907FB18453C363B4", name="DELL_STORAGE_AGENTS_DXE_GUID"
    ),
    UefiGuid(
        "787D4843-68AC-4C28-BE53BAC272BAECF9", name="DELL_VARIABLE_RUNTIME_DXE_GUID"
    ),
    UefiGuid("79E39A83-D013-4B2E-836AA99A7147351C", name="DELL_PBA_SCARD_DXE_GUID"),
    UefiGuid("7A08CB98-E9BC-41C3-BE19B302F3F1F523", name="DELL_SOL_POST_MESSAGE_GUID"),
    UefiGuid("7A08CB98-E9BC-41C3-BE19B302F3F1F595", name="DELL_TERMINAL_GUID"),
    UefiGuid("7BBB9525-97BA-4516-B6601B0EB78744A5", name="DELL_SM_BIOS_STRUC_D0_GUID"),
    UefiGuid("7C79AC8C-5E6C-4E3D-BA6FC260EE7C172E", name="DELL_SMM_RUNTIME_GUID"),
    UefiGuid("7CB919C8-D499-4629-ACB322769E343DDB", name="DELL_FLASH_IO_SMM_GUID"),
    UefiGuid("80AADD0D-8053-4BA7-941B36F48BB23001", name="DELL_SYSTEM_PW_SMM_GUID"),
    UefiGuid("80E66E0A-CCD1-43FA-A7B12D5EE0F13910", name="DELL_PCI_ROOT_BRIDGE_GUID"),
    UefiGuid("80F03C1F-89BD-4240-93A8AC99E9A929FC", name="DELL_POWER_OFF_CONFIG_GUID"),
    UefiGuid("82833414-C8F4-4B62-AB6AE4D5DAAB130F", name="DELL_DXE_HWM_DRIVER_GUID"),
    UefiGuid("82FBE26B-53D4-448F-924C7BE0126ECA4F", name="DELL_PXE_UNDI_ROM_GUID"),
    UefiGuid("83B698B9-0208-4ECF-BB83CE42D0BEFC4A", name="DELL_PW_BY_PASS_SMM_GUID"),
    UefiGuid(
        "8467C103-6FF6-4D66-86F6E1124C111E94", name="DELL_HOT_KEY_HANDLER_SMM_GUID"
    ),
    UefiGuid(
        "84991287-3FF0-4FCC-9C11C7E041862C76", name="DELL_POWER_MANAGEMENT_DXE_GUID"
    ),
    UefiGuid("858189AC-0ECA-4666-BAB89972409953C8", name="DELL_AC_RECOVERY_SMM_GUID"),
    UefiGuid("87C73C1C-1529-4EF5-A3FA08D73EF8C827", name="DELL_SMM_SIO_DASH_GUID"),
    UefiGuid("88D01BB9-247A-46F6-BB6EBBA49F1124F8", name="DELL_SMM_EVENT_DXE_GUID"),
    UefiGuid("899407D7-99FE-43D8-9A2179EC328CAC21", name="DELL_SETUP_PREP_GUID"),
    UefiGuid("89BF244F-E569-416F-B1108AE5EFD5171C", name="DELL_DXE_PREREQ_GUID"),
    UefiGuid("8A9F7CFC-5BFC-458F-986FBC3D1BA7AC89", name="DELL_SPI_PART_ST_MICRO_GUID"),
    UefiGuid("8B529353-1FCC-4AC0-A12190BE9F9CBF7A", name="DELL_DIAGS_LEGACY_GUID"),
    UefiGuid("8B5FBABD-F51F-4942-BF1616AAA38AE52B", name="DELL_ACPI_PLATFORM_GUID"),
    UefiGuid("8C783970-F02A-4A4D-AF098797A51EEC8D", name="DELL_POWER_MANAGEMENT_GUID"),
    UefiGuid("BC3245BD-B982-4F55-9F79056AD7E987C5", name="DELL_AHCI_SMM_GUID"),
    UefiGuid(
        "8D32716A-60DF-4BDE-86473D178CC82B6C", name="DELL_ME_PLATFORM_CONFIG_DXE_GUID"
    ),
    UefiGuid("8D6756B9-E55E-4D6A-A3A55E4D72DDF772", name="DELL_PCI_HOST_BRIDGE_GUID"),
    UefiGuid("8DCE1957-D774-4FDA-B90F0B231CB2E0CC", name="DELL_SMM_SERVICES_GUID"),
    UefiGuid("8E201427-463A-4D24-A0F425A875F61BAA", name="DELL_MOD_BAY_CONFIG_GUID"),
    UefiGuid("8EC38829-A59F-40E3-BAD085F815A157D1", name="DELL_SERVICE_RESET_SMM_GUID"),
    UefiGuid("8EC40FB3-4205-4D57-9039A192832C7CED", name="DELL_EC_CONFIG_DXE_GUID"),
    UefiGuid("8F439D8B-07B2-4121-AA958AC91571AA75", name="DELL_ERROR_HANDLER_DXE_GUID"),
    UefiGuid(
        "8F439D8B-07B2-4121-AA958AC91571FFFF", name="DELL_DEFAULT_BOOT_LIST_DXE_GUID"
    ),
    UefiGuid("8F5A2E02-538C-4D59-B920C4786ACBC552", name="DELL_AHCISRC_GUID"),
    UefiGuid("8FB82DA4-C3BE-42A6-8A707B94D60972A6", name="DELL_SB_ACCESS_SMM_GUID"),
    UefiGuid("903718F3-575D-4D9C-BC7865A11381A026", name="DELL_IMG_SRV_BASE_ROM_GUID"),
    UefiGuid("90CB75DB-71FC-489D-AACF943477EC7212", name="DELL_SMART_TIMER_GUID"),
    UefiGuid("91BA201E-1621-46FC-825AB0363ECAD4BC", name="DELL_PBA_FP_DXE_GUID"),
    UefiGuid("921CD783-3E22-4579-A71F00D74197FCC8", name="DELL_HECI_SMM_GUID"),
    UefiGuid("92C78E4E-C994-4F9E-82ABAF1DBA6858C9", name="DELL_SMM_COMPUTRACE_GUID"),
    UefiGuid("93022F8C-1F09-47EF-BBB25814FF609DF5", name="DELL_FILE_SYSTEM_GUID"),
    UefiGuid(
        "937710D0-EF10-42BE-92E18494722B17B5", name="DELL_DIRTY_SHUTDOWN_DXE_GUID"
    ),
    UefiGuid("94925FB1-EEF4-41ED-BD7ACDB04AAA81C7", name="DELL_SB_ACPI_CONFIG_GUID"),
    UefiGuid("94EDD12A-419B-447F-94349B3B70783903", name="DELL_TCG_SETUP_GUID"),
    UefiGuid("973BE267-0960-40C7-9856FA73C92A46BA", name="DELL_SERVICE_RESET_DXE_GUID"),
    UefiGuid(
        "97CC7188-79C9-449F-B969065B64BF9C69", name="DELL_BIOS_EXTENSION_LOADER_GUID"
    ),
    UefiGuid(
        "98D8A754-8482-46E2-828CBB632F34D9F7", name="DELL_HOT_SOS_DXE_DRIVER_GUID"
    ),
    UefiGuid("995D96C1-6651-46B0-BF3E83C017B43B1D", name="DELL_SM_BIOS_STRUC_DE_GUID"),
    UefiGuid("99A1E3F5-0887-43C7-B00AF14A646BC142", name="DELL_SB_SMM_HOOKS_GUID"),
    UefiGuid("9AB332EF-81D9-43D1-867A03DD574E8566", name="DELL_BBSMANAGER_DXE_GUID"),
    UefiGuid("9AF1DA25-F06C-43BB-BE88834550BF765B", name="DELL_GSET_CONFIG_GUID"),
    UefiGuid("9BFF300D-D647-480E-9FF32006D3D7CCC6", name="DELL_MULTI_DISPLAY_DXE_GUID"),
    UefiGuid("9F3A0016-AE55-4288-829DD22FD344C347", name="DELL_AMI_BOARD_INFO_GUID"),
    UefiGuid(
        "9FB63BF9-B522-4203-9A5320EC854A312A", name="DELL_DA_NON_VOLITILE_STORAGE_GUID"
    ),
    UefiGuid("9FCFD3EE-38C3-4AAF-9512EC416371F64A", name="DELL_DXE_MAIN_GUID"),
    UefiGuid(
        "A041B8D6-3F17-4582-AAC75A615672E295", name="DELL_SIO_POLICY_CONFIG_DXE_GUID"
    ),
    UefiGuid("A062CF1F-8473-4AA3-8793600BC4FFE9A8", name="DELL_CSMCORE_GUID"),
    UefiGuid("A0B87E92-6392-47FE-9AA090203147EC04", name="DELL_HDD_PW_SMM_GUID"),
    UefiGuid("A0BAD9F7-AB78-491B-B583C52B7F84B9E0", name="DELL_SMM_CONTROL_GUID"),
    UefiGuid("A19FB0EE-05F4-4CD6-8F2859B782FF95C6", name="DELL_PCI_EXPRESS_DXE_GUID"),
    UefiGuid("A287558B-D9DD-461B-A929EC2AD2FEF591", name="DELL_SPI_PART_SST_GUID"),
    UefiGuid(
        "A37A9E43-9077-4539-9B90BD719B032B23", name="DELL_SPI_CONTROLLER_DXE_GUID"
    ),
    UefiGuid("A3D93A29-501B-4F6E-869A6FD4BCEF90A7", name="DELL_SMB_FACT_DEFAULT_GUID"),
    UefiGuid(
        "A74B3AEF-2502-40C5-83A93524776273EA", name="DELL_SMM_SIO_GPIO_CONTROL_GUID"
    ),
    UefiGuid("A79DC347-DEB4-4A4E-8497CA5F8E7F0601", name="DELL_IMAGE_SERVER_SMM_GUID"),
    UefiGuid("A7B36DCF-C75A-4BE8-BE2FE4840CBE163B", name="DELL_PBA_MGR_DXE_GUID"),
    UefiGuid("A833441A-0CA0-4032-859A3CFF7B6646A7", name="DELL_IMAGE_SERVER_DXE_GUID"),
    UefiGuid("A8752510-D83E-400A-85DEFD1587882AFE", name="DELL_LOM_CONFIG_GUID"),
    UefiGuid("A8A2F6E1-6F62-4CFB-9E4793CFD6D6E476", name="DELL_LEGACY_USB_SMM_GUID"),
    UefiGuid("A8C67255-E029-4B1A-968EECA6E9C11C73", name="DELL_AMT_SMBIOS_GUID"),
    UefiGuid("AD416CE3-A483-45B1-94C24B4E4D575562", name="DELL_TCG_MOR_GUID"),
    UefiGuid(
        "AD4A0D05-806D-4FE4-96AD3BD62D8C4CE0", name="DELL_DO_SIODIAG_LED_DXE_GUID"
    ),
    UefiGuid("AE24851D-E414-4062-959D5F43EA99363C", name="DELL_AUDIO_DXE_GUID"),
    UefiGuid("AE717C2F-1A42-4F2B-886178B79CA07E07", name="DELL_FV_MAINNESTED_GUID"),
    UefiGuid(
        "AFE751C5-05D4-4898-BF941A4B50F35FD4", name="DELL_SMM_CRYPTO_ALGORITHM_GUID"
    ),
    UefiGuid(
        "B0D6ED53-B844-43F5-BD2F61095264E77E", name="DELL_PCH_SMI_DISPATCHER_GUID"
    ),
    UefiGuid("B13EDD38-684C-41ED-A305D7B7E32497DF", name="DELL_SMBIOS64_GUID"),
    UefiGuid("B1DA0ADF-4F77-4070-A88EBFFE1C60529A", name="DELL_AMITSE_GUID"),
    UefiGuid("B4186561-8967-4113-BC5F284B7670C4A0", name="DELL_DXE_DELAY_GUID"),
    UefiGuid(
        "B45B5FC4-D6A6-4523-A229E7772097EF14", name="DELL_LEG_USB_MEM_INFO_DXE_GUID"
    ),
    UefiGuid("B71ECD12-E01E-440B-AA524D6E59D45B3C", name="DELL_SECURITY_CONFIG_GUID"),
    UefiGuid(
        "B941CF11-E648-401C-92A1342A70C6B742", name="DELL_SMB_VERSION_MANIFEST_GUID"
    ),
    UefiGuid("B98999A4-E96F-475A-99FC762126F50F5A", name="DELL_SMBIOSUPDATE_DATA_GUID"),
    UefiGuid("BB1FBD4F-2E30-4793-9BED74F672BC8FFE", name="DELL_PCH_RESET_GUID"),
    UefiGuid("BB65942B-521F-4EC3-BAF9A92540CF60D2", name="DELL_SATA_CONTROLLER_GUID"),
    UefiGuid(
        "BB87B31D-F366-47E9-885EE816B09B97B6",
        name="DELL_ALERT_STANDARD_FORMAT_SMM_GUID",
    ),
    UefiGuid("BCABB6DD-3E9F-47E2-B8FB10B39AF6342C", name="DELL_DA_TAA_MAC_GUID"),
    UefiGuid(
        "BCBB3912-29FA-4B25-B6193C0D739FEF51", name="DELL_HOT_KEY_HANDLER_DXE_GUID"
    ),
    UefiGuid(
        "BD45A258-C7BD-4BF0-B254D0D45E02F877", name="DELL_CMOS_MANAGER_DXE_SRC_GUID"
    ),
    UefiGuid("BDCBB03C-545E-43C1-8AA8F058C4FA81A8", name="DELL_DXE_GPIO_CONTROL_GUID"),
    UefiGuid("BDE92229-F9A5-4897-81E48BAB6792A395", name="DELL_DT_LEG_USB_POLICY_GUID"),
    UefiGuid("BDFAAD26-4D7F-44A6-8719375ECB4AA407", name="DELL_PXE_BASE_ROM_GUID"),
    UefiGuid("BE216BA8-38C4-4535-A6CA5DCA5B43ADDF", name="DELL_SMI_VARIABLE_GUID"),
    UefiGuid("BF27A268-F369-43A2-B295242DEE7F3C29", name="DELL_POWER_BUTTON_DXE_GUID"),
    UefiGuid("BFD59D42-FE0F-4251-B7724B098A1AEC85", name="DELL_ACTIVE_BIOS_GUID"),
    UefiGuid(
        "C0B9206E-B6AB-4DF0-B3D6286AC76698BD", name="DELL_D010_ALEGACY_VIDEO_ROM_GUID"
    ),
    UefiGuid("C10A9E4D-BDB4-4B1A-82CCD901F86F5739", name="DELL_SMM_DIAG_LEDS_GUID"),
    UefiGuid(
        "C1443436-B954-43BA-8278C1E442C21539", name="DELL_FLASH_WRITE_PROTECT_GUID"
    ),
    UefiGuid("C1C418F9-591D-461C-82A2B9CD96DFEA86", name="DELL_LEGACY_INTERRUPT_GUID"),
    UefiGuid(
        "C41459C2-A281-40D3-B30A2B9DFF3F3DA0", name="DELL_SAVE_MEMORY_CONFIG_DXE_GUID"
    ),
    UefiGuid("C4331752-8BAC-4F2A-A9C3418ADFB033C4", name="DELL_SPI_PART_WINBOND_GUID"),
    UefiGuid("C4C7A1A9-D639-442D-B35809D06AE18BFC", name="DELL_DA_ASSET_TAG_GUID"),
    UefiGuid("C4F2D007-37FD-422D-B63D7ED73886E6CA", name="DELL_IDE_RCONTROLLER_GUID"),
    UefiGuid("C65AF0E1-82B3-416A-8F0ADDD0D3098C58", name="DELL_PBA_USH_GUID"),
    UefiGuid(
        "C67CBA49-B3C6-4CBA-9671BF2788712128", name="DELL_LEGACY_TCG_SUPPORT_GUID"
    ),
    UefiGuid(
        "C7B11F3C-89E9-445F-A972671B8B17580D", name="DELL_SMB_STRUC_B2_CONFIG_GUID"
    ),
    UefiGuid("C810485E-D0EC-4E98-AAB5120C7E554428", name="DELL_TDT_AM_GUID"),
    UefiGuid("C86181CD-B38C-49DD-A1767DC3591816A9", name="DELL_HOT_KEYS_CONFIG_GUID"),
    UefiGuid("C87DBB18-9ADD-41D5-80D74C8D795AD942", name="DELL_SMM_EVENT_SMM_GUID"),
    UefiGuid("C89CEC07-C4EE-440C-BFC5A22F43372F22", name="DELL_DASH_CONFIG_GUID"),
    UefiGuid(
        "C9D9CE44-708D-4E27-B989E35A33999C28", name="DELL_SMM_MULTI_PLATFORM_GUID"
    ),
    UefiGuid("CBC59C4A-383A-41EB-A8EE4498AEA567E4", name="DELL_RUNTIME_GUID"),
    UefiGuid("CCCC8702-BD68-45B1-8237C7DD004CDB37", name="DELL_SM_BIOS_STRUC_D2_GUID"),
    UefiGuid("CD84562C-6864-40A3-A081C8D35E82B920", name="DELL_CSP_LIB_DXE_GUID"),
    UefiGuid("CDC11AE9-01E7-42CB-88EBFDFFD8819893", name="DELL_TCG_SMM_GUID"),
    UefiGuid("CEF68C66-06AB-4FB3-A3ED5FFA885B5725", name="DELL_SMBIOS_BOARD_GUID"),
    UefiGuid("D0632C90-AFD7-4492-B186257C63143C61", name="DELL_SMM_BASE_GUID"),
    UefiGuid("D15170BE-D512-4894-B8639D0E6FF5C561", name="DELL_SLP_MARKER_CTRL_GUID"),
    UefiGuid("D472AB34-8E21-4343-8BFF01ECDE292375", name="DELL_THROTTLING_DRIVER_GUID"),
    UefiGuid(
        "D4A79A0D-B67D-4E83-8EFB455924372934", name="DELL_TERMINAL_PLATFORM_POLICY_GUID"
    ),
    UefiGuid("D4B61940-73AB-48A5-9E2653A4CA4A2C37", name="DELL_DIAGS_SMM_GUID"),
    UefiGuid("D61B6C7A-653A-45A9-8AF363A8A5076639", name="DELL_HDD_SMART_SMM_GUID"),
    UefiGuid(
        "D6AC4AC2-8BC2-4CDD-8776280E1469DE02", name="DELL_PCH_POLICY_INIT_DXE_GUID"
    ),
    UefiGuid("D700E3BC-4291-4749-87AF432A023DE658", name="DELL_NB_CONFIG_DXE_GUID"),
    UefiGuid("D739F969-FB2D-4BC2-AFE7081327D3FEDE", name="DELL_I_AMTDXE_GUID"),
    UefiGuid("D77C900D-A1C7-41C5-B9890C3D37FCA432", name="DELL_AMT_WRAPPER_DXE_GUID"),
    UefiGuid("D85A4A0C-2E73-4491-92E1DCEFC3882A68", name="DELL_PHYSICAL_PRESENCE_GUID"),
    UefiGuid("D9389EFA-2C9A-4696-8BB6FC5F398D8DF3", name="DELL_SMM_GPIO_CONTROL_GUID"),
    UefiGuid("D9CD0AA0-C20F-4810-999ED991FD0484D9", name="DELL_SM_BIOS_STRUC_D9_GUID"),
    UefiGuid(
        "DA451AF6-88D1-4FD7-B86E44BC47EE1F7E", name="DELL_STORAGE_AGENTS_SMM_GUID"
    ),
    UefiGuid("DC7063FC-E01D-476F-9BF3DE010884BC53", name="DELL_SMB_BBS_INFO_GUID"),
    UefiGuid("DC9CACE8-8C2E-4FE7-9E41E18C75FE9B66", name="DELL_SM_BIOS_STRUC_B2_GUID"),
    UefiGuid("DCAA4B60-408F-4BAD-99B9B880D4EF0950", name="DELL_TDT_DXE_GUID"),
    UefiGuid("DE23ACEE-CF55-4FB6-AA77984AB53DE811", name="DELL_SA_INIT_DXE_GUID"),
    UefiGuid("DE23ACEE-CF55-4FB6-AA77984AB53DE823", name="DELL_PCH_INIT_DXE_GUID"),
    UefiGuid(
        "DED7956D-7E20-4F20-91A1190439B04D5B", name="DELL_SMBIOS_GET_FLASH_DATA64_GUID"
    ),
    UefiGuid("DEFBFC7A-CC02-4225-88AD3C41E621785B", name="DELL_IDE_DXE_GUID"),
    UefiGuid(
        "DFE7EE64-D4FD-4424-BF1F85DA524236FB", name="DELL_MFG_MODE_DXE_DRIVER_GUID"
    ),
    UefiGuid("DFF4E6C8-234C-453E-A92DFA505E90D539", name="DELL_DA_OWNER_TAG_GUID"),
    UefiGuid("E0364FEE-1440-4A41-AD3E50E0B106A83D", name="DELL_DA_BFA_GUID"),
    UefiGuid("E03ABADF-E536-4E88-B3A0B77F78EB34FE", name="DELL_CPU_DXE_GUID"),
    UefiGuid("E052D8A6-224A-4C32-8D372E0AE162364D", name="DELL_PCH_SMBUS_DXE_GUID"),
    UefiGuid("E10539B8-1C35-4B84-859381555D065DD5", name="DELL_DA_PASSWORDS_GUID"),
    UefiGuid("E2CFB356-39A9-497B-B6B195B9D2EB2817", name="DELL_SLP_CONFIG_GUID"),
    UefiGuid(
        "E305E101-913A-4A51-8E0EB4618C5DB326", name="DELL_DXE_SB_GPIO_CONTROL_GUID"
    ),
    UefiGuid(
        "E374574A-2ADF-43DB-A77888A75FC08026", name="DELL_SMB_DA_TOKENS_CONFIG_GUID"
    ),
    UefiGuid(
        "E3932A34-5729-4F24-9FB1D7409B456A15", name="DELL_OEM_BADGING_SUPPORT_GUID"
    ),
    UefiGuid("E3ED7C21-9DB8-47C7-8E7D7DB97FE2063E", name="DELL_SMM_CIRC_BUFF_GUID"),
    UefiGuid("E566B097-4378-485F-91D01C097C190CE2", name="DELL_POWER_BUTTON_GUID"),
    UefiGuid("E72527CF-505B-4B50-99CDA32467FA4AA4", name="DELL_ASF_TABLE_GUID"),
    UefiGuid("E77E53A5-A5B8-45DC-BAEB66C86F1AC811", name="DELL_RUNTIME_ACPI_SMM_GUID"),
    UefiGuid("E807983E-D366-40C9-846A0ED7E6022C96", name="DELL_TAGS_CONFIG_GUID"),
    UefiGuid("E975314A-1464-411F-85E4041D903CB6C4", name="DELL_SM_BIOS_STRUC_FAN_GUID"),
    UefiGuid("EB78CE7E-4107-4EF5-86CB22E8D8AC4950", name="DELL_SMM_ASF_INIT_GUID"),
    UefiGuid("ECD81473-0533-4844-BC7748F0DE27519B", name="DELL_SM_BIOS_STRUC_D8_GUID"),
    UefiGuid("EDA39402-F375-4496-92D383B43CB8A76A", name="DELL_SM_BIOS_MEMORY_GUID"),
    UefiGuid("EFD652CC-0E99-40F0-96C0E08C089070FC", name="DELL_S3_RESUME_GUID"),
    UefiGuid("F0D8DEAA-C495-4139-B1FC78B70168260A", name="DELL_DXE_DIAG_LEDS_GUID"),
    UefiGuid("F0FFF864-8454-46E5-9F2BCF14F0A41A48", name="DELL_WARM_BOOT_SMM_GUID"),
    UefiGuid(
        "F1A18726-FD79-4DF5-A326E20B8CC46B06", name="DELL_BOOT_SCRIPT_SMI_HANDLER_GUID"
    ),
    UefiGuid("F317B29B-7DC9-4114-9086D7137EF4F118", name="DELL_TPM_SMM_GUID"),
    UefiGuid("F31C76B9-0B55-4456-99FE50A9C5EC116C", name="DELL_DXE_SIO_EMI_GUID"),
    UefiGuid(
        "F64EFABA-D0A9-42C3-B5E8391B48647BF1", name="DELL_GT2_LEGACY_VIDEO_ROM_GUID"
    ),
    UefiGuid(
        "F6CDAAE0-B301-4202-89F68623E6AED26E", name="DELL_SM_BIOS_STRUC_THERM_GUID"
    ),
    UefiGuid("F7D7F748-CCDE-481F-9B862EF5EDACC21D", name="DELL_VIDEO_CONFIG_DXE_GUID"),
    UefiGuid("F8BB41A2-485E-4B9D-88CF151E62F84DDE", name="DELL_SMB_RUN_ROM_GUID"),
    UefiGuid("F9109394-E5A3-4E61-B168DC34D534451E", name="DELL_BBSMANAGER_SMM_GUID"),
    UefiGuid(
        "F935FF1B-AFF0-40F3-B77AB795A7A7D714", name="DELL_PHYSICAL_PRESENCE_SMM_GUID"
    ),
    UefiGuid(
        "F94AB34A-8069-4A9F-AD089534EF37D9BE", name="DELL_ME_LOCAL_FW_UPDATE_DXE_GUID"
    ),
    UefiGuid("FB142B99-DF57-46CB-BC690BF858A734F9", name="DELL_PCI_SERIAL_GUID"),
    UefiGuid("FC1B7640-3466-4C06-B1CC1C935394B5C2", name="DELL_PCH_SERIAL_GPIO_GUID"),
    UefiGuid("FC87501F-F707-49A2-B67677717DD904DC", name="DELL_SMI_CPU_DECODE_GUID"),
    UefiGuid("0997E770-756E-4139-BF3F7FC2A308177E", name="DELL_SMM_S3_GUID"),
    UefiGuid(
        "1555ACF3-BD07-4685-B668A86945A4124D", name="DELL_CPU_PEI_BEFORE_MEM_GUID"
    ),
    UefiGuid("1D88C542-9DF7-424A-AA9002B61F286938", name="DELL_WDT_PEI_GUID"),
    UefiGuid("221D6760-4AC0-4C12-BA969C63E4D9C1C8", name="DELL_PEI_HWM_INIT_GUID"),
    UefiGuid("2BB5AFA9-FF33-417B-8497CB773C2B93BF", name="DELL_CPU_PEI_GUID"),
    UefiGuid("336CDDEA-AB28-4C4C-9F645FE0391FEBB8", name="DELL_AUDIO_CONFIG_PEI_GUID"),
    UefiGuid("34989D8E-930A-4A95-AB042E6CFDFF6631", name="DELL_TCG_PEI_GUID"),
    UefiGuid("3B42EF57-16D3-44CB-86329FDB06B41451", name="DELL_MEMORY_INIT_GUID"),
    UefiGuid(
        "3E4817FD-2742-4351-B59F91493280329C",
        name="DELL_ALERT_STANDARD_FORMAT_PEI_GUID",
    ),
    UefiGuid("49EAB1A6-AF2F-4064-9B29D9655B3B8DED", name="DELL_AMT_CONFIG_PEI_GUID"),
    UefiGuid("4FE772E8-FE3E-4086-B6388C493C490488", name="DELL_OP_PRESENCE_GUID"),
    UefiGuid("5479E09C-2E74-481B-89F8B0172E388D1F", name="DELL_START_WATCH_DOG_GUID"),
    UefiGuid("5924BE03-9DD8-4BAB-808FC21CABFE0B4B", name="DELL_ERROR_HANDLER_PEI_GUID"),
    UefiGuid(
        "643DF777-F312-42ED-81CC1B1F57E18AD6", name="DELL_PCH_SMBUS_ARP_DISABLED_GUID"
    ),
    UefiGuid("6B4FDBD2-47E1-4A09-BA8E8E041F208B95", name="DELL_PCH_USB_GUID"),
    UefiGuid("70E65212-F3AD-495F-B3A17A63542ED7D5", name="DELL_TCG_PEI2_GUID"),
    UefiGuid("795477D9-1786-4D8A-8B893FA7623F7EF8", name="DELL_SB_POWER_OFF_PEI_GUID"),
    UefiGuid(
        "7D7789CD-FD20-4C05-A57991253D5E3D9E", name="DELL_CMOS_MANAGER_PEI_SRC_GUID"
    ),
    UefiGuid(
        "81F0BCF2-F1AD-4DDE-9E5B75EB3427ABC4", name="DELL_MFG_MODE_PEI_DRIVER_GUID"
    ),
    UefiGuid("92685943-D810-47FF-A112CC8490776A1F", name="DELL_CORE_PEI_GUID"),
    UefiGuid("9CF30325-DC5C-4556-A8B074215C5F7FC4", name="DELL_HECI_PEI_GUID"),
    UefiGuid("A0D7E505-5DD4-4459-A4D336119563C4F9", name="DELL_AUDIO_PEI_GUID"),
    UefiGuid("A27E7C62-249F-4B7B-BD5C807202035DEC", name="DELL_FLASH_UPDATE_PEI_GUID"),
    UefiGuid("B178E5AA-0876-420A-B40FE39B4E6EE05B", name="DELL_WARM_BOOT_PEI_GUID"),
    UefiGuid("B1AEE818-B959-487B-A79516C2A54CB36E", name="DELL_PEI_MAIN_GUID"),
    UefiGuid(
        "B68EFDCA-BEE2-4BF6-B86E45DC3CCBFE3C", name="DELL_DO_SIODIAG_LED_PEI_GUID"
    ),
    UefiGuid("B6A2AFF3-767C-5658-C37AD1C82EF76543", name="DELL_PCH_ME_UMA_GUID"),
    UefiGuid("C7FCCF43-5859-4ABE-BCB54C1626ED8B03", name="DELL_PEI_DIAG_LEDS_GUID"),
    UefiGuid("CA9D8617-D652-403B-B6C5BA47570116AD", name="DELL_TXT_PEI_GUID"),
    UefiGuid("CDB65801-494D-472E-A6A8AFF5B76E8D1F", name="DELL_TXT_CONFIG_PEI_GUID"),
    UefiGuid("D041CFC0-7670-435A-A512F45C923B285D", name="DELL_PEI_SIO_EMI_GUID"),
    UefiGuid(
        "DE8A5A2C-D788-47FB-A0B520CA8E58DFEC", name="DELL_SYSTEM_ID_CONFIG_PEI_GUID"
    ),
    UefiGuid("E5205B53-9758-44AB-A44DDB3BE41D6742", name="DELL_NB_CONFIG_PEI_GUID"),
    UefiGuid(
        "E5652F81-EC09-479C-BAFFE756F1829016", name="DELL_DIRTY_SHUTDOWN_PEI_GUID"
    ),
    UefiGuid(
        "E747D8FF-1794-48C6-96D7A419D9C60F11", name="DELL_SIO_POLICY_CONFIG_PEI_GUID"
    ),
    UefiGuid("E9A60F94-7A8B-45BA-9C323485526B5716", name="DELL_ACLOSS_PEI_GUID"),
    UefiGuid("E9B60F94-7A0B-48CD-9C888484526C5719", name="DELL_SB_CONFIG_PEI_GUID"),
    UefiGuid("F665C81D-EFDE-4B5F-88E82160B748D2B4", name="DELL_VIDEO_CONFIG_PEI_GUID"),
    UefiGuid("FB8415B7-EA7E-4E6D-9381005C3BD1DAD7", name="DELL_EC_CONFIG_PEI_GUID"),
    UefiGuid("FD236AE7-0791-48C4-B29E29BDEEE1A811", name="DELL_SA_INIT_PEIM_GUID"),
    UefiGuid("FD236AE7-0791-48C4-B29E29BDEEE1A838", name="DELL_PCH_INIT_PEIM_GUID"),
    UefiGuid("BB11ECFE-820F-4968-BBA6F76AFE302596", name="ARM_TOKEN_SPACE_GUID"),
    UefiGuid("A4EE0728-E5D7-4AC5-B21E658ED857E834", name="ARM_MP_CORE_INFO_GUID"),
    UefiGuid("D7E5ABE9-33AB-418E-9F9172DAE2BA8E2F", name="ARM_SCMI_BASE_PROTOCOL_GUID"),
    UefiGuid(
        "91CE67A8-E0AA-4012-B99FB6FCF3048EAA", name="ARM_SCMI_CLOCK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B8D8CAF2-9E94-462C-A8346C99FC05EFCF", name="ARM_SCMI_CLOCK2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "09B8BA84-3DD3-49A6-A05A3134A5F07BAD", name="ARM_SCMI_PERFORMANCE_PROTOCOL_GUID"
    ),
    UefiGuid("6847CC74-E9EC-4F8F-A29DAB44E754A8FC", name="ARM_MP_CORE_INFO_PPI_GUID"),
    UefiGuid(
        "9C0AAED4-74C5-4043-B417A3223814CE76", name="ARM_PLATFORM_TOKEN_SPACE_GUID"
    ),
    UefiGuid("0B6F5CA7-4F53-445A-B76E2E365B806366", name="ARM_VIRT_TOKEN_SPACE_GUID"),
    UefiGuid(
        "B199DEA9-FD5C-4A84-80822F4170780305", name="EARLY_P_L011_BASE_ADDRESS_GUID"
    ),
    UefiGuid("50BEA1E5-A2C5-46E9-9B3A59596516B00A", name="ARM_VIRT_VARIABLE_GUID"),
    UefiGuid("E11FACA0-4710-4C8E-A7A201BAA2591B4C", name="FDT_CLIENT_PROTOCOL_GUID"),
    UefiGuid(
        "D85A4835-5A82-4894-AC02706F43D5978E",
        name="EDKII_CONFIGURATION_MANAGER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "91D1E327-FE5A-49B8-AB650ECE2DDB45EC",
        name="EDKII_DYNAMIC_TABLE_FACTORY_PROTOCOL_GUID",
    ),
    UefiGuid("E0D8CA17-4276-4386-BB7948CB813D3C4F", name="EMBEDDED_TOKEN_SPACE_GUID"),
    UefiGuid("B1B621D5-F19C-41A5-830BD9152C69AAE0", name="FDT_TABLE_GUID"),
    UefiGuid("16958446-19B7-480B-B0477485AD3F716D", name="FDT_HOB_GUID"),
    UefiGuid("25A4FD4A-9703-4BA9-A190B7C84EFB3E57", name="FDT_VARIABLE_GUID"),
    UefiGuid(
        "7EBB920D-1AAF-46D9-B2AF541E1DCE148B",
        name="EDKII_PLATFORM_HAS_DEVICE_TREE_GUID",
    ),
    UefiGuid("2B7A240D-D5AD-4FD6-BE1CDFA4415F5526", name="DT_PLATFORM_FORM_SET_GUID"),
    UefiGuid(
        "25462CDA-221F-47DF-AC1D259CFAA4E326", name="DT_PLATFORM_DEFAULT_DTB_FILE_GUID"
    ),
    UefiGuid("2D2358B4-E96C-484D-B2DD7C2EDFC7D56F", name="CONSOLE_PREF_FORM_SET_GUID"),
    UefiGuid(
        "D1A86E3F-0707-4C35-83CDDC2C29C891A3", name="EDKII_NV_VAR_STORE_FORMATTED_GUID"
    ),
    UefiGuid(
        "2890B3EA-053D-1643-AD0CD64808DA3FF1", name="HARDWARE_INTERRUPT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "32898322-2DA1-474A-BAAAF3F7CF569470", name="HARDWARE_INTERRUPT2_PROTOCOL_GUID"
    ),
    UefiGuid("BF4B9D10-13EC-43DD-8880E90B718F27DE", name="EMBEDDED_DEVICE_GUID"),
    UefiGuid(
        "735F8C64-D696-44D0-BDF2447FD05A5406",
        name="EMBEDDED_EXTERNAL_DEVICE_PROTOCOL_GUID",
    ),
    UefiGuid("17A0A3D7-C0A5-4635-BBD5072187DFE2EE", name="EMBEDDED_GPIO_PROTOCOL_GUID"),
    UefiGuid(
        "B323179B-97FB-477E-B0FED88591FA11AB", name="PE_COFF_LOADER_PROTOCOL_GUID"
    ),
    UefiGuid("3E591C00-9E4A-11DF-92440002A5D5C51B", name="EFI_MMC_HOST_PROTOCOL_GUID"),
    UefiGuid(
        "74BD9FE0-8902-11E3-B9D3F72238FC9A31",
        name="ANDROID_FASTBOOT_TRANSPORT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "524685A0-89A0-11E3-9D4DBFA9F6A40308",
        name="ANDROID_FASTBOOT_PLATFORM_PROTOCOL_GUID",
    ),
    UefiGuid("021BD2CA-51D2-11E3-8E56B75417C70B44", name="USB_DEVICE_PROTOCOL_GUID"),
    UefiGuid(
        "7197C8A7-6559-4C93-93D58A84F988798B",
        name="PLATFORM_BOOT_MANAGER_PROTOCOL_GUID",
    ),
    UefiGuid("52CE9845-5AF4-43E2-BAFD230812547AC2", name="PLATFORM_GPIO_PROTOCOL_GUID"),
    UefiGuid(
        "0E3606D2-1DC3-4E6F-BE65394982A26547",
        name="PLATFORM_VIRTUAL_KEYBOARD_PROTOCOL_GUID",
    ),
    UefiGuid(
        "9859BB19-407C-4F8B-BCE1F8DA6565F4A5", name="ANDROID_BOOT_IMG_PROTOCOL_GUID"
    ),
    UefiGuid(
        "21C3B115-4E0B-470C-85C7E105A575C97B", name="EDKII_EMBEDDED_GPIO_PPI_GUID"
    ),
    UefiGuid("5CF32E0B-8EDF-2E44-9CDA93205E99EC1C", name="EMU_THUNK_PROTOCOL_GUID"),
    UefiGuid("453368F6-7C85-434A-A98A72D1B7FFA926", name="EMU_IO_THUNK_PROTOCOL_GUID"),
    UefiGuid(
        "30FD316A-6728-2E41-A6900D1333D8CAC1", name="EMU_GRAPHICS_WINDOW_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3B1E4B7C-09D8-944F-A4081309EB8B4427", name="EMU_THREAD_THUNK_PROTOCOL_GUID"
    ),
    UefiGuid("6888A4AE-AFCE-E84B-9102F7B9DAE6A030", name="EMU_BLOCK_IO_PROTOCOL_GUID"),
    UefiGuid("FD5FBE54-8C35-B345-8A0F7AC8A5FD0521", name="EMU_SNP_PROTOCOL_GUID"),
    UefiGuid("E113F896-75CF-F640-817FC85A79E8AE67", name="EMU_THUNK_PPI_GUID"),
    UefiGuid(
        "4F792E68-E8C8-794E-B1D83703F3F2D5A5", name="EMULATOR_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("F8626165-6CEB-924A-BAFCF13AB9D65728", name="EMU_SYSTEM_CONFIG_GUID"),
    UefiGuid("F2BA331A-8985-11DB-A4060040D02B1835", name="EMU_VIRTUAL_DISKS_GUID"),
    UefiGuid("F2BDCC96-8985-11DB-87190040D02B1835", name="EMU_PHYSICAL_DISKS_GUID"),
    UefiGuid(
        "40B2D964-FE11-40DC-82832EFBDA295356", name="FMP_DEVICE_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "487784C5-6299-4BA6-B0965CC5277CF757",
        name="EDKII_CAPSULE_UPDATE_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "CD167C1E-6E0B-42B3-82F6E3E906199810",
        name="FSP_READY_FOR_NOTIFY_PHASE_PPI_GUID",
    ),
    UefiGuid("A1EEAB87-C859-479D-89B51461F4061A3E", name="FSP_IN_API_MODE_PPI_GUID"),
    UefiGuid("824D5A3A-AF92-4C0C-9F1919526DCA4ABB", name="FSPM_ARCH_CONFIG_PPI_GUID"),
    UefiGuid("BC1CFBDB-7E50-42BE-B48722E0A90CB052", name="FSP_TEMP_RAM_EXIT_PPI_GUID"),
    UefiGuid(
        "ED6E0531-F715-4A3D-9B12C1CA5EF698A2", name="INTEL_FSP2_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("912740BE-2284-4734-B97184B027353F0C", name="FSP_HEADER_FILE_GUID"),
    UefiGuid(
        "69A79759-1373-4367-A6C4C7F59EFD986E",
        name="FSP_RESERVED_MEMORY_RESOURCE_HOB_GUID",
    ),
    UefiGuid(
        "721ACF02-4D77-4C2A-B3DC270B7BA9E4B0", name="FSP_NON_VOLATILE_STORAGE_HOB_GUID"
    ),
    UefiGuid(
        "73FF4F56-AA8E-4451-B31636353667AD44", name="FSP_BOOT_LOADER_TOLUM_HOB_GUID"
    ),
    UefiGuid("56ED21B6-BA23-429E-8932376D8E182EE3", name="FSP_PERFORMANCE_DATA_GUID"),
    UefiGuid(
        "BD44F629-EAE7-4198-87F139FAB0FD717E", name="FSP_EVENT_END_OF_FIRMWARE_GUID"
    ),
    UefiGuid(
        "A34CF082-0F50-4F0D-898A3D39302BC51E",
        name="INTEL_FSP2_WRAPPER_TOKEN_SPACE_GUID",
    ),
    UefiGuid("C9122295-56ED-4D4E-06A6508D894D3E40", name="FSP_API_PERFORMANCE_GUID"),
    UefiGuid("6D86FB36-BA90-472C-B5833FBED3FB209A", name="FSP_HOB_GUID"),
    UefiGuid(
        "4EB6E09C-D256-4E1E-B50A874BD284B3DE", name="FSP_SILICON_INIT_DONE_PPI_GUID"
    ),
    UefiGuid(
        "2F3962B2-57C5-44EC-9EFCA69FD302032B", name="TOP_OF_TEMPORARY_RAM_PPI_GUID"
    ),
    UefiGuid(
        "C4A58D6D-3677-49CB-A00A9470765FB55E", name="ADD_PERF_RECORD_PROTOCOL_GUID"
    ),
    UefiGuid(
        "A1AFF049-FDEB-442A-B32013AB4CB72BBC",
        name="EFI_MDE_MODULE_PKG_TOKEN_SPACE_GUID",
    ),
    UefiGuid("EA296D92-0B69-423C-8C2833B4E0A91268", name="PCD_DATA_BASE_HOB_GUID"),
    UefiGuid(
        "3C7D193C-682C-4C14-A68F552DEA4F437E", name="PCD_DATA_BASE_SIGNATURE_GUID"
    ),
    UefiGuid("0F0B1735-87A0-4193-B266538C38AF48CE", name="EFI_IFR_TIANO_GUID"),
    UefiGuid("82DDD68B-9163-4187-9B2720A8FD60A71D", name="EDKII_IFR_BIT_VARSTORE_GUID"),
    UefiGuid("31CA5D1A-D511-4931-B782AE6B2B178CD7", name="EFI_IFR_FRAMEWORK_GUID"),
    UefiGuid("FFF12B8D-7696-4C8B-A9852747075B4F50", name="EFI_SYSTEM_NV_DATA_FV_GUID"),
    UefiGuid(
        "9E58292B-7C68-497D-A0CE6500FD9F1B95", name="EDKII_WORKING_BLOCK_SIGNATURE_GUID"
    ),
    UefiGuid(
        "1D3E9CB8-43AF-490B-830A3516AA532047", name="EDKII_FAULT_TOLERANT_WRITE_GUID"
    ),
    UefiGuid("D3B36F2C-D551-11D4-9A460090273FC14D", name="EFI_CONSOLE_OUT_DEVICE_GUID"),
    UefiGuid("D3B36F2B-D551-11D4-9A460090273FC14D", name="EFI_CONSOLE_IN_DEVICE_GUID"),
    UefiGuid(
        "4C19049F-4137-4DD3-9C108B97A83FFDFA", name="EFI_MEMORY_TYPE_INFORMATION_GUID"
    ),
    UefiGuid("711C703F-C285-4B10-A3B036ECBD3C8BE2", name="EFI_CAPSULE_VENDOR_GUID"),
    UefiGuid(
        "D3B36F2D-D551-11D4-9A460090273FC14D", name="EFI_STANDARD_ERROR_DEVICE_GUID"
    ),
    UefiGuid("DDCF3616-3275-4164-98B6FE85707FFE7D", name="EFI_VARIABLE_GUID"),
    UefiGuid(
        "AAF32C78-947B-439A-A1802E144EC37792", name="EFI_AUTHENTICATED_VARIABLE_GUID"
    ),
    UefiGuid(
        "8CFDB8C8-D6B2-40F3-8E9702307CC98B7C", name="EFI_VARIABLE_INDEX_TABLE_GUID"
    ),
    UefiGuid("93BA1826-DFFB-45DD-82A7E7DCAA3BBDF3", name="SMM_VARIABLE_WRITE_GUID"),
    UefiGuid("76B6BDFA-2ACD-4462-9E3FCB58C969D937", name="PERFORMANCE_PROTOCOL_GUID"),
    UefiGuid(
        "F866226A-EAA5-4F5A-A90A6CFBA57C588E", name="SMM_PERFORMANCE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "1EA81BEC-F01A-4D98-A2014A61CE2FC022", name="PERFORMANCE_EX_PROTOCOL_GUID"
    ),
    UefiGuid(
        "931FC048-C71D-4455-8930470630E30EE5", name="SMM_PERFORMANCE_EX_PROTOCOL_GUID"
    ),
    UefiGuid(
        "C85D06BE-5F75-48CE-A80F1236BA3B87B1",
        name="EDKII_PERFORMANCE_MEASUREMENT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "D56B6D73-1A7B-4015-9BB47B071729ED24",
        name="EDKII_SMM_PERFORMANCE_MEASUREMENT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "FC1BCDB0-7D31-49AA-936AA4600D9DD083",
        name="EFI_CRC32_GUIDED_SECTION_EXTRACTION_GUID",
    ),
    UefiGuid("E701458C-4900-4CA5-B7723D37949F7927", name="STATUS_CODE_CALLBACK_GUID"),
    UefiGuid(
        "060CC026-4C0D-4DDA-8F41595FEF00A502", name="MEMORY_STATUS_CODE_RECORD_GUID"
    ),
    UefiGuid(
        "9A4E9246-D553-11D5-87E200062945C3B9",
        name="EFI_STATUS_CODE_DATA_TYPE_DEBUG_GUID",
    ),
    UefiGuid(
        "2CA88B53-D296-4080-A4A5CAD9BAE24B09",
        name="LOAD_FIXED_ADDRESS_CONFIGURATION_TABLE_GUID",
    ),
    UefiGuid("59D1C24F-50F1-401A-B101F33E0DAED443", name="EFI_GENERIC_VARIABLE_GUID"),
    UefiGuid("3C8D294C-5FC3-4451-BB31C4C032295E6C", name="IDLE_LOOP_EVENT_GUID"),
    UefiGuid(
        "0FFBCE19-324C-4690-A00998C6AE2EB186", name="RECOVERY_ON_FAT_USB_DISK_GUID"
    ),
    UefiGuid(
        "B38573B6-6200-4AC5-B51D82E65938D783", name="RECOVERY_ON_FAT_IDE_DISK_GUID"
    ),
    UefiGuid(
        "2E3D2E75-9B2E-412D-B4B170416B8700FF", name="RECOVERY_ON_FAT_FLOPPY_DISK_GUID"
    ),
    UefiGuid("5CAC0099-0DC9-48E5-8068BB95F5400A9F", name="RECOVERY_ON_DATA_CD_GUID"),
    UefiGuid(
        "C770A27F-956A-497A-8548E06197588BF6", name="RECOVERY_ON_FAT_NVME_DISK_GUID"
    ),
    UefiGuid(
        "2A3CFEBD-27E8-4D0A-8B79D688C2A3E1C0",
        name="EFI_SMM_LOCK_BOX_COMMUNICATION_GUID",
    ),
    UefiGuid("AF9FFD67-EC10-488A-9DFC6CBF5EE22C2E", name="EFI_ACPI_VARIABLE_GUID"),
    UefiGuid("0EF98D3A-3E33-497A-A40177BE3EB74F38", name="EFI_ACPI_S3_CONTEXT_GUID"),
    UefiGuid(
        "3079818C-46D4-4A73-AEF3E3E46CF1EEDB",
        name="EFI_BOOT_SCRIPT_EXECUTOR_VARIABLE_GUID",
    ),
    UefiGuid(
        "79CB58C4-AC51-442F-AFD798E47D2E9908",
        name="EFI_BOOT_SCRIPT_EXECUTOR_CONTEXT_GUID",
    ),
    UefiGuid(
        "0C0F3B43-44DE-4907-B478225F6F6289DC", name="USB_KEYBOARD_LAYOUT_PACKAGE_GUID"
    ),
    UefiGuid(
        "3A4D7A7C-018A-4B42-81B3DC10E3B591BD", name="USB_KEYBOARD_LAYOUT_KEY_GUID"
    ),
    UefiGuid(
        "4F4EF7F0-AA29-4CE9-BA41643E0123A99F", name="HII_RESOURCE_SAMLE_FORM_SET_GUID"
    ),
    UefiGuid("A04A27F4-DF00-4D42-B55239511302113D", name="DRIVER_SAMPLE_FORM_SET_GUID"),
    UefiGuid(
        "B3F56470-6141-4621-8F19704E577AA9E8", name="DRIVER_SAMPLE_INVENTORY_GUID"
    ),
    UefiGuid("F5E655D9-02A6-46F2-9E76B8BE8E60AB22", name="EFI_IFR_REFRESH_ID_OP_GUID"),
    UefiGuid(
        "8614567D-35BE-4415-8D88BD7D0C9C70C0", name="PLATFORM_OVERRIDES_MANAGER_GUID"
    ),
    UefiGuid("00000000-0000-0000-0000000000000000", name="ZERO_GUID"),
    UefiGuid("EB704011-1402-11D3-8E7700A0C969723B", name="MTC_VENDOR_GUID"),
    UefiGuid(
        "C095791A-3001-47B2-80C9EAC7319F2FA4", name="EFI_FIRMWARE_PERFORMANCE_GUID"
    ),
    UefiGuid(
        "0DC65ADC-A973-4130-8DF02ADBEB9E4A31",
        name="FIRMWARE_PERFORMANCE_S3_POINTER_GUID",
    ),
    UefiGuid(
        "4F6C5507-232F-4787-B95E72F862490CB1",
        name="EVENT_EXIT_BOOT_SERVICES_FAILED_GUID",
    ),
    UefiGuid("DB4E8151-57ED-4BED-88336751B5D1A8D7", name="CONNECT_CON_IN_EVENT_GUID"),
    UefiGuid(
        "F6EE6DBB-D67F-4EA0-8B966A71B19D84AD",
        name="EDKII_STATUS_CODE_DATA_TYPE_VARIABLE_GUID",
    ),
    UefiGuid("821C9A09-541A-40F6-9F430AD193A12CFE", name="EDKII_MEMORY_PROFILE_GUID"),
    UefiGuid(
        "E22BBCCA-516A-46A8-80E26745E83693BD", name="EDKII_SMM_MEMORY_PROFILE_GUID"
    ),
    UefiGuid("04B37FE8-F6AE-480B-BDD537D98C5E89AA", name="EDKII_VAR_ERROR_FLAG_GUID"),
    UefiGuid(
        "3D532050-5CDA-4FD0-879E0F7F630D5AFB", name="BROTLI_CUSTOM_DECOMPRESS_GUID"
    ),
    UefiGuid("EE4E5898-3914-4259-9D6EDC7BD79403CF", name="LZMA_CUSTOM_DECOMPRESS_GUID"),
    UefiGuid(
        "D42AE6BD-1352-4BFB-909ACA72A6EAE889", name="LZMA_F86_CUSTOM_DECOMPRESS_GUID"
    ),
    UefiGuid("7D916D80-5BB1-458C-A48FE25FDD51EF94", name="EFI_TTY_TERM_GUID"),
    UefiGuid("E4364A7F-F825-430E-9D3A9C9BE6817CA5", name="EDKII_LINUX_TERM_GUID"),
    UefiGuid("FBFCA56B-BB36-4B78-AAABBE1B97EC7CCB", name="EDKII_XTERM_R6_GUID"),
    UefiGuid("8E46DDDD-3D49-4A9D-B8753C086F6AA2BD", name="EDKII_V_T400_GUID"),
    UefiGuid("FC7DD6E0-813C-434D-B4DA3BD649E9E15A", name="EDKII_S_C_O_TERM_GUID"),
    UefiGuid(
        "B2DEDC91-D59F-48D2-898A12490C74A4E0", name="EFI_IFR_BOOT_MAINTENANCE_GUID"
    ),
    UefiGuid("E58809F8-FBC1-48E2-883AA30FDC4B441E", name="EFI_IFR_FRONT_PAGE_GUID"),
    UefiGuid("2A46715F-3581-4A55-8E732B769AAA30C5", name="RAM_DISK_FORM_SET_GUID"),
    UefiGuid(
        "4E28CA50-D582-44AC-A11FE3D56526DB34",
        name="EDKII_PI_SMM_COMMUNICATION_REGION_TABLE_GUID",
    ),
    UefiGuid(
        "6B9FD3F7-16DF-45E8-BD39B94A66541A5D",
        name="EDKII_PI_SMM_MEMORY_ATTRIBUTES_TABLE_GUID",
    ),
    UefiGuid("49174342-7108-409B-8BBE65FDA85389F5", name="SMI_HANDLER_PROFILE_GUID"),
    UefiGuid(
        "C7D35798-E4D2-4A93-B14554889F02584B",
        name="EDKII_NON_DISCOVERABLE_AHCI_DEVICE_GUID",
    ),
    UefiGuid(
        "94440339-CC93-4506-B4C6EE8D0F4CA191",
        name="EDKII_NON_DISCOVERABLE_AMBA_DEVICE_GUID",
    ),
    UefiGuid(
        "EAEE5615-0CFD-45FC-8769A0D85695AF85",
        name="EDKII_NON_DISCOVERABLE_EHCI_DEVICE_GUID",
    ),
    UefiGuid(
        "C5F25542-2A79-4A26-81BB4EA63233B309",
        name="EDKII_NON_DISCOVERABLE_NVME_DEVICE_GUID",
    ),
    UefiGuid(
        "B20005B0-BB2D-496F-869C230B4479E7D1",
        name="EDKII_NON_DISCOVERABLE_OHCI_DEVICE_GUID",
    ),
    UefiGuid(
        "1DD1D619-F9B8-463E-8681D1DC7C07B72C",
        name="EDKII_NON_DISCOVERABLE_SDHCI_DEVICE_GUID",
    ),
    UefiGuid(
        "2EA77912-80A8-4947-BE69CDD00AFBE556",
        name="EDKII_NON_DISCOVERABLE_UFS_DEVICE_GUID",
    ),
    UefiGuid(
        "A8CDA0A2-4F37-4A1B-8E108EF3CC3BF3A8",
        name="EDKII_NON_DISCOVERABLE_UHCI_DEVICE_GUID",
    ),
    UefiGuid(
        "B1BE0BC5-6C28-442D-AA37151B4257BD78",
        name="EDKII_NON_DISCOVERABLE_XHCI_DEVICE_GUID",
    ),
    UefiGuid(
        "F0966B41-C23F-41B9-96040FF7E111965A", name="EDKII_PLATFORM_HAS_ACPI_GUID"
    ),
    UefiGuid(
        "3B387BFD-7ABC-4CF2-A0CAB6A16C1B1B25",
        name="EDKII_FPDT_EXTENDED_FIRMWARE_PERFORMANCE_GUID",
    ),
    UefiGuid("96F5296D-05F7-4F3C-8467E456890E0CB5", name="EDKII_END_OF_S3_RESUME_GUID"),
    UefiGuid("8F9D4825-797D-48FC-8471845025792EF6", name="EDKII_S3_SMM_INIT_DONE_GUID"),
    UefiGuid(
        "310E9B8C-CF90-421E-8E9B9EEFB617C8EF", name="S3_STORAGE_DEVICE_INIT_LIST_GUID"
    ),
    UefiGuid(
        "D3987D4B-971A-435F-8CAF4967EB627241", name="EDKII_SERIAL_PORT_LIB_VENDOR_GUID"
    ),
    UefiGuid(
        "98C80A4F-E16B-4D11-939AABE561260330", name="EDKII_CAPSULE_ON_DISK_NAME_GUID"
    ),
    UefiGuid("A45E60D1-C719-44AA-B07AAA777F85906D", name="PEI_ATA_CONTROLLER_PPI_GUID"),
    UefiGuid(
        "652B38A9-77F4-453F-89D5E7BDC352FC53", name="PEI_USB_HOST_CONTROLLER_PPI_GUID"
    ),
    UefiGuid(
        "FEDD6305-E2D7-4ED5-9FAADA080E336C22", name="PEI_USB2_HOST_CONTROLLER_PPI_GUID"
    ),
    UefiGuid("3BC1F6DE-693E-4547-A30021823CA420B2", name="PEI_USB_CONTROLLER_PPI_GUID"),
    UefiGuid("7C29785C-66B9-49FC-B7971CA5550EF283", name="PEI_USB_IO_PPI_GUID"),
    UefiGuid(
        "0ECC666B-4662-47F9-9DD5D096FF7DA49E", name="PEI_SEC_PERFORMANCE_PPI_GUID"
    ),
    UefiGuid(
        "AE933E1C-CC47-4E38-8F0EE2F61D2605DF", name="EFI_PEI_SMM_COMMUNICATION_PPI_GUID"
    ),
    UefiGuid("268F33A9-CCCD-48BE-881786053AC32ED6", name="PEI_SMM_ACCESS_PPI_GUID"),
    UefiGuid("61C68702-4D7E-4F43-8DEFA74305CE74C5", name="PEI_SMM_CONTROL_PPI_GUID"),
    UefiGuid(
        "88C9D306-0900-4EB5-82603E2DBEDA1F89", name="PEI_POST_SCRIPT_TABLE_PPI_GUID"
    ),
    UefiGuid("490E9D85-8AEF-4193-8E56F734A9FFAC8B", name="PEI_SERIAL_PORT_PPI_GUID"),
    UefiGuid(
        "DC54B283-1A77-4CD6-83BBFDDA469A2EC6",
        name="EDKII_PEI_UFS_HOST_CONTROLLER_PPI_GUID",
    ),
    UefiGuid("A9731431-D968-4277-B752A3A9A6AE1898", name="PEI_IPMI_PPI_GUID"),
    UefiGuid(
        "B30DFEED-947F-4396-B15ADFBDB916DC24",
        name="EDKII_PEI_SD_MMC_HOST_CONTROLLER_PPI_GUID",
    ),
    UefiGuid("70B0AF26-F847-4BB6-AAB9CDE84FC61431", name="EDKII_IO_MMU_PPI_GUID"),
    UefiGuid(
        "8C9F4DE3-7B90-47EF-9308287CECD66DE8",
        name="EDKII_PLATFORM_SPECIFIC_RESET_FILTER_PPI_GUID",
    ),
    UefiGuid(
        "E09F355D-DAE8-4910-B14A92780FDCF7CB",
        name="EDKII_PLATFORM_SPECIFIC_RESET_NOTIFICATION_PPI_GUID",
    ),
    UefiGuid(
        "75CF14AE-3441-49DC-AA10BB35A7BA8BAB",
        name="EDKII_PLATFORM_SPECIFIC_RESET_HANDLER_PPI_GUID",
    ),
    UefiGuid(
        "CAE3AA63-676F-4DA3-BD506CC5EDDE9AAD",
        name="EDKII_PEI_NVM_EXPRESS_HOST_CONTROLLER_PPI_GUID",
    ),
    UefiGuid(
        "61DD33EA-421F-4CC0-8929FFEEA9A1A261",
        name="EDKII_PEI_ATA_AHCI_HOST_CONTROLLER_PPI_GUID",
    ),
    UefiGuid(
        "35DE0B4E-30FB-46C3-BD841FDBA158BB56",
        name="EDKII_PEI_STORAGE_SECURITY_COMMAND_PPI_GUID",
    ),
    UefiGuid(
        "A16473FD-D474-4C89-AEC790B83C738609", name="EDKII_PEI_ATA_PASS_THRU_PPI_GUID"
    ),
    UefiGuid("999E699C-B013-475E-B17BF3A8AE5C4875", name="EDKII_DEBUG_PPI_GUID"),
    UefiGuid(
        "6AF31B2C-03BE-46C1-B12DEA4A36DFA74C",
        name="EDKII_PEI_NVM_EXPRESS_PASS_THRU_PPI_GUID",
    ),
    UefiGuid(
        "71A9EA61-5A35-4A5D-ACEF9CF86D6D67E0", name="EDKII_PEI_CAPSULE_ON_DISK_PPI_GUID"
    ),
    UefiGuid(
        "B08A11E4-E2B7-4B75-B515AF610668BFD1",
        name="EDKII_PEI_BOOT_IN_CAPSULE_ON_DISK_MODE_PPI_GUID",
    ),
    UefiGuid(
        "5CB5C776-60D5-45EE-883C452708CD743F", name="EFI_LOAD_PE_IMAGE_PROTOCOL_GUID"
    ),
    UefiGuid("F05976EF-83F1-4F3D-8619F7595D41E538", name="EFI_PRINT2_PROTOCOL_GUID"),
    UefiGuid("0CC252D2-C106-4661-B5BD3147A4F81F92", name="EFI_PRINT2_S_PROTOCOL_GUID"),
    UefiGuid(
        "309DE7F1-7F5E-4ACE-B49C531BE5AA95EF", name="EFI_GENERIC_MEM_TEST_PROTOCOL_GUID"
    ),
    UefiGuid(
        "577D959C-E967-4546-8620C778FAE5DA05",
        name="EFI_DEBUGGER_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "3EBD9E82-2C78-4DE6-97868D4BFCB7C881",
        name="EFI_FAULT_TOLERANT_WRITE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "3868FC3B-7E45-43A7-906C4BA47DE1754D",
        name="EFI_SMM_FAULT_TOLERANT_WRITE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "1259F60D-B754-468E-A7894DB85D55E87E",
        name="EFI_SWAP_ADDRESS_RANGE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "67C4F112-3385-4E55-9C5BC05B717C4228",
        name="EFI_SMM_SWAP_ADDRESS_RANGE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "ED32D533-99E6-4209-9CC02D72CDD998A7", name="EFI_SMM_VARIABLE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "CD3D0A05-9E24-437C-A8911EE053DB7638", name="EDKII_VARIABLE_LOCK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "AF23B340-97B4-4685-8D4FA3F28169B21D", name="EDKII_VAR_CHECK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B0D8F3C1-B7DE-4C11-BC892FB562C8C411", name="EDKII_SMM_VAR_CHECK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "D326D041-BD31-4C01-B5A8628BE87F0653",
        name="EFI_SMM_FIRMWARE_VOLUME_BLOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "4C8A2451-C207-405B-969499EA13251341", name="EFI_DEBUG_MASK_PROTOCOL_GUID"
    ),
    UefiGuid("BD445D79-B7AD-4F04-9AD829BD2040EB3C", name="EFI_LOCK_BOX_PROTOCOL_GUID"),
    UefiGuid(
        "1F73B18D-4630-43C1-A1DE6F80855D7DA4", name="EFI_FORM_BROWSER_EX_PROTOCOL_GUID"
    ),
    UefiGuid(
        "1F73B18D-4630-43C1-A1DE6F80855D7DA4",
        name="EDKII_FORM_BROWSER_EX_PROTOCOL_GUID",
    ),
    UefiGuid(
        "AAEACCFD-F27B-4C17-B61075CA1F2DFB52", name="EFI_EBC_VM_TEST_PROTOCOL_GUID"
    ),
    UefiGuid(
        "2A72D11E-7376-40F6-9C6823FA2FE363F1",
        name="EFI_EBC_SIMPLE_DEBUGGER_PROTOCOL_GUID",
    ),
    UefiGuid("CDEA2BD3-FC25-4C1C-B97CB31186064990", name="EFI_BOOT_LOGO_PROTOCOL_GUID"),
    UefiGuid(
        "4B5DC1DF-1EAA-48B2-A7E9EAC489A00B5C", name="EDKII_BOOT_LOGO2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "9BBE29E9-FDA1-41EC-AD52452213742D2E",
        name="EDKII_FORM_DISPLAY_ENGINE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A770C357-B693-4E6D-A6CFD21C728E550B",
        name="EDKII_FORM_BROWSER_EX2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "EBC01AF5-07A9-489E-B7CEDC089E459B2F",
        name="EDKII_UFS_HOST_CONTROLLER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "3D18BA13-D9B1-4DD4-B916D30796539ED8",
        name="EDKII_UFS_HC_PLATFORM_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A340C064-723C-4A9C-A4DDD5B47A26FBB0", name="ESRT_MANAGEMENT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "296EB418-C4C8-4E05-AB5939E8AF56F00A",
        name="EDKII_SMM_EXIT_BOOT_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "85A8AB57-0644-4110-850F981322047070",
        name="EDKII_SMM_LEGACY_BOOT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "6E057ECF-FA99-4F39-95BC59F9921D17E4",
        name="EDKII_SMM_READY_TO_BOOT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "53CD299F-2BC1-40C0-8C0723F64FDB30E0", name="EDKII_PLATFORM_LOGO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "2C03C536-4594-4515-9E7AD3D204FE1363", name="EFI_FILE_EXPLORER_PROTOCOL_GUID"
    ),
    UefiGuid("DBC6381F-5554-4D14-8FFD76D787B8ACBF", name="IPMI_PROTOCOL_GUID"),
    UefiGuid("5169AF60-8C5A-4243-B3E956C56D18EE26", name="SMM_IPMI_PROTOCOL_GUID"),
    UefiGuid(
        "4DF19259-DC71-4D46-BEF1357BB578C418", name="EFI_PS2_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0D51905B-B77E-452A-A2C0ECA0CC8D514A",
        name="EDKII_NON_DISCOVERABLE_DEVICE_PROTOCOL_GUID",
    ),
    UefiGuid("4E939DE9-D948-4B0F-88EDE6E1CE517C1E", name="EDKII_IO_MMU_PROTOCOL_GUID"),
    UefiGuid(
        "5D6B38C8-5510-4458-B48D9581CFA7B00D",
        name="EDKII_DEVICE_SECURITY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "2509B2F1-A022-4CCA-AF70F9D321FB6649",
        name="EDKII_DEVICE_IDENTIFIER_TYPE_PCI_GUID",
    ),
    UefiGuid(
        "7394F350-394D-488C-BB750CAB7B120AC5",
        name="EDKII_DEVICE_IDENTIFIER_TYPE_USB_GUID",
    ),
    UefiGuid(
        "69B792EA-39CE-402D-A2A6F721DE351DFE",
        name="EDKII_SMM_MEMORY_ATTRIBUTE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "EAF9E3C1-C9CD-46DB-A5E55A124C832323",
        name="EDKII_SD_MMC_OVERRIDE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "695D7835-8D47-4C11-AB22FA8ACCE7AE7A",
        name="EDKII_PLATFORM_SPECIFIC_RESET_FILTER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "2DF6BA0B-7092-440D-BD04FB091EC3F3C1",
        name="EDKII_PLATFORM_SPECIFIC_RESET_HANDLER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "1849BDA2-6952-4E86-A1DB559A3C479DF1",
        name="EDKII_FIRMWARE_MANAGEMENT_PROGRESS_PROTOCOL_GUID",
    ),
    UefiGuid(
        "E59CD769-5083-4F26-90946C919F916C4E",
        name="EDKII_ATA_ATAPI_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "96F46153-97A7-4793-ACC1FA19BF78EA97",
        name="EDKII_PE_COFF_IMAGE_EMULATOR_PROTOCOL_GUID",
    ),
    UefiGuid(
        "AA17ADD4-756C-460D-94B84388D7FB3E59",
        name="EDKII_PLATFORM_BOOT_MANAGER_PROTOCOL_GUID",
    ),
    UefiGuid("8BE4DF61-93CA-11D2-AA0D00E098032B8C", name="EFI_GLOBAL_VARIABLE_GUID"),
    UefiGuid("7BAEC70B-57E0-4C76-8E872F9E28088343", name="EFI_V_T100_PLUS_GUID"),
    UefiGuid("DFA66065-B419-11D3-9A2D0090273FC14D", name="EFI_V_T100_GUID"),
    UefiGuid("E0C14753-F9BE-11D2-9A0C0090273FC14D", name="EFI_PC_ANSI_GUID"),
    UefiGuid("AD15A0D6-8BEC-4ACF-A073D01DE77E2D88", name="EFI_V_T_U_T_F8_GUID"),
    UefiGuid("37499A9D-542F-4C89-A02635DA142094E4", name="EFI_UART_DEVICE_PATH_GUID"),
    UefiGuid("D487DDB4-008B-11D9-AFDC001083FFCA4D", name="EFI_SAS_DEVICE_PATH_GUID"),
    UefiGuid(
        "024DEE41-33E7-11D3-9D690008C781F39F", name="EFI_PART_TYPE_LEGACY_MBR_GUID"
    ),
    UefiGuid(
        "C12A7328-F81F-11D2-BA4B00A0C93EC93B", name="EFI_PART_TYPE_SYSTEM_PART_GUID"
    ),
    UefiGuid("00000000-0000-0000-0000000000000000", name="EFI_PART_TYPE_UNUSED_GUID"),
    UefiGuid(
        "49152E77-1ADA-4764-B7A27AFEFED95E8B", name="EFI_DEBUG_IMAGE_INFO_TABLE_GUID"
    ),
    UefiGuid("8868E871-E4F1-11D3-BC220080C73C8881", name="EFI_ACPI_TABLE_GUID"),
    UefiGuid("8868E871-E4F1-11D3-BC220080C73C8881", name="EFI_ACPI20_TABLE_GUID"),
    UefiGuid("EB9D2D30-2D88-11D3-9A160090273FC14D", name="EFI_ACPI10_TABLE_GUID"),
    UefiGuid("EB9D2D31-2D88-11D3-9A160090273FC14D", name="EFI_SMBIOS_TABLE_GUID"),
    UefiGuid("EB9D2D2F-2D88-11D3-9A160090273FC14D", name="EFI_MPS_TABLE_GUID"),
    UefiGuid(
        "C280C73E-15CA-11DA-B0CA001083FFCA4D", name="EFI_AUTHENTICATION_CHAP_LOCAL_GUID"
    ),
    UefiGuid(
        "D6062B50-15CA-11DA-9219001083FFCA4D",
        name="EFI_AUTHENTICATION_CHAP_RADIUS_GUID",
    ),
    UefiGuid(
        "DB47D7D3-FE81-11D3-9A350090273FC14D",
        name="EFI_FILE_SYSTEM_VOLUME_LABEL_INFO_ID_GUID",
    ),
    UefiGuid("09576E93-6D3F-11D2-8E3900A0C969723B", name="EFI_FILE_SYSTEM_INFO_GUID"),
    UefiGuid("09576E92-6D3F-11D2-8E3900A0C969723B", name="EFI_FILE_INFO_GUID"),
    UefiGuid(
        "EDD35E31-07B9-11D2-83A300A0C91FADCF",
        name="BOOT_OBJECT_AUTHORIZATION_PARMSET_GUID",
    ),
    UefiGuid(
        "345ECC0E-0CB6-4B75-BB571B129C47333E",
        name="EFI_PLATFORM_TO_DRIVER_CONFIGURATION_CLP_GUID",
    ),
    UefiGuid(
        "14982A4F-B0ED-45B8-A8115A7A9BC232DF", name="EFI_HII_KEY_BOARD_LAYOUT_GUID"
    ),
    UefiGuid(
        "0AF7C79C-65B5-4319-B0AE44EC484E4AD7", name="EFI_HASH_ALGORITHM_M_D5_GUID"
    ),
    UefiGuid(
        "CAA4381E-750C-4770-B8707A23B4E42130", name="EFI_HASH_ALGORITHM_SHA512_GUID"
    ),
    UefiGuid(
        "EFA96432-DE33-4DD2-AEE6328C33DF777A", name="EFI_HASH_ALGORITHM_SHA384_GUID"
    ),
    UefiGuid(
        "51AA59DE-FDF2-4EA3-BC63875FB7842EE9", name="EFI_HASH_ALGORITHM_SHA256_GUID"
    ),
    UefiGuid(
        "8DF01A06-9BD5-4BF7-B021DB4FD9CCF45B", name="EFI_HASH_ALGORITHM_SHA224_GUID"
    ),
    UefiGuid(
        "2AE9D80F-3FB2-4095-B7B1E93157B946B6", name="EFI_HASH_ALGORITHM_SHA1_GUID"
    ),
    UefiGuid(
        "7CE88FB3-4BD7-4679-87A8A8D8DEE50D2B", name="EFI_EVENT_READY_TO_BOOT_GUID"
    ),
    UefiGuid(
        "78BEE926-692F-48FD-9EDB01422EF0D7AB", name="EFI_EVENT_MEMORY_MAP_CHANGE_GUID"
    ),
    UefiGuid(
        "13FA7698-C831-49C7-87EA8F43FCC25196",
        name="EFI_EVENT_VIRTUAL_ADDRESS_CHANGE_GUID",
    ),
    UefiGuid(
        "27ABF055-B1B8-4C26-8048748F37BAA2DF", name="EFI_EVENT_EXIT_BOOT_SERVICES_GUID"
    ),
    UefiGuid(
        "EBA4E8D2-3858-41EC-A2812647BA9660D0", name="EFI_DEBUG_PORT_VARIABLE_GUID"
    ),
    UefiGuid(
        "EBA4E8D2-3858-41EC-A2812647BA9660D0", name="EFI_DEBUG_PORT_DEVICE_PATH_GUID"
    ),
    UefiGuid(
        "93039971-8545-4B04-B45E32EB8326040E",
        name="EFI_HII_PLATFORM_SETUP_FORMSET_GUID",
    ),
    UefiGuid(
        "F22FC20C-8CF4-45EB-8E06AD4E50B95DD3", name="EFI_HII_DRIVER_HEALTH_FORMSET_GUID"
    ),
    UefiGuid(
        "337F4407-5AEE-4B83-B2A74EADCA3088CD",
        name="EFI_HII_USER_CREDENTIAL_FORMSET_GUID",
    ),
    UefiGuid("3BD2F4EC-E524-46E4-A9D8510117425562", name="EFI_HII_STANDARD_FORM_GUID"),
    UefiGuid(
        "E20939BE-32D4-41BE-A150897F85D49829",
        name="EFI_MEMORY_OVERWRITE_CONTROL_DATA_GUID",
    ),
    UefiGuid(
        "BB983CCF-151D-40E1-A07B4A17BE168292",
        name="EFI_MEMORY_OVERWRITE_REQUEST_CONTROL_LOCK_GUID",
    ),
    UefiGuid(
        "A7717414-C616-4977-9420844712A735BF", name="EFI_CERT_TYPE_RSA2048_SHA256_GUID"
    ),
    UefiGuid(
        "2DCE8BB1-BDD7-450E-B9AD9CF4EBD4F890",
        name="EFI_EVENT_NOTIFICATION_TYPE_CMC_GUID",
    ),
    UefiGuid(
        "4E292F96-D843-4A55-A8C2D481F27EBEEE",
        name="EFI_EVENT_NOTIFICATION_TYPE_CPE_GUID",
    ),
    UefiGuid(
        "E8F56FFE-919C-4CC5-BA8865ABE14913BB",
        name="EFI_EVENT_NOTIFICATION_TYPE_MCE_GUID",
    ),
    UefiGuid(
        "CF93C01F-1A16-4DFC-B8BC9C4DAF67C104",
        name="EFI_EVENT_NOTIFICATION_TYPE_PCIE_GUID",
    ),
    UefiGuid(
        "CC5263E8-9308-454A-89D0340BD39BC98E",
        name="EFI_EVENT_NOTIFICATION_TYPE_INIT_GUID",
    ),
    UefiGuid(
        "5BAD89FF-B7E6-42C9-814ACF2485D6E98A",
        name="EFI_EVENT_NOTIFICATION_TYPE_NMI_GUID",
    ),
    UefiGuid(
        "3D61A466-AB40-409A-A698F362D464B38F",
        name="EFI_EVENT_NOTIFICATION_TYPE_BOOT_GUID",
    ),
    UefiGuid(
        "667DD791-C6B3-4C27-8A6B0F8E722DEB41",
        name="EFI_EVENT_NOTIFICATION_TYPE_DMAR_GUID",
    ),
    UefiGuid(
        "9876CCAD-47B4-4BDB-B65E16F193C4F3DB",
        name="EFI_PROCESSOR_GENERIC_ERROR_SECTION_GUID",
    ),
    UefiGuid(
        "DC3EA0B0-A144-4797-B95B53FA242B6E1D",
        name="EFI_PROCESSOR_SPECIFIC_ERROR_SECTION_GUID",
    ),
    UefiGuid(
        "DC3EA0B0-A144-4797-B95B53FA242B6E1D",
        name="EFI_IA32_X64_PROCESSOR_ERROR_SECTION_GUID",
    ),
    UefiGuid(
        "A5BC1114-6F64-4EDE-B8633E83ED7C83B1",
        name="EFI_PLATFORM_MEMORY_ERROR_SECTION_GUID",
    ),
    UefiGuid("D995E954-BBC1-430F-AD91B44DCB3C6F35", name="EFI_PCIE_ERROR_SECTION_GUID"),
    UefiGuid(
        "81212A96-09ED-4996-94718D729C8E69ED", name="EFI_FIRMWARE_ERROR_SECTION_GUID"
    ),
    UefiGuid(
        "C5753963-3B84-4095-BF78EDDAD3F9C9DD", name="EFI_PCI_BUS_ERROR_SECTION_GUID"
    ),
    UefiGuid(
        "EB5E4685-CA66-4769-B6A226068B001326", name="EFI_PCI_DEV_ERROR_SECTION_GUID"
    ),
    UefiGuid(
        "5B51FEF7-C79D-4434-8F1BAA62DE3E2C64",
        name="EFI_D_M_AR_GENERIC_ERROR_SECTION_GUID",
    ),
    UefiGuid(
        "71761D37-32B2-45CD-A7D0B0FEDD93E8CF",
        name="EFI_DIRECTED_IO_D_M_AR_ERROR_SECTION_GUID",
    ),
    UefiGuid(
        "036F84E1-7F37-428C-A79E575FDFAA84EC",
        name="EFI_IOMMU_D_M_AR_ERROR_SECTION_GUID",
    ),
    UefiGuid(
        "BAF1E6DE-209E-4ADB-8D96FD8B71F3F683",
        name="EFI_EVENT_USER_PROFILE_CHANGED_GUID",
    ),
    UefiGuid(
        "5CF32E68-7660-449B-80E67EA36E03F6A8",
        name="EFI_USER_CREDENTIAL_CLASS_UNKNOWN_GUID",
    ),
    UefiGuid(
        "F8E5058C-CCB6-4714-B2203F7E3A640BD1",
        name="EFI_USER_CREDENTIAL_CLASS_PASSWORD_GUID",
    ),
    UefiGuid(
        "5F03BA33-8C6B-4C24-AA2E14A2657BD454",
        name="EFI_USER_CREDENTIAL_CLASS_SMART_CARD_GUID",
    ),
    UefiGuid(
        "32CBA21F-F308-4CBC-9AB5F5A3699F044A",
        name="EFI_USER_CREDENTIAL_CLASS_FINGERPRINT_GUID",
    ),
    UefiGuid(
        "5917EF16-F723-4BB9-A64BD8C532F4D8B5",
        name="EFI_USER_CREDENTIAL_CLASS_HANDPRINT_GUID",
    ),
    UefiGuid(
        "8A6B4A83-42FE-45D2-A2EF46F06C7D9852",
        name="EFI_USER_CREDENTIAL_CLASS_SECURE_CARD_GUID",
    ),
    UefiGuid(
        "85B75607-F7CE-471E-B7E42AEA5F7232EE",
        name="EFI_USER_INFO_ACCESS_SETUP_ADMIN_GUID",
    ),
    UefiGuid(
        "1DB29AE0-9DCB-43BC-8D875DA14964DDE2",
        name="EFI_USER_INFO_ACCESS_SETUP_NORMAL_GUID",
    ),
    UefiGuid(
        "BDB38125-4D63-49F4-821261CF5A190AF8",
        name="EFI_USER_INFO_ACCESS_SETUP_RESTRICTED_GUID",
    ),
    UefiGuid(
        "D719B2CB-3D3A-4596-A3BCDAD00E67656F", name="EFI_IMAGE_SECURITY_DATABASE_GUID"
    ),
    UefiGuid("C1C41626-504C-4092-ACA941F936934328", name="EFI_CERT_SHA256_GUID"),
    UefiGuid("3C5766E8-269C-4E34-AA14ED776E85B3B6", name="EFI_CERT_RSA2048_GUID"),
    UefiGuid(
        "E2B36190-879B-4A3D-AD8DF2E7BBA32784", name="EFI_CERT_RSA2048_SHA256_GUID"
    ),
    UefiGuid("826CA512-CF10-4AC9-B187BE01496631BD", name="EFI_CERT_SHA1_GUID"),
    UefiGuid("67F8444F-8743-48F1-A3281EAAB8736080", name="EFI_CERT_RSA2048_SHA1_GUID"),
    UefiGuid("A5C059A1-94E4-4AA7-87B5AB155C2BF072", name="EFI_CERT_X509_GUID"),
    UefiGuid(
        "EC8A3D69-6DDF-4108-94767337FC522136", name="EFI_KMS_FORMAT_GENERIC128_GUID"
    ),
    UefiGuid(
        "A3B3E6F8-EFCA-4BC1-88FBCB87339B2579", name="EFI_KMS_FORMAT_GENERIC160_GUID"
    ),
    UefiGuid(
        "70F64793-C323-4261-AC2CD876F27C5345", name="EFI_KMS_FORMAT_GENERIC256_GUID"
    ),
    UefiGuid(
        "978FE043-D7AF-422E-8A922B48E463BDE6", name="EFI_KMS_FORMAT_GENERIC512_GUID"
    ),
    UefiGuid(
        "43BE0B44-874B-4EAD-B09C241A4FBD7EB3", name="EFI_KMS_FORMAT_GENERIC1024_GUID"
    ),
    UefiGuid(
        "40093F23-630C-4626-9C4840373B19CBBE", name="EFI_KMS_FORMAT_GENERIC2048_GUID"
    ),
    UefiGuid(
        "B9237513-6C44-4411-A99021E556E05ADE", name="EFI_KMS_FORMAT_GENERIC3072_GUID"
    ),
    UefiGuid("78BE11C4-EE44-4A22-9F0503852EC5C978", name="EFI_KMS_FORMAT_MD2128_GUID"),
    UefiGuid("F7AD60F8-EFA8-44A3-9113231F399EB4C7", name="EFI_KMS_FORMAT_MDC2128_GUID"),
    UefiGuid("D1C17AA1-CAC5-400F-BE17E2A2AE06677C", name="EFI_KMS_FORMAT_MD4128_GUID"),
    UefiGuid("3FA4F847-D8EB-4DF4-BD49103A0A847BBC", name="EFI_KMS_FORMAT_MDC4128_GUID"),
    UefiGuid("DCBC3662-9CDA-4B52-A04C82EB1D2348C7", name="EFI_KMS_FORMAT_MD5128_GUID"),
    UefiGuid(
        "1C178237-6897-459E-9D3667CE8EF94F76", name="EFI_KMS_FORMAT_MD5SHA128_GUID"
    ),
    UefiGuid("453C5E5A-482D-43F0-87C95941F3A38AC2", name="EFI_KMS_FORMAT_SHA1160_GUID"),
    UefiGuid(
        "6BB4F5CD-8022-448D-BC6D771BAE935FC6", name="EFI_KMS_FORMAT_SHA256256_GUID"
    ),
    UefiGuid(
        "2F240E12-E14D-475C-83B0EFFF22D77BE7", name="EFI_KMS_FORMAT_SHA512512_GUID"
    ),
    UefiGuid(
        "4776E33F-DB47-479A-A25FA1CD0AFAB38B", name="EFI_KMS_FORMAT_AESXTS128_GUID"
    ),
    UefiGuid(
        "DC7E8613-C4BB-4DB0-846213511357ABE2", name="EFI_KMS_FORMAT_AESXTS256_GUID"
    ),
    UefiGuid(
        "A0E8EE6A-0E92-44D4-861B0EAA4ACA44A2", name="EFI_KMS_FORMAT_AESCBC128_GUID"
    ),
    UefiGuid(
        "D7E69789-1F68-45E8-96EF3B6407A5B2DC", name="EFI_KMS_FORMAT_AESCBC256_GUID"
    ),
    UefiGuid(
        "56417BED-6BBE-4882-86A03AE8BB17F8F9", name="EFI_KMS_FORMAT_RSASHA11024_GUID"
    ),
    UefiGuid(
        "F66447D4-75A6-463E-A819077F2DDA05E9", name="EFI_KMS_FORMAT_RSASHA12048_GUID"
    ),
    UefiGuid(
        "A477AF13-877D-4060-BAA125D1BEA08AD3", name="EFI_KMS_FORMAT_RSASHA2562048_GUID"
    ),
    UefiGuid(
        "4E1356C2-0EED-463F-81479933ABDBC7D5", name="EFI_KMS_FORMAT_RSASHA2563072_GUID"
    ),
    UefiGuid("0B6E5233-A65C-44C9-9407D9AB83BFC8BD", name="EFI_CERT_SHA224_GUID"),
    UefiGuid("FF3E5307-9FD0-48C9-85F18AD56C701E01", name="EFI_CERT_SHA384_GUID"),
    UefiGuid("093E0FAE-A6C4-4F50-9F1BD41E2B89C19A", name="EFI_CERT_SHA512_GUID"),
    UefiGuid("4AAFD29D-68DF-49EE-8AA9347D375665A7", name="EFI_CERT_PKCS7_GUID"),
    UefiGuid(
        "24C5DC2F-53E2-40CA-9ED6A5D9A49F463B",
        name="EFI_HASH_ALGORITHM_SHA1_NO_PAD_GUID",
    ),
    UefiGuid(
        "8628752A-6CB7-4814-96FC24A815AC2226",
        name="EFI_HASH_ALGORITHM_SHA256_NO_PAD_GUID",
    ),
    UefiGuid("6DCBD5ED-E82D-4C44-BDA17194199AD92A", name="EFI_FMP_CAPSULE_GUID"),
    UefiGuid("3BD2A492-96C0-4079-B420FCF98EF103ED", name="EFI_CERT_X509_SHA256_GUID"),
    UefiGuid("7076876E-80C2-4EE6-AAD228B349A6865B", name="EFI_CERT_X509_SHA384_GUID"),
    UefiGuid("446DBF63-2502-4CDA-BCFA2465D2B0FE9D", name="EFI_CERT_X509_SHA512_GUID"),
    UefiGuid(
        "A7AF67CB-603B-4D42-BA2170BFB6293F96",
        name="EFI_RNG_ALGORITHM_SP80090_HASH256_GUID",
    ),
    UefiGuid(
        "C5149B43-AE85-4F53-9982B94335D3A9E7",
        name="EFI_RNG_ALGORITHM_SP80090_HMAC256_GUID",
    ),
    UefiGuid(
        "44F0DE6E-4D8C-4045-A8C74DD168856B9E",
        name="EFI_RNG_ALGORITHM_SP80090_CTR256_GUID",
    ),
    UefiGuid(
        "63C4785A-CA34-4012-A3C80B6A324F5546", name="EFI_RNG_ALGORITHM_X9313_DES_GUID"
    ),
    UefiGuid(
        "ACD03321-777E-4D3D-B1C820CFD88820C9", name="EFI_RNG_ALGORITHM_X931_AES_GUID"
    ),
    UefiGuid("E43176D7-B6E8-4827-B7847FFDC4B68561", name="EFI_RNG_ALGORITHM_RAW"),
    UefiGuid(
        "D7C74207-A831-4A26-B1F5D193065CE8B6", name="EFI_ADAPTER_INFO_MEDIA_STATE_GUID"
    ),
    UefiGuid(
        "1FBD2960-4130-41E5-94ACD2CF037FB37C", name="EFI_ADAPTER_INFO_NETWORK_BOOT_GUID"
    ),
    UefiGuid(
        "114DA5EF-2CF1-4E12-9BBBC470B55205D9",
        name="EFI_ADAPTER_INFO_SAN_MAC_ADDRESS_GUID",
    ),
    UefiGuid("39B68C46-F7FB-441B-B6EC16B0F69821F3", name="EFI_CAPSULE_REPORT_GUID"),
    UefiGuid("880AACA3-4ADC-4A04-9079B747340825E5", name="EFI_PROPERTIES_TABLE_GUID"),
    UefiGuid(
        "B122A263-3661-4F68-992978F8B0D62180", name="EFI_SYSTEM_RESOURCE_TABLE_GUID"
    ),
    UefiGuid(
        "4BD56BE3-4975-4D8A-A0ADC491204B5D4D",
        name="EFI_ADAPTER_INFO_UNDI_IPV6_SUPPORT_GUID",
    ),
    UefiGuid(
        "5F05B20F-4A56-C231-FA0BA7B1F110041D",
        name="EFI_REGEX_SYNTAX_TYPE_POSIX_EXTENDED_GUID",
    ),
    UefiGuid(
        "9A473A4A-4CEB-B95A-415E5BA0BC639B2E", name="EFI_REGEX_SYNTAX_TYPE_ECMA262_GUID"
    ),
    UefiGuid(
        "63E60A51-497D-D427-C4A5B8ABDC3AAEB6", name="EFI_REGEX_SYNTAX_TYPE_PERL_GUID"
    ),
    UefiGuid(
        "61EC04FC-48E6-D813-25C98DAA44750B12",
        name="EFI_PLATFORM_MEMORY2_ERROR_SECTION_GUID",
    ),
    UefiGuid(
        "2F87BA6A-5C04-4385-A780F3BF78A97BEC",
        name="EFI_BLOCK_IO_CRYPTO_ALGO_AES_XTS_GUID",
    ),
    UefiGuid(
        "689E4C62-70BF-4CF3-88BB33B318268670",
        name="EFI_BLOCK_IO_CRYPTO_ALGO_AES_CBC_MS_BITLOCKER_GUID",
    ),
    UefiGuid(
        "9317EC24-7CB0-4D0E-8B322ED9209CD8AF",
        name="EFI_PADDING_RSASSA_PKCS1_V1_P5_GUID",
    ),
    UefiGuid("7B2349E0-522D-4F8E-B92769D97C9E795F", name="EFI_PADDING_RSASSA_PSS_GUID"),
    UefiGuid("3629DDB1-228C-452E-B61609ED316A9700", name="EFI_PADDING_NONE_GUID"),
    UefiGuid(
        "E1C1D0A9-40B1-4632-BDCCD9D6E5295631", name="EFI_PADDING_RSAES_PKCS1_V1_P5_GUID"
    ),
    UefiGuid("C1E63AC4-D0CF-4CE6-835BEED0E6A8A45B", name="EFI_PADDING_RSAES_OAEP_GUID"),
    UefiGuid("F2FD1544-9794-4A2C-992EE5BBCF20E394", name="EFI_SMBIOS3_TABLE_GUID"),
    UefiGuid(
        "CAB0E94C-E15F-11E3-918DB8E8562CBAFA",
        name="EFI_BOOT_MANAGER_POLICY_CONSOLE_GUID",
    ),
    UefiGuid(
        "D04159DC-E15F-11E3-B261B8E8562CBAFA",
        name="EFI_BOOT_MANAGER_POLICY_NETWORK_GUID",
    ),
    UefiGuid(
        "113B2126-FC8A-11E3-BD6CB8E8562CBAFA",
        name="EFI_BOOT_MANAGER_POLICY_CONNECT_ALL_GUID",
    ),
    UefiGuid("77AB535A-45FC-624B-5560F7B281D1F96E", name="EFI_VIRTUAL_DISK_GUID"),
    UefiGuid("3D5ABD30-4175-87CE-6D64D2ADE523C4BB", name="EFI_VIRTUAL_CD_GUID"),
    UefiGuid(
        "5CEA02C9-4D07-69D3-269F4496FBE096F9", name="EFI_PERSISTENT_VIRTUAL_DISK_GUID"
    ),
    UefiGuid(
        "08018188-42CD-BB48-100F5387D53DED3D", name="EFI_PERSISTENT_VIRTUAL_CD_GUID"
    ),
    UefiGuid(
        "DCFA911D-26EB-469F-A22038B7DC461220", name="EFI_MEMORY_ATTRIBUTES_TABLE_GUID"
    ),
    UefiGuid(
        "E19E3D16-BC11-11E4-9CAAC2051D5D46B0",
        name="EFI_ARM_PROCESSOR_ERROR_SECTION_GUID",
    ),
    UefiGuid(
        "EFEFD093-0D9B-46EB-A85648350700C908",
        name="EFI_HII_IMAGE_DECODER_NAME_JPEG_GUID",
    ),
    UefiGuid(
        "AF060190-5E3A-4025-AFBDE1F905BFAA4C",
        name="EFI_HII_IMAGE_DECODER_NAME_PNG_GUID",
    ),
    UefiGuid("18633BFC-1735-4217-8AC917239282D3F8", name="EFI_BTT_ABSTRACTION_GUID"),
    UefiGuid("1B45CC0A-156A-428A-AF6249864DA0E6E6", name="PEI_APRIORI_FILE_NAME_GUID"),
    UefiGuid("FC510EE7-FFDC-11D4-BD410080C73C8881", name="APRIORI_GUID"),
    UefiGuid(
        "8C8CE578-8A3D-4F1C-9935896185C32DD3", name="EFI_FIRMWARE_FILE_SYSTEM2_GUID"
    ),
    UefiGuid(
        "1BA0062E-C779-4582-8566336AE8F78F09", name="EFI_FIRMWARE_VOLUME_TOP_FILE_GUID"
    ),
    UefiGuid(
        "F8E21975-0899-4F58-A4BE5525A9C6D77A", name="EFI_HOB_MEMORY_ALLOC_MODULE_GUID"
    ),
    UefiGuid(
        "4ED4BF27-4092-42E9-807D527B1D00C9BD", name="EFI_HOB_MEMORY_ALLOC_STACK_GUID"
    ),
    UefiGuid(
        "564B33CD-C92A-4593-90BF2473E43C6322",
        name="EFI_HOB_MEMORY_ALLOC_BSP_STORE_GUID",
    ),
    UefiGuid("2A571201-4966-47F6-8B86F31E41F32F10", name="EFI_EVENT_LEGACY_BOOT_GUID"),
    UefiGuid("7739F24C-93D7-11D4-9A3A0090273FC14D", name="EFI_HOB_LIST_GUID"),
    UefiGuid("05AD34BA-6F02-4214-952E4DA0398E2BB9", name="EFI_DXE_SERVICES_TABLE_GUID"),
    UefiGuid(
        "914AEBE7-4635-459B-AA1C11E219B03A10", name="EFI_MDE_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "414E6BDD-E47B-47CC-B244BB61020CF516", name="EFI_HARDWARE_ERROR_VARIABLE_GUID"
    ),
    UefiGuid("7081E22F-CAC6-4053-9468675782CF88E5", name="EFI_EVENT_DXE_DISPATCH_GUID"),
    UefiGuid(
        "5E948FE3-26D3-42B5-AF17610287188DEC", name="EFI_DISK_INFO_IDE_INTERFACE_GUID"
    ),
    UefiGuid(
        "08F74BAA-EA36-41D9-952121A70F8780BC", name="EFI_DISK_INFO_SCSI_INTERFACE_GUID"
    ),
    UefiGuid(
        "CB871572-C11A-47B5-B492675EAFA77727", name="EFI_DISK_INFO_USB_INTERFACE_GUID"
    ),
    UefiGuid(
        "9E498932-4ABC-45AF-A34D0247787BE7C6", name="EFI_DISK_INFO_AHCI_INTERFACE_GUID"
    ),
    UefiGuid(
        "92D11080-496F-4D95-BE7E037488382B0A",
        name="EFI_STATUS_CODE_DATA_TYPE_STRING_GUID",
    ),
    UefiGuid(
        "335984BD-E805-409A-B8F8D27ECE5FF7A6", name="EFI_STATUS_CODE_SPECIFIC_DATA_GUID"
    ),
    UefiGuid(
        "5473C07A-3DCB-4DCA-BD6F1E9689E7349A", name="EFI_FIRMWARE_FILE_SYSTEM3_GUID"
    ),
    UefiGuid(
        "02CE967A-DD7E-4FFC-9EE7810CF0470880", name="EFI_END_OF_DXE_EVENT_GROUP_GUID"
    ),
    UefiGuid(
        "0F9D89E8-9259-4F76-A5AF0C89E34023DF", name="EFI_FIRMWARE_CONTENTS_SIGNED_GUID"
    ),
    UefiGuid(
        "996EC11C-5397-4E73-B58F827E52906DEF", name="EFI_VECTOR_HANDOFF_TABLE_GUID"
    ),
    UefiGuid(
        "6BE272C7-1320-4CCD-9017D4612C012B25",
        name="ADAPTER_INFO_PLATFORM_SECURITY_GUID",
    ),
    UefiGuid(
        "3AB14680-5D3F-4A4D-BCDCCC380018C7F7", name="EFI_DISK_INFO_NVME_INTERFACE_GUID"
    ),
    UefiGuid("39F62CCE-6825-4669-BB56541ABA753A07", name="EFI_GRAPHICS_INFO_HOB_GUID"),
    UefiGuid(
        "4B3029CC-6B98-47FB-BC9676DCB80441F0", name="EFI_DISK_INFO_UFS_INTERFACE_GUID"
    ),
    UefiGuid(
        "E5CB2AC9-D35D-4430-936E1DE332478DE7", name="EFI_GRAPHICS_DEVICE_INFO_HOB_GUID"
    ),
    UefiGuid("6DADF1D1-D4CC-4910-BB6E82B1FD80FF3D", name="EFI_SMM_SMRAM_MEMORY_GUID"),
    UefiGuid(
        "8DEEC992-D39C-4A5C-AB6B986E14242B9D",
        name="EFI_DISK_INFO_SD_MMC_INTERFACE_GUID",
    ),
    UefiGuid("3B8C8162-188C-46A4-AEC9BE43F1D65697", name="WINDOWS_UX_CAPSULE_GUID"),
    UefiGuid(
        "A31280AD-481E-41B6-95E8127F4C984779", name="TIANO_CUSTOM_DECOMPRESS_GUID"
    ),
    UefiGuid(
        "A55701F5-E3EF-43DE-AC72249B573FAD2C",
        name="EFI_IA32_X64_ERROR_TYPE_CACHE_CHECK_GUID",
    ),
    UefiGuid(
        "FC06B535-5E1F-4562-9F250A3B9ADB63C3",
        name="EFI_IA32_X64_ERROR_TYPE_TLB_CHECK_GUID",
    ),
    UefiGuid(
        "1CF3F8B3-C5B1-49A2-AA595EEF92FFA63C",
        name="EFI_IA32_X64_ERROR_TYPE_BUS_CHECK_GUID",
    ),
    UefiGuid(
        "48AB7F57-DC34-4F6C-A7D3B0B5B0A74314",
        name="EFI_IA32_X64_ERROR_TYPE_MS_CHECK_GUID",
    ),
    UefiGuid(
        "7408D748-FC8C-4EE6-9288C4BEC092A410", name="EFI_PEI_MASTER_BOOT_MODE_PPI_GUID"
    ),
    UefiGuid("0AE8CE5D-E448-4437-A8D7EBF5F194F731", name="EFI_DXE_IPL_PPI_GUID"),
    UefiGuid(
        "F894643D-C449-42D1-8EA885BDD8C65BDE", name="EFI_PEI_MEMORY_DISCOVERED_PPI_GUID"
    ),
    UefiGuid(
        "17EE496A-D8E4-4B9A-94D1CE8272300850",
        name="EFI_PEI_BOOT_IN_RECOVERY_MODE_PPI_GUID",
    ),
    UefiGuid(
        "605EA650-C65C-42E1-BA8091A52AB618C6", name="EFI_END_OF_PEI_SIGNAL_PPI_GUID"
    ),
    UefiGuid("EF398D58-9DFD-4103-BF9478C6F4FE712F", name="EFI_PEI_RESET_PPI_GUID"),
    UefiGuid(
        "229832D3-7A30-4B36-B827F40CB7D45436", name="EFI_PEI_STATUS_CODE_PPI_GUID"
    ),
    UefiGuid("DCD0BE23-9586-40F4-B64306522CED4EDE", name="EFI_PEI_SECURITY2_PPI_GUID"),
    UefiGuid(
        "DBE23AA9-A345-4B97-85B6B226F1617389", name="EFI_TEMPORARY_RAM_SUPPORT_PPI_GUID"
    ),
    UefiGuid(
        "E6AF1F7B-FC3F-46DA-A828A3B457A44282", name="EFI_PEI_CPU_IO_PPI_INSTALLED_GUID"
    ),
    UefiGuid("057A449A-1FDC-4C06-BFC9F53F6A99BB92", name="EFI_PCI_CFG2_PPI_GUID"),
    UefiGuid("1F4C6F90-B06B-48D8-A201BAE5F1CD7D56", name="EFI_PEI_STALL_PPI_GUID"),
    UefiGuid(
        "2AB86EF5-ECB5-4134-B5563854CA1FE1B4",
        name="EFI_PEI_READ_ONLY_VARIABLE2_PPI_GUID",
    ),
    UefiGuid(
        "6F8C2B35-FEF4-448D-8256E11B19D61077",
        name="EFI_SEC_PLATFORM_INFORMATION_PPI_GUID",
    ),
    UefiGuid(
        "C1FCD448-6300-4458-B86428DF015364BC", name="EFI_PEI_LOADED_IMAGE_PPI_GUID"
    ),
    UefiGuid("9CA93627-B65B-4324-A202C0B461764543", name="EFI_PEI_SMBUS2_PPI_GUID"),
    UefiGuid(
        "49EDB1C1-BF21-4761-BB12EB0031AABB39",
        name="EFI_PEI_FIRMWARE_VOLUME_INFO_PPI_GUID",
    ),
    UefiGuid("B9E0ABFE-5979-4914-977F6DEE78C278A6", name="EFI_PEI_LOAD_FILE_PPI_GUID"),
    UefiGuid("1A36E4E7-FAB6-476A-8E75695A0576FDD7", name="EFI_PEI_DECOMPRESS_PPI_GUID"),
    UefiGuid("06E81C58-4AD7-44BC-8390F10265F72480", name="PCD_PPI_GUID"),
    UefiGuid("4D8B155B-C059-4C8F-892606FD4331DB8A", name="GET_PCD_INFO_PPI_GUID"),
    UefiGuid(
        "FB6D9542-612D-4F45-872F5CFF52E93DCF", name="EFI_PEI_RECOVERY_MODULE_PPI_GUID"
    ),
    UefiGuid(
        "0DE2CE25-446A-45A7-BFC937DA26344B37",
        name="EFI_PEI_DEVICE_RECOVERY_MODULE_PPI_GUID",
    ),
    UefiGuid(
        "695D8AA1-42EE-4C46-805C6EA6BCE799E3", name="EFI_PEI_VIRTUAL_BLOCK_IO_PPI_GUID"
    ),
    UefiGuid("6D582DBC-DB85-4514-8FCC5ADF6227B147", name="EFI_PEI_S3_RESUME2_PPI_GUID"),
    UefiGuid(
        "0065D394-9951-4144-82A30AFC8579C251", name="EFI_PEI_RSC_HANDLER_PPI_GUID"
    ),
    UefiGuid("01F34D25-4DE2-23AD-3FF336353FF323F1", name="EFI_PEI_PCD_PPI_GUID"),
    UefiGuid("A60C6B59-E459-425D-9C690BCC9CB27D81", name="EFI_GET_PCD_INFO_PPI_GUID"),
    UefiGuid(
        "CEAB683C-EC56-4A2D-A9064053FA4E9C16", name="EFI_TEMPORARY_RAM_DONE_PPI_GUID"
    ),
    UefiGuid(
        "3CD652B4-6D33-4DCE-89DB83DF9766FCCA", name="EFI_VECTOR_HANDOFF_INFO_PPI_GUID"
    ),
    UefiGuid("8D48BD70-C8A3-4C06-901B747946AAC358", name="EFI_ISA_HC_PPI_GUID"),
    UefiGuid("23A464AD-CB83-48B8-94AB1A6FEFCFE522", name="EFI_SIO_PPI_GUID"),
    UefiGuid("B3BFAB9B-9F9C-4E8B-AD377F8C51FC6280", name="EFI_PEI_I2C_MASTER_PPI_GUID"),
    UefiGuid(
        "EA7CA24B-DED5-4DAD-A389BF827E8F9B38",
        name="EFI_PEI_FIRMWARE_VOLUME_INFO2_PPI_GUID",
    ),
    UefiGuid("6ECD1463-4A4A-461B-AF5F5A33E3B2162B", name="EFI_PEI_GRAPHICS_PPI_GUID"),
    UefiGuid(
        "EE16160A-E8BE-47A6-820AC6900DB0250A", name="EFI_PEI_MP_SERVICES_PPI_GUID"
    ),
    UefiGuid("3ACF33EE-D892-40F4-A2FC3854D2E1323D", name="EFI_PEI_CAPSULE_PPI_GUID"),
    UefiGuid("3ACF33EE-D892-40F4-A2FC3854D2E1323D", name="PEI_CAPSULE_PPI_GUID"),
    UefiGuid("6CC45765-CCE4-42FD-BC56011AAAC6C9A8", name="EFI_PEI_RESET2_PPI_GUID"),
    UefiGuid(
        "26CC0FAD-BEB3-478A-91B20C188F726198", name="EFI_PEI_VIRTUAL_BLOCK_IO2_PPI_GUID"
    ),
    UefiGuid(
        "9E9F374B-8F16-4230-98245846EE766A97",
        name="EFI_SEC_PLATFORM_INFORMATION2_PPI_GUID",
    ),
    UefiGuid("3EBDAF20-6667-40D8-B4EEF5999AC1B71F", name="EFI_SEC_HOB_DATA_PPI_GUID"),
    UefiGuid("268F33A9-CCCD-48BE-881786053AC32ED6", name="EFI_PEI_MM_ACCESS_PPI_GUID"),
    UefiGuid("61C68702-4D7E-4F43-8DEFA74305CE74C5", name="EFI_PEI_MM_CONTROL_PPI_GUID"),
    UefiGuid(
        "52888EAE-5B10-47D0-A87FB822ABA0CAF4", name="EFI_PEI_CORE_FV_LOCATION_PPI_GUID"
    ),
    UefiGuid("11B34006-D85B-4D0A-A290D5A571310EF7", name="PCD_PROTOCOL_GUID"),
    UefiGuid("5BE40F57-FA68-4610-BBBFE9C5FCDAD365", name="GET_PCD_INFO_PROTOCOL_GUID"),
    UefiGuid("665E3FF6-46CC-11D4-9A380090273FC14D", name="EFI_BDS_ARCH_PROTOCOL_GUID"),
    UefiGuid("26BACCB1-6F42-11D4-BCE70080C73C8881", name="EFI_CPU_ARCH_PROTOCOL_GUID"),
    UefiGuid(
        "26BACCB2-6F42-11D4-BCE70080C73C8881", name="EFI_METRONOME_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "1DA97072-BDDC-4B30-99F172A0B56FFF2A",
        name="EFI_MONOTONIC_COUNTER_ARCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "27CFAC87-46CC-11D4-9A380090273FC14D",
        name="EFI_REAL_TIME_CLOCK_ARCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "27CFAC88-46CC-11D4-9A380090273FC14D", name="EFI_RESET_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B7DFB4E1-052F-449F-87BE9818FC91B733", name="EFI_RUNTIME_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "A46423E3-4617-49F1-B9FFD1BFA9115839", name="EFI_SECURITY_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "78E4D245-CD4D-4A05-A2BA4743E86CFCAB", name="EFI_SECURITY_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid(
        "26BACCB3-6F42-11D4-BCE70080C73C8881", name="EFI_TIMER_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "6441F818-6362-4E44-B5707DBA31DD2453",
        name="EFI_VARIABLE_WRITE_ARCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "1E5668E2-8481-11D4-BCF10080C73C8881", name="EFI_VARIABLE_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "665E3FF5-46CC-11D4-9A380090273FC14D",
        name="EFI_WATCHDOG_TIMER_ARCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "D2B2B828-0826-48A7-B3DF983C006024F0",
        name="EFI_STATUS_CODE_RUNTIME_PROTOCOL_GUID",
    ),
    UefiGuid("E49D33ED-513D-4634-B6986F55AA751C1B", name="EFI_SMBUS_HC_PROTOCOL_GUID"),
    UefiGuid(
        "220E73B6-6BDB-4413-8405B974B108619A", name="EFI_FIRMWARE_VOLUME2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "8F644FA9-E850-4DB1-9CE20B44698E8DA4",
        name="EFI_FIRMWARE_VOLUME_BLOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "5053697E-2CBC-4819-90D90580DEEE5754", name="EFI_CAPSULE_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3FDDA605-A76E-4F46-AD2912F4531B3D08", name="EFI_MP_SERVICE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "CF8034BE-6768-4D8B-B7397CCE683A9FBE",
        name="EFI_PCI_HOST_BRIDGE_RESOURCE_ALLOCATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "07D75280-27D4-4D69-90D05643E238B341", name="EFI_PCI_PLATFORM_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B5B35764-460C-4A06-99FC77A17C1B5CEB", name="EFI_PCI_OVERRIDE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "30CFE3E7-3DE1-4586-BE20DEABA1B3B793",
        name="EFI_PCI_ENUMERATION_COMPLETE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "EB23F55A-7863-4AC2-8D3D956535DE0375",
        name="EFI_INCOMPATIBLE_PCI_DEVICE_SUPPORT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "AA0E8BC1-DABC-46B0-A84437B8169B2BEA",
        name="EFI_PCI_HOT_PLUG_INIT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "19CB87AB-2CB9-4665-8360DDCF6054F79D",
        name="EFI_PCI_HOT_PLUG_REQUEST_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A1E37052-80D9-4E65-A3173E9A55C43EC9",
        name="EFI_IDE_CONTROLLER_INIT_PROTOCOL_GUID",
    ),
    UefiGuid("D432A67F-14DC-484B-B3BB3F0291849327", name="EFI_DISK_INFO_PROTOCOL_GUID"),
    UefiGuid("03583FF6-CB36-4940-947EB9B39F4AFAF7", name="EFI_SMBIOS_PROTOCOL_GUID"),
    UefiGuid(
        "E857CAF6-C046-45DC-BE3FEE0765FBA887", name="EFI_S3_SAVE_STATE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "320AFE62-E593-49CB-A9F1D4C2F4AF014C",
        name="EFI_S3_SMM_SAVE_STATE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "86212936-0E76-41C8-A03A2AF2FC1C39E2", name="EFI_RSC_HANDLER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "2FF29FA7-5E80-4ED9-B380017D3C554FF4", name="EFI_SMM_RSC_HANDLER_PROTOCOL_GUID"
    ),
    UefiGuid("EB97088E-CFDF-49C6-BE4BD906A5B20E86", name="EFI_ACPI_SDT_PROTOCOL_GUID"),
    UefiGuid("215FDD18-BD50-4FEB-890B58CA0B4739E9", name="EFI_SIO_PROTOCOL_GUID"),
    UefiGuid(
        "3242A9D8-CE70-4AA0-955D5E7B140DE4D2", name="EFI_SMM_CPU_IO2_PROTOCOL_GUID"
    ),
    UefiGuid("F4CCBFB7-F6E0-47FD-9DD410A8F150C191", name="EFI_SMM_BASE2_PROTOCOL_GUID"),
    UefiGuid(
        "C2702B74-800C-4131-87468FB5B89CE4AC", name="EFI_SMM_ACCESS2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "843DC720-AB1E-42CB-93578A0078F3561B", name="EFI_SMM_CONTROL2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "26EEB3DE-B689-492E-80F0BE8BD7DA4BA7",
        name="EFI_SMM_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "47B7FA8C-F4BD-4AF6-8200333086F0D2C8",
        name="EFI_SMM_READY_TO_LOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "60FF8964-E906-41D0-AFEDF241E974E08E",
        name="EFI_DXE_SMM_READY_TO_LOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "C68ED8E2-9DC6-4CBD-9D94DB65ACC5C332",
        name="EFI_SMM_COMMUNICATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "6AFD2B77-98C1-4ACD-A6F98A9439DE0FB1", name="EFI_SMM_STATUS_CODE_PROTOCOL_GUID"
    ),
    UefiGuid("EB346B97-975F-4A9F-8B22F8E92BB3D569", name="EFI_SMM_CPU_PROTOCOL_GUID"),
    UefiGuid(
        "8BC1714D-FFCB-41C3-89DC6C74D06D98EA",
        name="EFI_SMM_PCI_ROOT_BRIDGE_IO_PROTOCOL_GUID",
    ),
    UefiGuid(
        "18A3C6DC-5EEA-48C8-A1C1B53389F98999", name="EFI_SMM_SW_DISPATCH2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "456D2859-A84B-4E47-A2EE3276D886997D", name="EFI_SMM_SX_DISPATCH2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "4CEC368E-8E8E-4D71-8BE1958C45FC8A53",
        name="EFI_SMM_PERIODIC_TIMER_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "EE9B8D90-C5A6-40A2-BDE252558D33CCA1",
        name="EFI_SMM_USB_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "25566B03-B577-4CBF-958CED663EA24380",
        name="EFI_SMM_GPI_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7300C4A1-43F2-4017-A51BC81A7F40585B",
        name="EFI_SMM_STANDBY_BUTTON_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "1B1183FA-1823-46A7-88729C578755409D",
        name="EFI_SMM_POWER_BUTTON_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "58DC368D-7BFA-4E77-ABBC0E29418DF930",
        name="EFI_SMM_IO_TRAP_DISPATCH2_PROTOCOL_GUID",
    ),
    UefiGuid("13A3F0F6-264A-3EF0-F2E0DEC512342F34", name="EFI_PCD_PROTOCOL_GUID"),
    UefiGuid(
        "8F644FA9-E850-4DB1-9CE20B44698E8DA4",
        name="EFI_FIRMWARE_VOLUME_BLOCK2_PROTOCOL_GUID",
    ),
    UefiGuid("AD61F191-AE5F-4C0E-B9FAE869D288C64F", name="EFI_CPU_IO2_PROTOCOL_GUID"),
    UefiGuid(
        "70101EAF-0085-440C-B3568EE36FEF24F0", name="EFI_LEGACY_REGION2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "94AB2F58-1438-4EF1-915218941A3A0E68", name="EFI_SECURITY2_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "24E70042-D5C5-4260-8C390AD3AA32E93D", name="EFI_SMM_END_OF_DXE_PROTOCOL_GUID"
    ),
    UefiGuid("BCDAF080-1BDE-4E22-AE6A43541E128EC4", name="EFI_ISA_HC_PROTOCOL_GUID"),
    UefiGuid(
        "FAD7933A-6C21-4234-A4340A8A0D2B0781",
        name="EFI_ISA_HC_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "B91978DF-9FC1-427D-BB054C828455CA27", name="EFI_SIO_CONTROL_PROTOCOL_GUID"
    ),
    UefiGuid(
        "FD0F4478-0EFD-461D-BA2DE58C45FD5F5E", name="EFI_GET_PCD_INFO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "CD72881F-45B5-4FEB-98C8313DA8117462", name="EFI_I2C_MASTER_PROTOCOL_GUID"
    ),
    UefiGuid("B60A3E6B-18C4-46E5-A29AC9A10665A28E", name="EFI_I2C_IO_PROTOCOL_GUID"),
    UefiGuid(
        "DA8CD7C4-1C00-49E2-803E5214E701894C", name="EFI_I2C_ENUMERATE_PROTOCOL_GUID"
    ),
    UefiGuid("A5AAB9E3-C727-48CD-8BBF427233854948", name="EFI_I2C_HOST_PROTOCOL_GUID"),
    UefiGuid(
        "55B71FB5-17C6-410E-B5BD5FA2E3D4466B",
        name="EFI_I2C_BUS_CONFIGURATION_MANAGEMENT_PROTOCOL_GUID",
    ),
    UefiGuid("5D5450D7-990C-4180-A8038E63F0608307", name="EFI_MM_MP_PROTOCOL_GUID"),
    UefiGuid(
        "24E70042-D5C5-4260-8C390AD3AA32E93D", name="EFI_MM_END_OF_DXE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "58DC368D-7BFA-4E77-ABBC0E29418DF930",
        name="EFI_MM_IO_TRAP_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "1B1183FA-1823-46A7-88729C578755409D",
        name="EFI_MM_POWER_BUTTON_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7300C4A1-43F2-4017-A51BC81A7F40585B",
        name="EFI_MM_STANDBY_BUTTON_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "25566B03-B577-4CBF-958CED663EA24380", name="EFI_MM_GPI_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "EE9B8D90-C5A6-40A2-BDE252558D33CCA1", name="EFI_MM_USB_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "4CEC368E-8E8E-4D71-8BE1958C45FC8A53",
        name="EFI_MM_PERIODIC_TIMER_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "456D2859-A84B-4E47-A2EE3276D886997D", name="EFI_MM_SX_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "18A3C6DC-5EEA-48C8-A1C1B53389F98999", name="EFI_MM_SW_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "8BC1714D-FFCB-41C3-89DC6C74D06D98EA",
        name="EFI_MM_PCI_ROOT_BRIDGE_IO_PROTOCOL_GUID",
    ),
    UefiGuid("EB346B97-975F-4A9F-8B22F8E92BB3D569", name="EFI_MM_CPU_PROTOCOL_GUID"),
    UefiGuid(
        "6AFD2B77-98C1-4ACD-A6F98A9439DE0FB1", name="EFI_MM_STATUS_CODE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "60FF8964-E906-41D0-AFEDF241E974E08E",
        name="EFI_DXE_MM_READY_TO_LOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "26EEB3DE-B689-492E-80F0BE8BD7DA4BA7", name="EFI_MM_CONFIGURATION_PROTOCOL_GUID"
    ),
    UefiGuid(
        "47B7FA8C-F4BD-4AF6-8200333086F0D2C8", name="EFI_MM_READY_TO_LOCK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "843DC720-AB1E-42CB-93578A0078F3561B", name="EFI_MM_CONTROL_PROTOCOL_GUID"
    ),
    UefiGuid("C2702B74-800C-4131-87468FB5B89CE4AC", name="EFI_MM_ACCESS_PROTOCOL_GUID"),
    UefiGuid("F4CCBFB7-F6E0-47FD-9DD410A8F150C191", name="EFI_MM_BASE_PROTOCOL_GUID"),
    UefiGuid("3242A9D8-CE70-4AA0-955D5E7B140DE4D2", name="EFI_MM_CPU_IO_PROTOCOL_GUID"),
    UefiGuid(
        "2FF29FA7-5E80-4ED9-B380017D3C554FF4", name="EFI_MM_RSC_HANDLER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "C68ED8E2-9DC6-4CBD-9D94DB65ACC5C332", name="EFI_MM_COMMUNICATION_PROTOCOL_GUID"
    ),
    UefiGuid(
        "39136FC7-1A11-49DE-BF350E78DDB524FC",
        name="EFI_LEGACY_SPI_CONTROLLER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "F01BED57-04BC-4F3F-9660D6F2EA228259", name="EFI_LEGACY_SPI_FLASH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "62331B78-D8D0-4C8C-8CCBD27DFE32DB9B",
        name="EFI_LEGACY_SPI_SMM_CONTROLLER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "5E3848D4-0DB5-4FC0-97293F353D4F879F",
        name="EFI_LEGACY_SPI_SMM_FLASH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "85A6D3E6-B65B-4AFC-B38FC6D54AF6DDC8",
        name="EFI_SPI_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid("C74E5DB2-FA96-4AE2-B39915977FE3002D", name="EFI_SPI_HC_PROTOCOL_GUID"),
    UefiGuid(
        "B57EC3FE-F833-4BA6-85782A7D6A87444B", name="EFI_SPI_NOR_FLASH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "995C6ECA-171B-45FD-A3AAFD4C9C9DEF59",
        name="EFI_SPI_SMM_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "E9F02217-2093-4470-8A545C2CFFE73ECB", name="EFI_SPI_SMM_HC_PROTOCOL_GUID"
    ),
    UefiGuid(
        "AAB18F19-FE14-4666-860487FF6D662C9A",
        name="EFI_SPI_SMM_NOR_FLASH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "EBA4E8D2-3858-41EC-A2812647BA9660D0", name="EFI_DEBUG_PORT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "2755590C-6F3C-42FA-9EA4A3BA543CDA25", name="EFI_DEBUG_SUPPORT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "D8117CFE-94A6-11D4-9A3A0090273FC14D", name="EFI_DECOMPRESS_PROTOCOL_GUID"
    ),
    UefiGuid("AF6AC311-84C3-11D2-8E3C00A0C969723B", name="EFI_DEVICE_IO_PROTOCOL_GUID"),
    UefiGuid(
        "09576E91-6D3F-11D2-8E3900A0C969723B", name="EFI_DEVICE_PATH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "05C99A21-C70F-4AD2-8A5F35DF3343F51E",
        name="EFI_DEVICE_PATH_FROM_TEXT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "8B843E20-8132-4852-90CC551A4E4A7F1C",
        name="EFI_DEVICE_PATH_TO_TEXT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "0379BE4E-D706-437D-B037EDB82FB772A4",
        name="EFI_DEVICE_PATH_UTILITIES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "18A031AB-B443-4D1A-A5C00C09261E9F71", name="EFI_DRIVER_BINDING_PROTOCOL_GUID"
    ),
    UefiGuid(
        "6B30C738-A391-11D4-9A3B0090273FC14D",
        name="EFI_PLATFORM_DRIVER_OVERRIDE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "B1EE129E-DA36-4181-91F804A4923766A7",
        name="EFI_DRIVER_FAMILY_OVERRIDE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "3BC1B285-8A15-4A82-AABF4D7D13FB3265",
        name="EFI_BUS_SPECIFIC_DRIVER_OVERRIDE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "4D330321-025F-4AAC-90D85ED900173B63",
        name="EFI_DRIVER_DIAGNOSTICS2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "0784924F-E296-11D4-9A490090273FC14D",
        name="EFI_DRIVER_DIAGNOSTICS_PROTOCOL_GUID",
    ),
    UefiGuid(
        "6A7A5CFF-E8D9-4F70-BADA75AB3025CE14", name="EFI_COMPONENT_NAME2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "107A772C-D5E1-11D4-9A460090273FC14D", name="EFI_COMPONENT_NAME_PROTOCOL_GUID"
    ),
    UefiGuid(
        "BFD7DC1D-24F1-40D9-82E72E09BB6B4EBE",
        name="EFI_DRIVER_CONFIGURATION2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "107A772B-D5E1-11D4-9A460090273FC14D",
        name="EFI_DRIVER_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "642CD590-8059-4C0A-A958C5EC07D23C4B",
        name="EFI_PLATFORM_TO_DRIVER_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "5C198761-16A8-4E69-972C89D67954F81D",
        name="EFI_DRIVER_SUPPORTED_EFI_VERSION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "387477C1-69C7-11D2-8E3900A0C969723B", name="EFI_SIMPLE_TEXT_IN_PROTOCOL_GUID"
    ),
    UefiGuid(
        "DD9E7534-7762-4698-8C14F58517A625AA",
        name="EFI_SIMPLE_TEXT_INPUT_EX_PROTOCOL_GUID",
    ),
    UefiGuid(
        "387477C2-69C7-11D2-8E3900A0C969723B", name="EFI_SIMPLE_TEXT_OUT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "31878C87-0B75-11D5-9A4F0090273FC14D", name="EFI_SIMPLE_POINTER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "8D59D32B-C655-4AE9-9B15F25904992A43", name="EFI_ABSOLUTE_POINTER_PROTOCOL_GUID"
    ),
    UefiGuid("BB25CF6F-F1D4-11D2-9A0C0090273FC1FD", name="EFI_SERIAL_IO_PROTOCOL_GUID"),
    UefiGuid(
        "9042A9DE-23DC-4A38-96FB7ADED080516A", name="EFI_GRAPHICS_OUTPUT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "1C0C34F6-D380-41FA-A0498AD06C1A66AA", name="EFI_EDID_DISCOVERED_PROTOCOL_GUID"
    ),
    UefiGuid(
        "BD8C1056-9F36-44EC-92A8A6337F817986", name="EFI_EDID_ACTIVE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "48ECB431-FB72-45C0-A922F458FE040BD5", name="EFI_EDID_OVERRIDE_PROTOCOL_GUID"
    ),
    UefiGuid("61A4D49E-6F68-4F1B-B922A86EED0B07A2", name="EFI_UGA_IO_PROTOCOL_GUID"),
    UefiGuid("982C298B-F4FA-41CB-B83877AA688FB839", name="EFI_UGA_DRAW_PROTOCOL_GUID"),
    UefiGuid(
        "5B1B31A1-9562-11D2-8E3F00A0C969723B", name="EFI_LOADED_IMAGE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "BC62157E-3E33-4FEC-99202D3B36D750DF",
        name="EFI_LOADED_IMAGE_DEVICE_PATH_PROTOCOL_GUID",
    ),
    UefiGuid("56EC3091-954C-11D2-8E3F00A0C969723B", name="EFI_LOAD_FILE_PROTOCOL_GUID"),
    UefiGuid(
        "4006C0C1-FCB3-403E-996D4A6C8724E06D", name="EFI_LOAD_FILE2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "964E5B22-6459-11D2-8E3900A0C969723B",
        name="EFI_SIMPLE_FILE_SYSTEM_PROTOCOL_GUID",
    ),
    UefiGuid("1E93E633-D65A-459E-AB8493D9EC266D18", name="EFI_TAPE_IO_PROTOCOL_GUID"),
    UefiGuid("CE345171-BA0B-11D2-8E4F00A0C969723B", name="EFI_DISK_IO_PROTOCOL_GUID"),
    UefiGuid("964E5B21-6459-11D2-8E3900A0C969723B", name="EFI_BLOCK_IO_PROTOCOL_GUID"),
    UefiGuid(
        "1D85CD7F-F43D-11D2-9A0C0090273FC14D",
        name="EFI_UNICODE_COLLATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A4C751FC-23AE-4C3E-92E94964CF63F349",
        name="EFI_UNICODE_COLLATION2_PROTOCOL_GUID",
    ),
    UefiGuid(
        "2F707EBB-4A1A-11D4-9A380090273FC14D",
        name="EFI_PCI_ROOT_BRIDGE_IO_PROTOCOL_GUID",
    ),
    UefiGuid("4CF5B200-68B8-4CA5-9EECB23E3F50029A", name="EFI_PCI_IO_PROTOCOL_GUID"),
    UefiGuid("932F47E6-2362-4002-803E3CD54B138F85", name="EFI_SCSI_IO_PROTOCOL_GUID"),
    UefiGuid(
        "143B7632-B81B-4CB7-ABD3B625A5B9BFFE",
        name="EFI_EXT_SCSI_PASS_THRU_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A59E8FCF-BDA0-43BB-90B1D3732ECAA877", name="EFI_SCSI_PASS_THRU_PROTOCOL_GUID"
    ),
    UefiGuid(
        "59324945-EC44-4C0D-B1CD9DB139DF070C",
        name="EFI_I_SCSI_INITIATOR_NAME_PROTOCOL_GUID",
    ),
    UefiGuid("3E745226-9818-45B6-A2ACD7CD0E8BA2BC", name="EFI_USB2_HC_PROTOCOL_GUID"),
    UefiGuid("F5089266-1AA0-4953-97D8562F8A73B519", name="EFI_USB_HC_PROTOCOL_GUID"),
    UefiGuid("2B2F68D6-0CD2-44CF-8E8BBBA20B1B5B75", name="EFI_USB_IO_PROTOCOL_GUID"),
    UefiGuid(
        "FFE06BDD-6107-46A6-7BB25A9C7EC5275C", name="EFI_ACPI_TABLE_PROTOCOL_GUID"
    ),
    UefiGuid("13AC6DD1-73D0-11D4-B06B00AA00BD6DE7", name="EFI_EBC_PROTOCOL_GUID"),
    UefiGuid(
        "A19832B9-AC25-11D3-9A2D0090273FC14D", name="EFI_SIMPLE_NETWORK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "1ACED566-76ED-4218-BC81767F1F977A89",
        name="EFI_NETWORK_INTERFACE_IDENTIFIER_PROTOCOL_GUID_31",
    ),
    UefiGuid(
        "E18541CD-F755-4F73-928D643C8A79B229",
        name="EFI_NETWORK_INTERFACE_IDENTIFIER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "245DCA21-FB7B-11D3-8F0100A0C969723B",
        name="EFI_PXE_BASE_CODE_CALLBACK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "03C4E603-AC28-11D3-9A2D0090273FC14D", name="EFI_PXE_BASE_CODE_PROTOCOL_GUID"
    ),
    UefiGuid("0B64AAB0-5429-11D4-981600A0C91FADCF", name="EFI_BIS_PROTOCOL_GUID"),
    UefiGuid(
        "F36FF770-A7E1-42CF-9ED256F0F271F44C",
        name="EFI_MANAGED_NETWORK_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7AB33A91-ACE5-4326-B572E7EE33D39F16", name="EFI_MANAGED_NETWORK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "F44C00EE-1F2C-4A00-AA091C9F3E0800A3",
        name="EFI_ARP_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("F4B427BB-BA21-4F16-BC4E43E416AB619C", name="EFI_ARP_PROTOCOL_GUID"),
    UefiGuid(
        "9D9A39D8-BD42-4A73-A4D58EE94BE11380",
        name="EFI_DHCP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("8A219718-4EF5-4761-91C8C0F04BDA9E56", name="EFI_DHCP4_PROTOCOL_GUID"),
    UefiGuid(
        "00720665-67EB-4A99-BAF7D3C33A1C7CC9",
        name="EFI_TCP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("65530BC7-A359-410F-B0105AADC7EC2B62", name="EFI_TCP4_PROTOCOL_GUID"),
    UefiGuid(
        "C51711E7-B4BF-404A-BFB80A048EF1FFE4",
        name="EFI_IP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("41D94CD2-35B6-455A-8258D4E51334AADD", name="EFI_IP4_PROTOCOL_GUID"),
    UefiGuid(
        "3B95AA31-3793-434B-8667C8070892E05E", name="EFI_IP4_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid(
        "83F01464-99BD-45E5-B383AF6305D8E9E6",
        name="EFI_UDP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("3AD9DF29-4501-478D-B1F87F7FE70E50F3", name="EFI_UDP4_PROTOCOL_GUID"),
    UefiGuid(
        "2FE800BE-8F01-4AA6-946BD71388E1833F",
        name="EFI_MTFTP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("78247C57-63DB-4708-99C2A8B4A9A61F6B", name="EFI_MTFTP4_PROTOCOL_GUID"),
    UefiGuid(
        "7671D9D0-53DB-4173-AA692327F21F0BC7",
        name="EFI_AUTHENTICATION_INFO_PROTOCOL_GUID",
    ),
    UefiGuid(
        "42881C98-A4F3-44B0-A39DDFA18667D8CD",
        name="EFI_HASH_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("C5184932-DBA5-46DB-A5BACC0BDA9C1435", name="EFI_HASH_PROTOCOL_GUID"),
    UefiGuid("F541796D-A62E-4954-A7759584F61B9CDD", name="EFI_TCG_PROTOCOL_GUID"),
    UefiGuid("607F766C-7455-42BE-930BE4D76DB2720F", name="EFI_TR_E_E_PROTOCOL_GUID"),
    UefiGuid("607F766C-7455-42BE-930BE4D76DB2720F", name="EFI_TCG2_PROTOCOL_GUID"),
    UefiGuid(
        "1E2ED096-30E2-4254-BD89863BBEF82325", name="EFI_TCG2_FINAL_EVENTS_TABLE_GUID"
    ),
    UefiGuid(
        "B9D4C360-BCFB-4F9B-929853C136982258", name="EFI_FORM_BROWSER2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0FD96974-23AA-4CDC-B9CB98D17750322A", name="EFI_HII_STRING_PROTOCOL_GUID"
    ),
    UefiGuid("31A6406A-6BDF-4E46-B2A2EBAA89C40920", name="EFI_HII_IMAGE_PROTOCOL_GUID"),
    UefiGuid(
        "587E72D7-CC50-4F79-8209CA291FC1A10F",
        name="EFI_HII_CONFIG_ROUTING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "EF9FC172-A1B2-4693-B3276D32FC416042", name="EFI_HII_DATABASE_PROTOCOL_GUID"
    ),
    UefiGuid("E9CA4775-8657-47FC-97E77ED65A084324", name="EFI_HII_FONT_PROTOCOL_GUID"),
    UefiGuid(
        "330D4706-F2A0-4E4F-A369B66FA8D54385",
        name="EFI_HII_CONFIG_ACCESS_PROTOCOL_GUID",
    ),
    UefiGuid(
        "6A1EE763-D47A-43B4-AABEEF1DE2AB56FC", name="EFI_HII_PACKAGE_LIST_PROTOCOL_GUID"
    ),
    UefiGuid(
        "EC835DD3-FE0F-617B-A621B350C3E13388",
        name="EFI_IP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("2C8759D5-5C2D-66EF-925FB66C101957E2", name="EFI_IP6_PROTOCOL_GUID"),
    UefiGuid(
        "937FE521-95AE-4D1A-892948BCD90AD31A", name="EFI_IP6_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid(
        "D9760FF3-3CCA-4267-80F97527FAFA4223",
        name="EFI_MTFTP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("BF0A78BA-EC29-49CF-A1C97AE54EAB6A51", name="EFI_MTFTP6_PROTOCOL_GUID"),
    UefiGuid(
        "9FB9A8A1-2F4A-43A6-889CD0F7B6C47AD5",
        name="EFI_DHCP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("87C8BAD7-0595-4053-8297DEDE395F5D5B", name="EFI_DHCP6_PROTOCOL_GUID"),
    UefiGuid(
        "66ED4721-3C98-4D3E-81E3D03DD39A7254",
        name="EFI_UDP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("4F948815-B4B9-43CB-8A3390E060B34955", name="EFI_UDP6_PROTOCOL_GUID"),
    UefiGuid(
        "EC20EB79-6C1A-4664-9A0DD2E4CC16D664",
        name="EFI_TCP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("46E44855-BD60-4AB7-AB0DA679B9447D77", name="EFI_TCP6_PROTOCOL_GUID"),
    UefiGuid(
        "9E23D768-D2F3-4366-9FC33A7ABA864374", name="EFI_VLAN_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid("5D9F96DB-E731-4CAA-A00D72E187CD7762", name="EFI_EAP_PROTOCOL_GUID"),
    UefiGuid(
        "BB62E663-625D-40B2-A088BBE83623A245", name="EFI_EAP_MANAGEMENT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0FAAECB1-226E-4782-AACE7DB9BCBF4DAF",
        name="EFI_FTP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("EB338826-681B-4295-B3562B364C757B09", name="EFI_FTP4_PROTOCOL_GUID"),
    UefiGuid(
        "CE5E5929-C7A3-4602-AD9EC9DAF94EBFCF", name="EFI_IP_SEC_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid(
        "2A534210-9280-41D8-AE79CADA01A2B127", name="EFI_DRIVER_HEALTH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "15853D7C-3DDF-43E0-A1CBEBF85B8F872C",
        name="EFI_DEFERRED_IMAGE_LOAD_PROTOCOL_GUID",
    ),
    UefiGuid(
        "71EE5E94-65B9-45D5-821A3A4D86CFE6BE", name="EFI_USER_CREDENTIAL_PROTOCOL_GUID"
    ),
    UefiGuid(
        "6FD5B00C-D426-4283-98876CF5CF1CB1FE", name="EFI_USER_MANAGER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "1D3DE7F0-0807-424F-AA6911A54E19A46F", name="EFI_ATA_PASS_THRU_PROTOCOL_GUID"
    ),
    UefiGuid(
        "86C77A67-0B97-4633-A18749104D0685C7",
        name="EFI_FIRMWARE_MANAGEMENT_PROTOCOL_GUID",
    ),
    UefiGuid("DFB386F7-E100-43AD-9C9AED90D08A5E12", name="EFI_IP_SEC_PROTOCOL_GUID"),
    UefiGuid("A3979E64-ACE8-4DDC-BC074D66B8FD0977", name="EFI_IP_SEC2_PROTOCOL_GUID"),
    UefiGuid("EC3A978D-7C4E-48FA-9ABE6AD91CC8F811", name="EFI_KMS_PROTOCOL_GUID"),
    UefiGuid("A77B2472-E282-4E9F-A245C2C0E27BBCC1", name="EFI_BLOCK_IO2_PROTOCOL_GUID"),
    UefiGuid(
        "C88B0B6D-0DFC-49A7-9CB449074B4C3A78",
        name="EFI_STORAGE_SECURITY_COMMAND_PROTOCOL_GUID",
    ),
    UefiGuid(
        "E98ADB03-B8B9-4AF8-BA2026E9114CBCE5", name="EFI_USER_CREDENTIAL2_PROTOCOL_GUID"
    ),
    UefiGuid("151C8EAE-7F2C-472C-9E549828194F6A88", name="EFI_DISK_IO2_PROTOCOL_GUID"),
    UefiGuid("AFBFDE41-2E6E-4262-BA6562B9236E5495", name="EFI_TIMESTAMP_PROTOCOL_GUID"),
    UefiGuid("3152BCA5-EADE-433D-862EC01CDC291F44", name="EFI_RNG_PROTOCOL_GUID"),
    UefiGuid(
        "E5DD1403-D622-C24E-8488C71B17F5E802",
        name="EFI_ADAPTER_INFORMATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "52C78312-8EDC-4233-98F21A1AA5E388A5",
        name="EFI_NVM_EXPRESS_PASS_THRU_PROTOCOL_GUID",
    ),
    UefiGuid(
        "DA836F8D-217F-4CA0-99C21CA4E16077EA",
        name="EFI_HASH2_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("55B1D734-C5E1-49DB-9647B16AFB0E305B", name="EFI_HASH2_PROTOCOL_GUID"),
    UefiGuid(
        "A00490BA-3F1A-4B4C-AB904FA99726A1E8", name="EFI_BLOCK_IO_CRYPTO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "2A4D1ADF-21DC-4B81-A42F8B8EE2380060",
        name="EFI_SMART_CARD_READER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "D317F29B-A325-4712-9BF1C61954DC198C", name="EFI_SMART_CARD_EDGE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "32D2963A-FE5D-4F30-B6336E5DC55803CC", name="EFI_USB_FUNCTION_IO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B3930571-BEBA-4FC5-92039427242E6A43", name="EFI_BLUETOOTH_HC_PROTOCOL_GUID"
    ),
    UefiGuid(
        "388278D3-7B85-42F0-ABA9FB4BFD69F5AB",
        name="EFI_BLUETOOTH_IO_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "467313DE-4E30-43F1-943E323F89845DB5", name="EFI_BLUETOOTH_IO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "62960CF3-40FF-4263-A77CDFDEBD191B4B", name="EFI_BLUETOOTH_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B3F79D9A-436C-DC11-B052CD85DF524CE6",
        name="EFI_REGULAR_EXPRESSION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "FEDF8E0C-E147-11E3-9903B8E8562CBAFA",
        name="EFI_BOOT_MANAGER_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "0A8BADD5-03B8-4D19-B1287B8F0EDAA596",
        name="EFI_CONFIG_KEYWORD_HANDLER_PROTOCOL_GUID",
    ),
    UefiGuid("0DA55BC9-45F8-4BB4-87195224F18A4D45", name="EFI_WI_FI_PROTOCOL_GUID"),
    UefiGuid(
        "5E93C847-456D-40B3-A6B478B0C9CF7F20", name="EFI_EAP_MANAGEMENT2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E5B58DBB-7688-44B4-97BF5F1D4B7CC8DB",
        name="EFI_EAP_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "47889FB2-D671-4FAB-A0CADF0E44DF70D6", name="EFI_PKCS7_VERIFY_PROTOCOL_GUID"
    ),
    UefiGuid(
        "5B446ED1-E30B-4FAA-871A3654ECA36080", name="EFI_IP4_CONFIG2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B625B186-E063-44F7-89056A74DC6F52B4",
        name="EFI_DNS4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("AE3D28CC-E05B-4FA1-A0117EB55A3F1401", name="EFI_DNS4_PROTOCOL_GUID"),
    UefiGuid(
        "7F1647C8-B76E-44B2-A565F70FF19CD19E",
        name="EFI_DNS6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("CA37BC1F-A327-4AE9-828A8C40D8506A17", name="EFI_DNS6_PROTOCOL_GUID"),
    UefiGuid(
        "BDC8E6AF-D9BC-4379-A72AE0C4E75DAE1C",
        name="EFI_HTTP_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("7A59B29B-910B-4171-8242A85A0DF25B5B", name="EFI_HTTP_PROTOCOL_GUID"),
    UefiGuid(
        "3E35C163-4074-45DD-431E23989DD86B32", name="EFI_HTTP_UTILITIES_PROTOCOL_GUID"
    ),
    UefiGuid(
        "952CB795-FF36-48CF-A2494DF486D6AB8D",
        name="EFI_TLS_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("00CA959F-6CFA-4DB1-95BCE46C47514390", name="EFI_TLS_PROTOCOL_GUID"),
    UefiGuid(
        "1682FE44-BD7A-4407-B7C7DCA37CA3922D",
        name="EFI_TLS_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid("0DB48A36-4E54-EA9C-9B091EA5BE3A660B", name="EFI_REST_PROTOCOL_GUID"),
    UefiGuid(
        "45BCD98E-59AD-4174-9546344A07485898",
        name="EFI_SUPPLICANT_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "54FCC43E-AA89-4333-9A85CDEA24051E9E", name="EFI_SUPPLICANT_PROTOCOL_GUID"
    ),
    UefiGuid("1B0FB9BF-699D-4FDD-A7C32546681BF63B", name="EFI_WI_FI2_PROTOCOL_GUID"),
    UefiGuid("AB38A0DF-6873-44A9-87E6D4EB56148449", name="EFI_RAM_DISK_PROTOCOL_GUID"),
    UefiGuid(
        "9E66F251-727C-418C-BFD6C2B4252818EA",
        name="EFI_HII_IMAGE_DECODER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "1A1241E6-8F19-41A9-BC0EE8EF39E06546", name="EFI_HII_IMAGE_EX_PROTOCOL_GUID"
    ),
    UefiGuid(
        "716EF0D9-FF83-4F69-81E9518BD39A8E70", name="EFI_SD_MMC_PASS_THRU_PROTOCOL_GUID"
    ),
    UefiGuid(
        "95A9A93E-A86E-4926-AAEF9918E772D987", name="EFI_ERASE_BLOCK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "898890E9-84B2-4F3A-8C58D8577813E0AC",
        name="EFI_BLUETOOTH_ATTRIBUTE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "5639867A-8C8E-408D-AC2F4B61BDC0BBBB",
        name="EFI_BLUETOOTH_ATTRIBUTE_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "8F76DA58-1F99-4275-A4EC4756515B1CE8",
        name="EFI_BLUETOOTH_LE_CONFIG_PROTOCOL_GUID",
    ),
    UefiGuid(
        "B81BFAB0-0EB3-4CF9-84657FA986361664",
        name="EFI_UFS_DEVICE_CONFIG_PROTOCOL_GUID",
    ),
    UefiGuid(
        "BA23B311-343D-11E6-91855820B1D65299",
        name="EFI_HTTP_BOOT_CALLBACK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "9DA34AE0-EAF9-4BBF-8EC3FD60226C44BE",
        name="EFI_RESET_NOTIFICATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "8CF2F62C-BC9B-4821-808DEC9EC421A1A0", name="EFI_PARTITION_INFO_PROTOCOL_GUID"
    ),
    UefiGuid("4311EDC0-6054-46D4-9E40893EA952FCCC", name="EFI_HII_POPUP_PROTOCOL_GUID"),
    UefiGuid(
        "D40B6B80-97D5-4282-BB1D223A16918058", name="EFI_NVDIMM_LABEL_PROTOCOL_GUID"
    ),
    UefiGuid("6302D008-7F9B-4F30-87AC60C9FEF5DA4E", name="EFI_SHELL_PROTOCOL_GUID"),
    UefiGuid(
        "752F3136-4E16-4FDC-A22AE5F46812F4CA", name="EFI_SHELL_PARAMETERS_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3C7200E9-005F-4EA4-87DEA3DFAC8A27C3",
        name="EFI_SHELL_DYNAMIC_COMMAND_PROTOCOL_GUID",
    ),
    UefiGuid(
        "40E064B2-0AE0-48B1-A07DF8CF1E1A2310", name="EFI_NETWORK_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("02EEA107-98DB-400E-9830460A1542D799", name="IP6_CONFIG_NV_DATA_GUID"),
    UefiGuid("4B47D616-A8D6-4552-9D44CCAD2E0F4CF9", name="I_SCSI_CONFIG_GUID"),
    UefiGuid("4D20583A-7765-4E7A-8A67DCDE74EE3EC5", name="HTTP_BOOT_CONFIG_GUID"),
    UefiGuid("B0EAE4F8-9A04-4C6D-A748793DAA0F65DF", name="TLS_AUTH_CONFIG_GUID"),
    UefiGuid("FD2340D0-3DAB-4349-A6C73B4F12B48EAE", name="EFI_TLS_CA_CERTIFICATE_GUID"),
    UefiGuid(
        "46DDB415-5244-49C7-9374F0E298E7D386", name="EDKII_HTTP_TLS_CIPHER_LIST_GUID"
    ),
    UefiGuid("9F94D327-0B18-4245-8FF2832E300D2CEF", name="WIFI_CONFIG_GUID"),
    UefiGuid("9B942747-154E-4D29-A436BF7100C8B53B", name="IP4_CONFIG2_NV_DATA_GUID"),
    UefiGuid("D79DF6B0-EF44-43BD-979743E93BCF5FA8", name="VLAN_CONFIG_FORM_SET_GUID"),
    UefiGuid("6456ED61-3579-41C9-8A260A0BD62B78FC", name="IP4_I_SCSI_CONFIG_GUID"),
    UefiGuid(
        "786EC0AC-65AE-4D1B-B1370D110A483797", name="I_SCSI_C_H_A_P_AUTH_INFO_GUID"
    ),
    UefiGuid("480F8AE9-0C46-4AA9-BC89DB9FBA619806", name="EFI_DPC_PROTOCOL_GUID"),
    UefiGuid(
        "93BB96AF-B9F2-4EB8-9462E0BA74564236", name="UEFI_OVMF_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("D3B46F3B-D441-1244-9A120012273FC14D", name="EFI_XEN_INFO_GUID"),
    UefiGuid(
        "4E32566D-8E9E-4F52-81D35BB9715F9727", name="OVMF_PK_KEK1_APP_PREFIX_GUID"
    ),
    UefiGuid("7235C51C-0C80-4CAB-87AC3B084A6304B1", name="OVMF_PLATFORM_CONFIG_GUID"),
    UefiGuid("837DCA9E-E874-4D82-B29A23FE0E23D1E2", name="VIRTIO_MMIO_TRANSPORT_GUID"),
    UefiGuid("557423A1-63AB-406C-BE7E91CDBC08C457", name="QEMU_RAMFB_GUID"),
    UefiGuid("A732241F-383D-4D9C-8AE18E09837589D7", name="XEN_BUS_ROOT_DEVICE_GUID"),
    UefiGuid(
        "24A2D66F-EEDD-4086-9042F26E4797EE69",
        name="ROOT_BRIDGES_CONNECTED_EVENT_GROUP_GUID",
    ),
    UefiGuid("77FA9ABD-0359-4D32-BD6028F4E78F784B", name="MICROSOFT_VENDOR_GUID"),
    UefiGuid("2E3044AC-879F-490F-9760BBDFAF695F50", name="EFI_LEGACY_BIOS_GUID"),
    UefiGuid(
        "A56074DB-65FE-45F7-BD212D2BDD8E9652", name="EFI_LEGACY_DEV_ORDER_VARIABLE_GUID"
    ),
    UefiGuid("FA920010-6785-4941-B6EC498C579F160A", name="VIRTIO_DEVICE_PROTOCOL_GUID"),
    UefiGuid("3D3CA290-B9A5-11E3-B75DB8AC6F7D65E6", name="XEN_BUS_PROTOCOL_GUID"),
    UefiGuid("6EFAC84F-0AB0-4747-81BE855562590449", name="XEN_IO_PROTOCOL_GUID"),
    UefiGuid("F8775D50-8ABD-4ADF-92AC853E51F6C8DC", name="IO_MMU_ABSENT_PROTOCOL_GUID"),
    UefiGuid(
        "38321DBA-4FE0-4E17-8AEC413055EAEDC1", name="EFI_LEGACY8259_PROTOCOL_GUID"
    ),
    UefiGuid(
        "389F751F-1838-4388-8390CD8154BD27F8", name="EFI_FIRMWARE_VOLUME_PROTOCOL_GUID"
    ),
    UefiGuid("64A892DC-5561-4536-92C7799BFC183355", name="EFI_ISA_ACPI_PROTOCOL_GUID"),
    UefiGuid("7EE2BD44-3DA0-11D4-9A380090273FC14D", name="EFI_ISA_IO_PROTOCOL_GUID"),
    UefiGuid(
        "DB9A1E3D-45CB-4ABB-853BE5387FDB2E2D", name="EFI_LEGACY_BIOS_PROTOCOL_GUID"
    ),
    UefiGuid(
        "783658A3-4172-4421-A299E009079C0CB4",
        name="EFI_LEGACY_BIOS_PLATFORM_PROTOCOL_GUID",
    ),
    UefiGuid(
        "31CE593D-108A-485D-ADB278F21F2966BE", name="EFI_LEGACY_INTERRUPT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "C7735A2F-88F5-4882-AE63FAAC8C8B86B3", name="EFI_VGA_MINI_PORT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "326AE723-AE32-4589-98B8CAC23CDCC1B1", name="PC_AT_CHIPSET_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "0D3FB176-9569-4D51-A3EF7D61C64FEABA", name="EFI_SECURITY_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "F0A30BC7-AF08-4556-99C4001009C93A44",
        name="EFI_SECURE_BOOT_ENABLE_DISABLE_GUID",
    ),
    UefiGuid("C076EC0C-7028-4399-A07271EE5C448B9F", name="EFI_CUSTOM_MODE_ENABLE_GUID"),
    UefiGuid("9073E4E0-60EC-4B6E-99034C223C260F3C", name="EFI_VENDOR_KEYS_NV_GUID"),
    UefiGuid("D9BEE56E-75DC-49D9-B4D7B534210F637A", name="EFI_CERT_DB_GUID"),
    UefiGuid("2B9FFB52-1B13-416F-A87BBC930DEF92A8", name="TCG_EVENT_ENTRY_HOB_GUID"),
    UefiGuid("D26C221E-2430-4C8A-91703FCB4500413F", name="TCG_EVENT2_ENTRY_HOB_GUID"),
    UefiGuid("EF598499-B25E-473A-BFAFE7E57DCE82C4", name="TPM_ERROR_HOB_GUID"),
    UefiGuid(
        "0397B0C9-22E8-459E-A4FF99BC65270929", name="TPM2_STARTUP_LOCALITY_HOB_GUID"
    ),
    UefiGuid(
        "E2C3BC69-615C-4B5B-8E5CA033A9C25ED6",
        name="TCG800155_PLATFORM_ID_EVENT_HOB_GUID",
    ),
    UefiGuid("B2360B42-7173-420A-869646CA6BAB1060", name="MEASURED_FV_HOB_GUID"),
    UefiGuid("0F6499B1-E9AD-493D-B9C22F90815C6CBC", name="EFI_PHYSICAL_PRESENCE_GUID"),
    UefiGuid(
        "AEB9C5C1-94F1-4D02-BFD94602DB2D3C54", name="EFI_TCG2_PHYSICAL_PRESENCE_GUID"
    ),
    UefiGuid("3CCD3DD8-8D45-4FED-962D2B38CD82B3C4", name="USER_IDENTIFY_MANAGER_GUID"),
    UefiGuid("B0F901E4-C424-45DE-908195E20BDE6FB5", name="TCG_CONFIG_FORM_SET_GUID"),
    UefiGuid("6339D487-26BA-424B-9A5D687E25D740BC", name="TCG2_CONFIG_FORM_SET_GUID"),
    UefiGuid(
        "5DAF50A5-EA81-4DE2-8F9BCABDA9CF5C14", name="SECURE_BOOT_CONFIG_FORM_SET_GUID"
    ),
    UefiGuid(
        "00000000-0000-0000-0000000000000000", name="EFI_TPM_DEVICE_INSTANCE_NONE_GUID"
    ),
    UefiGuid(
        "8B01E5B6-4F19-46E8-AB931C53671B90CC", name="EFI_TPM_DEVICE_INSTANCE_TPM12_GUID"
    ),
    UefiGuid(
        "286BF25A-C2C3-408C-B3B425E6758B7317",
        name="EFI_TPM_DEVICE_INSTANCE_TPM20_DTPM_GUID",
    ),
    UefiGuid(
        "7F4158D3-074D-456D-8CB201F9C8F79DAA", name="EFI_TPM_DEVICE_SELECTED_GUID"
    ),
    UefiGuid(
        "44A2AD5D-612C-47B3-B06EC8F50BFBF07D", name="OPAL_EXTRA_INFO_VARIABLE_GUID"
    ),
    UefiGuid(
        "EF9AEFE5-2BD3-4031-AF7D5EFE5ABB9A0D",
        name="PEI_LOCK_PHYSICAL_PRESENCE_PPI_GUID",
    ),
    UefiGuid(
        "E9DB0D58-D48D-47F6-9C6E6F40E86C7B41", name="PEI_TPM_INITIALIZED_PPI_GUID"
    ),
    UefiGuid(
        "A030D115-54DD-447B-9064F206883D7CCC",
        name="PEI_TPM_INITIALIZATION_DONE_PPI_GUID",
    ),
    UefiGuid(
        "6E056FF9-C695-4364-9E2C6126F5CEEAAE",
        name="EFI_PEI_FIRMWARE_VOLUME_INFO_MEASUREMENT_EXCLUDED_PPI_GUID",
    ),
    UefiGuid(
        "3CE1E631-7008-477C-ADA75DCFC7C1494B",
        name="EDKII_PEI_FIRMWARE_VOLUME_INFO_PREHASHED_FV_PPI_GUID",
    ),
    UefiGuid(
        "7F5E4E31-81B1-47E5-9E211E4B5BC2F61D",
        name="EDKII_PEI_FIRMWARE_VOLUME_INFO_STORED_HASH_FV_PPI_GUID",
    ),
    UefiGuid(
        "D2C18636-40E5-4EB5-A31B36695FD42C87", name="EFI_SHELL_ENVIRONMENT2_EXT_GUID"
    ),
    UefiGuid(
        "171E9188-31D3-40F5-B10C539B2DB940CD", name="EFI_SHELL_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("158DEF5A-F656-419C-B0277A3192C079D2", name="SHELL_VARIABLE_GUID"),
    UefiGuid("51271E13-7DE3-43AF-8BC271AD3B824325", name="SHELL_MAP_GUID"),
    UefiGuid("0053D9D6-2659-4599-A26BEF4536E631A9", name="SHELL_ALIAS_GUID"),
    UefiGuid("B8969637-81DE-43AF-BC9A24D98913F2F6", name="HANDLE_PARSING_HII_GUID"),
    UefiGuid("25F200AA-D3CB-470A-BF51E7D162D22E6F", name="SHELL_DEBUG1_HII_GUID"),
    UefiGuid("0AF0B742-63EC-45BD-8DB671AD7F2FE8E8", name="SHELL_DRIVER1_HII_GUID"),
    UefiGuid("7D574D54-D364-4D4A-95E34945DB7AD3EE", name="SHELL_INSTALL1_HII_GUID"),
    UefiGuid("DEC5DAA4-6781-4820-9C63A7B0E4F1DB31", name="SHELL_LEVEL1_HII_GUID"),
    UefiGuid("F95A7CCC-4C55-4426-A7B4DC8961950BAE", name="SHELL_LEVEL2_HII_GUID"),
    UefiGuid("4344558D-4EF9-4725-B1E43376E8D6974F", name="SHELL_LEVEL3_HII_GUID"),
    UefiGuid("F3D301BB-F4A5-45A8-B0B7FA999C6237AE", name="SHELL_NETWORK1_HII_GUID"),
    UefiGuid("0174B2B5-F505-4B12-AA6059DFF8D6EA37", name="SHELL_NETWORK2_HII_GUID"),
    UefiGuid("738A9314-82C1-4592-8FF7C1BDF1B20ED4", name="SHELL_TFTP_HII_GUID"),
    UefiGuid("5F5F605D-1583-4A2D-A6B2EB12DAB4A2B6", name="SHELL_BCFG_HII_GUID"),
    UefiGuid("DA8CCDF4-ED8F-4FFC-B5EF2EF55E24932A", name="SHELL_ACPI_VIEW_HII_GUID"),
    UefiGuid("7C04A583-9E3E-4F1C-AD65E05268D0B4D1", name="UEFI_SHELL_FILE_GUID"),
    UefiGuid("47C7B221-C42A-11D2-8E5700A0C969723B", name="EFI_SHELL_ENVIRONMENT2_GUID"),
    UefiGuid("47C7B223-C42A-11D2-8E5700A0C969723B", name="EFI_SHELL_INTERFACE_GUID"),
    UefiGuid(
        "E1EB612F-1C6C-485D-9D06650844881569",
        name="EFI_SIGNED_CAPSULE_PKG_TOKEN_SPACE_GUID",
    ),
    UefiGuid(
        "90B2B846-CA6D-4D6E-A8D3C140A8E110AC",
        name="EDKII_SYSTEM_FIRMWARE_IMAGE_DESCRIPTOR_FILE_GUID",
    ),
    UefiGuid(
        "812136D3-4D3A-433A-941829BB9BF78F6E",
        name="EDKII_SYSTEM_FMP_CAPSULE_CONFIG_FILE_GUID",
    ),
    UefiGuid(
        "CE57B167-B0E4-41E8-A8975F4FEB781D40",
        name="EDKII_SYSTEM_FMP_CAPSULE_DRIVER_FV_FILE_GUID",
    ),
    UefiGuid("865A5A9B-B85D-474C-845565D1BE844BE2", name="EFI_DEBUG_AGENT_GUID"),
    UefiGuid(
        "865A5AAB-B85D-474C-845565D1BE844BE2",
        name="EFI_SOURCE_LEVEL_DEBUG_PKG_TOKEN_SPACE_GUID",
    ),
    UefiGuid(
        "18FE7632-F5C8-4E63-8DE817A55C5913BD", name="STANDALONE_MM_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("BA33F15D-4000-45C1-8E88F91692D457E3", name="MP_INFORMATION_HOB_GUID"),
    UefiGuid("B65694CC-09E3-4C3B-B5CD05F44D3CDBFF", name="MM_FV_DISPATCH_GUID"),
    UefiGuid("A160BF99-2AA4-4D7D-9993899CB12DF376", name="MM_CORE_DATA_HOB_GUID"),
    UefiGuid(
        "0703F912-BF8D-4E2A-BE07AB272525C592",
        name="EFI_MM_PEI_MMRAM_MEMORY_RESERVE_GUID",
    ),
    UefiGuid(
        "F00497E3-BFA2-41A1-9D2954C2E93721C5",
        name="EFI_STANDALONE_MM_NON_SECURE_BUFFER_GUID",
    ),
    UefiGuid(
        "6ECBD5A1-C0F8-4702-83014FC2C5470A51",
        name="EFI_ARM_TF_CPU_DRIVER_EP_DESCRIPTOR_GUID",
    ),
    UefiGuid(
        "AC05BF33-995A-4ED4-AAB8EF7AE80F5CB0", name="UEFI_CPU_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("5802BCE4-EEEE-4E33-A130EBAD27F0E439", name="MSEG_SMRAM_GUID"),
    UefiGuid(
        "A82485CE-AD6B-4101-99D3E1358C9E7E37", name="EDKII_CPU_FEATURES_SET_DONE_GUID"
    ),
    UefiGuid(
        "C77C3A41-61AB-4143-983E3339280628E5", name="EDKII_CPU_FEATURES_INIT_DONE_GUID"
    ),
    UefiGuid(
        "D178F11D-8716-418E-A131967D2AC42843", name="EDKII_MICROCODE_PATCH_HOB_GUID"
    ),
    UefiGuid(
        "1D202CAB-C8AB-4D5C-94F73CFCC0D3D335", name="EFI_SMM_CPU_SERVICE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "228F344D-B3DE-43BB-A4D7EA200B1B1482", name="EFI_SM_MONITOR_INIT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "5CB9CB3D-31A4-480C-949829D269BACFBA", name="EDKII_PEI_MP_SERVICES2_PPI_GUID"
    ),
    UefiGuid(
        "01D127EA-F6F1-4EF6-94158A000093F89D", name="UEFI_PAYLOAD_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("878AC2CC-5343-46F2-B56351F89DAF56BA", name="BMP_IMAGE_GUID"),
    UefiGuid("16C8A6D0-FE8A-4082-A208CF89C4290433", name="UEFI_SYSTEM_TABLE_INFO_GUID"),
    UefiGuid("0AD3D31B-B3D8-4506-AE712EF11006D90F", name="UEFI_ACPI_BOARD_INFO_GUID"),
    UefiGuid("6C6872FE-56A9-4403-BB98958D62DE87F1", name="UEFI_SERIAL_PORT_INFO_GUID"),
    UefiGuid("A1FF7424-7A1A-478E-A9E492F357D12832", name="LOADER_MEMORY_MAP_INFO_GUID"),
    UefiGuid(
        "DBE23AA1-A342-4B97-85B6B226F1617389", name="EFI_PAY_LOAD_HOB_BASE_PPI_GUID"
    ),
    UefiGuid("EC2E931B-3281-48A5-8107DF8A8BED3C5D", name="PLATFORM_G_O_P_POLICY_GUID"),
    UefiGuid("EB9D2D30-2D88-11D3-9A160090273FC14D", name="ACPI_TABLE_GUID"),
    UefiGuid(
        "2EA9743A-23D9-425E-872CF615AA195788",
        name="APPLE_REMOVABLE_MEDIA_PROTOCOL_GUID",
    ),
    UefiGuid(
        "AB1C1816-D542-4E6F-9B1E8ECD9253E2E7", name="ARM_GLOBAL_VARIABLE_PPI_GUID"
    ),
    UefiGuid(
        "C3253C90-A24F-4599-A6641F8813778FC9", name="ARM_HOB_GLOBAL_VARIABLE_GUID"
    ),
    UefiGuid("A4EE0728-E5D7-4AC5-B21E658ED857E834", name="ARM_MP_CORE_INFO_GUID"),
    UefiGuid("6847CC74-E9EC-4F8F-A29DAB44E754A8FC", name="ARM_MP_CORE_INFO_PPI_GUID"),
    UefiGuid("3B4D9B23-95AC-44F6-9FCD0E9594586C72", name="BDS_LIB_STRING_PACKAGE_GUID"),
    UefiGuid("CF31FAC5-C24E-11D2-85F300A0C93EC93B", name="BLOCKIO_VENDOR_GUID"),
    UefiGuid("6B558CE3-69E5-4C67-A634F7FE72ADBE84", name="BLOCK_MMIO_PROTOCOL_GUID"),
    UefiGuid("642237C7-35D4-472D-836512E0CCF27A22", name="BOOT_MAINT_FORMSET_GUID"),
    UefiGuid("847BC3FE-B974-446D-94495AD5412E993B", name="BOOT_MANAGER_FORMSET_GUID"),
    UefiGuid("DB4E8151-57ED-4BED-88336751B5D1A8D7", name="CONNECT_CONIN_EVENT_GUID"),
    UefiGuid("3EBFA8E6-511D-4B5B-A95FFB38260F1C27", name="DEVICE_MANAGER_FORMSET_GUID"),
    UefiGuid("EB832FD9-9089-4898-83C941618F5C48B9", name="DP_HII_GUID"),
    UefiGuid("F76E0A70-B5ED-4C38-AC9AE5F54BF16E34", name="DRIVER_HEALTH_FORMSET_GUID"),
    UefiGuid("A04A27F4-DF00-4D42-B55239511302113D", name="DRIVER_SAMPLE_FORMSET_GUID"),
    UefiGuid(
        "B3F56470-6141-4621-8F19704E577AA9E8", name="DRIVER_SAMPLE_INVENTORY_GUID"
    ),
    UefiGuid("ED150714-DF30-407D-B24A4B742FD5CEA2", name="DUET_CONSOLEOUT_CONFIG_GUID"),
    UefiGuid("D6A2CB7F-6A18-4E2F-B43B9920A733700A", name="DXE_CORE_FILE_NAME_GUID"),
    UefiGuid("05AD34BA-6F02-4214-952E4DA0398E2BB9", name="DXE_SERVICES_TABLE_GUID"),
    UefiGuid(
        "AEDA2428-9A22-4637-9B21545E28FBB829", name="EBL_ADD_COMMAND_PROTOCOL_GUID"
    ),
    UefiGuid("B0EE53D4-A049-4A79-B2FF19D9FAEFAA94", name="ECP_PEI_PCI_CFG_PPI_GUID"),
    UefiGuid("0EF98D3A-3E33-497A-A40177BE3EB74F38", name="EFI_ACPI_S3_CONTEXT_GUID"),
    UefiGuid(
        "C020489E-6DB2-4EF2-9AA5CA06FC11D36A",
        name="EFI_ACPI_VARIABLE_COMPATIBILITY_GUID",
    ),
    UefiGuid(
        "1D3DE7F0-0807-424F-AA6911A54E19A46F", name="EFI_ATA_PASS_THRU_PROTOCOL_GUID"
    ),
    UefiGuid(
        "AAF32C78-947B-439A-A1802E144EC37792", name="EFI_AUTHENTICATED_VARIABLE_GUID"
    ),
    UefiGuid(
        "C280C73E-15CA-11DA-B0CA001083FFCA4D", name="EFI_AUTHENTICATION_CHAP_LOCAL_GUID"
    ),
    UefiGuid(
        "D6062B50-15CA-11DA-9219001083FFCA4D",
        name="EFI_AUTHENTICATION_CHAP_RADIUS_GUID",
    ),
    UefiGuid("A77B2472-E282-4E9F-A245C2C0E27BBCC1", name="EFI_BLOCK_IO2_PROTOCOL_GUID"),
    UefiGuid("CDEA2BD3-FC25-4C1C-B97CB31186064990", name="EFI_BOOT_LOGO_PROTOCOL_GUID"),
    UefiGuid(
        "79CB58C4-AC51-442F-AFD798E47D2E9908",
        name="EFI_BOOT_SCRIPT_EXECUTOR_CONTEXT_GUID",
    ),
    UefiGuid(
        "3079818C-46D4-4A73-AEF3E3E46CF1EEDB",
        name="EFI_BOOT_SCRIPT_EXECUTOR_VARIABLE_GUID",
    ),
    UefiGuid("0B6E5233-A65C-44C9-9407D9AB83BFC8BD", name="EFI_CERT_SHA224_GUID"),
    UefiGuid("FF3E5307-9FD0-48C9-85F18AD56C701E01", name="EFI_CERT_SHA384_GUID"),
    UefiGuid("093E0FAE-A6C4-4F50-9F1BD41E2B89C19A", name="EFI_CERT_SHA512_GUID"),
    UefiGuid("A5C059A1-94E4-4AA7-87B5AB155C2BF072", name="EFI_CERT_X509_GUID"),
    UefiGuid(
        "F42F7782-012E-4C12-995649F94304F721", name="EFI_CONSOLE_CONTROL_PROTOCOL_GUID"
    ),
    UefiGuid(
        "FC1BCDB0-7D31-49AA-936AA4600D9DD083",
        name="EFI_CRC32_GUIDED_SECTION_EXTRACTION_GUID",
    ),
    UefiGuid(
        "D083E94C-6560-42E4-B6D42DF75ADF6A2A",
        name="EFI_DATA_HUB_STATUS_CODE_RECORD_GUID",
    ),
    UefiGuid("865A5A9B-B85D-474C-845565D1BE844BE2", name="EFI_DEBUG_AGENT_GUID"),
    UefiGuid(
        "E683DC4F-09ED-4F22-866B8E4046947C6C", name="EFI_DEBUG_SERIAL_IO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "9546E07C-2CBB-4C88-986CCD341086F044",
        name="EFI_DEBUG_SUPPORT_PERIODIC_CALLBACK_PROTOCOL_GUID",
    ),
    UefiGuid("7BB28B99-61BB-11D5-9A5D0090273FC14D", name="EFI_DEFAULT_BMP_LOGO_GUID"),
    UefiGuid("87C8BAD7-0595-4053-8297DEDE395F5D5B", name="EFI_DHCP6_PROTOCOL_GUID"),
    UefiGuid(
        "9FB9A8A1-2F4A-43A6-889CD0F7B6C47AD5",
        name="EFI_DHCP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("480F8AE9-0C46-4AA9-BC89DB9FBA619806", name="EFI_DPC_PROTOCOL_GUID"),
    UefiGuid(
        "BB62E663-625D-40B2-A088BBE83623A245", name="EFI_EAP_MANAGEMENT_PROTOCOL_GUID"
    ),
    UefiGuid("5D9F96DB-E731-4CAA-A00D72E187CD7762", name="EFI_EAP_PROTOCOL_GUID"),
    UefiGuid(
        "2A72D11E-7376-40F6-9C6823FA2FE363F1",
        name="EFI_EBC_SIMPLE_DEBUGGER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "AAEACCFD-F27B-4C17-B61075CA1F2DFB52", name="EFI_EBC_VM_TEST_PROTOCOL_GUID"
    ),
    UefiGuid("F2BA331A-8985-11DB-A4060040D02B1835", name="EFI_EMU_PHYSICAL_DISK_GUID"),
    UefiGuid("9C4FB516-3A1E-D847-A1A17058B6986732", name="EFI_EMU_SYSTEM_CONFIG_GUID"),
    UefiGuid("F2BA331A-8985-11DB-A4060040D02B1835", name="EFI_EMU_VIRTUAL_DISK_GUID"),
    UefiGuid(
        "02CE967A-DD7E-4FFC-9EE7810CF0470880", name="EFI_END_OF_DXE_EVENT_GROUP_GUID"
    ),
    UefiGuid(
        "71761D37-32B2-45CD-A7D0B0FEDD93E8CF",
        name="EFI_ERROR_SECTION_DIRECTED_IO_DMAR_GUID",
    ),
    UefiGuid(
        "5B51FEF7-C79D-4434-8F1BAA62DE3E2C64",
        name="EFI_ERROR_SECTION_DMAR_GENERIC_GUID",
    ),
    UefiGuid(
        "81212A96-09ED-4996-94718D729C8E69ED",
        name="EFI_ERROR_SECTION_FW_ERROR_RECORD_GUID",
    ),
    UefiGuid(
        "036F84E1-7F37-428C-A79E575FDFAA84EC", name="EFI_ERROR_SECTION_IOMMU_DMAR_GUID"
    ),
    UefiGuid("D995E954-BBC1-430F-AD91B44DCB3C6F35", name="EFI_ERROR_SECTION_PCIE_GUID"),
    UefiGuid(
        "EB5E4685-CA66-4769-B6A226068B001326", name="EFI_ERROR_SECTION_PCI_DEVICE_GUID"
    ),
    UefiGuid(
        "C5753963-3B84-4095-BF78EDDAD3F9C9DD",
        name="EFI_ERROR_SECTION_PCI_PCIX_BUS_GUID",
    ),
    UefiGuid(
        "A5BC1114-6F64-4EDE-B8633E83ED7C83B1",
        name="EFI_ERROR_SECTION_PLATFORM_MEMORY_GUID",
    ),
    UefiGuid(
        "9876CCAD-47B4-4BDB-B65E16F193C4F3DB",
        name="EFI_ERROR_SECTION_PROCESSOR_GENERIC_GUID",
    ),
    UefiGuid(
        "DC3EA0B0-A144-4797-B95B53FA242B6E1D",
        name="EFI_ERROR_SECTION_PROCESSOR_SPECIFIC_GUID",
    ),
    UefiGuid(
        "7081E22F-CAC6-4053-9468675782CF88E5", name="EFI_EVENT_GROUP_DXE_DISPATCH_GUID"
    ),
    UefiGuid(
        "3D61A466-AB40-409A-A698F362D464B38F",
        name="EFI_EVENT_NOTIFICATION_TYEP_BOOT_GUID",
    ),
    UefiGuid(
        "2DCE8BB1-BDD7-450E-B9AD9CF4EBD4F890",
        name="EFI_EVENT_NOTIFICATION_TYEP_CMC_GUID",
    ),
    UefiGuid(
        "4E292F96-D843-4A55-A8C2D481F27EBEEE",
        name="EFI_EVENT_NOTIFICATION_TYEP_CPE_GUID",
    ),
    UefiGuid(
        "667DD791-C6B3-4C27-8A6B0F8E722DEB41",
        name="EFI_EVENT_NOTIFICATION_TYEP_DMAR_GUID",
    ),
    UefiGuid(
        "CC5263E8-9308-454A-89D0340BD39BC98E",
        name="EFI_EVENT_NOTIFICATION_TYEP_INIT_GUID",
    ),
    UefiGuid(
        "E8F56FFE-919C-4CC5-BA8865ABE14913BB",
        name="EFI_EVENT_NOTIFICATION_TYEP_MCE_GUID",
    ),
    UefiGuid(
        "5BAD89FF-B7E6-42C9-814ACF2485D6E98A",
        name="EFI_EVENT_NOTIFICATION_TYEP_NMI_GUID",
    ),
    UefiGuid(
        "CF93C01F-1A16-4DFC-B8BC9C4DAF67C104",
        name="EFI_EVENT_NOTIFICATION_TYEP_PCIE_GUID",
    ),
    UefiGuid(
        "0F9D89E8-9259-4F76-A5AF0C89E34023DF", name="EFI_FIRMWARE_CONTENTS_SIGNED_GUID"
    ),
    UefiGuid(
        "5473C07A-3DCB-4DCA-BD6F1E9689E7349A", name="EFI_FIRMWARE_FILE_SYSTEM3_GUID"
    ),
    UefiGuid(
        "86C77A67-0B97-4633-A18749104D0685C7",
        name="EFI_FIRMWARE_MANAGEMENT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "C095791A-3001-47B2-80C9EAC7319F2FA4", name="EFI_FIRMWARE_PERFORMANCE_GUID"
    ),
    UefiGuid(
        "0FB7C852-ADCA-4853-8D0FFBA71B1CE11A",
        name="EFI_FORM_BROWSER_COMPATIBILITY_PROTOCOL_GUID",
    ),
    UefiGuid("EB338826-681B-4295-B3562B364C757B09", name="EFI_FTP4_PROTOCOL_GUID"),
    UefiGuid(
        "0FAAECB1-226E-4782-AACE7DB9BCBF4DAF",
        name="EFI_FTP4_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "24C5DC2F-53E2-40CA-9ED6A5D9A49F463B", name="EFI_HASH_ALGORITHM_SHA1_NOPAD_GUID"
    ),
    UefiGuid(
        "8628752A-6CB7-4814-96FC24A815AC2226",
        name="EFI_HASH_ALGORITHM_SHA256_NOPAD_GUID",
    ),
    UefiGuid(
        "5542CCE1-DF5C-4D1B-ABCA364F77D399FB",
        name="EFI_HII_COMPATIBILITY_PROTOCOL_GUID",
    ),
    UefiGuid("3BD2F4EC-E524-46E4-A9D8510117425562", name="EFI_HII_STANDARD_FORM_GUID"),
    UefiGuid(
        "1CF3F8B3-C5B1-49A2-AA595EEF92FFA63C",
        name="EFI_IA32_X64_ERROR_TYPE_BUS_CHECK_GUID",
    ),
    UefiGuid(
        "A55701F5-E3EF-43DE-AC72249B573FAD2C",
        name="EFI_IA32_X64_ERROR_TYPE_CACHE_CHECK_GUID",
    ),
    UefiGuid(
        "48AB7F57-DC34-4F6C-A7D3B0B5B0A74314",
        name="EFI_IA32_X64_ERROR_TYPE_MS_CHECK_GUID",
    ),
    UefiGuid(
        "FC06B535-5E1F-4562-9F250A3B9ADB63C3",
        name="EFI_IA32_X64_ERROR_TYPE_TLB_CHECK_GUID",
    ),
    UefiGuid("31CA5D1A-D511-4931-B782AE6B2B178CD7", name="EFI_IFR_FRAMEWORK_GUID"),
    UefiGuid("F5E655D9-02A6-46F2-9E76B8BE8E60AB22", name="EFI_IFR_REFRESH_ID_OP_GUID"),
    UefiGuid(
        "937FE521-95AE-4D1A-892948BCD90AD31A", name="EFI_IP6_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid("2C8759D5-5C2D-66EF-925FB66C101957E2", name="EFI_IP6_PROTOCOL_GUID"),
    UefiGuid(
        "EC835DD3-FE0F-617B-A621B350C3E13388",
        name="EFI_IP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("A3979E64-ACE8-4DDC-BC074D66B8FD0977", name="EFI_IPSEC2_PROTOCOL_GUID"),
    UefiGuid(
        "CE5E5929-C7A3-4602-AD9EC9DAF94EBFCF", name="EFI_IPSEC_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid("DFB386F7-E100-43AD-9C9AED90D08A5E12", name="EFI_IPSEC_PROTOCOL_GUID"),
    UefiGuid(
        "A0E8EE6A-0E92-44D4-861B0EAA4ACA44A2", name="EFI_KMS_FORMAT_AESCBC_128_GUID"
    ),
    UefiGuid(
        "D7E69789-1F68-45E8-96EF3B6407A5B2DC", name="EFI_KMS_FORMAT_AESCBC_256_GUID"
    ),
    UefiGuid(
        "4776E33F-DB47-479A-A25FA1CD0AFAB38B", name="EFI_KMS_FORMAT_AESXTS_128_GUID"
    ),
    UefiGuid(
        "DC7E8613-C4BB-4DB0-846213511357ABE2", name="EFI_KMS_FORMAT_AESXTS_256_GUID"
    ),
    UefiGuid(
        "43BE0B44-874B-4EAD-B09C241A4FBD7EB3", name="EFI_KMS_FORMAT_GENERIC_1024_GUID"
    ),
    UefiGuid(
        "EC8A3D69-6DDF-4108-94767337FC522136", name="EFI_KMS_FORMAT_GENERIC_128_GUID"
    ),
    UefiGuid(
        "A3B3E6F8-EFCA-4BC1-88FBCB87339B2579", name="EFI_KMS_FORMAT_GENERIC_160_GUID"
    ),
    UefiGuid(
        "40093F23-630C-4626-9C4840373B19CBBE", name="EFI_KMS_FORMAT_GENERIC_2048_GUID"
    ),
    UefiGuid(
        "70F64793-C323-4261-AC2CD876F27C5345", name="EFI_KMS_FORMAT_GENERIC_256_GUID"
    ),
    UefiGuid(
        "B9237513-6C44-4411-A99021E556E05ADE", name="EFI_KMS_FORMAT_GENERIC_3072_GUID"
    ),
    UefiGuid(
        "978FE043-D7AF-422E-8A922B48E463BDE6", name="EFI_KMS_FORMAT_GENERIC_512_GUID"
    ),
    UefiGuid("78BE11C4-EE44-4A22-9F0503852EC5C978", name="EFI_KMS_FORMAT_MD2_128_GUID"),
    UefiGuid("D1C17AA1-CAC5-400F-BE17E2A2AE06677C", name="EFI_KMS_FORMAT_MD4_128_GUID"),
    UefiGuid(
        "1C178237-6897-459E-9D3667CE8EF94F76", name="EFI_KMS_FORMAT_MD5SHA_128_GUID"
    ),
    UefiGuid("DCBC3662-9CDA-4B52-A04C82EB1D2348C7", name="EFI_KMS_FORMAT_MD5_128_GUID"),
    UefiGuid(
        "F7AD60F8-EFA8-44A3-9113231F399EB4C7", name="EFI_KMS_FORMAT_MDC2_128_GUID"
    ),
    UefiGuid(
        "3FA4F847-D8EB-4DF4-BD49103A0A847BBC", name="EFI_KMS_FORMAT_MDC4_128_GUID"
    ),
    UefiGuid(
        "56417BED-6BBE-4882-86A03AE8BB17F8F9", name="EFI_KMS_FORMAT_RSASHA1_1024_GUID"
    ),
    UefiGuid(
        "F66447D4-75A6-463E-A819077F2DDA05E9", name="EFI_KMS_FORMAT_RSASHA1_2048_GUID"
    ),
    UefiGuid(
        "A477AF13-877D-4060-BAA125D1BEA08AD3", name="EFI_KMS_FORMAT_RSASHA256_2048_GUID"
    ),
    UefiGuid(
        "453C5E5A-482D-43F0-87C95941F3A38AC2", name="EFI_KMS_FORMAT_SHA1_160_GUID"
    ),
    UefiGuid(
        "6BB4F5CD-8022-448D-BC6D771BAE935FC6", name="EFI_KMS_FORMAT_SHA256_256_GUID"
    ),
    UefiGuid(
        "2F240E12-E14D-475C-83B0EFFF22D77BE7", name="EFI_KMS_FORMAT_SHA512_512_GUID"
    ),
    UefiGuid("EC3A978D-7C4E-48FA-9ABE6AD91CC8F811", name="EFI_KMS_PROTOCOL_GUID"),
    UefiGuid(
        "A56074DB-65FE-45F7-BD212D2BDD8E9652", name="EFI_LEGACY_DEV_ORDER_VARIABLE_GUID"
    ),
    UefiGuid(
        "70101EAF-0085-440C-B3568EE36FEF24F0", name="EFI_LEGACY_REGION2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0FC9013A-0568-4BA9-9B7EC9C390A6609B", name="EFI_LEGACY_REGION_PROTOCOL_GUID"
    ),
    UefiGuid("56EC3091-954C-11D2-8E3F00A0C969723B", name="EFI_LOAD_FILE_PROTOCOL_GUID"),
    UefiGuid(
        "2CA88B53-D296-4080-A4A5CAD9BAE24B09",
        name="EFI_LOAD_FIXED_ADDRESS_CONFIGURATION_TABLE_GUID",
    ),
    UefiGuid("BD445D79-B7AD-4F04-9AD829BD2040EB3C", name="EFI_LOCK_BOX_PROTOCOL_GUID"),
    UefiGuid("B2360B42-7173-420A-869646CA6BAB1060", name="EFI_MEASURED_FV_HOB_GUID"),
    UefiGuid(
        "4C19049F-4137-4DD3-9C108B97A83FFDFA", name="EFI_MEMORY_TYPE_INFORMATION_GUID"
    ),
    UefiGuid("3E591C00-9E4A-11DF-92440002A5D5C51B", name="EFI_MMC_HOST_PROTOCOL_GUID"),
    UefiGuid(
        "1DA97072-BDDC-4B30-99F172A0B56FFF2A",
        name="EFI_MONOTONIC_COUNTER_ARCH_PROTOCOL_GUID",
    ),
    UefiGuid("BF0A78BA-EC29-49CF-A1C97AE54EAB6A51", name="EFI_MTFTP6_PROTOCOL_GUID"),
    UefiGuid(
        "D9760FF3-3CCA-4267-80F97527FAFA4223",
        name="EFI_MTFTP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid(
        "09D5B53F-F4B0-4F59-A0B17B57D35C0E05", name="EFI_NIC_IP4_CONFIG_NVDATA_GUID"
    ),
    UefiGuid("CCC53F6B-A03A-4ED8-839A03D99C02B4E3", name="EFI_NT_LOAD_AS_DLL_PPI_GUID"),
    UefiGuid("13A3F0F6-264A-3EF0-F2E0DEC512342F34", name="EFI_PCD_PROTOCOL_GUID"),
    UefiGuid(
        "30CFE3E7-3DE1-4586-BE20DEABA1B3B793", name="EFI_PCI_ENUMERATION_COMPLETE_GUID"
    ),
    UefiGuid(
        "CF8034BE-6768-4D8B-B7397CCE683A9FBE",
        name="EFI_PCI_HOST_BRIDGE_RESOURCE_ALLOCATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "19CB87AB-2CB9-4665-8360DDCF6054F79D",
        name="EFI_PCI_HOTPLUG_REQUEST_PROTOCOL_GUID",
    ),
    UefiGuid("B5B35764-460C-4A06-99FC77A17C1B5CEB", name="EFI_PCI_OVERRIDE_GUID"),
    UefiGuid(
        "ABD42895-78CF-4872-84441B5C180BFBFF",
        name="EFI_PEI_BOOT_SCRIPT_EXECUTER_PPI_GUID",
    ),
    UefiGuid("36164812-A023-44E5-BD8505BF3C7700AA", name="EFI_PEI_FIND_FV_PPI_GUID"),
    UefiGuid("01F34D25-4DE2-23AD-3FF336353FF323F1", name="EFI_PEI_PCD_PPI_GUID"),
    UefiGuid(
        "F894643D-C449-42D1-8EA885BDD8C65BDE",
        name="EFI_PEI_PERMANENT_MEMORY_INSTALLED_PPI_GUID",
    ),
    UefiGuid(
        "3CDC90C6-13FB-4A75-9E7959E9DD78B9FA",
        name="EFI_PEI_READ_ONLY_VARIABLE_ACCESS_PPI_GUID",
    ),
    UefiGuid(
        "695D8AA1-42EE-4C46-805C6EA6BCE799E3", name="EFI_PEI_RECOVERY_BLOCK_IO_PPI_GUID"
    ),
    UefiGuid("4426CCB2-E684-4A8A-AE4020D4B025B710", name="EFI_PEI_S3_RESUME_PPI_GUID"),
    UefiGuid("1388066E-3A57-4EFA-98F3C12F3A958A29", name="EFI_PEI_SECURITY_PPI_GUID"),
    UefiGuid("9CA93627-B65B-4324-A202C0B461764543", name="EFI_PEI_SMBUS2_PPI_GUID"),
    UefiGuid("ABD42895-78CF-4872-84441B5C180BFBDA", name="EFI_PEI_SMBUS_PPI_GUID"),
    UefiGuid(
        "AE933E1C-CC47-4E38-8F0EE2F61D2605DF", name="EFI_PEI_SMM_COMMUNICATION_PPI_GUID"
    ),
    UefiGuid(
        "CEAB683C-EC56-4A2D-A9064053FA4E9C16",
        name="EFI_PEI_TEMPORARY_RAM_DONE_PPI_GUID",
    ),
    UefiGuid(
        "3CD652B4-6D33-4DCE-89DB83DF9766FCCA",
        name="EFI_PEI_VECTOR_HANDOFF_INFO_PPI_GUID",
    ),
    UefiGuid(
        "0F6499B1-E9AD-493D-B9C22F90815C6CBC", name="EFI_PHYSICAL_PRESENCE_DATA_GUID"
    ),
    UefiGuid("F05976EF-83F1-4F3D-8619F7595D41E538", name="EFI_PRINT2_PROTOCOL_GUID"),
    UefiGuid(
        "86212936-0E76-41C8-A03A2AF2FC1C39E2", name="EFI_RSC_HANDLER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B7DFB4E1-052F-449F-87BE9818FC91B733", name="EFI_RUNTIME_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E1475E0C-1746-4802-862E011C2C2D9D86", name="EFI_RUNTIME_CRYPT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "B60DC6E8-3B6F-11D5-AF0900A0C944A05B", name="EFI_SAL_MCA_INIT_PMI_PROTOCOL_GUID"
    ),
    UefiGuid("D487DDB4-008B-11D9-AFDC001083FFCA4D", name="EFI_SAS_DEVICE_PATH_GUID"),
    UefiGuid("5261213D-3A3D-441E-B3AF21D3F7A4CA17", name="EFI_SCSI_BUS_PROTOCOL_GUID"),
    UefiGuid(
        "94AB2F58-1438-4EF1-915218941A3A0E68", name="EFI_SECURITY2_ARCH_PROTOCOL_GUID"
    ),
    UefiGuid("D2C18636-40E5-4EB5-A31B36695FD42C87", name="EFI_SE_EXT_SIGNATURE_GUID"),
    UefiGuid(
        "171E9188-31D3-40F5-B10C539B2DB940CD", name="EFI_SHELLPKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "752F3136-4E16-4FDC-A22AE5F46812F4CA", name="EFI_SHELL_PARAMETERS_PROTOCOL_GUID"
    ),
    UefiGuid("6302D008-7F9B-4F30-87AC60C9FEF5DA4E", name="EFI_SHELL_PROTOCOL_GUID"),
    UefiGuid(
        "31878C87-0B75-11D5-9A4F0090273FC14D", name="EFI_SIMPLE_POINTER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "387477C2-69C7-11D2-8E3900A0C969723B",
        name="EFI_SIMPLE_TEXT_OUTPUT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "C2702B74-800C-4131-87468FB5B89CE4AC", name="EFI_SMM_ACCESS2_PROTOCOL_GUID"
    ),
    UefiGuid("F4CCBFB7-F6E0-47FD-9DD410A8F150C191", name="EFI_SMM_BASE2_PROTOCOL_GUID"),
    UefiGuid(
        "910DCA07-1F94-4EE7-AF2FFF72F3154353",
        name="EFI_SMM_BASE_HELPER_READY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "26EEB3DE-B689-492E-80F0BE8BD7DA4BA7",
        name="EFI_SMM_CONFIGURATION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "843DC720-AB1E-42CB-93578A0078F3561B", name="EFI_SMM_CONTROL2_PROTOCOL_GUID"
    ),
    UefiGuid(
        "24E70042-D5C5-4260-8C390AD3AA32E93D", name="EFI_SMM_END_OF_DXE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3868FC3B-7E45-43A7-906C4BA47DE1754D",
        name="EFI_SMM_FAULT_TOLERANT_WRITE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "D326D041-BD31-4C01-B5A8628BE87F0653",
        name="EFI_SMM_FIRMWARE_VOLUME_BLOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "2A3CFEBD-27E8-4D0A-8B79D688C2A3E1C0",
        name="EFI_SMM_LOCK_BOX_COMMUNICATION_GUID",
    ),
    UefiGuid(
        "67C4F112-3385-4E55-9C5BC05B717C4228",
        name="EFI_SMM_SWAP_ADDRESS_RANGE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "14FC52BE-01DC-426C-91AEA23C3E220AE8", name="EFI_SMM_SX_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid("93BA1826-DFFB-45DD-82A7E7DCAA3BBDF3", name="EFI_SMM_VARIABLE_WRITE_GUID"),
    UefiGuid(
        "D2B2B828-0826-48A7-B3DF983C006024F0",
        name="EFI_STATUS_CODE_RUNTIME_PROTOCOL_GUID",
    ),
    UefiGuid("2E3044AC-879F-490F-9760BBDFAF695F50", name="EFI_TCG_EVENT_HOB_GUID"),
    UefiGuid("46E44855-BD60-4AB7-AB0DA679B9447D77", name="EFI_TCP6_PROTOCOL_GUID"),
    UefiGuid(
        "EC20EB79-6C1A-4664-9A0DD2E4CC16D664",
        name="EFI_TCP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("DBA6A7E3-BB57-4BE7-8AF8D578DB7E5687", name="EFI_TSC_FREQUENCY_GUID"),
    UefiGuid("37499A9D-542F-4C89-A02635DA142094E4", name="EFI_UART_DEVICE_PATH_GUID"),
    UefiGuid("4F948815-B4B9-43CB-8A3390E060B34955", name="EFI_UDP6_PROTOCOL_GUID"),
    UefiGuid(
        "66ED4721-3C98-4D3E-81E3D03DD39A7254",
        name="EFI_UDP6_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("F2CC5D06-8985-11DB-BB190040D02B1835", name="EFI_UNIX_CONSOLE_GUID"),
    UefiGuid("F2D3B330-8985-11DB-8AA30040D02B1835", name="EFI_UNIX_CPU_MODEL_GUID"),
    UefiGuid("F2D74E5A-8985-11DB-97050040D02B1835", name="EFI_UNIX_CPU_SPEED_GUID"),
    UefiGuid("F2C16B9E-8985-11DB-92C80040D02B1835", name="EFI_UNIX_FILE_SYSTEM_GUID"),
    UefiGuid("BACE07C2-8987-11DB-A59A0040D02B1835", name="EFI_UNIX_GOP_GUID"),
    UefiGuid("F2E23F54-8985-11DB-AC790040D02B1835", name="EFI_UNIX_IO_PROTOCOL_GUID"),
    UefiGuid("F2D006CC-8985-11DB-A4720040D02B1835", name="EFI_UNIX_MEMORY_GUID"),
    UefiGuid("081603B4-0F1D-4022-B6FD4CE35E09A1A6", name="EFI_UNIX_NETWORK_GUID"),
    UefiGuid(
        "F2BDCC96-8985-11DB-87190040D02B1835", name="EFI_UNIX_PHYSICAL_DISKS_GUID"
    ),
    UefiGuid("6D3A727D-66C8-4D19-87E60215861490F3", name="EFI_UNIX_SERIAL_PORT_GUID"),
    UefiGuid(
        "F2E98868-8985-11DB-9A590040D02B1835", name="EFI_UNIX_THUNK_PROTOCOL_GUID"
    ),
    UefiGuid("F2C8B80E-8985-11DB-93F10040D02B1835", name="EFI_UNIX_UGA_GUID"),
    UefiGuid(
        "F2E5E2C6-8985-11DB-A1910040D02B1835", name="EFI_UNIX_UGA_IO_PROTOCOL_GUID"
    ),
    UefiGuid("F2BA331A-8985-11DB-A4060040D02B1835", name="EFI_UNIX_VIRTUAL_DISKS_GUID"),
    UefiGuid("283FA2EE-532C-484D-93839F93B36F0B7E", name="EFI_UPDATE_DATA_FILE_GUID"),
    UefiGuid("2B2F68CC-0CD2-44CF-8E8BBBA20B1B5B75", name="EFI_USB_BUS_PROTOCOL_GUID"),
    UefiGuid(
        "E98ADB03-B8B9-4AF8-BA2026E9114CBCE5", name="EFI_USER_CREDENTIAL2_PROTOCOL_GUID"
    ),
    UefiGuid("375EA976-3CCD-4E74-A84526B9B324B13C", name="EFI_UXIX_SYSTEM_CONFIG_GUID"),
    UefiGuid("DDCF3616-3275-4164-98B6FE85707FFE7D", name="EFI_VARIABLE_GUID"),
    UefiGuid(
        "8CFDB8C8-D6B2-40F3-8E9702307CC98B7C", name="EFI_VARIABLE_INDEX_TABLE_GUID"
    ),
    UefiGuid(
        "9E23D768-D2F3-4366-9FC33A7ABA864374", name="EFI_VLAN_CONFIG_PROTOCOL_GUID"
    ),
    UefiGuid("BA73672C-A5D3-11D4-BD000080C73C8881", name="EFI_WIN_NT_CONSOLE_GUID"),
    UefiGuid("BEE9B6CE-2F8A-11D4-BD0D0080C73C8881", name="EFI_WIN_NT_CPU_MODEL_GUID"),
    UefiGuid("D4F29055-E1FB-11D4-BD0D0080C73C8881", name="EFI_WIN_NT_CPU_SPEED_GUID"),
    UefiGuid("0C95A935-A006-11D4-BCFA0080C73C8881", name="EFI_WIN_NT_FILE_SYSTEM_GUID"),
    UefiGuid("4E11E955-CCCA-11D4-BD0D0080C73C8881", name="EFI_WIN_NT_GOP_GUID"),
    UefiGuid("96EB4AD6-A32A-11D4-BCFD0080C73C8881", name="EFI_WIN_NT_IO_PROTOCOL_GUID"),
    UefiGuid("99042912-122A-11D4-BD0D0080C73C8881", name="EFI_WIN_NT_MEMORY_GUID"),
    UefiGuid(
        "CC664EB8-3C24-4086-B6F634E856BCE36E", name="EFI_WIN_NT_PASS_THROUGH_GUID"
    ),
    UefiGuid(
        "0C95A92F-A006-11D4-BCFA0080C73C8881", name="EFI_WIN_NT_PHYSICAL_DISKS_GUID"
    ),
    UefiGuid(
        "B347F047-AF8C-490E-AC070AA9B7E53858", name="EFI_WIN_NT_SYSTEM_CONFIG_GUID"
    ),
    UefiGuid("AB248E99-ABE1-11D4-BD0D0080C73C8881", name="EFI_WIN_NT_UGA_GUID"),
    UefiGuid(
        "0C95A928-A006-11D4-BCFA0080C73C8881", name="EFI_WIN_NT_VIRTUAL_DISKS_GUID"
    ),
    UefiGuid("D3B46F3B-D441-1244-9A120012273FC14D", name="EFI_XEN_INFO_GUID"),
    UefiGuid(
        "BF4B9D10-13EC-43DD-8880E90B718F27DE", name="EMBEDDED_DEVICE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "735F8C64-D696-44D0-BDF2447FD05A5406",
        name="EMBEDDED_EXTERNAL_DEVICE_PROTOCOL_GUID",
    ),
    UefiGuid("6888A4AE-AFCE-E84B-9102F7B9DAE6A030", name="EMU_BLOCK_IO_PROTOCOL_GUID"),
    UefiGuid(
        "30FD316A-6728-2E41-A6900D1333D8CAC1", name="EMU_GRAPHICS_WINDOW_PROTOCOL_GUID"
    ),
    UefiGuid("453368F6-7C85-434A-A98A72D1B7FFA926", name="EMU_IO_THUNK_PROTOCO_GUID"),
    UefiGuid("FD5FBE54-8C35-B345-8A0F7AC8A5FD0521", name="EMU_SNP_PROTOCOL_GUID"),
    UefiGuid("B958B78C-1D3E-EE40-8BF4F0632D063916", name="EMU_THUNK_PPI_GUID"),
    UefiGuid("5CF32E0B-8EDF-2E44-9CDA93205E99EC1C", name="EMU_THUNK_PROTOCOL_GUID"),
    UefiGuid(
        "DE0EE9A4-3C7A-44F2-B78BE3CCD69C3AF7",
        name="EXTENDED_SAL_BOOT_SERVICE_PROTOCOL_GUID",
    ),
    UefiGuid("AC05BF33-995A-4ED4-AAB8EF7AE80F5CB0", name="FFS_GUID"),
    UefiGuid("1F2D63E1-FEBD-4DC7-9CC5BA2B1CEF9C5B", name="FILE_EXPLORE_FORMSET_GUID"),
    UefiGuid("CBD2E4D5-7068-4FF5-B4629822B4AD8D60", name="FILE_GUID"),
    UefiGuid(
        "1F73B18D-4630-43C1-A1DE6F80855D7DA4",
        name="FORM_BROWSER_EXTENSION_PROTOCOL_GUID",
    ),
    UefiGuid(
        "9E0C30BC-3F06-4BA6-828809179B855DBE",
        name="FRAMEWORK_BDS_FRONTPAGE_FORMSET_GUID",
    ),
    UefiGuid(
        "F33261E7-23CB-11D5-BD5C0080C73C8881",
        name="FRAMEWORK_EFI_MP_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid("9E0C30BC-3F06-4BA6-828809179B855DBE", name="FRONT_PAGE_FORMSET_GUID"),
    UefiGuid("B8969637-81DE-43AF-BC9A24D98913F2F6", name="HANDLE_PARSING_HII_GUID"),
    UefiGuid(
        "FAB7E9E1-39DD-4F2B-8408E20E906CB6DE", name="HD_BOOT_DEVICE_PATH_VARIABLE_GUID"
    ),
    UefiGuid(
        "4F4EF7F0-AA29-4CE9-BA41643E0123A99F", name="HII_RESOURCE_SAMPLE_FORM_SET_GUID"
    ),
    UefiGuid("7739F24C-93D7-11D4-9A3A0090273FC14D", name="HOB_LIST_GUID"),
    UefiGuid("3C8D294C-5FC3-4451-BB31C4C032295E6C", name="IDLE_LOOP_EVENT_GUID"),
    UefiGuid(
        "D3705011-BC19-4AF7-BE16F68030378C15",
        name="INTEL_FRAMEWORK_MODULEPKG_TOKEN_SPACE_GUID",
    ),
    UefiGuid("6456ED61-3579-41C9-8A260A0BD62B78FC", name="IP4_ISCSI_CONFIG_GUID"),
    UefiGuid("02EEA107-98DB-400E-9830460A1542D799", name="IP6_CONFIG_NVDATA_GUID"),
    UefiGuid("786EC0AC-65AE-4D1B-B1370D110A483797", name="ISCSI_CHAP_AUTH_INFO_GUID"),
    UefiGuid("4B47D616-A8D6-4552-9D44CCAD2E0F4CF9", name="ISCSI_CONFIG_GUID"),
    UefiGuid("FA3CDE4C-87C2-427D-AEDE7DD096C88C58", name="ISCSI_V4_PRIVATE_GUID"),
    UefiGuid("28BE27E5-66CC-4A31-A315DB14C3744D85", name="ISCSI_V6_PRIVATE_GUID"),
    UefiGuid("0E8C545B-A2EE-470D-8E26BDA1A13C0AA3", name="LAST_ENUM_LANGUAGE_GUID"),
    UefiGuid("7701D7E5-7D1D-4432-A468673DAB8ADE60", name="LDR_MEMORY_DESCRIPTOR_GUID"),
    UefiGuid(
        "96EB4AD6-A32A-11D4-BCFD0080C73C8881",
        name="LOCAL_EFI_WIN_NT_BUS_DRIVER_IO_PROTOCOL_GUID",
    ),
    UefiGuid(
        "0C95A93D-A006-11D4-BCFA0080C73C8881", name="LOCAL_EFI_WIN_NT_SERIAL_PORT_GUID"
    ),
    UefiGuid(
        "58C518B1-76F3-11D4-BCEA0080C73C8881",
        name="LOCAL_EFI_WIN_NT_THUNK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "D42AE6BD-1352-4BFB-909ACA72A6EAE889", name="LZMAF86_CUSTOM_DECOMPRESS_GUID"
    ),
    UefiGuid("EE4E5898-3914-4259-9D6EDC7BD79403CF", name="LZMA_CUSTOM_DECOMPRESS_GUID"),
    UefiGuid(
        "A1AFF049-FDEB-442A-B32013AB4CB72BBC", name="MDEMODULEPKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("914AEBE7-4635-459B-AA1C11E219B03A10", name="MDEPKG_TOKEN_SPACE_GUID"),
    UefiGuid(
        "060CC026-4C0D-4DDA-8F41595FEF00A502", name="MEMORY_STATUS_CODE_RECORD_GUID"
    ),
    UefiGuid("EB704011-1402-11D3-8E7700A0C969723B", name="MTC_VENDOR_GUID"),
    UefiGuid("12345678-AABB-CCDD-EEFF112233445566", name="MY_GUID"),
    UefiGuid("4E76928F-50AD-4334-B06BA84213108A57", name="NT_FWH_PPI_GUID"),
    UefiGuid(
        "326AE723-AE32-4589-98B8CAC23CDCC1B1", name="PCATCHIPSET_TOKEN_SPACE_GUID"
    ),
    UefiGuid("EA296D92-0B69-423C-8C2833B4E0A91268", name="PCD_DATABASE_HOB_GUID"),
    UefiGuid("06E81C58-4AD7-44BC-8390F10265F72480", name="PCD_PPI_GUID"),
    UefiGuid("11B34006-D85B-4D0A-A290D5A571310EF7", name="PCD_PROTOCOL_GUID"),
    UefiGuid("5CB5C776-60D5-45EE-883C452708CD743F", name="PE32_IMAGE_PROTOCOL_GUID"),
    UefiGuid(
        "EF9AEFE5-2BD3-4031-AF7D5EFE5ABB9A0D",
        name="PEI_LOCK_PHYSICAL_PRESENCE_PPI_GUID",
    ),
    UefiGuid("98C281E5-F906-43DD-A92BB003BF2765DA", name="PEI_NT_THUNK_GUID"),
    UefiGuid("98C281E5-F906-43DD-A92BB003BF2765DA", name="PEI_NT_THUNK_PPI_GUID"),
    UefiGuid(
        "20A7378C-AA83-4CE1-821F4740EE1B3F9F", name="PEI_OPERATOR_PRESENCE_PPI_GUID"
    ),
    UefiGuid(
        "0ECC666B-4662-47F9-9DD5D096FF7DA49E", name="PEI_SEC_PERFORMANCE_PPI_GUID"
    ),
    UefiGuid("F2ED3D14-8985-11DB-B0570040D02B1835", name="PEI_UNIX_AUTOSCAN_PPI_GUID"),
    UefiGuid("F2F830F2-8985-11DB-806B0040D02B1835", name="PEI_UNIX_THUNK_PPI_GUID"),
    UefiGuid(
        "A7D09FE1-74D4-4BA5-847C12ED5B19ADE4", name="PEI_USB2_HOST_CONTROLLER_PPI_GUID"
    ),
    UefiGuid(
        "652B38A9-77F4-453F-89D5E7BDC352FC53", name="PEI_USB_HOST_CONTROLLER_PPI_GUID"
    ),
    UefiGuid("7C29785C-66B9-49FC-B7971CA5550EF283", name="PEI_USB_IO_PPI_GUID"),
    UefiGuid(
        "669346EF-FDAD-4AEB-08A621462D3FEF7D", name="PERFORMANCEPKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "1EA81BEC-F01A-4D98-A2014A61CE2FC022", name="PERFORMANCE_EX_PROTOCOL_GUID"
    ),
    UefiGuid("76B6BDFA-2ACD-4462-9E3FCB58C969D937", name="PERFORMANCE_PROTOCOL_GUID"),
    UefiGuid(
        "B323179B-97FB-477E-B0FED88591FA11AB", name="PE_COFF_LOADER_PROTOCOL_GUID"
    ),
    UefiGuid("8614567D-35BE-4415-8D88BD7D0C9C70C0", name="PLAT_OVER_MNGR_GUID"),
    UefiGuid(
        "385A982C-2F49-4043-A51E4901025C8B6B",
        name="PRE_PI_EXTRACT_GUIDED_SECTION_DATA_GUID",
    ),
    UefiGuid(
        "78B9EC8B-C000-46C5-AC9324A0C1BB00CE", name="PWD_CREDENTIAL_PROVIDER_GUID"
    ),
    UefiGuid("5CAC0099-0DC9-48E5-8068BB95F5400A9F", name="RECOVERY_ON_DATA_CD_GUID"),
    UefiGuid(
        "2E3D2E75-9B2E-412D-B4B170416B8700FF", name="RECOVERY_ON_FAT_FLOPPY_DISK_GUID"
    ),
    UefiGuid(
        "B38573B6-6200-4AC5-B51D82E65938D783", name="RECOVERY_ON_FAT_IDE_DISK_GUID"
    ),
    UefiGuid(
        "0FFBCE19-324C-4690-A00998C6AE2EB186", name="RECOVERY_ON_FAT_USB_DISK_GUID"
    ),
    UefiGuid("EB9D2D32-2D88-11D3-9A160090273FC14D", name="SAL_SYSTEM_TABLE_GUID"),
    UefiGuid(
        "5DAF50A5-EA81-4DE2-8F9BCABDA9CF5C14", name="SECUREBOOT_CONFIG_FORM_SET_GUID"
    ),
    UefiGuid(
        "0D3FB176-9569-4D51-A3EF7D61C64FEABA", name="SECURITYPKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "D2C18636-40E5-4EB5-A31B36695FD42C87", name="SHELLPKG_SHELL_ENV2_EXT_GUID"
    ),
    UefiGuid("0053D9D6-2659-4599-A26BEF4536E631A9", name="SHELL_ALIAS_VARIABLE_GUID"),
    UefiGuid("25F200AA-D3CB-470A-BF51E7D162D22E6F", name="SHELL_DEBUG1_HII_GUID"),
    UefiGuid("0AF0B742-63EC-45BD-8DB671AD7F2FE8E8", name="SHELL_DRIVER1_HII_GUID"),
    UefiGuid(
        "47C7B221-C42A-11D2-8E5700A0C969723B", name="SHELL_ENVIRONMENT_PROTOCOL_GUID"
    ),
    UefiGuid("7D574D54-D364-4D4A-95E34945DB7AD3EE", name="SHELL_INSTALL1_HII_GUID"),
    UefiGuid(
        "47C7B223-C42A-11D2-8E5700A0C969723B", name="SHELL_INTERFACE_PROTOCOL_GUID"
    ),
    UefiGuid("DEC5DAA4-6781-4820-9C63A7B0E4F1DB31", name="SHELL_LEVEL1_HII_GUID"),
    UefiGuid("F95A7CCC-4C55-4426-A7B4DC8961950BAE", name="SHELL_LEVEL2_HII_GUID"),
    UefiGuid("4344558D-4EF9-4725-B1E43376E8D6974F", name="SHELL_LEVEL3_HII_GUID"),
    UefiGuid("51271E13-7DE3-43AF-8BC271AD3B824325", name="SHELL_MAP_GUID"),
    UefiGuid("F3D301BB-F4A5-45A8-B0B7FA999C6237AE", name="SHELL_NETWORK1_HII_GUID"),
    UefiGuid("158DEF5A-F656-419C-B0277A3192C079D2", name="SHELL_VARIABLE_GUID"),
    UefiGuid("EB9D2D31-2D88-11D3-9A160090273FC14D", name="SMBIOS_TABLE_GUID"),
    UefiGuid(
        "931FC048-C71D-4455-8930470630E30EE5", name="SMM_PERFORMANCE_EX_PROTOCOL_GUID"
    ),
    UefiGuid(
        "F866226A-EAA5-4F5A-A90A6CFBA57C588E", name="SMM_PERFORMANCE_PROTOCOL_GUID"
    ),
    UefiGuid("E701458C-4900-4CA5-B7723D37949F7927", name="STATUS_CODE_CALLBACK_GUID"),
    UefiGuid("1547B4F3-3E8A-4FEF-81C8328ED647AB1A", name="SYSTEM_ROM_FILE_GUID"),
    UefiGuid("B0F901E4-C424-45DE-908195E20BDE6FB5", name="TCG_CONFIG_FORM_SET_GUID"),
    UefiGuid(
        "A31280AD-481E-41B6-95E8127F4C984779", name="TIANO_CUSTOM_DECOMPRESS_GUID"
    ),
    UefiGuid("F2F0DC30-8985-11DB-A15B0040D02B1835", name="UNIX_FWH_PPI_GUID"),
    UefiGuid("F2F48768-8985-11DB-B8DA0040D02B1835", name="UNIX_PEI_LOAD_FILE_GUID"),
    UefiGuid(
        "D0849ED1-A88C-4BA6-B1D6AB50E280B7A9", name="USB_CREDENTIAL_PROVIDER_GUID"
    ),
    UefiGuid(
        "0C0F3B43-44DE-4907-B478225F6F6289DC", name="USB_KEYBOARD_LAYOUT_PACKAGE_GUID"
    ),
    UefiGuid("3CCD3DD8-8D45-4FED-962D2B38CD82B3C4", name="USER_IDENTIFY_MANAGER_GUID"),
    UefiGuid("C35F272C-97C2-465A-A216696B668A8CFE", name="USER_PROFILE_MANAGER_GUID"),
    UefiGuid(
        "AD651C7D-3C22-4DBF-92E838A7CDAE87B2",
        name="VIRTUAL_UNCACHED_PAGES_PROTOCOL_GUID",
    ),
    UefiGuid("D79DF6B0-EF44-43BD-979743E93BCF5FA8", name="VLAN_CONFIG_FORM_SET_GUID"),
    UefiGuid(
        "0013BE6B-2198-43F1-93BA2A7ED7B1E1CC", name="LENOVO_SYSTEM_USB_SWITCH_DXE_GUID"
    ),
    UefiGuid(
        "0167CCC4-D0F7-4F21-A3EF9E64B7CDCE8B", name="LENOVO_SYSTEM_SCSI_BUS_DXE_GUID"
    ),
    UefiGuid(
        "0167CCC4-D0F7-4F21-A3EF9E64B7CDCE8C", name="LENOVO_SYSTEM_AHCI_BUS_DXE_GUID"
    ),
    UefiGuid("025BBFC7-E6A9-4B8B-82AD6815A1AEAF4A", name="LENOVO_MNP_DXE_BIN_GUID"),
    UefiGuid("03BF5679-5749-4BC5-A53F47471DA767C8", name="LENOVO_ISSC_SMM_GUID"),
    UefiGuid("03EBDB4B-96BA-4F40-83297F3AA8865707", name="LENOVO_EC_IO_SMM_GUID"),
    UefiGuid(
        "051A0B77-C066-4610-9904BF25C296D6E6",
        name="LENOVO_EN25_QH32_FLASH_PART_DXE_GUID",
    ),
    UefiGuid("06384570-1680-4620-9D006AD9E3CCB19F", name="LENOVO_MTM_FORMAT_SMM_GUID"),
    UefiGuid("06968B6E-F620-4E8D-A6A652552354DD75", name="LENOVO_DXE_RICOH_INIT_GUID"),
    UefiGuid(
        "06DE824A-A4E2-4295-A3F603B3FEF5B109", name="LENOVO_SECURITY_VARIABLE_DXE_GUID"
    ),
    UefiGuid(
        "0723F88B-810F-4E44-92A0A5C83B433698",
        name="LENOVO_IBEX_PEAK_FLASH_CONTROLLER_DXE_GUID",
    ),
    UefiGuid("0770E7A0-9C3A-499C-A3112F8385080556", name="LENOVO_BOARD_INFO_SMM_GUID"),
    UefiGuid(
        "07A9330A-F347-11D4-9A490090273FC14D", name="LENOVO_SYSTEM_METRONOME_DXE_GUID"
    ),
    UefiGuid("087E9760-1FB5-49F7-879A853D2B214CC7", name="LENOVO_SLP2_DXE_GUID"),
    UefiGuid(
        "08FBD32F-867F-452B-B134705161F73333", name="LENOVO_CRYPT_SERVICE_SMM_GUID"
    ),
    UefiGuid(
        "0A66E322-3740-4CCE-AD62BD172CECCA35", name="LENOVO_SYSTEM_SCSI_DISK_DXE_GUID"
    ),
    UefiGuid(
        "0B6A8DE0-6281-40A2-8E9A09F1A18A6D40", name="LENOVO_SYSTEM_USB_SWITCH_SMM_GUID"
    ),
    UefiGuid(
        "0C396FCA-6BDA-4A15-B6A3A6FA4544BDB7", name="LENOVO_EC_FW_UPDATE_DXE_GUID"
    ),
    UefiGuid(
        "0D648466-36BD-42C6-B2877C3BAA2575C0", name="LENOVO_SETUP_UNDER_OS_DXE_GUID"
    ),
    UefiGuid("0FE159B5-076F-4C36-BF26D724F2831252", name="LENOVO_BDS_CTRL_GUID"),
    UefiGuid(
        "10149626-5583-4364-822FA459B8121302",
        name="LENOVO_SYSTEM_FIRMWARE_DEVICE_DXE_GUID",
    ),
    UefiGuid("114CA60C-D965-4C13-BEF7C4062248E1FA", name="LENOVO_EC_IO_DXE_GUID"),
    UefiGuid(
        "124A2E7A-1949-483E-899F6032904CA0A7",
        name="LENOVO_SYSTEM_SMM_AHCI_ASPI_LEGACY_RT_GUID",
    ),
    UefiGuid("126A762D-5758-4FCA-8531201A7F57F850", name="LENOVO_BOOT_MENU_DXE_GUID"),
    UefiGuid("12E7691F-1EEE-4ED4-879369E2801D602A", name="LENOVO_FP_PBA_GUID"),
    UefiGuid("13AC6DD0-73D0-11D4-B06B00AA00BD6DE7", name="LENOVO_SYSTEM_EBC_DXE_GUID"),
    UefiGuid(
        "1504BA7B-58F8-4D12-9638B494A7044376",
        name="LENOVO_SYSTEM_SMM_SERVICES_SMM_GUID",
    ),
    UefiGuid("15C80344-F980-4BF5-AAA0BFBE027AEF16", name="LENOVO_EC_SERVICE_GUID"),
    UefiGuid(
        "15DD5676-2679-4E24-9CAA85B22DD893EB",
        name="LENOVO_SECURE_FLASH_VERIFY_SMM_GUID",
    ),
    UefiGuid(
        "170D9568-C969-4FA1-A533DB6779BFBE62",
        name="LENOVO_EN25_QH128_FLASH_PART_SMM_GUID",
    ),
    UefiGuid(
        "18578E75-D073-4203-90D28788A871E44D",
        name="LENOVO_COMPUTRACE_SMI_SERVICES_GUID",
    ),
    UefiGuid("19B6F165-7855-42E8-AA6F2361CA87E024", name="LENOVO_SMM_VIDEO_INT10_GUID"),
    UefiGuid("1A7E4468-2F55-4A56-903C01265EB7622B", name="LENOVO_TCP_DXE_BIN_GUID"),
    UefiGuid(
        "1C6B2FAF-D8BD-44D1-A91E7321B4C2F3D1",
        name="LENOVO_SYSTEM_BOOT_SCRIPT_SAVE_DXE_GUID",
    ),
    UefiGuid("1D201235-2F40-4FBC-86508502092D62AB", name="LENOVO_EAIA_DXE_GUID"),
    UefiGuid(
        "202A2B0E-9A31-4812-B2918747DF152439", name="LENOVO_SYSTEM_PS2_MOUSE_DXE_GUID"
    ),
    UefiGuid(
        "22BF0B85-4EB3-479B-A9E9A5F7CFCCA787", name="LENOVO_SMART_CARD_DETECT_DXE_GUID"
    ),
    UefiGuid(
        "2338337A-47B0-4C41-9CA70160FB94DFF4", name="LENOVO_PLATFORM_FLASH_DXE_GUID"
    ),
    UefiGuid(
        "238D654D-A6AD-41C8-A31B67A3208625C3",
        name="LENOVO_MX25_L3206_EFLASH_PART_SMM_GUID",
    ),
    UefiGuid("23D93EAA-0723-4559-B7685E22DF762DE4", name="LENOVO_TPM_ENABLER_DXE_GUID"),
    UefiGuid(
        "240612B7-A063-11D4-9A3A0090273FC14C",
        name="LENOVO_SYSTEM_USB_SUPPORT_POLICY_SMM_GUID",
    ),
    UefiGuid(
        "240612B7-A063-11D4-9A3A0090273FC14D", name="LENOVO_SYSTEM_USB_BUS_DXE_GUID"
    ),
    UefiGuid(
        "240612B7-A063-11D4-9A3A0090273FC14E",
        name="LENOVO_SYSTEM_USB_SUPPORT_POLICY_DXE_GUID",
    ),
    UefiGuid("243C8C8A-BBD0-4AA9-BE17CF9B583130EC", name="LENOVO_SMM_OEMINT15_GUID"),
    UefiGuid(
        "246F9F0A-11E3-459A-AE06372BDC4DE806",
        name="LENOVO_SYSTEM_STATUS_CODE_GENERIC_RT_GUID",
    ),
    UefiGuid("2554EF5E-C9CA-4A48-9D94249EB1E87C2D", name="LENOVO_CRYPT_SERVICE_GUID"),
    UefiGuid(
        "25F49067-A65B-48F5-BBBE35418C488836", name="LENOVO_TCG_SERVICES_DXE_GUID"
    ),
    UefiGuid("26DDBA9F-5B0D-4E80-86B280DAE4D01B0E", name="LENOVO_FDISK_OEM_GUID"),
    UefiGuid("27F4917B-A707-4AAD-967626DF168CBF0D", name="LENOVO_PCH_SPI_SMM_GUID"),
    UefiGuid("28B5A45C-CC5D-4221-9DFF86B0F42DDC01", name="LENOVO_IVB_GOP_DRIVER_GUID"),
    UefiGuid(
        "29A1A717-36E9-49E0-B381EA3B5FF70942",
        name="LENOVO_SYSTEM_STATUS_CODE_PORT80_RT_GUID",
    ),
    UefiGuid(
        "2A4224A1-2609-40A7-88E2A68DC0F1D180", name="LENOVO_W25_Q64_FLASH_PART_DXE_GUID"
    ),
    UefiGuid(
        "2ACB1E45-DFA4-4FF3-8F8191F7D787D22D",
        name="LENOVO_SYSTEM_USB_MEMORY_MANAGER_SMM_GUID",
    ),
    UefiGuid(
        "2BDED685-F733-455F-A84043A22B791FB3",
        name="LENOVO_SYSTEM_ACPI_S3_SAVE_DXE_GUID",
    ),
    UefiGuid(
        "2D1E361C-7B3F-4D15-8B1F66E551FABDC7", name="LENOVO_SA_LATE_INIT_SMM_GUID"
    ),
    UefiGuid(
        "2D2E62AA-9ECF-43B7-821994E7FC713DFE", name="LENOVO_SYSTEM_USB_MOUSE_DXE_GUID"
    ),
    UefiGuid(
        "2D2E62CF-9ECF-43B7-821994E7FC713DFE", name="LENOVO_SYSTEM_USB_KB_DXE_GUID"
    ),
    UefiGuid("2D6F37BF-9AFC-4AA5-A02632B2FCF30FB9", name="LENOVO_PLATFORM_DXE_GUID"),
    UefiGuid(
        "2EE81ACB-64B2-41AE-86357030D16C4AA8", name="LENOVO_PCH_BIOS_WRITE_PROTECT_GUID"
    ),
    UefiGuid(
        "301A0BC3-BA16-49F9-858BDEE05F91E7B8", name="LENOVO_TP_ACPI_NVS_INIT_DXE_GUID"
    ),
    UefiGuid("311032C4-98A6-4E06-8C03D815FD1FE709", name="LENOVO_SNB_GOP_DRIVER_GUID"),
    UefiGuid(
        "316190D1-5E48-4C15-AC7237532F3A1EC1",
        name="LENOVO_SYSTEM_USB_DATABASE_SMM_GUID",
    ),
    UefiGuid(
        "32442D09-1D11-4E27-8AAB90FE6ACB0489",
        name="LENOVO_SYSTEM_FORM_BROWSER_CORE_DXE_GUID",
    ),
    UefiGuid(
        "32C1C9F8-D53F-41C8-94D0F6739F231011", name="LENOVO_BIOS_EXTENSION_LOADER_GUID"
    ),
    UefiGuid(
        "348C4D62-BFBD-4882-9ECEC80BB1C4783B",
        name="LENOVO_SYSTEM_HII_DATABASE_DXE_GUID",
    ),
    UefiGuid("34D51EB9-76DB-4CAF-94AE934856E895F6", name="LENOVO_ATP_DXE_GUID"),
    UefiGuid("35B898CA-B6A9-49CE-8C72904735CC49B7", name="LENOVO_DXE_MAIN_GUID"),
    UefiGuid(
        "37087B94-EF41-4977-93E23F6ADFDD06E1", name="LENOVO_SYSTEM_SPLASH_DXE_GUID"
    ),
    UefiGuid(
        "371FA4B8-8042-4ED6-ADA7A987BD2CDF0A",
        name="LENOVO_SYSTEM_IMAGE_DISPLAY_DXE_GUID",
    ),
    UefiGuid("378D7B65-8DA9-4773-B6E4A47826A833E1", name="LENOVO_SYSTEM_RTC_RT_GUID"),
    UefiGuid(
        "37A0D020-CCD1-11DD-AD8B0800200C9A66",
        name="LENOVO_SYSTEM_IDE_ATA_ATAPI_PASS_THRU_DXE_GUID",
    ),
    UefiGuid("37AFCF55-2E8C-4722-B950B48B9165C56B", name="LENOVO_SETUP_MAIN_DXE_GUID"),
    UefiGuid("37DDC7E9-7C1D-4E1B-B2CB4ED5D12527B3", name="LENOVO_ME_PCI_UPDATE_GUID"),
    UefiGuid("37ECDF24-8453-476A-AB58F4CF8BFE6742", name="LENOVO_PHX_GOP_DXE_GUID"),
    UefiGuid(
        "389244DB-3D6E-4E50-835D5A775A325D89",
        name="LENOVO_MX25_L64_XFLASH_PART_DXE_GUID",
    ),
    UefiGuid("3A2AEC67-77B8-40AE-928103519C9E7F08", name="LENOVO_AOAC_SMM_GUID"),
    UefiGuid(
        "3C14C099-3143-46EF-AF18DE4AD2C3A799",
        name="LENOVO_SYSTEM_HII_IMAGE_DISPLAY_DXE_GUID",
    ),
    UefiGuid(
        "3DC82376-637B-40A6-A8FCA565417F2C38",
        name="LENOVO_SYSTEM_PS2_KEYBOARD_DXE_GUID",
    ),
    UefiGuid("3DFC255F-9C86-48BC-972DE522533768DB", name="LENOVO_REBOOT_CP_GUID"),
    UefiGuid(
        "3E44AD09-D88B-4C52-B41BB8C00536474A", name="LENOVO_SYSTEM_CRYPT_SVC_RT_GUID"
    ),
    UefiGuid(
        "3F7753EF-D36F-45CC-8A1A2EAAFF1038B6", name="LENOVO_PLATFORM_PCI_DXE_GUID"
    ),
    UefiGuid(
        "3F7E615B-0D45-4F80-88DC26B234958560", name="LENOVO_FIT_DIAGNOSTICS_LOADER_GUID"
    ),
    UefiGuid(
        "3FB856A5-C8B0-4760-B49A2EDD4FA35BFA",
        name="LENOVO_IBEX_PEAK_GLOBAL_SMI_CONTROL_GUID",
    ),
    UefiGuid(
        "401471E4-9E79-4664-BFB0179524230567", name="LENOVO_VPRO_IDER_ASF_BOOT_GUID"
    ),
    UefiGuid(
        "408EDCEC-CF6D-477C-A5A8B4844E3DE281",
        name="LENOVO_SYSTEM_CON_SPLITTER_DXE_GUID",
    ),
    UefiGuid("43172851-CF7E-4345-9FE0D7012BB17B88", name="LENOVO_I_FFS_SMM_GUID"),
    UefiGuid(
        "43B93232-AFBE-11D4-BD0F0080C73C8881", name="LENOVO_SYSTEM_PARTITION_DXE_GUID"
    ),
    UefiGuid("43BB9EAB-7D57-4DEF-B0A2A3A9F9C6EAE3", name="LENOVO_MAIL_BOX_DXE_GUID"),
    UefiGuid(
        "44CF7D20-DFF6-4209-9A1FF6CD5F5CE88B", name="LENOVO_VARIABLE_INIT_SMM_GUID"
    ),
    UefiGuid(
        "44F23590-DC88-11DD-AD8B0800200C9A66",
        name="LENOVO_SYSTEM_AHCI_ATA_ATAPI_PASS_THRU_DXE_GUID",
    ),
    UefiGuid(
        "4589CBF3-03F9-4998-9D6F26343C69562A", name="LENOVO_COMPUTRACE_LOADER_DXE_GUID"
    ),
    UefiGuid("46D8ACDD-29C3-4013-81E837A425AB4F99", name="LENOVO_SYSTEM_CMOS_SMM_GUID"),
    UefiGuid("4A5227D3-0BEF-4CAA-ACBDEC84446C5C6C", name="LENOVO_MISC_GA_IO_DXE_GUID"),
    UefiGuid(
        "4B6B21A1-F26B-404F-8694DB1C63A1D92A", name="LENOVO_AT24_RF08_EEPROM_SMM_GUID"
    ),
    UefiGuid(
        "4C7D1568-CF73-4676-A07916F7F9600CCC", name="LENOVO_SECURITY_SMI_DISPATCH_GUID"
    ),
    UefiGuid(
        "4D37DA42-3A0C-4EDA-B9EBBC0E1DB4713B",
        name="LENOVO_SYSTEM_PPIS_NEEDED_BY_DXE_CORE_GUID",
    ),
    UefiGuid(
        "4D62B5E9-71C8-412A-8604878C921D9AD1", name="LENOVO_SYSTEM_ERROR_LOG_SMM_GUID"
    ),
    UefiGuid("4ECEC19C-6E5F-4815-8342B55F2CC5DFC1", name="LENOVO_VPRO_HII_DXE_GUID"),
    UefiGuid(
        "4EFC51DA-23A6-4790-A2924985C7F5CDEF", name="LENOVO_COMPUTRACE_ENABLER_DXE_GUID"
    ),
    UefiGuid(
        "501BD93A-F1A9-4F4C-A9467867DF975537",
        name="LENOVO_EN25_QH64_FLASH_PART_SMM_GUID",
    ),
    UefiGuid("510550E0-BBCE-4EBF-BBA0946E3D18E705", name="LENOVO_PCH_EMULATE_KBC_GUID"),
    UefiGuid(
        "5112A2AA-E175-477E-A4E4D0B7E689BA9F",
        name="LENOVO_SYSTEM_SMBIOS_EVENT_LOG_DXE_GUID",
    ),
    UefiGuid("515B224B-B059-460B-85C7C28658F4B898", name="LENOVO_HPM_SMM_GUID"),
    UefiGuid(
        "51C9F40C-5243-4473-B265B3C8FFAFF9FA",
        name="LENOVO_SYSTEM_CRC32_SECTION_EXTRACT_DXE_GUID",
    ),
    UefiGuid(
        "51CCF399-4FDF-4E55-A45BE123F84D456A",
        name="LENOVO_SYSTEM_CON_PLATFORM_DXE_GUID",
    ),
    UefiGuid(
        "51D4BACC-E70A-4136-9426546973ED938C", name="LENOVO_PLATFORM_S3_SAVE_DXE_GUID"
    ),
    UefiGuid("520F9C58-9F9D-48C6-A0A0D9FE6D8FDE77", name="LENOVO_TDT_AM_GUID"),
    UefiGuid("529D3F93-E8E9-4E73-B1E1BDF6A9D50113", name="LENOVO_ARP_DXE_BIN_GUID"),
    UefiGuid("539D8AAD-C6AC-426C-B61F228E6D1501B6", name="LENOVO_HPM_DXE_GUID"),
    UefiGuid(
        "53BCC14F-C24F-434C-B2948ED2D4CC1860", name="LENOVO_SYSTEM_DATA_HUB_DXE_GUID"
    ),
    UefiGuid(
        "546AD295-0C64-4CE6-A06FFB9CD04B41CD",
        name="LENOVO_SYSTEM_FLASH_COMMUNICATION_DXE_GUID",
    ),
    UefiGuid(
        "54C32CBF-A3B9-4CCE-9C43E059E32255DF",
        name="LENOVO_SYSTEM_FIRMWARE_DEVICE_SMM_GUID",
    ),
    UefiGuid("550303D2-F033-4468-857A442C10E199E9", name="LENOVO_DRIVE_ERASE_DXE_GUID"),
    UefiGuid(
        "5552575A-7E00-4D61-A3A4F7547351B49E", name="LENOVO_SMM_BASE_RUNTIME_GUID"
    ),
    UefiGuid("55F9900A-BFB0-4B30-83EE7F437FAFAE33", name="LENOVO_DRIVE_ERASE_SMM_GUID"),
    UefiGuid("57F48613-300A-4101-A76D4F73C533B5B8", name="LENOVO_PRIOR_BOOT_GUID"),
    UefiGuid(
        "5A3F3BD1-B7A6-404B-A0F7285E1B898B00", name="LENOVO_DISK_CONTROLLER_SMBIOS_GUID"
    ),
    UefiGuid(
        "5B74C741-4A5F-4A8E-B689D804AB4368FD", name="LENOVO_DXE_REALTEK_CRC_INIT_GUID"
    ),
    UefiGuid(
        "5BDDE605-B107-419E-9510AA3C434EBBE4", name="LENOVO_PLATFORM_CSM_DXE_GUID"
    ),
    UefiGuid(
        "5BDEE236-0CB5-4D86-A608EE7B0C5F2956",
        name="LENOVO_SECURE_ERROR_MANAGER_DXE_GUID",
    ),
    UefiGuid("5BEDB5CC-D830-4EB2-87422D4CC9B54F2C", name="LENOVO_IP6_DXE_BIN_GUID"),
    UefiGuid(
        "5D758BFB-B323-4F6C-9017A3238CE65163", name="LENOVO_SETUP_RESTART_DXE_GUID"
    ),
    UefiGuid(
        "5F65D21A-8867-45D3-A41A526F9FE2C598",
        name="LENOVO_SYSTEM_IMAGE_DECODER_DXE_GUID",
    ),
    UefiGuid(
        "5FC8EEE0-9BF2-498F-B4D3C51ED4C87BB4", name="LENOVO_SETUP_CONFIG_DXE_GUID"
    ),
    UefiGuid(
        "605E41B8-E923-47CC-87B1A6E38526E78D", name="LENOVO_SMBIOS_COMPATI_INFO_GUID"
    ),
    UefiGuid(
        "61141CA0-1444-4C4E-A16DB601E0A09466", name="LENOVO_SYSTEM_KBC_INIT_DXE_GUID"
    ),
    UefiGuid(
        "621DE6C6-0F5E-4EE3-A1020BDE769A0AD4",
        name="LENOVO_REMOTE_CONFIG_UPDATE_DXE_GUID",
    ),
    UefiGuid("62D171CB-78CD-4480-8678C6A2A797A8DE", name="LENOVO_CPU_INIT_DXE_GUID"),
    UefiGuid(
        "63438C6E-971C-4E40-8F0B1109457BE333", name="LENOVO_CPU_INIT_PHNX_DXE_GUID"
    ),
    UefiGuid(
        "65246A3B-33EF-4F7E-B657A4A633F130B0", name="LENOVO_SYSTEM_VARIABLE_SMM_GUID"
    ),
    UefiGuid(
        "65A72030-B02E-4BF3-8424BA5F2FC56DE7", name="LENOVO_SETUP_UNDER_OS_SMM_GUID"
    ),
    UefiGuid("665D3F60-AD3E-4CAD-8E26DB46EEE9F1B5", name="LENOVO_RN_RCONFIG_GUID"),
    UefiGuid("66EECF40-6312-4A1A-A83AB3B2F8D8A71A", name="LENOVO_VARIABLE_DXE_GUID"),
    UefiGuid("67AFDE5F-EF16-47B8-BA19C21B3907DBF1", name="LENOVO_MISC_GA_IO_SMM_GUID"),
    UefiGuid(
        "68BC3095-8839-4A97-8FEABEE06ECA7BBB", name="LENOVO_W25_Q32_FLASH_PART_DXE_GUID"
    ),
    UefiGuid(
        "69FD8E47-A161-4550-B01A5594CEB2B2B2", name="LENOVO_SYSTEM_IDE_BUS_DXE_GUID"
    ),
    UefiGuid("6A628EFE-3682-4FDC-A31EC635BDF18CC8", name="LENOVO_OEM_MILESTONE_GUID"),
    UefiGuid(
        "6A7B1C86-590E-47A7-971AC49B408D5ED7", name="LENOVO_SETUP_SECURITY_DXE_GUID"
    ),
    UefiGuid("6A8A395F-4C07-49D1-B94C22ED50D425F8", name="LENOVO_SECURE_KEY_DXE_GUID"),
    UefiGuid("6D6963AB-906D-4A65-A7CABD40E5D6AF2B", name="LENOVO_UDP4_DXE_BIN_GUID"),
    UefiGuid(
        "6E32EAF6-32D1-43E0-B6E1DCF6DA81DF76",
        name="LENOVO_N25_Q064_FLASH_PART_SMM_GUID",
    ),
    UefiGuid("6E5228F3-933E-4961-95730F1E61B522AC", name="LENOVO_SMBIOS_VPRO_GUID"),
    UefiGuid(
        "6E6E4A75-EF14-47FB-B43A67EC43B87D22", name="LENOVO_W25_Q64_FLASH_PART_SMM_GUID"
    ),
    UefiGuid(
        "6E6F05AC-234B-465B-9450C36E6D7566C5", name="LENOVO_SYSTEM_ISA_KBC_DXE_GUID"
    ),
    UefiGuid(
        "706C1E0C-44FC-4012-9B713DBFEC762F35", name="LENOVO_PLATFORM_FLASH_SMM_GUID"
    ),
    UefiGuid(
        "707B3C6E-99A6-4E17-84A040284C1B3F38", name="LENOVO_PI_SAVE_STATE_ACCESS_GUID"
    ),
    UefiGuid(
        "717A0EAA-9B18-461A-B6ECAC09794CA631",
        name="LENOVO_SYSTEM_ACPI_SUPPORT_DXE_GUID",
    ),
    UefiGuid(
        "721C8B66-426C-4E86-8E993457C46AB0B9", name="LENOVO_SYSTEM_SETUP2_DXE_GUID"
    ),
    UefiGuid(
        "7391A960-CCD1-11DD-AD8B0800200C9A66",
        name="LENOVO_SYSTEM_IDE_ATA_ATAPI_PASS_THRU_SMM_GUID",
    ),
    UefiGuid(
        "73DAB30F-3F9C-4160-B064439CE16C3EDE",
        name="LENOVO_SYSTEM_BOOT_MANAGER_DXE_GUID",
    ),
    UefiGuid(
        "74A016D4-4E4B-45B5-AE974E6E33D65125", name="LENOVO_BLOCK_POWER_BUTTON_DXE_GUID"
    ),
    UefiGuid(
        "758AF8A7-2A04-4937-99E0837D1F97C76F", name="LENOVO_SYSTEM_ACPI_NUMA_DXE_GUID"
    ),
    UefiGuid(
        "77617ECB-E229-4E20-B8F9DF8245624D7A",
        name="LENOVO_SYSTEM_ACPI_TABLE_LOADER_DXE_GUID",
    ),
    UefiGuid(
        "77A6009E-116E-464D-8EF8B35201A022DD",
        name="LENOVO_DIGITAL_THERMAL_SENSOR_SMM_GUID",
    ),
    UefiGuid(
        "78FFCA55-4869-4393-BEE972E58BD3BE38",
        name="LENOVO_SYSTEM_FIXED_BUS_NUMBERS_DXE_GUID",
    ),
    UefiGuid(
        "793CBEA0-DA56-47F2-826424310CB75196", name="LENOVO_SYSTEM_VARIABLE_DXE_GUID"
    ),
    UefiGuid(
        "794610DA-D0EE-466B-81B38197213A6978",
        name="LENOVO_SYSTEM_SW_SMI_ALLOCATOR_RT_GUID",
    ),
    UefiGuid(
        "79CA4208-BBA1-4A9A-8456E1E66A81484E",
        name="LENOVO_SYSTEM8259_INTERRUPT_CONTROLLER_DXE_GUID",
    ),
    UefiGuid("79E0EDD7-9D1D-4F41-AE1AF896169E5216", name="LENOVO_WMA_POLICY_DXE_GUID"),
    UefiGuid("79F90154-83F4-478C-80BF878AC32E0A2C", name="LENOVO_LAN_UEFI_GUID"),
    UefiGuid("7B6F1DA6-EA4F-4938-A79F0C58E1442AA1", name="LENOVO_POP_MANAGER_DXE_GUID"),
    UefiGuid(
        "7BE725B2-F56C-41C7-9F031E7B56B65971",
        name="LENOVO_SYSTEM_USB_MEMORY_MANAGER_DXE_GUID",
    ),
    UefiGuid(
        "7CAB2E8A-B799-4540-851758644D03CC9F", name="LENOVO_ATP_SMI_SERVICES_GUID"
    ),
    UefiGuid(
        "7E715650-10E6-4B7B-896A4DC5FC742BC5", name="LENOVO_MODULES_SMM_THUNK_SMM_GUID"
    ),
    UefiGuid("7EA76D38-C57A-412D-AF1DD58BEE2B85B1", name="LENOVO_VPRO_PET_ALERT_GUID"),
    UefiGuid("7FED72EE-0170-4814-9878A8FB1864DFAF", name="LENOVO_SMM_RELOC_DXE_GUID"),
    UefiGuid("81334616-86CE-49C2-B6F91804E61C73F6", name="LENOVO_IGD_CUSTOMIZE_GUID"),
    UefiGuid(
        "8218965D-20C0-4DD6-81A0845C52270743", name="LENOVO_SETUP_DATE_TIME_DXE_GUID"
    ),
    UefiGuid(
        "826BCF56-BAC4-43F4-8EA18CDF0A121ABD", name="LENOVO_TRANSLATE_SERVICE_GUID"
    ),
    UefiGuid("82988420-7467-4490-9059FEB448DD1963", name="LENOVO_ME_CONFIG_GUID"),
    UefiGuid(
        "851A6355-CE10-4F67-9D1391B3281A95D3",
        name="LENOVO_SYSTEM_OEM_ACTIVATION_DXE_GUID",
    ),
    UefiGuid(
        "864B10FE-3D5E-4908-88CCF08D4EDC0050", name="LENOVO_SYSTEM_PCI_ISA_BUS_DXE_GUID"
    ),
    UefiGuid(
        "86E82AC3-D6AF-47A4-8C930C76C7B3CCD3", name="LENOVO_LEGACY_BIOS_MISC_SMM_GUID"
    ),
    UefiGuid("87E328F9-93A5-4375-BA6993A7A77A5D9B", name="LENOVO_INIT_ME_POLICY_GUID"),
    UefiGuid(
        "880D81BF-DA88-49E1-9543A58247FEAD1A", name="LENOVO_SETUP_AUTOMATION_SMM_GUID"
    ),
    UefiGuid(
        "886ACB20-78C0-4F88-9D3CBADCE37EDA47",
        name="LENOVO_N25_Q064_FLASH_PART_DXE_GUID",
    ),
    UefiGuid("89173692-9AC2-4C86-9ECCF37782DDEBAA", name="LENOVO_HDP_MANAGER_SMM_GUID"),
    UefiGuid(
        "8E5C4BEA-34FF-4E32-83583363DA01E628",
        name="LENOVO_SYSTEM_XHCI_RESET_SYSTEM_GUID",
    ),
    UefiGuid(
        "8E973F83-ABCA-49E4-A9404EAD5A6D8A1E", name="LENOVO_NATIONAL_LPC_PC87393_GUID"
    ),
    UefiGuid(
        "8F26EF0A-4F7F-4E4B-98028C22B700FFAC", name="LENOVO_SYSTEM_ENGLISH_DXE_GUID"
    ),
    UefiGuid(
        "8FEEECF1-BCFD-4A78-92314801566B3567",
        name="LENOVO_ABSOLUTE_COMPUTRACE_INSTALLER_WIN8_GUID",
    ),
    UefiGuid("90E01103-F784-4020-91ACC51E8BF553B2", name="LENOVO_WMA_PCI_DXE_GUID"),
    UefiGuid("918ABA30-3D8D-4BB5-B84945CC4FC7DE7C", name="LENOVO_DPTF_GUID"),
    UefiGuid(
        "93B80003-9FB3-11D4-9A3A0090273FC14D", name="LENOVO_SYSTEM_ISA_SERIAL_DXE_GUID"
    ),
    UefiGuid(
        "93B80004-9FB3-11D4-9A3A0090273FC14D", name="LENOVO_SYSTEM_PCI_BUS_DXE_GUID"
    ),
    UefiGuid("94734718-0BBC-47FB-96A5EE7A5AE6A2AD", name="LENOVO_DHCP4_DXE_BIN_GUID"),
    UefiGuid("949822BC-26D3-4BC6-954BF2C581342DE1", name="LENOVO_SYSTEM_XHCI_DXE_GUID"),
    UefiGuid("95BF86AD-A1E0-4143-B487004B1C2E05FA", name="LENOVO_DXE_CMOS_INIT_GUID"),
    UefiGuid(
        "95C051C5-F123-432E-9383B9CF113F082E", name="LENOVO_TCG_SERVICES_SMM_GUID"
    ),
    UefiGuid("95E3669D-34BE-4775-A6517EA41B69D89E", name="LENOVO_DHCP6_DXE_BIN_GUID"),
    UefiGuid(
        "961578FE-B6B7-44C3-AF356BC705CD2B1F",
        name="LENOVO_SYSTEM_ENHANCED_FAT_DXE_GUID",
    ),
    UefiGuid(
        "96B5C032-DF4C-4B6E-8232438DCF448D0E",
        name="LENOVO_SYSTEM_NULL_MEMORY_TEST_DXE_GUID",
    ),
    UefiGuid("970F9BDA-88FC-494C-9EA126419663CE11", name="LENOVO_SMM_CF9_TRAP_GUID"),
    UefiGuid("99F03B99-98D8-49DD-A8D33219D0FFE41E", name="LENOVO_MTFTP6_DXE_BIN_GUID"),
    UefiGuid(
        "9A9A912B-5F53-4586-8820704485A29D21",
        name="LENOVO_PLATFORM_RESET_RUNTIME_DXE_GUID",
    ),
    UefiGuid(
        "9B680FCE-AD6B-4F3A-B60BF59899003443", name="LENOVO_SYSTEM_DEVICE_PATH_DXE_GUID"
    ),
    UefiGuid(
        "9CC55D7D-FBFF-431C-BC14334EAEA6052B", name="LENOVO_SMM_CORE_DISPATCHER_GUID"
    ),
    UefiGuid("9CFD802C-09A1-43D6-8217AA49C1F90D2C", name="LENOVO_MEBX_GUID"),
    UefiGuid(
        "9D58058E-C8DD-4A9F-B176DFCB92354815",
        name="LENOVO_EN25_QH128_FLASH_PART_DXE_GUID",
    ),
    UefiGuid("9D5EB938-9A44-4568-8192C1563BEBAA01", name="LENOVO_ISSC_DXE_GUID"),
    UefiGuid(
        "9E863906-A40F-4875-977F5B93FF237FC6", name="LENOVO_SYSTEM_TERMINAL_DXE_GUID"
    ),
    UefiGuid(
        "9EA5DF0F-A35C-48C1-BAC9F63452B47C3E", name="LENOVO_SYSTEM_CAPSULE_RT_GUID"
    ),
    UefiGuid("9FA2F805-3D86-42BC-A9C32B26A5DF09F9", name="LENOVO_PCH_CONFIG_GUID"),
    UefiGuid("9FB1A1F3-3B71-4324-B39A745CBB015FFF", name="LENOVO_IP4_DXE_BIN_GUID"),
    UefiGuid(
        "9FB4B4A7-42C0-4BCD-85409BCC6711F83E",
        name="LENOVO_SYSTEM_USB_MASS_STORAGE_DXE_GUID",
    ),
    UefiGuid(
        "A196BA47-8ED3-4188-A765FA964DD7ED7A", name="LENOVO_SYSTEM_ERROR_LOG_DXE_GUID"
    ),
    UefiGuid("A210F973-229D-4F4D-AA379895E6C9EABA", name="LENOVO_DPC_DXE_BIN_GUID"),
    UefiGuid("A2F436EA-A127-4EF8-957C8048606FF670", name="LENOVO_SNP_DXE_BIN_GUID"),
    UefiGuid(
        "A4928C81-0703-4AD7-A32B7525BC3FE1F5", name="LENOVO_SECURE_UPDATE_DXE_GUID"
    ),
    UefiGuid("A4B9A806-B574-4354-BFC705217DC0FAF4", name="LENOVO_OEM_HOOK_SMM_GUID"),
    UefiGuid(
        "A4F2909C-5E2A-438A-91BA272B0923049A", name="LENOVO_PLATFORM_SETUP_DXE_GUID"
    ),
    UefiGuid(
        "A4FD00D3-821A-4519-85BB2E1987A6F4E1",
        name="LENOVO_MX25_L3206_EFLASH_PART_DXE_GUID",
    ),
    UefiGuid(
        "A6688890-BFF0-4F03-9B1E76AE86EEC5AB", name="LENOVO_SYSTEM_ACPI_TABLES_DXE_GUID"
    ),
    UefiGuid("A6A1C030-F956-4A9E-9540D0E81D2E33F2", name="LENOVO_SM_BIOS_HOLE_GUID"),
    UefiGuid(
        "A7D8D9A6-6AB0-4AEB-AD9D163E59A7A380",
        name="LENOVO_SYSTEM_DIAGNOSTIC_SPLASH_SCREEN_APP_GUID",
    ),
    UefiGuid("A89EC8E0-0BA1-40AA-A03EABDDA5295CDE", name="LENOVO_PCI_EXPRESS_GUID"),
    UefiGuid(
        "A94A6AEF-D69B-4010-BA24F91E1FE62ADA", name="LENOVO_SYSTEM_SECURE_BOOT_DXE_GUID"
    ),
    UefiGuid("AA382865-12B0-44E5-A7316DEF4DF34CE7", name="LENOVO_PROMPT_SERVICE_GUID"),
    UefiGuid(
        "AADFA1AC-E923-4673-B1B8714AD849F790",
        name="LENOVO_SYSTEM_FORM_BROWSER_SIMPLE_TEXT_VIEW_LAYOUT_DXE_GUID",
    ),
    UefiGuid("ABB74F50-FD2D-4072-A321CAFC72977EFA", name="LENOVO_SMM_RELOC_PEIM_GUID"),
    UefiGuid(
        "AC3435BB-B1D3-4EF8-957C8048606FF671",
        name="LENOVO_SYSTEM_FRAMEWORK_HII_DATABASE_DXE_GUID",
    ),
    UefiGuid(
        "AC5919D9-F137-4F87-869FD863EDA88ED2", name="LENOVO_INIT_VPRO_POLICY_GUID"
    ),
    UefiGuid(
        "AC60ED9F-523E-4F5B-94CA3961346A00BA", name="LENOVO_VARIABLE_INIT_DXE_GUID"
    ),
    UefiGuid(
        "AC6FD56A-3D41-4EFD-A1B9870293811A28", name="LENOVO_VPRO_MEBX_HOTKEY_GUID"
    ),
    UefiGuid(
        "ACA0F10C-04C9-4C4E-9D7BB7A097F5E1EC",
        name="LENOVO_EN25_QH64_FLASH_PART_DXE_GUID",
    ),
    UefiGuid("ACAEAA7A-C039-4424-88DAF42212EA0E55", name="LENOVO_PCH_PCIE_SMM_GUID"),
    UefiGuid(
        "AD608272-D07F-4964-801E7BD3B7888652",
        name="LENOVO_SYSTEM_MONOTONIC_COUNTER_RT_GUID",
    ),
    UefiGuid(
        "AD70855E-0CC5-4ABF-8979BE762A949EA3",
        name="LENOVO_INCOMPATIBLE_PCI_DEVICE_GUID",
    ),
    UefiGuid("AF59F2F5-5E28-4E03-80E24727545AF811", name="LENOVO_PCH_RESET_GUID"),
    UefiGuid("B2A23305-9455-489C-AA8121DB48C41EAB", name="LENOVO_AOAC_DXE_GUID"),
    UefiGuid(
        "B31FF060-8363-11DF-83950800200C9A66", name="LENOVO_SYSTEM_AHCI_BUS_SMM_GUID"
    ),
    UefiGuid(
        "B601F8C4-43B7-4784-95B1F4226CB40CEE", name="LENOVO_SYSTEM_RUNTIME_DXE_GUID"
    ),
    UefiGuid(
        "B62EFBBB-3923-4CB9-A6E8DB818E828A80", name="LENOVO_MEBX_SETUP_BROWSER_GUID"
    ),
    UefiGuid(
        "B63EBF4E-D196-40BA-AB631FA9A6068E84", name="LENOVO_OS_OPT_DEFAULT_DXE_GUID"
    ),
    UefiGuid("B65971BE-BABF-49ED-9DD248EC8DB4ABD3", name="LENOVO_SOUND_SERVICE_GUID"),
    UefiGuid("B6B9295F-CABF-4CEC-BB14FE4246F2173A", name="LENOVO_I_FFS_DXE_GUID"),
    UefiGuid(
        "B74BB37A-ECA2-4F79-A54490569ABE6B99",
        name="LENOVO_SYSTEM_SMM_COMMUNICATION_SMM_GUID",
    ),
    UefiGuid(
        "B8A6E7C5-B8FD-425C-A67E1009DF1F10B5", name="LENOVO_USER_MANAGER_DXE_GUID"
    ),
    UefiGuid(
        "B95E9FDA-26DE-48D2-88071F9107AC5E3A", name="LENOVO_UEFI_PXE_BC_DXE_BIN_GUID"
    ),
    UefiGuid("B9A3F174-1B36-4AEC-99E2F2855EB4C3DE", name="LENOVO_BOARD_INFO_DXE_GUID"),
    UefiGuid(
        "B9CABA9B-E5C5-41DC-86E2E7CEF87B81CB",
        name="LENOVO_SYSTEM_FLASH_COMMUNICATION_SMM_GUID",
    ),
    UefiGuid("B9F93638-35F5-447D-B908A2B852AA0B89", name="LENOVO_SMBIOS_TCG_DXE_GUID"),
    UefiGuid("BAE7599F-3C6B-43B7-BDF09CE07AA91AA6", name="LENOVO_CPU_IO_GUID"),
    UefiGuid(
        "BAF029A0-B2F3-45EE-8B52D402177BE6B8", name="LENOVO_OS_OPT_DEFAULT_SMM_GUID"
    ),
    UefiGuid(
        "BCB59C90-DC86-11DD-AD8B0800200C9A66",
        name="LENOVO_SYSTEM_AHCI_ATA_ATAPI_PASS_THRU_SMM_GUID",
    ),
    UefiGuid(
        "BD6736AC-B126-4FEA-9D1D174D4A899F22", name="LENOVO_SYSTEM_ERROR_MENU_DXE_GUID"
    ),
    UefiGuid("BDFE430E-8F2A-4DB0-99916F856594777E", name="LENOVO_SYSTEM_EHCI_DXE_GUID"),
    UefiGuid(
        "BF73D2BF-AC8C-4D83-A3B94A0F420074B3",
        name="LENOVO_STATUS_CODE_ME_DEBUG_DXE_GUID",
    ),
    UefiGuid(
        "BF89F10D-B205-474F-96E37A7BB1B4A407",
        name="LENOVO_SYSTEM_TEXT_CONSOLE_VGA_DXE_GUID",
    ),
    UefiGuid("BFEE74C2-60BD-467B-9261B72A881E49EE", name="LENOVO_MAIL_BOX_SMM_GUID"),
    UefiGuid("C07CCCFC-B4BF-4A30-A25F1F57C0522629", name="LENOVO_SMM_FDISK_OEM_GUID"),
    UefiGuid(
        "C0CFEB8B-6EE1-443B-BCC9854E7C9B416D",
        name="LENOVO_SYSTEM_STATUS_CODE_PORT80_SMM_GUID",
    ),
    UefiGuid("C194C6EA-B68C-4981-B64B9BD271474B20", name="LENOVO_PCH_SPI_RUNTIME_GUID"),
    UefiGuid("C3EE3EF0-63E6-4F59-A437AFA63C46C835", name="LENOVO_SMM_ASL_SMI_GUID"),
    UefiGuid(
        "C69BF28B-16C3-434F-A4A4292BDB0F4D12", name="LENOVO_VARIABLE_DEFAULT_GUID"
    ),
    UefiGuid("C7EA9787-CA0A-43B4-B1E525EF87391F8D", name="LENOVO_PCH_S3_SUPPORT_GUID"),
    UefiGuid("C80EA8F3-A3C9-4225-AA60769DD4C74E43", name="LENOVO_SVP_MANAGER_DXE_GUID"),
    UefiGuid(
        "C82919D6-33FD-43ED-800C635A3369D0F3",
        name="LENOVO_SYSTEM_DIAGNOSTIC_SUMMARY_SCREEN_DXE_GUID",
    ),
    UefiGuid(
        "C89851DD-5EA0-42CE-8934EB8EA608FBA9",
        name="LENOVO_SYSTEM_ACCELERATOR_KEY_DXE_GUID",
    ),
    UefiGuid(
        "C8AB0F4E-26FE-40F1-9579EA8D30D503A4",
        name="LENOVO_SYSTEM_FLASH_UPDATE_DRIVER_DXE_GUID",
    ),
    UefiGuid("C97809C0-5F90-4466-8B38341D7A67CF54", name="LENOVO_SMAPI_SMM_GUID"),
    UefiGuid(
        "CA261A26-7718-4B9B-8A075178B1AE3A02", name="LENOVO_SYSTEM_DISK_IO_DXE_GUID"
    ),
    UefiGuid(
        "CA515306-00CE-4032-874E11B755FF6866",
        name="LENOVO_SYSTEM_DATA_HUB_STD_ERR_DXE_GUID",
    ),
    UefiGuid(
        "CA7ACEF3-1181-4AE0-95A70F1FD8AFD2ED", name="LENOVO_SYSTEM_KEY_DESC_DXE_GUID"
    ),
    UefiGuid("CA9725C0-12E5-4FAC-AD58D9AAB03B8F11", name="LENOVO_HDP_MANAGER_DXE_GUID"),
    UefiGuid("CB73C3D9-2F01-4342-AE6704DDE5264092", name="LENOVO_PLATFORM_SMM_GUID"),
    UefiGuid(
        "CBCA3D88-CB1E-49C3-B4485FF287C58D40",
        name="LENOVO_EN25_QH32_FLASH_PART_SMM_GUID",
    ),
    UefiGuid("CC71B046-CF07-4DAE-AEAD7046845BCD8A", name="LENOVO_VIDEO_INIT_DXE_GUID"),
    UefiGuid("CC84D725-DA1E-46A7-9E754BACF1FD3902", name="LENOVO_SYSTEM_FONT_DXE_GUID"),
    UefiGuid(
        "CCABD229-51C4-4DE7-AE2F3AAF71F9ECE5",
        name="LENOVO_SYSTEM_SETUP_ADVANCED_DXE_GUID",
    ),
    UefiGuid(
        "CCCB0C28-4B24-11D5-9A5A0090273FC14D",
        name="LENOVO_SYSTEM_GRAPHICS_CONSOLE_DXE_GUID",
    ),
    UefiGuid(
        "CD92B9B9-2D4C-466B-845895447D8E4AFE", name="LENOVO_SYSTEM_SM_BIOS_SMM_GUID"
    ),
    UefiGuid(
        "CDC0191E-1891-4119-817E0B6A3685A511", name="LENOVO_SYSTEM_BUS_ISA_RTC_SMM_GUID"
    ),
    UefiGuid(
        "CF9668F7-B0F0-4D7F-B41EAEC794EEC7FC",
        name="LENOVO_SYSTEM_ACPI_SMI_SERVICES_DXE2_GUID",
    ),
    UefiGuid(
        "CFEF94C4-4167-466A-88938779459DFA86",
        name="LENOVO_PLATFORM_HII_ADVANCED_DXE_GUID",
    ),
    UefiGuid("D5196882-A970-4510-8FB9D7583B6906E7", name="LENOVO_VARIABLE_SMM_GUID"),
    UefiGuid(
        "D687E479-DB37-4BCE-864A02EEF6819DF1",
        name="LENOVO_SYSTEM_FORM_BROWSER_SIMPLE_TEXT_VIEW_DXE_GUID",
    ),
    UefiGuid(
        "D689F8C7-C354-4215-AA48B6AD66C77EAF",
        name="LENOVO_N25_Q032_FLASH_PART_SMM_GUID",
    ),
    UefiGuid(
        "D83977DF-34C3-4A21-A104369B8D4FA7B7", name="LENOVO_MFG_BENCH_EVENT_DXE_GUID"
    ),
    UefiGuid("D912C7BC-F098-4367-92BAE911083C7B0E", name="LENOVO_UDP6_DXE_BIN_GUID"),
    UefiGuid(
        "DA5D9983-033C-4823-93498B1B6A798030",
        name="LENOVO_SYSTEM_STATUS_CODE_GENERIC_SMM_GUID",
    ),
    UefiGuid(
        "DB4DB11A-0E5C-40B5-8E4617D04486A21B",
        name="LENOVO_MX25_L64_XFLASH_PART_SMM_GUID",
    ),
    UefiGuid("DC3641B8-2FA8-4ED3-BC1FF9962A03454B", name="LENOVO_MTFTP4_DXE_BIN_GUID"),
    UefiGuid(
        "DDB412A6-E3F3-4E9E-90A32A991270219C", name="LENOVO_I_FFS_DXE_POLICY_INIT_GUID"
    ),
    UefiGuid("E0ADB57E-E1B6-44EC-BF2E842874A26C83", name="LENOVO_WMA_USB_DXE_GUID"),
    UefiGuid(
        "E1AD4352-2610-4DD6-BB8F8BB2B03383A3",
        name="LENOVO_SYSTEM_SCSI_OPROM_PASS_THRU_DXE_GUID",
    ),
    UefiGuid("E4F272DA-237F-454C-8868FCF096CF1C6C", name="LENOVO_PASSWORD_CP_GUID"),
    UefiGuid(
        "E5D0BBDC-1BBC-49B7-A8B667AAF0A1CD9E",
        name="LENOVO_SYSTEM_DIAGNOSTIC_SPLASH_SCREEN_DXE_GUID",
    ),
    UefiGuid(
        "E84AEF95-7CA6-4B2F-A398B14D1C67A0B1",
        name="LENOVO_IBEX_PEAK_FLASH_CONTROLLER_SMM_GUID",
    ),
    UefiGuid("E8DC00BA-411F-4D52-B92550785436AC81", name="LENOVO_SMM_SLEEP_EVENT_GUID"),
    UefiGuid(
        "E94DB579-061B-437B-83C265C0AE7C14E2", name="LENOVO_AT24_RF08_EEPROM_DXE_GUID"
    ),
    UefiGuid("E9B4B126-4E13-41F5-9E4C9BF88B3C1B0C", name="LENOVO_KBD_CONFIG_GUID"),
    UefiGuid(
        "EA2EC402-2FD5-475F-922C98EAE0376312", name="LENOVO_SYSTEM_LEGACY_BIOS_DXE_GUID"
    ),
    UefiGuid(
        "EA9E92E1-22EF-4B5F-A5EE1317B1E77B10", name="LENOVO_VPRO_CIRA_HOT_KEY_GUID"
    ),
    UefiGuid("EAAF7109-EC2D-4A30-BF4F621408630E4C", name="LENOVO_OEM_HOOK_GUID"),
    UefiGuid(
        "EAF59C0E-BD46-413A-9AE9DD9F6D1A927D", name="LENOVO_SYSTEM_SMBIOS_DXE_GUID"
    ),
    UefiGuid("EB6EEBF4-1EC3-4D69-9D8265BF6C579B34", name="LENOVO_SECURE_KEY_SMM_GUID"),
    UefiGuid(
        "EC5A555B-6384-4C00-86252C1768B95B38", name="LENOVO_ME_CON_OUT_READY_GUID"
    ),
    UefiGuid("EE30FD26-1524-4CA2-B56D345830DC9CDB", name="LENOVO_FINGERPRINT_CP_GUID"),
    UefiGuid(
        "EF33C296-F64C-4146-AD04347899702C84",
        name="LENOVO_SYSTEM_SMM_USB_LEGACY_SMM_GUID",
    ),
    UefiGuid(
        "F099D67F-71AE-4C36-B2A3DCEB0EB2B7D8",
        name="LENOVO_SYSTEM_WATCHDOG_TIMER_DXE_GUID",
    ),
    UefiGuid(
        "F1BDCEC7-31D1-4F0F-AF62D02A44076026",
        name="LENOVO_N25_Q032_FLASH_PART_DXE_GUID",
    ),
    UefiGuid(
        "F1EFB523-3D59-4888-BB71EAA5A96628FA",
        name="LENOVO_SYSTEM_SECURITY_STUB_DXE_GUID",
    ),
    UefiGuid("F46D8EA6-4A67-4872-B0D1D4FDEA0B692F", name="LENOVO_SVP_MANAGER_SMM_GUID"),
    UefiGuid(
        "F46EE6F4-4785-43A3-923D7F786C3C8479", name="LENOVO_STARTUP_MENU_DXE_GUID"
    ),
    UefiGuid(
        "F5AC7057-5650-466E-B69276A47223EFB0", name="LENOVO_PLATFORM_S3_SAVE_SMM_GUID"
    ),
    UefiGuid("F642A9D7-26B7-47CB-ACC8135951AC93F8", name="LENOVO_SYSTEM_CMOS_DXE_GUID"),
    UefiGuid("F746D37F-F6C6-43C0-94DB466F5F10E030", name="LENOVO_FINGERPRINT_SMM_GUID"),
    UefiGuid(
        "F7731B4C-58A2-4DF4-89805645D39ECE58", name="LENOVO_POWER_MANAGEMENT2_GUID"
    ),
    UefiGuid(
        "F8CA70B4-6A85-4616-8BDE3EC569644AC3", name="LENOVO_W25_Q32_FLASH_PART_SMM_GUID"
    ),
    UefiGuid(
        "F94700A3-F49F-4CD2-BF2519E280B72E3B",
        name="LENOVO_SYSTEM_USB_DATABASE_DXE_GUID",
    ),
    UefiGuid("F94A048B-7FC4-4ABC-85A68616A6D3CDCE", name="LENOVO_TCG_SMM_GUID"),
    UefiGuid("FA4F6740-B95A-43F9-90B578D8147C0219", name="LENOVO_SLP2_SMM_GUID"),
    UefiGuid("FBB3F083-5787-45AF-BADC664854100F20", name="LENOVO_FIT_DIAGNOSTICS_GUID"),
    UefiGuid(
        "FCE47C4E-5ECC-4A41-B90E0BAB09968D02",
        name="LENOVO_SYSTEM_SECURE_FLASH_SLEEP_TRAP_SMM_GUID",
    ),
    UefiGuid("FF917E22-A228-448D-BDAA68EFCCDDA5D3", name="LENOVO_TXT_DXE_GUID"),
    UefiGuid("1E8DCCE9-341C-4219-A790F14C28B51CEE", name="LENOVO_EAIA_PEI_GUID"),
    UefiGuid(
        "233DF097-3218-47B2-9E09FE58C2B20D22", name="LENOVO_PLATFORM_STAGE2_PEI_GUID"
    ),
    UefiGuid("367712FD-6C5A-4600-BEA29A1F81AACC08", name="LENOVO_MISC_GA_IO_PEIM_GUID"),
    UefiGuid("52C05B14-0B98-496C-BC3B04B50211D680", name="LENOVO_PEI_MAIN_GUID"),
    UefiGuid(
        "53984C6A-1B4A-4174-9512A65E5BC8B278", name="LENOVO_PLATFORM_STAGE1_PEI_GUID"
    ),
    UefiGuid(
        "53AC1948-0ED0-428A-B4DDD2FFF2F5776F", name="LENOVO_FLASH_PROTECT_PEI_GUID"
    ),
    UefiGuid(
        "55A2532E-297A-4AFD-90BF139BFF5BB4D7", name="LENOVO_SYSTEM_ACPI_TABLES_PEI_GUID"
    ),
    UefiGuid("5920F406-5868-44F5-A9B96D4031481CC9", name="LENOVO_OEM_SEC_PEI_GUID"),
    UefiGuid(
        "8401A046-6F70-4505-84717015B40355E3", name="LENOVO_SYSTEM_USB_BOT_PEI_GUID"
    ),
    UefiGuid(
        "863095D8-3BF4-447C-AE1015B3AE9936FD", name="LENOVO_SYSTEM_BOOT_MODE_PEI_GUID"
    ),
    UefiGuid("8A78B107-0FDD-4CC8-B7BADC3E13CB8524", name="LENOVO_PEI_CPU_IO_GUID"),
    UefiGuid(
        "8BCEDDD7-E285-4168-9B3F09AF66C93FFE", name="LENOVO_SYSTEM_S3_RESUME_PEI_GUID"
    ),
    UefiGuid("9CCFE1AC-784C-463D-B145396D7014D559", name="LENOVO_VIDEO_INIT_PEIM_GUID"),
    UefiGuid(
        "9DD92425-6737-45CD-829C1AC2CD20C873", name="LENOVO_AT24_RF08_EEPROM_PEI_GUID"
    ),
    UefiGuid(
        "B9C37931-F069-418C-9E1931B0BC031887", name="LENOVO_POST_VALIDATOR_PEI_GUID"
    ),
    UefiGuid(
        "C2223A0B-80D3-4C35-891FBD0CC4E990F0", name="LENOVO_SYSTEM_FV_FILE_LOADER_GUID"
    ),
    UefiGuid("C311C600-84E9-4F1E-BF44BF36E3415158", name="LENOVO_MAIL_BOX_PEI_GUID"),
    UefiGuid("C37297CC-3FD9-4026-98907A1628184620", name="LENOVO_EC_IO_PEIM_GUID"),
    UefiGuid(
        "C7AD44B9-A775-4801-B772A964262BB92E", name="LENOVO_MODULE_S3_RESUME_PEIM_GUID"
    ),
    UefiGuid("C866BD71-7C79-4BF1-A93B066B830D8F9A", name="LENOVO_CPU_S3_PEIM_GUID"),
    UefiGuid(
        "E010BCCA-3A3B-4B03-BCF4AE1F64A85C89", name="LENOVO_USER_MANAGER_PEI_GUID"
    ),
    UefiGuid("EA42B5E6-5E01-4C6E-8189C1EFA6ACFD01", name="LENOVO_PROTECT_PBPEI_GUID"),
    UefiGuid("ECCBF582-EEE1-45B8-8926067F40B3E55E", name="LENOVO_ISSC_PEI_GUID"),
    UefiGuid(
        "ED52984E-6ED7-4445-9D5D200C3201F51E", name="LENOVO_PLATFORM_STAGE0_PEI_GUID"
    ),
    UefiGuid("E7E1EFA6-7607-4A78-A7DD43E4BD72C099", name="APP_PKG_TOKEN_SPACE_GUID"),
    UefiGuid("A1147A20-3144-4F8D-8295B48311C8E4A4", name="ARM_JUNO_TOKEN_SPACE_GUID"),
    UefiGuid(
        "AD651C7D-3C22-4DBF-92E838A7CDAE87B2",
        name="VIRTUAL_UNCACHED_PAGES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "CBD2E4D5-7068-4FF5-B4629822B4AD8D60", name="VARIABLE_RUNTIME_DXE_FILE_GUID"
    ),
    UefiGuid("C3253C90-A24F-4599-A6641F8813778FC9", name="ARM_GLOBAL_VARIABLE_GUID"),
    UefiGuid(
        "41E26B9C-ADA6-45B3-808E2357A35B60D6", name="ARM_BOOT_MON_FS_FILE_INFO_GUID"
    ),
    UefiGuid(
        "AFFE115B-8589-456D-BAB58F2EDA53AEB7", name="ARM_PLATFORM_UPDATE_FDT_EVENT_GUID"
    ),
    UefiGuid(
        "AB1C1816-D542-4E6F-9B1E8ECD9253E2E7", name="ARM_GLOBAL_VARIABLE_PPI_GUID"
    ),
    UefiGuid(
        "44577A0D-361A-45B2-B33DBB9EE60D5A4F",
        name="ARM_REAL_VIEW_EB_PKG_TOKEN_SPACE_GUID",
    ),
    UefiGuid(
        "9C0AAED4-74C5-4043-B417A3223814CE76", name="ARM_V_EXPRESS_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "0B6F5CA7-4F53-445A-B76E2E365B806366",
        name="ARM_VIRTUALIZATION_TOKEN_SPACE_GUID",
    ),
    UefiGuid(
        "6834FE45-4AEE-4FC6-BCB5FF45B7A871E2", name="BEAGLE_BOARD_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "E1475E0C-1746-4802-862E011C2C2D9D86", name="EFI_RUNTIME_CRYPT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "3677D529-326F-4603-A926EAACE01DCBB0", name="EFI_PCI_EXPRESS_BASE_ADDRESS_GUID"
    ),
    UefiGuid("3C699197-093C-4C69-B06B128AE3481DC9", name="EFI_ACPI_DESCRIPTION_GUID"),
    UefiGuid("B091E7D2-05A0-4198-94F074B7B8C55459", name="EFI_FLASH_MAP_HOB_GUID"),
    UefiGuid(
        "7462660F-1CBD-48DA-AD1191717913831C", name="EFI_PCI_OPTION_ROM_TABLE_GUID"
    ),
    UefiGuid(
        "ED150714-DF30-407D-B24A4B742FD5CEA2", name="DUET_CONSOLE_OUT_CONFIG_GUID"
    ),
    UefiGuid("D6A2CB7F-6A18-4E2F-B43B9920A733700A", name="DXE_CORE_FILE_NAME_GUID"),
    UefiGuid("7701D7E5-7D1D-4432-A468673DAB8ADE60", name="LDR_MEMORY_DESCRIPTOR_GUID"),
    UefiGuid(
        "6568A3D6-015F-4B4A-9C891D146314130A",
        name="EFI_SMM_BASE_THUNK_COMMUNICATION_GUID",
    ),
    UefiGuid("60B5E939-0FCF-4227-BA836BBED45BC0E3", name="EFI_BOOT_STATE_GUID"),
    UefiGuid(
        "2354D320-3EB3-4C81-99F4CC4AE143463E",
        name="EFI_EDK_COMPATIBILITY_PKG_TOKEN_SPACE_GUID",
    ),
    UefiGuid(
        "9E0C30BC-3F06-4BA6-828809179B855DBE",
        name="FRAMEWORK_BDS_FRONT_PAGE_FORMSET_GUID",
    ),
    UefiGuid("B0EE53D4-A049-4A79-B2FF19D9FAEFAA94", name="ECP_PEI_PCI_CFG_PPI_GUID"),
    UefiGuid("DF2D868E-32FC-4CF0-8E6BFFD95D1343D0", name="EFI_PRINT_PROTOCOL_GUID"),
    UefiGuid(
        "910DCA07-1F94-4EE7-AF2FFF72F3154353",
        name="EFI_SMM_BASE_HELPER_READY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "9546E07C-2CBB-4C88-986CCD341086F044",
        name="EFI_DEBUG_SUPPORT_PERIODIC_CALLBACK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "AEDA2428-9A22-4637-9B21545E28FBB829", name="EFI_EBL_ADD_COMMAND_PROTOCOL_GUID"
    ),
    UefiGuid(
        "D3705011-BC19-4AF7-BE16F68030378C15",
        name="EFI_INTEL_FRAMEWORK_MODULE_PKG_TOKEN_SPACE_GUID",
    ),
    UefiGuid(
        "D083E94C-6560-42E4-B6D42DF75ADF6A2A",
        name="EFI_DATA_HUB_STATUS_CODE_RECORD_GUID",
    ),
    UefiGuid(
        "C020489E-6DB2-4EF2-9AA5CA06FC11D36A",
        name="EFI_ACPI_VARIABLE_COMPATIBLITY_GUID",
    ),
    UefiGuid("283FA2EE-532C-484D-93839F93B36F0B7E", name="EFI_UPDATE_DATA_FILE_GUID"),
    UefiGuid("CF31FAC5-C24E-11D2-85F300A0C93EC93B", name="BLOCK_IO_VENDOR_GUID"),
    UefiGuid("9E0C30BC-3F06-4BA6-828809179B855DBE", name="FRONT_PAGE_FORM_SET_GUID"),
    UefiGuid("847BC3FE-B974-446D-94495AD5412E993B", name="BOOT_MANAGER_FORM_SET_GUID"),
    UefiGuid(
        "3EBFA8E6-511D-4B5B-A95FFB38260F1C27", name="DEVICE_MANAGER_FORM_SET_GUID"
    ),
    UefiGuid("F76E0A70-B5ED-4C38-AC9AE5F54BF16E34", name="DRIVER_HEALTH_FORM_SET_GUID"),
    UefiGuid("642237C7-35D4-472D-836512E0CCF27A22", name="BOOT_MAINT_FORM_SET_GUID"),
    UefiGuid("1F2D63E1-FEBD-4DC7-9CC5BA2B1CEF9C5B", name="FILE_EXPLORE_FORM_SET_GUID"),
    UefiGuid("3B4D9B23-95AC-44F6-9FCD0E9594586C72", name="BDS_LIB_STRING_PACKAGE_GUID"),
    UefiGuid("0E8C545B-A2EE-470D-8E26BDA1A13C0AA3", name="LAST_ENUM_LANG_GUID"),
    UefiGuid(
        "FAB7E9E1-39DD-4F2B-8408E20E906CB6DE", name="HD_BOOT_DEVICE_PATH_VARIABL_GUID"
    ),
    UefiGuid(
        "170E13C0-BF1B-4218-871D2ABDC6F887BC", name="EFI_O_E_M_BADGING_PROTOCOL_GUID"
    ),
    UefiGuid("D088A413-0A70-4217-BA559A3CB65C41B3", name="EXIT_PM_AUTH_PROTOCOL_GUID"),
    UefiGuid("7F0013A7-DC79-4B22-809911F75FDC829D", name="EFI_CACHE_SUB_CLASS_GUID"),
    UefiGuid("4E8F4EBB-64B9-4E05-9B184CFE49235097", name="EFI_MEMORY_SUB_CLASS_GUID"),
    UefiGuid("772484B2-7482-4B91-9F9AAD43F81C5881", name="EFI_MISC_SUB_CLASS_GUID"),
    UefiGuid(
        "26FDEB7E-B8AF-4CCF-AA9702633CE48CA7", name="EFI_PROCESSOR_SUB_CLASS_GUID"
    ),
    UefiGuid("3B6686BD-0D76-4030-B70EB5519E2FC5A0", name="EFI_CAPSULE_GUID"),
    UefiGuid("98B8D59B-E8BA-48EE-98DDC295392F1EDB", name="EFI_CONFIG_FILE_NAME_GUID"),
    UefiGuid(
        "6DADF1D1-D4CC-4910-BB6E82B1FD80FF3D",
        name="EFI_SMM_PEI_SMRAM_MEMORY_RESERVE_GUID",
    ),
    UefiGuid("F328E36C-23B6-4A95-854B32E19534CD75", name="SMM_COMMUNICATE_HEADER_GUID"),
    UefiGuid(
        "7A9354D9-0468-444A-81CE0BF617D890DF", name="EFI_FIRMWARE_FILE_SYSTEM_GUID"
    ),
    UefiGuid(
        "964E5B22-6459-11D2-8E3900A0C969723B", name="EFI_PEI_IDE_BLOCK_IO_PPI_GUID"
    ),
    UefiGuid(
        "DA6855BD-07B7-4C05-9ED8E259FD360E22",
        name="EFI_PEI144_FLOPPY_BLOCK_IO_PPI_GUID",
    ),
    UefiGuid(
        "ABD42895-78CF-4872-84441B5C180BFBFF",
        name="EFI_PEI_BOOT_SCRIPT_EXECUTER_PPI_GUID",
    ),
    UefiGuid("1388066E-3A57-4EFA-98F3C12F3A958A29", name="EFI_PEI_SECURITY_PPI_GUID"),
    UefiGuid("ABD42895-78CF-4872-84441B5C180BFBDA", name="EFI_PEI_SMBUS_PPI_GUID"),
    UefiGuid(
        "E1F2EBA0-F7B9-4A26-8620131221642A90",
        name="EFI_PCI_CFG_PPI_IN_SERVICE_TABLE_GUID",
    ),
    UefiGuid(
        "3CDC90C6-13FB-4A75-9E7959E9DD78B9FA",
        name="EFI_PEI_READ_ONLY_VARIABLE_PPI_GUID",
    ),
    UefiGuid(
        "4F89E208-E144-4804-9EC80F894F7E36D7",
        name="EFI_PEI_SECTION_EXTRACTION_PPI_GUID",
    ),
    UefiGuid(
        "7E1F0D85-04FF-4BB2-866A31A2996A48A8", name="EFI_PEI_FV_FILE_LOADER_PPI_GUID"
    ),
    UefiGuid("36164812-A023-44E5-BD8505BF3C7700AA", name="EFI_FIND_FV_PPI_GUID"),
    UefiGuid("4426CCB2-E684-4A8A-AE4020D4B025B710", name="EFI_PEI_S3_RESUME_PPI_GUID"),
    UefiGuid(
        "125F2DE1-FB85-440C-A54C4D99358A8D38", name="EFI_ACPI_S3_SAVE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "DBFF9D55-89B7-46DA-BDDF677D3DC0241D", name="EFI_ACPI_SUPPORT_PROTOCOL_GUID"
    ),
    UefiGuid(
        "470E1529-B79E-4E32-A0FE6A156D29F9B2", name="EFI_BOOT_SCRIPT_SAVE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0FC9013A-0568-4BA9-9B7EC9C390A6609B", name="EFI_LEGACY_REGION_PROTOCOL_GUID"
    ),
    UefiGuid("B0732526-38C8-4B40-887761C7B06AAC45", name="EFI_CPU_IO_PROTOCOL_GUID"),
    UefiGuid("AE80D021-618E-11D4-BCD70080C73C8881", name="EFI_DATA_HUB_PROTOCOL_GUID"),
    UefiGuid(
        "448F5DA4-6DD7-4FE1-930769224192215D",
        name="EFI_SECTION_EXTRACTION_PROTOCOL_GUID",
    ),
    UefiGuid("D7AD636E-B997-459B-BF3F8846897980E1", name="EFI_HII_PROTOCOL_GUID"),
    UefiGuid(
        "5542CCE1-DF5C-4D1B-ABCA364F77D399FB",
        name="EFI_HII_COMPATIBILITY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "F33261E7-23CB-11D5-BD5C0080C73C8881",
        name="FRAMEWORK_EFI_MP_SERVICE_PROTOCOL_GUID",
    ),
    UefiGuid("1390954D-DA95-4227-93287282C217DAA8", name="EFI_SMM_BASE_PROTOCOL_GUID"),
    UefiGuid(
        "3792095A-E309-4C1E-AA0185F5655A17F1", name="EFI_SMM_ACCESS_PROTOCOL_GUID"
    ),
    UefiGuid(
        "8D12E231-C667-4FD1-98F22449A7E7B2E5", name="EFI_SMM_CONTROL_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E541B773-DD11-420C-B026DF993653F8BF", name="EFI_SMM_SW_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "14FC52BE-01DC-426C-91AEA23C3E220AE8", name="EFI_SMM_SX_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "9CCA03FC-4C9E-4A19-9B06ED7B479BDE55",
        name="EFI_SMM_PERIODIC_TIMER_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A05B6FFD-87AF-4E42-95C96228B63CF3F3", name="EFI_SMM_USB_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E0744B81-9513-49CD-8CEAE9245E7039DA", name="EFI_SMM_GPI_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "78965B98-B0BF-449E-8B22D2914E498A98",
        name="EFI_SMM_STANDBY_BUTTON_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "B709EFA0-47A6-4B41-B93112ECE7A8EE56",
        name="EFI_SMM_POWER_BUTTON_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid(
        "C50B323E-9075-4F2A-AC8ED2596A1085CC",
        name="EFI_SMM_ICHN_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid("5F439A0B-45D8-4682-A4F4F0576B513441", name="EFI_SMM_CPU_IO_GUID"),
    UefiGuid(
        "F3E4543D-CF35-6CEF-35C44FE6344DFC54", name="EFI_FORM_CALLBACK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "E5A1333E-E1B4-4D55-CEEB35C3EF133443", name="EFI_FORM_BROWSER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "0FB7C852-ADCA-4853-8D0FFBA71B1CE11A",
        name="EFI_FORM_BROWSER_COMPATIBILITY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "DE28BC59-6228-41BD-BDF6A3B9ADB58DA1",
        name="FRAMERWORK_EFI_FIRMWARE_VOLUME_BLOCK_PROTOCOL_GUID",
    ),
    UefiGuid(
        "21F302AD-6E94-471B-84BCB14800403A1D",
        name="EFI_SMM_CPU_SAVE_STATE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "834C0C5F-ADB3-4372-AEEB03E4E9E6C591", name="INTEL_FSP_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "BBCFF46C-C8D3-4113-8985B9D4F3B3F64E",
        name="FSP_BOOT_LOADER_TEMPORARY_MEMORY_GUID",
    ),
    UefiGuid(
        "D038747C-D00C-4980-B319490199A47D55",
        name="FSP_RESERVED_MEMORY_RESOURCE_HOB_TSEG_GUID",
    ),
    UefiGuid(
        "9C7C3AA7-5332-4917-82B956A5F3E62A07",
        name="FSP_RESERVED_MEMORY_RESOURCE_HOB_GFX_GUID",
    ),
    UefiGuid(
        "00D6B14B-7DD0-4062-8821E5F96A2A1B00",
        name="FSP_RESERVED_MEMORY_RESOURCE_HOB_MISC_GUID",
    ),
    UefiGuid(
        "2BC1C74A-122F-40B2-B223082B7465225D", name="FSP_WRAPPER_TOKEN_SPACE_GUID"
    ),
    UefiGuid("F5EF05E4-D538-4774-8F1BE9773011E038", name="FSP_INIT_DONE_PPI_GUID"),
    UefiGuid(
        "D8944553-C4DD-41F4-9B30E1397CFB267B", name="EFI_NIC_IP4_CONFIG_VARIABLE_GUID"
    ),
    UefiGuid("09D5B53F-F4B0-4F59-A0B17B57D35C0E05", name="NIC_IP4_CONFIG_NV_DATA_GUID"),
    UefiGuid("EB9D2D32-2D88-11D3-9A160090273FC14D", name="EFI_SAL_SYSTEM_TABLE_GUID"),
    UefiGuid(
        "B60DC6E8-3B6F-11D5-AF0900A0C944A05B", name="EFI_SAL_MCA_INIT_PMI_PROTOCOL_GUID"
    ),
    UefiGuid(
        "DE0EE9A4-3C7A-44F2-B78BE3CCD69C3AF7",
        name="EFI_EXTENDED_SAL_BOOT_SERVICE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "5AEA42B5-31E1-4515-BC31B8D5257565A6",
        name="EFI_EXTENDED_SAL_BASE_IO_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "53A58D06-AC27-4D8C-B5E9F08A80654170",
        name="EFI_EXTENDED_SAL_STALL_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7E97A470-EFDB-4D02-8FCE6190D27BA296",
        name="EFI_EXTENDED_SAL_RTC_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "4ECB6C53-C641-4370-8CB23B0E496E8378",
        name="EFI_EXTENDED_SAL_VARIABLE_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "899AFD18-75E8-408B-A41A6E2E7ECDF454",
        name="EFI_EXTENDED_SAL_MTC_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7D019990-8CE1-46F5-A7763C5198676AA0",
        name="EFI_EXTENDED_SAL_RESET_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "00DBD91D-55E9-420F-96395E9F8437B44F",
        name="EFI_EXTENDED_SAL_STATUS_CODE_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A2271DF1-BCBB-4F1D-98A906BC172F071A",
        name="EFI_EXTENDED_SAL_FV_BLOCK_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "697D81A2-CF18-4DC0-9E0D06113B618A3F",
        name="EFI_EXTENDED_SAL_MP_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "E1CD9D21-0FC2-438D-970304E66D961E57",
        name="EFI_EXTENDED_SAL_PAL_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "D9E9FA06-0FE0-41C3-96FB83425A3394F8",
        name="EFI_EXTENDED_SAL_BASE_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "2A591128-6CC7-42B1-8AF058933B682DBB",
        name="EFI_EXTENDED_SAL_MCA_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "A46B1A31-AD66-4905-92F62B4659DC3063",
        name="EFI_EXTENDED_SAL_PCI_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "0EDC9494-2743-4BA5-88180AEF5213F188",
        name="EFI_EXTENDED_SAL_CACHE_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "CB3FD86E-38A3-4C03-9A5C90CFA3A2AB7A",
        name="EFI_EXTENDED_SAL_MCA_LOG_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "CC664EB8-3C24-4086-B6F634E856BCE36E", name="EFI_WIN_NT_PASS_THROUGH_GUID"
    ),
    UefiGuid("BA73672C-A5D3-11D4-BD000080C73C8881", name="EFI_WIN_NT_CONSOLE_GUID"),
    UefiGuid("AB248E99-ABE1-11D4-BD0D0080C73C8881", name="EFI_WIN_NT_UGA_GUID"),
    UefiGuid("4E11E955-CCCA-11D4-BD0D0080C73C8881", name="EFI_WIN_NT_GOP_GUID"),
    UefiGuid("0C95A93D-A006-11D4-BCFA0080C73C8881", name="EFI_WIN_NT_SERIAL_PORT_GUID"),
    UefiGuid("0C95A935-A006-11D4-BCFA0080C73C8881", name="EFI_WIN_NT_FILE_SYSTEM_GUID"),
    UefiGuid(
        "0C95A92F-A006-11D4-BCFA0080C73C8881", name="EFI_WIN_NT_PHYSICAL_DISKS_GUID"
    ),
    UefiGuid(
        "0C95A928-A006-11D4-BCFA0080C73C8881", name="EFI_WIN_NT_VIRTUAL_DISKS_GUID"
    ),
    UefiGuid(
        "0D79A645-1D91-40A6-A81F61E6982B32B4", name="EFI_NT32_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "B347F047-AF8C-490E-AC070AA9B7E53858", name="EFI_WIN_NT_SYSTEM_CONFIG_GUID"
    ),
    UefiGuid("98C281E5-F906-43DD-A92BB003BF2765DA", name="PEI_NT_THUNK_PPI_GUID"),
    UefiGuid("0DCE384D-007C-4BA5-94BD0F6EB64D2AA9", name="PEI_NT_AUTO_SCAN_PPI_GUID"),
    UefiGuid("FD0C65EB-0405-4CD2-8AEEF400EF13BAC2", name="NT_PEI_LOAD_FILE_PPI_GUID"),
    UefiGuid("4E76928F-50AD-4334-B06BA84213108A57", name="NT_FWH_PPI_GUID"),
    UefiGuid("0419F582-0625-4531-8A3385A9965C95BC", name="WIN_NT_BUS_DRIVER_GUID"),
    UefiGuid(
        "58C518B1-76F3-11D4-BCEA0080C73C8881", name="EFI_WIN_NT_THUNK_PROTOCOL_GUID"
    ),
    UefiGuid("96EB4AD6-A32A-11D4-BCFD0080C73C8881", name="EFI_WIN_NT_IO_PROTOCOL_GUID"),
    UefiGuid("24B09ABE-4E47-481C-A9ADCEF12C392327", name="OMAP35XX_TOKEN_SPACE_GUID"),
    UefiGuid(
        "1E43298F-3478-41A7-B57786064635C728", name="OPTION_ROM_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("6B558CE3-69E5-4C67-A634F7FE72ADBE84", name="BLOCK_MMIO_PROTOCOL_GUID"),
    UefiGuid(
        "669346EF-FDAD-4AEB-08A621462D3FEF7D", name="PERFORMANCE_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid("DBA6A7E3-BB57-4BE7-8AF8D578DB7E5687", name="EFI_TSC_FREQUENCY_GUID"),
    UefiGuid("EB9D2D30-2D88-11D3-9A160090273FC14D", name="ACPI10_TABLE_GUID"),
    UefiGuid("EB9D2D30-2D88-11D3-9A160090273FC14D", name="ACPI_TABLE_GUID"),
    UefiGuid("5CAC0099-0DC9-48E5-8068BB95F5400A9F", name="PEI_CAPSULE_ON_DATA_CD_GUID"),
    UefiGuid(
        "2E3D2E75-9B2E-412D-B4B170416B8700FF",
        name="PEI_CAPSULE_ON_FAT_FLOPPY_DISK_GUID",
    ),
    UefiGuid(
        "B38573B6-6200-4AC5-B51D82E65938D783", name="PEI_CAPSULE_ON_FAT_IDE_DISK_GUID"
    ),
    UefiGuid(
        "0FFBCE19-324C-4690-A00998C6AE2EB186", name="PEI_CAPSULE_ON_FAT_USB_DISK_GUID"
    ),
    UefiGuid(
        "31A4878F-230D-4FC2-90BBDA5F41A44B1B", name="DXE_DEBUGPORT_EXT_PROTOCOL_GUID"
    ),
    UefiGuid("CA3B3A50-5698-4551-8B18CEAEEF917D50", name="DXE_DEBUG_CMD_PROTOCOL_GUID"),
    UefiGuid(
        "301AF449-E0D7-43A3-8B1BBC16725D374B", name="DXE_DEBUG_DISPATCH_PROTOCOL_GUID"
    ),
    UefiGuid(
        "CC93A70B-EC27-49C5-8B3413931EFED6E2",
        name="EFI_ALERT_STANDARD_FORMAT_PROTOCOL_GUID",
    ),
    UefiGuid("EBA4E8D2-3858-41EC-A2812647BA9660D0", name="EFI_DEBUGPORT_PROTOCOL_GUID"),
    UefiGuid(
        "F33261E7-23CB-11D5-BD5C0080C73C8881", name="EFI_MP_SERVICES_PROTOCOL_GUID"
    ),
    UefiGuid("D8117CFF-94A6-11D4-9A3A0090273FC14D", name="EFI_PEI_PE_COFF_LOADER_GUID"),
    UefiGuid(
        "387477C1-69C7-11D2-8E3900A0C969723B",
        name="EFI_SIMPLE_TEXT_INPUT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "387477C2-69C7-11D2-8E3900A0C969723B",
        name="EFI_SIMPLE_TEXT_OUTPUT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "7A28436A-E113-406A-AFF90DBF7F643E02", name="EFI_TCG_PRIVATE_INTERFACE_GUID"
    ),
    UefiGuid("EB9D2D2F-2D88-11D3-9A160090273FC14D", name="MPS_TABLE_GUID"),
    UefiGuid("5F65DFDB-AC20-4F43-9B21269E81A0EA7E", name="PDM_WINDOW_BUTTON_GUID"),
    UefiGuid("60B27E1A-D65E-4DB0-B2BBC16FA71E44AB", name="PDM_WINDOW_SPIN_GUID"),
    UefiGuid("C153205A-E898-4C24-8689A4B4BCC5C8A2", name="PEI_CACHE_PPI_GUID"),
    UefiGuid("C9737920-C2AD-41C3-B1330F9C251B6743", name="PEI_DEBUG_DISPATCH_PPI_GUID"),
    UefiGuid(
        "88C9D306-0900-4EB5-82603E2DBEDA1F89",
        name="PEI_POST_BOOT_SCRIPT_TABLE_PPI_GUID",
    ),
    UefiGuid("63B6E435-32BC-49C6-81BDB7A1A0FE1A6C", name="PEI_SMBUS_POLICY_PPI_GUID"),
    UefiGuid(
        "87DD3539-0667-4BB7-9FB247D3C50B021B",
        name="PHOENIX_DESKTOP_MESSAGE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "6FFEC78D-7DBF-4F8B-968D2F43A080C13F",
        name="PHOENIX_DESKTOP_WINDOW_PROTOCOL_GUID",
    ),
    UefiGuid("3F5A74B2-BF80-4850-85914F2DF5F9CB2B", name="PHOENIX_GDI_PROTOCOL_GUID"),
    UefiGuid(
        "177E63C1-AED6-4D88-917E9B4FB9FE4092",
        name="PHOENIX_SMM_CORE_SERVICES_PROTOCOL_GUID",
    ),
    UefiGuid(
        "FF052503-1AF9-4AEB-83C4C2D4CEB10CA3",
        name="PHOENIX_EFI_SMM_SW_SMI_PROTOCOL_GUID",
    ),
    UefiGuid(
        "6FDE3AF8-37AD-43FC-B728F4D341F39240",
        name="PHOENIX_SMM_WATCHDOG_TIMER_PROTOCOL_GUID",
    ),
    UefiGuid("EB9D2D32-2D88-11D3-9A160090273FC14D", name="SAL_SYSTEM_TABLE_GUID"),
    UefiGuid("8D48BD70-C8A3-4C06-901B747946AAC358", name="SCT_ISA_HC_PPI_GUID"),
    UefiGuid("5FAC7360-D4F2-4C9E-A7EDF2595E670889", name="SCT_ISA_HC_PROTOCOL_GUID"),
    UefiGuid("EB9D2D31-2D88-11D3-9A160090273FC14D", name="SMBIOS_TABLE_GUID"),
    UefiGuid(
        "30AC275E-BB30-4B84-A1CD0AF1322C89C0", name="PEI_SPEAKER_INTERFACE_PPI_GUID"
    ),
    UefiGuid(
        "AB294A92-EAF5-4CF3-AB2B2D4BED4DB63D", name="PEI_MFG_MEMORY_TEST_PPI_GUID"
    ),
    UefiGuid("950E191B-8524-4F51-80A15C4F1B03F35C", name="PEI_SHA256_HASH_PPI_GUID"),
    UefiGuid("CE845704-1683-4D38-A4F97D0B50775793", name="EFI_PLATFORM_BOOT_MODE_GUID"),
    UefiGuid("1E2ACC41-E26A-483D-AFC7A056C34E087B", name="EFI_PLATFORM_INFO_GUID"),
    UefiGuid("80DBD530-B74C-4F11-8C03418665532831", name="EFI_MEMORY_CONFIG_DATA_GUID"),
    UefiGuid(
        "69D13BF0-AF91-4D96-AA9F2184C5CE3BC0", name="PLATFORM_MODULE_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "5FAD2389-2BC7-4BD2-83D3429FB6AEA33F", name="EFI_SERIAL_PORT_TOKEN_SPACE_GUID"
    ),
    UefiGuid("E38C11E3-968F-47B8-ACEFACC0693DB9FF", name="EFI_ICH_TOKEN_SPACE_GUID"),
    UefiGuid("89A1B278-A1A1-4DF7-B137DE5AD7C47913", name="EFI_PCH_TOKEN_SPACE_GUID"),
    UefiGuid("560BF58A-1E0D-4D7E-953F2980A261E031", name="EFI_SIO_VARIABLE_GUID"),
    UefiGuid("1BF06AEA-5BEC-4A8D-9576749B09562D30", name="PROCESSOR_PRODUCER_GUID"),
    UefiGuid("0468A601-C535-46FD-A95DBBAB991B178C", name="EFI_POWER_ON_HOB_GUID"),
    UefiGuid("BB9C7AB7-B8D9-4BF3-9C299BF341E217BC", name="EFI_PLATFORM_CPU_INFO_GUID"),
    UefiGuid("C3E36D09-8294-4B97-A857D5288FE33E28", name="EFI_BIOS_ID_GUID"),
    UefiGuid("94B9E8AE-8877-479A-9842F5974B82CED3", name="EFI_BOARD_FEATURES_GUID"),
    UefiGuid("3812723D-7E48-4E29-BC27F5A39AC94EF1", name="ITK_DATA_VAR_GUID"),
    UefiGuid("70E56C5E-280C-44B0-A49709681ABC375E", name="DMI_DATA_GUID"),
    UefiGuid("788E1D9F-1EAB-47D2-A2F378CAE87D6012", name="IDCC_DATA_HUB_GUID"),
    UefiGuid("EC87D643-EBA4-4BB5-A1E53F3E36B20DA9", name="EFI_SETUP_VARIABLE_GUID"),
    UefiGuid("EF14FD78-0793-4E2B-AC6D062847E01791", name="MFG_MODE_VARIABLE_GUID"),
    UefiGuid("7E374E25-8E01-4FEE-87F2390C23C606CD", name="EFI_ACPI_TABLE_STORAGE_GUID"),
    UefiGuid(
        "72234213-0FD7-48A1-A59FB41BC107FBCD",
        name="A_C_P_I_O_S_F_R_MFG_STRING_VARIABLE_GUID",
    ),
    UefiGuid(
        "72234213-0FD7-48A1-A59FB41BC107FBCD",
        name="A_C_P_I_O_S_F_R_REF_DATA_BLOCK_VARIABLE_GUID",
    ),
    UefiGuid(
        "CA1BCAD9-E021-4547-A1B05B22C7F687F4",
        name="A_C_P_I_O_S_F_R_MODEL_STRING_VARIABLE_GUID",
    ),
    UefiGuid("0D9A1427-E02A-437D-926BAA521FD722BA", name="EFI_PCI_LAN_INFO_GUID"),
    UefiGuid("EC87D643-EBA4-4BB5-A1E53F3E36B20DA9", name="EFI_NORMAL_SETUP_GUID"),
    UefiGuid("5E559C23-1FAA-4AE1-8D4AC6CF026C766F", name="FIRMWARE_ID_GUID"),
    UefiGuid("86843F56-675D-40A5-9530BC858372F103", name="OS_SELECTION_VARIABLE_GUID"),
    UefiGuid(
        "EBBE2D1B-1647-4BDA-AB9A7863E396D41A", name="EFI_ACTIVE_BIOS_PROTOCOL_GUID"
    ),
    UefiGuid(
        "BD26CDC9-A092-462A-877A5AB6ADCE4812", name="EFI_PLATFORM_CPU_PROTOCOL_GUID"
    ),
    UefiGuid(
        "4B0165A9-61D6-4E23-A0B53EC79C2E30D5",
        name="DXE_PCH_PLATFORM_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid(
        "DE161CFE-1E60-42A1-8CC3EE7EF0735212", name="EFI_TPM_MP_DRIVER_PROTOCOL_GUID"
    ),
    UefiGuid(
        "AB2BEE2F-C1A6-4399-853DC07C774FFD0D",
        name="EFI_LPC_WPCE791_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid("F617B358-12CF-414A-A06960677BDA13B4", name="USB_POLICY_GUID"),
    UefiGuid(
        "400B4476-3081-11D6-87ED00062945C3B9",
        name="EFI_SPEAKER_INTERFACE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "5BAB88BA-E0E2-4674-B6ADB812F6881CD6", name="DXE_VLV_PLATFORM_POLICY_GUID"
    ),
    UefiGuid(
        "EF7BF7D6-F8FF-4A76-8247C0D0D1CC49C0", name="EFI_SMBIOS_SLOT_POPULATION_GUID"
    ),
    UefiGuid("E227C522-D5FE-4A53-87B10FBE570F98E9", name="OBSERVABLE_PROTOCOL_GUID"),
    UefiGuid(
        "3C485EA4-449A-46CE-BB082A336EA96B4E", name="EFI_CK505_CLOCK_PLATFORM_INFO_GUID"
    ),
    UefiGuid(
        "D3ECC567-9FD5-44C1-86CF5DA7A24F4B5D",
        name="EFI_LPC_WPC83627_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid("A6A79162-E325-4C30-BCC359373064EFB3", name="EFI_TCO_RESET_PROTOCOL_GUID"),
    UefiGuid(
        "D5B06D16-2EA1-4DEF-98D0A05D40728417",
        name="EFI_WATCHDOG_TIMER_DRIVER_PROTOCOL_GUID",
    ),
    UefiGuid(
        "377C66A3-8FE7-4EE8-85B8F1A282569E3B",
        name="EFI_PLATFORM_IDE_INIT_PROTOCOL_GUID",
    ),
    UefiGuid(
        "91A1DDCF-5374-4939-8951D7293F1A786F", name="ENHANCED_SPEEDSTEP_PROTOCOL_GUID"
    ),
    UefiGuid(
        "6FF23F1D-877C-4B1B-93FCF142B2EEA6A7",
        name="EFI_GOP_DISPLAY_BRIGHTNESS_PROTOCOL_GUID",
    ),
    UefiGuid(
        "AD9C4381-1EDE-430C-8D4223767C465D52", name="EFI_USB_KEYBOARD_CONNECT_GUID"
    ),
    UefiGuid(
        "24486226-F8C2-41F5-B9DD783E9E56DEA0", name="EFI_MMIO_DEVICE_PROTOCOL_GUID"
    ),
    UefiGuid(
        "75032015-D156-423E-BFA37A65ABA47105",
        name="EFI_I2C_BUS_CONFIGURATION_MANAGEMENT_PROTOCOL2_GUID",
    ),
    UefiGuid("F30C2915-5782-4E6A-A84605BABCE7B6A0", name="EFI_I2C_ACPI_PROTOCOL_GUID"),
    UefiGuid(
        "578C315A-68CF-4E81-B5C622DB40D010BC", name="EFI_I2C_MASTER_PROTOCOL2_GUID"
    ),
    UefiGuid("70B221AF-FDFF-4FDE-99681AF623A956D9", name="EFI_I2C_HOST_PROTOCOL2_GUID"),
    UefiGuid("9FA1B225-3346-461B-A069ED01B673D240", name="EFI_I2C_BUS_PROTOCOL_GUID"),
    UefiGuid("F2C1910E-F5C9-4B72-B2436D59096A79F0", name="EFI_I2C_SLAVE_PROTOCOL_GUID"),
    UefiGuid("9F49A879-3D71-42B3-A0ADDDB1F33010A3", name="EFI_SPI_ACPI_PROTOCOL_GUID"),
    UefiGuid("951B65E5-8872-41ED-AD1DD5681F4AF033", name="EFI_SPI_HOST_PROTOCOL_GUID"),
    UefiGuid("137B3044-F6D7-473E-A6259FB92505C180", name="EFI_SPI_BUS_PROTOCOL_GUID"),
    UefiGuid("AF4CC162-D41C-455A-AB456DBCC1CD32F3", name="LPSS_DUMMY_PROTOCOL_GUID"),
    UefiGuid("1156EFC6-EA32-4396-B5D526932E83C313", name="EFI_SPI_PROTOCOL_GUID"),
    UefiGuid(
        "38DDFE8F-8991-44AA-988983F4918465B0", name="EFI_GPIO_OPERATION_PROTOCOL_GUID"
    ),
    UefiGuid(
        "4549AB47-6E60-4293-B91D31B610AD8056", name="EFI_ESRT_OPERATION_PROTOCOL_GUID"
    ),
    UefiGuid("3B8C8162-188C-46A4-AEC9BE43F1D65697", name="EFI_FW_DISPLAY_CAPSULE_GUID"),
    UefiGuid("B122A262-3551-4F48-889255F6C0614290", name="EFI_FIRMWARE_CLASS_GUID"),
    UefiGuid("0DC73AED-CBF6-4A25-A68D59C80F44C7C3", name="EFI_D_F_U_VER_GUID"),
    UefiGuid("B122A263-3661-4F68-992978F8B0D62180", name="EFI_ESRT_TABLE_GUID"),
    UefiGuid(
        "F3FF1468-04BA-4966-9FB2E4A790054650", name="EFI_CAPSULE_CRASH_LOG_VAR_GUID"
    ),
    UefiGuid("0E1D2972-65AF-4AC1-BFA3CEF4AB0C38FE", name="EFI_CAPSULE_CRASH_GUID"),
    UefiGuid(
        "78B9EC8B-C000-46C5-AC9324A0C1BB00CE", name="PWD_CREDENTIAL_PROVIDER_GUID"
    ),
    UefiGuid(
        "D0849ED1-A88C-4BA6-B1D6AB50E280B7A9", name="USB_CREDENTIAL_PROVIDER_GUID"
    ),
    UefiGuid("C35F272C-97C2-465A-A216696B668A8CFE", name="USER_PROFILE_MANAGER_GUID"),
    UefiGuid(
        "F24643C2-C622-494E-8A0D4632579C2D5B", name="EFI_TR_E_E_PHYSICAL_PRESENCE_GUID"
    ),
    UefiGuid("C54B425F-AA79-48B4-981F998B3C4B641C", name="TR_E_E_CONFIG_FORM_SET_GUID"),
    UefiGuid("447559F0-D02E-4CF1-99BCCA11654054C2", name="STD_LIB_TOKEN_SPACE_GUID"),
    UefiGuid("58E6ED63-1694-440B-9388E98FED6B65AF", name="EFI_SOCKET_PROTOCOL_GUID"),
    UefiGuid(
        "8AAEDB2A-A6BB-47C6-94CE1B8096423F2A",
        name="EFI_SOCKET_SERVICE_BINDING_PROTOCOL_GUID",
    ),
    UefiGuid("7D84B2C2-22A1-4372-B12CEBB232D3A6A3", name="VLV_POLICY_PPI_GUID"),
    UefiGuid("E767BF7F-4DB6-5B34-10114FBE4CA7AFD2", name="VLV_MMIO_POLICY_PPI_GUID"),
    UefiGuid("10E26DF1-8775-4EE1-B50A3AE82893703A", name="SE_CF_T_P_M_PPI_GUID"),
    UefiGuid("C02B0573-2B4E-4A31-A31A94567B50442C", name="PCH_USB_POLICY_PPI_GUID"),
    UefiGuid("09EA894A-BE0D-4230-A003EDC693B48E95", name="PCH_INIT_PPI_GUID"),
    UefiGuid(
        "15344673-D365-4BE2-85131497CC07611D", name="PCH_PLATFORM_POLICY_PPI_GUID"
    ),
    UefiGuid("A38C6898-2B5C-4FF6-93262E63212E56C2", name="PEI_SPI_PPI_GUID"),
    UefiGuid("09EA8911-BE0D-4230-A003EDC693B48E11", name="VLV_PEI_INIT_PPI_GUID"),
    UefiGuid("CBD86677-362F-4C04-9459A741326E05CF", name="SE_C_UMA_PPI_GUID"),
    UefiGuid(
        "7AE3CEB7-2EE2-48FA-AA493510BC83CABF", name="PEI_SE_C_PLATFORM_POLICY_PPI_GUID"
    ),
    UefiGuid("EE0EA811-FBD9-4777-B95ABA4F71101F74", name="PEI_HECI_PPI_GUID"),
    UefiGuid("F4EF9D7A-98C5-4C1A-B4D9D8D87265BE0C", name="PEI_SDHC_PPI_GUID"),
    UefiGuid("BC5FA650-EDBB-4D0D-B3A3D98907F847DF", name="PEI_BLOCK_IO_PPI_GUID"),
    UefiGuid("4FD1BA49-8F90-471A-A2C9173C7A732FD0", name="SE_CF_T_P_M_POLICY_PPI_GUID"),
    UefiGuid("ACB93B08-5CDC-4A8F-93D406E342DF182E", name="PCH_PEI_INIT_PPI_GUID"),
    UefiGuid(
        "074E1E48-8132-47A1-8C2C3F14AD9A66DC", name="EFI_GLOBAL_NVS_AREA_PROTOCOL_GUID"
    ),
    UefiGuid(
        "DDABFEAC-EF63-452C-8F39ED7FAED8265E", name="PPM_PLATFORM_POLICY_PROTOCOL_GUID"
    ),
    UefiGuid("6F20F7C8-E5EF-4F21-8D19EDC5F0C496AE", name="MEM_INFO_PROTOCOL_GUID"),
    UefiGuid(
        "B63F8EC7-A9C9-4472-A4C04D8BF365CC51", name="EFI_SD_HOST_IO_PROTOCOL_GUID"
    ),
    UefiGuid("D9072C35-EB8F-43AD-A22034D40E2A8285", name="EFI_SMM_SPI_PROTOCOL_GUID"),
    UefiGuid(
        "3920405B-C897-44DA-88F34C498A6FF736",
        name="EFI_SMM_ICHN_DISPATCH_EX_PROTOCOL_GUID",
    ),
    UefiGuid(
        "E287D20B-D897-4E1E-A5D9977763936A04", name="EFI_PCH_S3_SUPPORT_PROTOCOL_GUID"
    ),
    UefiGuid("DB63592C-B8CC-44C8-918C51F534598A5A", name="PCH_RESET_PROTOCOL_GUID"),
    UefiGuid(
        "3A3300AB-C929-487D-AB34159BC13562C0", name="PCH_RESET_CALLBACK_PROTOCOL_GUID"
    ),
    UefiGuid("D31F0400-7D16-4316-BF886065883B402B", name="EFI_PCH_INFO_PROTOCOL_GUID"),
    UefiGuid(
        "F0BBFCA0-684E-48B3-BAE26C84B89E5339",
        name="EFI_PCH_EXTENDED_RESET_PROTOCOL_GUID",
    ),
    UefiGuid(
        "F617B358-12CF-414A-A06960677BDA13B3",
        name="DXE_ICH_PLATFORM_POLICY_PROTOCOL_GUID",
    ),
    UefiGuid("D31F0400-7D16-4316-BF886065883B402B", name="EFI_ICH_INFO_PROTOCOL_GUID"),
    UefiGuid(
        "58DC368D-7BFA-4E77-ABBC0E29418DF930",
        name="EFI_SMM_IO_TRAP_DISPATCH_PROTOCOL_GUID",
    ),
    UefiGuid("72E40094-2EE1-497A-8F334C934A9E9C0C", name="EFI_SMM_SMBUS_PROTOCOL_GUID"),
    UefiGuid("CDC5DDDF-E79D-41EC-A9B06565490DB9D3", name="IGD_OP_REGION_PROTOCOL_GUID"),
    UefiGuid("CFB33810-6E87-4284-B203A66ABE07F6E8", name="EFI_HECI_PROTOCOL_GUID"),
    UefiGuid(
        "BC52476E-F67E-4301-B262369C4878AAC2", name="PLATFORM_SE_C_HOOK_PROTOCOL_GUID"
    ),
    UefiGuid(
        "11FBFDFB-10D2-43E6-B5B1B4386EDCCB9A", name="EFI_SE_C_RC_INFO_PROTOCOL_GUID"
    ),
    UefiGuid("0BF70067-D53B-42DF-B770E92C91C61411", name="EFI_TDT_PROTOCOL_GUID"),
    UefiGuid(
        "F8BFF014-18FB-4EF9-B10CAE22738DBEED", name="DXE_PLATFORM_SE_C_POLICY_GUID"
    ),
    UefiGuid(
        "1EBE5AB9-2129-49E7-84D7EEB9FCE5DEDD", name="EFI_EMMC_CARD_INFO_PROTOCOL_GUID"
    ),
    UefiGuid(
        "FD301BA4-5E62-4679-A06FE09AABDD2A91", name="EFI_TDT_OPERATION_PROTOCOL_GUID"
    ),
    UefiGuid("14A7C46F-BC02-4047-9F18A5D725D8BD19", name="EFI_D_F_U_RESULT_GUID"),
    UefiGuid("918211CE-A1D2-43A0-A04E75B5BF44500E", name="EFI_C_P_TOKEN_SPACE_GUID"),
    UefiGuid("707BE83E-0BF6-40A5-BE6434C03AA0B8E2", name="EFI_SMBUS_ARP_MAP_GUID"),
    UefiGuid("CA452C68-DF0C-45C9-82FBEAE42B312946", name="EFI_V_L_V_TOKEN_SPACE_GUID"),
    UefiGuid("BB929DA9-68F7-4035-B22CA3BB3F23DA55", name="SATA_CONTROLLER_DRIVER_GUID"),
    UefiGuid(
        "1A819E49-D8EE-48CB-9A9C0AA0D2810A38",
        name="DXE_PCH_POLICY_UPDATE_PROTOCOL_GUID",
    ),
    UefiGuid(
        "161BE597-E9C5-49DB-AE50C462AB54EEDA",
        name="POWER_MANAGEMENT_ACPI_TABLE_STORAGE_GUID",
    ),
    UefiGuid("E6C2F70A-B604-4877-85BADEEC89E117EB", name="PCH_INIT_VARIABLE_GUID"),
    UefiGuid(
        "85768E4A-6CDC-444E-93DF936685B5DFCC", name="VLV_REF_CODE_PKG_TOKEN_SPACE_GUID"
    ),
    UefiGuid(
        "03FDF171-1D67-4ACE-A9043E36D338FA74", name="SE_C_PLATFORM_READY_TO_BOOT_GUID"
    ),
    UefiGuid("40B09B5A-F0EF-4627-93D527F04B754D05", name="AMT_READY_TO_BOOT_GUID"),
    UefiGuid("10BA6BBE-A97E-41C3-9A07607AD9BD60E5", name="EFI_VLV2_VARIABLE_GUID"),
]

GUID_FROM_VALUE: Dict[str, UefiGuid] = dict(
    [(guid.value, guid) for guid in PROTOCOLS_GUIDS]
)
GUID_FROM_BYTES: Dict[bytes, UefiGuid] = dict(
    [(guid.bytes, guid) for guid in PROTOCOLS_GUIDS]
)
