"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack', { env });
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    tasks: [
        {
            listenerPort: 80,
            task: orderTask,
            desiredCount: 2,
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        { listenerPort: 8080, task: customerTask, desiredCount: 2 },
        { listenerPort: 9090, task: productTask, desiredCount: 2 },
    ],
    route53Ops: {
        zoneName,
        externalAlbRecordName,
        internalAlbRecordName,
    },
});
//# sourceMappingURL=data:application/json;base64,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