/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.prec;

import java.util.Arrays;

import mt.DenseVector;
import mt.Matrix;
import mt.Vector;
import smt.SparseVector;

/**
 * ILU preconditioner. Performs an incomplete LU decomposition as a
 * preconditioner. Applicable to unsymmetrical problems.
 */
public class ILU extends DecompositionPreconditioner {

    /**
     * Constructor for ILU
     * 
     * @param A
     *            Matrix to decompose. Not modified
     * @param initShift
     *            Initial diagonal shift on the factor matrix
     * @param shift
     *            Progressive diagonal shifts used every time a zero pivot is
     *            found
     */
    public ILU(Matrix A, double initShift, double shift) {
        super(A, initShift, shift);
    }

    /**
     * Constructor for ILU. Uses no initial shift, and a progressive shift of
     * <code>0.1</code>
     * 
     * @param A
     *            Matrix to decompose. Not modified
     */
    public ILU(Matrix A) {
        super(A);
    }

    public Vector apply(Vector b, Vector x) {
        if (!(x instanceof DenseVector))
            throw new IllegalArgumentException("x must be a DenseVector");

        // Copy b into x, and get the array
        x.set(b);
        double[] xdata = ((DenseVector) x).getData();

        // Solve L*U
        solveL(xdata, false);
        solveU(xdata, true);

        return x;
    }

    public Vector transApply(Vector b, Vector x) {
        if (!(x instanceof DenseVector))
            throw new IllegalArgumentException("x must be a DenseVector");

        // Copy b into x, and get the array
        x.set(b);
        double[] xdata = ((DenseVector) x).getData();

        // Solve U'*L'
        solveUT(xdata, true);
        solveLT(xdata, false);

        return x;
    }

    void factor(Matrix A) {
        double diagMod = initShift;
        int n = A.numRows();
        diagInd = new int[n];
        DenseVector curRow = new DenseVector(n);

        outer: while (true) {

            // Copy A into F, and ensure non-zero diagonal
            F.set(A).addDiagonal(diagMod);

            // Clear cache of diagonal indices
            Arrays.fill(diagInd, -1);
            if (n > 0)
                getDiagInd(0);

            // Row-based Gaussian elimination
            for (int i = 1; i < n; ++i) {

                // Get current row
                SparseVector curRowI = F.getRow(i);
                int[] curInd = curRowI.getIndex();
                int length = curRowI.used();

                // Expand current row into a dense vector
                scatter(curRow, curRowI);
                double[] curDense = curRow.getData();

                // Traverse all active rows
                for (int j = 0; j < length && curInd[j] < i; ++j) {

                    SparseVector actRow = F.getRow(curInd[j]);
                    int actLength = actRow.used();
                    int[] actInd = actRow.getIndex();
                    double[] actDat = actRow.getData();

                    // Check for missing or zero diagonal
                    int diag = getDiagInd(curInd[j], actInd, actLength);
                    if (diag < 0 || actDat[diag] == 0) {
                        diagMod += shift;
                        continue outer;
                    }

                    // Get multiplier (to be stored in L)
                    double mult = -curDense[curInd[j]] / actDat[diag];
                    curDense[curInd[j]] = -mult;

                    // Reduce everything on the upper diagonal of current row
                    for (int k = diag + 1; k < actLength; ++k)
                        curDense[actInd[k]] += mult * actDat[k];
                }

                // Retain only the original non-zero pattern
                gather(curRowI, curRow);
            }

            // Compress the matrix, and ensure that all diagonals exists with
            // correct indices
            F.compact();
            Arrays.fill(diagInd, -1);
            for (int i = 0; i < n; ++i) {
                getDiagInd(i);
                if (diagInd[i] < 0) {
                    diagMod += shift;
                    continue outer;
                }
            }

            // No errors detected
            break;
        }
    }

}
