/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.mixed;

import mt.AbstractMatrix;
import mt.Matrix;
import mt.Vector;
import smt.iter.CG;
import smt.iter.DefaultIterationMonitor;
import smt.iter.IterationMonitor;
import smt.iter.IterativeSolver;
import smt.iter.IterativeSolverNotConvergedException;

/**
 * Partial implementation of MixedSolver
 */
public abstract class AbstractMixedSolver implements MixedSolver {

    /**
     * Iterative solver to use
     */
    protected IterativeSolver solver;

    /**
     * Convergence tester
     */
    protected IterationMonitor iter;

    /**
     * Constructor for AbstractMixedSolver, with CG as linear solver and default
     * iteration monitor.
     * 
     * @param qTemplate
     *            Vector of size <i>m </i>, used as template in preallocating
     *            work vectors
     */
    public AbstractMixedSolver(Vector qTemplate) {
        iter = new DefaultIterationMonitor();
        solver = new CG(qTemplate);
    }

    public void setSubSolver(IterativeSolver solver) {
        this.solver = solver;
    }

    public IterativeSolver getSubSolver() {
        return solver;
    }

    public IterationMonitor getIterationMonitor() {
        return iter;
    }

    public void setIterationMonitor(IterationMonitor iter) {
        this.iter = iter;
    }

    public void solve(Matrix A, Matrix B, Matrix Bt, Matrix C, Vector q,
            Vector u, Vector f, Vector g)
            throws IterativeSolverNotConvergedException {
        checkSolveArguments(A, B, Bt, C, q, u, f, g);
        solveI(A, B, Bt, C, q, u, f, g);
    }

    public void solve(Matrix A, Matrix B, Matrix C, Vector q, Vector u,
            Vector f, Vector g) throws IterativeSolverNotConvergedException {
        solve(A, B, new TransposedMatrix(B), C, q, u, f, g);
    }

    /**
     * Checks that all sizes conform, or else throws an exception.
     */
    private void checkSolveArguments(Matrix A, Matrix B, Matrix Bt, Matrix C,
            Vector q, Vector u, Vector f, Vector g) {

        int m = B.numRows();
        int n = B.numColumns();

        if (n != A.numColumns())
            throw new IllegalArgumentException(
                    "B.numColumns() != A.numColumns()");
        if (q.size() != n)
            throw new IllegalArgumentException("q.size() != B.numColumns()");
        if (u.size() != m)
            throw new IllegalArgumentException("u.size() != B.numRows()");
        if (f.size() != n)
            throw new IllegalArgumentException("f.size() != B.numColumns()");
        if (g.size() != m)
            throw new IllegalArgumentException("g.size() != B.numRows()");
        if (Bt.numRows() != n || Bt.numColumns() != m)
            throw new IllegalArgumentException(
                    "Bt.numRows() != B.numColumns() || Bt.numColumns() != B.numRows()");
        if (!A.isSquare())
            throw new IllegalArgumentException("!A.isSquare()");
        if (!C.isSquare())
            throw new IllegalArgumentException("!C.isSquare()");
        if (C.numRows() != m)
            throw new IllegalArgumentException("C.numRows() != B.numRows()");
    }

    /**
     * Performs the actual solve after argument checking has been done
     */
    protected abstract void solveI(Matrix A, Matrix B, Matrix Bt, Matrix C,
            Vector q, Vector u, Vector f, Vector g)
            throws IterativeSolverNotConvergedException;

    /**
     * Changes matrix/vector products such that the matrix appears to be
     * transposed to an iterative solver
     */
    private class TransposedMatrix extends AbstractMatrix {

        private static final long serialVersionUID = 3760564187235168821L;

        private Matrix A;

        public TransposedMatrix(Matrix A) {
            super(A.numColumns(), A.numRows());
            this.A = A;
        }

        public Vector multAdd(double alpha, Vector x, double beta, Vector y,
                Vector z) {
            return A.transMultAdd(alpha, x, beta, y, z);
        }

        public Vector transMultAdd(double alpha, Vector x, double beta,
                Vector y, Vector z) {
            return A.multAdd(alpha, x, beta, y, z);
        }

    }

}
