/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.eig;

import mt.Matrix;
import mt.Vector;
import smt.iter.GMRES;
import smt.iter.IterativeSolver;
import smt.iter.IterativeSolverNotConvergedException;

/**
 * Performs both a shift and an inversion of the spectrum. The operator being
 * applied thus becomes <code>(A - sigma)<sup>-1</sup>x</code>
 */
public class ShiftInvertEigenvalueTransformation implements
        EigenvalueTransformation {

    /**
     * Current shift
     */
    protected double sigma;

    /**
     * Previous shift
     */
    protected double sigmaOld;

    /**
     * The shifted matrix used by the solver
     */
    protected Matrix Am;

    /**
     * The solver used for the inversion
     */
    protected IterativeSolver solver;

    /**
     * Constructor for ShiftInvertEigenvalueTransformation. The iterative solver
     * {@link smt.iter.GMRES GMRES}is used by default, but can be changed with
     * {@link #setSolver setSolver}.
     * 
     * @param A
     *            Matrix to apply transformation to. Used to create the initial
     *            shift/invert matrix by copying
     * @param template
     *            Vector to use as template for the work vectors needed in the
     *            underlying iterative solver
     */
    public ShiftInvertEigenvalueTransformation(Matrix A, Vector template) {
        Am = A.copy();
        solver = new GMRES(template);
    }

    public Vector apply(Matrix A, Vector x, Vector y) {
        // Undo the old shift, and add in the new shift
        Am.addDiagonal(sigmaOld - sigma);

        // Solve with the newly shifted matrix
        try {
            solver.solve(Am, x, y);
        } catch (IterativeSolverNotConvergedException e) {
            // Wrap the exception, pass it along
            throw new RuntimeException(e);
        }

        return y;
    }

    public void setShift(double sigma) {
        sigmaOld = this.sigma;
        this.sigma = sigma;
    }

    public double getShift() {
        return sigma;
    }

    public double eigenvalue(double e) {
        return sigma + 1. / e;
    }

    /**
     * Sets the iterative solver to use
     */
    public void setSolver(IterativeSolver solver) {
        this.solver = solver;
    }

    /**
     * Gets the current iterative solver
     */
    public IterativeSolver getSolver() {
        return solver;
    }

}
