/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.util;

/**
 * Generates linear indices
 */
public final class Index {

    private Index() {
        // No need to instantiate
    }

    /**
     * Creates a continuous linear index.
     * 
     * @param from
     *            Start, inclusive
     * @param to
     *            Stop, exclusive
     */
    public static int[] index(int from, int to) {
        int length = to - from;

        if (length < 0)
            length = 0;

        int[] index = new int[length];
        for (int i = from, j = 0; j < length; ++i, ++j)
            index[j] = i;
        return index;
    }

    /**
     * Creates a strided linear index.
     * 
     * @param from
     *            Start, inclusive
     * @param stride
     *            <code>stride=1</code> for continuous. Negative strides are
     *            allowed
     * @param to
     *            Stop, exclusive
     */
    public static int[] index(int from, int stride, int to) {
        if (stride == 1)
            return index(from, to);
        else if (stride == 0)
            return new int[0];

        if (to <= from && stride > 0)
            return new int[0];
        if (from <= to && stride < 0)
            return new int[0];

        int length = Math.abs((to - from) / stride);
        if (Math.abs((to - from) % stride) > 0)
            length++;

        if (length < 0)
            length = 0;

        int[] index = new int[length];
        for (int i = from, j = 0; j < length; i += stride, ++j)
            index[j] = i;
        return index;
    }

}
