/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.LowerSymmPackMatrix;
import mt.Matrix;
import mt.UpperSymmPackMatrix;
import mt.ll.BLASkernel.UpLo;
import mt.ll.Interface;
import mt.ll.LAPACKkernel.JobEig;

/**
 * Computes eigenvalues of symmetrical, packed matrices
 */
public class SymmPackEigenvalueComputer {

    /**
     * Double work array
     */
    private double[] work;

    /**
     * Integer work array
     */
    private int[] iwork;

    /**
     * Size of the matrix
     */
    private int n;

    /**
     * Upper or lower part stored
     */
    private UpLo uplo;

    /**
     * Job to do
     */
    private JobEig job;

    /**
     * Creates a new eigenvalue computer for symmetrical, packed matrices.
     * Computes all eigenvalues and eigenvectors
     * 
     * @param n
     *            Size of the matrix
     * @param uplo
     *            Lower or upper part stored?
     */
    public SymmPackEigenvalueComputer(int n, UpLo uplo) {
        this(n, uplo, JobEig.All);
    }

    /**
     * Creates a new eigenvalue computer for symmetrical, packed matrices
     * 
     * @param n
     *            Size of the matrix
     * @param uplo
     *            Lower or upper part stored?
     * @param job
     *            The type of computation to perform
     */
    public SymmPackEigenvalueComputer(int n, UpLo uplo, JobEig job) {
        this.n = n;
        this.uplo = uplo;
        this.job = job;

        // Find the needed workspace
        work = new double[1];
        iwork = new int[1];
        int info = Interface.lapack().spevd(job, uplo, n, new double[0],
                new double[0], new double[0], work, -1, iwork, -1);

        // Allocate workspace
        int lwork = 0, liwork = 0;
        if (info != 0) {
            if (job == JobEig.All) {
                lwork = 1 + 6 * n + n * n;
                liwork = 3 + 5 * n;
            } else {
                lwork = 2 * n;
                liwork = 1;
            }
        } else {
            lwork = (int) work[0];
            liwork = iwork[0];
        }
        lwork = Math.max(1, lwork);
        liwork = Math.max(1, liwork);
        work = new double[lwork];
        iwork = new int[liwork];
    }

    /**
     * Convience method for computing the full eigenvalue decomposition of the
     * given matrix
     * 
     * @param A
     *            Matrix to factorize. Upper part extracted, and the matrix is
     *            not modified
     * @return Newly allocated decomposition
     * @throws NotConvergedException
     */
    public static SymmEigenvalueDecomposition factorize(Matrix A)
            throws NotConvergedException {
        return new SymmPackEigenvalueComputer(A.numRows(), UpLo.Upper)
                .factor(new UpperSymmPackMatrix(A));
    }

    /**
     * Computes the eigenvalue decomposition of the given matrix
     * 
     * @param A
     *            Matrix to factorize. Overwritten on return
     * @return Newly allocated decomposition
     * @throws NotConvergedException
     */
    public SymmEigenvalueDecomposition factor(LowerSymmPackMatrix A)
            throws NotConvergedException {
        return factor(A, new SymmEigenvalueDecomposition(n, job));
    }

    /**
     * Computes the eigenvalue decomposition of the given matrix
     * 
     * @param A
     *            Matrix to factorize. Overwritten on return
     * @return Newly allocated decomposition
     * @throws NotConvergedException
     */
    public SymmEigenvalueDecomposition factor(UpperSymmPackMatrix A)
            throws NotConvergedException {
        return factor(A, new SymmEigenvalueDecomposition(n, job));
    }

    /**
     * Computes the eigenvalue decomposition of the given matrix
     * 
     * @param A
     *            Matrix to factorize. Overwritten on return
     * @param evd
     *            Factorization is written here
     * @return evd
     * @throws NotConvergedException
     */
    public SymmEigenvalueDecomposition factor(LowerSymmPackMatrix A,
            SymmEigenvalueDecomposition evd) throws NotConvergedException {
        if (uplo != UpLo.Lower)
            throw new IllegalArgumentException(
                    "Eigenvalue computer configured for lower-symmetrical matrices");

        return factor(A, A.getData(), evd);
    }

    /**
     * Computes the eigenvalue decomposition of the given matrix
     * 
     * @param A
     *            Matrix to factorize. Overwritten on return
     * @param evd
     *            Factorization is written here
     * @return evd
     * @throws NotConvergedException
     */
    public SymmEigenvalueDecomposition factor(UpperSymmPackMatrix A,
            SymmEigenvalueDecomposition evd) throws NotConvergedException {
        if (uplo != UpLo.Upper)
            throw new IllegalArgumentException(
                    "Eigenvalue computer configured for upper-symmetrical matrices");

        return factor(A, A.getData(), evd);
    }

    private SymmEigenvalueDecomposition factor(Matrix A, double[] data,
            SymmEigenvalueDecomposition evd) throws NotConvergedException {
        if (A.numRows() != n)
            throw new IllegalArgumentException("A.numRows() != n");
        else if (evd.getEigenvalues().length != n)
            throw new IllegalArgumentException(
                    "evd.getEigenvalues().length != n");
        else if (job == JobEig.All) {
            if (!evd.hasEigenvectors())
                throw new IllegalArgumentException(
                        "Eigenvalue decomposition does not store eigenvectors");
            else if (evd.getEigenvectors().numRows() != n)
                throw new IllegalArgumentException(
                        "evd.getEigenvectors().numRows() != n");
            else if (evd.getEigenvectors().numColumns() != n)
                throw new IllegalArgumentException(
                        "evd.getEigenvectors().numColumns() != n");
        }

        int info = Interface.lapack()
                .spevd(
                        job,
                        uplo,
                        n,
                        data,
                        evd.getEigenvalues(),
                        job == JobEig.All ? evd.getEigenvectors().getData()
                                : new double[0], work, work.length, iwork,
                        iwork.length);

        if (info > 0)
            throw new NotConvergedException(
                    NotConvergedException.Reason.Iterations);
        else if (info < 0)
            throw new IllegalArgumentException();

        return evd;
    }

}
