/*
 * Copyright (C) 2003-2005 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import java.io.Serializable;

import mt.DenseMatrix;
import mt.LowerTriangDenseMatrix;
import mt.UpperTriangDenseMatrix;

/**
 * Orthogonal matrix decomposition
 */
public class OrthogonalDecomposition implements Serializable {

    private static final long serialVersionUID = -1973650562208779301L;

    /**
     * The orthogonal matrix
     */
    private DenseMatrix Q;

    /**
     * Lower triangular factor. May not be present
     */
    private LowerTriangDenseMatrix L;

    /**
     * Upper triangular factor. May not be present
     */
    private UpperTriangDenseMatrix R;

    /**
     * Allocates storage for a new orthogonal decomposition.
     * 
     * @param m
     *            Number of rows
     * @param n
     *            Number of columns
     * @param upper
     *            True for storing an upper triangular factor, false for a lower
     *            triangular factor
     */
    public OrthogonalDecomposition(int m, int n, boolean upper) {
        Q = new DenseMatrix(m, n);
        if (upper)
            R = new UpperTriangDenseMatrix(Math.min(m, n));
        else
            L = new LowerTriangDenseMatrix(Math.min(m, n));
    }

    /**
     * Returns the orthogonal part of the factorization
     */
    public DenseMatrix getQ() {
        return Q;
    }

    /**
     * Returns the lower triangular factor, if present
     */
    public LowerTriangDenseMatrix getL() {
        return L;
    }

    /**
     * Returns the upper triangular factor, if present
     */
    public UpperTriangDenseMatrix getR() {
        return R;
    }

}
