/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

import mt.ll.BLASkernel.UpLo;
import mt.ll.Interface;

/**
 * Partial implementation of a symmetrical, packed matrix
 */
abstract class AbstractSymmPackMatrix extends AbstractPackMatrix {

    /**
     * Which part of the matrix which is stored
     */
    private UpLo uplo;

    /**
     * Constructor for AbstractSymmPackMatrix
     */
    AbstractSymmPackMatrix(int n, UpLo uplo) {
        super(n);
        this.uplo = uplo;
    }

    /**
     * Constructor for AbstractSymmPackMatrix
     */
    AbstractSymmPackMatrix(Matrix A, UpLo uplo) {
        this(A, true, uplo);
    }

    /**
     * Constructor for AbstractSymmPackMatrix
     */
    AbstractSymmPackMatrix(Matrix A, boolean deep, UpLo uplo) {
        super(A, deep);
        this.uplo = uplo;
    }

    public Vector multAdd(double alpha, Vector x, double beta, Vector y,
            Vector z) {
        if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
            return super.multAdd(alpha, x, beta, y, z);

        checkMultAdd(x, y, z);

        double[] xd = ((DenseVector) x).getData(), zd = ((DenseVector) z)
                .getData();

        z.set(y);

        Interface.blas().spmv(uplo, numRows, alpha, data, xd, beta, zd);

        return z;
    }

    public Vector transMultAdd(double alpha, Vector x, double beta, Vector y,
            Vector z) {
        return multAdd(alpha, x, beta, y, z);
    }

    public Matrix rank1(double alpha, Vector x, Vector y) {
        if (x != y)
            throw new IllegalArgumentException("x != y");
        if (!(x instanceof DenseVector))
            return super.rank1(alpha, x, y);

        checkRank1(x, y);

        double[] xd = ((DenseVector) x).getData();

        Interface.blas().spr(uplo, numRows, alpha, xd, data);

        return this;
    }

    public Matrix rank2(double alpha, Vector x, Vector y) {
        if (!(x instanceof DenseVector) || !(y instanceof DenseVector))
            return super.rank2(alpha, x, y);

        checkRank2(x, y);

        double[] xd = ((DenseVector) x).getData(), yd = ((DenseVector) y)
                .getData();

        Interface.blas().spr2(uplo, numRows, alpha, xd, yd, data);

        return this;
    }

    public Matrix solve(Matrix B, Matrix X) {
        if (!(X instanceof DenseMatrix))
            throw new UnsupportedOperationException("X must be a DenseMatrix");

        checkSolve(B, X);

        double[] Xd = ((DenseMatrix) X).getData();

        X.set(B);

        int[] ipiv = new int[numRows];

        int info = Interface.lapack().spsv(uplo, numRows, X.numColumns(),
                data.clone(), ipiv, Xd);

        if (info > 0)
            throw new MatrixSingularException();
        else if (info < 0)
            throw new IllegalArgumentException();

        return X;
    }

    public Vector solve(Vector b, Vector x) {
        DenseMatrix B = new DenseMatrix(b, false), X = new DenseMatrix(x, false);
        solve(B, X);
        return x;
    }

    public Matrix transSolve(Matrix B, Matrix X) {
        return solve(B, X);
    }

    public Vector transSolve(Vector b, Vector x) {
        return solve(b, x);
    }

    Matrix SPDsolve(Matrix B, Matrix X) {
        if (!(X instanceof DenseMatrix))
            throw new UnsupportedOperationException("X must be a DenseMatrix");

        checkSolve(B, X);

        double[] Xd = ((DenseMatrix) X).getData();

        X.set(B);

        int info = Interface.lapack().ppsv(uplo, numRows, X.numColumns(),
                data.clone(), Xd);

        if (info > 0)
            throw new MatrixNotSPDException();
        else if (info < 0)
            throw new IllegalArgumentException();

        return X;
    }

    public Matrix transpose() {
        return this;
    }

}
