/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MPP.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mpp;

/**
 * Contains predefined reductions
 */
public class Reductions {

    private Reductions() {
        // No need to instantiate
    }

    public static Reduction sum() {
        return new Sum();
    }

    public static Reduction product() {
        return new Product();
    }

    public static Reduction max() {
        return new Max();
    }

    public static Reduction min() {
        return new Min();
    }

    /**
     * Does a sum
     */
    private static class Sum implements Reduction {

        public void initByte(byte[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 0;
        }

        public void initChar(char[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 0;
        }

        public void initShort(short[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 0;
        }

        public void initInt(int[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 0;
        }

        public void initFloat(float[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 0;
        }

        public void initLong(long[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 0;
        }

        public void initDouble(double[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 0;
        }

        public void opByte(byte[] x, byte[] y) {
            assert x.length == y.length;
            for (int i = 0; i < x.length; ++i)
                x[i] += y[i];
        }

        public void opChar(char[] x, char[] y) {
            assert x.length == y.length;
            for (int i = 0; i < x.length; ++i)
                x[i] += y[i];
        }

        public void opShort(short[] x, short[] y) {
            assert x.length == y.length;
            for (int i = 0; i < x.length; ++i)
                x[i] += y[i];
        }

        public void opInt(int[] x, int[] y) {
            assert x.length == y.length;
            for (int i = 0; i < x.length; ++i)
                x[i] += y[i];
        }

        public void opFloat(float[] x, float[] y) {
            assert x.length == y.length;
            for (int i = 0; i < x.length; ++i)
                x[i] += y[i];
        }

        public void opLong(long[] x, long[] y) {
            assert x.length == y.length;
            for (int i = 0; i < x.length; ++i)
                x[i] += y[i];
        }

        public void opDouble(double[] x, double[] y) {
            assert x.length == y.length;
            for (int i = 0; i < x.length; ++i)
                x[i] += y[i];
        }

    }

    /**
     * Multiplies the elements
     */
    private static class Product implements Reduction {

        public void initByte(byte[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 1;
        }

        public void initChar(char[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 1;
        }

        public void initShort(short[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 1;
        }

        public void initInt(int[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 1;
        }

        public void initFloat(float[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 1;
        }

        public void initLong(long[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 1;
        }

        public void initDouble(double[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = 1;
        }

        public void opByte(byte[] x, byte[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] *= y[i];
        }

        public void opChar(char[] x, char[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] *= y[i];
        }

        public void opShort(short[] x, short[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] *= y[i];
        }

        public void opInt(int[] x, int[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] *= y[i];
        }

        public void opFloat(float[] x, float[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] *= y[i];
        }

        public void opLong(long[] x, long[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] *= y[i];
        }

        public void opDouble(double[] x, double[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] *= y[i];
        }

    }

    /**
     * Finds max
     */
    private static class Max implements Reduction {

        public void initByte(byte[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Byte.MIN_VALUE;
        }

        public void initChar(char[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Character.MIN_VALUE;
        }

        public void initShort(short[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Short.MIN_VALUE;
        }

        public void initInt(int[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Integer.MIN_VALUE;
        }

        public void initFloat(float[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Float.MIN_VALUE;
        }

        public void initLong(long[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Long.MIN_VALUE;
        }

        public void initDouble(double[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Double.MIN_VALUE;
        }

        public void opByte(byte[] x, byte[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = (byte) Math.max(x[i], y[i]);
        }

        public void opChar(char[] x, char[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = (char) Math.max(x[i], y[i]);
        }

        public void opShort(short[] x, short[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = (short) Math.max(x[i], y[i]);
        }

        public void opInt(int[] x, int[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Math.max(x[i], y[i]);
        }

        public void opFloat(float[] x, float[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Math.max(x[i], y[i]);
        }

        public void opLong(long[] x, long[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Math.max(x[i], y[i]);
        }

        public void opDouble(double[] x, double[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Math.max(x[i], y[i]);
        }

    }

    /**
     * Finds min
     */
    private static class Min implements Reduction {

        public void initByte(byte[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Byte.MAX_VALUE;
        }

        public void initChar(char[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Character.MAX_VALUE;
        }

        public void initShort(short[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Short.MAX_VALUE;
        }

        public void initInt(int[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Integer.MAX_VALUE;
        }

        public void initFloat(float[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Float.MAX_VALUE;
        }

        public void initLong(long[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Long.MAX_VALUE;
        }

        public void initDouble(double[] x) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Double.MAX_VALUE;
        }

        public void opByte(byte[] x, byte[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = (byte) Math.min(x[i], y[i]);
        }

        public void opChar(char[] x, char[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = (char) Math.min(x[i], y[i]);
        }

        public void opShort(short[] x, short[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = (short) Math.min(x[i], y[i]);
        }

        public void opInt(int[] x, int[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Math.min(x[i], y[i]);
        }

        public void opFloat(float[] x, float[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Math.min(x[i], y[i]);
        }

        public void opLong(long[] x, long[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Math.min(x[i], y[i]);
        }

        public void opDouble(double[] x, double[] y) {
            for (int i = 0; i < x.length; ++i)
                x[i] = Math.min(x[i], y[i]);
        }

    }

}
