# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['terra_sdk',
 'terra_sdk.client',
 'terra_sdk.client.lcd',
 'terra_sdk.client.lcd.api',
 'terra_sdk.core',
 'terra_sdk.core.auth',
 'terra_sdk.core.auth.data',
 'terra_sdk.core.authz',
 'terra_sdk.core.bank',
 'terra_sdk.core.distribution',
 'terra_sdk.core.feegrant',
 'terra_sdk.core.gov',
 'terra_sdk.core.ibc',
 'terra_sdk.core.ibc.data',
 'terra_sdk.core.ibc.msgs',
 'terra_sdk.core.ibc_transfer',
 'terra_sdk.core.market',
 'terra_sdk.core.oracle',
 'terra_sdk.core.params',
 'terra_sdk.core.slashing',
 'terra_sdk.core.staking',
 'terra_sdk.core.staking.data',
 'terra_sdk.core.treasury',
 'terra_sdk.core.upgrade',
 'terra_sdk.core.upgrade.data',
 'terra_sdk.core.wasm',
 'terra_sdk.key',
 'terra_sdk.util']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.8.1,<4.0.0',
 'attrs>=20.3.0,<21.0.0',
 'bech32>=1.2.0,<2.0.0',
 'betterproto==2.0.0b4',
 'bip32utils>=0.3.post4,<0.4',
 'boltons>=21.0.0,<22.0.0',
 'ecdsa>=0.17.0,<0.18.0',
 'furl>=2.1.3,<3.0.0',
 'mnemonic>=0.19,<0.20',
 'nest-asyncio>=1.5.1,<2.0.0',
 'protobuf>=3.19.1,<4.0.0',
 'terra-proto>=1.0.1,<2.0.0',
 'wrapt>=1.12.1,<2.0.0']

setup_kwargs = {
    'name': 'terra-sdk',
    'version': '2.0.2',
    'description': 'The Python SDK for Terra',
    'long_description': '<br/>\n<br/>\n\n<div  align="center"> <p > <img src="https://raw.githubusercontent.com/terra-money/terra-sdk-python/main/docs/img/logo.png" width=500 alt="py-sdk-logo"></p>\n\nThe Python SDK for Terra\n<br/>\n<p><sub>(Unfamiliar with Terra?  <a href="https://docs.terra.money/">Check out the Terra Docs</a>)</sub></p>\n\n  <p > <img alt="GitHub" src="https://img.shields.io/github/license/terra-money/terra-sdk-python">\n<img alt="Python" src="https://img.shields.io/pypi/pyversions/terra-sdk">\n  <img alt="pip" src="https://img.shields.io/pypi/v/terra-sdk"></p>\n<p>\n  <a href="https://terra-money.github.io/terra.py/index.html"><strong>Explore the Docs »</strong></a>\n<br/>\n  <a href="https://pypi.org/project/terra-sdk/">PyPI Package</a>\n  ·\n  <a href="https://github.com/terra-money/terra.py">GitHub Repository</a>\n</p></div>\n\n\nThe Terra Software Development Kit (SDK) in Python is a simple library toolkit for building software that can interact with the Terra blockchain and provides simple abstractions over core data structures, serialization, key management, and API request generation.\n\n## Features\n\n- Written in Python with extensive support libraries\n- Versatile support for key management solutions\n- Exposes the Terra API through LCDClient\n\n<br/>\n\n# Table of Contents\n- [API Reference](#api-reference)\n- [Getting Started](#getting-started)\n  - [Requirements](#requirements)\n  - [Installation](#installation)\n  - [Dependencies](#dependencies)\n  - [Tests](#tests)\n  - [Code Quality](#code-quality)\n- [Usage Examples](#usage-examples) \n  - [Getting Blockchain Information](#getting-blockchain-information)\n    - [Async Usage](#async-usage)\n  - [Building and Signing Transactions](#building-and-signing-transactions)\n      - [Example Using a Wallet](#example-using-a-wallet-recommended)\n- [Contributing](#contributing)\n  - [Reporting an Issue](#reporting-an-issue)\n  - [Requesting a Feature](#requesting-a-feature)\n  - [Contributing Code](#contributing-code)\n  - [Documentation Contributions](#documentation-contributions)\n- [License](#license)\n\n<br/>\n\n\n\n# API Reference\nAn intricate reference to the APIs on the Terra SDK can be found <a href="https://terra-money.github.io/terra.py/index.html">here</a>.\n\n<br/>\n\n# Getting Started\nA walk-through of the steps to get started with the Terra SDK alongside a few use case examples are provided below. Alternatively, a tutorial video is also available <a href="https://www.youtube.com/watch?v=GfasBlJHKIg">here</a> as reference.\n\n## Requirements\nTerra SDK requires <a href="https://www.python.org/downloads/">Python v3.7+</a>.\n\n## Installation\n\n<sub>**NOTE:** *All code starting with a `$` is meant to run on your terminal (a bash prompt). All code starting with a `>>>` is meant to run in a python interpreter, like <a href="https://pypi.org/project/ipython/">ipython</a>.*</sub>\n\nTerra SDK can be installed (preferably in a `virtual environment` from PyPI using `pip`) as follows:\n\n```\n$ pip install -U terra_sdk\n```\n<sub>*You might have `pip3` installed instead of `pip`; proceed according to your own setup.*<sub>\n\n## Dependencies\nTerra SDK uses <a href="https://python-poetry.org/">Poetry</a> to manage dependencies. To get set up with all the required dependencies, run:\n```\n$ pip install poetry\n$ poetry install\n```\n\n\n## Tests\nTerra SDK provides extensive tests for data classes and functions. To run them, after the steps in [Dependencies](#dependencies):\n```\n$ make test\n```\n\n## Code Quality\nTerra SDK uses <a href="https://black.readthedocs.io/en/stable/">Black</a>, <a href="https://isort.readthedocs.io/en/latest/">isort</a>, and <a href="https://mypy.readthedocs.io/en/stable/index.html">Mypy</a> for checking code quality and maintaining style. To reformat, after the steps in [Dependencies](#dependencies):\n```\n$ make qa && make format\n```\n\n<br/>\n\n# Usage Examples\nTerra SDK can help you read block data, sign and send transactions, deploy and interact with contracts, and many more.\nThe following examples are provided to help you get started. Use cases and functionalities of the Terra SDK are not limited to the following examples and can be found in full <a href="https://terra-money.github.io/terra.py/index.html">here</a>.\n\nIn order to interact with the Terra blockchain, you\'ll need a connection to a Terra node. This can be done through setting up an LCDClient (The LCDClient is an object representing an HTTP connection to a Terra LCD node.):\n\n```\n>>> from terra_sdk.client.lcd import LCDClient\n>>> terra = LCDClient(chain_id="columbus-5", url="https://lcd.terra.dev")\n```\n\n## Getting Blockchain Information\n\nOnce properly configured, the `LCDClient` instance will allow you to interact with the Terra blockchain. Try getting the latest block height:\n\n\n```\n>>> terra.tendermint.block_info()[\'block\'][\'header\'][\'height\']\n```\n\n`\'1687543\'`\n\n\n### Async Usage\n\nIf you want to make asynchronous, non-blocking LCD requests, you can use AsyncLCDClient. The interface is similar to LCDClient, except the module and wallet API functions must be awaited.\n\n<pre><code>\n>>> import asyncio \n>>> from terra_sdk.client.lcd import AsyncLCDClient\n\n>>> async def main():\n      <strong>terra = AsyncLCDClient("https://lcd.terra.dev", "columbus-5")</strong>\n      total_supply = await terra.bank.total()\n      print(total_supply)\n      <strong>await terra.session.close # you must close the session</strong>\n\n>>> asyncio.get_event_loop().run_until_complete(main())\n</code></pre>\n\n## Building and Signing Transactions\n\nIf you wish to perform a state-changing operation on the Terra blockchain such as sending tokens, swapping assets, withdrawing rewards, or even invoking functions on smart contracts, you must create a **transaction** and broadcast it to the network.\nTerra SDK provides functions that help create StdTx objects.\n\n### Example Using a Wallet (*recommended*)\n\nA `Wallet` allows you to create and sign a transaction in a single step by automatically fetching the latest information from the blockchain (chain ID, account number, sequence).\n\nUse `LCDClient.wallet()` to create a Wallet from any Key instance. The Key provided should correspond to the account you intend to sign the transaction with.\n\n\n```\n>>> from terra_sdk.client.lcd import LCDClient\n>>> from terra_sdk.key.mnemonic import MnemonicKey\n\n>>> mk = MnemonicKey(mnemonic=MNEMONIC) \n>>> terra = LCDClient("https://lcd.terra.dev", "columbus-5")\n>>> wallet = terra.wallet(mk)\n```\n\nOnce you have your Wallet, you can simply create a StdTx using `Wallet.create_and_sign_tx`.\n\n\n```\n>>> from terra_sdk.core.auth import Fee\n>>> from terra_sdk.core.bank import MsgSend\n>>> from terra_sdk.client.lcd.api.tx import CreateTxOptions\n\n>>> tx = wallet.create_and_sign_tx(CreateTxOptions(\n        msgs=[MsgSend(\n            wallet.key.acc_address,\n            RECIPIENT,\n            "1000000uluna"    # send 1 luna\n        )],\n        memo="test transaction!",\n        fee=Fee(200000, "120000uluna")\n    ))\n```\n\nYou should now be able to broadcast your transaction to the network.\n\n```\n>>> result = terra.tx.broadcast(tx)\n>>> print(result)\n```\n\n<br/>\n\n# Contributing\n\nCommunity contribution, whether it\'s a new feature, correction, bug report, additional documentation, or any other feedback is always welcome. Please read through this section to ensure that your contribution is in the most suitable format for us to effectively process.\n\n<br/>\n\n## Reporting an Issue \nFirst things first: **Do NOT report security vulnerabilities in public issues!** Please disclose responsibly by submitting your findings to the [Terra Bugcrowd submission form](https://www.terra.money/bugcrowd). The issue will be assessed as soon as possible. \nIf you encounter a different issue with the Python SDK, check first to see if there is an existing issue on the <a href="https://github.com/terra-money/terra-sdk-python/issues">Issues</a> page, or if there is a pull request on the <a href="https://github.com/terra-money/terra-sdk-python/pulls">Pull requests</a> page. Be sure to check both the Open and Closed tabs addressing the issue. \n\nIf there isn\'t a discussion on the topic there, you can file an issue. The ideal report includes:\n\n* A description of the problem / suggestion.\n* How to recreate the bug.\n* If relevant, including the versions of your:\n    * Python interpreter\n    * Terra SDK\n    * Optionally of the other dependencies involved\n* If possible, create a pull request with a (failing) test case demonstrating what\'s wrong. This makes the process for fixing bugs quicker & gets issues resolved sooner.\n</br>\n\n## Requesting a Feature\nIf you wish to request the addition of a feature, please first check out the <a href="https://github.com/terra-money/terra-sdk-python/issues">Issues</a> page and the <a href="https://github.com/terra-money/terra-sdk-python/pulls">Pull requests</a> page (both Open and Closed tabs). If you decide to continue with the request, think of the merits of the feature to convince the project\'s developers, and provide as much detail and context as possible in the form of filing an issue on the <a href="https://github.com/terra-money/terra-sdk-python/issues">Issues</a> page.\n\n\n<br/>\n\n## Contributing Code\nIf you wish to contribute to the repository in the form of patches, improvements, new features, etc., first scale the contribution. If it is a major development, like implementing a feature, it is recommended that you consult with the developers of the project before starting the development to avoid duplicating efforts. Once confirmed, you are welcome to submit your pull request.\n</br>\n\n### For new contributors, here is a quick guide: \n\n1. Fork the repository.\n2. Build the project using the [Dependencies](#dependencies) and [Tests](#tests) steps.\n3. Install a <a href="https://virtualenv.pypa.io/en/latest/index.html">virtualenv</a>.\n4. Develop your code and test the changes using the [Tests](#tests) and [Code Quality](#code-quality) steps.\n5. Commit your changes (ideally follow the <a href="https://github.com/angular/angular/blob/master/CONTRIBUTING.md#commit">Angular commit message guidelines</a>).\n6. Push your fork and submit a pull request to the repository\'s `main` branch to propose your code.\n   \n\nA good pull request:\n* Is clear and concise.\n* Works across all supported versions of Python. (3.7+)\n* Follows the existing style of the code base (<a href="https://pypi.org/project/flake8/">`Flake8`</a>).\n* Has comments included as needed.\n* Includes a test case that demonstrates the previous flaw that now passes with the included patch, or demonstrates the newly added feature.\n* Must include documentation for changing or adding any public APIs.\n* Must be appropriately licensed (MIT License).\n</br>\n\n## Documentation Contributions\nDocumentation improvements are always welcome. The documentation files live in the [docs](./docs) directory of the repository and are written in <a href="https://docutils.sourceforge.io/rst.html">reStructuredText</a> and use <a href="https://www.sphinx-doc.org/en/master/">Sphinx</a> to create the full suite of documentation.\n</br>\nWhen contributing documentation, please do your best to follow the style of the documentation files. This means a soft limit of 88 characters wide in your text files and a semi-formal, yet friendly and approachable, prose style. You can propose your improvements by submitting a pull request as explained above.\n\n### Need more information on how to contribute?\nYou can give this <a href="https://opensource.guide/how-to-contribute/#how-to-submit-a-contribution">guide</a> read for more insight.\n\n\n\n\n<br/>\n\n# License\n\nThis software is licensed under the MIT license. See [LICENSE](./LICENSE) for full disclosure.\n\n© 2021 Terraform Labs, PTE.\n\n<hr/>\n\n<p>&nbsp;</p>\n<p align="center">\n    <a href="https://terra.money/"><img src="https://assets.website-files.com/611153e7af981472d8da199c/61794f2b6b1c7a1cb9444489_symbol-terra-blue.svg" alt="Terra-logo" width=200/></a>\n<div align="center">\n  <sub><em>Powering the innovation of money.</em></sub>\n</div>\n\n',
    'author': 'Terraform Labs, PTE.',
    'author_email': 'engineering@terra.money',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/terra-money/terra.py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
