# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ps2_census']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2.23,<3.0', 'toml>=0.9,<0.10', 'websockets>=8.1,<9.0']

setup_kwargs = {
    'name': 'ps2-census',
    'version': '0.5.0',
    'description': 'Daybreak Games Census API client for Planetside 2 (UNOFFICIAL)',
    'long_description': '# ps2-census\n\nps2-census is a low-level client for Daybreak\'s Planetside 2 Census API written in Python >= 3.8.\n\n*Features*:\n- Build queries through method chaining\n- Join collections and nest them through method chaining\n- Get raw (deserialized) responses as Python dictionaries\n- Access common enums directly\n- Subscribe to event streams\n\nBy default the `s:example` service ID is used; however it is not recommended for production.\nYou should get your own service ID from the webside below and supply it to the client whenever needed.\n\nMore information about the Census API is available on the official Census documentation [here](http://census.daybreakgames.com/).\n\n[Iridar\'s blog](https://iridar-mirror.knyazev.io/index.html%3Fp=4127.html) is also recommended to understand\nthe quirks of this particular API as this knowledge is necessary to use ps2-census properly.\n\n## Installation\n\n`pip install ps2-census`\n\n## Query building\n\nQueries are made on collections with optional joins on other collections as well as various\ncommands that alter the output.\n\n### Basic query\n\nTo build a query, instantiate the `Query` class with a `Collection` and your service ID:\n```\nfrom ps2_census import Collection, Query\n\nquery: Query = Query(Collection.ITEM, service_id=YOUR_SERVICE_ID)\n```\n\nChain methods to alter the query further. Available methods are those detailed on the\nofficial Census API documentation website.\n```\nquery: Query = (\n    Query(Collection.ITEM, service_id=YOUR_SERVICE_ID)\n    .lang("en")\n    .sort(("item_id", 1), ("faction_id", -1))\n    .limit(30)\n)\n```\n\nExecute the query in one of the 2 ways made available by the Census API:\n\n- `.count()` to get the items count\n```\nquery.count()\n> {\'count\': 21048}\n```\n\n- `.get()` to get the results\n```\nquery.get()\n> {\'item_list\': [{...}, {...}, ...], \'returned\': 30}\n```\n\n### Single join\n\nIn order to perform joins instantiate the `Join` class with a `Collection` and pass it to the `Query`:\n```\nfrom ps2_census import Collection, Join, Query\n\nquery: Query = (\n    Query(Collection.XXX)\n    .yyy()\n    .join(\n        Join(Collection.WEAPON_DATASHEET)\n        .outer(0)\n        .on("item_id")\n        .to("item_id")\n        .inject_at("weapon_datasheet")\n    )\n)\n```\n\n### Nested join\n\nNested join are necessary in order to access data structures deeper in the collections tree.\nTo nest joins, instantiate the `Join` class multiple times and combine them through `join1.nest(join2)`\nwhere `join2` is nested within `join1`:\n```\nfrom ps2_census import Collection, Join, Query\n\nitem_to_weapon_join: Join = (\n    Join(Collection.ITEM_TO_WEAPON)\n    .on("item_id")\n    .to("item_id")\n    .inject_at("item_to_weapon")\n)\n\nweapon_join: Join (\n    Join(Collection.WEAPON)\n    .on("weapon_id")\n    .to("weapon_id")\n    .inject_at("weapon")\n)\n\nquery: Query = (\n    Query(Collection.ITEM)\n    .filter("item_type_id", ItemType.WEAPON)\n    .join(\n        item_to_weapon_join.nest(\n            weapon_join\n        )\n    )\n)\n```\n\nFor a deep join you might find it easier to first create the `Join` instances then nest them\nas shown above without having too much indentation depth.\n\n### Tree\n\nTrees are also built using their own class, `Tree`, then passed to the `Query` object:\n```\nfrom ps2_census import Collection, Query, Tree\n\nquery: Query = (\n    Query(Collection.ITEM)\n    .tree(\n        Tree("name.en")\n        .prefix("en_name_")\n    )\n)\n```\n\n## Common enums\n\nCensus API data uses a lot of integer enumerations that are collections themselves.\n\nFor example the `faction_id` key in items from `Collection.ITEM` is an integer that represents\na specific fation, refering to `Collection.FACTION`: Vanu is `1`, NC is `2` etc.\n\nIn order to reduce the amount of necessary joins, which are arguably the most complex part of Census queries,\nsome common enumerations are provided in `ps2_census.enums` as Python enum.IntEnum classes, including:\n- ArmorFacing\n- FacilityType\n- Faction\n- FireModeType\n- ItemType\n- ItemCategory\n- MetagameEventState\n- PlayerState\n- ProjectileFlightType\n- ResistType\n- ResourceType\n- RewardType\n- TargetType\n- Vehicle\n- World\n\nThese typically do not change often and ps2-census will be updated whenever there is such a change.\n\nThey can be used just for reference, but also in queries for filtering.\n\nSee the following example for filtering weapon items only using `ps2_census.enums.ItemType`:\n```\nfrom ps2_census.enums import ItemType\nquery = (\n    Query(Collection.ITEM, service_id=YOUR_SERVICE_ID)\n    .filter("item_type_id", ItemType.WEAPON)\n)\n```\n\n## Event Stream\n\nps2-census offers a client that handles connection to the WebSocket endpoint, subscription\nto various streams and reception of the events.\n\n*Note*: because the client uses the [websockets](https://github.com/aaugustin/websockets) library,\nwe need to handle async calls.\n\n### Usage\n\n`from ps2_census import EventStream`\n\nFirst you need to connect to the WebSocket endpoint; to do this, instantiate the `EventStream` class:\n```\nstream: EventStream = await EventStream(service_id=YOUR_SERVICE_ID)\n```\n\nThen, subscribe to events:\n```\nfrom ps2_census inport CharacterEvent, WorldEvent, EventStreamWorld\n\nawait stream.subscribe(\n    worlds=[EventStreamWorld.XXX, EventStreamWorld.YYY],\n    events=[WorldEvent.ZZZ, CharacterEvent.AAA],\n    characters=["1234", "5678"],\n    logical_and_characters_with_worlds=True\n)\n```\nWhere:\n- `worlds` is a list of `EventStreamWorld` objects. Use `[EventStreamWorld.ALL]` for all worlds\n- `events` is a list of `CharacterEvent`, `WorldEvent` or `GenericEvent` objects. Use `GenericEvent.ALL` to get all events (character and world)\n- `characters` is a list of character IDs as strings\n- `logical_and_characters_with_worlds` is True if you want to match all events concerning the characters *AND* the worlds; default is False, so it matches all events concerning the characters *OR* the worlds\n\nYou can perform multiple subscriptions one after another on the same `EventStream` object; they are additively merged.\n\nFinally, you need to handle received events from your subscription, for example:\n```\nwhile True:\n    print(await stream.receive())\n```\n\nMore information about the Planetside2 Census event stream can be found at [here](http://census.daybreakgames.com/#what-is-websocket).\n\n### Full example\n```\nfrom ps2_census import EventStream, CharacterEvent, WorldEvent, EventStreamWorld\n\nstream = await EventStream(service_id=xxx)\n\nawait stream.subscribe(worlds=[EventStreamWorld.SOLTECH], events=[WorldEvent.CONTINENT_LOCK, WorldEvent.CONTINENT_UNLOCK])\n\nwhile True:\n    print(await stream.receive())\n```\n\n## Development\n\nIn order to work on ps2-analysis:\n- Setup a virtual environment with python 3.8\n- Install [poetry](https://github.com/python-poetry/poetry)\n- Install dependencies with `poetry install`\n- Run tests with `pytest`\n- Update dependencies with `poetry update`\n',
    'author': 'Sylvain Pascou',
    'author_email': 'sylvain@pascou.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/spascou/ps2-census',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
