# This file is part of the Reproducible and Reusable Data Analysis Workflow
# Server (flowServ).
#
# Copyright (C) 2019-2020 NYU.
#
# flowServ is free software; you can redistribute it and/or modify it under the
# terms of the MIT License; see LICENSE file for more details.

"""Abstract class for file stores. The file store provides access to files that
are uploaded for workflow groups or that are generated by successful workflow
runs.

The file store also defines the folder structure of the file system for
workflows and their associated resources.

The folder structure is currently as follows:

/workflows/                  : Base directory
    {workflow_id}            : Folder for individual workflow
        groups/              : Folder for workflow groups
            {group_id}       : Folder for individual group
                files/       : Uploaded files for workflow group
                    {file_id}: Folder for uploaded file
        runs/                : Folder for all workflow runs
            {run_id}         : Result files for individual runs
        static/
"""

from abc import ABCMeta, abstractmethod
from typing import IO, List, Tuple, Union

import os


class FileStore(metaclass=ABCMeta):
    """Interface for the file store. Files are identified by unique keys (e.g.,
    relative paths). The key structure is implementation-dependent.
    """
    @abstractmethod
    def configuration(self) -> List[Tuple[str, str]]:
        """Get a list of tuples with the names of additional configuration
        variables and their current values.

        Returns
        -------
        list((string, string))
        """
        raise NotImplementedError()  # pragma: no cover

    @abstractmethod
    def copy_files(self, src: str, files: List[Tuple[str, str]]):
        """Copy a list of files or dirctories from a given source directory.
        The list of files contains tuples of relative file source and target
        path. The source path may reference existing files or directories.

        Parameters
        ----------
        src: string
            Path to source directory on disk.
        files: list((string, string))
            List of file source and target path. All path names are relative.
        """
        raise NotImplementedError()  # pragma: no cover

    @abstractmethod
    def delete_file(self, key: str):
        """Delete the file with the given key.

        Parameters
        ----------
        key: string
            Unique file key.
        """
        raise NotImplementedError()  # pragma: no cover

    @abstractmethod
    def download_archive(self, src: str, files: List[str]) -> IO:
        """Download all files in the given list from the specified source
        directory as a tar archive.

        Parameters
        ----------
        src: string
            Relative path to the files source directory.
        files: list(string)
            List of relative paths to files (or directories) in the specified
            source directory. Lists the files to include in the returned
            archive.

        Returns
        -------
        io.BytesIO
        """
        raise NotImplementedError()  # pragma: no cover

    @abstractmethod
    def download_files(self, files: List[Tuple[str, str]], dst: str):
        """Copy a list of files or dirctories from the file store to a given
        destination directory. The list of files contains tuples of relative
        file source and target path. The source path may reference files or
        directories.

        Parameters
        ----------
        files: list((string, string))
            List of file source and target path. All path names are relative.
        dst: string
            Path to target directory on disk.

        Raises
        ------
        ValueError
        """
        raise NotImplementedError()  # pragma: no cover

    def group_uploaddir(self, workflow_id: str, group_id: str) -> str:
        """Get base directory for files that are uploaded to a workflow group.

        Parameters
        ----------
        workflow_id: string
            Unique workflow identifier
        group_id: string
            Unique workflow group identifier

        Returns
        -------
        string
        """
        groupdir = self.workflow_groupdir(workflow_id, group_id)
        return os.path.join(groupdir, 'files')

    def group_uploadfile(
        self, workflow_id: str, group_id: str, file_id: str
    ) -> str:
        """Get path for a file that was uploaded for a workflow group.

        Parameters
        ----------
        workflow_id: string
            Unique workflow identifier
        group_id: string
            Unique workflow group identifier
        file_id: string
            Unique file identifier

        Returns
        -------
        string
        """
        uploaddir = self.group_uploaddir(workflow_id, group_id)
        return os.path.join(uploaddir, file_id)

    @abstractmethod
    def load_file(self, key: str) -> Union[str, IO]:
        """Get a file object for the given key. The type of the result is
        implmentation-dependent. The result may either be the path to a file on
        disk or a FileObject.

        Parameters
        ----------
        key: string
            Unique file key.

        Returns
        -------
        string or FileObject

        Raises
        ------
        flowserv.error.UnknownFileError
        """
        raise NotImplementedError()  # pragma: no cover

    def run_basedir(self, workflow_id: str, run_id: str) -> str:
        """Get path to the base directory for all files that are maintained for
        a workflow run.

        Parameters
        ----------
        workflow_id: string
            Unique workflow identifier
        run_id: string
            Unique run identifier

        Returns
        -------
        string
        """
        workflowdir = self.workflow_basedir(workflow_id)
        return os.path.join(workflowdir, 'runs', run_id)

    @abstractmethod
    def upload_file(self, file: Union[str, IO], dst: str) -> int:
        """Upload a given file object to the file store. The destination path
        is a relative path. The file may reference a file on the local file
        system or it is a fiile object (StringIO or BytesIO).

        Returns the size of the uploaded file.

        Paramaters
        ----------
        file: string or FileObject
            The input file is either a FileObject (buffer) or a reference to a
            file on the local file system.
        dst: string
            Relative target path for the stored file.

        Returns
        -------
        int
        """
        raise NotImplementedError()  # pragma: no cover

    def workflow_basedir(self, workflow_id: str) -> str:
        """Get base directory containing associated files for the workflow with
        the given identifier.

        Parameters
        ----------
        workflow_id: string
            Unique workflow identifier

        Returns
        -------
        string
        """
        return workflow_id

    def workflow_groupdir(self, workflow_id: str, group_id: str) -> str:
        """Get base directory containing files that are associated with a
        workflow group.

        Parameters
        ----------
        workflow_id: string
            Unique workflow identifier
        group_id: string
            Unique workflow group identifier

        Returns
        -------
        string
        """
        workflowdir = self.workflow_basedir(workflow_id)
        return os.path.join(workflowdir, 'groups', group_id)

    def workflow_staticdir(self, workflow_id: str) -> str:
        """Get base directory containing static files that are associated with
        a workflow template.

        Parameters
        ----------
        workflow_id: string
            Unique workflow identifier

        Returns
        -------
        string
        """
        return os.path.join(self.workflow_basedir(workflow_id), 'static')
