# Copyright 2020 Software Factory Labs, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Defines an Abstract Base Class for `JobStore`."""

from abc import ABC, abstractmethod
from datetime import datetime
from typing import Optional, TYPE_CHECKING
from ..util import get_class_logger
from ..job import Job

if TYPE_CHECKING:
    from ..scheduler import Scheduler

MIN_INSTANCE_ID = 10000
MAX_INSTANCE_ID = 99999


class JobStore(ABC):
    """Abstract Base Class for `JobStore`s.
    
    Utilizes instance ids to track a `Job` and its associated data when running.

    Attributes:
        _logger (logging.Logger): The class' logger.
        _scheduler (Scheduler): The parent `Scheduler`.
        last_instance_id (int): The last instance id assigned.
    """

    def __init__(self) -> None:
        self._logger = get_class_logger(self)
        self._scheduler: Optional['Scheduler'] = None

        self.last_instance_id = 0

    def setup(self) -> None:
        """Any basic setup necessary for the `JobStore`."""
        pass

    def teardown(self) -> None:
        """Cleans up `Job`s and their instances."""
        pass

    def get_due_jobs(self, latest: datetime) -> list[Job]:
        """Returns a list of all `Job`s that are due to run.

        Args:
            latest (datetime): Filters out all `Job`s scheduled to run after this time.

        Returns:
            List[Job]: A list of `Job`s scheduled to run before `latest`.
        """
        jobs = self.get_jobs()
        pending = list(filter(lambda job: job.active and job.next_run_time is not None and\
            job.next_run_time <= latest, jobs))
        return pending

    def get_next_run_time(self) -> Optional[datetime]:
        """Returns the next upcoming run time of the soonest `Job`, if any is scheduled."""
        jobs = self.get_jobs()
        return jobs[0].next_run_time if jobs else None

    @abstractmethod
    def add_job(self, job: Job, replace_existing: bool) -> None:
        """Adds a `Job` to store and track.

        Args:
            job (Job): A `Job` to store and track.
            replace_existing (bool): Determines if a `Job` matching the one provided should be
                replaced.
        """
        pass

    @abstractmethod
    def update_job(self, job: Job) -> None:
        """Updates an existing `Job` matching `job.id` with the new data.

        Args:
            job (Job): Used to update the current `Job` matching the provided `Job`'s id.
        """
        pass

    @abstractmethod
    def remove_job(self, job_id: str) -> None:
        """Removes a single `Job`.

        Args:
            job_id (str): Identifier for the `Job` to remove.
        """
        pass

    @abstractmethod
    def remove_all_jobs(self) -> None:
        """Removes all `Job`s."""
        pass

    @abstractmethod
    def get_job(self, job_id: str) -> Job:
        """Returns the `Job` associated with the `job_id`.

        Args:
            job_id (str): Identifier for the `Job` to retrieve.

        Returns:
            Job: The `Job` associated with the `job_id`.
        """
        pass

    @abstractmethod
    def get_jobs(self, pattern: str = None) -> list[Job]:
        """Returns a list of `Jobs` matching the pattern, or all if no pattern.

        Args:
            pattern (str): A regular expression pattern to match against `Job` ids.

        Returns:
            List[Job]: A list of `Job`s matching the pattern, or all if no pattern.
        """
        pass

    @abstractmethod
    def contains_job(self, job_id: str) -> bool:
        """Returns if the `JobStore` currently has a `Job`.

        Args:
            job_id (str): The identifier to look for.

        Returns:
            bool: True of the `JobStore` currently has a `Job` with the `job_id`, False otherwise.
        """
        pass

    def get_new_instance_id(self) -> int:
        """Returns the next instance id to use.

        Instance ids are in the range `[MIN_INSTANCE_ID, MAX_INSTANCE_ID]` and loop back around
        when the max is reached.

        Returns:
            int: The next instance id to use.
        """
        if self.last_instance_id == 0:
            self._logger.debug('first time generating a new instance ID. getting last one from jobstore')
            self.last_instance_id = self._get_stored_instance_id()
            if self.last_instance_id == 0:
                self._logger.debug(f'no last instance ID in jobstore. using minimum value ({MIN_INSTANCE_ID})')
                # we use MIN - 1 here because of the increment a few lines down
                self.last_instance_id = MIN_INSTANCE_ID - 1
            else:
                self._logger.debug(f'got last instance ID from jobstore ({self.last_instance_id})')

        if self.last_instance_id < MIN_INSTANCE_ID - 1 or self.last_instance_id > MAX_INSTANCE_ID:
            self._logger.error(f'Last instance ID ({self.last_instance_id}) was out of acceptable range '
                               f'([{MIN_INSTANCE_ID}, {MAX_INSTANCE_ID}])! Resetting to minimum value '
                               f'({MIN_INSTANCE_ID})')

            # we use MIN - 1 here because of the increment in the next line of code
            self.last_instance_id = MIN_INSTANCE_ID - 1

        self.last_instance_id += 1

        if self.last_instance_id > MAX_INSTANCE_ID:
            self._logger.debug(f'new instance ID at max ({MAX_INSTANCE_ID}), resetting to min ({MIN_INSTANCE_ID})')
            self.last_instance_id = MIN_INSTANCE_ID

        self._save_instance_id(self.last_instance_id)

        self._logger.debug(f'generated a new instance ID ({self.last_instance_id})')

        return self.last_instance_id

    @abstractmethod
    def _get_stored_instance_id(self) -> int:
        """Returns the last used instance id that was stored.
        
        Returns:
            int: The last used instance id that was stored.
        """
        pass

    @abstractmethod
    def _save_instance_id(self, instance_id: int) -> None:
        """Saves the last used instance id."""
        pass


class JobAlreadyExistsException(Exception):
    """Raised when a `Job` with the provided identifier already exists."""

    def __init__(self, job_id: str) -> None:
        """Initializes a JobAlreadyExistsException.

        Args:
            job_id (str): Identifier for a `Job` that was already in use.
        """
        super().__init__(f'Job "{job_id}" already exists')


class JobDoesNotExistException(Exception):
    """Raised when a `Job` identifier was used to retrieve a `Job`, but it did not exist."""

    def __init__(self, job_id: str) -> None:
        """Initializes a JobDoesNotExistException.

        Args:
            job_id (str): Identifier for a `Job` which was not in the `JobStore`.
        """
        super().__init__(f'Job "{job_id}" does not exist')
