"""Parser for KSP save file"""
from typing import Any, Dict, List, Optional, Union

BACKEND = "Python"


def parse_savefile(sfs: str, sfs_is_path: bool = True) -> Dict[str, Any]:
    """Parses an SFS file

    Params:
        sfs: str; the path to the SFS file to read or a string containing data read from an sfs.
        sfs_is_path (optional, default True): bool; whether the 'sfs' param is a path or raw data.
    Raises:
        No specific exceptions.
    Returns:
        Dictionary containing the data in the SFS.
    Extra information:
        All values are strings as SFS files do not reveal data to be any type.
        The SFS format is particularly bad and this leads to the returned dictionary
        containing data that is unusually structured. If the SFS contains multiple keys of any
        kind with the same name (this can be a 'node' header or values in a node), then the data
        contained within these keys will formatted as the common name of the keys as a key
        in a dict, and the values as a list. This data will always be in the exact order
        that they were in in the SFS (dictionaries are ordered in python3.6+). Example:

        --SFS format--
        NODE
        {
            x = 1
            x = 2
            y = 3
        }
        NODE
        {
            value = 1
        }
        OTHER
        {
            z = 4
        }

        --Python structure--
        {
            "NODE": [
                {"x": ["1","2"], "y": "3"},
                {"value": "1"}
            ],
            "OTHER": {
                "z": "4"
            }
        }
    """
    if sfs_is_path:
        with open(sfs, "r") as file:
            data = file.read()
    else:
        data = sfs
    # removes all tabs, spaces after newlines and spaces around equals
    data = data.replace("\t", "").replace("\n ", "\n").replace(" = ", "=")
    # in_nodes tracks the location of data being parsed (what nodes the parser is inside)
    in_nodes = []
    out_dict: Dict = {}
    # key_read contains the start index of the key being read
    key_read = 0
    # value_read contains the start index of the value being read
    value_read = 0
    for index, char in enumerate(data):
        # check if the char is one of the chars which leads to an action
        # this is an optimisation only
        if char in {"\n", "}", "="}:
            if char == "\n":
                # if the last character wasn't a bracket, we have a node
                if data[index - 1] not in {"{", "}"}:
                    # if next char is an open bracket, save it as a new node
                    if data[index + 1] == "{":
                        in_nodes.append(data[key_read:index])
                        set_value(out_dict, in_nodes, {})
                    # else it is a value in an existing node
                    else:
                        write_list = in_nodes[:]
                        # use value_read - 1 as the endpoint as the key must end 2 chars before the value starts
                        write_list.append(data[key_read : value_read - 1])
                        set_value(out_dict, write_list, data[value_read:index])
                # read the key from the beginning of the next line
                key_read = index + 1
            # pop the end of the 'stack' used to track attribute location
            # when the end of a node is found
            elif char == "}":
                in_nodes.pop()
            # due to the set check, the char must be =
            # set the start index of the value (the end index of the key must be one less than this)
            else:
                value_read = index + 1
    return out_dict


def set_value(
    dict_nested: Dict, address_list: List[str], value: Union[str, Dict]
) -> None:
    """Sets a value in a nested dict
    WARNING - modifies the dictionary passed as an arg"""
    # references the main dict
    current = dict_nested
    # locate the desired node to write to through iterating through the keys
    # while selecting the last element of any list found, as the data is in order
    for path_item in address_list[:-1]:
        if isinstance(current, list):
            current = current[-1][path_item]
        else:
            current = current[path_item]
    # if current is a list, then take the last entry as that's what will be modified
    if isinstance(current, list):
        current = current[-1]
    # if the node already exists
    prev_node = address_list[-1]
    if prev_node in current:
        # if it's a list simply append it to the list
        if isinstance(current[prev_node], list):
            current[prev_node].append(value)
        # else convert the existing dict to a list
        else:
            current[prev_node] = [current[prev_node], value]
    # if it doesn't exist
    else:
        # guaranteed to be a dict thanks to earlier list check, so insert the key into the dict
        current[prev_node] = value


def writeout_savefile(
    parsed_data: Dict, destination_file: Optional[str] = None
) -> Optional[str]:
    """Writes out the parsed data back into the SFS format

    Params:
        parsed_data: dict; the parsed dictionary generated by parse_savefile.
        destination_file (optional): str; the destination file to write the SFS to.
    Raises:
        No specific exceptions
    Returns:
        str containg the generated SFS if a destination file is not specified
        None if a destination file is specified
    Extra information:
        This function will generate a byte perfect copy of the original SFS parsed assuming
        the data is not modified. All abnormalities of the SFS format are addressed and
        represented correctly.
    """
    out_data: List[str] = []
    serialise_data(parsed_data, out_data, -1)
    out_str = "".join(out_data)
    if not destination_file:
        return out_str
    with open(destination_file, "w") as file:
        file.write(out_str)
    return None


def serialise_data(obj: Union[List, Dict], out_data: List[str], indents: int) -> None:
    """Recursively serialises data"""
    # indent upon each recurse
    indents += 1
    indent_str = "\t" * indents
    # set up the buffer list
    if isinstance(obj, list):
        buffer_list = obj
    else:
        buffer_list = [obj]
    for item in buffer_list:
        # it is a dict, so iterate through
        for key, value in item.items():
            # if value is a string, it must be a value to write to a node
            if isinstance(value, str):
                out_data.extend((indent_str, key, " = ", value, "\n"))
            # if it's a dict, it's another node, so recurse
            elif isinstance(value, dict):
                write_new_node(out_data, indent_str, indents, key, value)
            # if it's a list it could be multiple things
            else:
                # if the first element is a string, they will all be
                # it is a multi value node
                if isinstance(value[0], str):
                    # write out each value in the node
                    for res in value:
                        out_data.extend((indent_str, key, " = ", res, "\n"))
                # else just write out each subdict in the list
                else:
                    for subdict in value:
                        write_new_node(out_data, indent_str, indents, key, subdict)


def write_new_node(
    out_data: List[str],
    indent_str: str,
    indents: int,
    sect_name: str,
    value: Union[List, Dict],
) -> None:
    """Write a new node to the SFS"""
    # adds the header
    out_data.extend((indent_str, sect_name, "\n", indent_str, "{\n"))
    # adds data through recursion
    serialise_data(value, out_data, indents)
    # closes the block
    out_data.extend((indent_str, "}\n"))
