import * as notifications from '@aws-cdk/aws-codestarnotifications';
import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * Additional options to pass to the notification rule.
 *
 * @stability stable
 */
export interface RepositoryNotifyOnOptions extends notifications.NotificationRuleOptions {
    /**
     * A list of event types associated with this notification rule for CodeCommit repositories.
     *
     * For a complete list of event types and IDs, see Notification concepts in the Developer Tools Console User Guide.
     *
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#concepts-api
     * @stability stable
     */
    readonly events: RepositoryNotificationEvents[];
}
/**
 * @stability stable
 */
export interface IRepository extends IResource, notifications.INotificationRuleSource {
    /**
     * The ARN of this Repository.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryArn: string;
    /**
     * The human-visible name of this Repository.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryName: string;
    /**
     * The HTTP clone URL.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryCloneUrlHttp: string;
    /**
     * The SSH clone URL.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryCloneUrlSsh: string;
    /**
     * The HTTPS (GRC) clone URL.
     *
     * HTTPS (GRC) is the protocol to use with git-remote-codecommit (GRC).
     *
     * It is the recommended method for supporting connections made with federated
     * access, identity providers, and temporary credentials.
     *
     * @see https://docs.aws.amazon.com/codecommit/latest/userguide/setting-up-git-remote-codecommit.html
     * @stability stable
     */
    readonly repositoryCloneUrlGrc: string;
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit Repository State Change" event occurs.
     *
     * @stability stable
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is created (i.e. a new branch/tag is created) to the repository.
     *
     * @stability stable
     */
    onReferenceCreated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     *
     * @stability stable
     */
    onReferenceUpdated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is delete (i.e. a branch/tag is deleted) from the repository.
     *
     * @stability stable
     */
    onReferenceDeleted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     *
     * @stability stable
     */
    onPullRequestStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     *
     * @stability stable
     */
    onCommentOnPullRequest(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     *
     * @stability stable
     */
    onCommentOnCommit(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     *
     * @stability stable
     */
    onCommit(id: string, options?: OnCommitOptions): events.Rule;
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to pull this repository.
     *
     * @stability stable
     */
    grantPull(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to pull and push this repository.
     *
     * @stability stable
     */
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to read this repository.
     *
     * @stability stable
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Defines a CodeStar Notification rule triggered when the project events specified by you are emitted. Similar to `onEvent` API.
     *
     * You can also use the methods to define rules for the specific event emitted.
     * eg: `notifyOnPullRequstCreated`.
     *
     * @returns CodeStar Notifications rule associated with this repository.
     * @stability stable
     */
    notifyOn(id: string, target: notifications.INotificationRuleTarget, options: RepositoryNotifyOnOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a comment is made on a pull request.
     *
     * @stability stable
     */
    notifyOnPullRequestComment(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when an approval status is changed.
     *
     * @stability stable
     */
    notifyOnApprovalStatusChanged(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when an approval rule is overridden.
     *
     * @stability stable
     */
    notifyOnApprovalRuleOverridden(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a pull request is created.
     *
     * @stability stable
     */
    notifyOnPullRequestCreated(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a pull request is merged.
     *
     * @stability stable
     */
    notifiyOnPullRequestMerged(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a new branch or tag is created.
     *
     * @stability stable
     */
    notifyOnBranchOrTagCreated(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a branch or tag is deleted.
     *
     * @stability stable
     */
    notifyOnBranchOrTagDeleted(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
}
/**
 * Options for the onCommit() method.
 *
 * @stability stable
 */
export interface OnCommitOptions extends events.OnEventOptions {
    /**
     * The branch to monitor.
     *
     * @default - All branches
     * @stability stable
     */
    readonly branches?: string[];
}
/**
 * Represents a reference to a CodeCommit Repository.
 *
 * If you want to create a new Repository managed alongside your CDK code,
 * use the {@link Repository} class.
 *
 * If you want to reference an already existing Repository,
 * use the {@link Repository.import} method.
 */
declare abstract class RepositoryBase extends Resource implements IRepository {
    /** The ARN of this Repository. */
    abstract readonly repositoryArn: string;
    /** The human-visible name of this Repository. */
    abstract readonly repositoryName: string;
    /** The HTTP clone URL */
    abstract readonly repositoryCloneUrlHttp: string;
    /** The SSH clone URL */
    abstract readonly repositoryCloneUrlSsh: string;
    abstract readonly repositoryCloneUrlGrc: string;
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit Repository State Change" event occurs.
     *
     * @stability stable
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is created (i.e. a new branch/tag is created) to the repository.
     *
     * @stability stable
     */
    onReferenceCreated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     *
     * @stability stable
     */
    onReferenceUpdated(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a reference is delete (i.e. a branch/tag is deleted) from the repository.
     *
     * @stability stable
     */
    onReferenceDeleted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     *
     * @stability stable
     */
    onPullRequestStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     *
     * @stability stable
     */
    onCommentOnPullRequest(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     *
     * @stability stable
     */
    onCommentOnCommit(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     *
     * @stability stable
     */
    onCommit(id: string, options?: OnCommitOptions): events.Rule;
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to pull this repository.
     *
     * @stability stable
     */
    grantPull(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to pull and push this repository.
     *
     * @stability stable
     */
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to read this repository.
     *
     * @stability stable
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Defines a CodeStar Notification rule triggered when the project events specified by you are emitted. Similar to `onEvent` API.
     *
     * You can also use the methods to define rules for the specific event emitted.
     * eg: `notifyOnPullRequstCreated`.
     *
     * @stability stable
     */
    notifyOn(id: string, target: notifications.INotificationRuleTarget, options: RepositoryNotifyOnOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a comment is made on a pull request.
     *
     * @stability stable
     */
    notifyOnPullRequestComment(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when an approval status is changed.
     *
     * @stability stable
     */
    notifyOnApprovalStatusChanged(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when an approval rule is overridden.
     *
     * @stability stable
     */
    notifyOnApprovalRuleOverridden(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a pull request is created.
     *
     * @stability stable
     */
    notifyOnPullRequestCreated(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a pull request is merged.
     *
     * @stability stable
     */
    notifiyOnPullRequestMerged(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a new branch or tag is created.
     *
     * @stability stable
     */
    notifyOnBranchOrTagCreated(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar Notification rule which triggers when a branch or tag is deleted.
     *
     * @stability stable
     */
    notifyOnBranchOrTagDeleted(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Returns a source configuration for notification rule.
     *
     * @stability stable
     */
    bindAsNotificationRuleSource(_scope: Construct): notifications.NotificationRuleSourceConfig;
}
/**
 * @stability stable
 */
export interface RepositoryProps {
    /**
     * Name of the repository.
     *
     * This property is required for all CodeCommit repositories.
     *
     * @stability stable
     */
    readonly repositoryName: string;
    /**
     * A description of the repository.
     *
     * Use the description to identify the
     * purpose of the repository.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
}
/**
 * Provides a CodeCommit Repository.
 *
 * @stability stable
 */
export declare class Repository extends RepositoryBase {
    /**
     * Imports a codecommit repository.
     *
     * @param repositoryArn (e.g. `arn:aws:codecommit:us-east-1:123456789012:MyDemoRepo`).
     * @stability stable
     */
    static fromRepositoryArn(scope: Construct, id: string, repositoryArn: string): IRepository;
    /**
     * @stability stable
     */
    static fromRepositoryName(scope: Construct, id: string, repositoryName: string): IRepository;
    /**
     * The ARN of this Repository.
     *
     * @stability stable
     */
    readonly repositoryArn: string;
    /**
     * The human-visible name of this Repository.
     *
     * @stability stable
     */
    readonly repositoryName: string;
    /**
     * The HTTP clone URL.
     *
     * @stability stable
     */
    readonly repositoryCloneUrlHttp: string;
    /**
     * The SSH clone URL.
     *
     * @stability stable
     */
    readonly repositoryCloneUrlSsh: string;
    /**
     * The HTTPS (GRC) clone URL.
     *
     * HTTPS (GRC) is the protocol to use with git-remote-codecommit (GRC).
     *
     * It is the recommended method for supporting connections made with federated
     * access, identity providers, and temporary credentials.
     *
     * @stability stable
     */
    readonly repositoryCloneUrlGrc: string;
    private readonly triggers;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: RepositoryProps);
    /**
     * Create a trigger to notify another service to run actions on repository events.
     *
     * @param arn Arn of the resource that repository events will notify.
     * @param options Trigger options to run actions.
     * @stability stable
     */
    notify(arn: string, options?: RepositoryTriggerOptions): Repository;
}
/**
 * Creates for a repository trigger to an SNS topic or Lambda function.
 *
 * @stability stable
 */
export interface RepositoryTriggerOptions {
    /**
     * A name for the trigger.Triggers on a repository must have unique names.
     *
     * @stability stable
     */
    readonly name?: string;
    /**
     * The repository events for which AWS CodeCommit sends information to the target, which you specified in the DestinationArn property.If you don't specify events, the trigger runs for all repository events.
     *
     * @stability stable
     */
    readonly events?: RepositoryEventTrigger[];
    /**
     * The names of the branches in the AWS CodeCommit repository that contain events that you want to include in the trigger.
     *
     * If you don't specify at
     * least one branch, the trigger applies to all branches.
     *
     * @stability stable
     */
    readonly branches?: string[];
    /**
     * When an event is triggered, additional information that AWS CodeCommit includes when it sends information to the target.
     *
     * @stability stable
     */
    readonly customData?: string;
}
/**
 * Repository events that will cause the trigger to run actions in another service.
 *
 * @stability stable
 */
export declare enum RepositoryEventTrigger {
    /**
     * @stability stable
     */
    ALL = "all",
    /**
     * @stability stable
     */
    UPDATE_REF = "updateReference",
    /**
     * @stability stable
     */
    CREATE_REF = "createReference",
    /**
     * @stability stable
     */
    DELETE_REF = "deleteReference"
}
/**
 * List of event types for AWS CodeCommit.
 *
 * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-repositories
 * @stability stable
 */
export declare enum RepositoryNotificationEvents {
    /**
     * Trigger notication when comment made on commit.
     *
     * @stability stable
     */
    COMMIT_COMMENT = "codecommit-repository-comments-on-commits",
    /**
     * Trigger notification when comment made on pull request.
     *
     * @stability stable
     */
    PULL_REQUEST_COMMENT = "codecommit-repository-comments-on-pull-requests",
    /**
     * Trigger notification when approval status changed.
     *
     * @stability stable
     */
    APPROVAL_STATUS_CHANGED = "codecommit-repository-approvals-status-changed",
    /**
     * Trigger notifications when approval rule is overridden.
     *
     * @stability stable
     */
    APPROVAL_RULE_OVERRIDDEN = "codecommit-repository-approvals-rule-override",
    /**
     * Trigger notification when pull request created.
     *
     * @stability stable
     */
    PULL_REQUEST_CREATED = "codecommit-repository-pull-request-created",
    /**
     * Trigger notification when pull request source updated.
     *
     * @stability stable
     */
    PULL_REQUEST_SOURCE_UPDATED = "codecommit-repository-pull-request-source-updated",
    /**
     * Trigger notification when pull request status is changed.
     *
     * @stability stable
     */
    PULL_REQUEST_STATUS_CHANGED = "codecommit-repository-pull-request-status-changed",
    /**
     * Trigger notification when pull requset is merged.
     *
     * @stability stable
     */
    PULL_REQUEST_MERGED = "codecommit-repository-pull-request-merged",
    /**
     * Trigger notification when a branch or tag is created.
     *
     * @stability stable
     */
    BRANCH_OR_TAG_CREATED = "codecommit-repository-branches-and-tags-created",
    /**
     * Trigger notification when a branch or tag is deleted.
     *
     * @stability stable
     */
    BRANCH_OR_TAG_DELETED = "codecommit-repository-branches-and-tags-deleted",
    /**
     * Trigger notification when a branch or tag is updated.
     *
     * @stability stable
     */
    BRANCH_OR_TAG_UPDATED = "codecommit-repository-branches-and-tags-updated"
}
export {};
