"use strict";
const assert_1 = require("@aws-cdk/assert");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
// CDK parameters
const CONSTRUCT_NAME = 'MyTable';
// DynamoDB table parameters
const TABLE_NAME = 'MyTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: lib_1.AttributeType.STRING };
const TABLE_SORT_KEY = { name: 'sortKey', type: lib_1.AttributeType.NUMBER };
// DynamoDB global secondary index parameters
const GSI_NAME = 'MyGSI';
const GSI_PARTITION_KEY = { name: 'gsiHashKey', type: lib_1.AttributeType.STRING };
const GSI_SORT_KEY = { name: 'gsiSortKey', type: lib_1.AttributeType.BINARY };
const GSI_NON_KEY = 'gsiNonKey';
function* GSI_GENERATOR() {
    let n = 0;
    while (true) {
        const globalSecondaryIndexProps = {
            indexName: `${GSI_NAME}${n}`,
            partitionKey: { name: `${GSI_PARTITION_KEY.name}${n}`, type: GSI_PARTITION_KEY.type }
        };
        yield globalSecondaryIndexProps;
        n++;
    }
}
function* NON_KEY_ATTRIBUTE_GENERATOR(nonKeyPrefix) {
    let n = 0;
    while (true) {
        yield `${nonKeyPrefix}${n}`;
        n++;
    }
}
// DynamoDB local secondary index parameters
const LSI_NAME = 'MyLSI';
const LSI_SORT_KEY = { name: 'lsiSortKey', type: lib_1.AttributeType.NUMBER };
const LSI_NON_KEY = 'lsiNonKey';
function* LSI_GENERATOR() {
    let n = 0;
    while (true) {
        const localSecondaryIndexProps = {
            indexName: `${LSI_NAME}${n}`,
            sortKey: { name: `${LSI_SORT_KEY.name}${n}`, type: LSI_SORT_KEY.type }
        };
        yield localSecondaryIndexProps;
        n++;
    }
}
function testGrant(test, expectedActions, invocation) {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
    const user = new iam.User(stack, 'user');
    // WHEN
    invocation(user, table);
    // THEN
    const action = expectedActions.length > 1 ? expectedActions.map(a => `dynamodb:${a}`) : `dynamodb:${expectedActions[0]}`;
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        "PolicyDocument": {
            "Statement": [
                {
                    "Action": action,
                    "Effect": "Allow",
                    "Resource": [
                        { "Fn::GetAtt": ["mytable0324D45C", "Arn"] },
                        { "Ref": "AWS::NoValue" }
                    ]
                }
            ],
            "Version": "2012-10-17"
        },
        "Users": [{ "Ref": "user2C2B57AE" }]
    }));
    test.done();
}
module.exports = {
    'default properties': {
        'hash key only'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [{ AttributeName: 'hashKey', AttributeType: 'S' }],
                KeySchema: [{ AttributeName: 'hashKey', KeyType: 'HASH' }],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.RETAIN }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'removalPolicy is DESTROY'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, removalPolicy: core_1.RemovalPolicy.DESTROY });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.DELETE }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'hash + range key'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            }));
            test.done();
        },
        'hash + range key can also be specified in props'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            }));
            test.done();
        },
        'point-in-time recovery is not enabled'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            test.done();
        },
        'server-side encryption is not enabled'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            test.done();
        },
        'stream is not enabled'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            test.done();
        },
        'ttl is not enabled'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            test.done();
        },
        'can specify new and old images'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                readCapacity: 42,
                writeCapacity: 1337,
                stream: lib_1.StreamViewType.NEW_AND_OLD_IMAGES,
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                StreamSpecification: { StreamViewType: 'NEW_AND_OLD_IMAGES' },
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
                TableName: 'MyTable'
            }));
            test.done();
        },
        'can specify new images only'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                readCapacity: 42,
                writeCapacity: 1337,
                stream: lib_1.StreamViewType.NEW_IMAGE,
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                StreamSpecification: { StreamViewType: 'NEW_IMAGE' },
                TableName: 'MyTable',
            }));
            test.done();
        },
        'can specify old images only'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                readCapacity: 42,
                writeCapacity: 1337,
                stream: lib_1.StreamViewType.OLD_IMAGE,
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                StreamSpecification: { StreamViewType: 'OLD_IMAGE' },
                TableName: 'MyTable',
            }));
            test.done();
        }
    },
    'when specifying every property'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            pointInTimeRecovery: true,
            serverSideEncryption: true,
            billingMode: lib_1.BillingMode.PROVISIONED,
            stream: lib_1.StreamViewType.KEYS_ONLY,
            timeToLiveAttribute: 'timeToLive',
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: {
                ReadCapacityUnits: 42,
                WriteCapacityUnits: 1337
            },
            PointInTimeRecoverySpecification: { PointInTimeRecoveryEnabled: true },
            SSESpecification: { SSEEnabled: true },
            StreamSpecification: { StreamViewType: 'KEYS_ONLY' },
            TableName: 'MyTable',
            Tags: [{ Key: 'Environment', Value: 'Production' }],
            TimeToLiveSpecification: { AttributeName: 'timeToLive', Enabled: true }
        }));
        test.done();
    },
    'when specifying PAY_PER_REQUEST billing mode'(test) {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
            ],
            BillingMode: 'PAY_PER_REQUEST',
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
            ],
            TableName: 'MyTable',
        }));
        test.done();
    },
    'error when specifying read or write capacity with a PAY_PER_REQUEST billing mode'(test) {
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY,
            readCapacity: 1
        }));
        test.throws(() => new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY,
            writeCapacity: 1
        }));
        test.throws(() => new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY,
            readCapacity: 1,
            writeCapacity: 1
        }));
        test.done();
    },
    'when adding a global secondary index with hash key only'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY
        });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            readCapacity: 42,
            writeCapacity: 1337
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 }
                }
            ]
        }));
        test.done();
    },
    'when adding a global secondary index with hash + range key'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY
        });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.ALL,
            readCapacity: 42,
            writeCapacity: 1337
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
                { AttributeName: 'gsiSortKey', AttributeType: 'B' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                        { AttributeName: 'gsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 }
                }
            ]
        }));
        test.done();
    },
    'when adding a global secondary index with projection type KEYS_ONLY'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY
        });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.KEYS_ONLY,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
                { AttributeName: 'gsiSortKey', AttributeType: 'B' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                        { AttributeName: 'gsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { ProjectionType: 'KEYS_ONLY' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                }
            ]
        }));
        test.done();
    },
    'when adding a global secondary index with projection type INCLUDE'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value, gsiNonKeyAttributeGenerator.next().value],
            readCapacity: 42,
            writeCapacity: 1337
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
                { AttributeName: 'gsiSortKey', AttributeType: 'B' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                        { AttributeName: 'gsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { NonKeyAttributes: ['gsiNonKey0', 'gsiNonKey1'], ProjectionType: 'INCLUDE' },
                    ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 }
                }
            ]
        }));
        test.done();
    },
    'when adding a global secondary index on a table with PAY_PER_REQUEST billing mode'(test) {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY
        }).addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            ],
            BillingMode: 'PAY_PER_REQUEST',
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' }
                }
            ]
        }));
        test.done();
    },
    'error when adding a global secondary index with projection type INCLUDE, but without specifying non-key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE
        }), /non-key attributes should be specified when using INCLUDE projection type/);
        test.done();
    },
    'error when adding a global secondary index with projection type ALL, but with non-key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value]
        }), /non-key attributes should not be specified when not using INCLUDE projection type/);
        test.done();
    },
    'error when adding a global secondary index with projection type KEYS_ONLY, but with non-key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            projectionType: lib_1.ProjectionType.KEYS_ONLY,
            nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value]
        }), /non-key attributes should not be specified when not using INCLUDE projection type/);
        test.done();
    },
    'error when adding a global secondary index with projection type INCLUDE, but with more than 20 non-key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
        const gsiNonKeyAttributes = [];
        for (let i = 0; i < 21; i++) {
            gsiNonKeyAttributes.push(gsiNonKeyAttributeGenerator.next().value);
        }
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: gsiNonKeyAttributes
        }), /a maximum number of nonKeyAttributes across all of secondary indexes is 20/);
        test.done();
    },
    'error when adding a global secondary index with projection type INCLUDE, but with key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: [GSI_NON_KEY, TABLE_PARTITION_KEY.name]
            // tslint:disable-next-line:max-line-length
        }), /a key attribute, hashKey, is part of a list of non-key attributes, gsiNonKey,hashKey, which is not allowed since all key attributes are added automatically and this configuration causes stack creation failure/);
        test.done();
    },
    'error when adding a global secondary index with read or write capacity on a PAY_PER_REQUEST table'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST
        });
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            readCapacity: 1
        }));
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            writeCapacity: 1
        }));
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            readCapacity: 1,
            writeCapacity: 1
        }));
        test.done();
    },
    'when adding multiple global secondary indexes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiGenerator = GSI_GENERATOR();
        for (let i = 0; i < 5; i++) {
            table.addGlobalSecondaryIndex(gsiGenerator.next().value);
        }
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey0', AttributeType: 'S' },
                { AttributeName: 'gsiHashKey1', AttributeType: 'S' },
                { AttributeName: 'gsiHashKey2', AttributeType: 'S' },
                { AttributeName: 'gsiHashKey3', AttributeType: 'S' },
                { AttributeName: 'gsiHashKey4', AttributeType: 'S' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI0',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey0', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
                {
                    IndexName: 'MyGSI1',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey1', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
                {
                    IndexName: 'MyGSI2',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey2', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
                {
                    IndexName: 'MyGSI3',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey3', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
                {
                    IndexName: 'MyGSI4',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey4', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
            ]
        }));
        test.done();
    },
    'when adding a global secondary index without specifying read and write capacity'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                }
            ]
        }));
        test.done();
    },
    'when adding a local secondary index with hash + range key'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        table.addLocalSecondaryIndex({
            indexName: LSI_NAME,
            sortKey: LSI_SORT_KEY,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'lsiSortKey', AttributeType: 'N' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            LocalSecondaryIndexes: [
                {
                    IndexName: 'MyLSI',
                    KeySchema: [
                        { AttributeName: 'hashKey', KeyType: 'HASH' },
                        { AttributeName: 'lsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { ProjectionType: 'ALL' },
                }
            ],
        }));
        test.done();
    },
    'when adding a local secondary index with projection type KEYS_ONLY'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        table.addLocalSecondaryIndex({
            indexName: LSI_NAME,
            sortKey: LSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.KEYS_ONLY
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'lsiSortKey', AttributeType: 'N' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            LocalSecondaryIndexes: [
                {
                    IndexName: 'MyLSI',
                    KeySchema: [
                        { AttributeName: 'hashKey', KeyType: 'HASH' },
                        { AttributeName: 'lsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { ProjectionType: 'KEYS_ONLY' },
                }
            ],
        }));
        test.done();
    },
    'when adding a local secondary index with projection type INCLUDE'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const lsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(LSI_NON_KEY);
        table.addLocalSecondaryIndex({
            indexName: LSI_NAME,
            sortKey: LSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: [lsiNonKeyAttributeGenerator.next().value, lsiNonKeyAttributeGenerator.next().value]
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'lsiSortKey', AttributeType: 'N' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            LocalSecondaryIndexes: [
                {
                    IndexName: 'MyLSI',
                    KeySchema: [
                        { AttributeName: 'hashKey', KeyType: 'HASH' },
                        { AttributeName: 'lsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { NonKeyAttributes: ['lsiNonKey0', 'lsiNonKey1'], ProjectionType: 'INCLUDE' },
                }
            ],
        }));
        test.done();
    },
    'error when adding more than 5 local secondary indexes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const lsiGenerator = LSI_GENERATOR();
        for (let i = 0; i < 5; i++) {
            table.addLocalSecondaryIndex(lsiGenerator.next().value);
        }
        test.throws(() => table.addLocalSecondaryIndex(lsiGenerator.next().value), /a maximum number of local secondary index per table is 5/);
        test.done();
    },
    'error when adding a local secondary index with the name of a global secondary index'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        table.addGlobalSecondaryIndex({
            indexName: 'SecondaryIndex',
            partitionKey: GSI_PARTITION_KEY
        });
        test.throws(() => table.addLocalSecondaryIndex({
            indexName: 'SecondaryIndex',
            sortKey: LSI_SORT_KEY
        }), /a duplicate index name, SecondaryIndex, is not allowed/);
        test.done();
    },
    'error when validating construct if a local secondary index exists without a sort key of the table'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
        table.addLocalSecondaryIndex({
            indexName: LSI_NAME,
            sortKey: LSI_SORT_KEY
        });
        const errors = core_1.ConstructNode.validate(table.node);
        test.strictEqual(1, errors.length);
        test.strictEqual('a sort key of the table must be specified to add local secondary indexes', errors[0].message);
        test.done();
    },
    'can enable Read AutoScaling'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        // WHEN
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            MaxCapacity: 500,
            MinCapacity: 50,
            ScalableDimension: 'dynamodb:table:ReadCapacityUnits',
            ServiceNamespace: 'dynamodb'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: 'TargetTrackingScaling',
            TargetTrackingScalingPolicyConfiguration: {
                PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBReadCapacityUtilization' },
                TargetValue: 75
            }
        }));
        test.done();
    },
    'can enable Write AutoScaling'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        // WHEN
        table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            MaxCapacity: 500,
            MinCapacity: 50,
            ScalableDimension: 'dynamodb:table:WriteCapacityUnits',
            ServiceNamespace: 'dynamodb'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: 'TargetTrackingScaling',
            TargetTrackingScalingPolicyConfiguration: {
                PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBWriteCapacityUtilization' },
                TargetValue: 75
            }
        }));
        test.done();
    },
    'cannot enable AutoScaling twice on the same property'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
        // WHEN
        test.throws(() => {
            table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
        });
        test.done();
    },
    'error when enabling AutoScaling on the PAY_PER_REQUEST table'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { billingMode: lib_1.BillingMode.PAY_PER_REQUEST, partitionKey: TABLE_PARTITION_KEY });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY
        });
        // WHEN
        test.throws(() => {
            table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
        });
        test.throws(() => {
            table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 });
        });
        test.throws(() => table.autoScaleGlobalSecondaryIndexReadCapacity(GSI_NAME, {
            minCapacity: 1,
            maxCapacity: 5
        }));
        test.done();
    },
    'error when specifying Read Auto Scaling with invalid scalingTargetValue < 10'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        // THEN
        test.throws(() => {
            table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 5 });
        });
        test.done();
    },
    'error when specifying Read Auto Scaling with invalid minimumCapacity'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        // THEN
        test.throws(() => table.autoScaleReadCapacity({ minCapacity: 10, maxCapacity: 5 }));
        test.done();
    },
    'can autoscale on a schedule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            readCapacity: 42,
            writeCapacity: 1337,
            partitionKey: { name: 'Hash', type: lib_1.AttributeType.STRING }
        });
        // WHEN
        const scaling = table.autoScaleReadCapacity({ minCapacity: 1, maxCapacity: 100 });
        scaling.scaleOnSchedule('SaveMoneyByNotScalingUp', {
            schedule: appscaling.Schedule.cron({}),
            maxCapacity: 10
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            ScheduledActions: [
                {
                    ScalableTargetAction: { "MaxCapacity": 10 },
                    Schedule: "cron(* * * * ? *)",
                    ScheduledActionName: "SaveMoneyByNotScalingUp"
                }
            ]
        }));
        test.done();
    },
    'metrics': {
        'Can use metricConsumedReadCapacityUnits on a Dynamodb Table'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'Table', {
                partitionKey: { name: 'id', type: lib_1.AttributeType.STRING }
            });
            // THEN
            test.deepEqual(stack.resolve(table.metricConsumedReadCapacityUnits()), {
                period: { amount: 5, unit: { label: 'minutes', inSeconds: 60 } },
                dimensions: { TableName: { Ref: 'TableCD117FA1' } },
                namespace: 'AWS/DynamoDB',
                metricName: 'ConsumedReadCapacityUnits',
                statistic: 'Sum',
            });
            test.done();
        },
        'Can use metricConsumedWriteCapacityUnits on a Dynamodb Table'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'Table', {
                partitionKey: { name: 'id', type: lib_1.AttributeType.STRING }
            });
            // THEN
            test.deepEqual(stack.resolve(table.metricConsumedWriteCapacityUnits()), {
                period: { amount: 5, unit: { label: 'minutes', inSeconds: 60 } },
                dimensions: { TableName: { Ref: 'TableCD117FA1' } },
                namespace: 'AWS/DynamoDB',
                metricName: 'ConsumedWriteCapacityUnits',
                statistic: 'Sum',
            });
            test.done();
        },
        'Can use metricSystemErrors on a Dynamodb Table'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'Table', {
                partitionKey: { name: 'id', type: lib_1.AttributeType.STRING }
            });
            // THEN
            test.deepEqual(stack.resolve(table.metricSystemErrors()), {
                period: { amount: 5, unit: { label: 'minutes', inSeconds: 60 } },
                dimensions: { TableName: { Ref: 'TableCD117FA1' } },
                namespace: 'AWS/DynamoDB',
                metricName: 'SystemErrors',
                statistic: 'Sum',
            });
            test.done();
        },
        'Can use metricUserErrors on a Dynamodb Table'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'Table', {
                partitionKey: { name: 'id', type: lib_1.AttributeType.STRING }
            });
            // THEN
            test.deepEqual(stack.resolve(table.metricUserErrors()), {
                period: { amount: 5, unit: { label: 'minutes', inSeconds: 60 } },
                dimensions: { TableName: { Ref: 'TableCD117FA1' } },
                namespace: 'AWS/DynamoDB',
                metricName: 'UserErrors',
                statistic: 'Sum',
            });
            test.done();
        },
        'Can use metricConditionalCheckFailedRequests on a Dynamodb Table'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'Table', {
                partitionKey: { name: 'id', type: lib_1.AttributeType.STRING }
            });
            // THEN
            test.deepEqual(stack.resolve(table.metricConditionalCheckFailedRequests()), {
                period: { amount: 5, unit: { label: 'minutes', inSeconds: 60 } },
                dimensions: { TableName: { Ref: 'TableCD117FA1' } },
                namespace: 'AWS/DynamoDB',
                metricName: 'ConditionalCheckFailedRequests',
                statistic: 'Sum',
            });
            test.done();
        },
        'Can use metricSuccessfulRequestLatency on a Dynamodb Table'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'Table', {
                partitionKey: { name: 'id', type: lib_1.AttributeType.STRING }
            });
            // THEN
            test.deepEqual(stack.resolve(table.metricSuccessfulRequestLatency()), {
                period: { amount: 5, unit: { label: 'minutes', inSeconds: 60 } },
                dimensions: { TableName: { Ref: 'TableCD117FA1' } },
                namespace: 'AWS/DynamoDB',
                metricName: 'SuccessfulRequestLatency',
                statistic: 'Average',
            });
            test.done();
        },
    },
    'grants': {
        '"grant" allows adding arbitrary actions associated with this table resource'(test) {
            testGrant(test, ['action1', 'action2'], (p, t) => t.grant(p, 'dynamodb:action1', 'dynamodb:action2'));
        },
        '"grantReadData" allows the principal to read data from the table'(test) {
            testGrant(test, ['BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan'], (p, t) => t.grantReadData(p));
        },
        '"grantWriteData" allows the principal to write data to the table'(test) {
            testGrant(test, [
                'BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'
            ], (p, t) => t.grantWriteData(p));
        },
        '"grantReadWriteData" allows the principal to read/write data'(test) {
            testGrant(test, [
                'BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan',
                'BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'
            ], (p, t) => t.grantReadWriteData(p));
        },
        '"grantFullAccess" allows the principal to perform any action on the table ("*")'(test) {
            testGrant(test, ['*'], (p, t) => t.grantFullAccess(p));
        },
        '"Table.grantListStreams" allows principal to list all streams'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const user = new iam.User(stack, 'user');
            // WHEN
            lib_1.Table.grantListStreams(user);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "dynamodb:ListStreams",
                            "Effect": "Allow",
                            "Resource": "*"
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Users": [{ "Ref": "user2C2B57AE" }]
            }));
            test.done();
        },
        '"grantTableListStreams" should fail if streaming is not enabled on table"'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                }
            });
            const user = new iam.User(stack, 'user');
            // WHEN
            test.throws(() => table.grantTableListStreams(user), /DynamoDB Streams must be enabled on the table my-table/);
            test.done();
        },
        '"grantTableListStreams" allows principal to list all streams for this table'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                },
                stream: lib_1.StreamViewType.NEW_IMAGE
            });
            const user = new iam.User(stack, 'user');
            // WHEN
            table.grantTableListStreams(user);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "dynamodb:ListStreams",
                            "Effect": "Allow",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["mytable0324D45C", "Arn"] }, "/stream/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Users": [{ "Ref": "user2C2B57AE" }]
            }));
            test.done();
        },
        '"grantStreamRead" should fail if streaming is not enabled on table"'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                }
            });
            const user = new iam.User(stack, 'user');
            // WHEN
            test.throws(() => table.grantStreamRead(user), /DynamoDB Streams must be enabled on the table my-table/);
            test.done();
        },
        '"grantStreamRead" allows principal to read and describe the table stream"'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                },
                stream: lib_1.StreamViewType.NEW_IMAGE
            });
            const user = new iam.User(stack, 'user');
            // WHEN
            table.grantStreamRead(user);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "dynamodb:ListStreams",
                            "Effect": "Allow",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["mytable0324D45C", "Arn"] }, "/stream/*"]] }
                        },
                        {
                            "Action": [
                                "dynamodb:DescribeStream",
                                "dynamodb:GetRecords",
                                "dynamodb:GetShardIterator"
                            ],
                            "Effect": "Allow",
                            "Resource": {
                                "Fn::GetAtt": [
                                    "mytable0324D45C",
                                    "StreamArn"
                                ]
                            }
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Users": [{ "Ref": "user2C2B57AE" }]
            }));
            test.done();
        },
        'if table has an index grant gives access to the index'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
            table.addGlobalSecondaryIndex({ indexName: 'MyIndex', partitionKey: { name: 'Age', type: lib_1.AttributeType.NUMBER } });
            const user = new iam.User(stack, 'user');
            // WHEN
            table.grantReadData(user);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                'dynamodb:BatchGetItem',
                                'dynamodb:GetRecords',
                                'dynamodb:GetShardIterator',
                                'dynamodb:Query',
                                'dynamodb:GetItem',
                                'dynamodb:Scan'
                            ],
                            "Effect": "Allow",
                            "Resource": [
                                { "Fn::GetAtt": ["mytable0324D45C", "Arn"] },
                                { "Fn::Join": ["", [{ "Fn::GetAtt": ["mytable0324D45C", "Arn"] }, "/index/*"]] }
                            ]
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Users": [{ "Ref": "user2C2B57AE" }]
            }));
            test.done();
        }
    },
    'import': {
        'report error when importing an external/existing table from invalid arn missing resource name'(test) {
            const stack = new core_1.Stack();
            const tableArn = 'arn:aws:dynamodb:us-east-1::table/';
            // WHEN
            test.throws(() => lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn), /ARN for DynamoDB table must be in the form: .../);
            test.done();
        },
        'static import(ref) allows importing an external/existing table from arn'(test) {
            const stack = new core_1.Stack();
            const tableArn = 'arn:aws:dynamodb:us-east-1:11111111:table/MyTable';
            const table = lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn);
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            table.grantReadData(role);
            // it is possible to obtain a permission statement for a ref
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                'dynamodb:BatchGetItem',
                                'dynamodb:GetRecords',
                                'dynamodb:GetShardIterator',
                                'dynamodb:Query',
                                'dynamodb:GetItem',
                                'dynamodb:Scan'
                            ],
                            "Effect": "Allow",
                            "Resource": [
                                tableArn,
                                { "Ref": "AWS::NoValue" }
                            ]
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "PolicyName": 'NewRoleDefaultPolicy90E8F49D',
                "Roles": [{ "Ref": 'NewRole99763075' }]
            }));
            test.deepEqual(table.tableArn, tableArn);
            test.deepEqual(stack.resolve(table.tableName), 'MyTable');
            test.done();
        },
        'static import(ref) allows importing an external/existing table from table name'(test) {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const table = lib_1.Table.fromTableName(stack, 'ImportedTable', tableName);
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            table.grantReadWriteData(role);
            // it is possible to obtain a permission statement for a ref
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                'dynamodb:BatchGetItem',
                                'dynamodb:GetRecords',
                                'dynamodb:GetShardIterator',
                                'dynamodb:Query',
                                'dynamodb:GetItem',
                                'dynamodb:Scan',
                                'dynamodb:BatchWriteItem',
                                'dynamodb:PutItem',
                                'dynamodb:UpdateItem',
                                'dynamodb:DeleteItem'
                            ],
                            "Effect": "Allow",
                            "Resource": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition"
                                            },
                                            ":dynamodb:",
                                            {
                                                "Ref": "AWS::Region"
                                            },
                                            ":",
                                            {
                                                "Ref": "AWS::AccountId"
                                            },
                                            ":table/MyTable"
                                        ]
                                    ]
                                },
                                {
                                    "Ref": "AWS::NoValue"
                                }
                            ]
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "PolicyName": 'NewRoleDefaultPolicy90E8F49D',
                "Roles": [{ "Ref": 'NewRole99763075' }]
            }));
            test.deepEqual(table.tableArn, 'arn:${Token[AWS::Partition.3]}:dynamodb:${Token[AWS::Region.4]}:${Token[AWS::AccountId.0]}:table/MyTable');
            test.deepEqual(stack.resolve(table.tableName), tableName);
            test.done();
        },
    },
    'global': {
        'create replicas'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Table(stack, 'Table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                },
                replicationRegions: [
                    'eu-west-2',
                    'eu-central-1'
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::DynamoDBReplica', {
                Properties: {
                    ServiceToken: {
                        'Fn::GetAtt': [
                            'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                            'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn'
                        ]
                    },
                    TableName: {
                        Ref: 'TableCD117FA1'
                    },
                    Region: 'eu-west-2'
                },
                Condition: 'TableStackRegionNotEqualseuwest2A03859E7'
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('Custom::DynamoDBReplica', {
                Properties: {
                    ServiceToken: {
                        'Fn::GetAtt': [
                            'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                            'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn'
                        ]
                    },
                    TableName: {
                        Ref: 'TableCD117FA1'
                    },
                    Region: 'eu-central-1'
                },
                Condition: 'TableStackRegionNotEqualseucentral199D46FC0'
            }, assert_1.ResourcePart.CompleteDefinition));
            test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Conditions, {
                TableStackRegionNotEqualseuwest2A03859E7: {
                    'Fn::Not': [
                        { 'Fn::Equals': ['eu-west-2', { Ref: 'AWS::Region' }] }
                    ]
                },
                TableStackRegionNotEqualseucentral199D46FC0: {
                    'Fn::Not': [
                        { 'Fn::Equals': ['eu-central-1', { Ref: 'AWS::Region' }] }
                    ]
                }
            });
            test.done();
        },
        'throws with PROVISIONED billing mode'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // THEN
            test.throws(() => new lib_1.Table(stack, 'Table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                },
                replicationRegions: [
                    'eu-west-2',
                    'eu-central-1'
                ],
                billingMode: lib_1.BillingMode.PROVISIONED,
            }), /`PAY_PER_REQUEST`/);
            test.done();
        },
        'throws when stream is set and not set to NEW_AND_OLD_IMAGES'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // THEN
            test.throws(() => new lib_1.Table(stack, 'Table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                },
                replicationRegions: [
                    'eu-west-2',
                    'eu-central-1'
                ],
                stream: lib_1.StreamViewType.OLD_IMAGE,
            }), /`NEW_AND_OLD_IMAGES`/);
            test.done();
        },
        'throws with replica in same region as stack'(test) {
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'Stack', {
                env: { region: 'us-east-1' }
            });
            // THEN
            test.throws(() => new lib_1.Table(stack, 'Table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                },
                replicationRegions: [
                    'eu-west-1',
                    'us-east-1',
                    'eu-west-2',
                ],
            }), /`replicationRegions` cannot include the region where this stack is deployed/);
            test.done();
        },
        'no conditions when region is known'(test) {
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'Stack', {
                env: { region: 'eu-west-1' }
            });
            // WHEN
            new lib_1.Table(stack, 'Table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                },
                replicationRegions: [
                    'eu-west-2',
                    'eu-central-1'
                ],
            });
            // THEN
            test.equal(assert_1.SynthUtils.toCloudFormation(stack).Conditions, undefined);
            test.done();
        },
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5keW5hbW9kYi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuZHluYW1vZGIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLDRDQUFpRjtBQUNqRixrRUFBa0U7QUFDbEUsd0NBQXdDO0FBQ3hDLHdDQUFpRztBQUVqRyxnQ0FTZ0I7QUFFaEIsMkNBQTJDO0FBRTNDLGlCQUFpQjtBQUNqQixNQUFNLGNBQWMsR0FBRyxTQUFTLENBQUM7QUFFakMsNEJBQTRCO0FBQzVCLE1BQU0sVUFBVSxHQUFHLFNBQVMsQ0FBQztBQUM3QixNQUFNLG1CQUFtQixHQUFjLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsbUJBQWEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztBQUN2RixNQUFNLGNBQWMsR0FBYyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTSxFQUFFLENBQUM7QUFFbEYsNkNBQTZDO0FBQzdDLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQztBQUN6QixNQUFNLGlCQUFpQixHQUFjLEVBQUUsSUFBSSxFQUFFLFlBQVksRUFBRSxJQUFJLEVBQUUsbUJBQWEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztBQUN4RixNQUFNLFlBQVksR0FBYyxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTSxFQUFFLENBQUM7QUFDbkYsTUFBTSxXQUFXLEdBQUcsV0FBVyxDQUFDO0FBQ2hDLFFBQVEsQ0FBQyxDQUFDLGFBQWE7SUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ1YsT0FBTyxJQUFJLEVBQUU7UUFDWCxNQUFNLHlCQUF5QixHQUE4QjtZQUMzRCxTQUFTLEVBQUUsR0FBRyxRQUFRLEdBQUcsQ0FBQyxFQUFFO1lBQzVCLFlBQVksRUFBRSxFQUFFLElBQUksRUFBRSxHQUFHLGlCQUFpQixDQUFDLElBQUksR0FBRyxDQUFDLEVBQUUsRUFBRSxJQUFJLEVBQUUsaUJBQWlCLENBQUMsSUFBSSxFQUFFO1NBQ3RGLENBQUM7UUFDRixNQUFNLHlCQUF5QixDQUFDO1FBQ2hDLENBQUMsRUFBRSxDQUFDO0tBQ0w7QUFDSCxDQUFDO0FBQ0QsUUFBUSxDQUFDLENBQUMsMkJBQTJCLENBQUMsWUFBb0I7SUFDeEQsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ1YsT0FBTyxJQUFJLEVBQUU7UUFDWCxNQUFNLEdBQUcsWUFBWSxHQUFHLENBQUMsRUFBRSxDQUFDO1FBQzVCLENBQUMsRUFBRSxDQUFDO0tBQ0w7QUFDSCxDQUFDO0FBRUQsNENBQTRDO0FBQzVDLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQztBQUN6QixNQUFNLFlBQVksR0FBYyxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTSxFQUFFLENBQUM7QUFDbkYsTUFBTSxXQUFXLEdBQUcsV0FBVyxDQUFDO0FBQ2hDLFFBQVEsQ0FBQyxDQUFDLGFBQWE7SUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ1YsT0FBTyxJQUFJLEVBQUU7UUFDWCxNQUFNLHdCQUF3QixHQUE2QjtZQUN6RCxTQUFTLEVBQUUsR0FBRyxRQUFRLEdBQUcsQ0FBQyxFQUFFO1lBQzVCLE9BQU8sRUFBRSxFQUFFLElBQUksRUFBRyxHQUFHLFlBQVksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxFQUFFLEVBQUUsSUFBSSxFQUFFLFlBQVksQ0FBQyxJQUFJLEVBQUU7U0FDeEUsQ0FBQztRQUNGLE1BQU0sd0JBQXdCLENBQUM7UUFDL0IsQ0FBQyxFQUFFLENBQUM7S0FDTDtBQUNILENBQUM7QUFnb0RELFNBQVMsU0FBUyxDQUFDLElBQVUsRUFBRSxlQUF5QixFQUFFLFVBQXdEO0lBQ2hILFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxZQUFZLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRyxtQkFBYSxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMxRyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBRXpDLE9BQU87SUFDUCxVQUFVLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXhCLE9BQU87SUFDUCxNQUFNLE1BQU0sR0FBRyxlQUFlLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsWUFBWSxlQUFlLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUN6SCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7UUFDaEQsZ0JBQWdCLEVBQUU7WUFDaEIsV0FBVyxFQUFFO2dCQUNYO29CQUNFLFFBQVEsRUFBRSxNQUFNO29CQUNoQixRQUFRLEVBQUUsT0FBTztvQkFDakIsVUFBVSxFQUFFO3dCQUNWLEVBQUUsWUFBWSxFQUFFLENBQUUsaUJBQWlCLEVBQUUsS0FBSyxDQUFFLEVBQUU7d0JBQzlDLEVBQUUsS0FBSyxFQUFHLGNBQWMsRUFBRTtxQkFDM0I7aUJBQ0Y7YUFDRjtZQUNELFNBQVMsRUFBRSxZQUFZO1NBQ3hCO1FBQ0QsT0FBTyxFQUFFLENBQUUsRUFBRSxLQUFLLEVBQUUsY0FBYyxFQUFFLENBQUU7S0FDdkMsQ0FBQyxDQUFDLENBQUM7SUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7QUFDZCxDQUFDO0FBMXBERCxpQkFBUztJQUNQLG9CQUFvQixFQUFFO1FBQ3BCLGVBQWUsQ0FBQyxJQUFVO1lBQ3hCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxDQUFDLENBQUM7WUFFeEUsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO2dCQUNwRCxvQkFBb0IsRUFBRSxDQUFDLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFLENBQUM7Z0JBQ3hFLFNBQVMsRUFBRSxDQUFDLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFLENBQUM7Z0JBQzFELHFCQUFxQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxFQUFFLGtCQUFrQixFQUFFLENBQUMsRUFBRTthQUN2RSxDQUFDLENBQUMsQ0FBQztZQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFBRSxFQUFFLGNBQWMsRUFBRSx3QkFBaUIsQ0FBQyxNQUFNLEVBQUUsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUV0SSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsMEJBQTBCLENBQUMsSUFBVTtZQUNuQyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsRUFBRSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsYUFBYSxFQUFFLG9CQUFhLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztZQUU5RyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUUsRUFBRSxjQUFjLEVBQUUsd0JBQWlCLENBQUMsTUFBTSxFQUFFLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7WUFFdEksSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELGtCQUFrQixDQUFDLElBQVU7WUFDM0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO2dCQUMvQixZQUFZLEVBQUUsbUJBQW1CO2dCQUNqQyxPQUFPLEVBQUUsY0FBYzthQUN4QixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7Z0JBQzlDLG9CQUFvQixFQUFFO29CQUNwQixFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtvQkFDaEQsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7aUJBQ2pEO2dCQUNELFNBQVMsRUFBRTtvQkFDVCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtvQkFDN0MsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUU7aUJBQy9DO2dCQUNELHFCQUFxQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxFQUFFLGtCQUFrQixFQUFFLENBQUMsRUFBRTthQUM3RSxDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxpREFBaUQsQ0FBQyxJQUFVO1lBQzFELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFFMUIsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtnQkFDL0IsWUFBWSxFQUFFLG1CQUFtQjtnQkFDakMsT0FBTyxFQUFFLGNBQWM7YUFDeEIsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUNsRDtnQkFDRSxvQkFBb0IsRUFBRTtvQkFDcEIsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7b0JBQ2hELEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2lCQUNqRDtnQkFDRCxTQUFTLEVBQUU7b0JBQ1QsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7b0JBQzdDLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFO2lCQUMvQztnQkFDRCxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7YUFDdkUsQ0FBQyxDQUFDLENBQUM7WUFFTixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsdUNBQXVDLENBQUMsSUFBVTtZQUNoRCxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQy9CLFlBQVksRUFBRSxtQkFBbUI7Z0JBQ2pDLE9BQU8sRUFBRSxjQUFjO2FBQ3hCLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFDbEQ7Z0JBQ0Usb0JBQW9CLEVBQUU7b0JBQ3BCLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO29CQUNoRCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtpQkFDakQ7Z0JBQ0QsU0FBUyxFQUFFO29CQUNULEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO29CQUM3QyxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRTtpQkFDL0M7Z0JBQ0QscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxFQUFFO2FBQ3ZFLENBQ0YsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHVDQUF1QyxDQUFDLElBQVU7WUFDaEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO2dCQUMvQixZQUFZLEVBQUUsbUJBQW1CO2dCQUNqQyxPQUFPLEVBQUUsY0FBYzthQUN4QixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQ2xEO2dCQUNFLG9CQUFvQixFQUFFO29CQUNwQixFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtvQkFDaEQsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7aUJBQ2pEO2dCQUNELFNBQVMsRUFBRTtvQkFDVCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtvQkFDN0MsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUU7aUJBQy9DO2dCQUNELHFCQUFxQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxFQUFFLGtCQUFrQixFQUFFLENBQUMsRUFBRTthQUN2RSxDQUNGLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx1QkFBdUIsQ0FBQyxJQUFVO1lBQ2hDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtnQkFDL0IsWUFBWSxFQUFFLG1CQUFtQjtnQkFDakMsT0FBTyxFQUFFLGNBQWM7YUFDeEIsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUNsRDtnQkFDRSxvQkFBb0IsRUFBRTtvQkFDcEIsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7b0JBQ2hELEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2lCQUNqRDtnQkFDRCxTQUFTLEVBQUU7b0JBQ1QsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7b0JBQzdDLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFO2lCQUMvQztnQkFDRCxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7YUFDdkUsQ0FDRixDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsb0JBQW9CLENBQUMsSUFBVTtZQUM3QixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQy9CLFlBQVksRUFBRSxtQkFBbUI7Z0JBQ2pDLE9BQU8sRUFBRSxjQUFjO2FBQ3hCLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFDbEQ7Z0JBQ0Usb0JBQW9CLEVBQUU7b0JBQ3BCLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO29CQUNoRCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtpQkFDakQ7Z0JBQ0QsU0FBUyxFQUFFO29CQUNULEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO29CQUM3QyxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRTtpQkFDL0M7Z0JBQ0QscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxFQUFFO2FBQ3ZFLENBQ0YsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELGdDQUFnQyxDQUFDLElBQVU7WUFDekMsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUUxQixJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO2dCQUMvQixTQUFTLEVBQUUsVUFBVTtnQkFDckIsWUFBWSxFQUFFLEVBQUU7Z0JBQ2hCLGFBQWEsRUFBRSxJQUFJO2dCQUNuQixNQUFNLEVBQUUsb0JBQWMsQ0FBQyxrQkFBa0I7Z0JBQ3pDLFlBQVksRUFBRSxtQkFBbUI7Z0JBQ2pDLE9BQU8sRUFBRSxjQUFjO2FBQ3hCLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFDbEQ7Z0JBQ0Usb0JBQW9CLEVBQUU7b0JBQ3BCLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO29CQUNoRCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtpQkFDakQ7Z0JBQ0QsbUJBQW1CLEVBQUUsRUFBRSxjQUFjLEVBQUUsb0JBQW9CLEVBQUU7Z0JBQzdELFNBQVMsRUFBRTtvQkFDVCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtvQkFDN0MsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUU7aUJBQy9DO2dCQUNELHFCQUFxQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsRUFBRSxFQUFFLGtCQUFrQixFQUFFLElBQUksRUFBRTtnQkFDMUUsU0FBUyxFQUFFLFNBQVM7YUFDckIsQ0FDRixDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNkJBQTZCLENBQUMsSUFBVTtZQUN0QyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQy9CLFNBQVMsRUFBRSxVQUFVO2dCQUNyQixZQUFZLEVBQUUsRUFBRTtnQkFDaEIsYUFBYSxFQUFFLElBQUk7Z0JBQ25CLE1BQU0sRUFBRSxvQkFBYyxDQUFDLFNBQVM7Z0JBQ2hDLFlBQVksRUFBRSxtQkFBbUI7Z0JBQ2pDLE9BQU8sRUFBRSxjQUFjO2FBQ3hCLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFDbEQ7Z0JBQ0UsU0FBUyxFQUFFO29CQUNULEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO29CQUM3QyxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRTtpQkFDL0M7Z0JBQ0QscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxFQUFFLEVBQUUsa0JBQWtCLEVBQUUsSUFBSSxFQUFFO2dCQUMxRSxvQkFBb0IsRUFBRTtvQkFDcEIsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7b0JBQ2hELEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2lCQUNqRDtnQkFDRCxtQkFBbUIsRUFBRSxFQUFFLGNBQWMsRUFBRSxXQUFXLEVBQUU7Z0JBQ3BELFNBQVMsRUFBRSxTQUFTO2FBQ3JCLENBQ0YsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDZCQUE2QixDQUFDLElBQVU7WUFDdEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUUxQixJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO2dCQUMvQixTQUFTLEVBQUUsVUFBVTtnQkFDckIsWUFBWSxFQUFFLEVBQUU7Z0JBQ2hCLGFBQWEsRUFBRSxJQUFJO2dCQUNuQixNQUFNLEVBQUUsb0JBQWMsQ0FBQyxTQUFTO2dCQUNoQyxZQUFZLEVBQUUsbUJBQW1CO2dCQUNqQyxPQUFPLEVBQUUsY0FBYzthQUN4QixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQ2xEO2dCQUNFLFNBQVMsRUFBRTtvQkFDVCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtvQkFDN0MsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUU7aUJBQy9DO2dCQUNELHFCQUFxQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsRUFBRSxFQUFFLGtCQUFrQixFQUFFLElBQUksRUFBRTtnQkFDMUUsb0JBQW9CLEVBQUU7b0JBQ3BCLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO29CQUNoRCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtpQkFDakQ7Z0JBQ0QsbUJBQW1CLEVBQUUsRUFBRSxjQUFjLEVBQUUsV0FBVyxFQUFFO2dCQUNwRCxTQUFTLEVBQUUsU0FBUzthQUNyQixDQUNGLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELGdDQUFnQyxDQUFDLElBQVU7UUFDekMsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQzdDLFNBQVMsRUFBRSxVQUFVO1lBQ3JCLFlBQVksRUFBRSxFQUFFO1lBQ2hCLGFBQWEsRUFBRSxJQUFJO1lBQ25CLG1CQUFtQixFQUFFLElBQUk7WUFDekIsb0JBQW9CLEVBQUUsSUFBSTtZQUMxQixXQUFXLEVBQUUsaUJBQVcsQ0FBQyxXQUFXO1lBQ3BDLE1BQU0sRUFBRSxvQkFBYyxDQUFDLFNBQVM7WUFDaEMsbUJBQW1CLEVBQUUsWUFBWTtZQUNqQyxZQUFZLEVBQUUsbUJBQW1CO1lBQ2pDLE9BQU8sRUFBRSxjQUFjO1NBQ3hCLENBQUMsQ0FBQztRQUNILEtBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksVUFBRyxDQUFDLGFBQWEsRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDO1FBRTdELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFDbEQ7WUFDRSxvQkFBb0IsRUFBRTtnQkFDcEIsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7Z0JBQ2hELEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2FBQ2pEO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO2dCQUM3QyxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRTthQUMvQztZQUNELHFCQUFxQixFQUFFO2dCQUNyQixpQkFBaUIsRUFBRSxFQUFFO2dCQUNyQixrQkFBa0IsRUFBRSxJQUFJO2FBQ3pCO1lBQ0QsZ0NBQWdDLEVBQUUsRUFBRSwwQkFBMEIsRUFBRSxJQUFJLEVBQUU7WUFDdEUsZ0JBQWdCLEVBQUUsRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFO1lBQ3RDLG1CQUFtQixFQUFFLEVBQUUsY0FBYyxFQUFFLFdBQVcsRUFBRTtZQUNwRCxTQUFTLEVBQUUsU0FBUztZQUNwQixJQUFJLEVBQUUsQ0FBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUUsS0FBSyxFQUFFLFlBQVksRUFBRSxDQUFFO1lBQ3JELHVCQUF1QixFQUFFLEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFO1NBQ3hFLENBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhDQUE4QyxDQUFDLElBQVU7UUFDdkQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQy9CLFNBQVMsRUFBRSxVQUFVO1lBQ3JCLFdBQVcsRUFBRSxpQkFBVyxDQUFDLGVBQWU7WUFDeEMsWUFBWSxFQUFFLG1CQUFtQjtTQUNsQyxDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQ2xEO1lBQ0UsU0FBUyxFQUFFO2dCQUNULEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO2FBQzlDO1lBQ0QsV0FBVyxFQUFFLGlCQUFpQjtZQUM5QixvQkFBb0IsRUFBRTtnQkFDcEIsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7YUFDakQ7WUFDRCxTQUFTLEVBQUUsU0FBUztTQUNyQixDQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrRkFBa0YsQ0FBQyxJQUFVO1FBQzNGLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQ2pELFNBQVMsRUFBRSxVQUFVO1lBQ3JCLFdBQVcsRUFBRSxpQkFBVyxDQUFDLGVBQWU7WUFDeEMsWUFBWSxFQUFFLG1CQUFtQjtZQUNqQyxZQUFZLEVBQUUsQ0FBQztTQUNoQixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUNqRCxTQUFTLEVBQUUsVUFBVTtZQUNyQixXQUFXLEVBQUUsaUJBQVcsQ0FBQyxlQUFlO1lBQ3hDLFlBQVksRUFBRSxtQkFBbUI7WUFDakMsYUFBYSxFQUFFLENBQUM7U0FDakIsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDakQsU0FBUyxFQUFFLFVBQVU7WUFDckIsV0FBVyxFQUFFLGlCQUFXLENBQUMsZUFBZTtZQUN4QyxZQUFZLEVBQUUsbUJBQW1CO1lBQ2pDLFlBQVksRUFBRSxDQUFDO1lBQ2YsYUFBYSxFQUFFLENBQUM7U0FDakIsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseURBQXlELENBQUMsSUFBVTtRQUNsRSxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDN0MsWUFBWSxFQUFFLG1CQUFtQjtZQUNqQyxPQUFPLEVBQUUsY0FBYztTQUN4QixDQUFDLENBQUM7UUFFSCxLQUFLLENBQUMsdUJBQXVCLENBQUM7WUFDNUIsU0FBUyxFQUFFLFFBQVE7WUFDbkIsWUFBWSxFQUFFLGlCQUFpQjtZQUMvQixZQUFZLEVBQUUsRUFBRTtZQUNoQixhQUFhLEVBQUUsSUFBSTtTQUNwQixDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQ2xEO1lBQ0Usb0JBQW9CLEVBQUU7Z0JBQ3BCLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNoRCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtnQkFDaEQsRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7YUFDcEQ7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7Z0JBQzdDLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFO2FBQy9DO1lBQ0QscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxFQUFFO1lBQ3RFLHNCQUFzQixFQUFFO2dCQUN0QjtvQkFDRSxTQUFTLEVBQUUsT0FBTztvQkFDbEIsU0FBUyxFQUFFO3dCQUNULEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO3FCQUNqRDtvQkFDRCxVQUFVLEVBQUUsRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFO29CQUNyQyxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLEVBQUUsRUFBRSxrQkFBa0IsRUFBRSxJQUFJLEVBQUU7aUJBQzNFO2FBQ0Y7U0FDRixDQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0REFBNEQsQ0FBQyxJQUFVO1FBQ3JFLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUM3QyxZQUFZLEVBQUUsbUJBQW1CO1lBQ2pDLE9BQU8sRUFBRSxjQUFjO1NBQ3hCLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyx1QkFBdUIsQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixZQUFZLEVBQUUsaUJBQWlCO1lBQy9CLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLGNBQWMsRUFBRSxvQkFBYyxDQUFDLEdBQUc7WUFDbEMsWUFBWSxFQUFFLEVBQUU7WUFDaEIsYUFBYSxFQUFFLElBQUk7U0FDcEIsQ0FBQyxDQUFDO1FBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUNsRDtZQUNFLG9CQUFvQixFQUFFO2dCQUNwQixFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtnQkFDaEQsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7Z0JBQ2hELEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNuRCxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTthQUNwRDtZQUNELFNBQVMsRUFBRTtnQkFDVCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtnQkFDN0MsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUU7YUFDL0M7WUFDRCxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7WUFDdEUsc0JBQXNCLEVBQUU7Z0JBQ3RCO29CQUNFLFNBQVMsRUFBRSxPQUFPO29CQUNsQixTQUFTLEVBQUU7d0JBQ1QsRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7d0JBQ2hELEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFO3FCQUNsRDtvQkFDRCxVQUFVLEVBQUUsRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFO29CQUNyQyxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLEVBQUUsRUFBRSxrQkFBa0IsRUFBRSxJQUFJLEVBQUU7aUJBQzNFO2FBQ0Y7U0FDRixDQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxRUFBcUUsQ0FBQyxJQUFVO1FBQzlFLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUM3QyxZQUFZLEVBQUUsbUJBQW1CO1lBQ2pDLE9BQU8sRUFBRSxjQUFjO1NBQ3hCLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyx1QkFBdUIsQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixZQUFZLEVBQUUsaUJBQWlCO1lBQy9CLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLGNBQWMsRUFBRSxvQkFBYyxDQUFDLFNBQVM7U0FDekMsQ0FBQyxDQUFDO1FBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUNsRDtZQUNFLG9CQUFvQixFQUFFO2dCQUNwQixFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtnQkFDaEQsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7Z0JBQ2hELEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNuRCxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTthQUNwRDtZQUNELFNBQVMsRUFBRTtnQkFDVCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtnQkFDN0MsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUU7YUFDL0M7WUFDRCxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7WUFDdEUsc0JBQXNCLEVBQUU7Z0JBQ3RCO29CQUNFLFNBQVMsRUFBRSxPQUFPO29CQUNsQixTQUFTLEVBQUU7d0JBQ1QsRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7d0JBQ2hELEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFO3FCQUNsRDtvQkFDRCxVQUFVLEVBQUUsRUFBRSxjQUFjLEVBQUUsV0FBVyxFQUFFO29CQUMzQyxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7aUJBQ3ZFO2FBQ0Y7U0FDRixDQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtRUFBbUUsQ0FBQyxJQUFVO1FBQzVFLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUMvRyxNQUFNLDJCQUEyQixHQUFHLDJCQUEyQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQzdFLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixZQUFZLEVBQUUsaUJBQWlCO1lBQy9CLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLGNBQWMsRUFBRSxvQkFBYyxDQUFDLE9BQU87WUFDdEMsZ0JBQWdCLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxLQUFLLEVBQUUsMkJBQTJCLENBQUMsSUFBSSxFQUFFLENBQUMsS0FBSyxDQUFDO1lBQ3RHLFlBQVksRUFBRSxFQUFFO1lBQ2hCLGFBQWEsRUFBRSxJQUFJO1NBQ3BCLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFDbEQ7WUFDRSxvQkFBb0IsRUFBRTtnQkFDcEIsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7Z0JBQ2hELEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNoRCxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtnQkFDbkQsRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7YUFDcEQ7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7Z0JBQzdDLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFO2FBQy9DO1lBQ0QscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxFQUFFO1lBQ3RFLHNCQUFzQixFQUFFO2dCQUN0QjtvQkFDRSxTQUFTLEVBQUUsT0FBTztvQkFDbEIsU0FBUyxFQUFFO3dCQUNULEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO3dCQUNoRCxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRTtxQkFDbEQ7b0JBQ0QsVUFBVSxFQUFFLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxZQUFZLEVBQUUsWUFBWSxDQUFDLEVBQUUsY0FBYyxFQUFFLFNBQVMsRUFBRTtvQkFDekYscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxFQUFFLEVBQUUsa0JBQWtCLEVBQUUsSUFBSSxFQUFFO2lCQUMzRTthQUNGO1NBQ0YsQ0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsbUZBQW1GLENBQUMsSUFBVTtRQUM1RixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDL0IsV0FBVyxFQUFFLGlCQUFXLENBQUMsZUFBZTtZQUN4QyxZQUFZLEVBQUUsbUJBQW1CO1lBQ2pDLE9BQU8sRUFBRSxjQUFjO1NBQ3hCLENBQUMsQ0FBQyx1QkFBdUIsQ0FBQztZQUN6QixTQUFTLEVBQUUsUUFBUTtZQUNuQixZQUFZLEVBQUUsaUJBQWlCO1NBQ2hDLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFDbEQ7WUFDRSxvQkFBb0IsRUFBRTtnQkFDcEIsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7Z0JBQ2hELEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNoRCxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTthQUNwRDtZQUNELFdBQVcsRUFBRSxpQkFBaUI7WUFDOUIsU0FBUyxFQUFFO2dCQUNULEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO2dCQUM3QyxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRTthQUMvQztZQUNELHNCQUFzQixFQUFFO2dCQUN0QjtvQkFDRSxTQUFTLEVBQUUsT0FBTztvQkFDbEIsU0FBUyxFQUFFO3dCQUNULEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO3FCQUNqRDtvQkFDRCxVQUFVLEVBQUUsRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFO2lCQUN0QzthQUNGO1NBQ0YsQ0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsb0hBQW9ILENBQUMsSUFBVTtRQUM3SCxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsRUFBRSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7UUFDL0csSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsdUJBQXVCLENBQUM7WUFDOUMsU0FBUyxFQUFFLFFBQVE7WUFDbkIsWUFBWSxFQUFFLGlCQUFpQjtZQUMvQixPQUFPLEVBQUUsWUFBWTtZQUNyQixjQUFjLEVBQUUsb0JBQWMsQ0FBQyxPQUFPO1NBQ3ZDLENBQUMsRUFBRSwyRUFBMkUsQ0FBQyxDQUFDO1FBRWpGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrR0FBa0csQ0FBQyxJQUFVO1FBQzNHLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUMvRyxNQUFNLDJCQUEyQixHQUFHLDJCQUEyQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBRTdFLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLHVCQUF1QixDQUFDO1lBQzlDLFNBQVMsRUFBRSxRQUFRO1lBQ25CLFlBQVksRUFBRSxpQkFBaUI7WUFDL0IsZ0JBQWdCLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxLQUFLLENBQUM7U0FDN0QsQ0FBQyxFQUFFLG1GQUFtRixDQUFDLENBQUM7UUFFekYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHdHQUF3RyxDQUFDLElBQVU7UUFDakgsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEVBQUUsWUFBWSxFQUFFLG1CQUFtQixFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO1FBQy9HLE1BQU0sMkJBQTJCLEdBQUcsMkJBQTJCLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFN0UsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsdUJBQXVCLENBQUM7WUFDOUMsU0FBUyxFQUFFLFFBQVE7WUFDbkIsWUFBWSxFQUFFLGlCQUFpQjtZQUMvQixjQUFjLEVBQUUsb0JBQWMsQ0FBQyxTQUFTO1lBQ3hDLGdCQUFnQixFQUFFLENBQUMsMkJBQTJCLENBQUMsSUFBSSxFQUFFLENBQUMsS0FBSyxDQUFDO1NBQzdELENBQUMsRUFBRSxtRkFBbUYsQ0FBQyxDQUFDO1FBRXpGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtSEFBbUgsQ0FBQyxJQUFVO1FBQzVILE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUMvRyxNQUFNLDJCQUEyQixHQUFHLDJCQUEyQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQzdFLE1BQU0sbUJBQW1CLEdBQWEsRUFBRSxDQUFDO1FBQ3pDLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxFQUFFLEVBQUU7WUFDM0IsbUJBQW1CLENBQUMsSUFBSSxDQUFDLDJCQUEyQixDQUFDLElBQUksRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ3BFO1FBRUQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsdUJBQXVCLENBQUM7WUFDOUMsU0FBUyxFQUFFLFFBQVE7WUFDbkIsWUFBWSxFQUFFLGlCQUFpQjtZQUMvQixPQUFPLEVBQUUsWUFBWTtZQUNyQixjQUFjLEVBQUUsb0JBQWMsQ0FBQyxPQUFPO1lBQ3RDLGdCQUFnQixFQUFFLG1CQUFtQjtTQUN0QyxDQUFDLEVBQUUsNEVBQTRFLENBQUMsQ0FBQztRQUVsRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0dBQWtHLENBQUMsSUFBVTtRQUMzRyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsRUFBRSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7UUFFL0csSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsdUJBQXVCLENBQUM7WUFDOUMsU0FBUyxFQUFFLFFBQVE7WUFDbkIsWUFBWSxFQUFFLGlCQUFpQjtZQUMvQixPQUFPLEVBQUUsWUFBWTtZQUNyQixjQUFjLEVBQUUsb0JBQWMsQ0FBQyxPQUFPO1lBQ3RDLGdCQUFnQixFQUFFLENBQUMsV0FBVyxFQUFFLG1CQUFtQixDQUFDLElBQUksQ0FBQztZQUN6RCwyQ0FBMkM7U0FDNUMsQ0FBQyxFQUFFLGtOQUFrTixDQUFDLENBQUM7UUFFeE4sSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1HQUFtRyxDQUFDLElBQVU7UUFDNUcsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQzdDLFlBQVksRUFBRSxtQkFBbUI7WUFDakMsV0FBVyxFQUFFLGlCQUFXLENBQUMsZUFBZTtTQUN6QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQztZQUM5QyxTQUFTLEVBQUUsUUFBUTtZQUNuQixZQUFZLEVBQUUsaUJBQWlCO1lBQy9CLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLFlBQVksRUFBRSxDQUFDO1NBQ2hCLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsdUJBQXVCLENBQUM7WUFDOUMsU0FBUyxFQUFFLFFBQVE7WUFDbkIsWUFBWSxFQUFFLGlCQUFpQjtZQUMvQixPQUFPLEVBQUUsWUFBWTtZQUNyQixhQUFhLEVBQUUsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLHVCQUF1QixDQUFDO1lBQzlDLFNBQVMsRUFBRSxRQUFRO1lBQ25CLFlBQVksRUFBRSxpQkFBaUI7WUFDL0IsT0FBTyxFQUFFLFlBQVk7WUFDckIsWUFBWSxFQUFFLENBQUM7WUFDZixhQUFhLEVBQUUsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrQ0FBK0MsQ0FBQyxJQUFVO1FBQ3hELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUMvRyxNQUFNLFlBQVksR0FBRyxhQUFhLEVBQUUsQ0FBQztRQUNyQyxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFO1lBQzFCLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDMUQ7UUFFRCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQ2xEO1lBQ0Usb0JBQW9CLEVBQUU7Z0JBQ3BCLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNoRCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtnQkFDaEQsRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7Z0JBQ3BELEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNwRCxFQUFFLGFBQWEsRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtnQkFDcEQsRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7Z0JBQ3BELEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2FBQ3JEO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO2dCQUM3QyxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRTthQUMvQztZQUNELHFCQUFxQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxFQUFFLGtCQUFrQixFQUFFLENBQUMsRUFBRTtZQUN0RSxzQkFBc0IsRUFBRTtnQkFDdEI7b0JBQ0UsU0FBUyxFQUFFLFFBQVE7b0JBQ25CLFNBQVMsRUFBRTt3QkFDVCxFQUFFLGFBQWEsRUFBRSxhQUFhLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtxQkFDbEQ7b0JBQ0QsVUFBVSxFQUFFLEVBQUUsY0FBYyxFQUFFLEtBQUssRUFBRTtvQkFDckMscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxFQUFFO2lCQUN2RTtnQkFDRDtvQkFDRSxTQUFTLEVBQUUsUUFBUTtvQkFDbkIsU0FBUyxFQUFFO3dCQUNULEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO3FCQUNsRDtvQkFDRCxVQUFVLEVBQUUsRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFO29CQUNyQyxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7aUJBQ3ZFO2dCQUNEO29CQUNFLFNBQVMsRUFBRSxRQUFRO29CQUNuQixTQUFTLEVBQUU7d0JBQ1QsRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7cUJBQ2xEO29CQUNELFVBQVUsRUFBRSxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUU7b0JBQ3JDLHFCQUFxQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxFQUFFLGtCQUFrQixFQUFFLENBQUMsRUFBRTtpQkFDdkU7Z0JBQ0Q7b0JBQ0UsU0FBUyxFQUFFLFFBQVE7b0JBQ25CLFNBQVMsRUFBRTt3QkFDVCxFQUFFLGFBQWEsRUFBRSxhQUFhLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtxQkFDbEQ7b0JBQ0QsVUFBVSxFQUFFLEVBQUUsY0FBYyxFQUFFLEtBQUssRUFBRTtvQkFDckMscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxFQUFFO2lCQUN2RTtnQkFDRDtvQkFDRSxTQUFTLEVBQUUsUUFBUTtvQkFDbkIsU0FBUyxFQUFFO3dCQUNULEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO3FCQUNsRDtvQkFDRCxVQUFVLEVBQUUsRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFO29CQUNyQyxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7aUJBQ3ZFO2FBQ0Y7U0FDRixDQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpRkFBaUYsQ0FBQyxJQUFVO1FBQzFGLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUUvRyxLQUFLLENBQUMsdUJBQXVCLENBQUM7WUFDNUIsU0FBUyxFQUFFLFFBQVE7WUFDbkIsWUFBWSxFQUFFLGlCQUFpQjtTQUNoQyxDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQ2xEO1lBQ0Usb0JBQW9CLEVBQUU7Z0JBQ3BCLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNoRCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtnQkFDaEQsRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7YUFDcEQ7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7Z0JBQzdDLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFO2FBQy9DO1lBQ0QscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxFQUFFO1lBQ3RFLHNCQUFzQixFQUFFO2dCQUN0QjtvQkFDRSxTQUFTLEVBQUUsT0FBTztvQkFDbEIsU0FBUyxFQUFFO3dCQUNULEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO3FCQUNqRDtvQkFDRCxVQUFVLEVBQUUsRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFO29CQUNyQyxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7aUJBQ3ZFO2FBQ0Y7U0FDRixDQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwyREFBMkQsQ0FBQyxJQUFVO1FBQ3BFLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUUvRyxLQUFLLENBQUMsc0JBQXNCLENBQUM7WUFDM0IsU0FBUyxFQUFFLFFBQVE7WUFDbkIsT0FBTyxFQUFFLFlBQVk7U0FDdEIsQ0FBQyxDQUFDO1FBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUNsRDtZQUNFLG9CQUFvQixFQUFFO2dCQUNwQixFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtnQkFDaEQsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7Z0JBQ2hELEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2FBQ3BEO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO2dCQUM3QyxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRTthQUMvQztZQUNELHFCQUFxQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxFQUFFLGtCQUFrQixFQUFFLENBQUMsRUFBRTtZQUN0RSxxQkFBcUIsRUFBRTtnQkFDckI7b0JBQ0UsU0FBUyxFQUFFLE9BQU87b0JBQ2xCLFNBQVMsRUFBRTt3QkFDVCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTt3QkFDN0MsRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUU7cUJBQ2xEO29CQUNELFVBQVUsRUFBRSxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUU7aUJBQ3RDO2FBQ0Y7U0FDRixDQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxvRUFBb0UsQ0FBQyxJQUFVO1FBQzdFLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUMvRyxLQUFLLENBQUMsc0JBQXNCLENBQUM7WUFDM0IsU0FBUyxFQUFFLFFBQVE7WUFDbkIsT0FBTyxFQUFFLFlBQVk7WUFDckIsY0FBYyxFQUFFLG9CQUFjLENBQUMsU0FBUztTQUN6QyxDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQ2xEO1lBQ0Usb0JBQW9CLEVBQUU7Z0JBQ3BCLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNoRCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTtnQkFDaEQsRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7YUFDcEQ7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7Z0JBQzdDLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFO2FBQy9DO1lBQ0QscUJBQXFCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxFQUFFO1lBQ3RFLHFCQUFxQixFQUFFO2dCQUNyQjtvQkFDRSxTQUFTLEVBQUUsT0FBTztvQkFDbEIsU0FBUyxFQUFFO3dCQUNULEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFO3dCQUM3QyxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRTtxQkFDbEQ7b0JBQ0QsVUFBVSxFQUFFLEVBQUUsY0FBYyxFQUFFLFdBQVcsRUFBRTtpQkFDNUM7YUFDRjtTQUNGLENBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGtFQUFrRSxDQUFDLElBQVU7UUFDM0UsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEVBQUUsWUFBWSxFQUFFLG1CQUFtQixFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO1FBQy9HLE1BQU0sMkJBQTJCLEdBQUcsMkJBQTJCLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDN0UsS0FBSyxDQUFDLHNCQUFzQixDQUFDO1lBQzNCLFNBQVMsRUFBRSxRQUFRO1lBQ25CLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLGNBQWMsRUFBRSxvQkFBYyxDQUFDLE9BQU87WUFDdEMsZ0JBQWdCLEVBQUUsQ0FBRSwyQkFBMkIsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxLQUFLLEVBQUUsMkJBQTJCLENBQUMsSUFBSSxFQUFFLENBQUMsS0FBSyxDQUFFO1NBQ3pHLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFDbEQ7WUFDRSxvQkFBb0IsRUFBRTtnQkFDcEIsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7Z0JBQ2hELEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxFQUFFO2dCQUNoRCxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRTthQUNwRDtZQUNELFNBQVMsRUFBRTtnQkFDVCxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtnQkFDN0MsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUU7YUFDL0M7WUFDRCxxQkFBcUIsRUFBRSxFQUFFLGlCQUFpQixFQUFFLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7WUFDdEUscUJBQXFCLEVBQUU7Z0JBQ3JCO29CQUNFLFNBQVMsRUFBRSxPQUFPO29CQUNsQixTQUFTLEVBQUU7d0JBQ1QsRUFBRSxhQUFhLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUU7d0JBQzdDLEVBQUUsYUFBYSxFQUFFLFlBQVksRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFO3FCQUNsRDtvQkFDRCxVQUFVLEVBQUUsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLFlBQVksRUFBRSxZQUFZLENBQUMsRUFBRSxjQUFjLEVBQUUsU0FBUyxFQUFFO2lCQUMxRjthQUNGO1NBQ0YsQ0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsdURBQXVELENBQUMsSUFBVTtRQUNoRSxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsRUFBRSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7UUFDL0csTUFBTSxZQUFZLEdBQUcsYUFBYSxFQUFFLENBQUM7UUFDckMsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRTtZQUMxQixLQUFLLENBQUMsc0JBQXNCLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ3pEO1FBRUQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsc0JBQXNCLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxDQUFDLEtBQUssQ0FBQyxFQUN2RSwwREFBMEQsQ0FBQyxDQUFDO1FBRTlELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxRkFBcUYsQ0FBQyxJQUFVO1FBQzlGLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUMvRyxLQUFLLENBQUMsdUJBQXVCLENBQUM7WUFDNUIsU0FBUyxFQUFFLGdCQUFnQjtZQUMzQixZQUFZLEVBQUUsaUJBQWlCO1NBQ2hDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDO1lBQzdDLFNBQVMsRUFBRSxnQkFBZ0I7WUFDM0IsT0FBTyxFQUFFLFlBQVk7U0FDdEIsQ0FBQyxFQUFFLHdEQUF3RCxDQUFDLENBQUM7UUFFOUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1HQUFtRyxDQUFDLElBQVU7UUFDNUcsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEVBQUUsWUFBWSxFQUFFLG1CQUFtQixFQUFFLENBQUMsQ0FBQztRQUV0RixLQUFLLENBQUMsc0JBQXNCLENBQUM7WUFDM0IsU0FBUyxFQUFFLFFBQVE7WUFDbkIsT0FBTyxFQUFFLFlBQVk7U0FDdEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxNQUFNLEdBQUcsb0JBQWEsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRWxELElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNuQyxJQUFJLENBQUMsV0FBVyxDQUFDLDBFQUEwRSxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVoSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkJBQTZCLENBQUMsSUFBVTtRQUN0QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEVBQUUsWUFBWSxFQUFFLEVBQUUsRUFBRSxhQUFhLEVBQUUsSUFBSSxFQUFFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxDQUFDLENBQUM7UUFFN0gsT0FBTztRQUNQLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFLFdBQVcsRUFBRSxFQUFFLEVBQUUsV0FBVyxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsa0JBQWtCLENBQUMsRUFBRSx3QkFBd0IsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRXhILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNkNBQTZDLEVBQUU7WUFDM0UsV0FBVyxFQUFFLEdBQUc7WUFDaEIsV0FBVyxFQUFFLEVBQUU7WUFDZixpQkFBaUIsRUFBRSxrQ0FBa0M7WUFDckQsZ0JBQWdCLEVBQUUsVUFBVTtTQUM3QixDQUFDLENBQUMsQ0FBQztRQUNKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyw0Q0FBNEMsRUFBRTtZQUMxRSxVQUFVLEVBQUUsdUJBQXVCO1lBQ25DLHdDQUF3QyxFQUFFO2dCQUN4Qyw2QkFBNkIsRUFBRSxFQUFFLG9CQUFvQixFQUFFLGlDQUFpQyxFQUFFO2dCQUMxRixXQUFXLEVBQUUsRUFBRTthQUNoQjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhCQUE4QixDQUFDLElBQVU7UUFDdkMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxFQUFFLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDO1FBRTdILE9BQU87UUFDUCxLQUFLLENBQUMsc0JBQXNCLENBQUMsRUFBRSxXQUFXLEVBQUUsRUFBRSxFQUFFLFdBQVcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLEVBQUUsd0JBQXdCLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUV6SCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDZDQUE2QyxFQUFFO1lBQzNFLFdBQVcsRUFBRSxHQUFHO1lBQ2hCLFdBQVcsRUFBRSxFQUFFO1lBQ2YsaUJBQWlCLEVBQUUsbUNBQW1DO1lBQ3RELGdCQUFnQixFQUFFLFVBQVU7U0FDN0IsQ0FBQyxDQUFDLENBQUM7UUFDSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNENBQTRDLEVBQUU7WUFDMUUsVUFBVSxFQUFFLHVCQUF1QjtZQUNuQyx3Q0FBd0MsRUFBRTtnQkFDeEMsNkJBQTZCLEVBQUUsRUFBRSxvQkFBb0IsRUFBRSxrQ0FBa0MsRUFBRTtnQkFDM0YsV0FBVyxFQUFFLEVBQUU7YUFDaEI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxzREFBc0QsQ0FBQyxJQUFVO1FBQy9ELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsRUFBRSxZQUFZLEVBQUUsRUFBRSxFQUFFLGFBQWEsRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFLG1CQUFtQixFQUFFLENBQUMsQ0FBQztRQUM3SCxLQUFLLENBQUMscUJBQXFCLENBQUMsRUFBRSxXQUFXLEVBQUUsRUFBRSxFQUFFLFdBQVcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLEVBQUUsd0JBQXdCLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUV4SCxPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixLQUFLLENBQUMscUJBQXFCLENBQUMsRUFBRSxXQUFXLEVBQUUsRUFBRSxFQUFFLFdBQVcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ3JFLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhEQUE4RCxDQUFDLElBQVU7UUFDdkUsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFdBQVcsRUFBRSxpQkFBVyxDQUFDLGVBQWUsRUFBRSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDO1FBQ2hJLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixZQUFZLEVBQUUsaUJBQWlCO1NBQ2hDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNmLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFLFdBQVcsRUFBRSxFQUFFLEVBQUUsV0FBVyxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDckUsQ0FBQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNmLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLFdBQVcsRUFBRSxFQUFFLEVBQUUsV0FBVyxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDdEUsQ0FBQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyx5Q0FBeUMsQ0FBQyxRQUFRLEVBQUU7WUFDMUUsV0FBVyxFQUFFLENBQUM7WUFDZCxXQUFXLEVBQUUsQ0FBQztTQUNmLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhFQUE4RSxDQUFDLElBQVU7UUFDdkYsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxFQUFFLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDO1FBRTdILE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNmLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFLFdBQVcsRUFBRSxFQUFFLEVBQUUsV0FBVyxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsa0JBQWtCLENBQUMsRUFBRSx3QkFBd0IsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ3pILENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNFQUFzRSxDQUFDLElBQVU7UUFDL0UsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxFQUFFLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDO1FBRTdILE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFLFdBQVcsRUFBRSxFQUFFLEVBQUUsV0FBVyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUVwRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkJBQTZCLENBQUMsSUFBVTtRQUN0QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQzdDLFlBQVksRUFBRSxFQUFFO1lBQ2hCLGFBQWEsRUFBRSxJQUFJO1lBQ25CLFlBQVksRUFBRSxFQUFFLElBQUksRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTSxFQUFFO1NBQzNELENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMscUJBQXFCLENBQUMsRUFBRSxXQUFXLEVBQUUsQ0FBQyxFQUFFLFdBQVcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ2xGLE9BQU8sQ0FBQyxlQUFlLENBQUMseUJBQXlCLEVBQUU7WUFDakQsUUFBUSxFQUFFLFVBQVUsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztZQUN0QyxXQUFXLEVBQUUsRUFBRTtTQUNoQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDZDQUE2QyxFQUFFO1lBQzNFLGdCQUFnQixFQUFFO2dCQUNoQjtvQkFDRSxvQkFBb0IsRUFBRSxFQUFFLGFBQWEsRUFBRSxFQUFFLEVBQUU7b0JBQzNDLFFBQVEsRUFBRSxtQkFBbUI7b0JBQzdCLG1CQUFtQixFQUFFLHlCQUF5QjtpQkFDL0M7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELFNBQVMsRUFBRTtRQUNULDZEQUE2RCxDQUFDLElBQVU7WUFDdEUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtnQkFDdEMsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsbUJBQWEsQ0FBQyxNQUFNLEVBQUU7YUFDekQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsK0JBQStCLEVBQUUsQ0FBQyxFQUFFO2dCQUNyRSxNQUFNLEVBQUUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxFQUFFLElBQUksRUFBRSxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLEVBQUUsRUFBRSxFQUFFO2dCQUNoRSxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsZUFBZSxFQUFFLEVBQUU7Z0JBQ25ELFNBQVMsRUFBRSxjQUFjO2dCQUN6QixVQUFVLEVBQUUsMkJBQTJCO2dCQUN2QyxTQUFTLEVBQUUsS0FBSzthQUNqQixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsOERBQThELENBQUMsSUFBVTtZQUN2RSxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO2dCQUN0QyxZQUFZLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxtQkFBYSxDQUFDLE1BQU0sRUFBRTthQUN6RCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxnQ0FBZ0MsRUFBRSxDQUFDLEVBQUU7Z0JBQ3RFLE1BQU0sRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxTQUFTLEVBQUUsRUFBRSxFQUFFLEVBQUU7Z0JBQ2hFLFVBQVUsRUFBRSxFQUFFLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxlQUFlLEVBQUUsRUFBRTtnQkFDbkQsU0FBUyxFQUFFLGNBQWM7Z0JBQ3pCLFVBQVUsRUFBRSw0QkFBNEI7Z0JBQ3hDLFNBQVMsRUFBRSxLQUFLO2FBQ2pCLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxnREFBZ0QsQ0FBQyxJQUFVO1lBQ3pELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7Z0JBQ3RDLFlBQVksRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTSxFQUFFO2FBQ3pELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLGtCQUFrQixFQUFFLENBQUMsRUFBRTtnQkFDeEQsTUFBTSxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUUsRUFBRTtnQkFDaEUsVUFBVSxFQUFFLEVBQUUsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLGVBQWUsRUFBRSxFQUFFO2dCQUNuRCxTQUFTLEVBQUUsY0FBYztnQkFDekIsVUFBVSxFQUFFLGNBQWM7Z0JBQzFCLFNBQVMsRUFBRSxLQUFLO2FBQ2pCLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw4Q0FBOEMsQ0FBQyxJQUFVO1lBQ3ZELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7Z0JBQ3RDLFlBQVksRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTSxFQUFFO2FBQ3pELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLGdCQUFnQixFQUFFLENBQUMsRUFBRTtnQkFDdEQsTUFBTSxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUUsRUFBRTtnQkFDaEUsVUFBVSxFQUFFLEVBQUUsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLGVBQWUsRUFBRSxFQUFFO2dCQUNuRCxTQUFTLEVBQUUsY0FBYztnQkFDekIsVUFBVSxFQUFFLFlBQVk7Z0JBQ3hCLFNBQVMsRUFBRSxLQUFLO2FBQ2pCLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxrRUFBa0UsQ0FBQyxJQUFVO1lBQzNFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7Z0JBQ3RDLFlBQVksRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTSxFQUFFO2FBQ3pELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLG9DQUFvQyxFQUFFLENBQUMsRUFBRTtnQkFDMUUsTUFBTSxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUUsRUFBRTtnQkFDaEUsVUFBVSxFQUFFLEVBQUUsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLGVBQWUsRUFBRSxFQUFFO2dCQUNuRCxTQUFTLEVBQUUsY0FBYztnQkFDekIsVUFBVSxFQUFFLGdDQUFnQztnQkFDNUMsU0FBUyxFQUFFLEtBQUs7YUFDakIsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDREQUE0RCxDQUFDLElBQVU7WUFDckUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtnQkFDdEMsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsbUJBQWEsQ0FBQyxNQUFNLEVBQUU7YUFDekQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsOEJBQThCLEVBQUUsQ0FBQyxFQUFFO2dCQUNwRSxNQUFNLEVBQUUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxFQUFFLElBQUksRUFBRSxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLEVBQUUsRUFBRSxFQUFFO2dCQUNoRSxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsZUFBZSxFQUFFLEVBQUU7Z0JBQ25ELFNBQVMsRUFBRSxjQUFjO2dCQUN6QixVQUFVLEVBQUUsMEJBQTBCO2dCQUN0QyxTQUFTLEVBQUUsU0FBUzthQUNyQixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBRUY7SUFFRCxRQUFRLEVBQUU7UUFFUiw2RUFBNkUsQ0FBQyxJQUFVO1lBQ3RGLFNBQVMsQ0FBQyxJQUFJLEVBQ1osQ0FBRSxTQUFTLEVBQUUsU0FBUyxDQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7UUFDNUYsQ0FBQztRQUVELGtFQUFrRSxDQUFDLElBQVU7WUFDM0UsU0FBUyxDQUFDLElBQUksRUFDWixDQUFFLGNBQWMsRUFBRSxZQUFZLEVBQUUsa0JBQWtCLEVBQUUsT0FBTyxFQUFFLFNBQVMsRUFBRSxNQUFNLENBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNwSCxDQUFDO1FBRUQsa0VBQWtFLENBQUMsSUFBVTtZQUMzRSxTQUFTLENBQUMsSUFBSSxFQUFFO2dCQUNkLGdCQUFnQixFQUFFLFNBQVMsRUFBRSxZQUFZLEVBQUUsWUFBWTthQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDOUYsQ0FBQztRQUVELDhEQUE4RCxDQUFDLElBQVU7WUFDdkUsU0FBUyxDQUFDLElBQUksRUFBRTtnQkFDZCxjQUFjLEVBQUUsWUFBWSxFQUFFLGtCQUFrQixFQUFFLE9BQU8sRUFBRSxTQUFTLEVBQUUsTUFBTTtnQkFDNUUsZ0JBQWdCLEVBQUUsU0FBUyxFQUFFLFlBQVksRUFBRSxZQUFZO2FBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2xHLENBQUM7UUFFRCxpRkFBaUYsQ0FBQyxJQUFVO1lBQzFGLFNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBRSxHQUFHLENBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMzRCxDQUFDO1FBRUQsK0RBQStELENBQUMsSUFBVTtZQUN4RSxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBRXpDLE9BQU87WUFDUCxXQUFLLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFN0IsT0FBTztZQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtnQkFDaEQsZ0JBQWdCLEVBQUU7b0JBQ2hCLFdBQVcsRUFBRTt3QkFDWDs0QkFDRSxRQUFRLEVBQUUsc0JBQXNCOzRCQUNoQyxRQUFRLEVBQUUsT0FBTzs0QkFDakIsVUFBVSxFQUFFLEdBQUc7eUJBQ2hCO3FCQUNGO29CQUNELFNBQVMsRUFBRSxZQUFZO2lCQUN4QjtnQkFDRCxPQUFPLEVBQUUsQ0FBRSxFQUFFLEtBQUssRUFBRSxjQUFjLEVBQUUsQ0FBRTthQUN2QyxDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwyRUFBMkUsQ0FBQyxJQUFVO1lBQ3BGLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7Z0JBQ3pDLFlBQVksRUFBRTtvQkFDWixJQUFJLEVBQUUsSUFBSTtvQkFDVixJQUFJLEVBQUUsbUJBQWEsQ0FBQyxNQUFNO2lCQUMzQjthQUNGLENBQUMsQ0FBQztZQUNILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFFekMsT0FBTztZQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQyxFQUFFLHdEQUF3RCxDQUFDLENBQUM7WUFFL0csSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDZFQUE2RSxDQUFDLElBQVU7WUFDdEYsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtnQkFDekMsWUFBWSxFQUFFO29CQUNaLElBQUksRUFBRSxJQUFJO29CQUNWLElBQUksRUFBRSxtQkFBYSxDQUFDLE1BQU07aUJBQzNCO2dCQUNELE1BQU0sRUFBRSxvQkFBYyxDQUFDLFNBQVM7YUFDakMsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztZQUV6QyxPQUFPO1lBQ1AsS0FBSyxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBRWxDLE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELGdCQUFnQixFQUFFO29CQUNoQixXQUFXLEVBQUU7d0JBQ1g7NEJBQ0UsUUFBUSxFQUFFLHNCQUFzQjs0QkFDaEMsUUFBUSxFQUFFLE9BQU87NEJBQ2pCLFVBQVUsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFFLEVBQUUsRUFBRSxDQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsaUJBQWlCLEVBQUUsS0FBSyxDQUFFLEVBQUUsRUFBRSxXQUFXLENBQUUsQ0FBRSxFQUFFO3lCQUNwRztxQkFDRjtvQkFDRCxTQUFTLEVBQUUsWUFBWTtpQkFDeEI7Z0JBQ0QsT0FBTyxFQUFFLENBQUUsRUFBRSxLQUFLLEVBQUUsY0FBYyxFQUFFLENBQUU7YUFDdkMsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQscUVBQXFFLENBQUMsSUFBVTtZQUM5RSxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO2dCQUN6QyxZQUFZLEVBQUU7b0JBQ1osSUFBSSxFQUFFLElBQUk7b0JBQ1YsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTTtpQkFDM0I7YUFDRixDQUFDLENBQUM7WUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBRXpDLE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLEVBQUUsd0RBQXdELENBQUMsQ0FBQztZQUV6RyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsMkVBQTJFLENBQUMsSUFBVTtZQUNwRixRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO2dCQUN6QyxZQUFZLEVBQUU7b0JBQ1osSUFBSSxFQUFFLElBQUk7b0JBQ1YsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTTtpQkFDM0I7Z0JBQ0QsTUFBTSxFQUFFLG9CQUFjLENBQUMsU0FBUzthQUNqQyxDQUFDLENBQUM7WUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBRXpDLE9BQU87WUFDUCxLQUFLLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBRTVCLE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELGdCQUFnQixFQUFFO29CQUNoQixXQUFXLEVBQUU7d0JBQ1g7NEJBQ0UsUUFBUSxFQUFFLHNCQUFzQjs0QkFDaEMsUUFBUSxFQUFFLE9BQU87NEJBQ2pCLFVBQVUsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFFLEVBQUUsRUFBRSxDQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsaUJBQWlCLEVBQUUsS0FBSyxDQUFFLEVBQUUsRUFBRSxXQUFXLENBQUUsQ0FBRSxFQUFFO3lCQUNwRzt3QkFDRDs0QkFDRSxRQUFRLEVBQUU7Z0NBQ1IseUJBQXlCO2dDQUN6QixxQkFBcUI7Z0NBQ3JCLDJCQUEyQjs2QkFDNUI7NEJBQ0QsUUFBUSxFQUFFLE9BQU87NEJBQ2pCLFVBQVUsRUFBRTtnQ0FDVixZQUFZLEVBQUU7b0NBQ1osaUJBQWlCO29DQUNqQixXQUFXO2lDQUNaOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNELFNBQVMsRUFBRSxZQUFZO2lCQUN4QjtnQkFDRCxPQUFPLEVBQUUsQ0FBRSxFQUFFLEtBQUssRUFBRSxjQUFjLEVBQUUsQ0FBRTthQUN2QyxDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCx1REFBdUQsQ0FBQyxJQUFVO1lBQ2hFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxZQUFZLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxtQkFBYSxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztZQUN6RyxLQUFLLENBQUMsdUJBQXVCLENBQUMsRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLFlBQVksRUFBRSxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTSxFQUFFLEVBQUMsQ0FBQyxDQUFDO1lBQ2xILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFFekMsT0FBTztZQUNQLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFMUIsT0FBTztZQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtnQkFDaEQsZ0JBQWdCLEVBQUU7b0JBQ2hCLFdBQVcsRUFBRTt3QkFDWDs0QkFDRSxRQUFRLEVBQUU7Z0NBQ1IsdUJBQXVCO2dDQUN2QixxQkFBcUI7Z0NBQ3JCLDJCQUEyQjtnQ0FDM0IsZ0JBQWdCO2dDQUNoQixrQkFBa0I7Z0NBQ2xCLGVBQWU7NkJBQ2hCOzRCQUNELFFBQVEsRUFBRSxPQUFPOzRCQUNqQixVQUFVLEVBQUU7Z0NBQ1YsRUFBRSxZQUFZLEVBQUUsQ0FBQyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsRUFBRTtnQ0FDNUMsRUFBRSxVQUFVLEVBQUUsQ0FBRSxFQUFFLEVBQUUsQ0FBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGlCQUFpQixFQUFFLEtBQUssQ0FBRSxFQUFFLEVBQUUsVUFBVSxDQUFFLENBQUUsRUFBRTs2QkFDdkY7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsU0FBUyxFQUFFLFlBQVk7aUJBQ3hCO2dCQUNELE9BQU8sRUFBRSxDQUFFLEVBQUUsS0FBSyxFQUFFLGNBQWMsRUFBRSxDQUFFO2FBQ3ZDLENBQUMsQ0FBQyxDQUFDO1lBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsUUFBUSxFQUFFO1FBQ1IsK0ZBQStGLENBQUMsSUFBVTtZQUN4RyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLE1BQU0sUUFBUSxHQUFHLG9DQUFvQyxDQUFDO1lBQ3RELE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLFdBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRSxRQUFRLENBQUMsRUFBRSxpREFBaUQsQ0FBQyxDQUFDO1lBRTNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCx5RUFBeUUsQ0FBQyxJQUFVO1lBQ2xGLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFFMUIsTUFBTSxRQUFRLEdBQUcsbURBQW1ELENBQUM7WUFDckUsTUFBTSxLQUFLLEdBQUcsV0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBRW5FLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO2dCQUMxQyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7YUFDL0QsQ0FBQyxDQUFDO1lBQ0gsS0FBSyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUUxQiw0REFBNEQ7WUFDNUQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO2dCQUNoRCxnQkFBZ0IsRUFBRTtvQkFDaEIsV0FBVyxFQUFFO3dCQUNYOzRCQUNFLFFBQVEsRUFBRTtnQ0FDUix1QkFBdUI7Z0NBQ3ZCLHFCQUFxQjtnQ0FDckIsMkJBQTJCO2dDQUMzQixnQkFBZ0I7Z0NBQ2hCLGtCQUFrQjtnQ0FDbEIsZUFBZTs2QkFDaEI7NEJBQ0QsUUFBUSxFQUFFLE9BQU87NEJBQ2pCLFVBQVUsRUFBRTtnQ0FDVixRQUFRO2dDQUNSLEVBQUUsS0FBSyxFQUFFLGNBQWMsRUFBRTs2QkFDMUI7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsU0FBUyxFQUFFLFlBQVk7aUJBQ3hCO2dCQUNELFlBQVksRUFBRSw4QkFBOEI7Z0JBQzVDLE9BQU8sRUFBRSxDQUFFLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFFLENBQUU7YUFDMUMsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDekMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsRUFBRSxTQUFTLENBQUMsQ0FBQztZQUMxRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0QsZ0ZBQWdGLENBQUMsSUFBVTtZQUN6RixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLE1BQU0sU0FBUyxHQUFHLFNBQVMsQ0FBQztZQUM1QixNQUFNLEtBQUssR0FBRyxXQUFLLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFFckUsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7Z0JBQzFDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyx5QkFBeUIsQ0FBQzthQUMvRCxDQUFDLENBQUM7WUFDSCxLQUFLLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFL0IsNERBQTREO1lBQzVELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtnQkFDaEQsZ0JBQWdCLEVBQUU7b0JBQ2hCLFdBQVcsRUFBRTt3QkFDWDs0QkFDRSxRQUFRLEVBQUU7Z0NBQ1IsdUJBQXVCO2dDQUN2QixxQkFBcUI7Z0NBQ3JCLDJCQUEyQjtnQ0FDM0IsZ0JBQWdCO2dDQUNoQixrQkFBa0I7Z0NBQ2xCLGVBQWU7Z0NBQ2YseUJBQXlCO2dDQUN6QixrQkFBa0I7Z0NBQ2xCLHFCQUFxQjtnQ0FDckIscUJBQXFCOzZCQUN0Qjs0QkFDRCxRQUFRLEVBQUUsT0FBTzs0QkFDakIsVUFBVSxFQUFFO2dDQUNWO29DQUNFLFVBQVUsRUFBRTt3Q0FDVixFQUFFO3dDQUNGOzRDQUNFLE1BQU07NENBQ047Z0RBQ0UsS0FBSyxFQUFFLGdCQUFnQjs2Q0FDeEI7NENBQ0QsWUFBWTs0Q0FDWjtnREFDRSxLQUFLLEVBQUUsYUFBYTs2Q0FDckI7NENBQ0QsR0FBRzs0Q0FDSDtnREFDRSxLQUFLLEVBQUUsZ0JBQWdCOzZDQUN4Qjs0Q0FDRCxnQkFBZ0I7eUNBQ2pCO3FDQUNGO2lDQUNGO2dDQUNEO29DQUNFLEtBQUssRUFBRSxjQUFjO2lDQUN0Qjs2QkFDRjt5QkFDRjtxQkFDRjtvQkFDRCxTQUFTLEVBQUUsWUFBWTtpQkFDeEI7Z0JBQ0QsWUFBWSxFQUFFLDhCQUE4QjtnQkFDNUMsT0FBTyxFQUFFLENBQUUsRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsQ0FBRTthQUMxQyxDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSwwR0FBMEcsQ0FBQyxDQUFDO1lBQzNJLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFDMUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsUUFBUSxFQUFFO1FBQ1IsaUJBQWlCLENBQUMsSUFBVTtZQUMxQixRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUUxQixPQUFPO1lBQ1AsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtnQkFDeEIsWUFBWSxFQUFFO29CQUNaLElBQUksRUFBRSxJQUFJO29CQUNWLElBQUksRUFBRSxtQkFBYSxDQUFDLE1BQU07aUJBQzNCO2dCQUNELGtCQUFrQixFQUFFO29CQUNsQixXQUFXO29CQUNYLGNBQWM7aUJBQ2Y7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHlCQUF5QixFQUFFO2dCQUN2RCxVQUFVLEVBQUU7b0JBQ1YsWUFBWSxFQUFFO3dCQUNaLFlBQVksRUFBRTs0QkFDWix3R0FBd0c7NEJBQ3hHLHFFQUFxRTt5QkFDdEU7cUJBQ0Y7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULEdBQUcsRUFBRSxlQUFlO3FCQUNyQjtvQkFDRCxNQUFNLEVBQUUsV0FBVztpQkFDcEI7Z0JBQ0QsU0FBUyxFQUFFLDBDQUEwQzthQUN0RCxFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1lBRXJDLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtnQkFDdkQsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRTt3QkFDWixZQUFZLEVBQUU7NEJBQ1osd0dBQXdHOzRCQUN4RyxxRUFBcUU7eUJBQ3RFO3FCQUNGO29CQUNELFNBQVMsRUFBRTt3QkFDVCxHQUFHLEVBQUUsZUFBZTtxQkFDckI7b0JBQ0QsTUFBTSxFQUFFLGNBQWM7aUJBQ3ZCO2dCQUNELFNBQVMsRUFBRSw2Q0FBNkM7YUFDekQsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUVyQyxJQUFJLENBQUMsU0FBUyxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsVUFBVSxFQUFFO2dCQUM1RCx3Q0FBd0MsRUFBRTtvQkFDeEMsU0FBUyxFQUFFO3dCQUNULEVBQUUsWUFBWSxFQUFFLENBQUUsV0FBVyxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxDQUFFLEVBQUU7cUJBQzFEO2lCQUNGO2dCQUNELDJDQUEyQyxFQUFFO29CQUMzQyxTQUFTLEVBQUU7d0JBQ1QsRUFBRSxZQUFZLEVBQUUsQ0FBRSxjQUFjLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFLENBQUUsRUFBRTtxQkFDN0Q7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsc0NBQXNDLENBQUMsSUFBVTtZQUMvQyxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUUxQixPQUFPO1lBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO2dCQUMxQyxZQUFZLEVBQUU7b0JBQ1osSUFBSSxFQUFFLElBQUk7b0JBQ1YsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTTtpQkFDM0I7Z0JBQ0Qsa0JBQWtCLEVBQUU7b0JBQ2xCLFdBQVc7b0JBQ1gsY0FBYztpQkFDZjtnQkFDRCxXQUFXLEVBQUUsaUJBQVcsQ0FBQyxXQUFXO2FBQ3JDLENBQUMsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO1lBRXpCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw2REFBNkQsQ0FBQyxJQUFVO1lBQ3RFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7Z0JBQzFDLFlBQVksRUFBRTtvQkFDWixJQUFJLEVBQUUsSUFBSTtvQkFDVixJQUFJLEVBQUUsbUJBQWEsQ0FBQyxNQUFNO2lCQUMzQjtnQkFDRCxrQkFBa0IsRUFBRTtvQkFDbEIsV0FBVztvQkFDWCxjQUFjO2lCQUNmO2dCQUNELE1BQU0sRUFBRSxvQkFBYyxDQUFDLFNBQVM7YUFDakMsQ0FBQyxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFFNUIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDZDQUE2QyxDQUFDLElBQVU7WUFDdEQsUUFBUTtZQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7WUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRTtnQkFDcEMsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTthQUM3QixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO2dCQUMxQyxZQUFZLEVBQUU7b0JBQ1osSUFBSSxFQUFFLElBQUk7b0JBQ1YsSUFBSSxFQUFFLG1CQUFhLENBQUMsTUFBTTtpQkFDM0I7Z0JBQ0Qsa0JBQWtCLEVBQUU7b0JBQ2xCLFdBQVc7b0JBQ1gsV0FBVztvQkFDWCxXQUFXO2lCQUNaO2FBQ0YsQ0FBQyxFQUFFLDZFQUE2RSxDQUFDLENBQUM7WUFFbkYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELG9DQUFvQyxDQUFDLElBQVU7WUFDN0MsUUFBUTtZQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7WUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRTtnQkFDcEMsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTthQUM3QixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtnQkFDeEIsWUFBWSxFQUFFO29CQUNaLElBQUksRUFBRSxJQUFJO29CQUNWLElBQUksRUFBRSxtQkFBYSxDQUFDLE1BQU07aUJBQzNCO2dCQUNELGtCQUFrQixFQUFFO29CQUNsQixXQUFXO29CQUNYLGNBQWM7aUJBQ2Y7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsSUFBSSxDQUFDLEtBQUssQ0FBQyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLFVBQVUsRUFBRSxTQUFTLENBQUMsQ0FBQztZQUVyRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhwZWN0LCBoYXZlUmVzb3VyY2UsIFJlc291cmNlUGFydCwgU3ludGhVdGlscyB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgKiBhcyBhcHBzY2FsaW5nIGZyb20gJ0Bhd3MtY2RrL2F3cy1hcHBsaWNhdGlvbmF1dG9zY2FsaW5nJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IEFwcCwgQ2ZuRGVsZXRpb25Qb2xpY3ksIENvbnN0cnVjdE5vZGUsIFJlbW92YWxQb2xpY3ksIFN0YWNrLCBUYWcgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQge1xuICBBdHRyaWJ1dGUsXG4gIEF0dHJpYnV0ZVR5cGUsXG4gIEJpbGxpbmdNb2RlLFxuICBHbG9iYWxTZWNvbmRhcnlJbmRleFByb3BzLFxuICBMb2NhbFNlY29uZGFyeUluZGV4UHJvcHMsXG4gIFByb2plY3Rpb25UeXBlLFxuICBTdHJlYW1WaWV3VHlwZSxcbiAgVGFibGUsXG59IGZyb20gJy4uL2xpYic7XG5cbi8vIHRzbGludDpkaXNhYmxlOm9iamVjdC1saXRlcmFsLWtleS1xdW90ZXNcblxuLy8gQ0RLIHBhcmFtZXRlcnNcbmNvbnN0IENPTlNUUlVDVF9OQU1FID0gJ015VGFibGUnO1xuXG4vLyBEeW5hbW9EQiB0YWJsZSBwYXJhbWV0ZXJzXG5jb25zdCBUQUJMRV9OQU1FID0gJ015VGFibGUnO1xuY29uc3QgVEFCTEVfUEFSVElUSU9OX0tFWTogQXR0cmlidXRlID0geyBuYW1lOiAnaGFzaEtleScsIHR5cGU6IEF0dHJpYnV0ZVR5cGUuU1RSSU5HIH07XG5jb25zdCBUQUJMRV9TT1JUX0tFWTogQXR0cmlidXRlID0geyBuYW1lOiAnc29ydEtleScsIHR5cGU6IEF0dHJpYnV0ZVR5cGUuTlVNQkVSIH07XG5cbi8vIER5bmFtb0RCIGdsb2JhbCBzZWNvbmRhcnkgaW5kZXggcGFyYW1ldGVyc1xuY29uc3QgR1NJX05BTUUgPSAnTXlHU0knO1xuY29uc3QgR1NJX1BBUlRJVElPTl9LRVk6IEF0dHJpYnV0ZSA9IHsgbmFtZTogJ2dzaUhhc2hLZXknLCB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklORyB9O1xuY29uc3QgR1NJX1NPUlRfS0VZOiBBdHRyaWJ1dGUgPSB7IG5hbWU6ICdnc2lTb3J0S2V5JywgdHlwZTogQXR0cmlidXRlVHlwZS5CSU5BUlkgfTtcbmNvbnN0IEdTSV9OT05fS0VZID0gJ2dzaU5vbktleSc7XG5mdW5jdGlvbiogR1NJX0dFTkVSQVRPUigpOiBHZW5lcmF0b3I8R2xvYmFsU2Vjb25kYXJ5SW5kZXhQcm9wcywgbmV2ZXI+IHtcbiAgbGV0IG4gPSAwO1xuICB3aGlsZSAodHJ1ZSkge1xuICAgIGNvbnN0IGdsb2JhbFNlY29uZGFyeUluZGV4UHJvcHM6IEdsb2JhbFNlY29uZGFyeUluZGV4UHJvcHMgPSB7XG4gICAgICBpbmRleE5hbWU6IGAke0dTSV9OQU1FfSR7bn1gLFxuICAgICAgcGFydGl0aW9uS2V5OiB7IG5hbWU6IGAke0dTSV9QQVJUSVRJT05fS0VZLm5hbWV9JHtufWAsIHR5cGU6IEdTSV9QQVJUSVRJT05fS0VZLnR5cGUgfVxuICAgIH07XG4gICAgeWllbGQgZ2xvYmFsU2Vjb25kYXJ5SW5kZXhQcm9wcztcbiAgICBuKys7XG4gIH1cbn1cbmZ1bmN0aW9uKiBOT05fS0VZX0FUVFJJQlVURV9HRU5FUkFUT1Iobm9uS2V5UHJlZml4OiBzdHJpbmcpOiBHZW5lcmF0b3I8c3RyaW5nLCBuZXZlcj4ge1xuICBsZXQgbiA9IDA7XG4gIHdoaWxlICh0cnVlKSB7XG4gICAgeWllbGQgYCR7bm9uS2V5UHJlZml4fSR7bn1gO1xuICAgIG4rKztcbiAgfVxufVxuXG4vLyBEeW5hbW9EQiBsb2NhbCBzZWNvbmRhcnkgaW5kZXggcGFyYW1ldGVyc1xuY29uc3QgTFNJX05BTUUgPSAnTXlMU0knO1xuY29uc3QgTFNJX1NPUlRfS0VZOiBBdHRyaWJ1dGUgPSB7IG5hbWU6ICdsc2lTb3J0S2V5JywgdHlwZTogQXR0cmlidXRlVHlwZS5OVU1CRVIgfTtcbmNvbnN0IExTSV9OT05fS0VZID0gJ2xzaU5vbktleSc7XG5mdW5jdGlvbiogTFNJX0dFTkVSQVRPUigpOiBHZW5lcmF0b3I8TG9jYWxTZWNvbmRhcnlJbmRleFByb3BzLCBuZXZlcj4ge1xuICBsZXQgbiA9IDA7XG4gIHdoaWxlICh0cnVlKSB7XG4gICAgY29uc3QgbG9jYWxTZWNvbmRhcnlJbmRleFByb3BzOiBMb2NhbFNlY29uZGFyeUluZGV4UHJvcHMgPSB7XG4gICAgICBpbmRleE5hbWU6IGAke0xTSV9OQU1FfSR7bn1gLFxuICAgICAgc29ydEtleTogeyBuYW1lIDogYCR7TFNJX1NPUlRfS0VZLm5hbWV9JHtufWAsIHR5cGU6IExTSV9TT1JUX0tFWS50eXBlIH1cbiAgICB9O1xuICAgIHlpZWxkIGxvY2FsU2Vjb25kYXJ5SW5kZXhQcm9wcztcbiAgICBuKys7XG4gIH1cbn1cblxuZXhwb3J0ID0ge1xuICAnZGVmYXVsdCBwcm9wZXJ0aWVzJzoge1xuICAgICdoYXNoIGtleSBvbmx5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwgeyBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVkgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkR5bmFtb0RCOjpUYWJsZScsIHtcbiAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IFt7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgQXR0cmlidXRlVHlwZTogJ1MnIH1dLFxuICAgICAgICBLZXlTY2hlbWE6IFt7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH1dLFxuICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDUsIFdyaXRlQ2FwYWNpdHlVbml0czogNSB9XG4gICAgICB9KSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkR5bmFtb0RCOjpUYWJsZScsIHsgRGVsZXRpb25Qb2xpY3k6IENmbkRlbGV0aW9uUG9saWN5LlJFVEFJTiB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAncmVtb3ZhbFBvbGljeSBpcyBERVNUUk9ZJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwgeyBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RHluYW1vREI6OlRhYmxlJywgeyBEZWxldGlvblBvbGljeTogQ2ZuRGVsZXRpb25Qb2xpY3kuREVMRVRFIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdoYXNoICsgcmFuZ2Uga2V5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gICAgICAgIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RHluYW1vREI6OlRhYmxlJywge1xuICAgICAgICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH1cbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ3NvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgUHJvdmlzaW9uZWRUaHJvdWdocHV0OiB7IFJlYWRDYXBhY2l0eVVuaXRzOiA1LCBXcml0ZUNhcGFjaXR5VW5pdHM6IDUgfSxcbiAgICAgIH0pKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnaGFzaCArIHJhbmdlIGtleSBjYW4gYWxzbyBiZSBzcGVjaWZpZWQgaW4gcHJvcHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgIG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHtcbiAgICAgICAgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLFxuICAgICAgICBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWVxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkR5bmFtb0RCOjpUYWJsZScsXG4gICAgICAgIHtcbiAgICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9LFxuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ3NvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNSwgV3JpdGVDYXBhY2l0eVVuaXRzOiA1IH0sXG4gICAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdwb2ludC1pbi10aW1lIHJlY292ZXJ5IGlzIG5vdCBlbmFibGVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gICAgICAgIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RHluYW1vREI6OlRhYmxlJyxcbiAgICAgICAge1xuICAgICAgICAgIEF0dHJpYnV0ZURlZmluaXRpb25zOiBbXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgQXR0cmlidXRlVHlwZTogJ1MnIH0sXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEtleVR5cGU6ICdSQU5HRScgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgUHJvdmlzaW9uZWRUaHJvdWdocHV0OiB7IFJlYWRDYXBhY2l0eVVuaXRzOiA1LCBXcml0ZUNhcGFjaXR5VW5pdHM6IDUgfVxuICAgICAgICB9XG4gICAgICApKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnc2VydmVyLXNpZGUgZW5jcnlwdGlvbiBpcyBub3QgZW5hYmxlZCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHtcbiAgICAgICAgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLFxuICAgICAgICBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWSxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAgICB7XG4gICAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ3NvcnRLZXknLCBBdHRyaWJ1dGVUeXBlOiAnTicgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgS2V5VHlwZTogJ1JBTkdFJyB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDUsIFdyaXRlQ2FwYWNpdHlVbml0czogNSB9XG4gICAgICAgIH1cbiAgICAgICkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdzdHJlYW0gaXMgbm90IGVuYWJsZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7XG4gICAgICAgIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSxcbiAgICAgICAgc29ydEtleTogVEFCTEVfU09SVF9LRVksXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RHluYW1vREI6OlRhYmxlJyxcbiAgICAgICAge1xuICAgICAgICAgIEF0dHJpYnV0ZURlZmluaXRpb25zOiBbXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgQXR0cmlidXRlVHlwZTogJ1MnIH0sXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEtleVR5cGU6ICdSQU5HRScgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgUHJvdmlzaW9uZWRUaHJvdWdocHV0OiB7IFJlYWRDYXBhY2l0eVVuaXRzOiA1LCBXcml0ZUNhcGFjaXR5VW5pdHM6IDUgfVxuICAgICAgICB9XG4gICAgICApKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAndHRsIGlzIG5vdCBlbmFibGVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gICAgICAgIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkR5bmFtb0RCOjpUYWJsZScsXG4gICAgICAgIHtcbiAgICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9LFxuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ3NvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNSwgV3JpdGVDYXBhY2l0eVVuaXRzOiA1IH1cbiAgICAgICAgfVxuICAgICAgKSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2NhbiBzcGVjaWZ5IG5ldyBhbmQgb2xkIGltYWdlcycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgICB0YWJsZU5hbWU6IFRBQkxFX05BTUUsXG4gICAgICAgIHJlYWRDYXBhY2l0eTogNDIsXG4gICAgICAgIHdyaXRlQ2FwYWNpdHk6IDEzMzcsXG4gICAgICAgIHN0cmVhbTogU3RyZWFtVmlld1R5cGUuTkVXX0FORF9PTERfSU1BR0VTLFxuICAgICAgICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gICAgICAgIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RHluYW1vREI6OlRhYmxlJyxcbiAgICAgICAge1xuICAgICAgICAgIEF0dHJpYnV0ZURlZmluaXRpb25zOiBbXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgQXR0cmlidXRlVHlwZTogJ1MnIH0sXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIFN0cmVhbVNwZWNpZmljYXRpb246IHsgU3RyZWFtVmlld1R5cGU6ICdORVdfQU5EX09MRF9JTUFHRVMnIH0sXG4gICAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgS2V5VHlwZTogJ1JBTkdFJyB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDQyLCBXcml0ZUNhcGFjaXR5VW5pdHM6IDEzMzcgfSxcbiAgICAgICAgICBUYWJsZU5hbWU6ICdNeVRhYmxlJ1xuICAgICAgICB9XG4gICAgICApKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIHNwZWNpZnkgbmV3IGltYWdlcyBvbmx5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgICBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7XG4gICAgICAgIHRhYmxlTmFtZTogVEFCTEVfTkFNRSxcbiAgICAgICAgcmVhZENhcGFjaXR5OiA0MixcbiAgICAgICAgd3JpdGVDYXBhY2l0eTogMTMzNyxcbiAgICAgICAgc3RyZWFtOiBTdHJlYW1WaWV3VHlwZS5ORVdfSU1BR0UsXG4gICAgICAgIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSxcbiAgICAgICAgc29ydEtleTogVEFCTEVfU09SVF9LRVlcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAgICB7XG4gICAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgS2V5VHlwZTogJ1JBTkdFJyB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDQyLCBXcml0ZUNhcGFjaXR5VW5pdHM6IDEzMzcgfSxcbiAgICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9LFxuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBTdHJlYW1TcGVjaWZpY2F0aW9uOiB7IFN0cmVhbVZpZXdUeXBlOiAnTkVXX0lNQUdFJyB9LFxuICAgICAgICAgIFRhYmxlTmFtZTogJ015VGFibGUnLFxuICAgICAgICB9XG4gICAgICApKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIHNwZWNpZnkgb2xkIGltYWdlcyBvbmx5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgICBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7XG4gICAgICAgIHRhYmxlTmFtZTogVEFCTEVfTkFNRSxcbiAgICAgICAgcmVhZENhcGFjaXR5OiA0MixcbiAgICAgICAgd3JpdGVDYXBhY2l0eTogMTMzNyxcbiAgICAgICAgc3RyZWFtOiBTdHJlYW1WaWV3VHlwZS5PTERfSU1BR0UsXG4gICAgICAgIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSxcbiAgICAgICAgc29ydEtleTogVEFCTEVfU09SVF9LRVlcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAgICB7XG4gICAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgS2V5VHlwZTogJ1JBTkdFJyB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDQyLCBXcml0ZUNhcGFjaXR5VW5pdHM6IDEzMzcgfSxcbiAgICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9LFxuICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBTdHJlYW1TcGVjaWZpY2F0aW9uOiB7IFN0cmVhbVZpZXdUeXBlOiAnT0xEX0lNQUdFJyB9LFxuICAgICAgICAgIFRhYmxlTmFtZTogJ015VGFibGUnLFxuICAgICAgICB9XG4gICAgICApKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH1cbiAgfSxcblxuICAnd2hlbiBzcGVjaWZ5aW5nIGV2ZXJ5IHByb3BlcnR5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHtcbiAgICAgIHRhYmxlTmFtZTogVEFCTEVfTkFNRSxcbiAgICAgIHJlYWRDYXBhY2l0eTogNDIsXG4gICAgICB3cml0ZUNhcGFjaXR5OiAxMzM3LFxuICAgICAgcG9pbnRJblRpbWVSZWNvdmVyeTogdHJ1ZSxcbiAgICAgIHNlcnZlclNpZGVFbmNyeXB0aW9uOiB0cnVlLFxuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBST1ZJU0lPTkVELFxuICAgICAgc3RyZWFtOiBTdHJlYW1WaWV3VHlwZS5LRVlTX09OTFksXG4gICAgICB0aW1lVG9MaXZlQXR0cmlidXRlOiAndGltZVRvTGl2ZScsXG4gICAgICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gICAgICBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWSxcbiAgICB9KTtcbiAgICB0YWJsZS5ub2RlLmFwcGx5QXNwZWN0KG5ldyBUYWcoJ0Vudmlyb25tZW50JywgJ1Byb2R1Y3Rpb24nKSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAge1xuICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH1cbiAgICAgICAgXSxcbiAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ3NvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgXSxcbiAgICAgICAgUHJvdmlzaW9uZWRUaHJvdWdocHV0OiB7XG4gICAgICAgICAgUmVhZENhcGFjaXR5VW5pdHM6IDQyLFxuICAgICAgICAgIFdyaXRlQ2FwYWNpdHlVbml0czogMTMzN1xuICAgICAgICB9LFxuICAgICAgICBQb2ludEluVGltZVJlY292ZXJ5U3BlY2lmaWNhdGlvbjogeyBQb2ludEluVGltZVJlY292ZXJ5RW5hYmxlZDogdHJ1ZSB9LFxuICAgICAgICBTU0VTcGVjaWZpY2F0aW9uOiB7IFNTRUVuYWJsZWQ6IHRydWUgfSxcbiAgICAgICAgU3RyZWFtU3BlY2lmaWNhdGlvbjogeyBTdHJlYW1WaWV3VHlwZTogJ0tFWVNfT05MWScgfSxcbiAgICAgICAgVGFibGVOYW1lOiAnTXlUYWJsZScsXG4gICAgICAgIFRhZ3M6IFsgeyBLZXk6ICdFbnZpcm9ubWVudCcsIFZhbHVlOiAnUHJvZHVjdGlvbicgfSBdLFxuICAgICAgICBUaW1lVG9MaXZlU3BlY2lmaWNhdGlvbjogeyBBdHRyaWJ1dGVOYW1lOiAndGltZVRvTGl2ZScsIEVuYWJsZWQ6IHRydWUgfVxuICAgICAgfVxuICAgICkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd3aGVuIHNwZWNpZnlpbmcgUEFZX1BFUl9SRVFVRVNUIGJpbGxpbmcgbW9kZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgdGFibGVOYW1lOiBUQUJMRV9OQU1FLFxuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCxcbiAgICAgIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWVxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RHluYW1vREI6OlRhYmxlJyxcbiAgICAgIHtcbiAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICBdLFxuICAgICAgICBCaWxsaW5nTW9kZTogJ1BBWV9QRVJfUkVRVUVTVCcsXG4gICAgICAgIEF0dHJpYnV0ZURlZmluaXRpb25zOiBbXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9LFxuICAgICAgICBdLFxuICAgICAgICBUYWJsZU5hbWU6ICdNeVRhYmxlJyxcbiAgICAgIH1cbiAgICApKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZXJyb3Igd2hlbiBzcGVjaWZ5aW5nIHJlYWQgb3Igd3JpdGUgY2FwYWNpdHkgd2l0aCBhIFBBWV9QRVJfUkVRVUVTVCBiaWxsaW5nIG1vZGUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHtcbiAgICAgIHRhYmxlTmFtZTogVEFCTEVfTkFNRSxcbiAgICAgIGJpbGxpbmdNb2RlOiBCaWxsaW5nTW9kZS5QQVlfUEVSX1JFUVVFU1QsXG4gICAgICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gICAgICByZWFkQ2FwYWNpdHk6IDFcbiAgICB9KSk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgdGFibGVOYW1lOiBUQUJMRV9OQU1FLFxuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCxcbiAgICAgIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSxcbiAgICAgIHdyaXRlQ2FwYWNpdHk6IDFcbiAgICB9KSk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgdGFibGVOYW1lOiBUQUJMRV9OQU1FLFxuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCxcbiAgICAgIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSxcbiAgICAgIHJlYWRDYXBhY2l0eTogMSxcbiAgICAgIHdyaXRlQ2FwYWNpdHk6IDFcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3doZW4gYWRkaW5nIGEgZ2xvYmFsIHNlY29uZGFyeSBpbmRleCB3aXRoIGhhc2gga2V5IG9ubHknKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7XG4gICAgICBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksXG4gICAgICBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWVxuICAgIH0pO1xuXG4gICAgdGFibGUuYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgaW5kZXhOYW1lOiBHU0lfTkFNRSxcbiAgICAgIHBhcnRpdGlvbktleTogR1NJX1BBUlRJVElPTl9LRVksXG4gICAgICByZWFkQ2FwYWNpdHk6IDQyLFxuICAgICAgd3JpdGVDYXBhY2l0eTogMTMzN1xuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RHluYW1vREI6OlRhYmxlJyxcbiAgICAgIHtcbiAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgQXR0cmlidXRlVHlwZTogJ1MnIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9LFxuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaUhhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgXSxcbiAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ3NvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgXSxcbiAgICAgICAgUHJvdmlzaW9uZWRUaHJvdWdocHV0OiB7IFJlYWRDYXBhY2l0eVVuaXRzOiA1LCBXcml0ZUNhcGFjaXR5VW5pdHM6IDUgfSxcbiAgICAgICAgR2xvYmFsU2Vjb25kYXJ5SW5kZXhlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEluZGV4TmFtZTogJ015R1NJJyxcbiAgICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdnc2lIYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgUHJvamVjdGlvbjogeyBQcm9qZWN0aW9uVHlwZTogJ0FMTCcgfSxcbiAgICAgICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNDIsIFdyaXRlQ2FwYWNpdHlVbml0czogMTMzNyB9XG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9XG4gICAgKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3doZW4gYWRkaW5nIGEgZ2xvYmFsIHNlY29uZGFyeSBpbmRleCB3aXRoIGhhc2ggKyByYW5nZSBrZXknKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLFxuICAgICAgc29ydEtleTogVEFCTEVfU09SVF9LRVlcbiAgICB9KTtcblxuICAgIHRhYmxlLmFkZEdsb2JhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgIGluZGV4TmFtZTogR1NJX05BTUUsXG4gICAgICBwYXJ0aXRpb25LZXk6IEdTSV9QQVJUSVRJT05fS0VZLFxuICAgICAgc29ydEtleTogR1NJX1NPUlRfS0VZLFxuICAgICAgcHJvamVjdGlvblR5cGU6IFByb2plY3Rpb25UeXBlLkFMTCxcbiAgICAgIHJlYWRDYXBhY2l0eTogNDIsXG4gICAgICB3cml0ZUNhcGFjaXR5OiAxMzM3XG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAge1xuICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpSGFzaEtleScsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9LFxuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaVNvcnRLZXknLCBBdHRyaWJ1dGVUeXBlOiAnQicgfVxuICAgICAgICBdLFxuICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEtleVR5cGU6ICdSQU5HRScgfVxuICAgICAgICBdLFxuICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDUsIFdyaXRlQ2FwYWNpdHlVbml0czogNSB9LFxuICAgICAgICBHbG9iYWxTZWNvbmRhcnlJbmRleGVzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgSW5kZXhOYW1lOiAnTXlHU0knLFxuICAgICAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaUhhc2hLZXknLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpU29ydEtleScsIEtleVR5cGU6ICdSQU5HRScgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFByb2plY3Rpb246IHsgUHJvamVjdGlvblR5cGU6ICdBTEwnIH0sXG4gICAgICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDQyLCBXcml0ZUNhcGFjaXR5VW5pdHM6IDEzMzcgfVxuICAgICAgICAgIH1cbiAgICAgICAgXVxuICAgICAgfVxuICAgICkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd3aGVuIGFkZGluZyBhIGdsb2JhbCBzZWNvbmRhcnkgaW5kZXggd2l0aCBwcm9qZWN0aW9uIHR5cGUgS0VZU19PTkxZJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHtcbiAgICAgIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSxcbiAgICAgIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZXG4gICAgfSk7XG5cbiAgICB0YWJsZS5hZGRHbG9iYWxTZWNvbmRhcnlJbmRleCh7XG4gICAgICBpbmRleE5hbWU6IEdTSV9OQU1FLFxuICAgICAgcGFydGl0aW9uS2V5OiBHU0lfUEFSVElUSU9OX0tFWSxcbiAgICAgIHNvcnRLZXk6IEdTSV9TT1JUX0tFWSxcbiAgICAgIHByb2plY3Rpb25UeXBlOiBQcm9qZWN0aW9uVHlwZS5LRVlTX09OTFksXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAge1xuICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpSGFzaEtleScsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9LFxuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaVNvcnRLZXknLCBBdHRyaWJ1dGVUeXBlOiAnQicgfVxuICAgICAgICBdLFxuICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEtleVR5cGU6ICdSQU5HRScgfVxuICAgICAgICBdLFxuICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDUsIFdyaXRlQ2FwYWNpdHlVbml0czogNSB9LFxuICAgICAgICBHbG9iYWxTZWNvbmRhcnlJbmRleGVzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgSW5kZXhOYW1lOiAnTXlHU0knLFxuICAgICAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaUhhc2hLZXknLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpU29ydEtleScsIEtleVR5cGU6ICdSQU5HRScgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFByb2plY3Rpb246IHsgUHJvamVjdGlvblR5cGU6ICdLRVlTX09OTFknIH0sXG4gICAgICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDUsIFdyaXRlQ2FwYWNpdHlVbml0czogNSB9XG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9XG4gICAgKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3doZW4gYWRkaW5nIGEgZ2xvYmFsIHNlY29uZGFyeSBpbmRleCB3aXRoIHByb2plY3Rpb24gdHlwZSBJTkNMVURFJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHsgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLCBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWSB9KTtcbiAgICBjb25zdCBnc2lOb25LZXlBdHRyaWJ1dGVHZW5lcmF0b3IgPSBOT05fS0VZX0FUVFJJQlVURV9HRU5FUkFUT1IoR1NJX05PTl9LRVkpO1xuICAgIHRhYmxlLmFkZEdsb2JhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgIGluZGV4TmFtZTogR1NJX05BTUUsXG4gICAgICBwYXJ0aXRpb25LZXk6IEdTSV9QQVJUSVRJT05fS0VZLFxuICAgICAgc29ydEtleTogR1NJX1NPUlRfS0VZLFxuICAgICAgcHJvamVjdGlvblR5cGU6IFByb2plY3Rpb25UeXBlLklOQ0xVREUsXG4gICAgICBub25LZXlBdHRyaWJ1dGVzOiBbZ3NpTm9uS2V5QXR0cmlidXRlR2VuZXJhdG9yLm5leHQoKS52YWx1ZSwgZ3NpTm9uS2V5QXR0cmlidXRlR2VuZXJhdG9yLm5leHQoKS52YWx1ZV0sXG4gICAgICByZWFkQ2FwYWNpdHk6IDQyLFxuICAgICAgd3JpdGVDYXBhY2l0eTogMTMzN1xuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RHluYW1vREI6OlRhYmxlJyxcbiAgICAgIHtcbiAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgQXR0cmlidXRlVHlwZTogJ1MnIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9LFxuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaUhhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdnc2lTb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ0InIH1cbiAgICAgICAgXSxcbiAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnaGFzaEtleScsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ3NvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgXSxcbiAgICAgICAgUHJvdmlzaW9uZWRUaHJvdWdocHV0OiB7IFJlYWRDYXBhY2l0eVVuaXRzOiA1LCBXcml0ZUNhcGFjaXR5VW5pdHM6IDUgfSxcbiAgICAgICAgR2xvYmFsU2Vjb25kYXJ5SW5kZXhlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEluZGV4TmFtZTogJ015R1NJJyxcbiAgICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdnc2lIYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaVNvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBQcm9qZWN0aW9uOiB7IE5vbktleUF0dHJpYnV0ZXM6IFsnZ3NpTm9uS2V5MCcsICdnc2lOb25LZXkxJ10sIFByb2plY3Rpb25UeXBlOiAnSU5DTFVERScgfSxcbiAgICAgICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNDIsIFdyaXRlQ2FwYWNpdHlVbml0czogMTMzNyB9XG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9XG4gICAgKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3doZW4gYWRkaW5nIGEgZ2xvYmFsIHNlY29uZGFyeSBpbmRleCBvbiBhIHRhYmxlIHdpdGggUEFZX1BFUl9SRVFVRVNUIGJpbGxpbmcgbW9kZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCxcbiAgICAgIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSxcbiAgICAgIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZXG4gICAgfSkuYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgaW5kZXhOYW1lOiBHU0lfTkFNRSxcbiAgICAgIHBhcnRpdGlvbktleTogR1NJX1BBUlRJVElPTl9LRVksXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAge1xuICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpSGFzaEtleScsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9LFxuICAgICAgICBdLFxuICAgICAgICBCaWxsaW5nTW9kZTogJ1BBWV9QRVJfUkVRVUVTVCcsXG4gICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgS2V5VHlwZTogJ1JBTkdFJyB9XG4gICAgICAgIF0sXG4gICAgICAgIEdsb2JhbFNlY29uZGFyeUluZGV4ZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBJbmRleE5hbWU6ICdNeUdTSScsXG4gICAgICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpSGFzaEtleScsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFByb2plY3Rpb246IHsgUHJvamVjdGlvblR5cGU6ICdBTEwnIH1cbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH1cbiAgICApKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZXJyb3Igd2hlbiBhZGRpbmcgYSBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4IHdpdGggcHJvamVjdGlvbiB0eXBlIElOQ0xVREUsIGJ1dCB3aXRob3V0IHNwZWNpZnlpbmcgbm9uLWtleSBhdHRyaWJ1dGVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHsgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLCBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWSB9KTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiB0YWJsZS5hZGRHbG9iYWxTZWNvbmRhcnlJbmRleCh7XG4gICAgICBpbmRleE5hbWU6IEdTSV9OQU1FLFxuICAgICAgcGFydGl0aW9uS2V5OiBHU0lfUEFSVElUSU9OX0tFWSxcbiAgICAgIHNvcnRLZXk6IEdTSV9TT1JUX0tFWSxcbiAgICAgIHByb2plY3Rpb25UeXBlOiBQcm9qZWN0aW9uVHlwZS5JTkNMVURFXG4gICAgfSksIC9ub24ta2V5IGF0dHJpYnV0ZXMgc2hvdWxkIGJlIHNwZWNpZmllZCB3aGVuIHVzaW5nIElOQ0xVREUgcHJvamVjdGlvbiB0eXBlLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZXJyb3Igd2hlbiBhZGRpbmcgYSBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4IHdpdGggcHJvamVjdGlvbiB0eXBlIEFMTCwgYnV0IHdpdGggbm9uLWtleSBhdHRyaWJ1dGVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHsgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLCBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWSB9KTtcbiAgICBjb25zdCBnc2lOb25LZXlBdHRyaWJ1dGVHZW5lcmF0b3IgPSBOT05fS0VZX0FUVFJJQlVURV9HRU5FUkFUT1IoR1NJX05PTl9LRVkpO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gdGFibGUuYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgaW5kZXhOYW1lOiBHU0lfTkFNRSxcbiAgICAgIHBhcnRpdGlvbktleTogR1NJX1BBUlRJVElPTl9LRVksXG4gICAgICBub25LZXlBdHRyaWJ1dGVzOiBbZ3NpTm9uS2V5QXR0cmlidXRlR2VuZXJhdG9yLm5leHQoKS52YWx1ZV1cbiAgICB9KSwgL25vbi1rZXkgYXR0cmlidXRlcyBzaG91bGQgbm90IGJlIHNwZWNpZmllZCB3aGVuIG5vdCB1c2luZyBJTkNMVURFIHByb2plY3Rpb24gdHlwZS8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Vycm9yIHdoZW4gYWRkaW5nIGEgZ2xvYmFsIHNlY29uZGFyeSBpbmRleCB3aXRoIHByb2plY3Rpb24gdHlwZSBLRVlTX09OTFksIGJ1dCB3aXRoIG5vbi1rZXkgYXR0cmlidXRlcycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7IHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSwgc29ydEtleTogVEFCTEVfU09SVF9LRVkgfSk7XG4gICAgY29uc3QgZ3NpTm9uS2V5QXR0cmlidXRlR2VuZXJhdG9yID0gTk9OX0tFWV9BVFRSSUJVVEVfR0VORVJBVE9SKEdTSV9OT05fS0VZKTtcblxuICAgIHRlc3QudGhyb3dzKCgpID0+IHRhYmxlLmFkZEdsb2JhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgIGluZGV4TmFtZTogR1NJX05BTUUsXG4gICAgICBwYXJ0aXRpb25LZXk6IEdTSV9QQVJUSVRJT05fS0VZLFxuICAgICAgcHJvamVjdGlvblR5cGU6IFByb2plY3Rpb25UeXBlLktFWVNfT05MWSxcbiAgICAgIG5vbktleUF0dHJpYnV0ZXM6IFtnc2lOb25LZXlBdHRyaWJ1dGVHZW5lcmF0b3IubmV4dCgpLnZhbHVlXVxuICAgIH0pLCAvbm9uLWtleSBhdHRyaWJ1dGVzIHNob3VsZCBub3QgYmUgc3BlY2lmaWVkIHdoZW4gbm90IHVzaW5nIElOQ0xVREUgcHJvamVjdGlvbiB0eXBlLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZXJyb3Igd2hlbiBhZGRpbmcgYSBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4IHdpdGggcHJvamVjdGlvbiB0eXBlIElOQ0xVREUsIGJ1dCB3aXRoIG1vcmUgdGhhbiAyMCBub24ta2V5IGF0dHJpYnV0ZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwgeyBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZIH0pO1xuICAgIGNvbnN0IGdzaU5vbktleUF0dHJpYnV0ZUdlbmVyYXRvciA9IE5PTl9LRVlfQVRUUklCVVRFX0dFTkVSQVRPUihHU0lfTk9OX0tFWSk7XG4gICAgY29uc3QgZ3NpTm9uS2V5QXR0cmlidXRlczogc3RyaW5nW10gPSBbXTtcbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IDIxOyBpKyspIHtcbiAgICAgIGdzaU5vbktleUF0dHJpYnV0ZXMucHVzaChnc2lOb25LZXlBdHRyaWJ1dGVHZW5lcmF0b3IubmV4dCgpLnZhbHVlKTtcbiAgICB9XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB0YWJsZS5hZGRHbG9iYWxTZWNvbmRhcnlJbmRleCh7XG4gICAgICBpbmRleE5hbWU6IEdTSV9OQU1FLFxuICAgICAgcGFydGl0aW9uS2V5OiBHU0lfUEFSVElUSU9OX0tFWSxcbiAgICAgIHNvcnRLZXk6IEdTSV9TT1JUX0tFWSxcbiAgICAgIHByb2plY3Rpb25UeXBlOiBQcm9qZWN0aW9uVHlwZS5JTkNMVURFLFxuICAgICAgbm9uS2V5QXR0cmlidXRlczogZ3NpTm9uS2V5QXR0cmlidXRlc1xuICAgIH0pLCAvYSBtYXhpbXVtIG51bWJlciBvZiBub25LZXlBdHRyaWJ1dGVzIGFjcm9zcyBhbGwgb2Ygc2Vjb25kYXJ5IGluZGV4ZXMgaXMgMjAvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdlcnJvciB3aGVuIGFkZGluZyBhIGdsb2JhbCBzZWNvbmRhcnkgaW5kZXggd2l0aCBwcm9qZWN0aW9uIHR5cGUgSU5DTFVERSwgYnV0IHdpdGgga2V5IGF0dHJpYnV0ZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwgeyBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZIH0pO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gdGFibGUuYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgaW5kZXhOYW1lOiBHU0lfTkFNRSxcbiAgICAgIHBhcnRpdGlvbktleTogR1NJX1BBUlRJVElPTl9LRVksXG4gICAgICBzb3J0S2V5OiBHU0lfU09SVF9LRVksXG4gICAgICBwcm9qZWN0aW9uVHlwZTogUHJvamVjdGlvblR5cGUuSU5DTFVERSxcbiAgICAgIG5vbktleUF0dHJpYnV0ZXM6IFtHU0lfTk9OX0tFWSwgVEFCTEVfUEFSVElUSU9OX0tFWS5uYW1lXVxuICAgICAgLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm1heC1saW5lLWxlbmd0aFxuICAgIH0pLCAvYSBrZXkgYXR0cmlidXRlLCBoYXNoS2V5LCBpcyBwYXJ0IG9mIGEgbGlzdCBvZiBub24ta2V5IGF0dHJpYnV0ZXMsIGdzaU5vbktleSxoYXNoS2V5LCB3aGljaCBpcyBub3QgYWxsb3dlZCBzaW5jZSBhbGwga2V5IGF0dHJpYnV0ZXMgYXJlIGFkZGVkIGF1dG9tYXRpY2FsbHkgYW5kIHRoaXMgY29uZmlndXJhdGlvbiBjYXVzZXMgc3RhY2sgY3JlYXRpb24gZmFpbHVyZS8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Vycm9yIHdoZW4gYWRkaW5nIGEgZ2xvYmFsIHNlY29uZGFyeSBpbmRleCB3aXRoIHJlYWQgb3Igd3JpdGUgY2FwYWNpdHkgb24gYSBQQVlfUEVSX1JFUVVFU1QgdGFibGUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwge1xuICAgICAgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLFxuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVFxuICAgIH0pO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gdGFibGUuYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgaW5kZXhOYW1lOiBHU0lfTkFNRSxcbiAgICAgIHBhcnRpdGlvbktleTogR1NJX1BBUlRJVElPTl9LRVksXG4gICAgICBzb3J0S2V5OiBHU0lfU09SVF9LRVksXG4gICAgICByZWFkQ2FwYWNpdHk6IDFcbiAgICB9KSk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gdGFibGUuYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgaW5kZXhOYW1lOiBHU0lfTkFNRSxcbiAgICAgIHBhcnRpdGlvbktleTogR1NJX1BBUlRJVElPTl9LRVksXG4gICAgICBzb3J0S2V5OiBHU0lfU09SVF9LRVksXG4gICAgICB3cml0ZUNhcGFjaXR5OiAxXG4gICAgfSkpO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IHRhYmxlLmFkZEdsb2JhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgIGluZGV4TmFtZTogR1NJX05BTUUsXG4gICAgICBwYXJ0aXRpb25LZXk6IEdTSV9QQVJUSVRJT05fS0VZLFxuICAgICAgc29ydEtleTogR1NJX1NPUlRfS0VZLFxuICAgICAgcmVhZENhcGFjaXR5OiAxLFxuICAgICAgd3JpdGVDYXBhY2l0eTogMVxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd3aGVuIGFkZGluZyBtdWx0aXBsZSBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4ZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwgeyBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZIH0pO1xuICAgIGNvbnN0IGdzaUdlbmVyYXRvciA9IEdTSV9HRU5FUkFUT1IoKTtcbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IDU7IGkrKykge1xuICAgICAgdGFibGUuYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoZ3NpR2VuZXJhdG9yLm5leHQoKS52YWx1ZSk7XG4gICAgfVxuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RHluYW1vREI6OlRhYmxlJyxcbiAgICAgIHtcbiAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgQXR0cmlidXRlVHlwZTogJ1MnIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9LFxuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaUhhc2hLZXkwJywgQXR0cmlidXRlVHlwZTogJ1MnIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpSGFzaEtleTEnLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdnc2lIYXNoS2V5MicsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9LFxuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaUhhc2hLZXkzJywgQXR0cmlidXRlVHlwZTogJ1MnIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpSGFzaEtleTQnLCBBdHRyaWJ1dGVUeXBlOiAnUycgfVxuICAgICAgICBdLFxuICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnc29ydEtleScsIEtleVR5cGU6ICdSQU5HRScgfVxuICAgICAgICBdLFxuICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDUsIFdyaXRlQ2FwYWNpdHlVbml0czogNSB9LFxuICAgICAgICBHbG9iYWxTZWNvbmRhcnlJbmRleGVzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgSW5kZXhOYW1lOiAnTXlHU0kwJyxcbiAgICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdnc2lIYXNoS2V5MCcsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFByb2plY3Rpb246IHsgUHJvamVjdGlvblR5cGU6ICdBTEwnIH0sXG4gICAgICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDUsIFdyaXRlQ2FwYWNpdHlVbml0czogNSB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBJbmRleE5hbWU6ICdNeUdTSTEnLFxuICAgICAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaUhhc2hLZXkxJywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgUHJvamVjdGlvbjogeyBQcm9qZWN0aW9uVHlwZTogJ0FMTCcgfSxcbiAgICAgICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNSwgV3JpdGVDYXBhY2l0eVVuaXRzOiA1IH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEluZGV4TmFtZTogJ015R1NJMicsXG4gICAgICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpSGFzaEtleTInLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBQcm9qZWN0aW9uOiB7IFByb2plY3Rpb25UeXBlOiAnQUxMJyB9LFxuICAgICAgICAgICAgUHJvdmlzaW9uZWRUaHJvdWdocHV0OiB7IFJlYWRDYXBhY2l0eVVuaXRzOiA1LCBXcml0ZUNhcGFjaXR5VW5pdHM6IDUgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgSW5kZXhOYW1lOiAnTXlHU0kzJyxcbiAgICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdnc2lIYXNoS2V5MycsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFByb2plY3Rpb246IHsgUHJvamVjdGlvblR5cGU6ICdBTEwnIH0sXG4gICAgICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDUsIFdyaXRlQ2FwYWNpdHlVbml0czogNSB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBJbmRleE5hbWU6ICdNeUdTSTQnLFxuICAgICAgICAgICAgS2V5U2NoZW1hOiBbXG4gICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2dzaUhhc2hLZXk0JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgUHJvamVjdGlvbjogeyBQcm9qZWN0aW9uVHlwZTogJ0FMTCcgfSxcbiAgICAgICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNSwgV3JpdGVDYXBhY2l0eVVuaXRzOiA1IH1cbiAgICAgICAgICB9LFxuICAgICAgICBdXG4gICAgICB9XG4gICAgKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3doZW4gYWRkaW5nIGEgZ2xvYmFsIHNlY29uZGFyeSBpbmRleCB3aXRob3V0IHNwZWNpZnlpbmcgcmVhZCBhbmQgd3JpdGUgY2FwYWNpdHknKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwgeyBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVksIHNvcnRLZXk6IFRBQkxFX1NPUlRfS0VZIH0pO1xuXG4gICAgdGFibGUuYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgaW5kZXhOYW1lOiBHU0lfTkFNRSxcbiAgICAgIHBhcnRpdGlvbktleTogR1NJX1BBUlRJVElPTl9LRVksXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAge1xuICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpSGFzaEtleScsIEF0dHJpYnV0ZVR5cGU6ICdTJyB9XG4gICAgICAgIF0sXG4gICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgS2V5VHlwZTogJ1JBTkdFJyB9XG4gICAgICAgIF0sXG4gICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNSwgV3JpdGVDYXBhY2l0eVVuaXRzOiA1IH0sXG4gICAgICAgIEdsb2JhbFNlY29uZGFyeUluZGV4ZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBJbmRleE5hbWU6ICdNeUdTSScsXG4gICAgICAgICAgICBLZXlTY2hlbWE6IFtcbiAgICAgICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnZ3NpSGFzaEtleScsIEtleVR5cGU6ICdIQVNIJyB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFByb2plY3Rpb246IHsgUHJvamVjdGlvblR5cGU6ICdBTEwnIH0sXG4gICAgICAgICAgICBQcm92aXNpb25lZFRocm91Z2hwdXQ6IHsgUmVhZENhcGFjaXR5VW5pdHM6IDUsIFdyaXRlQ2FwYWNpdHlVbml0czogNSB9XG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9XG4gICAgKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3doZW4gYWRkaW5nIGEgbG9jYWwgc2Vjb25kYXJ5IGluZGV4IHdpdGggaGFzaCArIHJhbmdlIGtleScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7IHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSwgc29ydEtleTogVEFCTEVfU09SVF9LRVkgfSk7XG5cbiAgICB0YWJsZS5hZGRMb2NhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgIGluZGV4TmFtZTogTFNJX05BTUUsXG4gICAgICBzb3J0S2V5OiBMU0lfU09SVF9LRVksXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAge1xuICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnbHNpU29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9XG4gICAgICAgIF0sXG4gICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgS2V5VHlwZTogJ1JBTkdFJyB9XG4gICAgICAgIF0sXG4gICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNSwgV3JpdGVDYXBhY2l0eVVuaXRzOiA1IH0sXG4gICAgICAgIExvY2FsU2Vjb25kYXJ5SW5kZXhlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEluZGV4TmFtZTogJ015TFNJJyxcbiAgICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2xzaVNvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBQcm9qZWN0aW9uOiB7IFByb2plY3Rpb25UeXBlOiAnQUxMJyB9LFxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgIH1cbiAgICApKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnd2hlbiBhZGRpbmcgYSBsb2NhbCBzZWNvbmRhcnkgaW5kZXggd2l0aCBwcm9qZWN0aW9uIHR5cGUgS0VZU19PTkxZJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHsgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLCBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWSB9KTtcbiAgICB0YWJsZS5hZGRMb2NhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgIGluZGV4TmFtZTogTFNJX05BTUUsXG4gICAgICBzb3J0S2V5OiBMU0lfU09SVF9LRVksXG4gICAgICBwcm9qZWN0aW9uVHlwZTogUHJvamVjdGlvblR5cGUuS0VZU19PTkxZXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAge1xuICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnbHNpU29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9XG4gICAgICAgIF0sXG4gICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgS2V5VHlwZTogJ1JBTkdFJyB9XG4gICAgICAgIF0sXG4gICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNSwgV3JpdGVDYXBhY2l0eVVuaXRzOiA1IH0sXG4gICAgICAgIExvY2FsU2Vjb25kYXJ5SW5kZXhlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEluZGV4TmFtZTogJ015TFNJJyxcbiAgICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2xzaVNvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBQcm9qZWN0aW9uOiB7IFByb2plY3Rpb25UeXBlOiAnS0VZU19PTkxZJyB9LFxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgIH1cbiAgICApKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnd2hlbiBhZGRpbmcgYSBsb2NhbCBzZWNvbmRhcnkgaW5kZXggd2l0aCBwcm9qZWN0aW9uIHR5cGUgSU5DTFVERScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7IHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSwgc29ydEtleTogVEFCTEVfU09SVF9LRVkgfSk7XG4gICAgY29uc3QgbHNpTm9uS2V5QXR0cmlidXRlR2VuZXJhdG9yID0gTk9OX0tFWV9BVFRSSUJVVEVfR0VORVJBVE9SKExTSV9OT05fS0VZKTtcbiAgICB0YWJsZS5hZGRMb2NhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgIGluZGV4TmFtZTogTFNJX05BTUUsXG4gICAgICBzb3J0S2V5OiBMU0lfU09SVF9LRVksXG4gICAgICBwcm9qZWN0aW9uVHlwZTogUHJvamVjdGlvblR5cGUuSU5DTFVERSxcbiAgICAgIG5vbktleUF0dHJpYnV0ZXM6IFsgbHNpTm9uS2V5QXR0cmlidXRlR2VuZXJhdG9yLm5leHQoKS52YWx1ZSwgbHNpTm9uS2V5QXR0cmlidXRlR2VuZXJhdG9yLm5leHQoKS52YWx1ZSBdXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEeW5hbW9EQjo6VGFibGUnLFxuICAgICAge1xuICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBBdHRyaWJ1dGVUeXBlOiAnUycgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgQXR0cmlidXRlVHlwZTogJ04nIH0sXG4gICAgICAgICAgeyBBdHRyaWJ1dGVOYW1lOiAnbHNpU29ydEtleScsIEF0dHJpYnV0ZVR5cGU6ICdOJyB9XG4gICAgICAgIF0sXG4gICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2hhc2hLZXknLCBLZXlUeXBlOiAnSEFTSCcgfSxcbiAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdzb3J0S2V5JywgS2V5VHlwZTogJ1JBTkdFJyB9XG4gICAgICAgIF0sXG4gICAgICAgIFByb3Zpc2lvbmVkVGhyb3VnaHB1dDogeyBSZWFkQ2FwYWNpdHlVbml0czogNSwgV3JpdGVDYXBhY2l0eVVuaXRzOiA1IH0sXG4gICAgICAgIExvY2FsU2Vjb25kYXJ5SW5kZXhlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEluZGV4TmFtZTogJ015TFNJJyxcbiAgICAgICAgICAgIEtleVNjaGVtYTogW1xuICAgICAgICAgICAgICB7IEF0dHJpYnV0ZU5hbWU6ICdoYXNoS2V5JywgS2V5VHlwZTogJ0hBU0gnIH0sXG4gICAgICAgICAgICAgIHsgQXR0cmlidXRlTmFtZTogJ2xzaVNvcnRLZXknLCBLZXlUeXBlOiAnUkFOR0UnIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBQcm9qZWN0aW9uOiB7IE5vbktleUF0dHJpYnV0ZXM6IFsnbHNpTm9uS2V5MCcsICdsc2lOb25LZXkxJ10sIFByb2plY3Rpb25UeXBlOiAnSU5DTFVERScgfSxcbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICB9XG4gICAgKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Vycm9yIHdoZW4gYWRkaW5nIG1vcmUgdGhhbiA1IGxvY2FsIHNlY29uZGFyeSBpbmRleGVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHsgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLCBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWSB9KTtcbiAgICBjb25zdCBsc2lHZW5lcmF0b3IgPSBMU0lfR0VORVJBVE9SKCk7XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCA1OyBpKyspIHtcbiAgICAgIHRhYmxlLmFkZExvY2FsU2Vjb25kYXJ5SW5kZXgobHNpR2VuZXJhdG9yLm5leHQoKS52YWx1ZSk7XG4gICAgfVxuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gdGFibGUuYWRkTG9jYWxTZWNvbmRhcnlJbmRleChsc2lHZW5lcmF0b3IubmV4dCgpLnZhbHVlKSxcbiAgICAgIC9hIG1heGltdW0gbnVtYmVyIG9mIGxvY2FsIHNlY29uZGFyeSBpbmRleCBwZXIgdGFibGUgaXMgNS8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Vycm9yIHdoZW4gYWRkaW5nIGEgbG9jYWwgc2Vjb25kYXJ5IGluZGV4IHdpdGggdGhlIG5hbWUgb2YgYSBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHsgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZLCBzb3J0S2V5OiBUQUJMRV9TT1JUX0tFWSB9KTtcbiAgICB0YWJsZS5hZGRHbG9iYWxTZWNvbmRhcnlJbmRleCh7XG4gICAgICBpbmRleE5hbWU6ICdTZWNvbmRhcnlJbmRleCcsXG4gICAgICBwYXJ0aXRpb25LZXk6IEdTSV9QQVJUSVRJT05fS0VZXG4gICAgfSk7XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB0YWJsZS5hZGRMb2NhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgIGluZGV4TmFtZTogJ1NlY29uZGFyeUluZGV4JyxcbiAgICAgIHNvcnRLZXk6IExTSV9TT1JUX0tFWVxuICAgIH0pLCAvYSBkdXBsaWNhdGUgaW5kZXggbmFtZSwgU2Vjb25kYXJ5SW5kZXgsIGlzIG5vdCBhbGxvd2VkLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZXJyb3Igd2hlbiB2YWxpZGF0aW5nIGNvbnN0cnVjdCBpZiBhIGxvY2FsIHNlY29uZGFyeSBpbmRleCBleGlzdHMgd2l0aG91dCBhIHNvcnQga2V5IG9mIHRoZSB0YWJsZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7IHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSB9KTtcblxuICAgIHRhYmxlLmFkZExvY2FsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgaW5kZXhOYW1lOiBMU0lfTkFNRSxcbiAgICAgIHNvcnRLZXk6IExTSV9TT1JUX0tFWVxuICAgIH0pO1xuXG4gICAgY29uc3QgZXJyb3JzID0gQ29uc3RydWN0Tm9kZS52YWxpZGF0ZSh0YWJsZS5ub2RlKTtcblxuICAgIHRlc3Quc3RyaWN0RXF1YWwoMSwgZXJyb3JzLmxlbmd0aCk7XG4gICAgdGVzdC5zdHJpY3RFcXVhbCgnYSBzb3J0IGtleSBvZiB0aGUgdGFibGUgbXVzdCBiZSBzcGVjaWZpZWQgdG8gYWRkIGxvY2FsIHNlY29uZGFyeSBpbmRleGVzJywgZXJyb3JzWzBdLm1lc3NhZ2UpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NhbiBlbmFibGUgUmVhZCBBdXRvU2NhbGluZycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHsgcmVhZENhcGFjaXR5OiA0Miwgd3JpdGVDYXBhY2l0eTogMTMzNywgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIHRhYmxlLmF1dG9TY2FsZVJlYWRDYXBhY2l0eSh7IG1pbkNhcGFjaXR5OiA1MCwgbWF4Q2FwYWNpdHk6IDUwMCB9KS5zY2FsZU9uVXRpbGl6YXRpb24oeyB0YXJnZXRVdGlsaXphdGlvblBlcmNlbnQ6IDc1IH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkFwcGxpY2F0aW9uQXV0b1NjYWxpbmc6OlNjYWxhYmxlVGFyZ2V0Jywge1xuICAgICAgTWF4Q2FwYWNpdHk6IDUwMCxcbiAgICAgIE1pbkNhcGFjaXR5OiA1MCxcbiAgICAgIFNjYWxhYmxlRGltZW5zaW9uOiAnZHluYW1vZGI6dGFibGU6UmVhZENhcGFjaXR5VW5pdHMnLFxuICAgICAgU2VydmljZU5hbWVzcGFjZTogJ2R5bmFtb2RiJ1xuICAgIH0pKTtcbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcHBsaWNhdGlvbkF1dG9TY2FsaW5nOjpTY2FsaW5nUG9saWN5Jywge1xuICAgICAgUG9saWN5VHlwZTogJ1RhcmdldFRyYWNraW5nU2NhbGluZycsXG4gICAgICBUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIFByZWRlZmluZWRNZXRyaWNTcGVjaWZpY2F0aW9uOiB7IFByZWRlZmluZWRNZXRyaWNUeXBlOiAnRHluYW1vREJSZWFkQ2FwYWNpdHlVdGlsaXphdGlvbicgfSxcbiAgICAgICAgVGFyZ2V0VmFsdWU6IDc1XG4gICAgICB9XG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NhbiBlbmFibGUgV3JpdGUgQXV0b1NjYWxpbmcnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7IHJlYWRDYXBhY2l0eTogNDIsIHdyaXRlQ2FwYWNpdHk6IDEzMzcsIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSB9KTtcblxuICAgIC8vIFdIRU5cbiAgICB0YWJsZS5hdXRvU2NhbGVXcml0ZUNhcGFjaXR5KHsgbWluQ2FwYWNpdHk6IDUwLCBtYXhDYXBhY2l0eTogNTAwIH0pLnNjYWxlT25VdGlsaXphdGlvbih7IHRhcmdldFV0aWxpemF0aW9uUGVyY2VudDogNzUgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXBwbGljYXRpb25BdXRvU2NhbGluZzo6U2NhbGFibGVUYXJnZXQnLCB7XG4gICAgICBNYXhDYXBhY2l0eTogNTAwLFxuICAgICAgTWluQ2FwYWNpdHk6IDUwLFxuICAgICAgU2NhbGFibGVEaW1lbnNpb246ICdkeW5hbW9kYjp0YWJsZTpXcml0ZUNhcGFjaXR5VW5pdHMnLFxuICAgICAgU2VydmljZU5hbWVzcGFjZTogJ2R5bmFtb2RiJ1xuICAgIH0pKTtcbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcHBsaWNhdGlvbkF1dG9TY2FsaW5nOjpTY2FsaW5nUG9saWN5Jywge1xuICAgICAgUG9saWN5VHlwZTogJ1RhcmdldFRyYWNraW5nU2NhbGluZycsXG4gICAgICBUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIFByZWRlZmluZWRNZXRyaWNTcGVjaWZpY2F0aW9uOiB7IFByZWRlZmluZWRNZXRyaWNUeXBlOiAnRHluYW1vREJXcml0ZUNhcGFjaXR5VXRpbGl6YXRpb24nIH0sXG4gICAgICAgIFRhcmdldFZhbHVlOiA3NVxuICAgICAgfVxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW5ub3QgZW5hYmxlIEF1dG9TY2FsaW5nIHR3aWNlIG9uIHRoZSBzYW1lIHByb3BlcnR5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCBDT05TVFJVQ1RfTkFNRSwgeyByZWFkQ2FwYWNpdHk6IDQyLCB3cml0ZUNhcGFjaXR5OiAxMzM3LCBwYXJ0aXRpb25LZXk6IFRBQkxFX1BBUlRJVElPTl9LRVkgfSk7XG4gICAgdGFibGUuYXV0b1NjYWxlUmVhZENhcGFjaXR5KHsgbWluQ2FwYWNpdHk6IDUwLCBtYXhDYXBhY2l0eTogNTAwIH0pLnNjYWxlT25VdGlsaXphdGlvbih7IHRhcmdldFV0aWxpemF0aW9uUGVyY2VudDogNzUgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgdGFibGUuYXV0b1NjYWxlUmVhZENhcGFjaXR5KHsgbWluQ2FwYWNpdHk6IDUwLCBtYXhDYXBhY2l0eTogNTAwIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Vycm9yIHdoZW4gZW5hYmxpbmcgQXV0b1NjYWxpbmcgb24gdGhlIFBBWV9QRVJfUkVRVUVTVCB0YWJsZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHsgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCwgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZIH0pO1xuICAgIHRhYmxlLmFkZEdsb2JhbFNlY29uZGFyeUluZGV4KHtcbiAgICAgIGluZGV4TmFtZTogR1NJX05BTUUsXG4gICAgICBwYXJ0aXRpb25LZXk6IEdTSV9QQVJUSVRJT05fS0VZXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgdGFibGUuYXV0b1NjYWxlUmVhZENhcGFjaXR5KHsgbWluQ2FwYWNpdHk6IDUwLCBtYXhDYXBhY2l0eTogNTAwIH0pO1xuICAgIH0pO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIHRhYmxlLmF1dG9TY2FsZVdyaXRlQ2FwYWNpdHkoeyBtaW5DYXBhY2l0eTogNTAsIG1heENhcGFjaXR5OiA1MDAgfSk7XG4gICAgfSk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gdGFibGUuYXV0b1NjYWxlR2xvYmFsU2Vjb25kYXJ5SW5kZXhSZWFkQ2FwYWNpdHkoR1NJX05BTUUsIHtcbiAgICAgIG1pbkNhcGFjaXR5OiAxLFxuICAgICAgbWF4Q2FwYWNpdHk6IDVcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZXJyb3Igd2hlbiBzcGVjaWZ5aW5nIFJlYWQgQXV0byBTY2FsaW5nIHdpdGggaW52YWxpZCBzY2FsaW5nVGFyZ2V0VmFsdWUgPCAxMCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgQ09OU1RSVUNUX05BTUUsIHsgcmVhZENhcGFjaXR5OiA0Miwgd3JpdGVDYXBhY2l0eTogMTMzNywgcGFydGl0aW9uS2V5OiBUQUJMRV9QQVJUSVRJT05fS0VZIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIHRhYmxlLmF1dG9TY2FsZVJlYWRDYXBhY2l0eSh7IG1pbkNhcGFjaXR5OiA1MCwgbWF4Q2FwYWNpdHk6IDUwMCB9KS5zY2FsZU9uVXRpbGl6YXRpb24oeyB0YXJnZXRVdGlsaXphdGlvblBlcmNlbnQ6IDUgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZXJyb3Igd2hlbiBzcGVjaWZ5aW5nIFJlYWQgQXV0byBTY2FsaW5nIHdpdGggaW52YWxpZCBtaW5pbXVtQ2FwYWNpdHknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7IHJlYWRDYXBhY2l0eTogNDIsIHdyaXRlQ2FwYWNpdHk6IDEzMzcsIHBhcnRpdGlvbktleTogVEFCTEVfUEFSVElUSU9OX0tFWSB9KTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB0YWJsZS5hdXRvU2NhbGVSZWFkQ2FwYWNpdHkoeyBtaW5DYXBhY2l0eTogMTAsIG1heENhcGFjaXR5OiA1IH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW4gYXV0b3NjYWxlIG9uIGEgc2NoZWR1bGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssIENPTlNUUlVDVF9OQU1FLCB7XG4gICAgICByZWFkQ2FwYWNpdHk6IDQyLFxuICAgICAgd3JpdGVDYXBhY2l0eTogMTMzNyxcbiAgICAgIHBhcnRpdGlvbktleTogeyBuYW1lOiAnSGFzaCcsIHR5cGU6IEF0dHJpYnV0ZVR5cGUuU1RSSU5HIH1cbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBzY2FsaW5nID0gdGFibGUuYXV0b1NjYWxlUmVhZENhcGFjaXR5KHsgbWluQ2FwYWNpdHk6IDEsIG1heENhcGFjaXR5OiAxMDAgfSk7XG4gICAgc2NhbGluZy5zY2FsZU9uU2NoZWR1bGUoJ1NhdmVNb25leUJ5Tm90U2NhbGluZ1VwJywge1xuICAgICAgc2NoZWR1bGU6IGFwcHNjYWxpbmcuU2NoZWR1bGUuY3Jvbih7fSksXG4gICAgICBtYXhDYXBhY2l0eTogMTBcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcHBsaWNhdGlvbkF1dG9TY2FsaW5nOjpTY2FsYWJsZVRhcmdldCcsIHtcbiAgICAgIFNjaGVkdWxlZEFjdGlvbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFNjYWxhYmxlVGFyZ2V0QWN0aW9uOiB7IFwiTWF4Q2FwYWNpdHlcIjogMTAgfSxcbiAgICAgICAgICBTY2hlZHVsZTogXCJjcm9uKCogKiAqICogPyAqKVwiLFxuICAgICAgICAgIFNjaGVkdWxlZEFjdGlvbk5hbWU6IFwiU2F2ZU1vbmV5QnlOb3RTY2FsaW5nVXBcIlxuICAgICAgICB9XG4gICAgICBdXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ21ldHJpY3MnOiB7XG4gICAgJ0NhbiB1c2UgbWV0cmljQ29uc3VtZWRSZWFkQ2FwYWNpdHlVbml0cyBvbiBhIER5bmFtb2RiIFRhYmxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCAnVGFibGUnLCB7XG4gICAgICAgIHBhcnRpdGlvbktleTogeyBuYW1lOiAnaWQnLCB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklORyB9XG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh0YWJsZS5tZXRyaWNDb25zdW1lZFJlYWRDYXBhY2l0eVVuaXRzKCkpLCB7XG4gICAgICAgIHBlcmlvZDogeyBhbW91bnQ6IDUsIHVuaXQ6IHsgbGFiZWw6ICdtaW51dGVzJywgaW5TZWNvbmRzOiA2MCB9IH0sXG4gICAgICAgIGRpbWVuc2lvbnM6IHsgVGFibGVOYW1lOiB7IFJlZjogJ1RhYmxlQ0QxMTdGQTEnIH0gfSxcbiAgICAgICAgbmFtZXNwYWNlOiAnQVdTL0R5bmFtb0RCJyxcbiAgICAgICAgbWV0cmljTmFtZTogJ0NvbnN1bWVkUmVhZENhcGFjaXR5VW5pdHMnLFxuICAgICAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnQ2FuIHVzZSBtZXRyaWNDb25zdW1lZFdyaXRlQ2FwYWNpdHlVbml0cyBvbiBhIER5bmFtb2RiIFRhYmxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCAnVGFibGUnLCB7XG4gICAgICAgIHBhcnRpdGlvbktleTogeyBuYW1lOiAnaWQnLCB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklORyB9XG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh0YWJsZS5tZXRyaWNDb25zdW1lZFdyaXRlQ2FwYWNpdHlVbml0cygpKSwge1xuICAgICAgICBwZXJpb2Q6IHsgYW1vdW50OiA1LCB1bml0OiB7IGxhYmVsOiAnbWludXRlcycsIGluU2Vjb25kczogNjAgfSB9LFxuICAgICAgICBkaW1lbnNpb25zOiB7IFRhYmxlTmFtZTogeyBSZWY6ICdUYWJsZUNEMTE3RkExJyB9IH0sXG4gICAgICAgIG5hbWVzcGFjZTogJ0FXUy9EeW5hbW9EQicsXG4gICAgICAgIG1ldHJpY05hbWU6ICdDb25zdW1lZFdyaXRlQ2FwYWNpdHlVbml0cycsXG4gICAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgICB9KTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdDYW4gdXNlIG1ldHJpY1N5c3RlbUVycm9ycyBvbiBhIER5bmFtb2RiIFRhYmxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCAnVGFibGUnLCB7XG4gICAgICAgIHBhcnRpdGlvbktleTogeyBuYW1lOiAnaWQnLCB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklORyB9XG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh0YWJsZS5tZXRyaWNTeXN0ZW1FcnJvcnMoKSksIHtcbiAgICAgICAgcGVyaW9kOiB7IGFtb3VudDogNSwgdW5pdDogeyBsYWJlbDogJ21pbnV0ZXMnLCBpblNlY29uZHM6IDYwIH0gfSxcbiAgICAgICAgZGltZW5zaW9uczogeyBUYWJsZU5hbWU6IHsgUmVmOiAnVGFibGVDRDExN0ZBMScgfSB9LFxuICAgICAgICBuYW1lc3BhY2U6ICdBV1MvRHluYW1vREInLFxuICAgICAgICBtZXRyaWNOYW1lOiAnU3lzdGVtRXJyb3JzJyxcbiAgICAgICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ0NhbiB1c2UgbWV0cmljVXNlckVycm9ycyBvbiBhIER5bmFtb2RiIFRhYmxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCAnVGFibGUnLCB7XG4gICAgICAgIHBhcnRpdGlvbktleTogeyBuYW1lOiAnaWQnLCB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklORyB9XG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh0YWJsZS5tZXRyaWNVc2VyRXJyb3JzKCkpLCB7XG4gICAgICAgIHBlcmlvZDogeyBhbW91bnQ6IDUsIHVuaXQ6IHsgbGFiZWw6ICdtaW51dGVzJywgaW5TZWNvbmRzOiA2MCB9IH0sXG4gICAgICAgIGRpbWVuc2lvbnM6IHsgVGFibGVOYW1lOiB7IFJlZjogJ1RhYmxlQ0QxMTdGQTEnIH0gfSxcbiAgICAgICAgbmFtZXNwYWNlOiAnQVdTL0R5bmFtb0RCJyxcbiAgICAgICAgbWV0cmljTmFtZTogJ1VzZXJFcnJvcnMnLFxuICAgICAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnQ2FuIHVzZSBtZXRyaWNDb25kaXRpb25hbENoZWNrRmFpbGVkUmVxdWVzdHMgb24gYSBEeW5hbW9kYiBUYWJsZScodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgJ1RhYmxlJywge1xuICAgICAgICBwYXJ0aXRpb25LZXk6IHsgbmFtZTogJ2lkJywgdHlwZTogQXR0cmlidXRlVHlwZS5TVFJJTkcgfVxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUodGFibGUubWV0cmljQ29uZGl0aW9uYWxDaGVja0ZhaWxlZFJlcXVlc3RzKCkpLCB7XG4gICAgICAgIHBlcmlvZDogeyBhbW91bnQ6IDUsIHVuaXQ6IHsgbGFiZWw6ICdtaW51dGVzJywgaW5TZWNvbmRzOiA2MCB9IH0sXG4gICAgICAgIGRpbWVuc2lvbnM6IHsgVGFibGVOYW1lOiB7IFJlZjogJ1RhYmxlQ0QxMTdGQTEnIH0gfSxcbiAgICAgICAgbmFtZXNwYWNlOiAnQVdTL0R5bmFtb0RCJyxcbiAgICAgICAgbWV0cmljTmFtZTogJ0NvbmRpdGlvbmFsQ2hlY2tGYWlsZWRSZXF1ZXN0cycsXG4gICAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgICB9KTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdDYW4gdXNlIG1ldHJpY1N1Y2Nlc3NmdWxSZXF1ZXN0TGF0ZW5jeSBvbiBhIER5bmFtb2RiIFRhYmxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCAnVGFibGUnLCB7XG4gICAgICAgIHBhcnRpdGlvbktleTogeyBuYW1lOiAnaWQnLCB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklORyB9XG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh0YWJsZS5tZXRyaWNTdWNjZXNzZnVsUmVxdWVzdExhdGVuY3koKSksIHtcbiAgICAgICAgcGVyaW9kOiB7IGFtb3VudDogNSwgdW5pdDogeyBsYWJlbDogJ21pbnV0ZXMnLCBpblNlY29uZHM6IDYwIH0gfSxcbiAgICAgICAgZGltZW5zaW9uczogeyBUYWJsZU5hbWU6IHsgUmVmOiAnVGFibGVDRDExN0ZBMScgfSB9LFxuICAgICAgICBuYW1lc3BhY2U6ICdBV1MvRHluYW1vREInLFxuICAgICAgICBtZXRyaWNOYW1lOiAnU3VjY2Vzc2Z1bFJlcXVlc3RMYXRlbmN5JyxcbiAgICAgICAgc3RhdGlzdGljOiAnQXZlcmFnZScsXG4gICAgICB9KTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICB9LFxuXG4gICdncmFudHMnOiB7XG5cbiAgICAnXCJncmFudFwiIGFsbG93cyBhZGRpbmcgYXJiaXRyYXJ5IGFjdGlvbnMgYXNzb2NpYXRlZCB3aXRoIHRoaXMgdGFibGUgcmVzb3VyY2UnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3RHcmFudCh0ZXN0LFxuICAgICAgICBbICdhY3Rpb24xJywgJ2FjdGlvbjInIF0sIChwLCB0KSA9PiB0LmdyYW50KHAsICdkeW5hbW9kYjphY3Rpb24xJywgJ2R5bmFtb2RiOmFjdGlvbjInKSk7XG4gICAgfSxcblxuICAgICdcImdyYW50UmVhZERhdGFcIiBhbGxvd3MgdGhlIHByaW5jaXBhbCB0byByZWFkIGRhdGEgZnJvbSB0aGUgdGFibGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3RHcmFudCh0ZXN0LFxuICAgICAgICBbICdCYXRjaEdldEl0ZW0nLCAnR2V0UmVjb3JkcycsICdHZXRTaGFyZEl0ZXJhdG9yJywgJ1F1ZXJ5JywgJ0dldEl0ZW0nLCAnU2NhbicgXSwgKHAsIHQpID0+IHQuZ3JhbnRSZWFkRGF0YShwKSk7XG4gICAgfSxcblxuICAgICdcImdyYW50V3JpdGVEYXRhXCIgYWxsb3dzIHRoZSBwcmluY2lwYWwgdG8gd3JpdGUgZGF0YSB0byB0aGUgdGFibGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3RHcmFudCh0ZXN0LCBbXG4gICAgICAgICdCYXRjaFdyaXRlSXRlbScsICdQdXRJdGVtJywgJ1VwZGF0ZUl0ZW0nLCAnRGVsZXRlSXRlbScgXSwgKHAsIHQpID0+IHQuZ3JhbnRXcml0ZURhdGEocCkpO1xuICAgIH0sXG5cbiAgICAnXCJncmFudFJlYWRXcml0ZURhdGFcIiBhbGxvd3MgdGhlIHByaW5jaXBhbCB0byByZWFkL3dyaXRlIGRhdGEnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3RHcmFudCh0ZXN0LCBbXG4gICAgICAgICdCYXRjaEdldEl0ZW0nLCAnR2V0UmVjb3JkcycsICdHZXRTaGFyZEl0ZXJhdG9yJywgJ1F1ZXJ5JywgJ0dldEl0ZW0nLCAnU2NhbicsXG4gICAgICAgICdCYXRjaFdyaXRlSXRlbScsICdQdXRJdGVtJywgJ1VwZGF0ZUl0ZW0nLCAnRGVsZXRlSXRlbScgXSwgKHAsIHQpID0+IHQuZ3JhbnRSZWFkV3JpdGVEYXRhKHApKTtcbiAgICB9LFxuXG4gICAgJ1wiZ3JhbnRGdWxsQWNjZXNzXCIgYWxsb3dzIHRoZSBwcmluY2lwYWwgdG8gcGVyZm9ybSBhbnkgYWN0aW9uIG9uIHRoZSB0YWJsZSAoXCIqXCIpJyh0ZXN0OiBUZXN0KSB7XG4gICAgICB0ZXN0R3JhbnQodGVzdCwgWyAnKicgXSwgKHAsIHQpID0+IHQuZ3JhbnRGdWxsQWNjZXNzKHApKTtcbiAgICB9LFxuXG4gICAgJ1wiVGFibGUuZ3JhbnRMaXN0U3RyZWFtc1wiIGFsbG93cyBwcmluY2lwYWwgdG8gbGlzdCBhbGwgc3RyZWFtcycodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCB1c2VyID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAndXNlcicpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBUYWJsZS5ncmFudExpc3RTdHJlYW1zKHVzZXIpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBcImR5bmFtb2RiOkxpc3RTdHJlYW1zXCIsXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiBcIipcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgXCJWZXJzaW9uXCI6IFwiMjAxMi0xMC0xN1wiXG4gICAgICAgIH0sXG4gICAgICAgIFwiVXNlcnNcIjogWyB7IFwiUmVmXCI6IFwidXNlcjJDMkI1N0FFXCIgfSBdXG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ1wiZ3JhbnRUYWJsZUxpc3RTdHJlYW1zXCIgc2hvdWxkIGZhaWwgaWYgc3RyZWFtaW5nIGlzIG5vdCBlbmFibGVkIG9uIHRhYmxlXCInKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssICdteS10YWJsZScsIHtcbiAgICAgICAgcGFydGl0aW9uS2V5OiB7XG4gICAgICAgICAgbmFtZTogJ2lkJyxcbiAgICAgICAgICB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklOR1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICAgIGNvbnN0IHVzZXIgPSBuZXcgaWFtLlVzZXIoc3RhY2ssICd1c2VyJyk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IHRhYmxlLmdyYW50VGFibGVMaXN0U3RyZWFtcyh1c2VyKSwgL0R5bmFtb0RCIFN0cmVhbXMgbXVzdCBiZSBlbmFibGVkIG9uIHRoZSB0YWJsZSBteS10YWJsZS8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ1wiZ3JhbnRUYWJsZUxpc3RTdHJlYW1zXCIgYWxsb3dzIHByaW5jaXBhbCB0byBsaXN0IGFsbCBzdHJlYW1zIGZvciB0aGlzIHRhYmxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCAnbXktdGFibGUnLCB7XG4gICAgICAgIHBhcnRpdGlvbktleToge1xuICAgICAgICAgIG5hbWU6ICdpZCcsXG4gICAgICAgICAgdHlwZTogQXR0cmlidXRlVHlwZS5TVFJJTkdcbiAgICAgICAgfSxcbiAgICAgICAgc3RyZWFtOiBTdHJlYW1WaWV3VHlwZS5ORVdfSU1BR0VcbiAgICAgIH0pO1xuICAgICAgY29uc3QgdXNlciA9IG5ldyBpYW0uVXNlcihzdGFjaywgJ3VzZXInKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgdGFibGUuZ3JhbnRUYWJsZUxpc3RTdHJlYW1zKHVzZXIpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBcImR5bmFtb2RiOkxpc3RTdHJlYW1zXCIsXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiB7IFwiRm46OkpvaW5cIjogWyBcIlwiLCBbIHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJteXRhYmxlMDMyNEQ0NUNcIiwgXCJBcm5cIiBdIH0sIFwiL3N0cmVhbS8qXCIgXSBdIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICB9LFxuICAgICAgICBcIlVzZXJzXCI6IFsgeyBcIlJlZlwiOiBcInVzZXIyQzJCNTdBRVwiIH0gXVxuICAgICAgfSkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdcImdyYW50U3RyZWFtUmVhZFwiIHNob3VsZCBmYWlsIGlmIHN0cmVhbWluZyBpcyBub3QgZW5hYmxlZCBvbiB0YWJsZVwiJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCAnbXktdGFibGUnLCB7XG4gICAgICAgIHBhcnRpdGlvbktleToge1xuICAgICAgICAgIG5hbWU6ICdpZCcsXG4gICAgICAgICAgdHlwZTogQXR0cmlidXRlVHlwZS5TVFJJTkdcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgICBjb25zdCB1c2VyID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAndXNlcicpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB0YWJsZS5ncmFudFN0cmVhbVJlYWQodXNlciksIC9EeW5hbW9EQiBTdHJlYW1zIG11c3QgYmUgZW5hYmxlZCBvbiB0aGUgdGFibGUgbXktdGFibGUvKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdcImdyYW50U3RyZWFtUmVhZFwiIGFsbG93cyBwcmluY2lwYWwgdG8gcmVhZCBhbmQgZGVzY3JpYmUgdGhlIHRhYmxlIHN0cmVhbVwiJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhYmxlID0gbmV3IFRhYmxlKHN0YWNrLCAnbXktdGFibGUnLCB7XG4gICAgICAgIHBhcnRpdGlvbktleToge1xuICAgICAgICAgIG5hbWU6ICdpZCcsXG4gICAgICAgICAgdHlwZTogQXR0cmlidXRlVHlwZS5TVFJJTkdcbiAgICAgICAgfSxcbiAgICAgICAgc3RyZWFtOiBTdHJlYW1WaWV3VHlwZS5ORVdfSU1BR0VcbiAgICAgIH0pO1xuICAgICAgY29uc3QgdXNlciA9IG5ldyBpYW0uVXNlcihzdGFjaywgJ3VzZXInKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgdGFibGUuZ3JhbnRTdHJlYW1SZWFkKHVzZXIpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBcImR5bmFtb2RiOkxpc3RTdHJlYW1zXCIsXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiB7IFwiRm46OkpvaW5cIjogWyBcIlwiLCBbIHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJteXRhYmxlMDMyNEQ0NUNcIiwgXCJBcm5cIiBdIH0sIFwiL3N0cmVhbS8qXCIgXSBdIH1cbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiQWN0aW9uXCI6IFtcbiAgICAgICAgICAgICAgICBcImR5bmFtb2RiOkRlc2NyaWJlU3RyZWFtXCIsXG4gICAgICAgICAgICAgICAgXCJkeW5hbW9kYjpHZXRSZWNvcmRzXCIsXG4gICAgICAgICAgICAgICAgXCJkeW5hbW9kYjpHZXRTaGFyZEl0ZXJhdG9yXCJcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBcIlJlc291cmNlXCI6IHtcbiAgICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgICAgXCJteXRhYmxlMDMyNEQ0NUNcIixcbiAgICAgICAgICAgICAgICAgIFwiU3RyZWFtQXJuXCJcbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICB9LFxuICAgICAgICBcIlVzZXJzXCI6IFsgeyBcIlJlZlwiOiBcInVzZXIyQzJCNTdBRVwiIH0gXVxuICAgICAgfSkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnaWYgdGFibGUgaGFzIGFuIGluZGV4IGdyYW50IGdpdmVzIGFjY2VzcyB0byB0aGUgaW5kZXgnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShzdGFjaywgJ215LXRhYmxlJywgeyBwYXJ0aXRpb25LZXk6IHsgbmFtZTogJ0lEJywgdHlwZTogQXR0cmlidXRlVHlwZS5TVFJJTkcgfSB9KTtcbiAgICAgIHRhYmxlLmFkZEdsb2JhbFNlY29uZGFyeUluZGV4KHsgaW5kZXhOYW1lOiAnTXlJbmRleCcsIHBhcnRpdGlvbktleTogeyBuYW1lOiAnQWdlJywgdHlwZTogQXR0cmlidXRlVHlwZS5OVU1CRVIgfX0pO1xuICAgICAgY29uc3QgdXNlciA9IG5ldyBpYW0uVXNlcihzdGFjaywgJ3VzZXInKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgdGFibGUuZ3JhbnRSZWFkRGF0YSh1c2VyKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogW1xuICAgICAgICAgICAgICAgICdkeW5hbW9kYjpCYXRjaEdldEl0ZW0nLFxuICAgICAgICAgICAgICAgICdkeW5hbW9kYjpHZXRSZWNvcmRzJyxcbiAgICAgICAgICAgICAgICAnZHluYW1vZGI6R2V0U2hhcmRJdGVyYXRvcicsXG4gICAgICAgICAgICAgICAgJ2R5bmFtb2RiOlF1ZXJ5JyxcbiAgICAgICAgICAgICAgICAnZHluYW1vZGI6R2V0SXRlbScsXG4gICAgICAgICAgICAgICAgJ2R5bmFtb2RiOlNjYW4nXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiBbXG4gICAgICAgICAgICAgICAgeyBcIkZuOjpHZXRBdHRcIjogW1wibXl0YWJsZTAzMjRENDVDXCIsIFwiQXJuXCJdIH0sXG4gICAgICAgICAgICAgICAgeyBcIkZuOjpKb2luXCI6IFsgXCJcIiwgWyB7IFwiRm46OkdldEF0dFwiOiBbIFwibXl0YWJsZTAzMjRENDVDXCIsIFwiQXJuXCIgXSB9LCBcIi9pbmRleC8qXCIgXSBdIH1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgXCJWZXJzaW9uXCI6IFwiMjAxMi0xMC0xN1wiXG4gICAgICAgIH0sXG4gICAgICAgIFwiVXNlcnNcIjogWyB7IFwiUmVmXCI6IFwidXNlcjJDMkI1N0FFXCIgfSBdXG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH0sXG5cbiAgJ2ltcG9ydCc6IHtcbiAgICAncmVwb3J0IGVycm9yIHdoZW4gaW1wb3J0aW5nIGFuIGV4dGVybmFsL2V4aXN0aW5nIHRhYmxlIGZyb20gaW52YWxpZCBhcm4gbWlzc2luZyByZXNvdXJjZSBuYW1lJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgICBjb25zdCB0YWJsZUFybiA9ICdhcm46YXdzOmR5bmFtb2RiOnVzLWVhc3QtMTo6dGFibGUvJztcbiAgICAgIC8vIFdIRU5cbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IFRhYmxlLmZyb21UYWJsZUFybihzdGFjaywgJ0ltcG9ydGVkVGFibGUnLCB0YWJsZUFybiksIC9BUk4gZm9yIER5bmFtb0RCIHRhYmxlIG11c3QgYmUgaW4gdGhlIGZvcm06IC4uLi8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdzdGF0aWMgaW1wb3J0KHJlZikgYWxsb3dzIGltcG9ydGluZyBhbiBleHRlcm5hbC9leGlzdGluZyB0YWJsZSBmcm9tIGFybicodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgY29uc3QgdGFibGVBcm4gPSAnYXJuOmF3czpkeW5hbW9kYjp1cy1lYXN0LTE6MTExMTExMTE6dGFibGUvTXlUYWJsZSc7XG4gICAgICBjb25zdCB0YWJsZSA9IFRhYmxlLmZyb21UYWJsZUFybihzdGFjaywgJ0ltcG9ydGVkVGFibGUnLCB0YWJsZUFybik7XG5cbiAgICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdOZXdSb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWNzLXRhc2tzLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIH0pO1xuICAgICAgdGFibGUuZ3JhbnRSZWFkRGF0YShyb2xlKTtcblxuICAgICAgLy8gaXQgaXMgcG9zc2libGUgdG8gb2J0YWluIGEgcGVybWlzc2lvbiBzdGF0ZW1lbnQgZm9yIGEgcmVmXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBbXG4gICAgICAgICAgICAgICAgJ2R5bmFtb2RiOkJhdGNoR2V0SXRlbScsXG4gICAgICAgICAgICAgICAgJ2R5bmFtb2RiOkdldFJlY29yZHMnLFxuICAgICAgICAgICAgICAgICdkeW5hbW9kYjpHZXRTaGFyZEl0ZXJhdG9yJyxcbiAgICAgICAgICAgICAgICAnZHluYW1vZGI6UXVlcnknLFxuICAgICAgICAgICAgICAgICdkeW5hbW9kYjpHZXRJdGVtJyxcbiAgICAgICAgICAgICAgICAnZHluYW1vZGI6U2NhbidcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBcIlJlc291cmNlXCI6IFtcbiAgICAgICAgICAgICAgICB0YWJsZUFybixcbiAgICAgICAgICAgICAgICB7IFwiUmVmXCI6IFwiQVdTOjpOb1ZhbHVlXCIgfVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgfSxcbiAgICAgICAgXCJQb2xpY3lOYW1lXCI6ICdOZXdSb2xlRGVmYXVsdFBvbGljeTkwRThGNDlEJyxcbiAgICAgICAgXCJSb2xlc1wiOiBbIHsgXCJSZWZcIjogJ05ld1JvbGU5OTc2MzA3NScgfSBdXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHRhYmxlLnRhYmxlQXJuLCB0YWJsZUFybik7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHRhYmxlLnRhYmxlTmFtZSksICdNeVRhYmxlJyk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdzdGF0aWMgaW1wb3J0KHJlZikgYWxsb3dzIGltcG9ydGluZyBhbiBleHRlcm5hbC9leGlzdGluZyB0YWJsZSBmcm9tIHRhYmxlIG5hbWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IHRhYmxlTmFtZSA9ICdNeVRhYmxlJztcbiAgICAgIGNvbnN0IHRhYmxlID0gVGFibGUuZnJvbVRhYmxlTmFtZShzdGFjaywgJ0ltcG9ydGVkVGFibGUnLCB0YWJsZU5hbWUpO1xuXG4gICAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnTmV3Um9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2Vjcy10YXNrcy5hbWF6b25hd3MuY29tJyksXG4gICAgICB9KTtcbiAgICAgIHRhYmxlLmdyYW50UmVhZFdyaXRlRGF0YShyb2xlKTtcblxuICAgICAgLy8gaXQgaXMgcG9zc2libGUgdG8gb2J0YWluIGEgcGVybWlzc2lvbiBzdGF0ZW1lbnQgZm9yIGEgcmVmXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBbXG4gICAgICAgICAgICAgICAgJ2R5bmFtb2RiOkJhdGNoR2V0SXRlbScsXG4gICAgICAgICAgICAgICAgJ2R5bmFtb2RiOkdldFJlY29yZHMnLFxuICAgICAgICAgICAgICAgICdkeW5hbW9kYjpHZXRTaGFyZEl0ZXJhdG9yJyxcbiAgICAgICAgICAgICAgICAnZHluYW1vZGI6UXVlcnknLFxuICAgICAgICAgICAgICAgICdkeW5hbW9kYjpHZXRJdGVtJyxcbiAgICAgICAgICAgICAgICAnZHluYW1vZGI6U2NhbicsXG4gICAgICAgICAgICAgICAgJ2R5bmFtb2RiOkJhdGNoV3JpdGVJdGVtJyxcbiAgICAgICAgICAgICAgICAnZHluYW1vZGI6UHV0SXRlbScsXG4gICAgICAgICAgICAgICAgJ2R5bmFtb2RiOlVwZGF0ZUl0ZW0nLFxuICAgICAgICAgICAgICAgICdkeW5hbW9kYjpEZWxldGVJdGVtJ1xuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFwiUmVzb3VyY2VcIjogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgXCJSZWZcIjogXCJBV1M6OlBhcnRpdGlvblwiXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICBcIjpkeW5hbW9kYjpcIixcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBcIlJlZlwiOiBcIkFXUzo6UmVnaW9uXCJcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAgIFwiOlwiLFxuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpBY2NvdW50SWRcIlxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICAgXCI6dGFibGUvTXlUYWJsZVwiXG4gICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpOb1ZhbHVlXCJcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICB9LFxuICAgICAgICBcIlBvbGljeU5hbWVcIjogJ05ld1JvbGVEZWZhdWx0UG9saWN5OTBFOEY0OUQnLFxuICAgICAgICBcIlJvbGVzXCI6IFsgeyBcIlJlZlwiOiAnTmV3Um9sZTk5NzYzMDc1JyB9IF1cbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kZWVwRXF1YWwodGFibGUudGFibGVBcm4sICdhcm46JHtUb2tlbltBV1M6OlBhcnRpdGlvbi4zXX06ZHluYW1vZGI6JHtUb2tlbltBV1M6OlJlZ2lvbi40XX06JHtUb2tlbltBV1M6OkFjY291bnRJZC4wXX06dGFibGUvTXlUYWJsZScpO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh0YWJsZS50YWJsZU5hbWUpLCB0YWJsZU5hbWUpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcblxuICAnZ2xvYmFsJzoge1xuICAgICdjcmVhdGUgcmVwbGljYXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgVGFibGUoc3RhY2ssICdUYWJsZScsIHtcbiAgICAgICAgcGFydGl0aW9uS2V5OiB7XG4gICAgICAgICAgbmFtZTogJ2lkJyxcbiAgICAgICAgICB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklOR1xuICAgICAgICB9LFxuICAgICAgICByZXBsaWNhdGlvblJlZ2lvbnM6IFtcbiAgICAgICAgICAnZXUtd2VzdC0yJyxcbiAgICAgICAgICAnZXUtY2VudHJhbC0xJ1xuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdDdXN0b206OkR5bmFtb0RCUmVwbGljYScsIHtcbiAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgIFNlcnZpY2VUb2tlbjoge1xuICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICdhd3NjZGthd3NkeW5hbW9kYlJlcGxpY2FQcm92aWRlck5lc3RlZFN0YWNrYXdzY2RrYXdzZHluYW1vZGJSZXBsaWNhUHJvdmlkZXJOZXN0ZWRTdGFja1Jlc291cmNlMThFM0YxMkQnLFxuICAgICAgICAgICAgICAnT3V0cHV0cy5hd3NjZGthd3NkeW5hbW9kYlJlcGxpY2FQcm92aWRlcmZyYW1ld29ya29uRXZlbnRGOTUwNDY5MUFybidcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFRhYmxlTmFtZToge1xuICAgICAgICAgICAgUmVmOiAnVGFibGVDRDExN0ZBMSdcbiAgICAgICAgICB9LFxuICAgICAgICAgIFJlZ2lvbjogJ2V1LXdlc3QtMidcbiAgICAgICAgfSxcbiAgICAgICAgQ29uZGl0aW9uOiAnVGFibGVTdGFja1JlZ2lvbk5vdEVxdWFsc2V1d2VzdDJBMDM4NTlFNydcbiAgICAgIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0N1c3RvbTo6RHluYW1vREJSZXBsaWNhJywge1xuICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgU2VydmljZVRva2VuOiB7XG4gICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgJ2F3c2Nka2F3c2R5bmFtb2RiUmVwbGljYVByb3ZpZGVyTmVzdGVkU3RhY2thd3NjZGthd3NkeW5hbW9kYlJlcGxpY2FQcm92aWRlck5lc3RlZFN0YWNrUmVzb3VyY2UxOEUzRjEyRCcsXG4gICAgICAgICAgICAgICdPdXRwdXRzLmF3c2Nka2F3c2R5bmFtb2RiUmVwbGljYVByb3ZpZGVyZnJhbWV3b3Jrb25FdmVudEY5NTA0NjkxQXJuJ1xuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgVGFibGVOYW1lOiB7XG4gICAgICAgICAgICBSZWY6ICdUYWJsZUNEMTE3RkExJ1xuICAgICAgICAgIH0sXG4gICAgICAgICAgUmVnaW9uOiAnZXUtY2VudHJhbC0xJ1xuICAgICAgICB9LFxuICAgICAgICBDb25kaXRpb246ICdUYWJsZVN0YWNrUmVnaW9uTm90RXF1YWxzZXVjZW50cmFsMTk5RDQ2RkMwJ1xuICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuXG4gICAgICB0ZXN0LmRlZXBFcXVhbChTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24oc3RhY2spLkNvbmRpdGlvbnMsIHtcbiAgICAgICAgVGFibGVTdGFja1JlZ2lvbk5vdEVxdWFsc2V1d2VzdDJBMDM4NTlFNzoge1xuICAgICAgICAgICdGbjo6Tm90JzogW1xuICAgICAgICAgICAgeyAnRm46OkVxdWFscyc6IFsgJ2V1LXdlc3QtMicsIHsgUmVmOiAnQVdTOjpSZWdpb24nIH0gXSB9XG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBUYWJsZVN0YWNrUmVnaW9uTm90RXF1YWxzZXVjZW50cmFsMTk5RDQ2RkMwOiB7XG4gICAgICAgICAgJ0ZuOjpOb3QnOiBbXG4gICAgICAgICAgICB7ICdGbjo6RXF1YWxzJzogWyAnZXUtY2VudHJhbC0xJywgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSBdIH1cbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3Rocm93cyB3aXRoIFBST1ZJU0lPTkVEIGJpbGxpbmcgbW9kZScodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBUYWJsZShzdGFjaywgJ1RhYmxlJywge1xuICAgICAgICBwYXJ0aXRpb25LZXk6IHtcbiAgICAgICAgICBuYW1lOiAnaWQnLFxuICAgICAgICAgIHR5cGU6IEF0dHJpYnV0ZVR5cGUuU1RSSU5HXG4gICAgICAgIH0sXG4gICAgICAgIHJlcGxpY2F0aW9uUmVnaW9uczogW1xuICAgICAgICAgICdldS13ZXN0LTInLFxuICAgICAgICAgICdldS1jZW50cmFsLTEnXG4gICAgICAgIF0sXG4gICAgICAgIGJpbGxpbmdNb2RlOiBCaWxsaW5nTW9kZS5QUk9WSVNJT05FRCxcbiAgICAgIH0pLCAvYFBBWV9QRVJfUkVRVUVTVGAvKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICd0aHJvd3Mgd2hlbiBzdHJlYW0gaXMgc2V0IGFuZCBub3Qgc2V0IHRvIE5FV19BTkRfT0xEX0lNQUdFUycodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBUYWJsZShzdGFjaywgJ1RhYmxlJywge1xuICAgICAgICBwYXJ0aXRpb25LZXk6IHtcbiAgICAgICAgICBuYW1lOiAnaWQnLFxuICAgICAgICAgIHR5cGU6IEF0dHJpYnV0ZVR5cGUuU1RSSU5HXG4gICAgICAgIH0sXG4gICAgICAgIHJlcGxpY2F0aW9uUmVnaW9uczogW1xuICAgICAgICAgICdldS13ZXN0LTInLFxuICAgICAgICAgICdldS1jZW50cmFsLTEnXG4gICAgICAgIF0sXG4gICAgICAgIHN0cmVhbTogU3RyZWFtVmlld1R5cGUuT0xEX0lNQUdFLFxuICAgICAgfSksIC9gTkVXX0FORF9PTERfSU1BR0VTYC8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3Rocm93cyB3aXRoIHJlcGxpY2EgaW4gc2FtZSByZWdpb24gYXMgc3RhY2snKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdTdGFjaycsIHtcbiAgICAgICAgZW52OiB7IHJlZ2lvbjogJ3VzLWVhc3QtMScgfVxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBUYWJsZShzdGFjaywgJ1RhYmxlJywge1xuICAgICAgICBwYXJ0aXRpb25LZXk6IHtcbiAgICAgICAgICBuYW1lOiAnaWQnLFxuICAgICAgICAgIHR5cGU6IEF0dHJpYnV0ZVR5cGUuU1RSSU5HXG4gICAgICAgIH0sXG4gICAgICAgIHJlcGxpY2F0aW9uUmVnaW9uczogW1xuICAgICAgICAgICdldS13ZXN0LTEnLFxuICAgICAgICAgICd1cy1lYXN0LTEnLFxuICAgICAgICAgICdldS13ZXN0LTInLFxuICAgICAgICBdLFxuICAgICAgfSksIC9gcmVwbGljYXRpb25SZWdpb25zYCBjYW5ub3QgaW5jbHVkZSB0aGUgcmVnaW9uIHdoZXJlIHRoaXMgc3RhY2sgaXMgZGVwbG95ZWQvKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdubyBjb25kaXRpb25zIHdoZW4gcmVnaW9uIGlzIGtub3duJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2snLCB7XG4gICAgICAgIGVudjogeyByZWdpb246ICdldS13ZXN0LTEnIH1cbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgVGFibGUoc3RhY2ssICdUYWJsZScsIHtcbiAgICAgICAgcGFydGl0aW9uS2V5OiB7XG4gICAgICAgICAgbmFtZTogJ2lkJyxcbiAgICAgICAgICB0eXBlOiBBdHRyaWJ1dGVUeXBlLlNUUklOR1xuICAgICAgICB9LFxuICAgICAgICByZXBsaWNhdGlvblJlZ2lvbnM6IFtcbiAgICAgICAgICAnZXUtd2VzdC0yJyxcbiAgICAgICAgICAnZXUtY2VudHJhbC0xJ1xuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZXF1YWwoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKS5Db25kaXRpb25zLCB1bmRlZmluZWQpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9XG59O1xuXG5mdW5jdGlvbiB0ZXN0R3JhbnQodGVzdDogVGVzdCwgZXhwZWN0ZWRBY3Rpb25zOiBzdHJpbmdbXSwgaW52b2NhdGlvbjogKHVzZXI6IGlhbS5JUHJpbmNpcGFsLCB0YWJsZTogVGFibGUpID0+IHZvaWQpIHtcbiAgLy8gR0lWRU5cbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgY29uc3QgdGFibGUgPSBuZXcgVGFibGUoc3RhY2ssICdteS10YWJsZScsIHsgcGFydGl0aW9uS2V5OiB7IG5hbWU6ICdJRCcsIHR5cGU6ICBBdHRyaWJ1dGVUeXBlLlNUUklORyB9IH0pO1xuICBjb25zdCB1c2VyID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAndXNlcicpO1xuXG4gIC8vIFdIRU5cbiAgaW52b2NhdGlvbih1c2VyLCB0YWJsZSk7XG5cbiAgLy8gVEhFTlxuICBjb25zdCBhY3Rpb24gPSBleHBlY3RlZEFjdGlvbnMubGVuZ3RoID4gMSA/IGV4cGVjdGVkQWN0aW9ucy5tYXAoYSA9PiBgZHluYW1vZGI6JHthfWApIDogYGR5bmFtb2RiOiR7ZXhwZWN0ZWRBY3Rpb25zWzBdfWA7XG4gIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICB7XG4gICAgICAgICAgXCJBY3Rpb25cIjogYWN0aW9uLFxuICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICBcIlJlc291cmNlXCI6IFtcbiAgICAgICAgICAgIHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJteXRhYmxlMDMyNEQ0NUNcIiwgXCJBcm5cIiBdIH0sXG4gICAgICAgICAgICB7IFwiUmVmXCIgOiBcIkFXUzo6Tm9WYWx1ZVwiIH1cbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICB9LFxuICAgIFwiVXNlcnNcIjogWyB7IFwiUmVmXCI6IFwidXNlcjJDMkI1N0FFXCIgfSBdXG4gIH0pKTtcbiAgdGVzdC5kb25lKCk7XG59XG4iXX0=