"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisStreamsToLambda = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-solutions-constructs/core");
const core_2 = require("@aws-cdk/core");
/**
 * @summary The KinesisStreamsToLambda class.
 */
class KinesisStreamsToLambda extends core_2.Construct {
    /**
     * @summary Constructs a new instance of the KinesisStreamsToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Kinesis Stream
        this.kinesisStream = defaults.buildKinesisStream(this, {
            kinesisStreamProps: props.kinesisStreamProps
        });
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        // Add the Lambda event source mapping
        const eventSourceProps = (props.eventSourceProps) ?
            core_1.overrideProps(defaults.DefaultKinesisEventSourceProps(this.kinesisStream.streamArn), props.eventSourceProps) :
            defaults.DefaultKinesisEventSourceProps(this.kinesisStream.streamArn);
        this.lambdaFunction.addEventSourceMapping('LambdaKinesisEventSourceMapping', eventSourceProps);
        // Add permissions for the Lambda function to access Kinesis
        const policy = new iam.Policy(this, 'LambdaFunctionPolicy');
        this.kinesisStreamRole = this.lambdaFunction.role;
        policy.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [this.kinesisStream.streamArn],
            actions: [
                'kinesis:GetRecords',
                'kinesis:GetShardIterator',
                'kinesis:DescribeStream'
            ]
        }));
        policy.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'kinesis:ListStreams',
            ]
        }));
        policy.attachToRole(this.kinesisStreamRole);
        this.kinesisStream.grantRead(this.lambdaFunction.grantPrincipal);
        // Add appropriate cfn_nag metadata
        const cfnCustomPolicy = policy.node.defaultChild;
        cfnCustomPolicy.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W12",
                        reason: "The kinesis:ListStreams action requires a wildcard resource."
                    }
                ]
            }
        };
    }
}
exports.KinesisStreamsToLambda = KinesisStreamsToLambda;
//# sourceMappingURL=data:application/json;base64,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