from collections import defaultdict

import h5py
import numpy as np
import xarray as xr

from .backend.db import DamnitDB, DB_NAME
from .backend.extract_data import add_to_db, ReducedData
from .ctxsupport.ctxrunner import generate_thumbnail, add_to_h5_file, DataType


def migrate_images(new_db, db_dir, dry_run):
    """
    Image thumbnails were previously generated by the GUI, now they're generated
    by the backend. This function will convert old image summaries (2D arrays) into
    thumbnails (RGBA arrays) that can be used directly by the GUI.

    Be careful to pass in the *new database* handle instead of the old one,
    since this will modify the DB even when dry_run=False.
    """
    proposal = new_db.metameta.get("proposal")
    if proposal is None:
        raise RuntimeError("Database must have a proposal configured for it to be migrated.")

    reduced_data = defaultdict(dict)
    files = list((db_dir / "extracted_data").glob("*.h5"))
    n_files = len(files)
    files_modified = set()
    suffix = "s" if n_files > 1 or n_files == 0 else ""

    print(f"Looking through {n_files} HDF5 file{suffix}...")

    # First we modify all of the image summaries in the HDF5 files
    for h5_path in files:
        with add_to_h5_file(h5_path) as f:
            if ".reduced" in f:
                reduced = f[".reduced"]
                run = int(h5_path.stem.split("_")[1][1:])

                for ds_name, dset in reduced.items():
                    if dset.ndim == 2 or (dset.ndim == 3 and dset.shape[2] == 4):
                        # Generate a new thumbnail
                        image = reduced[ds_name][()]
                        image = generate_thumbnail(image)
                        reduced_data[run][ds_name] = ReducedData(image.data)
                        files_modified.add(h5_path)

                        if not dry_run:
                            # Overwrite the dataset
                            del reduced[ds_name]
                            dset = reduced.create_dataset(
                                ds_name, data=np.frombuffer(image.data, dtype=np.uint8)
                            )

    # And then update the summaries in the database
    for run, run_reduced_data in reduced_data.items():
        add_to_db(run_reduced_data, new_db, proposal, run)

    info = f"updated {len(reduced_data)} variables in {len(files_modified)} files"
    if dry_run:
        print(f"Dry run: would have {info}.")
    else:
        print(f"Migration completed successfully, {info}")

def dataarray_from_group(group):
    data = group["data"][()]
    coords = { ds_name: group[ds_name][()] for ds_name in group.keys()
               if ds_name != "data" }
    scalar_coords = {name: co for (name, co) in coords.items() if np.ndim(co) == 0}

    # Attempt to map the coords to the right dimensions. This
    # will fail if there are two coordinates/dimensions with the
    # same length.
    length_to_dim = {len(co): name for name, co in coords.items()
                     if name not in scalar_coords}
    if len(set(length_to_dim.keys())) != len(length_to_dim):
        return None

    dims = [length_to_dim[l] if l in length_to_dim else f"dim_{i}"
            for i, l in enumerate(data.shape)]
    return xr.DataArray(data,
                        dims=dims,
                        coords={ dim: coords[dim] for dim in dims
                                 if dim in coords } | scalar_coords)


def migrate_dataarrays(db, db_dir, dry_run):
    files = list((db_dir / "extracted_data").glob("*.h5"))
    n_files = len(files)
    suffix = "s" if n_files > 1 or n_files == 0 else ""

    would = "(would) " if dry_run else ""

    print(f"Looking through {n_files} HDF5 file{suffix}...")

    files_modified = set()
    total_groups = 0
    for h5_path in files:
        groups_to_replace = []
        with h5py.File(h5_path, "a") as f:
            for name, grp in f.items():
                if name == '.reduced':
                    continue

                if isinstance(grp, h5py.Group) and 'data' in grp and len(grp) > 1:
                    dataarray = dataarray_from_group(grp)
                    if dataarray is None:
                        raise RuntimeError(
                            f"Error: could not convert v0 array for '{name}' to a DataArray automatically"
                        )
                    groups_to_replace.append((name, dataarray))

            for name, _ in groups_to_replace:
                print(f"{would}Delete {name} in {h5_path.relative_to(db_dir)}")
                if not dry_run:
                    f[name].clear()
                    f[name].attrs.clear()
                    f[name].attrs['_damnit_objtype'] = DataType.DataArray.value

        for name, arr in groups_to_replace:
            print(f"{would}Save {name} in {h5_path.relative_to(db_dir)}")
            if not dry_run:
                arr.to_netcdf(h5_path, mode="a", format="NETCDF4", group=name, engine="h5netcdf")

            files_modified.add(h5_path)
            total_groups += 1

    print(("(would have) " if dry_run else "") +
          f"Modified {total_groups} groups in {len(files_modified)} files")
    if dry_run:
        print("Dry run - no files were changed")


def main_dataset(grp: h5py.Group):
    """Get the main dataset from a group representing an xarray or numpy array.

    For an xarray DataArray, this is the one that's not coordinate labels.
    We're assuming these groups have already been converted to NetCDF format.
    """
    candidates = {name for (name, dset) in grp.items()
                  if dset.attrs.get('CLASS', b'') != b'DIMENSION_SCALE'}
    if len(candidates) == 1:
        return grp[candidates.pop()]


def copy_table(table, old_db, new_db):
    """Copy an entire table from one database to another."""
    rows = old_db.conn.execute(f"SELECT * FROM {table}").fetchall()
    if len(rows) == 0:
        return

    placeholder = ", ".join(["?" for _ in rows[0]])
    new_db.conn.executemany(f"""
    INSERT INTO {table}
    VALUES ({placeholder})
    """, rows)

def migrate_v0_to_v1(db, db_dir, dry_run):
    """
    For reference, see the V0_SCHEMA variable in db.py.
    In the v1 schema, the runs table was deleted and replaced with a view of
    the new run_variables table. The run_info table also needs to be created,
    but that can be done by executing the v1 schema.
    """
    migrate_dataarrays(db, db_dir, dry_run)

    # Get all column and variable names
    column_names = [rec[0] for rec in
                    db.conn.execute("SELECT name FROM PRAGMA_TABLE_INFO('runs')").fetchall()]
    variable_names = [name for name in column_names
                      if name not in ["proposal", "runnr", "start_time", "added_at"]]

    # And then read all run data. This is what we'll need to copy into the new
    # `run_variables` table.
    runs = [rec for rec in db.conn.execute("SELECT * FROM runs").fetchall()]

    print(f"Found {len(runs)} runs, with these variables:")
    print(variable_names)
    print()

    # Scan HDF5 files to get timestamps (from mtime) & max diff for 1D arrays
    timestamps = {}  # keys (proposal, run)
    max_diffs = {}  # keys (proposal, run, variable)
    for record in runs:
        proposal = record["proposal"]
        run_no = record["runnr"]
        h5_path = db_dir / f"extracted_data/p{proposal}_r{run_no}.h5"

        if not h5_path.exists():
            print(f"Skipping variables for run {run_no} because {h5_path} does not exist")
            continue

        timestamps[(proposal, run_no)] = h5_path.stat().st_mtime

        with h5py.File(h5_path, "a") as f:
            for name in variable_names:
                if name not in f:
                    continue

                if (ds := main_dataset(f[name])) is None:
                    print(f"Couldn't identify main dataset for {name} in "
                          f"{h5_path}, skipping max_diff")
                    continue

                if 'max_diff' in ds.attrs:
                    max_diffs[(proposal, run_no, name)] = ds.attrs['max_diff'].item()
                elif ds.ndim == 1 and ds.size > 0 and np.issubdtype(ds.dtype, np.number):
                    data = ds[()]
                    max_diff = abs(np.nanmax(data) - np.nanmin(data)).item()
                    max_diffs[(proposal, run_no, name)] = max_diff
                    ds.attrs['max_diff'] = max_diff

    print(f"Found max difference for {len(max_diffs)} variables")

    new_db_path = db_dir / "runs.v1.sqlite"
    new_db_path.unlink(missing_ok=True)  # Clear any previous attempt
    new_db = DamnitDB(new_db_path)

    # Copy the metadata
    for k, v in db.metameta.items():
        if k != "data_format_version":
            new_db.metameta[k] = v

    # Copy the user-editable variables and standalone comments
    for table in ["variables", "time_comments"]:
        copy_table(table, db, new_db)

    # Load the data into the new database
    total_vars = 0
    with new_db.conn:
        for record in runs:
            run = dict(zip(column_names, record))
            proposal = run["proposal"]
            run_no = run["runnr"]

            # Add the run info to the `run_info` table
            new_db.conn.execute("""
                INSERT INTO run_info
                VALUES (:proposal, :runnr, :start_time, :added_at)
                """, run)

            for name in variable_names:
                value = record[name]
                if value is None:
                    continue

                max_diff = max_diffs.get((proposal, run_no, name))
                timestamp = timestamps.get((proposal, run_no))

                variable = {
                    "proposal": proposal,
                    "run": run_no,
                    "name": name,
                    "version": 1,
                    "value": value,
                    "timestamp": timestamp,
                    "max_diff": max_diff
                }
                new_db.conn.execute("""
                    INSERT INTO run_variables (proposal, run, name, version, value, timestamp, max_diff)
                    VALUES (:proposal, :run, :name, :version, :value, :timestamp, :max_diff)
                    """, variable)
                total_vars += 1

        # And now that we're done, we need to recreate the `runs` view
        new_db.update_views()

    # Last step: migrate the images from the old format to the new format
    migrate_images(new_db, db_dir, dry_run)

    new_db.close()
    db.close()

    if dry_run:
        print(f"Dry-run: new format DB created at {new_db_path.name}")
        print("If all seems OK, re-run the migration without --dry-run.")
    else:
        db_path = db_dir / DB_NAME
        backup_path = db_dir / "runs.v0-backup.sqlite"
        db_path.rename(backup_path)
        new_db_path.rename(db_path)
        print(f"New format DB created and moved to {db_path.name}")
        print(f"Old database backed up as {backup_path.name}")

def migrate_intermediate_v1(db, db_dir, dry_run):
    """Migrate intermediate v1 (v0.5) databases.

    Before v1 rose over the world, resplendent and glorious, there was a humble
    antecedent that was used for some proposals:
    - p3338 (FXE)
    - p6616 (FXE)
    - p4507 (FXE)
    - p5639 (SXP)
    - p4656 (MID)
    - p3118 (MID)
    - p6976 (MID)
    - p4559 (MID)
    - p5397 (MID)
    - p4239 (MID)
    - p4442 (MID)
    - p2956 (SCS)

    To push these databases into their destiny of v1 we must make some changes:
    - Remove the `run_variables.stored_type` column
    - Re-do image migration to convert the thumbnails to PNGs
    - Move the `stored_type` attribute on `.reduced/<var>` datasets to a
      `_damnit_objtype` attribute on the `<var>` group.
    """
    # Create a new database, overwriting any previous attempts
    new_db_path = db_dir / "runs.v1.sqlite"
    new_db_path.unlink(missing_ok=True)
    new_db = DamnitDB(new_db_path)

    # Copy everything but `run_variables` to the new database
    for k, v in db.metameta.items():
        if k != "data_format_version":
            new_db.metameta[k] = v

    for table in ["run_info", "time_comments", "variables"]:
        copy_table(table, db, new_db)

    # Note that we skip the `stored_type` column because that was removed
    run_variables = db.conn.execute("""
        SELECT proposal, run, name, version, value, timestamp, max_diff, provenance
        FROM run_variables
    """).fetchall()
    for row in run_variables:
        new_db.conn.execute("""
        INSERT INTO run_variables (proposal, run, name, version, value, timestamp, max_diff, provenance)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        """, row)

    new_db.update_views()

    # Convert the old `stored_type` attribute into `_damnit_objtype`
    runs = db.conn.execute("SELECT proposal, run FROM runs").fetchall()
    for proposal, run_no in runs:
        h5_path = db_dir / "extracted_data" / f"p{proposal}_r{run_no}.h5"
        if not h5_path.is_file():
            continue

        with add_to_h5_file(h5_path) as f:
            reduced = f[".reduced"]
            for ds_name, dset in reduced.items():
                if "stored_type" in dset.attrs:
                    stored_type = dset.attrs["stored_type"]

                    obj_type = None
                    if stored_type in ["DataArray", "Dataset", "image", "timestamp"]:
                        obj_type = stored_type.lower()

                    if not dry_run:
                        if obj_type is not None:
                            f[ds_name].attrs["_damnit_objtype"] = obj_type
                        del dset.attrs["stored_type"]

    # Migrate images to use PNGs for thumbnails
    migrate_images(new_db, db_dir, dry_run)

    new_db.close()
    db.close()

    if dry_run:
        print(f"Dry-run: new format DB created at {new_db_path.name}")
        print("If all seems OK, re-run the migration without --dry-run.")
    else:
        db_path = db_dir / DB_NAME
        backup_path = db_dir / "runs.intermediate-v1-backup.sqlite"
        db_path.rename(backup_path)
        new_db_path.rename(db_path)
        print(f"New format DB created and moved to {db_path.name}")
        print(f"Old database backed up as {backup_path.name}")
