"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Database = exports.WordPress = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const efs = require("@aws-cdk/aws-efs");
const logs = require("@aws-cdk/aws-logs");
const rds = require("@aws-cdk/aws-rds");
const cdk = require("@aws-cdk/core");
const patterns = require("./index");
class WordPress extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : getOrCreateVpc(this);
        this.db = this.addDatabase({
            vpc: this.vpc,
            databaseSubnets: props.databaseSubnets,
            instanceType: props.databaseInstanceType,
            instanceEngine: props.instanceEngine,
            clusterEngine: props.clusterEngine,
            auroraServerless: props.auroraServerless,
            singleDbInstance: props.singleDbInstance,
            backupRetention: props.backupRetention,
        });
        const logGroup = new logs.LogGroup(this, 'LogGroup', {
            retention: logs.RetentionDays.ONE_MONTH,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const task = new ecs.FargateTaskDefinition(this, 'Task', {
            cpu: 256,
            memoryLimitMiB: 512,
        });
        task.addContainer('wordpress', {
            image: ecs.ContainerImage.fromRegistry('wordpress:latest'),
            portMappings: [{ containerPort: 80 }],
            environment: {
                WORDPRESS_DB_NAME: 'wordpress',
            },
            logging: new ecs.AwsLogDriver({
                streamPrefix: 'wordpress-fargate',
                logGroup,
            }),
            secrets: {
                WORDPRESS_DB_HOST: ecs.Secret.fromSecretsManager(this.db.secret, 'host'),
                WORDPRESS_DB_USER: ecs.Secret.fromSecretsManager(this.db.secret, 'username'),
                WORDPRESS_DB_PASSWORD: ecs.Secret.fromSecretsManager(this.db.secret, 'password'),
            },
        });
        const healthCheck = {
            path: '/wp-includes/images/blank.gif',
            interval: cdk.Duration.minutes(1),
        };
        this.svc = new patterns.DualAlbFargateService(this, 'ALBFargateService', {
            vpc: this.vpc,
            spot: props.spot,
            enableExecuteCommand: props.enableExecuteCommand,
            tasks: props.serviceProps ? [props.serviceProps] : [
                {
                    external: { port: 80 },
                    task,
                    healthCheck,
                },
            ],
            route53Ops: {
                enableLoadBalancerAlias: false,
            },
        });
        // EFS volume
        const filesystem = new efs.FileSystem(this, 'FileSystem', {
            vpc: this.vpc,
            encrypted: true,
        });
        const volumeName = 'efs';
        this.svc.service[0].taskDefinition.addVolume({
            name: volumeName,
            efsVolumeConfiguration: {
                fileSystemId: filesystem.fileSystemId,
            },
        });
        (_b = this.svc.service[0].taskDefinition.defaultContainer) === null || _b === void 0 ? void 0 : _b.addMountPoints({
            containerPath: '/var/www/html',
            readOnly: false,
            sourceVolume: volumeName,
        });
        filesystem.connections.allowFrom(new ec2.Connections({ securityGroups: this.svc.service[0].connections.securityGroups }), ec2.Port.tcp(2049), 'allow wordpress to connect efs');
        this.db.connections.allowFrom(this.svc.service[0], this.db.connections.defaultPort, `allow ${this.svc.service[0].serviceName} to connect db`);
    }
    addDatabase(props) {
        return new Database(this, 'Database', {
            ...props,
        });
    }
}
exports.WordPress = WordPress;
/**
 * Represents the database instance or database cluster
 */
class Database extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this._mysqlListenerPort = 3306;
        this.vpc = props.vpc;
        const config = props.auroraServerless ? this._createServerlessCluster(props)
            : props.singleDbInstance ? this._createRdsInstance(props) : this._createRdsCluster(props);
        this.secret = config.secret;
        // allow internally from the same security group
        config.connections.allowInternally(ec2.Port.tcp(this._mysqlListenerPort));
        // allow from the fargate service or the whole vpc cidr
        config.connections.allowFrom((_a = props.allowFrom) !== null && _a !== void 0 ? _a : ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(this._mysqlListenerPort));
        this.clusterEndpointHostname = config.endpoint;
        this.clusterIdentifier = config.identifier;
        this.connections = config.connections;
        printOutput(this, 'DBSecretArn', config.secret.secretArn);
        printOutput(this, 'clusterEndpointHostname', this.clusterEndpointHostname);
        printOutput(this, 'clusterIdentifier', this.clusterIdentifier);
    }
    _createRdsInstance(props) {
        var _a, _b, _c;
        const dbInstance = new rds.DatabaseInstance(this, 'DBInstance', {
            vpc: props.vpc,
            vpcSubnets: props.databaseSubnets,
            engine: (_a = props.instanceEngine) !== null && _a !== void 0 ? _a : rds.DatabaseInstanceEngine.mysql({
                version: rds.MysqlEngineVersion.VER_8_0_23,
            }),
            storageEncrypted: true,
            backupRetention: (_b = props.backupRetention) !== null && _b !== void 0 ? _b : cdk.Duration.days(7),
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            instanceType: (_c = props.instanceType) !== null && _c !== void 0 ? _c : new ec2.InstanceType('r5.large'),
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.mysql8.0'),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        return {
            connections: dbInstance.connections,
            endpoint: dbInstance.dbInstanceEndpointAddress,
            identifier: dbInstance.instanceIdentifier,
            secret: dbInstance.secret,
        };
    }
    // create a RDS for MySQL DB cluster
    _createRdsCluster(props) {
        var _a, _b, _c;
        const dbCluster = new rds.DatabaseCluster(this, 'DBCluster', {
            engine: (_a = props.clusterEngine) !== null && _a !== void 0 ? _a : rds.DatabaseClusterEngine.auroraMysql({
                version: rds.AuroraMysqlEngineVersion.VER_2_09_1,
            }),
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            defaultDatabaseName: 'wordpress',
            instanceProps: {
                vpc: props.vpc,
                vpcSubnets: props.databaseSubnets,
                instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : new ec2.InstanceType('r5.large'),
            },
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
            backup: {
                retention: (_c = props.backupRetention) !== null && _c !== void 0 ? _c : cdk.Duration.days(7),
            },
            storageEncrypted: true,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
    _createServerlessCluster(props) {
        var _a;
        const dbCluster = new rds.ServerlessCluster(this, 'AuroraServerlessCluster', {
            engine: rds.DatabaseClusterEngine.AURORA_MYSQL,
            vpc: props.vpc,
            vpcSubnets: props.databaseSubnets,
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            backupRetention: (_a = props.backupRetention) !== null && _a !== void 0 ? _a : cdk.Duration.days(7),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
            defaultDatabaseName: 'wordpress',
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
}
exports.Database = Database;
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
function printOutput(scope, id, key) {
    new cdk.CfnOutput(scope, id, { value: String(key) });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid29yZHByZXNzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL3dvcmRwcmVzcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4QywwQ0FBMEM7QUFDMUMsd0NBQXdDO0FBRXhDLHFDQUFxQztBQUNyQyxvQ0FBb0M7QUFnRXBDLE1BQWEsU0FBVSxTQUFRLEdBQUcsQ0FBQyxTQUFTO0lBSTFDLFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsUUFBd0IsRUFBRTs7UUFDdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsR0FBRyxTQUFHLEtBQUssQ0FBQyxHQUFHLG1DQUFJLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QyxJQUFJLENBQUMsRUFBRSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUM7WUFDekIsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO1lBQ2IsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO1lBQ3RDLFlBQVksRUFBRSxLQUFLLENBQUMsb0JBQW9CO1lBQ3hDLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztZQUNwQyxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7WUFDbEMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtZQUN4QyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZTtTQUN2QyxDQUFDLENBQUM7UUFFSCxNQUFNLFFBQVEsR0FBRyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNuRCxTQUFTLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO1lBQ3ZDLGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU87U0FDekMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRTtZQUN2RCxHQUFHLEVBQUUsR0FBRztZQUNSLGNBQWMsRUFBRSxHQUFHO1NBQ3BCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFO1lBQzdCLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQztZQUMxRCxZQUFZLEVBQUUsQ0FBQyxFQUFFLGFBQWEsRUFBRSxFQUFFLEVBQUUsQ0FBQztZQUNyQyxXQUFXLEVBQUU7Z0JBQ1gsaUJBQWlCLEVBQUUsV0FBVzthQUMvQjtZQUNELE9BQU8sRUFBRSxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUM7Z0JBQzVCLFlBQVksRUFBRSxtQkFBbUI7Z0JBQ2pDLFFBQVE7YUFDVCxDQUFDO1lBQ0YsT0FBTyxFQUFFO2dCQUNQLGlCQUFpQixFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsa0JBQWtCLENBQzlDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUNkLE1BQU0sQ0FDUDtnQkFDRCxpQkFBaUIsRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLGtCQUFrQixDQUM5QyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFDZCxVQUFVLENBQ1g7Z0JBQ0QscUJBQXFCLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsQ0FDbEQsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQ2QsVUFBVSxDQUNYO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLFdBQVcsR0FBRztZQUNsQixJQUFJLEVBQUUsK0JBQStCO1lBQ3JDLFFBQVEsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7U0FDbEMsQ0FBQztRQUVGLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxRQUFRLENBQUMscUJBQXFCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQ3ZFLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNiLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNoQixvQkFBb0IsRUFBRSxLQUFLLENBQUMsb0JBQW9CO1lBQ2hELEtBQUssRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2pEO29CQUNFLFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRSxFQUFFLEVBQUU7b0JBQ3RCLElBQUk7b0JBQ0osV0FBVztpQkFDWjthQUNGO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLHVCQUF1QixFQUFFLEtBQUs7YUFDL0I7U0FDRixDQUFDLENBQUM7UUFFSCxhQUFhO1FBQ2IsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDeEQsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO1lBQ2IsU0FBUyxFQUFFLElBQUk7U0FDaEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUM7WUFDM0MsSUFBSSxFQUFFLFVBQVU7WUFDaEIsc0JBQXNCLEVBQUU7Z0JBQ3RCLFlBQVksRUFBRSxVQUFVLENBQUMsWUFBWTthQUN0QztTQUNGLENBQUMsQ0FBQztRQUNILE1BQUEsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQiwwQ0FBRSxjQUFjLENBQUM7WUFDbEUsYUFBYSxFQUFFLGVBQWU7WUFDOUIsUUFBUSxFQUFFLEtBQUs7WUFDZixZQUFZLEVBQUUsVUFBVTtTQUN6QixFQUFFO1FBRUgsVUFBVSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUUsSUFBSSxHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsY0FBYyxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLGdDQUFnQyxDQUFDLENBQUM7UUFFakwsSUFBSSxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLFdBQVksRUFBRSxTQUFTLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsZ0JBQWdCLENBQUMsQ0FBQztJQUVqSixDQUFDO0lBQ08sV0FBVyxDQUFDLEtBQW9CO1FBQ3RDLE9BQU8sSUFBSSxRQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwQyxHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUF6R0QsOEJBeUdDO0FBOEVEOztHQUVHO0FBQ0gsTUFBYSxRQUFTLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFRekMsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUFvQjs7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUhGLHVCQUFrQixHQUFXLElBQUksQ0FBQztRQUlqRCxJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUM7UUFDckIsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsd0JBQXdCLENBQUMsS0FBSyxDQUFDO1lBQzFFLENBQUMsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzVGLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQztRQUM1QixnREFBZ0Q7UUFDaEQsTUFBTSxDQUFDLFdBQVcsQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztRQUMxRSx1REFBdUQ7UUFDdkQsTUFBTSxDQUFDLFdBQVcsQ0FBQyxTQUFTLE9BQUMsS0FBSyxDQUFDLFNBQVMsbUNBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1FBQzlILElBQUksQ0FBQyx1QkFBdUIsR0FBRyxNQUFNLENBQUMsUUFBUSxDQUFDO1FBQy9DLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDO1FBQzNDLElBQUksQ0FBQyxXQUFXLEdBQUcsTUFBTSxDQUFDLFdBQVcsQ0FBQztRQUN0QyxXQUFXLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzFELFdBQVcsQ0FBQyxJQUFJLEVBQUUseUJBQXlCLEVBQUUsSUFBSSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDM0UsV0FBVyxDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBQ08sa0JBQWtCLENBQUMsS0FBb0I7O1FBQzdDLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDOUQsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO1lBQ2QsVUFBVSxFQUFFLEtBQUssQ0FBQyxlQUFlO1lBQ2pDLE1BQU0sUUFBRSxLQUFLLENBQUMsY0FBYyxtQ0FBSSxHQUFHLENBQUMsc0JBQXNCLENBQUMsS0FBSyxDQUFDO2dCQUMvRCxPQUFPLEVBQUUsR0FBRyxDQUFDLGtCQUFrQixDQUFDLFVBQVU7YUFDM0MsQ0FBQztZQUNGLGdCQUFnQixFQUFFLElBQUk7WUFDdEIsZUFBZSxRQUFFLEtBQUssQ0FBQyxlQUFlLG1DQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztZQUM5RCxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPLENBQUM7WUFDekQsWUFBWSxRQUFFLEtBQUssQ0FBQyxZQUFZLG1DQUFJLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUM7WUFDcEUsY0FBYyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsc0JBQXNCLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFLGtCQUFrQixDQUFDO1lBQ3JHLGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU87U0FDekMsQ0FBQyxDQUFDO1FBQ0gsT0FBTztZQUNMLFdBQVcsRUFBRSxVQUFVLENBQUMsV0FBVztZQUNuQyxRQUFRLEVBQUUsVUFBVSxDQUFDLHlCQUF5QjtZQUM5QyxVQUFVLEVBQUUsVUFBVSxDQUFDLGtCQUFrQjtZQUN6QyxNQUFNLEVBQUUsVUFBVSxDQUFDLE1BQU87U0FDM0IsQ0FBQztJQUNKLENBQUM7SUFDRCxvQ0FBb0M7SUFDNUIsaUJBQWlCLENBQUMsS0FBb0I7O1FBQzVDLE1BQU0sU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFO1lBQzNELE1BQU0sUUFBRSxLQUFLLENBQUMsYUFBYSxtQ0FBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsV0FBVyxDQUFDO2dCQUNuRSxPQUFPLEVBQUUsR0FBRyxDQUFDLHdCQUF3QixDQUFDLFVBQVU7YUFDakQsQ0FBQztZQUNGLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQztZQUN6RCxtQkFBbUIsRUFBRSxXQUFXO1lBQ2hDLGFBQWEsRUFBRTtnQkFDYixHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7Z0JBQ2QsVUFBVSxFQUFFLEtBQUssQ0FBQyxlQUFlO2dCQUNqQyxZQUFZLFFBQUUsS0FBSyxDQUFDLFlBQVksbUNBQUksSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQzthQUNyRTtZQUNELGNBQWMsRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLHNCQUFzQixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSx5QkFBeUIsQ0FBQztZQUM1RyxNQUFNLEVBQUU7Z0JBQ04sU0FBUyxRQUFFLEtBQUssQ0FBQyxlQUFlLG1DQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQzthQUN6RDtZQUNELGdCQUFnQixFQUFFLElBQUk7WUFDdEIsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsT0FBTztTQUN6QyxDQUFDLENBQUM7UUFDSCxPQUFPO1lBQ0wsV0FBVyxFQUFFLFNBQVMsQ0FBQyxXQUFXO1lBQ2xDLFFBQVEsRUFBRSxTQUFTLENBQUMsZUFBZSxDQUFDLFFBQVE7WUFDNUMsVUFBVSxFQUFFLFNBQVMsQ0FBQyxpQkFBaUI7WUFDdkMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxNQUFPO1NBQzFCLENBQUM7SUFDSixDQUFDO0lBQ08sd0JBQXdCLENBQUMsS0FBb0I7O1FBQ25ELE1BQU0sU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFBRTtZQUMzRSxNQUFNLEVBQUUsR0FBRyxDQUFDLHFCQUFxQixDQUFDLFlBQVk7WUFDOUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO1lBQ2QsVUFBVSxFQUFFLEtBQUssQ0FBQyxlQUFlO1lBQ2pDLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQztZQUN6RCxlQUFlLFFBQUUsS0FBSyxDQUFDLGVBQWUsbUNBQUksR0FBRyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQzlELGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU87WUFDeEMsY0FBYyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsc0JBQXNCLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFLHlCQUF5QixDQUFDO1lBQzVHLG1CQUFtQixFQUFFLFdBQVc7U0FDakMsQ0FBQyxDQUFDO1FBQ0gsT0FBTztZQUNMLFdBQVcsRUFBRSxTQUFTLENBQUMsV0FBVztZQUNsQyxRQUFRLEVBQUUsU0FBUyxDQUFDLGVBQWUsQ0FBQyxRQUFRO1lBQzVDLFVBQVUsRUFBRSxTQUFTLENBQUMsaUJBQWlCO1lBQ3ZDLE1BQU0sRUFBRSxTQUFTLENBQUMsTUFBTztTQUMxQixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBM0ZELDRCQTJGQztBQUVELFNBQVMsY0FBYyxDQUFDLEtBQW9CO0lBQzFDLDBDQUEwQztJQUMxQyxPQUFPLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLEtBQUssR0FBRztXQUNyRCxPQUFPLENBQUMsR0FBRyxDQUFDLG1CQUFtQixLQUFLLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDcEcsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUN0QyxHQUFHLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3JGLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFBRSxXQUFXLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQztBQUMvRCxDQUFDO0FBRUQsU0FBUyxXQUFXLENBQUMsS0FBb0IsRUFBRSxFQUFVLEVBQUUsR0FBb0I7SUFDekUsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQztBQUN2RCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZWNzIGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3MnO1xuaW1wb3J0ICogYXMgZWZzIGZyb20gJ0Bhd3MtY2RrL2F3cy1lZnMnO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyByZHMgZnJvbSAnQGF3cy1jZGsvYXdzLXJkcyc7XG5pbXBvcnQgKiBhcyBzZWNyZXRzbWFuYWdlciBmcm9tICdAYXdzLWNkay9hd3Mtc2VjcmV0c21hbmFnZXInO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgcGF0dGVybnMgZnJvbSAnLi9pbmRleCc7XG5cblxuZXhwb3J0IGludGVyZmFjZSBXb3JkUHJlc3NQcm9wcyB7XG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuICAvKipcbiAgICogVlBDIHN1Ym5ldHMgZm9yIGRhdGFiYXNlXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVlBDIGlzb2xhdGVkIHN1Ym5ldHNcbiAgICovXG4gIHJlYWRvbmx5IGRhdGFiYXNlU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG4gIC8qKlxuICAgKiBEYXRhYmFzZSBpbnN0YW5jZSB0eXBlXG4gICAqXG4gICAqIEBkZWZhdWx0IHI1LmxhcmdlXG4gICAqL1xuICByZWFkb25seSBkYXRhYmFzZUluc3RhbmNlVHlwZT86IGVjMi5JbnN0YW5jZVR5cGU7XG4gIC8qKlxuICAgKiBUaGUgZGF0YWJhc2UgaW5zdGFuY2UgZW5naW5lXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTXlTUUwgOC4wLjIxXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZUVuZ2luZT86IHJkcy5JSW5zdGFuY2VFbmdpbmU7XG4gIC8qKlxuICAgKiBUaGUgZGF0YWJhc2UgY2x1c3RlciBlbmdpbmVcbiAgICpcbiAgICogQGRlZmF1bHQgcmRzLkF1cm9yYU15c3FsRW5naW5lVmVyc2lvbi5WRVJfMl8wOV8xXG4gICAqL1xuICByZWFkb25seSBjbHVzdGVyRW5naW5lPzogcmRzLklDbHVzdGVyRW5naW5lO1xuICAvKipcbiAgICogV2hldGhlciB0byB1c2UgYXVyb3JhIHNlcnZlcmxlc3MuIFdoZW4gZW5hYmxlZCwgdGhlIGBkYXRhYmFzZUluc3RhbmNlVHlwZWAgYW5kXG4gICAqIGBlbmdpbmVgIHdpbGwgYmUgaWdub3JlZC4gVGhlIGByZHMuRGF0YWJhc2VDbHVzdGVyRW5naW5lLkFVUk9SQV9NWVNRTGAgd2lsbCBiZSB1c2VkIGFzXG4gICAqIHRoZSBkZWZhdWx0IGNsdXN0ZXIgZW5naW5lIGluc3RlYWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBhdXJvcmFTZXJ2ZXJsZXNzPzogYm9vbGVhbjtcbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gdXNlIHNpbmdsZSBSRFMgaW5zdGFuY2UgcmF0aGVyIHRoYW4gUkRTIGNsdXN0ZXIuIE5vdCByZWNvbW1lbmRlZCBmb3IgcHJvZHVjdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHNpbmdsZURiSW5zdGFuY2U/OiBib29sZWFuO1xuICAvKipcbiAgICogZGF0YWJhc2UgYmFja3VwIHJldGVuc2lvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDcgZGF5c1xuICAgKi9cbiAgcmVhZG9ubHkgYmFja3VwUmV0ZW50aW9uPzogY2RrLkR1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiB0YXNrIG9wdGlvbnMgZm9yIHRoZSBXb3JkUHJlc3MgZmFyZ2F0ZSBzZXJ2aWNlXG4gICAqL1xuICByZWFkb25seSBzZXJ2aWNlUHJvcHM/OiBwYXR0ZXJucy5GYXJnYXRlVGFza1Byb3BzO1xuICAvKipcbiAgICogZW5hYmxlIGZhcmdhdGUgc3BvdFxuICAgKi9cbiAgcmVhZG9ubHkgc3BvdD86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBlbmFibGUgRUNTIEV4ZWNcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZUV4ZWN1dGVDb21tYW5kPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGNsYXNzIFdvcmRQcmVzcyBleHRlbmRzIGNkay5Db25zdHJ1Y3Qge1xuICByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuICByZWFkb25seSBkYj86IERhdGFiYXNlO1xuICByZWFkb25seSBzdmM6IHBhdHRlcm5zLkR1YWxBbGJGYXJnYXRlU2VydmljZTtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBXb3JkUHJlc3NQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMudnBjID0gcHJvcHMudnBjID8/IGdldE9yQ3JlYXRlVnBjKHRoaXMpO1xuICAgIHRoaXMuZGIgPSB0aGlzLmFkZERhdGFiYXNlKHtcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICBkYXRhYmFzZVN1Ym5ldHM6IHByb3BzLmRhdGFiYXNlU3VibmV0cyxcbiAgICAgIGluc3RhbmNlVHlwZTogcHJvcHMuZGF0YWJhc2VJbnN0YW5jZVR5cGUsXG4gICAgICBpbnN0YW5jZUVuZ2luZTogcHJvcHMuaW5zdGFuY2VFbmdpbmUsXG4gICAgICBjbHVzdGVyRW5naW5lOiBwcm9wcy5jbHVzdGVyRW5naW5lLFxuICAgICAgYXVyb3JhU2VydmVybGVzczogcHJvcHMuYXVyb3JhU2VydmVybGVzcyxcbiAgICAgIHNpbmdsZURiSW5zdGFuY2U6IHByb3BzLnNpbmdsZURiSW5zdGFuY2UsXG4gICAgICBiYWNrdXBSZXRlbnRpb246IHByb3BzLmJhY2t1cFJldGVudGlvbixcbiAgICB9KTtcblxuICAgIGNvbnN0IGxvZ0dyb3VwID0gbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgJ0xvZ0dyb3VwJywge1xuICAgICAgcmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRILFxuICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHRhc2sgPSBuZXcgZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbih0aGlzLCAnVGFzaycsIHtcbiAgICAgIGNwdTogMjU2LFxuICAgICAgbWVtb3J5TGltaXRNaUI6IDUxMixcbiAgICB9KTtcblxuICAgIHRhc2suYWRkQ29udGFpbmVyKCd3b3JkcHJlc3MnLCB7XG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnd29yZHByZXNzOmxhdGVzdCcpLFxuICAgICAgcG9ydE1hcHBpbmdzOiBbeyBjb250YWluZXJQb3J0OiA4MCB9XSxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIFdPUkRQUkVTU19EQl9OQU1FOiAnd29yZHByZXNzJyxcbiAgICAgIH0sXG4gICAgICBsb2dnaW5nOiBuZXcgZWNzLkF3c0xvZ0RyaXZlcih7XG4gICAgICAgIHN0cmVhbVByZWZpeDogJ3dvcmRwcmVzcy1mYXJnYXRlJyxcbiAgICAgICAgbG9nR3JvdXAsXG4gICAgICB9KSxcbiAgICAgIHNlY3JldHM6IHtcbiAgICAgICAgV09SRFBSRVNTX0RCX0hPU1Q6IGVjcy5TZWNyZXQuZnJvbVNlY3JldHNNYW5hZ2VyKFxuICAgICAgICAgIHRoaXMuZGIuc2VjcmV0LFxuICAgICAgICAgICdob3N0JyxcbiAgICAgICAgKSxcbiAgICAgICAgV09SRFBSRVNTX0RCX1VTRVI6IGVjcy5TZWNyZXQuZnJvbVNlY3JldHNNYW5hZ2VyKFxuICAgICAgICAgIHRoaXMuZGIuc2VjcmV0LFxuICAgICAgICAgICd1c2VybmFtZScsXG4gICAgICAgICksXG4gICAgICAgIFdPUkRQUkVTU19EQl9QQVNTV09SRDogZWNzLlNlY3JldC5mcm9tU2VjcmV0c01hbmFnZXIoXG4gICAgICAgICAgdGhpcy5kYi5zZWNyZXQsXG4gICAgICAgICAgJ3Bhc3N3b3JkJyxcbiAgICAgICAgKSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBoZWFsdGhDaGVjayA9IHtcbiAgICAgIHBhdGg6ICcvd3AtaW5jbHVkZXMvaW1hZ2VzL2JsYW5rLmdpZicsXG4gICAgICBpbnRlcnZhbDogY2RrLkR1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgfTtcblxuICAgIHRoaXMuc3ZjID0gbmV3IHBhdHRlcm5zLkR1YWxBbGJGYXJnYXRlU2VydmljZSh0aGlzLCAnQUxCRmFyZ2F0ZVNlcnZpY2UnLCB7XG4gICAgICB2cGM6IHRoaXMudnBjLFxuICAgICAgc3BvdDogcHJvcHMuc3BvdCxcbiAgICAgIGVuYWJsZUV4ZWN1dGVDb21tYW5kOiBwcm9wcy5lbmFibGVFeGVjdXRlQ29tbWFuZCxcbiAgICAgIHRhc2tzOiBwcm9wcy5zZXJ2aWNlUHJvcHMgPyBbcHJvcHMuc2VydmljZVByb3BzXSA6IFtcbiAgICAgICAge1xuICAgICAgICAgIGV4dGVybmFsOiB7IHBvcnQ6IDgwIH0sXG4gICAgICAgICAgdGFzayxcbiAgICAgICAgICBoZWFsdGhDaGVjayxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICByb3V0ZTUzT3BzOiB7XG4gICAgICAgIGVuYWJsZUxvYWRCYWxhbmNlckFsaWFzOiBmYWxzZSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBFRlMgdm9sdW1lXG4gICAgY29uc3QgZmlsZXN5c3RlbSA9IG5ldyBlZnMuRmlsZVN5c3RlbSh0aGlzLCAnRmlsZVN5c3RlbScsIHtcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICBlbmNyeXB0ZWQ6IHRydWUsXG4gICAgfSk7XG5cbiAgICBjb25zdCB2b2x1bWVOYW1lID0gJ2Vmcyc7XG4gICAgdGhpcy5zdmMuc2VydmljZVswXS50YXNrRGVmaW5pdGlvbi5hZGRWb2x1bWUoe1xuICAgICAgbmFtZTogdm9sdW1lTmFtZSxcbiAgICAgIGVmc1ZvbHVtZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgZmlsZVN5c3RlbUlkOiBmaWxlc3lzdGVtLmZpbGVTeXN0ZW1JZCxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdGhpcy5zdmMuc2VydmljZVswXS50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyPy5hZGRNb3VudFBvaW50cyh7XG4gICAgICBjb250YWluZXJQYXRoOiAnL3Zhci93d3cvaHRtbCcsXG4gICAgICByZWFkT25seTogZmFsc2UsXG4gICAgICBzb3VyY2VWb2x1bWU6IHZvbHVtZU5hbWUsXG4gICAgfSk7XG5cbiAgICBmaWxlc3lzdGVtLmNvbm5lY3Rpb25zLmFsbG93RnJvbSggbmV3IGVjMi5Db25uZWN0aW9ucyh7IHNlY3VyaXR5R3JvdXBzOiB0aGlzLnN2Yy5zZXJ2aWNlWzBdLmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzIH0pLCBlYzIuUG9ydC50Y3AoMjA0OSksICdhbGxvdyB3b3JkcHJlc3MgdG8gY29ubmVjdCBlZnMnKTtcblxuICAgIHRoaXMuZGIuY29ubmVjdGlvbnMuYWxsb3dGcm9tKHRoaXMuc3ZjLnNlcnZpY2VbMF0sIHRoaXMuZGIuY29ubmVjdGlvbnMuZGVmYXVsdFBvcnQhLCBgYWxsb3cgJHt0aGlzLnN2Yy5zZXJ2aWNlWzBdLnNlcnZpY2VOYW1lfSB0byBjb25uZWN0IGRiYCk7XG5cbiAgfVxuICBwcml2YXRlIGFkZERhdGFiYXNlKHByb3BzOiBEYXRhYmFzZVByb3BzKTogRGF0YWJhc2Uge1xuICAgIHJldHVybiBuZXcgRGF0YWJhc2UodGhpcywgJ0RhdGFiYXNlJywge1xuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cbn1cblxuZXhwb3J0IGludGVyZmFjZSBEYXRhYmFzZVByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBWUEMgZm9yIHRoZSBkYXRhYmFzZVxuICAgKi9cbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcbiAgLyoqXG4gICAqIFZQQyBzdWJuZXRzIGZvciBkYXRhYmFzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGF0YWJhc2VTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcbiAgLyoqXG4gICAqIFRoZSBkYXRhYmFzZSBpbnN0YW5jZSB0eXBlXG4gICAqXG4gICAqIEBkZWZhdWx0IHI1LmxhcmdlXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVR5cGU/OiBlYzIuSW5zdGFuY2VUeXBlO1xuICAvKipcbiAgICogVGhlIGRhdGFiYXNlIGluc3RhbmNlIGVuZ2luZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE15U1FMIDguMC4yMVxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VFbmdpbmU/OiByZHMuSUluc3RhbmNlRW5naW5lO1xuICAvKipcbiAgICogVGhlIGRhdGFiYXNlIGNsdXN0ZXIgZW5naW5lXG4gICAqXG4gICAqIEBkZWZhdWx0IHJkcy5BdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uVkVSXzJfMDlfMVxuICAgKi9cbiAgcmVhZG9ubHkgY2x1c3RlckVuZ2luZT86IHJkcy5JQ2x1c3RlckVuZ2luZTtcbiAgLyoqXG4gICAqIGVuYWJsZSBhdXJvcmEgc2VydmVybGVzc1xuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgYXVyb3JhU2VydmVybGVzcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gdXNlIHNpbmdsZSBSRFMgaW5zdGFuY2UgcmF0aGVyIHRoYW4gUkRTIGNsdXN0ZXIuIE5vdCByZWNvbW1lbmRlZCBmb3IgcHJvZHVjdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHNpbmdsZURiSW5zdGFuY2U/OiBib29sZWFuO1xuICAvKipcbiAgICogZGF0YWJhc2UgYmFja3VwIHJldGVuc2lvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDcgZGF5c1xuICAgKi9cbiAgcmVhZG9ubHkgYmFja3VwUmV0ZW50aW9uPzogY2RrLkR1cmF0aW9uO1xuICAvKipcbiAgICogQWxsb3cgZGF0YWJhc2UgY29ubmVjdGlvbi5cbiAgICogQGRlZmF1bHQgLSB0aGUgd2hvbGUgVlBDIENJRFJcbiAgICovXG4gIHJlYWRvbmx5IGFsbG93RnJvbT86IGVjMi5JQ29ubmVjdGFibGU7XG59XG5cbi8qKlxuICogRGF0YWJhc2UgY29uZmlndXJhdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIERhdGFiYXNlQ29maWcge1xuICAvKipcbiAgICogVGhlIGRhdGFiYXNlIHNlY3JldC5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3JldDogc2VjcmV0c21hbmFnZXIuSVNlY3JldDtcbiAgLyoqXG4gICAqIFRoZSBkYXRhYmFzZSBjb25ubmVjdGlvbnMuXG4gICAqL1xuICByZWFkb25seSBjb25uZWN0aW9uczogZWMyLkNvbm5lY3Rpb25zO1xuICAvKipcbiAgICogVGhlIGVuZHBvaW50IGFkZHJlc3MgZm9yIHRoZSBkYXRhYmFzZS5cbiAgICovXG4gIHJlYWRvbmx5IGVuZHBvaW50OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgZGF0YWJhc2FlIGlkZW50aWZpZXIuXG4gICAqL1xuICByZWFkb25seSBpZGVudGlmaWVyOiBzdHJpbmc7XG59XG5cblxuLyoqXG4gKiBSZXByZXNlbnRzIHRoZSBkYXRhYmFzZSBpbnN0YW5jZSBvciBkYXRhYmFzZSBjbHVzdGVyXG4gKi9cbmV4cG9ydCBjbGFzcyBEYXRhYmFzZSBleHRlbmRzIGNkay5Db25zdHJ1Y3Qge1xuICByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuICByZWFkb25seSBjbHVzdGVyRW5kcG9pbnRIb3N0bmFtZTogc3RyaW5nO1xuICByZWFkb25seSBjbHVzdGVySWRlbnRpZmllcjogc3RyaW5nO1xuICByZWFkb25seSBzZWNyZXQ6IHNlY3JldHNtYW5hZ2VyLklTZWNyZXQ7XG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBlYzIuQ29ubmVjdGlvbnM7XG4gIHByaXZhdGUgcmVhZG9ubHkgX215c3FsTGlzdGVuZXJQb3J0OiBudW1iZXIgPSAzMzA2O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGF0YWJhc2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgdGhpcy52cGMgPSBwcm9wcy52cGM7XG4gICAgY29uc3QgY29uZmlnID0gcHJvcHMuYXVyb3JhU2VydmVybGVzcyA/IHRoaXMuX2NyZWF0ZVNlcnZlcmxlc3NDbHVzdGVyKHByb3BzKVxuICAgICAgOiBwcm9wcy5zaW5nbGVEYkluc3RhbmNlID8gdGhpcy5fY3JlYXRlUmRzSW5zdGFuY2UocHJvcHMpIDogdGhpcy5fY3JlYXRlUmRzQ2x1c3Rlcihwcm9wcyk7XG4gICAgdGhpcy5zZWNyZXQgPSBjb25maWcuc2VjcmV0O1xuICAgIC8vIGFsbG93IGludGVybmFsbHkgZnJvbSB0aGUgc2FtZSBzZWN1cml0eSBncm91cFxuICAgIGNvbmZpZy5jb25uZWN0aW9ucy5hbGxvd0ludGVybmFsbHkoZWMyLlBvcnQudGNwKHRoaXMuX215c3FsTGlzdGVuZXJQb3J0KSk7XG4gICAgLy8gYWxsb3cgZnJvbSB0aGUgZmFyZ2F0ZSBzZXJ2aWNlIG9yIHRoZSB3aG9sZSB2cGMgY2lkclxuICAgIGNvbmZpZy5jb25uZWN0aW9ucy5hbGxvd0Zyb20ocHJvcHMuYWxsb3dGcm9tID8/IGVjMi5QZWVyLmlwdjQocHJvcHMudnBjLnZwY0NpZHJCbG9jayksIGVjMi5Qb3J0LnRjcCh0aGlzLl9teXNxbExpc3RlbmVyUG9ydCkpO1xuICAgIHRoaXMuY2x1c3RlckVuZHBvaW50SG9zdG5hbWUgPSBjb25maWcuZW5kcG9pbnQ7XG4gICAgdGhpcy5jbHVzdGVySWRlbnRpZmllciA9IGNvbmZpZy5pZGVudGlmaWVyO1xuICAgIHRoaXMuY29ubmVjdGlvbnMgPSBjb25maWcuY29ubmVjdGlvbnM7XG4gICAgcHJpbnRPdXRwdXQodGhpcywgJ0RCU2VjcmV0QXJuJywgY29uZmlnLnNlY3JldC5zZWNyZXRBcm4pO1xuICAgIHByaW50T3V0cHV0KHRoaXMsICdjbHVzdGVyRW5kcG9pbnRIb3N0bmFtZScsIHRoaXMuY2x1c3RlckVuZHBvaW50SG9zdG5hbWUpO1xuICAgIHByaW50T3V0cHV0KHRoaXMsICdjbHVzdGVySWRlbnRpZmllcicsIHRoaXMuY2x1c3RlcklkZW50aWZpZXIpO1xuICB9XG4gIHByaXZhdGUgX2NyZWF0ZVJkc0luc3RhbmNlKHByb3BzOiBEYXRhYmFzZVByb3BzKTogRGF0YWJhc2VDb2ZpZyB7XG4gICAgY29uc3QgZGJJbnN0YW5jZSA9IG5ldyByZHMuRGF0YWJhc2VJbnN0YW5jZSh0aGlzLCAnREJJbnN0YW5jZScsIHtcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgdnBjU3VibmV0czogcHJvcHMuZGF0YWJhc2VTdWJuZXRzLFxuICAgICAgZW5naW5lOiBwcm9wcy5pbnN0YW5jZUVuZ2luZSA/PyByZHMuRGF0YWJhc2VJbnN0YW5jZUVuZ2luZS5teXNxbCh7XG4gICAgICAgIHZlcnNpb246IHJkcy5NeXNxbEVuZ2luZVZlcnNpb24uVkVSXzhfMF8yMyxcbiAgICAgIH0pLFxuICAgICAgc3RvcmFnZUVuY3J5cHRlZDogdHJ1ZSxcbiAgICAgIGJhY2t1cFJldGVudGlvbjogcHJvcHMuYmFja3VwUmV0ZW50aW9uID8/IGNkay5EdXJhdGlvbi5kYXlzKDcpLFxuICAgICAgY3JlZGVudGlhbHM6IHJkcy5DcmVkZW50aWFscy5mcm9tR2VuZXJhdGVkU2VjcmV0KCdhZG1pbicpLFxuICAgICAgaW5zdGFuY2VUeXBlOiBwcm9wcy5pbnN0YW5jZVR5cGUgPz8gbmV3IGVjMi5JbnN0YW5jZVR5cGUoJ3I1LmxhcmdlJyksXG4gICAgICBwYXJhbWV0ZXJHcm91cDogcmRzLlBhcmFtZXRlckdyb3VwLmZyb21QYXJhbWV0ZXJHcm91cE5hbWUodGhpcywgJ1BhcmFtZXRlckdyb3VwJywgJ2RlZmF1bHQubXlzcWw4LjAnKSxcbiAgICAgIHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgfSk7XG4gICAgcmV0dXJuIHtcbiAgICAgIGNvbm5lY3Rpb25zOiBkYkluc3RhbmNlLmNvbm5lY3Rpb25zLFxuICAgICAgZW5kcG9pbnQ6IGRiSW5zdGFuY2UuZGJJbnN0YW5jZUVuZHBvaW50QWRkcmVzcyxcbiAgICAgIGlkZW50aWZpZXI6IGRiSW5zdGFuY2UuaW5zdGFuY2VJZGVudGlmaWVyLFxuICAgICAgc2VjcmV0OiBkYkluc3RhbmNlLnNlY3JldCEsXG4gICAgfTtcbiAgfVxuICAvLyBjcmVhdGUgYSBSRFMgZm9yIE15U1FMIERCIGNsdXN0ZXJcbiAgcHJpdmF0ZSBfY3JlYXRlUmRzQ2x1c3Rlcihwcm9wczogRGF0YWJhc2VQcm9wcyk6IERhdGFiYXNlQ29maWcge1xuICAgIGNvbnN0IGRiQ2x1c3RlciA9IG5ldyByZHMuRGF0YWJhc2VDbHVzdGVyKHRoaXMsICdEQkNsdXN0ZXInLCB7XG4gICAgICBlbmdpbmU6IHByb3BzLmNsdXN0ZXJFbmdpbmUgPz8gcmRzLkRhdGFiYXNlQ2x1c3RlckVuZ2luZS5hdXJvcmFNeXNxbCh7XG4gICAgICAgIHZlcnNpb246IHJkcy5BdXJvcmFNeXNxbEVuZ2luZVZlcnNpb24uVkVSXzJfMDlfMSxcbiAgICAgIH0pLFxuICAgICAgY3JlZGVudGlhbHM6IHJkcy5DcmVkZW50aWFscy5mcm9tR2VuZXJhdGVkU2VjcmV0KCdhZG1pbicpLFxuICAgICAgZGVmYXVsdERhdGFiYXNlTmFtZTogJ3dvcmRwcmVzcycsXG4gICAgICBpbnN0YW5jZVByb3BzOiB7XG4gICAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgICB2cGNTdWJuZXRzOiBwcm9wcy5kYXRhYmFzZVN1Ym5ldHMsXG4gICAgICAgIGluc3RhbmNlVHlwZTogcHJvcHMuaW5zdGFuY2VUeXBlID8/IG5ldyBlYzIuSW5zdGFuY2VUeXBlKCdyNS5sYXJnZScpLFxuICAgICAgfSxcbiAgICAgIHBhcmFtZXRlckdyb3VwOiByZHMuUGFyYW1ldGVyR3JvdXAuZnJvbVBhcmFtZXRlckdyb3VwTmFtZSh0aGlzLCAnUGFyYW1ldGVyR3JvdXAnLCAnZGVmYXVsdC5hdXJvcmEtbXlzcWw1LjcnKSxcbiAgICAgIGJhY2t1cDoge1xuICAgICAgICByZXRlbnRpb246IHByb3BzLmJhY2t1cFJldGVudGlvbiA/PyBjZGsuRHVyYXRpb24uZGF5cyg3KSxcbiAgICAgIH0sXG4gICAgICBzdG9yYWdlRW5jcnlwdGVkOiB0cnVlLFxuICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICB9KTtcbiAgICByZXR1cm4ge1xuICAgICAgY29ubmVjdGlvbnM6IGRiQ2x1c3Rlci5jb25uZWN0aW9ucyxcbiAgICAgIGVuZHBvaW50OiBkYkNsdXN0ZXIuY2x1c3RlckVuZHBvaW50Lmhvc3RuYW1lLFxuICAgICAgaWRlbnRpZmllcjogZGJDbHVzdGVyLmNsdXN0ZXJJZGVudGlmaWVyLFxuICAgICAgc2VjcmV0OiBkYkNsdXN0ZXIuc2VjcmV0ISxcbiAgICB9O1xuICB9XG4gIHByaXZhdGUgX2NyZWF0ZVNlcnZlcmxlc3NDbHVzdGVyKHByb3BzOiBEYXRhYmFzZVByb3BzKTogRGF0YWJhc2VDb2ZpZyB7XG4gICAgY29uc3QgZGJDbHVzdGVyID0gbmV3IHJkcy5TZXJ2ZXJsZXNzQ2x1c3Rlcih0aGlzLCAnQXVyb3JhU2VydmVybGVzc0NsdXN0ZXInLCB7XG4gICAgICBlbmdpbmU6IHJkcy5EYXRhYmFzZUNsdXN0ZXJFbmdpbmUuQVVST1JBX01ZU1FMLFxuICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy5kYXRhYmFzZVN1Ym5ldHMsXG4gICAgICBjcmVkZW50aWFsczogcmRzLkNyZWRlbnRpYWxzLmZyb21HZW5lcmF0ZWRTZWNyZXQoJ2FkbWluJyksXG4gICAgICBiYWNrdXBSZXRlbnRpb246IHByb3BzLmJhY2t1cFJldGVudGlvbiA/PyBjZGsuRHVyYXRpb24uZGF5cyg3KSxcbiAgICAgIHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICBwYXJhbWV0ZXJHcm91cDogcmRzLlBhcmFtZXRlckdyb3VwLmZyb21QYXJhbWV0ZXJHcm91cE5hbWUodGhpcywgJ1BhcmFtZXRlckdyb3VwJywgJ2RlZmF1bHQuYXVyb3JhLW15c3FsNS43JyksXG4gICAgICBkZWZhdWx0RGF0YWJhc2VOYW1lOiAnd29yZHByZXNzJyxcbiAgICB9KTtcbiAgICByZXR1cm4ge1xuICAgICAgY29ubmVjdGlvbnM6IGRiQ2x1c3Rlci5jb25uZWN0aW9ucyxcbiAgICAgIGVuZHBvaW50OiBkYkNsdXN0ZXIuY2x1c3RlckVuZHBvaW50Lmhvc3RuYW1lLFxuICAgICAgaWRlbnRpZmllcjogZGJDbHVzdGVyLmNsdXN0ZXJJZGVudGlmaWVyLFxuICAgICAgc2VjcmV0OiBkYkNsdXN0ZXIuc2VjcmV0ISxcbiAgICB9O1xuICB9XG59XG5cbmZ1bmN0aW9uIGdldE9yQ3JlYXRlVnBjKHNjb3BlOiBjZGsuQ29uc3RydWN0KTogZWMyLklWcGMge1xuICAvLyB1c2UgYW4gZXhpc3RpbmcgdnBjIG9yIGNyZWF0ZSBhIG5ldyBvbmVcbiAgcmV0dXJuIHNjb3BlLm5vZGUudHJ5R2V0Q29udGV4dCgndXNlX2RlZmF1bHRfdnBjJykgPT09ICcxJ1xuICAgIHx8IHByb2Nlc3MuZW52LkNES19VU0VfREVGQVVMVF9WUEMgPT09ICcxJyA/IGVjMi5WcGMuZnJvbUxvb2t1cChzY29wZSwgJ1ZwYycsIHsgaXNEZWZhdWx0OiB0cnVlIH0pIDpcbiAgICBzY29wZS5ub2RlLnRyeUdldENvbnRleHQoJ3VzZV92cGNfaWQnKSA/XG4gICAgICBlYzIuVnBjLmZyb21Mb29rdXAoc2NvcGUsICdWcGMnLCB7IHZwY0lkOiBzY29wZS5ub2RlLnRyeUdldENvbnRleHQoJ3VzZV92cGNfaWQnKSB9KSA6XG4gICAgICBuZXcgZWMyLlZwYyhzY29wZSwgJ1ZwYycsIHsgbWF4QXpzOiAzLCBuYXRHYXRld2F5czogMSB9KTtcbn1cblxuZnVuY3Rpb24gcHJpbnRPdXRwdXQoc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGtleTogc3RyaW5nIHwgbnVtYmVyKSB7XG4gIG5ldyBjZGsuQ2ZuT3V0cHV0KHNjb3BlLCBpZCwgeyB2YWx1ZTogU3RyaW5nKGtleSkgfSk7XG59XG5cblxuIl19