"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack2', { env });
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
// order service
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
// customer service
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
// product service
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
// nginx service
const nginxTask = new ecs.FargateTaskDefinition(stack, 'nginxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nginxTask.addContainer('nginx', {
    image: ecs.ContainerImage.fromRegistry('nginx:latest'),
    portMappings: [
        { containerPort: 80 },
    ],
});
// php service
const phpTask = new ecs.FargateTaskDefinition(stack, 'phpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
phpTask.addContainer('php', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nginx-php')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// laravel-nginx-php-fpm service
const laravelNginxPhpFpmTask = new ecs.FargateTaskDefinition(stack, 'laravelNginxPhpFpmTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelNginxPhpFpmTask.addContainer('laravel-nginx-php-fpm', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-nginx-php-fpm')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// // laravel-bitnami service
// const laravelBitnamiTask = new ecs.FargateTaskDefinition(stack, 'laravelBitnamiTask', {
//   cpu: 256,
//   memoryLimitMiB: 512,
// });
// laravelBitnamiTask.addContainer('laravel-bitnami', {
//   image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-bitnami')),
//   portMappings: [
//     { containerPort: 3000 },
//   ],
// });
// NuxtJS service
const nuxtTask = new ecs.FargateTaskDefinition(stack, 'nuxtTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nuxtTask.addContainer('nuxt', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nuxt')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// Node.js service
const nodeTask = new ecs.FargateTaskDefinition(stack, 'nodeTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nodeTask.addContainer('node', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/node')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// java spring boot service
const javaTask = new ecs.FargateTaskDefinition(stack, 'javaTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
javaTask.addContainer('java', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/java-spring-boot')),
    portMappings: [
        { containerPort: 8080 },
    ],
});
const certArn = stack.node.tryGetContext('ACM_CERT_ARN');
const cert = certArn ? acm.Certificate.fromCertificateArn(stack, 'Cert', certArn) : undefined;
const svc = new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            task: orderTask,
            desiredCount: 2,
            internal: { port: 80 },
            external: cert ? { port: 443, certificate: [cert] } : { port: 80 },
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            task: customerTask,
            desiredCount: 1,
            internal: { port: 8080 },
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        {
            task: productTask,
            desiredCount: 1,
            internal: { port: 9090 },
        },
        // The nginx service(external only)
        {
            task: nginxTask,
            desiredCount: 1,
            external: { port: 9091 },
        },
        // The nginx-php-fpm service(external/internal)
        {
            task: phpTask,
            desiredCount: 1,
            internal: { port: 9092 },
            external: { port: 9092 },
        },
        // The NuxtJS service(external/internal)
        {
            task: nuxtTask,
            desiredCount: 1,
            internal: { port: 9093 },
            external: { port: 9093 },
        },
        // The node service(external/internal)
        {
            task: nodeTask,
            desiredCount: 1,
            internal: { port: 9094 },
            external: { port: 9094 },
        },
        // The laravel-nginx-php-fpm service(external/internal)
        {
            task: laravelNginxPhpFpmTask,
            desiredCount: 1,
            internal: { port: 9095 },
            external: { port: 9095 },
        },
        // // The laravel-bitnami service(external/internal)
        // {
        //   task: laravelBitnamiTask,
        //   desiredCount: 1,
        //   internal: { port: 9096 },
        //   external: { port: 9096 },
        // },
        // java spring boot service(external/internal)
        {
            task: javaTask,
            desiredCount: 1,
            internal: { port: 9097 },
            external: { port: 9097 },
            healthCheck: { path: '/hello-world' },
        },
    ],
    route53Ops: {
        zoneName,
        externalAlbRecordName,
        internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,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