'''
*atmdata: Table of atomic data*
---------------------------------

The entries here are:

  XrayFF: a dict of form factor coefficients

  AtmSize: atom Sizes, bond radii, angle radii, H-bond radii

  AtmBlens: atom masses & neutron scattering length (b,b'), sig(incoh) @ 1A

  MagFF: neutron magnetic form factor coeff: M for j<0> & N for j<2>

Sources:
  
           Exponential scattering factor curve coeficients, Cromer and
           Waber(1971) Int. Tables Vol. IV. Delta f' and delta f" terms,
           Cromer(1971) Int. Tables Vol. IV Atomic weights from CRC 56th
           Edition.

           Neutron scattering lengths & abs. cross sections from
           H. Rauch & W. Waschowski, Neutron Data Booklet, 2003.  X-ray
           <j0> & <j2> coeff. from Intl. Tables for Cryst, Vol. C
           5-d <j0> & <j2> from Kobayashi K, Nagao T, Ito M.
           Acta Crystallogr A67, 473-480 (2011)

           Neutron anomalous coeff (LS) from fitting Lynn & Seeger, At. Data & Nuc. Data Tables, 44, 191-207(1990)

           O2- x-ray scattering factor from Tokonami (1965) Acta Cryst 19, 486

           At wts from 14th ed Nuclides & Isotopes, 1989 GE Co.

'''
XrayFF = { 
#form factor coefficients
    'H':    {'Z':1,'fa':[ .493002,  .322912,  .140191,  .040810],'fb':[  10.5109,  26.1257,  3.14236,  57.7997],'fc':  .003038},
    'H-1':  {'Z':1,'fa':[0.897661, 0.565616,  .415815,  .116973],'fb':[  53.1368,   15.187,  186.576,  3.56709],'fc':  .002389},
    'D':    {'Z':1,'fa':[ .493002,  .322912,  .140191,  .040810],'fb':[  10.5109,  26.1257,  3.14236,  57.7997],'fc':  .003038},
    'D-1':  {'Z':1,'fa':[0.897661, 0.565616,  .415815,  .116973],'fb':[  53.1368,   15.187,  186.576,  3.56709],'fc':  .002389},
    'T':    {'Z':1,'fa':[ .493002,  .322912,  .140191,  .040810],'fb':[  10.5109,  26.1257,  3.14236,  57.7997],'fc':  .003038},
    'T-1':  {'Z':1,'fa':[0.897661, 0.565616,  .415815,  .116973],'fb':[  53.1368,   15.187,  186.576,  3.56709],'fc':  .002389},
    'He':   {'Z':2,'fa':[  0.8734,   0.6309,   0.3112,    0.178],'fb':[   9.1037,   3.3568,  22.9276,   0.9821],'fc':   0.0064},
    'Li':   {'Z':3,'fa':[  1.1282,   0.7508,   0.6175,   0.4653],'fb':[   3.9546,   1.0524,  85.3905,  168.261],'fc':   0.0377},
    'Li+1': {'Z':3,'fa':[  0.6968,   0.7888,   0.3414,   0.1563],'fb':[   4.6237,   1.9557,   0.6316,  10.0953],'fc':   0.0167},
    'Be':   {'Z':4,'fa':[  1.5919,   1.1278,   0.5391,   0.7029],'fb':[  43.6427,   1.8623,  103.483,    0.542],'fc':   0.0385},
    'Be+2': {'Z':4,'fa':[  6.2603,   0.8849,   0.7993,   0.1647],'fb':[   0.0027,   0.8313,   2.2758,   5.1146],'fc':  -6.1092},
    'B':    {'Z':5,'fa':[  2.0545,   1.3326,   1.0979,   0.7068],'fb':[  23.2185,    1.021,  60.3498,   0.1403],'fc':   0.1932},
    'C':    {'Z':6,'fa':[    2.31,     1.02,   1.5886,    0.865],'fb':[  20.8439,  10.2075,   0.5687,  51.6512],'fc':   0.2156},
#   'CV':   {'Z':6,'fa':[ 2.26069,  1.56165,  1.05075, 0.839259],'fb':[  22.6907, 0.656665,  9.75618,  55.5949],'fc': 0.286977},
    'N':    {'Z':7,'fa':[ 12.2126,   3.1322,   2.0125,   1.1663],'fb':[   0.0057,   9.8933,  28.9975,   0.5826],'fc':  -11.529},
    'O':    {'Z':8,'fa':[  3.0485,   2.2868,   1.5463,    0.867],'fb':[  13.2771,   5.7011,   0.3239,  32.9089],'fc':   0.2508},
    'O-1':  {'Z':8,'fa':[  4.1916,  1.63969,  1.52673,  -20.307],'fb':[  12.8573,  4.17236,  47.0179, -0.01404],'fc':  21.9412},
    'O-2':  {'Z':8,'fa':[ 3.76936,  2.53342,  1.91579,  1.55094],'fb':[  15.0797,  6.30430,  40.3445,  0.31291],'fc':  0.22789},
    'F':    {'Z':9,'fa':[  3.5392,   2.6412,    1.517,   1.0243],'fb':[  10.2825,   4.2944,   0.2615,  26.1476],'fc':   0.2776},
    'F-1':  {'Z':9,'fa':[  3.6322,  3.51057,  1.26064, 0.940706],'fb':[  5.27756,  14.7353, 0.442258,  47.3437],'fc': 0.653396},
    'Ne':   {'Z':10,'fa':[  3.9553,   3.1125,   1.4546,   1.1251],'fb':[   8.4042,   3.4262,   0.2306,  21.7184],'fc':   0.3515},
    'Na':   {'Z':11,'fa':[  4.7626,   3.1736,   1.2674,   1.1128],'fb':[    3.285,   8.8422,   0.3136,  129.424],'fc':    0.676},
    'Na+1': {'Z':11,'fa':[  3.2565,   3.9362,   1.3998,   1.0032],'fb':[   2.6671,   6.1153,   0.2001,   14.039],'fc':    0.404},
    'Mg':   {'Z':12,'fa':[  5.4204,   2.1735,   1.2269,   2.3073],'fb':[   2.8275,  79.2611,   0.3808,   7.1937],'fc':   0.8584},
    'Mg+2': {'Z':12,'fa':[  3.4988,   3.8378,   1.3284,   0.8497],'fb':[   2.1676,   4.7542,    0.185,  10.1411],'fc':   0.4853},
    'Al':   {'Z':13,'fa':[  6.4202,   1.9002,   1.5936,   1.9646],'fb':[   3.0387,   0.7426,  31.5472,  85.0886],'fc':   1.1151},
    'Al+3': {'Z':13,'fa':[ 4.17448,   3.3876,  1.20296, 0.528137],'fb':[  1.93816,  4.14553, 0.228753,  8.28524],'fc': 0.706786},
    'Si':   {'Z':14,'fa':[  6.2915,   3.0353,   1.9891,    1.541],'fb':[   2.4386,  32.3337,   0.6785,  81.6937],'fc':   1.1407},
    'Si+4': {'Z':14,'fa':[ 4.43918,  3.20345,  1.19453,  0.41653],'fb':[  1.64167,  3.43757,   0.2149,  6.65365],'fc': 0.746297},
#   'SiV':  {'Z':14,'fa':[ 5.66269,  3.07164,  2.62446,   1.3932],'fb':[   2.6652,  38.6634, 0.916946,  93.5458],'fc':  1.24707},
    'P':    {'Z':15,'fa':[  6.4345,   4.1791,     1.78,   1.4908],'fb':[   1.9067,   27.157,    0.526,  68.1645],'fc':   1.1149},
    'S':    {'Z':16,'fa':[  6.9053,   5.2034,   1.4379,   1.5863],'fb':[   1.4679,  22.2151,   0.2536,   56.172],'fc':   0.8669},
    'Cl':   {'Z':17,'fa':[ 11.4604,   7.1964,   6.2556,   1.6455],'fb':[   0.0104,   1.1662,  18.5194,  47.7784],'fc':  -9.5574},
    'Cl-1': {'Z':17,'fa':[ 18.2915,   7.2084,   6.5337,   2.3386],'fb':[   0.0066,   1.1717,  19.5424,  60.4486],'fc':  -16.378},
    'Ar':   {'Z':18,'fa':[  7.4845,   6.7723,   0.6539,   1.6442],'fb':[   0.9072,  14.8407,  43.8983,  33.3929],'fc':   1.4445},
    'K':    {'Z':19,'fa':[  8.2186,   7.4398,   1.0519,   0.8659],'fb':[  12.7949,   0.7748,  213.187,  41.6841],'fc':   1.4228},
    'K+1':  {'Z':19,'fa':[  7.9578,   7.4917,    6.359,   1.1915],'fb':[  12.6331,   0.7674,   -0.002,  31.9128],'fc':  -4.9978},
    'Ca':   {'Z':20,'fa':[  8.6266,   7.3873,   1.5899,   1.0211],'fb':[  10.4421,   0.6599,  85.7484,  178.437],'fc':   1.3751},
    'Ca+2': {'Z':20,'fa':[ 15.6348,   7.9518,   8.4372,   0.8537],'fb':[  -0.0074,   0.6089,  10.3116,  25.9905],'fc':  -14.875},
    'Sc':   {'Z':21,'fa':[   9.189,   7.3679,   1.6409,    1.468],'fb':[   9.0213,   0.5729,  136.108,  51.3531],'fc':   1.3329},
    'Sc+3': {'Z':21,'fa':[ 13.4008,   8.0273,  1.65943,  1.57936],'fb':[  0.29854,   7.9629, -0.28604,  16.0662],'fc':  -6.6667},
    'Ti':   {'Z':22,'fa':[  9.7595,   7.3558,   1.6991,   1.9021],'fb':[   7.8508,      0.5,  35.6338,  116.105],'fc':   1.2807},
    'Ti+2': {'Z':22,'fa':[ 9.11423,  7.62174,   2.2793, 0.087899],'fb':[   7.5243, 0.457585,  19.5361,  61.6558],'fc': 0.897155},
    'Ti+3': {'Z':22,'fa':[ 17.7344,  8.73816,  5.25691,  1.92134],'fb':[  0.22061,  7.04716, -0.15762,  15.9768],'fc':  -14.652},
    'Ti+4': {'Z':22,'fa':[ 19.5114,  8.23473,  2.01341,   1.5208],'fb':[ 0.178847,  6.67018, -0.29263,  12.9464],'fc':   -13.28},
    'V':    {'Z':23,'fa':[ 10.2971,   7.3511,   2.0703,   2.0571],'fb':[   6.8657,   0.4385,  26.8938,  102.478],'fc':   1.2199},
    'V+2':  {'Z':23,'fa':[  10.106,   7.3541,   2.2884,   0.0223],'fb':[   6.8818,   0.4409,  20.3004,  115.122],'fc':   1.2298},
    'V+3':  {'Z':23,'fa':[ 9.43141,   7.7419,  2.15343, 0.016865],'fb':[  6.39535, 0.383349,  15.1908,  63.969 ],'fc': 0.656565},
    'V+5':  {'Z':23,'fa':[ 15.6887,  8.14208,  2.03081,   -9.576],'fb':[ 0.679003,  5.40135,  9.97278, 0.940464],'fc':   1.7143},
    'Cr':   {'Z':24,'fa':[ 10.6406,   7.3537,    3.324,   1.4922],'fb':[   6.1038,    0.392,  20.2626,  98.7399],'fc':   1.1832},
    'Cr+2': {'Z':24,'fa':[ 9.54034,   7.7509,  3.58274, 0.509107],'fb':[  5.66078, 0.344261,  13.3075,  32.4224],'fc': 0.616898},
    'Cr+3': {'Z':24,'fa':[  9.6809,  7.81136,  2.87603, 0.113575],'fb':[  5.59463, 0.334393,  12.8288,  32.8761],'fc': 0.518275},
    'Mn':   {'Z':25,'fa':[ 11.2819,   7.3573,   3.0193,   2.2441],'fb':[   5.3409,   0.3432,  17.8674,  83.7543],'fc':   1.0896},
    'Mn+2': {'Z':25,'fa':[ 10.8061,    7.362,   3.5268,   0.2184],'fb':[   5.2796,   0.3435,   14.343,  41.3235],'fc':   1.0874},
    'Mn+3': {'Z':25,'fa':[ 9.84521,  7.87194,  3.56531, 0.323613],'fb':[  4.91797, 0.294393,  10.8171,  24.1281],'fc': 0.393974},
    'Mn+4': {'Z':25,'fa':[ 9.96253,  7.97057,  2.76067, 0.054447],'fb':[   4.8485, 0.283303,  10.4852,   27.573],'fc': 0.251877},
    'Fe':   {'Z':26,'fa':[ 11.7695,   7.3573,   3.5222,   2.3045],'fb':[   4.7611,   0.3072,  15.3535,  76.8805],'fc':   1.0369},
    'Fe+2': {'Z':26,'fa':[ 11.0424,    7.374,   4.1346,   0.4399],'fb':[   4.6538,   0.3053,  12.0546,  31.2809],'fc':   1.0097},
    'Fe+3': {'Z':26,'fa':[ 11.1764,   7.3863,   3.3948,   0.0724],'fb':[   4.6147,   0.3005,  11.6729,  38.5566],'fc':   0.9707},
    'Co':   {'Z':27,'fa':[ 12.2841,   7.3409,   4.0034,   2.3488],'fb':[   4.2791,   0.2784,  13.5359,  71.1692],'fc':   1.0118},
    'Co+2': {'Z':27,'fa':[ 11.2296,   7.3883,   4.7393,   0.7108],'fb':[   4.1231,   0.2726,  10.2443,  25.6466],'fc':   0.9324},
    'Co+3': {'Z':27,'fa':[  10.338,  7.88173,  4.76795, 0.725591],'fb':[  3.90969, 0.238668,  8.35583,  18.3491],'fc': 0.286667},
    'Ni':   {'Z':28,'fa':[ 12.8376,    7.292,   4.4438,     2.38],'fb':[   3.8785,   0.2565,  12.1763,  66.3421],'fc':   1.0341},
    'Ni+2': {'Z':28,'fa':[ 11.4166,   7.4005,   5.3442,   0.9773],'fb':[   3.6766,   0.2449,    8.873,  22.1626],'fc':   0.8614},
    'Ni+3': {'Z':28,'fa':[ 10.7806,  7.75868,  5.22746, 0.847114],'fb':[   3.5477,  0.22314,  7.64468,  16.9673],'fc': 0.386044},
    'Cu':   {'Z':29,'fa':[  13.338,   7.1676,   5.6158,   1.6735],'fb':[   3.5828,    0.247,  11.3966,  64.8126],'fc':    1.191},
    'Cu+1': {'Z':29,'fa':[ 11.9475,   7.3573,   6.2455,   1.5578],'fb':[   3.3669,   0.2274,   8.6625,  25.8487],'fc':     0.89},
    'Cu+2': {'Z':29,'fa':[ 11.8168,  7.11181,  5.78135,  1.14523],'fb':[  3.37484, 0.244078,   7.9876,   19.897],'fc':  1.14431},
    'Zn':   {'Z':30,'fa':[ 14.0743,   7.0318,   5.1652,     2.41],'fb':[   3.2655,   0.2333,  10.3163,  58.7097],'fc':   1.3041},
    'Zn+2': {'Z':30,'fa':[ 11.9719,   7.3862,   6.4668,    1.394],'fb':[   2.9946,   0.2031,   7.0826,  18.0995],'fc':   0.7807},
    'Ga':   {'Z':31,'fa':[ 15.2354,   6.7006,   4.3591,   2.9623],'fb':[   3.0669,   0.2412,  10.7805,  61.4135],'fc':   1.7189},
    'Ga+3': {'Z':31,'fa':[  12.692,  6.69883,  6.06692,   1.0066],'fb':[  2.81262,  0.22789,  6.36441,  14.4122],'fc':  1.53545},
    'Ge':   {'Z':32,'fa':[ 16.0816,   6.3747,   3.7068,    3.683],'fb':[   2.8509,   0.2516,  11.4468,  54.7625],'fc':   2.1313},
    'Ge+4': {'Z':32,'fa':[ 12.9172,  6.70003,  6.06791, 0.859041],'fb':[  2.53718, 0.205855,  5.47913,   11.603],'fc':  1.45572},
    'As':   {'Z':33,'fa':[ 16.6723,   6.0701,   3.4313,   4.2779],'fb':[   2.6345,   0.2647,  12.9479,  47.7972],'fc':    2.531},
    'Se':   {'Z':34,'fa':[ 17.0006,   5.8196,   3.9731,   4.3543],'fb':[   2.4098,   0.2726,  15.2372,  43.8163],'fc':   2.8409},
    'Br':   {'Z':35,'fa':[ 17.1789,   5.2358,   5.6377,   3.9851],'fb':[   2.1723,  16.5796,   0.2609,  41.4328],'fc':   2.9557},
    'Br-1': {'Z':35,'fa':[ 17.1718,   6.3338,   5.5754,   3.7272],'fb':[   2.2059,  19.3345,   0.2871,  58.1535],'fc':   3.1776},
    'Kr':   {'Z':36,'fa':[ 17.3555,   6.7286,   5.5493,   3.5375],'fb':[   1.9384,  16.5623,   0.2261,  39.3972],'fc':    2.825},
    'Rb':   {'Z':37,'fa':[ 17.1784,   9.6435,   5.1399,   1.5292],'fb':[   1.7888,  17.3151,   0.2748,  164.934],'fc':   3.4873},
    'Rb+1': {'Z':37,'fa':[ 17.5816,   7.6598,   5.8981,   2.7817],'fb':[   1.7139,  14.7957,   0.1603,  31.2087],'fc':   2.0782},
    'Sr':   {'Z':38,'fa':[ 17.5663,   9.8184,    5.422,   2.6694],'fb':[   1.5564,  14.0988,   0.1664,  132.376],'fc':   2.5064},
    'Sr+2': {'Z':38,'fa':[ 18.0874,   8.1373,   2.5654,  -34.193],'fb':[   1.4907,  12.6963,  24.5651,  -0.0138],'fc':  41.4025},
    'Y':    {'Z':39,'fa':[  17.776,  10.2946,  5.72629,  3.26588],'fb':[   1.4029,  12.8006, 0.125599,  104.354],'fc':  1.91213},
    'Y+3':  {'Z':39,'fa':[ 17.9268,   9.1531,  1.76795,  -33.108],'fb':[  1.35417,  11.2145,  22.6599, -0.01319],'fc':  40.2602},
    'Zr':   {'Z':40,'fa':[ 17.8765,   10.948,  5.41732,  3.65721],'fb':[  1.27618,   11.916, 0.117622,  87.6627],'fc':  2.06929},
    'Zr+4': {'Z':40,'fa':[ 18.1668,  10.0562,  1.01118,  -2.6479],'fb':[   1.2148,  10.1483,  21.6054, -0.10276],'fc':  9.41454},
    'Nb':   {'Z':41,'fa':[ 17.6142,  12.0144,  4.04183,  3.53346],'fb':[  1.18865,   11.766, 0.204785,  69.7957],'fc':  3.75591},
    'Nb+3': {'Z':41,'fa':[ 19.8812,  18.0653,  11.0177,  1.94715],'fb':[ 0.019175,  1.13305,  10.1621,  28.3389],'fc':  -12.912},
    'Nb+5': {'Z':41,'fa':[ 17.9163,  13.3417,   10.799, 0.337905],'fb':[  1.12446, 0.028781,  9.28206,  25.7228],'fc':  -6.3934},
    'Mo':   {'Z':42,'fa':[  3.7025,  17.2356,  12.8876,   3.7429],'fb':[   0.2772,   1.0958,   11.004,  61.6584],'fc':   4.3875},
    'Mo+3': {'Z':42,'fa':[ 21.1664,  18.2017,  11.7423,  2.30951],'fb':[ 0.014734,  1.03031,  9.53659,  26.6307],'fc':  -14.421},
    'Mo+5': {'Z':42,'fa':[ 21.0149,  18.0992,  11.4632, 0.740625],'fb':[ 0.014345,  1.02238,  8.78809,  23.3452],'fc':  -14.316},
    'Mo+6': {'Z':42,'fa':[ 17.8871,   11.175,  6.57891,      0.0],'fb':[  1.03649,  8.48061, 0.058881,      0.0],'fc': 0.344941},
    'Tc':   {'Z':43,'fa':[ 19.1301,  11.0948,  4.64901,  2.71263],'fb':[ 0.864132,  8.14487,  21.5707,  86.8472],'fc':  5.40428},
    'Ru':   {'Z':44,'fa':[ 19.2674,  12.9182,  4.86337,  1.56756],'fb':[  0.80852,  8.43467,  24.7997,  94.2928],'fc':  5.37874},
    'Ru+3': {'Z':44,'fa':[ 18.5638,  13.2885,  9.32602,  3.00964],'fb':[ 0.847329,  8.37164, 0.017662,   22.887],'fc':  -3.1892},
    'Ru+4': {'Z':44,'fa':[ 18.5003,  13.1787,  4.71304,  2.18535],'fb':[ 0.844582,  8.12534,  0.36495,  20.8504],'fc':  1.42357},
    'Rh':   {'Z':45,'fa':[ 19.2957,  14.3501,  4.73425,  1.28918],'fb':[ 0.751536,  8.21758,  25.8749,  98.6062],'fc':    5.328},
    'Rh+3': {'Z':45,'fa':[ 18.8785,  14.1259,  3.32515,  -6.1989],'fb':[ 0.764252,  7.84438,  21.2487, -0.01036],'fc':  11.8678},
    'Rh+4': {'Z':45,'fa':[ 18.8545,  13.9806,  2.53464,  -5.6526],'fb':[ 0.760825,  7.62436,  19.3317,  -0.0102],'fc':  11.2835},
    'Pd':   {'Z':46,'fa':[ 19.3319,  15.5017,  5.29537, 0.605844],'fb':[ 0.698655,  7.98929,  25.2052,  76.8986],'fc':  5.26593},
    'Pd+2': {'Z':46,'fa':[ 19.1701,  15.2096,  4.32234,      0.0],'fb':[ 0.696219,  7.55573,  22.5057,      0.0],'fc':   5.2916},
    'Pd+4': {'Z':46,'fa':[ 19.2493,    14.79,  2.89289,  -7.9492],'fb':[ 0.683839,  7.14833,  17.9144, 0.005127],'fc':  13.0174},
    'Ag':   {'Z':47,'fa':[ 19.2808,  16.6885,   4.8045,   1.0463],'fb':[   0.6446,   7.4726,  24.6605,  99.8156],'fc':    5.179},
    'Ag+1': {'Z':47,'fa':[ 19.1812,  15.9719,  5.27475, 0.357534],'fb':[ 0.646179,  7.19123,  21.7326,  66.1147],'fc':  5.21572},
    'Ag+2': {'Z':47,'fa':[ 19.1643,  16.2456,   4.3709,      0.0],'fb':[ 0.645643,  7.18544,  21.4072,      0.0],'fc':  5.21404},
    'Cd':   {'Z':48,'fa':[ 19.2214,  17.6444,    4.461,   1.6029],'fb':[   0.5946,   6.9089,  24.7008,  87.4825],'fc':   5.0694},
    'Cd+2': {'Z':48,'fa':[ 19.1514,  17.2535,  4.47128,      0.0],'fb':[ 0.597922,  6.80639,  20.2521,      0.0],'fc':  5.11937},
    'In':   {'Z':49,'fa':[ 19.1624,  18.5596,   4.2948,   2.0396],'fb':[   0.5476,   6.3776,  25.8499,  92.8029],'fc':   4.9391},
    'In+3': {'Z':49,'fa':[ 19.1045,  18.1108,  3.78897,      0.0],'fb':[ 0.551522,   6.3247,  17.3595,      0.0],'fc':  4.99635},
    'Sn':   {'Z':50,'fa':[ 19.1889,  19.1005,   4.4585,   2.4663],'fb':[   5.8303,   0.5031,  26.8909,  83.9571],'fc':   4.7821},
    'Sn+2': {'Z':50,'fa':[ 19.1094,  19.0548,   4.5648,    0.487],'fb':[   0.5036,   5.8378,  23.3752,  62.2061],'fc':   4.7861},
    'Sn+4': {'Z':50,'fa':[ 18.9333,  19.7131,   3.4182,   0.0193],'fb':[    5.764,   0.4655,  14.0049,  -0.7583],'fc':   3.9182},
    'Sb':   {'Z':51,'fa':[ 19.6418,  19.0455,   5.0371,   2.6827],'fb':[   5.3034,   0.4607,  27.9074,  75.2825],'fc':   4.5909},
    'Sb+3': {'Z':51,'fa':[ 18.9755,   18.933,  5.10789, 0.288753],'fb':[ 0.467196,  5.22126,  19.5902,  55.5113],'fc':  4.69626},
    'Sb+5': {'Z':51,'fa':[ 19.8685,  19.0302,  2.41253,      0.0],'fb':[  5.44853, 0.467973,  14.1259,      0.0],'fc':  4.69263},
    'Te':   {'Z':52,'fa':[ 19.9644,  19.0138,  6.14487,   2.5239],'fb':[  4.81742, 0.420885,  28.5284,  70.8403],'fc':    4.352},
    'I':    {'Z':53,'fa':[ 20.1472,  18.9949,   7.5138,   2.2735],'fb':[    4.347,   0.3814,   27.766,  66.8776],'fc':   4.0712},
    'I-1':  {'Z':53,'fa':[ 20.2332,   18.997,   7.8069,   2.8868],'fb':[   4.3579,   0.3815,  29.5259,  84.9304],'fc':   4.0714},
    'Xe':   {'Z':54,'fa':[ 20.2933,  19.0298,   8.9767,     1.99],'fb':[   3.9282,    0.344,  26.4659,  64.2658],'fc':   3.7118},
    'Cs':   {'Z':55,'fa':[ 20.3892,  19.1062,   10.662,   1.4953],'fb':[    3.569,   0.3107,  24.3879,  213.904],'fc':   3.3352},
    'Cs+1': {'Z':55,'fa':[ 20.3524,  19.1278,  10.2821,   0.9615],'fb':[    3.552,   0.3086,  23.7128,  59.4565],'fc':   3.2791},
    'Ba':   {'Z':56,'fa':[ 20.3361,   19.297,   10.888,   2.6959],'fb':[    3.216,   0.2756,  20.2073,  167.202],'fc':   2.7731},
    'Ba+2': {'Z':56,'fa':[ 20.1807,  19.1136,  10.9054,  0.77634],'fb':[  3.21367,  0.28331,  20.0558,   51.746],'fc':  3.02902},
    'La':   {'Z':57,'fa':[  20.578,   19.599,  11.3727,  3.28719],'fb':[  2.94817, 0.244475,  18.7726,  133.124],'fc':  2.14678},
    'La+3': {'Z':57,'fa':[ 20.2489,  19.3763,  11.6323, 0.336048],'fb':[   2.9207, 0.250698,  17.8211,  54.9453],'fc':   2.4086},
    'Ce':   {'Z':58,'fa':[ 21.1671,  19.7695,  11.8513,  3.33049],'fb':[  2.81219, 0.226836,  17.6083,  127.113],'fc':  1.86264},
    'Ce+3': {'Z':58,'fa':[ 20.8036,   19.559,  11.9369, 0.612376],'fb':[  2.77691,  0.23154,  16.5408,  43.1692],'fc':  2.09013},
    'Ce+4': {'Z':58,'fa':[ 20.3235,  19.8186,  12.1233, 0.144583],'fb':[  2.65941,  0.21885,  15.7992,  62.2355],'fc':   1.5918},
    'Pr':   {'Z':59,'fa':  [22.044,  19.6697,  12.3856,  2.82428],'fb':[   2.77393, 0.222087,  16.7669,  143.644],'fc':   2.0583},
    'Pr+3': {'Z':59,'fa':[ 21.3727,  19.7491,  12.1329,  0.97518],'fb':[   2.6452, 0.214299,   15.323,  36.4065],'fc':  1.77132},
    'Pr+4': {'Z':59,'fa':[ 20.9413,  20.0539,  12.4668, 0.296689],'fb':[  2.54467, 0.202481,  14.8137,  45.4643],'fc':  1.24285},
    'Nd':   {'Z':60,'fa':[ 22.6845,  19.6847,   12.774,  2.85137],'fb':[  2.66248, 0.210628,   15.885,  137.903],'fc':  1.98486},
    'Nd+3': {'Z':60,'fa':[  21.961,  19.9339,    12.12,  1.51031],'fb':[  2.52722, 0.199237,  14.1783,  30.8717],'fc':  1.47588},
    'Pm':   {'Z':61,'fa':[ 23.3405,  19.6095,  13.1235,  2.87516],'fb':[   2.5627, 0.202088,  15.1009,  132.721],'fc':  2.02876},
    'Pm+3': {'Z':61,'fa':[ 22.5527,  20.1108,  12.0671,  2.07492],'fb':[   2.4174, 0.185769,  13.1275,  27.4491],'fc':  1.19499},
    'Sm':   {'Z':62,'fa':[ 24.0042,  19.4258,  13.4396,  2.89604],'fb':[  2.47274, 0.196451,  14.3996,  128.007],'fc':  2.20963},
    'Sm+3': {'Z':62,'fa':[ 23.1504,  20.2599,  11.9202,  2.71488],'fb':[  2.31641, 0.174081,  12.1571,  24.8242],'fc': 0.954586},
    'Eu':   {'Z':63,'fa':[ 24.6274,  19.0886,  13.7603,   2.9227],'fb':[   2.3879,   0.1942,  13.7546,  123.174],'fc':   2.5745},
    'Eu+2': {'Z':63,'fa':[ 24.0063,  19.9504,  11.8034,  3.87243],'fb':[  2.27783,  0.17353,  11.6096,  26.5156],'fc':  1.36389},
    'Eu+3': {'Z':63,'fa':[ 23.7497,  20.3745,  11.8509,  3.26503],'fb':[  2.22258,  0.16394,   11.311,  22.9966],'fc': 0.759344},
    'Gd':   {'Z':64,'fa':[ 25.0709,  19.0798,  13.8518,  3.54545],'fb':[  2.25341, 0.181951,  12.9331,  101.398],'fc':   2.4196},
    'Gd+3': {'Z':64,'fa':[ 24.3466,  20.4208,  11.8708,   3.7149],'fb':[  2.13553, 0.155525,  10.5782,  21.7029],'fc': 0.645089},
    'Tb':   {'Z':65,'fa':[ 25.8976,  18.2185,  14.3167,  2.95354],'fb':[  2.24256, 0.196143,  12.6648,  115.362],'fc':  3.58324},
    'Tb+3': {'Z':65,'fa':[ 24.9559,  20.3271,  12.2471,    3.773],'fb':[  2.05601, 0.149525,  10.0499,  21.2773],'fc': 0.691967},
    'Dy':   {'Z':66,'fa':[  26.507,  17.6383,  14.5596,  2.96577],'fb':[   2.1802, 0.202172,  12.1899,  111.874],'fc':  4.29728},
    'Dy+3': {'Z':66,'fa':[ 25.5395,  20.2861,  11.9812,  4.50073],'fb':[   1.9804, 0.143384,  9.34972,   19.581],'fc':  0.68969},
    'Ho':   {'Z':67,'fa':[ 26.9049,   17.294,  14.5583,  3.63837],'fb':[  2.07051,  0.19794,  11.4407,  92.6566],'fc':  4.56796},
    'Ho+3': {'Z':67,'fa':[ 26.1296,  20.0994,  11.9788,  4.93676],'fb':[  1.91072, 0.139358,  8.80018,  18.5908],'fc': 0.852795},
    'Er':   {'Z':68,'fa':[ 27.6563,  16.4285,  14.9779,  2.98233],'fb':[  2.07356, 0.223545,  11.3604,  105.703],'fc':  5.92046},
    'Er+3': {'Z':68,'fa':[  26.722,  19.7748,  12.1506,  5.17379],'fb':[  1.84659,  0.13729,  8.36225,  17.8974],'fc':  1.17613},
    'Tm':   {'Z':69,'fa':[ 28.1819,  15.8851,  15.1542,  2.98706],'fb':[  2.02859, 0.238849,  10.9975,  102.961],'fc':  6.75621},
    'Tm+3': {'Z':69,'fa':[ 27.3083,   19.332,  12.3339,  5.38348],'fb':[  1.78711, 0.136974,  7.96778,  17.2922],'fc':  1.63929},
    'Yb':   {'Z':70,'fa':[ 28.6641,  15.4345,  15.3087,  2.98963],'fb':[   1.9889, 0.257119,  10.6647,  100.417],'fc':  7.56672},
    'Yb+2': {'Z':70,'fa':[ 28.1209,  17.6817,  13.3335,  5.14657],'fb':[  1.78503,  0.15997,  8.18304,    20.39],'fc':  3.70983},
    'Yb+3': {'Z':70,'fa':[ 27.8917,  18.7614,  12.6072,  5.47647],'fb':[  1.73272,  0.13879,  7.64412,  16.8153],'fc':  2.26001},
    'Lu':   {'Z':71,'fa':[ 28.9476,  15.2208,     15.1,  3.71601],'fb':[  1.90182,  9.98519, 0.261033,  84.3298],'fc':  7.97628},
    'Lu+3': {'Z':71,'fa':[ 28.4628,   18.121,  12.8429,  5.59415],'fb':[  1.68216, 0.142292,  7.33727,  16.3535],'fc':  2.97573},
    'Hf':   {'Z':72,'fa':[  29.144,  15.1726,  14.7586,  4.30013],'fb':[  1.83262,   9.5999, 0.275116,   72.029],'fc':  8.58154},
    'Hf+4': {'Z':72,'fa':[ 28.8131,  18.4601,  12.7285,  5.59927],'fb':[  1.59136, 0.128903,  6.76232,  14.0366],'fc':  2.39699},
    'Ta':   {'Z':73,'fa':[ 29.2024,  15.2293,  14.5135,  4.76492],'fb':[  1.77333,  9.37046, 0.295977,  63.3644],'fc':  9.24354},
    'Ta+5': {'Z':73,'fa':[ 29.1587,  18.8407,  12.8268,  5.38695],'fb':[  1.50711, 0.116741,  6.31524,  12.4244],'fc':  1.78555},
    'W':    {'Z':74,'fa':[ 29.0818,    15.43,  14.4327,  5.11982],'fb':[  1.72029,   9.2259, 0.321703,   57.056],'fc':   9.8875},
    'W+6':  {'Z':74,'fa':[ 29.4936,  19.3763,  13.0544,  5.06412],'fb':[  1.42755, 0.104621,  5.93667,  11.1972],'fc':  1.01074},
    'Re':   {'Z':75,'fa':[ 28.7621,  15.7189,  14.5564,  5.44174],'fb':[  1.67191,  9.09227,   0.3505,  52.0861],'fc':   10.472},
    'Os':   {'Z':76,'fa':[ 28.1894,   16.155,  14.9305,  5.67589],'fb':[  1.62903,  8.97948, 0.382661,  48.1647],'fc':  11.0005},
    'Os+4': {'Z':76,'fa':[  30.419,  15.2637,  14.7458,  5.06795],'fb':[  1.37113,  6.84706, 0.165191,   18.003],'fc':  6.49804},
    'Ir':   {'Z':77,'fa':[ 27.3049,  16.7296,  15.6115,  5.83377],'fb':[  1.59279,  8.86553, 0.417916,  45.0011],'fc':  11.4722},
    'Ir+3': {'Z':77,'fa':[ 30.4156,   15.862,  13.6145,  5.82008],'fb':[  1.34323,  7.10909, 0.204633,  20.3254],'fc':  8.27903},
    'Ir+4': {'Z':77,'fa':[ 30.7058,  15.5512,  14.2326,  5.53672],'fb':[  1.30923,  6.71983, 0.167252,  17.4911],'fc':  6.96824},
    'Pt':   {'Z':78,'fa':[ 27.0059,  17.7639,  15.7131,   5.7837],'fb':[  1.51293,  8.81174, 0.424593,  38.6103],'fc':  11.6883},
    'Pt+2': {'Z':78,'fa':[ 29.8429,  16.7224,  13.2153,  6.35234],'fb':[  1.32927,  7.38979, 0.263297,  22.9426],'fc':  9.85329},
    'Pt+4': {'Z':78,'fa':[ 30.9612,  15.9829,  13.7348,  5.92034],'fb':[  1.24813,  6.60834,  0.16864,  16.9392],'fc':  7.39534},
    'Au':   {'Z':79,'fa':[ 16.8819,  18.5913,  25.5582,     5.86],'fb':[   0.4611,   8.6216,   1.4826,  36.3956],'fc':  12.0658},
    'Au+1': {'Z':79,'fa':[ 28.0109,  17.8204,  14.3359,  6.58077],'fb':[  1.35321,   7.7395, 0.356752,  26.4043],'fc':  11.2299},
    'Au+3': {'Z':79,'fa':[ 30.6886,  16.9029,  12.7801,  6.52354],'fb':[   1.2199,  6.82872, 0.212867,   18.659],'fc':   9.0968},
    'Hg':   {'Z':80,'fa':[ 20.6809,  19.0417,  21.6575,   5.9676],'fb':[    0.545,   8.4484,   1.5729,  38.3246],'fc':  12.6089},
    'Hg+1': {'Z':80,'fa':[ 25.0853,  18.4973,  16.8883,  6.48216],'fb':[  1.39507,  7.65105, 0.443378,  28.2262],'fc':  12.0205},
    'Hg+2': {'Z':80,'fa':[ 29.5641,    18.06,  12.8374,  6.89912],'fb':[  1.21152,  7.05639, 0.284738,  20.7482],'fc':  10.6268},
    'Tl':   {'Z':81,'fa':[ 27.5446,  19.1584,   15.538,  5.52593],'fb':[  0.65515,  8.70751,  1.96347,  45.8149],'fc':  13.1746},
    'Tl+1': {'Z':81,'fa':[ 21.3985,  20.4723,  18.7478,  6.82847],'fb':[   1.4711, 0.517394,  7.43463,  28.8482],'fc':  12.5258},
    'Tl+3': {'Z':81,'fa':[ 30.8695,  18.3481,  11.9328,  7.00574],'fb':[   1.1008,  6.53852, 0.219074,  17.2114],'fc':   9.8027},
    'Pb':   {'Z':82,'fa':[ 31.0617,  13.0637,   18.442,   5.9696],'fb':[   0.6902,   2.3576,    8.618,  47.2579],'fc':  13.4118},
    'Pb+2': {'Z':82,'fa':[ 21.7886,  19.5682,  19.1406,  7.01107],'fb':[   1.3366, 0.488383,   6.7727,  23.8132],'fc':  12.4734},
    'Pb+4': {'Z':82,'fa':[ 32.1244,  18.8003,  12.0175,  6.96886],'fb':[  1.00566,  6.10926, 0.147041,   14.714],'fc':  8.08428},
    'Bi':   {'Z':83,'fa':[ 33.3689,   12.951,  16.5877,   6.4692],'fb':[    0.704,   2.9238,   8.7937,  48.0093],'fc':  13.5782},
    'Bi+3': {'Z':83,'fa':[ 21.8053,  19.5026,  19.1053,  7.10295],'fb':[   1.2356,  6.24149, 0.469999,  20.3185],'fc':  12.4711},
    'Bi+5': {'Z':83,'fa':[ 33.5364,  25.0946,  19.2497,  6.91555],'fb':[  0.91654,  0.39042,  5.71414,  12.8285],'fc':  -6.7994},
    'Po':   {'Z':84,'fa':[ 34.6726,  15.4733,  13.1138,  7.02588],'fb':[ 0.700999,  3.55078,  9.55642,  47.0045],'fc':   13.677},
    'At':   {'Z':85,'fa':[ 35.3163,  19.0211,  9.49887,  7.42518],'fb':[  0.68587,  3.97458,  11.3824,  45.4715],'fc':  13.7108},
    'Rn':   {'Z':86,'fa':[ 35.5631,  21.2816,   8.0037,   7.4433],'fb':[   0.6631,   4.0691,  14.0422,  44.2473],'fc':  13.6905},
    'Fr':   {'Z':87,'fa':[ 35.9299,  23.0547,  12.1439,  2.11253],'fb':[ 0.646453,  4.17619,  23.1052,  150.645],'fc':  13.7247},
    'Ra':   {'Z':88,'fa':[  35.763,  22.9064,  12.4739,  3.21097],'fb':[ 0.616341,  3.87135,  19.9887,  142.325],'fc':  13.6211},
    'Ra+2': {'Z':88,'fa':[  35.215,    21.67,  7.91342,  7.65078],'fb':[ 0.604909,   3.5767,   12.601,  29.8436],'fc':  13.5431},
    'Ac':   {'Z':89,'fa':[ 35.6597,  23.1032,  12.5977,  4.08655],'fb':[ 0.589092,  3.65155,   18.599,   117.02],'fc':  13.5266},
    'Ac+3': {'Z':89,'fa':[ 35.1736,  22.1112,  8.19216,  7.05545],'fb':[ 0.579689,  3.41437,  12.9187,  25.9443],'fc':  13.4637},
    'Th':   {'Z':90,'fa':[ 35.5645,  23.4219,  12.7473,  4.80703],'fb':[ 0.563359,  3.46204,  17.8309,  99.1722],'fc':  13.4314},
    'Th+4': {'Z':90,'fa':[ 35.1007,  22.4418,  9.78554,  5.29444],'fb':[ 0.555054,  3.24498,  13.4661,  23.9533],'fc':   13.376},
    'Pa':   {'Z':91,'fa':[ 35.8847,  23.2948,  14.1891,  4.17287],'fb':[ 0.547751,  3.41519,  16.9235,  105.251],'fc':  13.4287},
    'U':    {'Z':92,'fa':[ 36.0228,  23.4128,  14.9491,    4.188],'fb':[   0.5293,   3.3253,  16.0927,  100.613],'fc':  13.3966},
    'U+3':  {'Z':92,'fa':[ 35.5747,  22.5259,  12.2165,  5.37073],'fb':[  0.52048,  3.12293,  12.7148,  26.3394],'fc':  13.3092},
    'U+4':  {'Z':92,'fa':[ 35.3715,  22.5326,  12.0291,   4.7984],'fb':[ 0.516598,  3.05053,  12.5723,  23.4582],'fc':  13.2671},
    'U+6':  {'Z':92,'fa':[ 34.8509,  22.7584,  14.0099,  1.21457],'fb':[ 0.507079,   2.8903,  13.1767,  25.2017],'fc':  13.1665},
    'Np':   {'Z':93,'fa':[ 36.1874,  23.5964,  15.6402,   4.1855],'fb':[ 0.511929,  3.25396,  15.3622,  97.4908],'fc':  13.3573},
    'Np+3': {'Z':93,'fa':[ 35.7074,   22.613,  12.9898,  5.43227],'fb':[ 0.502322,  3.03807,  12.1449,  25.4928],'fc':  13.2544},
    'Np+4': {'Z':93,'fa':[ 35.5103,  22.5787,  12.7766,  4.92159],'fb':[ 0.498626,  2.96627,  11.9484,  22.7502],'fc':  13.2116},
    'Np+6': {'Z':93,'fa':[ 35.0136,  22.7286,  14.3884,  1.75669],'fb':[  0.48981,  2.81099,    12.33,  22.6581],'fc':  13.113 },
    'Pu':   {'Z':94,'fa':[ 36.5254,  23.8083,  16.7707,  3.47947],'fb':[ 0.499384,  3.26371,  14.9455,   105.98],'fc':  13.3812},
    'Pu+3': {'Z':94,'fa':[   35.84,  22.7169,  13.5807,  5.66016],'fb':[ 0.484938,  2.96118,  11.5331,  24.3992],'fc':  13.1991},
    'Pu+4': {'Z':94,'fa':[ 35.6493,   22.646,  13.3595,  5.18831],'fb':[ 0.481422,   2.8902,   11.316,  21.8301],'fc':  13.1555},
    'Pu+6': {'Z':94,'fa':[ 35.1736,  22.7181,  14.7635,  2.28678],'fb':[ 0.473204,  2.73848,   11.553,  20.9303],'fc':  13.0582},
    'Am':   {'Z':95,'fa':[ 36.6706,  24.0992,  17.3415,  3.49331],'fb':[ 0.483629,  3.20647,  14.3136,  102.273],'fc':  13.3592},
    'Cm':   {'Z':96,'fa':[ 36.6488,  24.4096,   17.399,  4.21665],'fb':[ 0.465154,  3.08997,  13.4346,  88.4834],'fc':  13.2887},
    'Bk':   {'Z':97,'fa':[ 36.7881,  24.7736,  17.8919,  4.23284],'fb':[ 0.451018,  3.04619,  12.8946,   86.003],'fc':  13.2754},
    'Cf':   {'Z':98,'fa':[ 36.9185,  25.1995,  18.3317,  4.24391],'fb':[ 0.437533,  3.00775,  12.4044,  83.7881],'fc':  13.2674},
        }
# atom Sizes, bond radii, angle radii, H-bond radii
AtmSize = {
    'H': [0.98,  0.78, 1.20, 0],
    'D': [0.98,  0.78, 1.20, 0],
    'T': [0.98,  0.78, 1.20, 0],
    'He':[1.45,  1.25, 1.40, 0],
    'Li':[1.76,  1.56, 1.82, 0],
    'Be':[1.33,  1.13, 1.70, 0],
    'B': [1.18,  0.98, 2.08, 0],
    'C': [1.12,  0.92, 1.95, 0],
    'N': [1.08,  0.88, 1.85, 1.30],
    'O': [1.09,  0.89, 1.70, 1.40],
    'F': [1.30,  1.10, 1.73, 0],
    'Ne':[1.50,  1.30, 1.54, 0],
    'Na':[2.10,  1.91, 2.27, 0],
    'Mg':[1.80,  1.60, 1.73, 0],                                      
    'Al':[1.60,  1.43, 2.05, 0],
    'Si':[1.52,  1.32, 2.10, 0],
    'P': [1.48,  1.28, 2.08, 0],
    'S': [1.47,  1.27, 2.00, 0],
    'Cl':[1.70,  1.50, 1.97, 0],
    'Ar':[2.00,  1.80, 1.88, 0],
    'K': [2.58,  2.38, 2.75, 0],
    'Ca':[2.17,  1.97, 1.97, 0],
    'Sc':[1.84,  1.64, 1.70, 0],
    'Ti':[1.66,  1.46, 1.70, 0],
    'V': [1.55,  1.35, 1.70, 0],
    'Cr':[1.56,  1.36, 1.70, 0],
    'Mn':[1.54,  1.30, 1.70, 0],
    'Fe':[1.47,  1.27, 1.70, 0],
    'Co':[1.45,  1.25, 1.70, 0],
    'Ni':[1.45,  1.25, 1.63, 0],
    'Cu':[1.48,  1.28, 1.40, 0],
    'Zn':[1.59,  1.39, 1.39, 0],
    'Ga':[1.61,  1.41, 1.87, 0],
    'Ge':[1.57,  1.37, 1.70, 0],
    'As':[1.59,  1.39, 1.85, 0],
    'Se':[1.60,  1.40, 1.90, 0],
    'Br':[1.80,  1.60, 2.10, 0],
    'Kr':[2.10,  1.90, 2.02, 0],
    'Rb':[2.75,  2.55, 1.70, 0],
    'Sr':[2.35,  2.15, 1.70, 0],
    'Y': [2.00,  1.80, 1.70, 0],
    'Zr':[1.80,  1.60, 1.70, 0],
    'Nb':[1.67,  1.47, 1.70, 0],
    'Mo':[1.60,  1.40, 1.70, 0],
    'Tc':[1.56,  1.36, 1.70, 0],
    'Ru':[1.54,  1.34, 1.70, 0],
    'Rh':[1.54,  1.34, 1.70, 0],
    'Pd':[1.58,  1.38, 1.63, 0],
    'Ag':[1.64,  1.44, 1.72, 0],
    'Cd':[1.77,  1.57, 1.58, 0],
    'In':[1.86,  1.66, 1.93, 0],
    'Sn':[1.82,  1.62, 2.17, 0],
    'Sb':[1.79,  1.59, 2.20, 0],
    'Te':[1.80,  1.60, 2.06, 0],
    'I': [1.90,  1.70, 2.15, 0],
    'Xe':[2.30,  2.10, 2.16, 0],
    'Cs':[2.93,  2.73, 1.70, 0],
    'Ba':[2.44,  2.24, 1.70, 0],
    'La':[2.08,  1.88, 1.70, 0],
    'Ce':[2.02,  1.82, 1.70, 0],
    'Pr':[2.03,  1.83, 1.70, 0],
    'Nd':[2.02,  1.82, 1.70, 0],
    'Pm':[2.01,  1.81, 1.70, 0],
    'Sm':[2.00,  1.80, 1.70, 0],
    'Eu':[2.24,  2.04, 1.70, 0],
    'Gd':[2.00,  1.80, 1.70, 0],
    'Tb':[1.98,  1.78, 1.70, 0],
    'Dy':[1.97,  1.77, 1.70, 0],
    'Ho':[1.98,  1.78, 1.70, 0],
    'Er':[1.96,  1.76, 1.70, 0],
    'Tm':[1.95,  1.75, 1.70, 0],
    'Yb':[2.10,  1.90, 1.70, 0],
    'Lu':[1.93,  1.73, 1.70, 0],
    'Hf':[1.78,  1.58, 1.70, 0],
    'Ta':[1.67,  1.47, 1.70, 0],
    'W': [1.61,  1.41, 1.70, 0],
    'Re':[1.58,  1.38, 1.70, 0],
    'Os':[1.55,  1.35, 1.70, 0],
    'Ir':[1.56,  1.36, 1.70, 0],
    'Pt':[1.59,  1.39, 1.72, 0],
    'Au':[1.64,  1.44, 1.66, 0],
    'Hg':[1.77,  1.57, 1.55, 0],
    'Tl':[1.92,  1.72, 1.96, 0],
    'Pb':[1.95,  1.75, 2.02, 0],
    'Bi':[1.90,  1.70, 1.70, 0],
    'Po':[1.96,  1.76, 1.70, 0],
    'At':[2.00,  1.80, 1.70, 0],
    'Rn':[2.40,  2.20, 1.70, 0],
    'Fr':[3.00,  2.80, 1.70, 0],
    'Ra':[2.46,  2.26, 1.70, 0],
    'Ac':[2.09,  1.88, 1.70, 0],
    'Th':[2.00,  1.80, 1.70, 0],
    'Pa':[1.83,  1.63, 1.70, 0],
    'U': [1.76,  1.56, 1.86, 0],
    'Np':[1.80,  1.60, 1.70, 0],
    'Pu':[1.84,  1.64, 1.70, 0],
    'Am':[2.01,  1.81, 1.70, 0],
    'Cm':[2.20,  2.00, 1.70, 0],
    'Bk':[2.20,  2.00, 1.70, 0],
    'Cf':[2.20,  2.00, 1.70, 0],
        }
AtmBlens = {
# atom masses & neutron scattering length (b,b'), sig(incoh) @ 1A
    'H_':    {'Mass':  1.008,'SL':[ -.3741,      0],'SA':       20.6},
    'H_1':   {'Mass':  1.008,'SL':[ -.3742,      0],'SA':       20.6},
    'H_2':   {'Mass':  2.014,'SL':[  .6674,      0],'SA':     .00052},
    'H_3':   {'Mass':  3.016,'SL':[  .4792,      0],'SA':        0.0},
    'D_':    {'Mass':  2.014,'SL':[  .6674,      0],'SA':     .00052},
    'D_2':   {'Mass':  2.014,'SL':[  .6674,      0],'SA':     .00052},
    'T_':    {'Mass':  3.016,'SL':[  .4792,      0],'SA':        0.0},
    'T_3':   {'Mass':  3.016,'SL':[  .4792,      0],'SA':        0.0},
    'He_':   {'Mass':  4.003,'SL':[   .326,      0],'SA':      .0075},
    'He_3':  {'Mass':  3.016,'SL':[   .574, -.1483],'SA':      5333.},
    'He_4':  {'Mass':  4.003,'SL':[   .326,      0],'SA':        0.0},
    'Li_':   {'Mass':  6.941,'SL':[  -.190,      0],'SA':       70.5},
    'Li_6':  {'Mass':  6.015,'SL':[   .200, -.0261],'SA':       940.},
    'Li_7':  {'Mass':  7.016,'SL':[  -.222,      0],'SA':      .0454},
    'Be_':   {'Mass':  9.012,'SL':[   .779,      0],'SA':      .0076},
    'B_':    {'Mass':  10.81,'SL':[    .53, -.0213],'SA':       767.},
    'B_10':  {'Mass': 10.013,'SL':[  -0.02, -.1066],'SA':      3835.},
    'B_11':  {'Mass': 11.009,'SL':[   .665,      0],'SA':      .0055},
    'C_':    {'Mass': 12.011,'SL':[  .6648,      0],'SA':      .0035},
    'C_12':  {'Mass': 12.000,'SL':[  .6654,      0],'SA':     .00353},
    'C_13':  {'Mass': 13.003,'SL':[   .619,      0],'SA':     .00137},
    'N_':    {'Mass': 14.007,'SL':[   .936,      0],'SA':       1.90},
    'N_14':  {'Mass': 14.003,'SL':[   .937,      0],'SA':       1.91},
    'N_15':  {'Mass': 15.000,'SL':[   .644,      0],'SA':     .00002},
    'O_':    {'Mass': 15.999,'SL':[  .5805,      0],'SA':     .00019},
    'O_16':  {'Mass': 15.995,'SL':[  .5805,      0],'SA':     .00010},
    'O_17':  {'Mass': 16.999,'SL':[    .56,      0],'SA':       .236},
    'O_18':  {'Mass': 17.999,'SL':[   .584,      0],'SA':     .00016},
    'F_':    {'Mass': 18.998,'SL':[  .5654,      0],'SA':      .0096},
    'F_19':  {'Mass': 18.998,'SL':[  .5654,      0],'SA':      .0096},
    'Ne_':   {'Mass': 20.179,'SL':[    .46,      0],'SA':       .039},
    'Ne_20': {'Mass': 19.992,'SL':[  .4631,      0],'SA':       .036},
    'Ne_21': {'Mass': 20.994,'SL':[   .666,      0],'SA':        .67},
    'Ne_22': {'Mass': 21.991,'SL':[   .387,      0],'SA':       .046},
    'Na_':   {'Mass': 22.990,'SL':[   .363,      0],'SA':       .530},
    'Na_23': {'Mass': 22.990,'SL':[   .363,      0],'SA':       .530},
    'Mg_':   {'Mass': 24.305,'SL':[  .5375,      0],'SA':       .063},
    'Mg_24': {'Mass': 23.985,'SL':[   .549,      0],'SA':       .050},
    'Mg_25': {'Mass': 24.986,'SL':[   .362,      0],'SA':        .19},
    'Mg_26': {'Mass': 25.983,'SL':[   .489,      0],'SA':      .0382},
    'Al_':   {'Mass': 26.982,'SL':[  .3449,      0],'SA':       .231},
    'Al_27': {'Mass': 26.982,'SL':[  .3449,      0],'SA':       .231},
    'Si_':   {'Mass': 28.086,'SL':[ .41507,      0],'SA':       .171},
    'Si_28': {'Mass': 27.977,'SL':[  .4107,      0],'SA':       .177},
    'Si_29': {'Mass': 28.976,'SL':[    .47,      0],'SA':       .101},
    'Si_30': {'Mass': 29.974,'SL':[   .458,      0],'SA':       .107},
    'P_':    {'Mass': 30.974,'SL':[   .513,      0],'SA':       .172},
    'P_31':  {'Mass': 30.974,'SL':[   .513,      0],'SA':       .172},
    'S_':    {'Mass':  32.06,'SL':[  .2847,      0],'SA':        .53},
    'S_32':  {'Mass': 31.972,'SL':[  .2804,      0],'SA':        .54},
    'S_33':  {'Mass': 32.971,'SL':[   .474,      0],'SA':        .54},
    'S_34':  {'Mass': 33.968,'SL':[   .348,      0],'SA':       .227},
    'Cl_':   {'Mass': 35.453,'SL':[  .9579,      0],'SA':       33.5},
    'Cl_35': {'Mass': 34.969,'SL':[   1.17,      0],'SA':       44.1},
    'Cl_37': {'Mass': 36.966,'SL':[   .308,      0],'SA':       .433},
    'Ar_':   {'Mass': 39.948,'SL':[  .1909,      0],'SA':       .675},
    'Ar_36': {'Mass': 35.968,'SL':[   2.49,      0],'SA':        5.2},
    'Ar_40': {'Mass': 39.962,'SL':[    .17,      0],'SA':       .660},
    'K_':    {'Mass': 39.098,'SL':[   .367,      0],'SA':        2.1},
    'K_39':  {'Mass': 38.964,'SL':[   .379,      0],'SA':        2.1},
    'K_41':  {'Mass': 40.962,'SL':[   .269,      0],'SA':       1.46},
    'Ca_':   {'Mass':  40.08,'SL':[    .47,      0],'SA':        .43},
    'Ca_40': {'Mass': 39.962,'SL':[   .478,      0],'SA':        .41},
    'Ca_44': {'Mass': 43.955,'SL':[   .142,      0],'SA':        .88},
    'Sc_':   {'Mass': 44.956,'SL':[   1.21,      0],'SA':       27.5},
    'Sc_45': {'Mass': 44.956,'SL':[   1.21,      0],'SA':       27.5},
    'Ti_':   {'Mass':  47.90,'SL':[ -.3370,      0],'SA':       6.09},
    'Ti_46': {'Mass': 45.953,'SL':[   .472,      0],'SA':        .59},
    'Ti_47': {'Mass': 46.952,'SL':[   .353,      0],'SA':        1.7},
    'Ti_48': {'Mass': 47.948,'SL':[  -.586,      0],'SA':       7.84},
    'Ti_49': {'Mass': 48.948,'SL':[   .098,      0],'SA':        2.2},
    'Ti_50': {'Mass': 49.945,'SL':[   .588,      0],'SA':       .179},
    'V_':    {'Mass': 50.942,'SL':[ -.0443,      0],'SA':       5.08},
    'V_51':  {'Mass': 50.944,'SL':[ -.0402,      0],'SA':        4.9},
    'Cr_':   {'Mass': 51.996,'SL':[  .3635,      0],'SA':       3.05},
    'Cr_50': {'Mass': 49.946,'SL':[   -.45,      0],'SA':       15.8},
    'Cr_52': {'Mass': 51.941,'SL':[  .4914,      0],'SA':        .76},
    'Cr_53': {'Mass': 52.941,'SL':[   -.42,      0],'SA':       18.1},
    'Cr_54': {'Mass': 53.939,'SL':[   .455,      0],'SA':        .36},
    'Mn_':   {'Mass': 54.938,'SL':[  -.375,      0],'SA':       13.3},
    'Mn_55': {'Mass': 54.938,'SL':[  -.375,      0],'SA':       13.3},
    'Fe_':   {'Mass': 55.847,'SL':[   .945,      0],'SA':       2.56},
    'Fe_54': {'Mass': 53.940,'SL':[    .42,      0],'SA':       2.25},
    'Fe_56': {'Mass': 55.935,'SL':[   1.01,      0],'SA':       2.59},
    'Fe_57': {'Mass': 56.935,'SL':[    .23,      0],'SA':       2.48},
    'Fe_58': {'Mass': 57.933,'SL':[    1.5,      0],'SA':       1.28},
    'Co_':   {'Mass': 58.933,'SL':[   .249,      0],'SA':      37.18},
    'Co_59': {'Mass': 58.933,'SL':[   .249,      0],'SA':      37.18},
    'Ni_':   {'Mass':  58.70,'SL':[   1.03,      0],'SA':       4.49},
    'Ni_58': {'Mass': 57.935,'SL':[   1.44,      0],'SA':        4.6},
    'Ni_60': {'Mass': 58.931,'SL':[    .28,      0],'SA':        2.9},
    'Ni_61': {'Mass': 60.931,'SL':[    .76,      0],'SA':        2.5},
    'Ni_62': {'Mass': 61.928,'SL':[   -.87,      0],'SA':       14.5},
    'Ni_64': {'Mass': 63.928,'SL':[  -.037,      0],'SA':       1.52},
    'Cu_':   {'Mass': 63.546,'SL':[  .7718,      0],'SA':       3.78},
    'Cu_63': {'Mass': 62.930,'SL':[  .6477,      0],'SA':       4.50},
    'Cu_65': {'Mass': 64.928,'SL':[  1.020,      0],'SA':       2.17},
    'Zn_':   {'Mass':  65.38,'SL':[   .568,      0],'SA':       1.11},
    'Zn_64': {'Mass': 63.929,'SL':[   .523,      0],'SA':        .93},
    'Zn_66': {'Mass': 65.926,'SL':[   .598,      0],'SA':        .62},
    'Zn_67': {'Mass': 65.927,'SL':[   .758,      0],'SA':        6.8},
    'Zn_68': {'Mass': 67.925,'SL':[   .604,      0],'SA':        1.1},
    'Ga_':   {'Mass':  69.72,'SL':[  .7288,      0],'SA':       2.75},
    'Ge_':   {'Mass':  72.59,'SL':[  .8185,      0],'SA':       2.20},
    'As_':   {'Mass': 74.922,'SL':[   .658,      0],'SA':        4.5},
    'As_75': {'Mass': 74.922,'SL':[   .658,      0],'SA':        4.5},
    'Se_':   {'Mass':  78.96,'SL':[   .797,      0],'SA':       11.7},
    'Br_':   {'Mass': 79.904,'SL':[   .679,      0],'SA':        6.9},
    'Br_79': {'Mass': 78.918,'SL':[   .679,      0],'SA':       11.0},
    'Br_81': {'Mass': 80.916,'SL':[   .678,      0],'SA':        2.7},
    'Kr_':   {'Mass':  83.80,'SL':[   .781,      0],'SA':       25.0},
    'Rb_':   {'Mass': 85.468,'SL':[   .708,      0],'SA':        .38},
    'Sr_':   {'Mass':  87.62,'SL':[   .702,      0],'SA':       1.28},
    'Y_':    {'Mass': 88.907,'SL':[   .775,      0],'SA':       1.28},
    'Y_89 ': {'Mass': 88.907,'SL':[   .775,      0],'SA':       1.28},
    'Zr_':   {'Mass':  91.22,'SL':[   .716,      0],'SA':       .185},
    'Nb_':   {'Mass': 92.906,'SL':[  .7054,      0],'SA':       1.15},
    'Nb_93': {'Mass': 92.906,'SL':[  .7054,      0],'SA':       1.15},
    'Mo_':   {'Mass':  95.94,'SL':[  .6715,      0],'SA':       2.48},
    'Tc_98': {'Mass':   98.0,'SL':[    .68,      0],'SA':       20.0},
    'Ru_':   {'Mass': 101.07,'SL':[   .702,      0],'SA':       2.56},
    'Rh_':   {'Mass':102.906,'SL':[      0,      0],
        'BW-LS':[   0.59,      78.92, 6155.6, 1257., 78.0,0,0,0,0]},     #BW
    'Rh_103':{'Mass':102.906,'SL':[      0,      0],
        'BW-LS':[   0.59,      78.92, 6155.6, 1257., 78.0,0,0,0,0]},     #BW
    'Pd_':   {'Mass':  106.4,'SL':[   .591,      0],'SA':         6.9},
    'Ag_':   {'Mass':107.868,'SL':[  .5922,      0],'SA':        63.3},
    'Cd_':   {'Mass': 112.41,'SL':[      0,      0],
        'BW-LS':[    .483,      -.070, 2520.,0,0,0,0,0,0]},
    'Cd_112':{'Mass':111.903,'SL':[   .634,      0],'SA':         2.2},
    'Cd_113':{'Mass':112.904,'SL':[      0,      0],
        'BW-LS':[  0.625,      267.0, 15218., 177.6, 57.0,0,0,0,0]},     #BW
    'Cd_114':{'Mass':113.903,'SL':[   .748,      0],'SA':         .34},
    'Cd_116':{'Mass':115.905,'SL':[   .626,      0],'SA':        .075},
#    'In_':   {'Mass': 114.82,'SL':[      0,      0],
#        'BW-LS':[  .4065,     -.0539,  194.,0,0,0,0,0,0]},
    'In_':   {'Mass': 114.82,'SL':[  .4065,      0],'SA':       193.8},
    'In_113':{'Mass':112.904,'SL':[   .539,      0],'SA':         12.},
#    'In_115':{'Mass':114.904,'SL':[      0,      0],
#        'BW-LS':[  0.400,      0.354, 10.793, 97.30, 30.5,0,0,0,0]},     #BW
    'In_115':{'Mass':114.904,'SL':[   .400,      0],'SA':        202.},
    'Sn_':   {'Mass': 118.69,'SL':[  .6225,      0],'SA':        .626},
    'Sb_':   {'Mass': 121.75,'SL':[   .557,      0],'SA':        4.91},
    'Te_':   {'Mass': 127.60,'SL':[   .568,      0],'SA':         4.7},
    'I_':    {'Mass':126.904,'SL':[   .528,      0],'SA':        6.15},
    'I_127': {'Mass':126.904,'SL':[   .528,      0],'SA':        6.15},
    'Xe_':   {'Mass': 131.30,'SL':[   .469,      0],'SA':        23.9},
    'Cs_':   {'Mass':132.905,'SL':[   .542,      0],'SA':        29.0},
    'Cs_133':{'Mass':132.905,'SL':[   .542,      0],'SA':        29.0},
    'Ba_':   {'Mass': 137.33,'SL':[   .507,      0],'SA':         1.1},
    'La_':   {'Mass':138.906,'SL':[   .824,      0],'SA':        8.97},
    'Ce_':   {'Mass': 140.12,'SL':[   .484,      0],'SA':         .63},
    'Pr_':   {'Mass':140.908,'SL':[   .458,      0],'SA':        11.5},
    'Pr_141':{'Mass':140.908,'SL':[   .458,      0],'SA':        11.5},
    'Nd_':   {'Mass': 144.24,'SL':[   .769,      0],'SA':        50.5},
    'Pm_147':{'Mass':   147.,'SL':[   1.26,      0],'SA':       168.4},
    'Sm_':   {'Mass':  150.4,'SL':[      0,      0],
        'BW-LS':[  0.418,      29.72, 1009.4, 97.23, 32.91,0,0,0,0]},     #LS
    'Sm_149':{'Mass':148.917,'SL':[      0,      0],
        'BW-LS':[  0.525,      213.7, 7256.4, 97.23, 32.90,0,0,0,0]},     #LS
    'Sm_152':{'Mass':151.920,'SL':[   -.50,      0],'SA':        206.},
    'Sm_154':{'Mass':153.922,'SL':[    .80,      0],'SA':         8.4},
    'Eu_':   {'Mass': 151.96,'SL':[      0,      0],
        'BW-LS':[  0.748,      61.91, 2742.6, 460.2, 43.74, .1350, 321.1, 0.229, -36.4]},  #LS
    'Eu_151':{'Mass':150.920,'SL':[      0,      0],
        'BW-LS':[  0.704,      129.4, 5658.4, 460.2, 43.64, .1342, 321.2, 0.191, -28.9]},  #LS
    'Eu_153':{'Mass':152.921,'SL':[   .822,      0],'SA':        312.},
    'Gd_':   {'Mass': 157.25,'SL':[      0,      0],
        'BW-LS':[  0.685,      72.74, 3967.4, 30.58, 53.13,0,0,0,0]},     #LS
    'Gd_155':{'Mass':154.923,'SL':[      0,      0],
        'BW-LS':[  0.683,      88.78, 4684.9, 28.11, 52.61,0,0,0,0]},     #LS
    'Gd_160':{'Mass':159.927,'SL':[   .915,      0],'SA':         .77},
    'Tb_':  {'Mass':158.925,'SL':[   .734,      0],'SA':        23.4},
    'Tb_159':{'Mass':158.925,'SL':[   .734,      0],'SA':        23.4},
    'Dy_':   {'Mass': 162.50,'SL':[   1.69,      0],'SA':        994.},
    'Ho_':   {'Mass':164.930,'SL':[   .844,      0],'SA':        64.7},
    'Ho_165':{'Mass':164.930,'SL':[   .844,      0],'SA':        64.7},
    'Er_':   {'Mass': 167.26,'SL':[      0,      0],
        'BW-LS':[  0.853,      11.97,  533.9, 460.0, 44.17, .6448, 584.3,0,0]}, #LS
    'Er_167':{'Mass':166.932,'SL':[      0,      0],
        'BW-LS':[  0.564,      52.22, 2298.0, 460.0, 44.02, .6451, 584.2,0,0]}, #LS
    'Tm_':   {'Mass':168.934,'SL':[   .707,      0],'SA':        100.},
    'Tm_169':{'Mass':168.934,'SL':[   .707,      0],'SA':        100.},
    'Yb_':   {'Mass': 173.04,'SL':[      0,      0],
        'BW-LS':[  1.229,      0.560,  34.28, 597.1, 33.08,0,0,0,0]},    #LS
    'Yb_168':{'Mass':167.934,'SL':[      0,      0],
        'BW-LS':[  0.699,      636.9, 21095., 597.1, 33.08,0,0,0,0]},    #LS
    'Lu_':   {'Mass':174.967,'SL':[   .721,      0],'SA':        74.0},
    'Lu_176':{'Mass':175.943,'SL':[      0,      0],
        'BW-LS':[  0.818,      26.17,  850.3, 141.3, 31.51,0,0,0,0]},    #LS
    'Hf_':   {'Mass': 178.49,'SL':[   .777,      0],'SA':       104.1},
    'Ta_':   {'Mass':180.948,'SL':[   .691,      0],'SA':        20.6},
    'Ta_181':{'Mass':180.948,'SL':[   .691,      0],'SA':        20.6},
    'W_':    {'Mass': 183.85,'SL':[  .4755,      0],'SA':        18.3},
    'Re_':   {'Mass':186.207,'SL':[    .92,      0],'SA':        89.7},
    'Os_':   {'Mass':  190.2,'SL':[   1.07,      0],'SA':        16.0},
    'Ir_':   {'Mass': 192.22,'SL':[   1.06,      0],'SA':        425.},
    'Pt_':   {'Mass': 195.09,'SL':[    .96,      0],'SA':        10.3},
    'Au_':   {'Mass':196.967,'SL':[   .790,      0],'SA':       98.65},
    'Au_197':{'Mass':196.967,'SL':[   .790,      0],'SA':       98.65},
    'Hg_':   {'Mass': 200.59,'SL':[  1.266,      0],'SA':       372.3},
    'Tl_':   {'Mass': 204.37,'SL':[  .8776,      0],'SA':        3.43},
    'Pb_':   {'Mass':  207.2,'SL':[  .9401,      0],'SA':        .171},
    'Bi_':   {'Mass':208.980,'SL':[  .8532,      0],'SA':       .0338},
    'Bi_209':{'Mass':208.980,'SL':[  .8532,      0],'SA':       .0338},
    'Po_209':{'Mass':208.982,'SL':[     .8,      0],'SA':         .03},
    'At_210':{'Mass': 210.00,'SL':[     .8,      0],'SA':           0},
    'Rn_222':{'Mass':222.018,'SL':[     .8,      0],'SA':         .72},
    'Fr_223':{'Mass': 223.00,'SL':[     .8,      0],'SA':           0},
    'Ra_226':{'Mass':226.025,'SL':[    1.0,      0],'SA':        12.8},
    'Ac_227':{'Mass':227.028,'SL':[     .8,      0],'SA':        515.},
    'Th_':   {'Mass':232.038,'SL':[  1.031,      0],'SA':        7.37},
    'Th_232':{'Mass':232.038,'SL':[  1.031,      0],'SA':        7.37},
    'Pa_':   {'Mass':231.036,'SL':[   0.91,      0],'SA':       200.6},
    'Pa_231':{'Mass':231.036,'SL':[   0.91,      0],'SA':       200.6},
    'U_':    {'Mass':238.029,'SL':[  .8417,      0],'SA':        7.57},
    'U_235': {'Mass':235.044,'SL':[   1.05,      0],'SA':       680.9},
    'U_238': {'Mass':238.051,'SL':[  .8407,      0],'SA':        2.68},
    'Np_237':{'Mass':237.048,'SL':[  1.055,      0],'SA':       175.9},
    'Pu_239':{'Mass':239.052,'SL':[      0,      0],
        'BW-LS':[  0.896,      22.68, 1157.0, 296.0, 51.00,0,0,0,0]},   #BW
    'Pu_240':{'Mass':240.054,'SL':[      0,      0],
        'BW-LS':[  0.874,      516.0, 8591.3, 1056., 16.65,0,0,0,0]},   #BW
    'Pu_242':{'Mass':242.059,'SL':[    .81,      0],'SA':        18.5},
    'Am_243':{'Mass':243.061,'SL':[    .83,      0],'SA':        75.3},
    'Cm_244':{'Mass':244.063,'SL':[    .95,      0],'SA':        16.2},
    'Bk_247':{'Mass':247.070,'SL':[     .8,      0],'SA':           0},
    'Cf_249':{'Mass':249.075,'SL':[     .8,      0],'SA':           0},
        }
MagFF = {
#neutron magnetic form factor coeff: M for j<0> & N for j<2>
    'Sc':{'M':  [     .2512, 90.030,  .3290, 39.402,   .4235, 14.322,  .0000,   .000, -.0043],
        'N' :  [   10.8172, 51.327, 4.7353, 14.847,   .6071,  4.218,  .0000,   .000,  .0011, 2.00]},
    'Sc+1':{'M':[     .4889, 51.160,  .5203, 14.076,  -.0286,   .179,  .0000,   .000,  .0185],
        'N':[    8.5021, 34.285, 3.2116, 10.994,   .4244,  3.605,  .0000,   .000,  .0009, 2.00]},
    'Sc+3':{'M':[     .5048, 31.404,  .5186, 10.990,  -.0241,  1.183,  .0000,   .000,  .0000],
        'N':[    4.3683, 28.654, 3.7231, 10.823,   .6074,  3.668,  .0000,   .000,  .0014, 2.00]},
    'Ti':{'M':  [     .4657, 33.590,  .5490,  9.879,  -.0291,   .323,  .0000,   .000,  .0123],
        'N':  [    4.3583, 36.056, 3.8230, 11.133,   .6855,  3.469,  .0000,   .000,  .0020, 2.00]},
    'Ti+1':{'M':[     .5093, 36.703,  .5032, 10.371,  -.0263,   .311,  .0000,   .000,  .0116],
        'N':[    6.1567, 27.275, 2.6833,  8.983,   .4070,  3.052,  .0000,   .000,  .0011, 2.00]},
    'Ti+2':{'M':[     .5091, 24.976,  .5162,  8.757,  -.0281,   .916,  .0000,   .000,  .0015],
        'N':[    4.3107, 18.348, 2.0960,  6.797,   .2984,  2.548,  .0000,   .000,  .0007, 2.00]},
    'Ti+3':{'M':[     .3571, 22.841,  .6688,  8.931,  -.0354,   .483,  .0000,   .000,  .0099],
        'N':[    3.3717, 14.444, 1.8258,  5.713,   .2470,  2.265,  .0000,   .000,  .0005, 2.00]},
    'V':{'M':   [     .4086, 28.811,  .6077,  8.544,  -.0295,   .277,  .0000,   .000,  .0123],
        'N':   [    3.7600, 21.831, 2.4026,  7.546,   .4464,  2.663,  .0000,   .000,  .0017, 2.00]},
    'V+1':{'M': [     .4444, 32.648,  .5683,  9.097,  -.2285,   .022,  .0000,   .000,  .2150],
        'N': [    4.7474, 23.323, 2.3609,  7.808,   .4105,  2.706,  .0000,   .000,  .0014, 2.00]},
    'V+2':{'M': [     .4085, 23.853,  .6091,  8.246,  -.1676,   .041,  .0000,   .000,  .1496],
        'N': [    3.4386, 16.530, 1.9638,  6.141,   .2997,  2.267,  .0000,   .000,  .0009, 2.00]},
    'V+3':{'M': [     .3598, 19.336,  .6632,  7.617,  -.3064,   .030,  .0000,   .000,  .2835],
        'N': [    2.3005, 14.682, 2.0364,  6.130,   .4099,  2.382,  .0000,   .000,  .0014, 2.00]},
    'V+4':{'M': [     .3106, 16.816,  .7198,  7.049,  -.0521,   .302,  .0000,   .000,  .0221],
        'N': [    1.8377, 12.267, 1.8247,  5.458,   .3979,  2.248,  .0000,   .000,  .0012, 2.00]},
    'Cr':{'M':  [     .1135, 45.199,  .3481, 19.493,   .5477,  7.354,  .0000,   .000, -.0092],
        'N':  [    3.4085, 20.127, 2.1006,  6.802,   .4266,  2.394,  .0000,   .000,  .0019, 2.00]},
    'Cr+1':{'M':[    -.0977,   .047,  .4544, 26.005,   .5579,  7.489,  .0000,   .000,  .0831],
        'N':[    3.7768, 20.346, 2.1028,  6.893,   .4010,  2.411,  .0000,   .000,  .0017, 2.00]},
    'Cr+2':{'M':[    1.2024,  -.005,  .4158, 20.548,   .6032,  6.956,  .0000,   .000,-1.2218],
        'N':[    2.6422, 16.060, 1.9198,  6.235,   .4446,  2.372,  .0000,   .000,  .0020, 2.00]},
    'Cr+3':{'M':[    -.3094,   .027,  .3680, 17.035,   .6559,  6.524,  .0000,   .000,  .2856],
        'N':[    1.6262, 15.066, 2.0618,  6.284,   .5281,  2.368,  .0000,   .000,  .0023, 2.00]},
    'Cr+4':{'M':[    -.2320,   .043,  .3101, 14.952,   .7182,  6.173,  .0000,   .000,  .2042],
        'N':[    1.0293, 13.950, 1.9933,  6.059,   .5974,  2.346,  .0000,   .000,  .0027, 2.00]},
    'Mn':{'M':  [     .2438, 24.963,  .1472, 15.673,   .6189,  6.540,  .0000,   .000, -.0105],
        'N':  [    2.6681, 16.060, 1.7561,  5.640,   .3675,  2.049,  .0000,   .000,  .0017, 2.00]},
    'Mn+1':{'M':[    -.0138,   .421,  .4231, 24.668,   .5905,  6.655,  .0000,   .000, -.0010],
        'N':[    3.2953, 18.695, 1.8792,  6.240,   .3927,  2.201,  .0000,   .000,  .0022, 2.00]},
    'Mn+2':{'M':[     .4220, 17.684,  .5948,  6.005,   .0043,  -.609,  .0000,   .000, -.0219],
        'N':[    2.0515, 15.556, 1.8841,  6.063,   .4787,  2.232,  .0000,   .000,  .0027, 2.00]},
    'Mn+3':{'M':[     .4198, 14.283,  .6054,  5.469,   .9241,  -.009,  .0000,   .000, -.9498],
        'N':[    1.2427, 14.997, 1.9567,  6.118,   .5732,  2.258,  .0000,   .000,  .0031, 2.00]},
    'Mn+4':{'M':[     .3760, 12.566,  .6602,  5.133,  -.0372,   .563,  .0000,   .000,  .0011],
        'N':[     .7879, 13.886, 1.8717,  5.743,   .5981,  2.182,  .0000,   .000,  .0034, 2.00]},
    'Fe':{'M':  [     .0706, 35.008,  .3589, 15.358,   .5819,  5.561,  .0000,   .000, -.0114],
        'N':  [    1.9405, 18.473, 1.9566,  6.323,   .5166,  2.161,  .0000,   .000,  .0036, 2.00]},
    'Fe+1':{'M':[     .1251, 34.963,  .3629, 15.514,   .5223,  5.591,  .0000,   .000, -.0105],
        'N':[    2.6290, 18.660, 1.8704,  6.331,   .4690,  2.163,  .0000,   .000,  .0031, 2.00]},
    'Fe+2':{'M':[     .0263, 34.960,  .3668, 15.943,   .6188,  5.594,  .0000,   .000, -.0119],
        'N':[    1.6490, 16.559, 1.9064,  6.133,   .5206,  2.137,  .0000,   .000,  .0035, 2.00]},
    'Fe+3':{'M':[     .3972, 13.244,  .6295,  4.903,  -.0314,   .350,  .0000,   .000,  .0044],
        'N':[    1.3602, 11.998, 1.5188,  5.003,   .4705,  1.991,  .0000,   .000,  .0038, 2.00]},
    'Fe+4':{'M':[     .3782, 11.380,  .6556,  4.592,  -.0346,   .483,  .0000,   .000,  .0005],
        'N':[    1.5582,  8.275, 1.1863,  3.279,   .1366,  1.107,  .0000,   .000, -.0022, 2.00]},
    'Co':{'M':  [     .4139, 16.162,  .6013,  4.780,  -.1518,   .021,  .0000,   .000,  .1345],
        'N':  [    1.9678, 14.170, 1.4911,  4.948,   .3844,  1.797,  .0000,   .000,  .0027, 2.00]},
    'Co+1':{'M':[     .0990, 33.125,  .3645, 15.177,   .5470,  5.008,  .0000,   .000, -.0109],
        'N':[    2.4097, 16.161, 1.5780,  5.460,   .4095,  1.914,  .0000,   .000,  .0031, 2.00]},
    'Co+2':{'M':[     .4332, 14.355,  .5857,  4.608,  -.0382,   .134,  .0000,   .000,  .0179],
        'N':[    1.9049, 11.644, 1.3159,  4.357,   .3146,  1.645,  .0000,   .000,  .0017, 2.00]},
    'Co+3':{'M':[     .3902, 12.508,  .6324,  4.457,  -.1500,   .034,  .0000,   .000,  .1272],
        'N':[    1.7058,  8.859, 1.1409,  3.309,   .1474,  1.090,  .0000,   .000, -.0025, 2.00]},
    'Co+4':{'M':[     .3515, 10.778,  .6778,  4.234,  -.0389,   .241,  .0000,   .000,  .0098],
        'N':[    1.3110,  8.025, 1.1551,  3.179,   .1608,  1.130,  .0000,   .000, -.0011, 2.00]},
    'Ni':{'M':  [    -.0172, 35.739,  .3174, 14.269,   .7136,  4.566,  .0000,   .000, -.0143],
        'N':  [    1.0302, 12.252, 1.4669,  4.745,   .4521,  1.744,  .0000,   .000,  .0036, 2.00]},
    'Ni+1':{'M':[     .0705, 35.856,  .3984, 13.804,   .5427,  4.397,  .0000,   .000, -.0118],
        'N':[    2.1040, 14.866, 1.4302,  5.071,   .4031,  1.778,  .0000,   .000,  .0034, 2.00]},
    'Ni+2':{'M':[     .0163, 35.883,  .3916, 13.223,   .6052,  4.339,  .0000,   .000, -.0133],
        'N':[    1.7080, 11.016, 1.2147,  4.103,   .3150,  1.533,  .0000,   .000,  .0018, 2.00]},
    'Ni+3':{'M':[     .0012, 35.000,  .3468, 11.987,   .6667,  4.252,  .0000,   .000, -.0148],
        'N':[    1.4683,  8.671,  .1794,  1.106,  1.1068,  3.257,  .0000,   .000, -.0023, 2.00]},
    'Ni+4':{'M':[    -.0090, 35.861,  .2776, 11.790,   .7474,  4.201,  .0000,   .000, -.0163],
        'N':[    1.1612,  7.700, 1.0027,  3.263,   .2719,  1.378,  .0000,   .000,  .0025, 2.00]},
    'Cu':{'M':  [     .0909, 34.984,  .4088, 11.443,   .5128,  3.825,  .0000,   .000, -.0124],
        'N':  [    1.9182, 14.490, 1.3329,  4.730,   .3842,  1.639,  .0000,   .000,  .0035, 2.00]},
    'Cu+1':{'M':[     .0749, 34.966,  .4147, 11.764,   .5238,  3.850,  .0000,   .000, -.0127],
        'N':[    1.8814, 13.433, 1.2809,  4.545,   .3646,  1.602,  .0000,   .000,  .0033, 2.00]},
    'Cu+2':{'M':[     .0232, 34.969,  .4023, 11.564,   .5882,  3.843,  .0000,   .000, -.0137],
        'N':[    1.5189, 10.478, 1.1512,  3.813,   .2918,  1.398,  .0000,   .000,  .0017, 2.00]},
    'Cu+3':{'M':[     .0031, 34.907,  .3582, 10.914,   .6531,  3.828,  .0000,   .000, -.0147],
        'N':[    1.2797,  8.450, 1.0315,  3.280,   .2401,  1.250,  .0000,   .000,  .0015, 2.00]},
    'Cu+4':{'M':[    -.0132, 30.682,  .2801, 11.163,   .7490,  3.817,  .0000,   .000, -.0165],
        'N':[     .9568,  7.448,  .9099,  3.396,   .3729,  1.494,  .0000,   .000,  .0049, 2.00]},
    'Y':{'M':   [     .5915, 67.608, 1.5123, 17.900, -1.1130, 14.136,  .0000,   .000,  .0080],
        'N':   [   14.4084, 44.658, 5.1045, 14.904,  -.0535,  3.319,  .0000,   .000,  .0028, 2.00]},
    'Zr':{'M':  [     .4106, 59.996, 1.0543, 18.648,  -.4751, 10.540,  .0000,   .000,  .0106],
        'N':  [   10.1378, 35.337, 4.7734, 12.545,  -.0489,  2.672,  .0000,   .000,  .0036, 2.00]},
    'Zr+1':{'M':[     .4532, 59.595,  .7834, 21.436,  -.2451,  9.036,  .0000,   .000,  .0098],
        'N':[   11.8722, 34.920, 4.0502, 12.127,  -.0632,  2.828,  .0000,   .000,  .0034, 2.00]},
    'Nb':{'M':  [     .3946, 49.230, 1.3197, 14.822,  -.7269,  9.616,  .0000,   .000,  .0129],
        'N':  [    7.4796, 33.179, 5.0884, 11.571,  -.0281,  1.564,  .0000,   .000,  .0047, 2.00]},
    'Nb+1':{'M':[     .4572, 49.918, 1.0274, 15.726,  0.4962,  9.157,  .0000,   .000,  .0118],
        'N':[    8.7735, 33.285, 4.6556, 11.605,  -.0268,  1.539,  .0000,   .000,  .0044, 2.00]},
    'Mo':{'M':  [     .1806, 49.057, 1.2306, 14.786,  -.4268,  6.987,  .0000,   .000,  .0171],
        'N':  [    5.1180, 23.422, 4.1809,  9.208,  -.0505,  1.743,  .0000,   .000,  .0053, 2.00]},
    'Mo+1':{'M':[     .3500, 48.035, 1.0305, 15.060,  0.3929,  7.479,  .0000,   .000,  .0139],
        'N':[    7.2367, 28.128, 4.0705,  9.923,  -.0317,  1.455,  .0000,   .000,  .0049, 2.00]},
    'Tc':{'M':  [     .1298, 49.661, 1.1656, 14.131,  -.3134,  5.513,  .0000,   .000,  .0195],
        'N':  [    4.2441, 21.397, 3.9439,  8.375,  -.0371,  1.187,  .0000,   .000,  .0066, 2.00]},
    'Tc+1':{'M':[     .2674, 48.957,  .9569, 15.141,  -.2387,  5.458,  .0000,   .000,  .0160],
        'N':[    6.4056, 24.824, 3.5400,  8.611,  -.0366,  1.485,  .0000,   .000,  .0044, 2.00]},
    'Ru':{'M':  [     .1069, 49.424, 1.1912, 12.742,  -.3176,  4.912,  .0000,   .000,  .0213],
        'N':  [    3.7445, 18.613, 3.4749,  7.420,  -.0363,  1.007,  .0000,   .000,  .0073, 2.00]},
    'Ru+1':{'M':[     .4410, 33.309, 1.4775,  9.553,  -.9361,  6.722,  .0000,   .000,  .0176],
        'N':[    5.2826, 23.683, 3.5813,  8.152,  -.0257,   .426,  .0000,   .000,  .0131, 2.00]},
    'Rh':{'M':  [     .0976, 49.882, 1.1601, 11.831,  -.2789,  4.127,  .0000,   .000,  .0234],
        'N':  [    3.3651, 17.344, 3.2121,  6.804,  -.0350,   .503,  .0000,   .000,  .0146, 2.00]},
    'Rh+1':{'M':[     .3342, 29.756, 1.2209,  9.438,  -.5755,  5.332,  .0000,   .000,  .0210],
        'N':[    4.0260, 18.950, 3.1663,  7.000,  -.0296,   .486,  .0000,   .000,  .0127, 2.00]},
    'Pd':{'M':  [     .2003, 29.363, 1.1446,  9.599,  -.3689,  4.042,  .0000,   .000,  .0251],
        'N':  [    3.3105, 14.726, 2.6332,  5.862,  -.0437,  1.130,  .0000,   .000,  .0053, 2.00]},
    'Pd+1':{'M':[     .5033, 24.504, 1.9982,  6.908, -1.5240,  5.513,  .0000,   .000,  .0213],
        'N':[    4.2749, 17.900, 2.7021,  6.354,  -.0258,   .700,  .0000,   .000,  .0071, 2.00]},
    #3rd row transition elements - take the 6s0 configs.
    'Hf':{'M':[0.4229, 50.465, 0.7333, 23.865, 0.3798 ,4.051, 0.2252, 2.497, 0.0018],
        'N':[9.6670, 33.435, 5.2429, 13.529, 0.5533, 1.402, 0.4934, 1.254, 0.0033, 2.00]},      #really Hf2+
    'Hf+2':{'M':[0.4229, 50.465, 0.7333, 23.865, 0.3798 ,4.051, 0.2252, 2.497, 0.0018],
        'N':[9.6670, 33.435, 5.2429, 13.529, 0.5533, 1.402, 0.4934, 1.254, 0.0033, 2.00]},
    'Hf+3':{'M':[0.3555, 40.954, 0.8483, 21.726, 0.4116, 4.305, 0.2101, 2.349, 0.0023],
        'N':[7.5646, 27.367, 5.0743, 12.402, 0.4133, 1.742, 0.3163, 1.437, 0.0012, 2.00]},
    'Ta':{'M':[0.3976, 45.095, 0.7746, 21.028, 0.6098, 3.471, 0.4395, 2.570, 0.0020],
        'N':[8.1746, 29.871, 4.9405, 12.188, 1.1294, 1.254, 1.0658, 1.181, 0.0046, 2.00]},      #really Ta+2
    'Ta+2':{'M':[0.3976, 45.095, 0.7746, 21.028, 0.6098, 3.471, 0.4395, 2.570, 0.0020],
        'N':[8.1746, 29.871, 4.9405, 12.188, 1.1294, 1.254, 1.0658, 1.181, 0.0046, 2.00]},
    'Ta+3':{'M':[0.3611, 36.921, 0.8579, 19.195, 0.4945, 3.857, 0.2781, 2.303, 0.0026],
        'N':[6.5998, 25.026, 4.7913, 11.303, 0.7801, 1.529, 0.6820, 1.382, 0.0019, 2.00]},
    'Ta+4':{'M':[0.3065, 31.817, 0.9611, 17.749, 0.5463, 3.979, 0.2816, 2.232, 0.0030],
        'N':[5.5048, 21.620, 4.6734, 10.556, 0.2181, 2.100, 0.0810, 1.049, 0.0016, 2.00]},
    'W':{'M':[0.3990, 73.810, 0.7138, 22.815, 2.0436, 2.710, 1.9319, 2.559, 0.0023],
        'N':[8.4216, 64.652, 7.6983, 23.124, 2.3605, 8.583, 0.3876, 4.620, 0.0033, 2.00]},
    'W+1':{'M':[0.4077, 51.367, 0.7436, 20.256, 9.8283, 2.780, 9.6788, 2.740, 0.0021],
        'N':[5.6854, 44.743, 6.8903, 18.753, 7.8523, 6.308, 6.3447, 5.842, 0.0038, 2.00]},
    'W+2':{'M':[0.3876, 40.340, 0.8008, 18.621, 1.3911, 2.995, 1.2048, 2.627, 0.0023],
        'N':[7.0301, 26.990, 4.6604, 11.083, 0.8220, 1.145, 0.7581, 1.042, 0.0068, 2.00]},
    'W+3':{'M':[0.3610, 33.519, 0.8717, 17.176, 0.6183, 3.445, 0.3883, 2.276, 0.0028],
        'N':[5.8036, 22.969, 4.5243, 10.361, 0.7897, 1.388, 0.6927, 1.248, 0.0031, 2.00]},
    'W+4':{'M':[0.3221, 29.047, 0.9574, 15.979, 0.6287, 3.597, 0.3525, 2.174, 0.0033],
        'N':[4.9006, 20.117, 4.4360, 9.765, 2.0009, 1.497, 1.8714, 1.431, 0.0014, 2.00]},
    'W+5':{'M':[0.2725, 25.966, 1.0558, 14.954, 0.6990, 3.643, 0.3745, 2.145, 0.0037],
        'N':[4.1973, 17.967, 4.3791, 9.255, 1.8830, 1.603, 1.7205, 1.518, 0.000, 2.00]},
    'Re':{'M':[0.3807, 63.042, 0.7497, 19.967, 6.5300, 2.501, 6.4013, 2.451, 0.0028],
        'N':[6.7574, 55.529, 6.7931, 20.125, 2.3113, 7.529, 0.5004, 4.412, 0.0037, 2.00]},
    'Re+1':{'M':[0.3944, 45.427, 0.7742, 17.948, 3.1692, 2.653, 3.0028, 2.521, 0.0026],
        'N':[4.7231, 40.130, 6.2139, 16.902, 8.8240, 5.766, 7.3608, 5.396, 0.0042, 2.00]},
    'Re+2':{'M':[0.3825, 36.336, 0.8218, 16.636, 8.7220, 2.657, 8.5201, 2.601, 0.0026],
        'N':[3.0708, 33.896, 5.9941, 15.817, 6.7816, 5.768, 5.2056, 5.231, 0.0047, 2.00]},
    'Re+3':{'M':[0.3585, 30.671, 0.8863, 15.527, 0.8682, 3.047, 0.6263, 2.280, 0.0030],
        'N':[1.7870, 31.248, 5.9068, 15.160, 17.4262, 5.551, 15.6856, 5.339, 0.0053, 2.00]},
    'Re+4':{'M':[0.2974, 27.372, 0.9826, 14.807, 1.8869, 2.840, 1.6100, 2.476, 0.0031],
        'N':[4.4033, 18.733, 4.1971, 9.047, 2.3106, 1.352, 2.1866, 1.299, 0.0025, 2.00]},
    'Re+5':{'M':[0.3143, 23.522, 1.0276, 13.505, 0.7438, 3.393, 0.4059, 2.030, 0.0041],
        'N':[3.8162, 16.843, 4.1400, 8.605, 4.0882, 1.443, 3.9333, 1.408, 0.0012, 2.00]},
    'Re+6':{'M':[0.2146, 22.496, 1.1616, 13.064, 1.0455, 3.162, 0.6734, 2.196, 0.0041],
        'N':[3.3349, 15.341, 4.1015, 8.213, 2.3511, 1.531, 2.1651, 1.459, 0.0003, 2.00]},
    'Os':{'M':[0.3676, 54.835, 0.7793, 17.716, 2.0669, 2.418, 1.9224, 2.247, 0.0034],
        'N':[5.5418, 48.893, 6.0803, 17.984, 2.2542, 6.853, 0.5285, 4.095, 0.0040, 2.00]},
    'Os+1':{'M':[0.3837, 40.665, 0.8006, 16.096, 3.5305, 2.487, 3.3488, 2.366, 0.0030],
        'N':[4.0236, 36.272, 5.6349, 15.338, 8.6568, 5.318, 7.2505, 4.980, 0.0044, 2.00]},
    'Os+2':{'M':[0.3786, 33.005, 0.8412, 14.990, 7.0632, 2.503, 6.8462, 2.433, 0.0030],
        'N':[2.7289, 30.900, 5.4357, 14.396, 18.6546, 5.151, 17.1708, 4.993, 0.0050, 2.00]},
    'Os+3':{'M':[0.3557, 28.222, 0.9002, 14.140, 2.5972, 2.601, 2.3444, 2.376, 0.0032],
        'N':[4.5913, 19.692, 4.0615, 8.862, 0.9950, 1.086, 0.9081, 0.986, 0.0077, 2.00]},
    'Os+4':{'M':[0.3337, 24.723, 0.9655, 13.288, 0.9653, 2.906, 0.6698, 2.117, 0.0037],
        'N':[3.9724, 17.514, 3.9817, 8.421, 0.7719, 1.264, 0.6552, 1.107, 0.0042, 2.00]},
    'Os+5':{'M':[0.3055, 22.152, 1.0395, 12.529, 0.9158, 3.016, 0.5750, 2.032, 0.0042],
        'N':[3.4764, 15.826, 3.9241, 8.032, 0.6854, 1.397, 0.5394, 1.184, 0.0024, 2.00]},
    'Os+6':{'M':[0.2714, 20.218, 1.1211, 11.851, 0.9773, 3.050, 0.5894, 2.005, 0.004],
        'N':[3.0642, 14.479, 3.8847, 7.686, 4.7235, 1.384, 4.5485, 1.352, 0.0012, 2.00]},
    'Os+7':{'M':[0.2101, 19.108, 1.2240, 11.347, 1.2543, 2.933, 0.8250, 2.088, 0.0048],
        'N':[2.7164, 13.366, 3.8554, 7.370, 5.0211, 1.439, 4.8166, 1.405, 0.0004, 2.00]},
    'Ir':{'M':[0.3564, 48.464, 0.8049, 15.923, 2.5258, 2.265, 2.3675, 2.121, 0.0040],
        'N':[4.6102, 43.878, 5.4892, 16.394, 2.1707, 6.423, 0.4761, 3.722, 0.0043, 2.00]},
    'Ir+1':{'M':[0.3744, 36.764, 0.8240, 14.576, 8.8616, 2.303, 8.6664, 2.255, 0.0035],
        'N':[3.4956, 32.991, 5.1369, 13.998, 8.3991, 4.924, 7.0561, 4.612, 0.0047, 2.00]},
    'Ir+2':{'M':[0.3802, 30.032, 0.8550, 13.567, 1.6185, 2.488, 1.3866, 2.162, 0.0035],
        'N':[2.4419, 28.356, 4.9470, 13.222, 4.7478, 5.029, 3.3259, 4.393, 0.0052, 2.00]},
    'Ir+3':{'M':[0.3678, 25.828, 0.9065, 12.788, 0.8587, 2.745, 0.5883, 1.960, 0.0040],
        'N':[1.5883, 25.969, 4.8472, 12.711, 5.6507, 4.949, 4.1190, 4.388, 0.0056, 2.00]},
    'Ir+4':{'M':[0.3969, 22.050, 0.9310, 11.768, 0.7090, 3.017, 0.3857, 1.778, 0.0047],
        'N':[3.5964, 16.439, 3.7872, 7.873, 1.2303, 1.092, 1.1232, 1.003, 0.0068, 2.00]},
    'Ir+5':{'M':[0.3059, 20.759, 1.0432, 11.622, 1.3457, 2.655, 1.0008, 2.066, 0.0043],
        'N':[3.1710, 14.923, 3.7320, 7.529, 1.5400, 1.200, 1.4048, 1.119, 0.0040, 2.00]},
    'Ir+6':{'M':[0.2854, 18.867, 1.1123, 10.982, 1.1403, 2.782, 0.7475, 1.959, 0.004],
        'N':[2.8163, 13.697, 3.6906, 7.219, 5.2957, 1.254, 5.1328, 1.228, 0.0024, 2.00]},
    'Pt':{'M':[0.3664, 33.503, 0.8449, 13.303, 2.6192, 2.227, 2.4116, 2.059, 0.0041],
        'N':[3.0783, 30.198, 4.7065, 12.857, 5.7270, 4.650, 4.4425, 4.222, 0.004, 2.00]},       #really Pt+1
    'Pt+1':{'M':[0.3664, 33.503, 0.8449, 13.303, 2.6192, 2.227, 2.4116, 2.059, 0.0041],
        'N':[3.0783, 30.198, 4.7065, 12.857, 5.7270, 4.650, 4.4425, 4.222, 0.004, 2.00]},
    'Pt+2':{'M':[0.3701, 27.799, 0.8761, 12.480, 3.2959, 2.258, 3.0535, 2.106, 0.004],
        'N':[2.2407, 26.003, 4.5300, 12.118, 6.7189, 4.557, 5.3954, 4.187, 0.0053, 2.00]},
    'Pt+3':{'M':[0.3606, 24.052, 0.9205, 11.833, 4.0612, 2.271, 3.7841, 2.133, 0.0041],
        'N':[1.5391, 23.689, 4.4263, 11.645, 8.5673, 4.482, 7.1591, 4.181, 0.0057, 2.00]},
    'Pt+4':{'M':[0.3429, 21.358, 0.9753, 11.261, 2.2859, 2.357, 1.9719, 2.074, 0.0043],
        'N':[0.9477, 22.775, 4.3337, 11.319, 7.2374, 4.500, 5.7086, 4.106, 0.0062, 2.00]},
    'Pt+5':{'M':[0.3649, 18.762, 1.0102, 10.456, 0.8785, 2.765, 0.5087, 1.737, 0.0053],
        'N':[0.5132, 23.124, 4.1835, 11.009, 7.4312, 4.478, 5.7759, 4.065, 0.0066, 2.00]},
    'Pt+6':{'M':[0.2877, 17.780, 1.1111, 10.263, 1.9416, 2.432, 1.5477, 2.015, 0.0049],
        'N':[2.5907, 12.991, 3.5155, 6.801, 0.7511, 1.206, 0.6014, 1.022, 0.0043, 2.00]},
    'Au':{'M':[0.3475, 28.294, 0.8964, 11.794, 4.5383, 2.086, 4.2988, 1.983, 0.0047],
        'N':[2.3189, 25.954, 4.2267, 11.444, 9.0997, 4.215, 7.8523, 3.981, 0.0052, 2.00]},      #really Au+1
    'Au+1':{'M':[0.3475, 28.294, 0.8964, 11.794, 4.5383, 2.086, 4.2988, 1.983, 0.0047],
        'N':[2.3189, 25.954, 4.2267, 11.444, 9.0997, 4.215, 7.8523, 3.981, 0.0052, 2.00]},
    'Au+2':{'M':[0.3664, 25.714, 0.8919, 11.488, 1.7302, 2.207, 1.4763, 1.908, 0.0046],
        'N':[2.0676, 23.960, 4.1678, 11.164, 13.3827, 4.151, 12.1447, 3.996, 0.0054, 2.00]},
    'Au+3':{'M':[0.3639, 22.330, 0.9313, 10.883, 1.1359, 2.355, 0.8455, 1.809, 0.004],
        'N':[1.4776, 21.784, 4.0585, 10.746, 5.9059, 4.242, 4.5905, 3.842, 0.0058, 2.00]},
    'Au+4':{'M':[0.3472, 19.944, 0.9807, 10.424, 3.8228, 2.169, 3.4995, 2.009, 0.0047],
        'N':[2.9775, 14.632, 3.4472, 6.956, 1.9784, 0.794, 1.8980, 0.745, 0.0182, 2.00]},
    'Au+5':{'M':[0.3292, 18.073, 1.0376, 9.957 , 2.2028, 2.257, 1.8410, 1.938, 0.0050],
        'N':[0.5763, 20.517, 3.8572, 10.171, 6.0222, 4.213, 4.5020, 3.760, 0.0067, 2.00]},
        
            
    
    'Ce':{'M':[     .2953, 17.685,  .2923,  6.733,   .4313,  5.383,  .0000,   .000, -.0194],
        'N':[     .9809, 18.063, 1.8413,  7.769,   .9905,  2.845,  .0000,   .000,  .0120, 2.00]},       #really Ce+2
    'Ce+2':{'M':[   .2953, 17.685,  .2923,  6.733,   .4313,  5.383,  .0000,   .000, -.0194],
        'N':[     .9809, 18.063, 1.8413,  7.769,   .9905,  2.845,  .0000,   .000,  .0120, 2.00]},
    'Pr':{'M':[    .050, 24.999,  0.257, 12.038,    .714,  5.004, -0.022,   .084,  .0000],              #really Pr+3 - from J2K
        'N':[     .873,  18.988, 1.559,   6.087,   .814,   2.415,   .011,   .044,  .0000, 2.00]},
    'Pr+3':{'M':[    .050, 24.999,  0.257, 12.038,    .714,  5.004, -0.022,   .084,  .0000],
        'N':[     .873,  18.988, 1.559,   6.087,   .814,   2.415,   .011,   .044,  .0000, 2.00]},
    'Nd':{'M':[     .1645, 25.045,  .2522, 11.978,   .6012,  4.946,  .0000,   .000, -.0180],
        'N':[    1.4530, 18.340, 1.6196,  7.285,   .8752,  2.622,  .0000,   .000,  .0126, 2.00]},       #really Nd+2
    'Nd+2':{'M':[     .1645, 25.045,  .2522, 11.978,   .6012,  4.946,  .0000,   .000, -.0180],
        'N':[    1.4530, 18.340, 1.6196,  7.285,   .8752,  2.622,  .0000,   .000,  .0126, 2.00]},
    'Nd+3':{'M':[     .0540, 25.029,  .3101, 12.102,   .6575,  4.722,  .0000,   .000, -.0216],
        'N':[     .6751, 18.342, 1.6272,  7.260,   .9644,  2.602,  .0000,   .000,  .0150, 2.00]},
    'Sm':{'M':[     .0909, 25.203,  .3037, 11.856,   .6250,  4.237,  .0000,   .000, -.0200],
        'N':[    1.0360, 18.425, 1.4769,  7.032,   .8810,  2.437,  .0000,   .000,  .0152, 2.00]},       #really Sm+2
    'Sm+2':{'M':[     .0909, 25.203,  .3037, 11.856,   .6250,  4.237,  .0000,   .000, -.0200],
        'N':[    1.0360, 18.425, 1.4769,  7.032,   .8810,  2.437,  .0000,   .000,  .0152, 2.00]},
    'Sm+3':{'M':[     .0288, 25.207,  .2973, 11.831,   .6954,  4.212,  .0000,   .000, -.0213],
        'N':[     .4707, 18.430, 1.4261,  7.034,   .9574,  2.439,  .0000,   .000,  .0182, 2.00]},
    'Eu':{'M':[     .0755, 25.296,  .3001, 11.599,   .6438,  4.025,  .0000,   .000, -.0196],
        'N':[     .8970, 18.443, 1.3769,  7.005,   .9060,  2.421,  .0000,   .000,  .0190, 2.00]},       #really Eu+2
    'Eu+2':{'M':[     .0755, 25.296,  .3001, 11.599,   .6438,  4.025,  .0000,   .000, -.0196],
        'N':[     .8970, 18.443, 1.3769,  7.005,   .9060,  2.421,  .0000,   .000,  .0190, 2.00]},
    'Eu+3':{'M':[     .0204, 25.308,  .3010, 11.474,   .7005,  3.942,  .0000,   .000, -.0220],
        'N':[     .3985, 18.451, 1.3307,  6.956,   .9603,  2.378,  .0000,   .000,  .0197, 2.00]},
    'Gd':{'M':[     .0636, 25.382,  .3033, 11.212,   .6528,  3.788,  .0000,   .000, -.0199],
        'N':[     .7756, 18.469, 1.3124,  6.899,   .8956,  2.338,  .0000,   .000,  .0199, 2.00]},       #really Gd+2
    'Gd+2':{'M':[     .0636, 25.382,  .3033, 11.212,   .6528,  3.788,  .0000,   .000, -.0199],
        'N':[     .7756, 18.469, 1.3124,  6.899,   .8956,  2.338,  .0000,   .000,  .0199, 2.00]},
    'Gd+3':{'M':[     .0186, 25.387,  .2895, 11.142,   .7135,  3.752,  .0000,   .000, -.0217],
        'N':[     .3347, 18.476, 1.2465,  6.877,   .9537,  2.318,  .0000,   .000,  .0217, 2.00]},
    'Tb':{'M':[     .0547, 25.509,  .3171, 10.591,   .6490,  3.517,  .0000,   .000, -.0212],
        'N':[     .6688, 18.491, 1.2487,  6.822,   .8888,  2.275,  .0000,   .000,  .0215, 2.00]},       #really Tb+2
    'Tb+2':{'M':[     .0547, 25.509,  .3171, 10.591,   .6490,  3.517,  .0000,   .000, -.0212],
        'N':[     .6688, 18.491, 1.2487,  6.822,   .8888,  2.275,  .0000,   .000,  .0215, 2.00]},
    'Tb+3':{'M':[     .0177, 25.510,  .2921, 10.577,   .7133,  3.512,  .0000,   .000, -.0231],
        'N':[     .3347, 18.476, 1.2465,  6.877,   .9537,  2.318,  .0000,   .000,  .0217, 2.00]},
    'Dy':{'M':[     .1308, 18.316,  .3118,  7.665,   .5795,  3.147,  .0000,   .000, -.0226],
        'N':[     .5917, 18.511, 1.1828,  6.747,   .8801,  2.214,  .0000,   .000,  .0229, 2.00]},       #really Dy+2
    'Dy+2':{'M':[     .1308, 18.316,  .3118,  7.665,   .5795,  3.147,  .0000,   .000, -.0226],
        'N':[     .5917, 18.511, 1.1828,  6.747,   .8801,  2.214,  .0000,   .000,  .0229, 2.00]},
    'Dy+3':{'M':[     .1157, 15.073,  .3270,  6.799,   .5821,  3.020,  .0000,   .000, -.0249],
        'N':[     .2523, 18.517, 1.0914,  6.736,   .9345,  2.208,  .0000,   .000,  .0250, 2.00]},
    'Ho':{'M':[     .0995, 18.176,  .3305,  7.856,   .5921,  2.980,  .0000,   .000, -.0230],
        'N':[     .5094, 18.515, 1.1234,  6.706,   .8727,  2.159,  .0000,   .000,  .0242, 2.00]},       #really Ho+2
    'Ho+2':{'M':[     .0995, 18.176,  .3305,  7.856,   .5921,  2.980,  .0000,   .000, -.0230],
        'N':[     .5094, 18.515, 1.1234,  6.706,   .8727,  2.159,  .0000,   .000,  .0242, 2.00]},
    'Ho+3':{'M':[     .0566, 18.318,  .3365,  7.688,   .6317,  2.943,  .0000,   .000, -.0248],
        'N':[     .2188, 18.516, 1.0240,  6.707,   .9251,  2.161,  .0000,   .000,  .0268, 2.00]},
    'Er':{'M':[     .1122, 18.122,  .3462,  6.911,   .5649,  2.761,  .0000,   .000, -.0235],
        'N':[     .4693, 18.528, 1.0545,  6.649,   .8679,  2.120,  .0000,   .000,  .0261, 2.00]},       #really Er+2
    'Er+2':{'M':[     .1122, 18.122,  .3462,  6.911,   .5649,  2.761,  .0000,   .000, -.0235],
        'N':[     .4693, 18.528, 1.0545,  6.649,   .8679,  2.120,  .0000,   .000,  .0261, 2.00]},
    'Er+3':{'M':[     .0586, 17.980,  .3540,  7.096,   .6126,  2.748,  .0000,   .000, -.0251],
        'N':[     .1710, 18.534,  .9879,  6.625,   .9044,  2.100,  .0000,   .000,  .0278, 2.00]},
    'Tm':{'M':[     .0983, 18.324,  .3380,  6.918,   .5875,  2.662,  .0000,   .000, -.0241],
        'N':[     .4198, 18.542,  .9959,  6.600,   .8593,  2.082,  .0000,   .000,  .0284, 2.00]},       #really TTm+2
    'Tm+2':{'M':[     .0983, 18.324,  .3380,  6.918,   .5875,  2.662,  .0000,   .000, -.0241],
        'N':[     .4198, 18.542,  .9959,  6.600,   .8593,  2.082,  .0000,   .000,  .0284, 2.00]},
    'Tm+3':{'M':[     .0581, 15.092,  .2787,  7.801,   .6854,  2.793,  .0000,   .000, -.0224],
        'N':[     .1760, 18.542,  .9105,  6.579,   .8970,  2.062,  .0000,   .000,  .0294, 2.00]},
    'Yb':{'M':[     .0855, 18.512,  .2943,  7.373,   .6412,  2.678,  .0000,   .000, -.0213],
        'N':[     .3852, 18.550,  .9415,  6.551,   .8492,  2.043,  .0000,   .000,  .0301, 2.00]},
    'Yb+2':{'M':[     .0855, 18.512,  .2943,  7.373,   .6412,  2.678,  .0000,   .000, -.0213],          #really Yb+2
        'N':[     .3852, 18.550,  .9415,  6.551,   .8492,  2.043,  .0000,   .000,  .0301, 2.00]},
    'Yb+3':{'M':[     .0416, 16.095,  .2849,  7.834,   .6961,  2.672,  .0000,   .000, -.0229],
        'N':[     .1570, 18.555,  .8484,  6.540,   .8880,  2.037,  .0000,   .000,  .0318, 2.00]},
    'U':{'M': [     .5058, 23.288, 1.3464,  7.003,  -.8724,  4.868,  .0000,   .000,  .0192],
        'N': [    4.1582, 16.534, 2.4675,  5.952,  -.0252,   .765,  .0000,   .000,  .0057, 2.00]},      #really U+3
    'U+3':{'M': [     .5058, 23.288, 1.3464,  7.003,  -.8724,  4.868,  .0000,   .000,  .0192],
        'N': [    4.1582, 16.534, 2.4675,  5.952,  -.0252,   .765,  .0000,   .000,  .0057, 2.00]},
    'U+4':{'M': [     .3291, 23.548, 1.0836,  8.454,  -.4340,  4.120,  .0000,   .000,  .0214],
        'N': [    3.7449, 13.894, 2.6453,  4.863,  -.5218,  3.192,  .0000,   .000,  .0009, 2.00]},
    'U+5':{'M': [     .3650, 19.804, 3.2199,  6.282, -2.6077,  5.301,  .0000,   .000,  .0233],
        'N': [    3.0724, 12.546, 2.3076,  5.231,  -.0644,  1.474,  .0000,   .000,  .0035, 2.00]},
    'Np':{'M':[     .5157, 20.865, 2.2784,  5.893, -1.8163,  4.846,  .0000,   .000,  .0211],
        'N':[    3.7170, 15.133, 2.3216,  5.503,  -.0275,   .800,  .0000,   .000,  .0052, 2.00]},       #really Np+3
    'Np+3':{'M':[     .5157, 20.865, 2.2784,  5.893, -1.8163,  4.846,  .0000,   .000,  .0211],
        'N':[    3.7170, 15.133, 2.3216,  5.503,  -.0275,   .800,  .0000,   .000,  .0052, 2.00]},
    'Np+4':{'M':[     .4206, 19.805, 2.8004,  5.978, -2.2436,  4.985,  .0000,   .000,  .0228],
        'N':[    2.9203, 14.646, 2.5979,  5.559,  -.0301,   .367,  .0000,   .000,  .0141, 2.00]},
    'Np+5':{'M':[     .3692, 18.190, 3.1510,  5.850, -2.5446,  4.916,  .0000,   .000,  .0248],
        'N':[    2.3308, 13.654, 2.7219,  5.494,  -.1357,   .049,  .0000,   .000,  .1224, 2.00]},
    'Np+6':{'M':[     .2929, 17.561, 3.4866,  5.785, -2.8066,  4.871,  .0000,   .000,  .0267],
        'N':[    1.8245, 13.180, 2.8508,  5.407,  -.1579,   .044,  .0000,   .000,  .1438, 2.00]},
    'Pu':{'M':[     .3840, 16.679, 3.1049,  5.421, -2.5148,  4.551,  .0000,   .000,  .0263],
        'N':[    2.0885, 12.871, 2.5961,  5.190,  -.1465,   .039,  .0000,   .000,  .1343, 2.00]},       #really Pu+3
    'Pu+3':{'M':[     .3840, 16.679, 3.1049,  5.421, -2.5148,  4.551,  .0000,   .000,  .0263],
        'N':[    2.0885, 12.871, 2.5961,  5.190,  -.1465,   .039,  .0000,   .000,  .1343, 2.00]},
    'Pu+4':{'M':[     .4934, 16.836, 1.6394,  5.638, -1.1581,  4.140,  .0000,   .000,  .0248],
        'N':[    2.7244, 12.926, 2.3387,  5.163,  -.1300,   .046,  .0000,   .000,  .1177, 2.00]},
    'Pu+5':{'M':[     .3888, 16.559, 2.0362,  5.657, -1.4515,  4.255,  .0000,   .000,  .0267],
        'N':[    2.1409, 12.832, 2.5664,  5.132,  -.1338,   .046,  .0000,   .000,  .1210, 2.00]},
    'Pu+6':{'M':[     .3172, 16.051, 3.4654,  5.351, -2.8102,  4.513,  .0000,   .000,  .0281],
        'N':[    1.7262, 12.324, 2.6652,  5.066,  -.1695,   .041,  .0000,   .000,  .1550, 2.00]},
    'Am':{'M':[     .4743, 21.776, 1.5800,  5.690, -1.0779,  4.145,  .0000,   .000,  .0218],
        'N':[    3.5237, 15.955, 2.2855,  5.195,  -.0142,   .585,  .0000,   .000,  .0033, 2.00]},       #really Am+3
    'Am+2':{'M':[     .4743, 21.776, 1.5800,  5.690, -1.0779,  4.145,  .0000,   .000,  .0218],
        'N':[    3.5237, 15.955, 2.2855,  5.195,  -.0142,   .585,  .0000,   .000,  .0033, 2.00]},
    'Am+3':{'M':[     .4239, 19.574, 1.4573,  5.872,  -.9052,  3.968,  .0000,   .000,  .0238],
        'N':[    2.8622, 14.733, 2.4099,  5.144,  -.1326,   .031,  .0000,   .000,  .1233, 2.00]},
    'Am+4':{'M':[     .3737, 17.862, 1.3521,  6.043,  -.7514,  3.720,  .0000,   .000,  .0258],
        'N':[    2.4141, 12.948, 2.3687,  4.945,  -.2490,   .022,  .0000,   .000,  .2371, 2.00]},
    'Am+5':{'M':[     .2956, 17.372, 1.4525,  6.073,  -.7755,  3.662,  .0000,   .000,  .0277],
        'N':[    2.0109, 12.053, 2.4155,  4.836,  -.2264,   .027,  .0000,   .000,  .2128, 2.00]},
    'Am+6':{'M':[     .2302, 16.953, 1.4864,  6.116,  -.7457,  3.543,  .0000,   .000,  .0294],
        'N':[    1.6778, 11.337, 2.4531,  4.725,  -.2043,   .034,  .0000,   .000,  .1892, 2.00]},
    'Am+7':{'M':[     .3601, 12.730, 1.9640,  5.120, -1.3560,  3.714,  .0000,   .000,  .0316],
        'N':[    1.8845,  9.161, 2.0746,  4.042,  -.1318,  1.723,  .0000,   .000,  .0020, 2.00]},
}
'''
unused j<0>
W_6s15d5    FFj0A=0.3811 FFj0a=62.707 FFj0B=0.7523 FFj0b=21.434 FFj0C=12.5449 FFj0c=2.702 FFj0D=12.4130 FFj0d=2.674 FFj0E=0.0023 0.0365
W_6s25d4    FFj0A=0.3653 FFj0a=53.965 FFj0B=0.7926 FFj0b=20.078 FFj0C=0.8142 FFj0c=3.030 FFj0D=0.6581 FFj0d=2.476 FFj0E=0.0023 0.0247
W1p_6s15d4  FFj0A=0.3834 FFj0a=46.233 FFj0B=0.7890 FFj0b=19.278 FFj0C=1.4650 FFj0c=2.947 FFj0D=1.2945 FFj0d=2.628 FFj0E=0.0022 0.0138
Re_6s15d6   FFj0A=0.3691 FFj0a=53.934 FFj0B=0.7837 FFj0b=18.790 FFj0C=9.1491 FFj0c=2.558 FFj0D=8.9983 FFj0d=2.517 FFj0E=0.0027 0.0333
Re_6s25d5   FFj0A=0.3548 FFj0a=47.108 FFj0B=0.8210 FFj0b=17.769 FFj0C=9.8674 FFj0c=2.599 FFj0D=9.6938 FFj0d=2.556 FFj0E=0.0027 0.0248
Re1p_6s15d5 FFj0A=0.3736 FFj0a=41.151 FFj0B=0.8160 FFj0b=17.158 FFj0C=7.0396 FFj0c=2.642 FFj0D=6.8523 FFj0d=2.577 FFj0E=0.0026 0.0139
Os_6s15d7   FFj0A=0.3571 FFj0a=47.458 FFj0B=0.8123 FFj0b=16.770 FFj0C=1.2072 FFj0c=2.556 FFj0D=1.0404 FFj0d=2.211 FFj0E=0.0033 0.0314
Os_6s25d6   FFj0A=0.3467 FFj0a=41.778 FFj0B=0.8458 FFj0b=15.918 FFj0C=5.6370 FFj0c=2.459 FFj0D=5.4472 FFj0d=2.381 FFj0E=0.0032 0.0222
Os1p_6s15d6 FFj0A=0.3666 FFj0a=36.997 FFj0B=0.8390 FFj0b=15.425 FFj0C=2.6944 FFj0c=2.537 FFj0D=2.4916 FFj0d=2.360 FFj0E=0.0031 0.0140
Ir_6s15d8   FFj0A=0.3492 FFj0a=42.195 FFj0B=0.8350 FFj0b=15.113 FFj0C=5.1496 FFj0c=2.279 FFj0D=4.9686 FFj0d=2.201 FFj0E=0.0038 0.0296
Ir_6s25d7   FFj0A=0.3400 FFj0a=37.499 FFj0B=0.8675 FFj0b=14.402 FFj0C=2.3703 FFj0c=2.370 FFj0D=2.1661 FFj0d=2.177 FFj0E=0.0037 0.0214
Ir1p_6s15d7 FFj0A=0.3604 FFj0a=33.570 FFj0B=0.8597 FFj0b=13.993 FFj0C=2.1686 FFj0c=2.412 FFj0D=1.9518 FFj0d=2.188 FFj0E=0.0036 0.0142
j<2>
W_6s15d5 FFj2A=6.9289 FFj2a=53.425 FFj2B=7.2157 FFj2b=20.016 FFj2C=5.7754 FFj2c=6.543 FFj2D=4.2114 FFj2d=5.854 FFj2E=0.0036 0.0398 
W_6s25d4 FFj2A=5.4017 FFj2a=47.110 FFj2B=6.9113 FFj2b=18.760 FFj2C=7.0586 FFj2c=6.363 FFj2D=5.4525 FFj2d=5.802 FFj2E=0.0040 0.0400 
W1p_6s15d4 FFj2A=4.5118 FFj2a=41.281 FFj2B=6.7136 FFj2b=17.964 FFj2C=7.6331 FFj2c=6.269 FFj2D=6.0365 FFj2d=5.758 FFj2E=0.0042 0.0405 
Re_6s15d6 FFj2A=5.5830 FFj2a=46.852 FFj2B=6.4516 FFj2b=17.855 FFj2C=5.0609 FFj2c=6.037 FFj2D=3.5427 FFj2d=5.312 FFj2E=0.0040 0.0334 
Re_6s25d5 FFj2A=4.4322 FFj2a=41.798 FFj2B=6.2063 FFj2b=16.844 FFj2C=9.8763 FFj2c=5.751 FFj2D=8.3294 FFj2d=5.406 FFj2E=0.0043 0.0327 
Re1p_6s15d5 FFj2A=3.7875 FFj2a=37.254 FFj2B=6.0587 FFj2b=16.264 FFj2C=6.9896 FFj2c=5.786 FFj2D=5.4457 FFj2d=5.279 FFj2E=0.0045 0.0329 
Os_6s15d7 FFj2A=4.6511 FFj2a=41.610 FFj2B=5.8194 FFj2b=16.062 FFj2C=6.7172 FFj2c=5.447 FFj2D=5.2689 FFj2d=4.986 FFj2E=0.0043 0.0274 
Os_6s25d6 FFj2A=3.7421 FFj2a=37.491 FFj2B=5.6137 FFj2b=15.256 FFj2C=10.0488 FFj2c=5.293 FFj2D=8.5683 FFj2d=4.991 FFj2E=0.0046 0.0259 
Os1p_6s15d6 FFj2A=3.2611 FFj2a=33.818 FFj2B=5.4945 FFj2b=14.808 FFj2C=5.7610 FFj2c=5.391 FFj2D=4.2811 FFj2d=4.827 FFj2E=0.0048 0.0261 
Ir_6s15d8 FFj2A=3.9372 FFj2a=37.508 FFj2B=5.2846 FFj2b=14.661 FFj2C=3.7267 FFj2c=5.263 FFj2D=2.3158 FFj2d=4.416 FFj2E=0.0045 0.0217 
Ir_6s25d7 FFj2A=3.2263 FFj2a=33.922 FFj2B=5.1086 FFj2b=13.921 FFj2C=6.5993 FFj2c=4.978 FFj2D=5.1841 FFj2d=4.549 FFj2E=0.0048 0.0203 
Ir1p_6s15d7 FFj2A=2.8732 FFj2a=30.809 FFj2B=5.0094 FFj2b=13.522 FFj2C=6.8656 FFj2c=4.933 FFj2D=5.4669 FFj2d=4.526 FFj2E=0.0050 0.0199 
    
'''
