"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginAccessIdentity = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
class OriginAccessIdentityBase extends cdk.Resource {
    /**
     * The ARN to include in S3 bucket policy to allow CloudFront access
     */
    arn() {
        return cdk.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            account: 'cloudfront',
            resource: 'user',
            resourceName: `CloudFront Origin Access Identity ${this.originAccessIdentityName}`,
        });
    }
}
/**
 * An origin access identity is a special CloudFront user that you can
 * associate with Amazon S3 origins, so that you can secure all or just some of
 * your Amazon S3 content.
 *
 * @resource AWS::CloudFront::CloudFrontOriginAccessIdentity
 */
class OriginAccessIdentity extends OriginAccessIdentityBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new cloudfront_generated_1.CfnCloudFrontOriginAccessIdentity(this, 'Resource', {
            cloudFrontOriginAccessIdentityConfig: {
                comment: (props && props.comment) || 'Allows CloudFront to reach the bucket',
            },
        });
        // physical id - OAI name
        this.originAccessIdentityName = this.getResourceNameAttribute(this.resource.ref);
        // Canonical user to whitelist in S3 Bucket Policy
        this.cloudFrontOriginAccessIdentityS3CanonicalUserId = this.resource.attrS3CanonicalUserId;
        // The principal for must be either the canonical user or a special ARN
        // with the CloudFront Origin Access Id (see `arn()` method). For
        // import/export the OAI is anyway required so the principal is constructed
        // with it. But for the normal case the S3 Canonical User as a nicer
        // interface and does not require constructing the ARN.
        this.grantPrincipal = new iam.CanonicalUserPrincipal(this.cloudFrontOriginAccessIdentityS3CanonicalUserId);
    }
    /**
     * Creates a OriginAccessIdentity by providing the OriginAccessIdentityName
     */
    static fromOriginAccessIdentityName(scope, id, originAccessIdentityName) {
        class Import extends OriginAccessIdentityBase {
            constructor(s, i) {
                super(s, i, { physicalName: originAccessIdentityName });
                this.originAccessIdentityName = originAccessIdentityName;
                this.grantPrincipal = new iam.ArnPrincipal(this.arn());
            }
        }
        return new Import(scope, id);
    }
}
exports.OriginAccessIdentity = OriginAccessIdentity;
//# sourceMappingURL=data:application/json;base64,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