# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/03_signals.ipynb.

# %% ../nbs/03_signals.ipynb 1
from __future__ import annotations
from typing import Callable, Optional, Set, Any, NamedTuple, Protocol, Tuple, Dict, TypeVar
from fastcore.test import test_eq, test_fail
from .utils import Bunch

# %% auto 0
__all__ = ['T', 'Getter', 'Setter', 'Subscriber', 'Unsubscriber', 'context', 'observable', 'cell', 'observer', 'callback',
           'effect', 'view', 'derived', 'formula', 'Observable', 'Observer', 'Readable', 'Writable', 'Signal',
           'Callback', 'signal', 'reaction', 'writable', 'pipe', 'computed', 'readonly']

# %% ../nbs/03_signals.ipynb 3
T = Optional[TypeVar("T")]
Getter = Callable[[],T]
Setter = Callable[[T], None]
Subscriber = Callable[[T], None] # a callback
Unsubscriber = Callable[[], None] # a callback to be used upon termination of the subscription
class Observable(Protocol):
    def subscribe(self, 
                  observer: Callable[[Any], None]
                  ) -> None: ...
class Observer(Protocol):
    def run(self) -> None: ...
    def cancel(self) -> None: ...
class Readable(Bunch):
    ''' An observable that can be read.'''
    def __init__(self, 
                 read: Getter
                 ): self.read = read
    def __repr__(self): 
        return f'${(self.read()).__class__.__name__}: {self.read().__repr__()}'
    def asTuple(self):
        return self.__dict__.values()
    
class Writable(Readable):
    ''' An observable that can be read and set.'''
    def __init__(self, read, 
                 write:Setter): 
        super().__init__(read)
        self.write = write

class Signal(Writable, Observable):
    ''' A signal (aka store) stores a value and run callbacks when the value changes.
    '''
    def __init__(self, 
                 read: Getter, 
                 write:Setter,
                 subscribe: Callable): 
        super().__init__(read, write)
        self.subscribe = subscribe
    subscribe = lambda self, observer: self.subscribe(observer)


# Signal = Tuple[Getter, Setter, Callable[[Subscriber], None]]
class Callback(Bunch, Observer):
    ''' A callback is a function that is called when a signal changes.'''
    def __init__(self, 
                 run:Callable[[], None], # the function to be called
                 cancel:Callable[[], None], # cancels the subscription
                 subscriptions:Set[Any] # observables the callback is subscribed to
                 ): 
        self.run = run
        self.cancel = cancel
        self.subscriptions = subscriptions
    run = lambda self: self.run()
    cancel = lambda self: self.cancel()

# %% ../nbs/03_signals.ipynb 4
context = [] # a stack of dependencies of the current reaction


def signal(
    value: T = None # initial value
    ) -> Signal:
    ''' Signal factory.'''
    subscribers = set()

    def subscribe(
        callback: Subscriber # callback to be called when the store value changes
        ) -> None:
        '''Add callback to list of subscribers to be executed when the signal value changes.
        Also adds unsubscribe function to the callback's subscriptions.
        '''
        subscribers.add(callback.run)
        callback.subscriptions.add(lambda: subscribers.discard(callback.run))

    def read() -> T: # signal getter
        callback = context[-1] if context else None
        if callback: subscribe(callback)
        return value

    def write(newValue: T) -> None: # signal setter
        nonlocal value
        value = newValue
        # run can change the subscribers set, so we need to copy it
        for run in subscribers.copy():run()

    return Signal(read, write, subscribe)


def reaction(fn: Callable) -> Callback:
    ''' Reaction factory. A reaction is a callback that is called when a signal changes.\n
    Also known as: effect, observer, callback, computed, formula, derived.'''
    def cancel():
        nonlocal callback
        # unsubscribe function can change the set, so we need to copy it
        for u in callback.subscriptions.copy(): u()
        callback = Callback(callback.run, callback.cancel, set())

    def run():
        cancel()
        context.append(callback)
        try: fn()
        finally: context.pop()

    callback = Callback(run, cancel, set())
    run()
    return callback

def writable(value:T=None) -> Writable:
    ''' Writable factory. A writable is an interface to a signal.'''
    s = signal(value)
    return Writable(s.read, s.write)

from .utils import compose
from functools import reduce

def pipe(*fns):
    ''' Composes a list of functions.'''
    w = writable()
    fn = compose(*fns)
    _ = reduce(lambda fn: w.write(fn()), [*fns])
    # _ = reaction(lambda: w.write(fn()))
    return w.read

def computed(fn) -> Getter:
    ''' A computed is a signal that is derived from other signals. It is a kind of cache of a reaction.'''
    w = writable()
    _ = reaction(lambda: w.write(fn()))
    return w.read



def readonly(value:T=None) -> Getter:
    w = writable(value)
    return w.read
    # return Readable(read)


observable = cell = signal
observer = callback = effect = view = reaction
derived = formula = computed

