# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['suretime']

package_data = \
{'': ['*']}

install_requires = \
['defusedxml>=0.7,<1.0', 'ntplib>=0.4,<1.0']

setup_kwargs = {
    'name': 'suretime',
    'version': '0.2.0',
    'description': 'Handle IANA timezones and fully resolved timestamps and intervals on all platforms',
    'long_description': '# Suretime\n\n[![Version status](https://img.shields.io/pypi/status/suretime)](https://pypi.org/project/suretime)\n[![License](https://img.shields.io/badge/License-Apache%202.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)\n[![Python version compatibility](https://img.shields.io/pypi/pyversions/suretime)](https://pypi.org/project/suretime)\n[![Version on GitHub](https://img.shields.io/github/v/release/dmyersturnbull/suretime?include_prereleases&label=GitHub)](https://github.com/dmyersturnbull/suretime/releases)\n[![Version on PyPi](https://img.shields.io/pypi/v/suretime)](https://pypi.org/project/suretime)\n[![Build (GitHub Actions)](https://img.shields.io/github/workflow/status/dmyersturnbull/suretime/Build%20&%20test?label=Build%20&%20test)](https://github.com/dmyersturnbull/suretime/actions)\n[![Test coverage (coveralls)](https://coveralls.io/repos/github/dmyersturnbull/suretime/badge.svg?branch=main&service=github)](https://coveralls.io/github/dmyersturnbull/suretime?branch=main)\n[![Maintainability (Code Climate)](https://api.codeclimate.com/v1/badges/14b23b28b0d9c37a0ebf/maintainability)](https://codeclimate.com/github/dmyersturnbull/suretime/maintainability)\n[![Code Quality (Scrutinizer)](https://scrutinizer-ci.com/g/dmyersturnbull/suretime/badges/quality-score.png?b=main)](https://scrutinizer-ci.com/g/dmyersturnbull/suretime/?branch=main)\n[![Created with Tyrannosaurus](https://img.shields.io/badge/Created_with-Tyrannosaurus-0000ff.svg)](https://github.com/dmyersturnbull/tyrannosaurus)\n\nMap nonstandard timezones to IANA zones on all platforms.\nRecord timestamps and intervals that behave correctly,\neven if the timezone changes in the middle of a calculation.\n`pip install suretime tzdata`.\n\n### Timezone mapping\n\nTo get the local zone in IANA:\n\n```python\nfrom suretime import Suretime, datetime\n\ndatetime.now().tzname()  # "Pacific Standard Time"\nSuretime.zones.first_local()  # ZoneInfo[America/Los_Angeles])\n```\n\nTo map a nonstandard zone from elsewhere:\n\n```python\nfrom suretime import Suretime\n\nSuretime.zones.only("Europe/Tiraspol")  # ZoneInfo[Europe/Tiraspol]\nSuretime.zones.first("Central Pacific Standard Time")  # ZoneInfo[Pacific/Guadalcanal]\nSuretime.zones.first("Central Pacific Standard Time", territory="AQ")  # ZoneInfo[Antarctica/Casey]\n```\n\nNote that there is no 1-1 mapping between Windows and IANA timezones,\nso suretime can fail despite its name.\n\n### "Tagged" datetimes and intervals\n\nSuretime also has models to represent timestamps and intervals as accurately as the system permits.\nFor example, a `TaggedInterval` contains the wall time, IANA zone, original (unmapped) timezone info\nfrom the system, monotonic (typically boottime) clock time, and the clock used.\n\nTimezone-resolved datetimes and intervals know both real and calendar times, correctly representing\nthe ground truth even if a timezone shift occurs between events – such as from a daylight savings change\nor the user boarding a flight.\n\n```python\nfrom suretime import Suretime\n\ntagged = Suretime.tagged.now()\ntagged.to_utc  # TaggedDateTime[...]\ntagged.clock.name  # "boottime" on most systems\ntagged == tagged  # same point in time\ntagged.iso_with_zone  # "2021-01-20T22:24:13.219253-07:00 [America/Los_Angeles]"\n```\n\n\n### Comparison to [tzlocal](https://github.com/regebro/tzlocal)\n\ntzlocal is a bit different. It:\n- ... only handles your current system’s timezone\n- ... is highly OS-specific\n- ... requires many system calls, making it typically *much* slower\n- ... is compatible with Windows 2000, XP, and 7 and below\n- ... is compatible with Python 3.6, 3.7, and 3.8\n- ... very rarely, can access timezones on incorrectly configured POSIX systems\n\nYou can combine both packages, falling back to tzlocal if suretime fails.\n(See the example below).\n\n### Full example\n\n```python\nfrom suretime import Suretime, datetime\n\n# Get your local system, non-IANA timezone\nsystem_time = datetime.now().astimezone()\nsystem_timezone = system_time.tzname()  # e.g. Pacific Standard Time\n\n# Get an IANA timezone instead:\nSuretime.zones.only_local()  # ZoneInfo[America/Los_Angeles]\n# Or for an arbitrary system timezone name:\nSuretime.zones.first(system_timezone)  # ZoneInfo[America/Los_Angeles]\n# Of course, it maps IANA zones to themselves:\nSuretime.zones.only("America/Los_Angeles")  # ZoneInfo[America/Los_Angeles]\n\n# Get all IANA timezones that could match a zone\n# The first uses the primary/null territory\n# The second uses the territory "AQ"\nSuretime.zones.all("Central Pacific Standard Time")  # {ZoneInfo[Pacific/Guadalcanal]}\nSuretime.zones.all("Central Pacific Standard Time", "AQ")  # {ZoneInfo[Antarctica/Casey]}\n\n# Get 1 matching IANA zone; "get" means optional\nSuretime.zones.first("Central Pacific Standard Time", "AQ")  # ZoneInfo[Pacific/Casey]\nSuretime.zones.first("nonexistent zone")  # None\nSuretime.zones.only("nonexistent zone")  # errors\nSuretime.zones.only("Central Pacific Standard Time", "any")  # fails (multiple possible IANA zones)\n\n# Get a fully resolved "tagged datetime"\n# It contains:\n# - The zoned datetime\n# - The primary IANA ZoneInfo\n# - The original system timezone\n# - A system wall time (`time.monotonic_ns`)\ntagged = Suretime.tagged.now_local_sys()  # TaggedDatetime[ ... ]\nprint(tagged.clock.name)  # e.g. "boottime"\n# or NTP:\ntagged = Suretime.tagged.now_local_ntp(server="north-america", kind="server-received")  # TaggedDatetime[ ... ]\n# or fully reliable but not keeping the local zone:\ntagged = Suretime.tagged.now_utc_ntp()  # TaggedDatetime[ ... ]\n# or NTP:\ntagged = Suretime.tagged.now_local_sys()  # TaggedDatetime[ ... ]\nprint(tagged.clock.name)  # "ntp:..."\nprint(tagged.clock.info.resolution)  # e.g. -7\n\n# 2021-01-20T22:24:13.219253-07:00 [America/Los_Angeles]\nprint(tagged.iso_with_zone)  # <datetime> [zone]\nprint(tagged.source.territory)  # "primary"\n\n# if you only need the real time:\nSuretime.clocks.sys()\n# or:\nSuretime.clocks.ntp()\n# or for all of the NTP clock times:\nntp_data = Suretime.clocks.ntp_raw()\nprint(ntp_data.root_dispersion)\nprint(ntp_data.server_sent, ntp_data.client_received)\nprint(ntp_data.round_trip.total_seconds())\n\n# Adjust a tagged time to a new real time\ntagged = Suretime.tagged.now_utc_sys()  # TaggedDatetime[ ... ]\nsys_now = Suretime.clocks.sys()\ntagged.at(sys_now)\n# now the datetime is adjusted to where it should be at the new real time\n\n# Compare tagged datetimes\nprint(tagged < tagged)  # False\nprint(tagged == tagged)  # True: They\'re the same point in time\nprint(tagged == system_time)  # True: They\'re the same point in time\nprint(tagged.is_identical_to(tagged))  # True: They\'re exactly the same\n\n# Get a "tagged duration" with the start and end, and monotonic real time in nanoseconds\nthen = Suretime.tagged.now()  # TaggedDatetime [ ... ]\nfor i in list(range(10000)): i += 1  # Just waiting a little\nnow = Suretime.tagged.now()  # TaggedInterval [ ... ]\ninterval = Suretime.tagged.interval(then, now)  # TaggedInterval [ ... ]\nprint(interval.real_delta)  # Actual time passed\nprint(interval.wall_delta)  # Simple end - start\nprint(interval.iso)  # start--end in ISO 8601 format\nprint(interval.real_str)  # days:HH:mm:ss.millis.micros.nanos\nprint(interval.duration.iso)  # e.g. P0Y3M5DT14H22M35.223051S\n\n# use suretime, fall back to tzlocal\nimport tzlocal\n\n\ndef get_local() -> Suretime.Types.TaggedDatetime:\n  try:\n    return Suretime.tagged.now()\n  except Suretime.Errors.CannotMapTzError:\n    zone = tzlocal.get_localzone()\n    return Suretime.tagged.exact(datetime.now(zone))\n\n```\n\n### 🍁 Contributing\n\nLicensed under the terms of the [Apache License 2.0](https://spdx.org/licenses/Apache-2.0.html).\n[New issues](https://github.com/dmyersturnbull/suretime/issues) and pull requests are welcome.\nPlease refer to the [contributing guide](https://github.com/dmyersturnbull/suretime/blob/main/CONTRIBUTING.md)\nand [security policy](https://github.com/dmyersturnbull/suretime/blob/main/SECURITY.md).\n\nGenerated with [Tyrannosaurus](https://github.com/dmyersturnbull/tyrannosaurus).\n',
    'author': 'Douglas Myers-Turnbull',
    'author_email': None,
    'maintainer': 'Douglas Myers-Turnbull',
    'maintainer_email': None,
    'url': 'https://github.com/dmyersturnbull/suretime',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
