from ..test import ExperimentTest, Results


class HasCorrectItems(ExperimentTest):
    """Passes if an ANTS resource is found and such resource contains the main
    expected items."""

    passing = 'BBRCDEV_E01613',
    failing = 'BBRCDEV_E02443',

    def run(self, experiment_id):
        expected_items = {'*_dn_bc_T1space_affine.txt': 1,
                          '*_dn_bc_T1space.nii.gz': 1,
                          '*_dn_bc.nii.gz': 2}
        data = []
        result = True

        res = self.xnat_instance.select.experiment(experiment_id)\
            .resource('ANTS')

        for k, v in expected_items.items():
            nfiles = len(res.files(k).get())
            if nfiles != v:
                data.append('`%s`: %i file(s) expected, %i file(s) found<br>'
                            % (k, v, nfiles))
                result = False

        return Results(result, data)


class HasCorrectANTsVersion(ExperimentTest):
    """This test checks the version of ANTs used. Passes if ANTS outputs were
    created using the expected version (`2.3.1.0.post149-gd15f7`)."""

    passing = 'BBRCDEV_E01613',
    failing = 'BBRCDEV_E00272',

    def run(self, experiment_id):
        import logging as log

        expected_version = 'ANTs Version: 2.3.1.0.post149-gd15f7'
        data = []
        result = True

        columns = ['label', 'project', 'xnat:mrsessiondata/subject_id']
        labels = self.xnat_instance.array.mrsessions(experiment_id=experiment_id,
                                                     columns=['label']).data
        exp_label, project, subject_id = [labels[0][e] for e in columns]

        res = self.xnat_instance.select.project(project).subject(subject_id)\
            .experiment(experiment_id).resource('ANTS')
        if not res.exists():
            log.error('ANTS resource not found for %s' % experiment_id)
            return Results(False, data=['ANTS resource not found'])

        log = res.file('LOGS/%s.log' % exp_label)
        log_data = self.xnat_instance.get(log.attributes()['URI']).text
        version = [line for line in log_data.splitlines()
                   if line.startswith('ANTs Version')]

        if not version or not version[0].startswith(expected_version):
            result = False
            data = ['%s' % version[0]]

        return Results(result, data)


class ANTSSnapshot(ExperimentTest):
    """This test creates a snapshot of the results generated by ANTs. The
    snapshot includes a T1w image as background (grayscale) and a coregistered
    T2w image as additional overlay (red colormap). Test passes if the snapshot
    is created successfully, fails otherwise. Does not tell anything on the
    segmentation quality."""

    passing = 'BBRCDEV_E01613',
    failing = 'BBRCDEV_E00754',  # has no ANTS resource

    def run(self, experiment_id):
        import os
        import tempfile
        import logging as log

        if os.getenv('SKIP_SNAPSHOTS_TESTS') == 'True':
            return Results(experiment_id == self.passing[0],
                           data=['Skipping it. (SKIP_SNAPSHOTS_TESTS)'])

        r = self.xnat_instance.select.experiment(experiment_id).resource('ANTS')
        if not r.exists():
            log.error('ANTS resource not found for %s' % experiment_id)
            return Results(False, data=['ANTS resource not found'])

        f = r.files('*T1space.nii.gz')[0]

        fd1, t2_mri_path = tempfile.mkstemp(suffix='.nii.gz')
        f.get(t2_mri_path)

        from ..sanity import data
        p = data.HasPreferredT1(self.lut, self.xnat_instance)
        e = p.preferred_t1(experiment_id)

        scan_files = list(self.xnat_instance.select.experiment(experiment_id)
                          .scan(e).resource('NIFTI').files('*.nii.gz'))

        if len(scan_files) == 0:
            return Results(False, data=['T1 not found.'])

        fd2, t1_mri_path = tempfile.mkstemp(suffix='.nii.gz')
        scan_files[0].get(t1_mri_path)

        from . import ants_snapshot
        res = ants_snapshot(t1_mri_path, t2_mri_path)

        return Results(True, res)

    def report(self):
        report = []
        for path in self.results.data:
            report.append('![snapshot](%s)' % path)
        return report
