#	x509sak - The X.509 Swiss Army Knife white-hat certificate toolkit
#	Copyright (C) 2018-2018 Johannes Bauer
#
#	This file is part of x509sak.
#
#	x509sak is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; this program is ONLY licensed under
#	version 3 of the License, later versions are explicitly excluded.
#
#	x509sak is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with x509sak; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#	Johannes Bauer <JohannesBauer@gmx.de>

from x509sak.tests import BaseTest
from x509sak.estimate import AnalysisOptions
from x509sak.estimate.EstimateRSA import RSASecurityEstimator

class RSASecurityEstimatorTests(BaseTest):
	def test_rsa_modulus_bitlength(self):
		# NIST SP800-57 Part 1 revision 4 (January 2016)
		expected_rsa_security = {
			1024:	80,
			2048:	112,
			3072:	128,
			7680:	192,
			15360:	256,
		}
		nontrivially_factorable_moduli = {
			1024:	11806812585384728620706951598473666388298277121619179787397705267844756965795861017112861169301246554960036504817449514058384399516746245699885208636289237 * 12485510381322057478594292531446616371031948495646810728264046277812188796992044728689644845492036571404332627618421262503169441079325896088171468793697737,
			2048:	158539917544424793425906670006226660866514434532884958272856606178817708538543006868624516931926067308086826318789585642070730648901358108854409094262141210511380662596559387300703344822810443806545725751615962248272238156185696290183040676587344500291113769543029535390119688481409607447151370671128834269667 * 178487773869652946431247862258820414496497925292883317395528014931419161217856246591619132193882034251506788306119209316604713480309641179574436063040381061678027299272808144688678886265713297852669229796109210285758084074367141357004686785921755514411420202241925666046759362924573617188336885461106417508063,
			3072:	0xd7c0df1102af20364d02253f9d374936e37fcedc2e94e6323fbdc02cbbe8449455326452c67641af12584228422048f460927ec083fd5fa51f674aef6d45d1496e6f511a38c924181243ae834c89a57c483cb638289f8caa9961c5824b191d5ab3c10e0077b1cae0f286476592042602d9b7093df1bb1be6edbbacc228477282f83024c7804602d4f0211a1b93541b3a9138b5f89c8ede0d0e7646c74d79547650b390258d7f06e47859952f75d970f5f63f6853a4b2968bb4d6c0e96e0955d5 * 0xd814f10c924277f35df339453845a1aba24a785a5cb4953ac1f38c7fd1af69298597e5156c145c825aed0767e886fb38cea81a1dce360c0e86ed7a19effd0fd1d50c8fb66c486724f8fd2426fc3ac13e37101e035f833bf38916f70963894731a81460c848ff29102f09ebc69538322536676f18e391608d04f077a9e1e04c8e319f4934a49792d814c91897fdf63d870f3b059bdd53b813da5b29cc30c9625609adc93db27001c8a7dd663aed6cab16a63c3003f2ee3f595b59d83b1e09ca4b,
			7680:	0xe695a129c8941256e6fe693a19f6606f3e9d9ab8e11709c1dbefe75573f7f6b23a29837111babbe8c1ee3dfe9664031e4006fdc0ad01c33342babcf4814de6e3202ef29ce9f4d58f1ac08be192ef168be6ddd703dfcb2396b98d1318f3f8f4d1ab412271ee5c4c08f533a34a58322b19f5b9a91711fda76c49fb66a76c5baaafd03d36869a599b5716d94754c7cada22203c86009ae47bd6e6dbc9ffc2bba1f98a194df95013516a380c53a792d92aa8f201ab97c052c8b6c21a0b2f0e4b9886288bc0e7cbf8b8388b1554e5a014e944ecc0a59458ce548aa6c316f61ebab7c900a24abe655abdc70f146f7197c4a67012b3d43a42574966e09f1f3bb04ea2d6749b2493133d67e6a9c62651dd0d22bce1353ed38488d3163ab42b63887b8c6927243684a972c434b256ef53b9eea0597fc44c59134d1389b8659607b3eb198154dc8f3ad9eec6fff9de98e7e54a7adadf220771e7862160b3b976991ed0fc6bb286064038539e7f8240ec0f1e84a18fd500d5182bd76c9aaf77fd3eb68d97e928394ff2452ed558834addca8ca698fda478bb6777fe857c846f28cfa24bb18fd03f56afb00bfdfce75011e3488679aeaea101e4eefe28348ca1eeca87b930d88623fcc61fb6176edf4614e36552ff08a4c647aab6a84b3ccf204c03fdafac51 * 0xc738967f6217d42d0303027cc2c39168670af5050a33d31b869a4a11f23edaef30fe654db1808360a7d71366eec431801378112f2894f27e17fb2ba5999081222ab342be9a211f1a088ba9518f50b5f4c592a08f6bcbdbb7492c85484d838547537d3ae298480a9ca1f0ff68ecd28d147ea7025f0c4f3af55412efa74886ac32c59b12681652b4c20d3dea0615a45fa5659aa4b280907e7ee349ab364cf9fa0f1500ec1b17a575aa7afc586d1003d2f8429a5b59fafd905673c88b36b44164bb11dafefe472ab7d695aeadba1d5bd686ef23ee4ac4089789c93d290b59dc66f21143e6c329bf1597960a98e5e9ccab67488c0ae42ffca2b7f5ee925c32ef9e7d534ffd2129be365b68c1715ad4ebd94f0ef96758c9a816e7ae954b117a21235b6c009aea958b68ee426aec5925a879bcf4886173db8eda792d79f38fecde7434527905dd759224cc475ac80bffa99eb620dc741e0cd77cbbd8656d632cbd35bad29d7db451eac48c5789d8002b36b57a63772bf8ee38cdf6f6822d004f9e6d677a4385627ea94e9f07ac4a7ec4a929f70fb7f3ab692d1d0a1e6ff90b32acfa50693f1d26848493ec6064b36459366ddae01e430038edc274e5bc91932f295fc64a600d83562c588d9c1d344b7796c35342ecb668295a53ce0a6928a3aa30187f,
			15360:	0xdd8b25d77df1c18a9e142832de75c8c0623b7be01909a0ad608dbc3cd4f7840015f166a8b473eb4d03a4f853e0d71cceb47993daa15eb223372b60319ca64ce06cb7410fe136311153b351a5c2b5121c5d758c033f2c73f0c72979d0862777a8d27f65c8f3767e025a83f1821c4177d3d056d163998c6339d4af9501dc704440a4dcd86d3ad566be46309d482bb7b5df1334668eb52a8a5701e9d49dc6223f51f6991a11598fed80814989e40fdd50e0335a55a64709611c4d2c2821a7c0cd8cefb86edf3b2a4f2449a1b62003e597be88c51828f32d685f0ddd93bbac75a64adceb082a2d6ba3f623282697b2c01d7897f330e02c2d436ba9eb3e9b17febd50b4af52baef54c7f37ab8586b62125a56faabe4be16b0c8996bb1737fe4ddeae017359c86c809888f31c889781462872a146b7e8bdb6f05bcc03e3ae3b8f547f65bad0888f92f425c3e8a0930d2a57af2fe52aa00821b120eadc8c364ffd54404b6f60db2df0698730de8aaaabc889807ff13355ac27812a8fe34c66a3a768c7b671e174cd7b66f897a0f2cb1a72cba987b83a3be6087b7f9d86212501f76be20ba44101548a04f73c37141767013f80443239f3be23424408b58f7864659cd69f1549043f2cb85d632ddd346524e3666900dd43f1e15a4d83ded788b8d9048f27732f2e1b7b017d70f5adc405515410aa0db68a52d49ae73e72be2babf355923c06a346e3f31c8817d9b691b8f283b8b446ca6bc2da0643a483bc3c70894849f6d2c6f3298942417cd3d9331427fc1619297f24d427f4223b9cea63c90404f3c3108230fe216699d6b4c71fb2817185f554e89919a9dda59202774650db7c999e41338923156673811df4afbbc701a5b6a42fba22cbc5ac8a21adbc779b5cd13591f1f1cc6fa982d5e3605d46160f3ab35b9d1b1a1111d5ce550ed45b6b3545ade9ef2e7ab3d755179010986fcbeeb5dbad3894ad03272813db7297f7354d011f0e0082a0714f98dcce5c04398dd1003b449f4f04e5504b21de9edd1c34a20fdb6ffa50918d28def28e71a30a6e53df7ae200306ec7584605f16b2892c39e2d3c4842704160fa2e82275135faedbb30002e9cd177829205898a7112d87bde68e221ab91e33de31c73e6ce7e55357322a58687a5fa20699e362ea2e3145cfa404d07a75f51e9826cec8a6e67dbf498396e8224109b0e26821cd71e06f0d9e49b03410aba911425eb8dd12226b874afbbcd07f6a91aed99d4b28b5c34b5aafdcf1a20370d8fe0c845ab10c105f542ed76df078d737d698ee88d0bb72f2c72d9efdf8f2f41f6ce268a6325f62b55111f44e95c5bcb6786a645b3be5cbbe1f1f4213 * 0xd076764533770944e5d751cb3b8824ad3837fbac1f89683afdc72ed20cc3045cd411024a97f7d26551ebcb09a4fd1915e51c1954e8e8f1b9bec345e31ddf10929a35abdc62b73f31d6193363e161884dcb8a186e629d57e27187c69891949ea491e4c15b8ac8f4156ddfb3edfcdd5147ff57b433955d65e226ffef61d7220dc3681c033e7ed8307ba872c9036ade6a5b2a8aba2312272a85a7604b19d67a5844584c09d5ea6166ab46cf43a8837d289a2f41919ef9c31e53c13533485aaccec16b9414c095330e8fce8abbd35bd8129738b7bfcaf6050f8bbe80e723a44b95de053039c627fe3088c6abbb12a9abd19d1e88b94d3e87a99e4985c6258b3f9cbd0bbc9e32e7840aa45eb7e9715eb70a8ef9d74aa628531fe56b2d616157d40afc40fa8081f697e5bbe77a51a7335017ff5b931d20526226666be073f880f87167d7f4ccc10e674f7d211480be8882d5b5347dcd05f62a91964d25da85f8471e532212e9a612b1ff878eb120250f1a1e8b275e5749ba42c4fbe8aa6c54737346e7c79b9570faff333d6ff35bf7d1126b089dcd10991a9c33d6b382ac6ffa0a539cba42163af63a3ab74eddf1ceb14647e98d14b767ab98982cf014fe27e9fc92fdb1b85637428583c17070bf01fad3b2c7ecf019bfd1ee1113e421d4df7b601869dac26faa63a3083c2fa131ae340b01cbaa5aaa6a0ebbed289f848a155fc643991f908f56e132dc657a2304cdd0b30889aff009891bb44844f6ba858916e772166f92fcda4084eb52dcc31378c493d83911fe7b95daa0cb2ab15dc6aa19ffd701c6c9ca5688ff848bc8e84b76af9eebf19561bf2d1a40355ca9f9ac6d211801fd2d679d850948ddce2060955908d0f1c2cc60c04c5008a334c3e730390c4cf992f55863429080e9a8cfaa7d07beee0b08499421bd7a3a8db38c176574240e0b45d7c5f691260ac18176156622bdb9121473add70f0346c1740e36c6b42cee663d11da1f36a54ca604667538db9796ea13945ae93cf3f74fef71f346e8edd8a69a29ccf7d857c2b77d81e7f37f66a09cb48829bf71f987aef5d9661d9ab51ccc417505b53220fb1ccfaa2b553be23565a612ea0e649be940c7d1c4f66b23cf3e62fdcdcfa8ead7f9fdef58d43e781dcf4b4aa71ec006be91045986e474a21778a19ecc5fe88adf4f5910b284fd20d96c44867d19731ab8f5875815e0aef084cc505361b8b5f2222bdd33458ca6407bdf2de2af1aa8a22e6fd7232de8a6c0ac462b088d7bb5aa35c53b52751fa7e29f913ca9372cbc0bbee9f64da996e64524a8f75124cb82899f2e11cd8db4c6afe2a45e78d24c807168baf881c08036c7c7adb1,
		}
		for (modulus_bitlength, expected_security) in expected_rsa_security.items():
			n = nontrivially_factorable_moduli[modulus_bitlength]
			self.assertEqual(n.bit_length(), modulus_bitlength)
			if modulus_bitlength <= 1024:
				analysis_options = AnalysisOptions(rsa_testing = AnalysisOptions.RSATesting.Some)
			else:
				# Skip this for large moduli, too slow.
				analysis_options = AnalysisOptions(rsa_testing = AnalysisOptions.RSATesting.Fast)
			analysis = RSASecurityEstimator(analysis_options).analyze_n(n)
			deviation = analysis.bits - expected_security
			self.assertLessEqual(abs(deviation), 8)
