# Welcome to the Open Source DIY Hyperspectral Imager Library
> Library to calibrate, capture and process data cubes for the open source DIY hyperspectral camera. 


![](https://github.com/openhsi/openhsi/actions/workflows/main.yml/badge.svg)

This Python library is licensed under the [Apache v2 License](https://www.apache.org/licenses/LICENSE-2.0). The documentation is licensed under a <a rel="license" href="http://creativecommons.org/licenses/by/3.0/au/">Creative Commons Attribution 3.0 Australia License</a>.

Documentation can be found here: [https://openhsi.github.io/openhsi/](https://openhsi.github.io/openhsi/).

## Install

`pip install openhsi` 

or

`conda install -c conda-forge openhsi`

The source code can be found on [GitHub](https://github.com/openhsi/openhsi).

## Requirements

- Python 3.7+ 

Depending on your camera sensor, install:
- Ximea SDK (See https://www.ximea.com/support/wiki/apis/Python)
- FLIR Spinnaker SDK with the python package (See https://www.flir.com/products/spinnaker-sdk/)
- LUCID SDK (See https://thinklucid.com/downloads-hub/)
{% include note.html content='A descriptive installation guide on Linux platforms can be found at https://openhsi.github.io/openhsi/tutorial_installing_linux.html' %}

## Development and Contributions

This whole software library, testing suite, documentation website, and PyPI/conda package was developed in Jupyter Notebooks using [nbdev](https://nbdev.fast.ai/). 
{% include important.html content='This library is under active development and new features are still being added. ' %}

## Citation

If OpenHSI has been useful for your research, please acknowledge the project in your academic publication. We have a publication in progress. 

```
@Article{        mao2022openhsi,
 title         = {OpenHSI: A complete open-source hyperspectral imaging solution for everyone},
 author        = {Yiwei Mao, and Christopher H. Betters, et al.},
 year          = {2022},
 journal       = {},
}
```


## How to use
{% include tip.html content='For more detailed instructions, please see the tutorials in the sidebar of the documentation site. ' %}

### Taking a single hyperspectral datacube

The example shown here uses a *simulated* camera for testing purposes. Replace `SimulatedCamera` with the appropriate Python class for your own camera to work with real hardware. For example, use `LucidCamera` imported from `openhsi.cameras` inplace of `SimulatedCamera`. 

```python

from openhsi.capture import *

with SimulatedCamera(img_path="assets/rocky_beach.png", n_lines=1024, processing_lvl = 3,
                    json_path="assets/cam_settings.json",pkl_path="assets/cam_calibration.pkl") as cam:
    cam.collect()
    fig = cam.show(plot_lib="matplotlib",robust=True)

```

    Allocated 417.15 MB of RAM.


    100%|██████████| 1024/1024 [00:22<00:00, 46.17it/s]










```python
fig.opts(fig_inches=7,title="simulated hyperspectral datacube")
```




<img src='data:image/png;base64,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' style='max-width:100%; margin: auto; display: block; '/>



{% include tip.html content='For more information on how to use this library, check out our [Quick Start Guide](https://openhsi.github.io/openhsi/tutorial_camera.html).' %}

## Hardware cameras

![](https://raw.githubusercontent.com/openhsi/openhsi/master/assets/openhsi_cam.png "The hardware consists of a collimator tube with a slit (1) mounted in a 3D printed housing (2). A diffraction grating (3) is used to split the incoming light into its component colours to be detected on the camera sensor (4).")

The hardware consists of a collimator tube with a slit (1) mounted in a 3D printed housing (2). A diffraction grating (3) is used to split the incoming light into its component colours to be detected on the camera sensor (4).  

We have the following implementations in [`cameras` module](https://openhsi.github.io/openhsi/cameras.html):
- `WebCamera`
- `XimeaCamera`
- `LucidCamera`
- `FlirCamera`

These all have the same interface so in principle, these OpenHSI cameras can be used interchangeably as long as you have the right calibration files. 

## Frequently Asked Questions

**I'm having trouble with the software install. Do you have a guide?**

Check out our [Linux Installation Guide](https://openhsi.github.io/openhsi/tutorial_installing_linux.html).

**Where can I get a quick overview of what `openhsi` can do?**

Our [Quick Start Guide](https://openhsi.github.io/openhsi/tutorial_camera.html) is the best place to start. The sidebar also includes documentation for each software module in more detail. 


**The OpenHSI camera is a pushbroom sensor and requires motion to scan a 2D space. What kind of motion should I apply?**

Any motion that allows you to scan a 2D space is okay. This can be from translating the camera is space or from applying a rotation. (or both translation and rotation)
The developers of `openhsi` has flown the OpenHSI camera on a drone which sweeps across an area of interest in multiple overlapping swaths. You can fly this camera on other platforms, vehicles, etc... 

**How fast should I move the camera?**

It'll depend on what your case is. This answer assumes you want square pixels.
Assuming the cross-track (scan line) spatial resolution is 0.42 mrad in the field of view, and your altitude is 120 m, the ground sample distance is:

GSD $\approx$ 0.00042 $\times$ 120 (using the small angle approximation) = 5 cm

Assuming your frame rate is 98 FPS at your desired processing level, and you want to get square pixels, you want to be flying at speed

$v$ = 98 $\times$ 0.05 = 4.9 m/s

If you fly faster/slower than this, your datacube will appear to be stretched spatially.
