# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dunebuggy', 'dunebuggy.core', 'dunebuggy.models']

package_data = \
{'': ['*']}

install_requires = \
['httpx>=0.18.2',
 'numpy>=1.21.0,<1.22.0',
 'pandas>=1.2.4',
 'pydantic>=1.8.2',
 'sqlparse>=0.4.2']

setup_kwargs = {
    'name': 'dunebuggy',
    'version': '1.0.5',
    'description': 'A lightweight (unofficial) Python SDK for Dune.xyz',
    'long_description': '<!-- # Dunebuggy\n\n![Alt text](./dune-buggy.svg) -->\n\n<h1 align="center">\n  <br>\n  <img src="./assets/dune-buggy.svg" alt="Dunebuggy" width="300" height="225">\n  <br>\n  <div style="sans-serif"><b>Dunebuggy</b></p>\n</h1>\n\n<div align="center">\nA lightweight (unofficial) Python SDK for <a href=https://dune.xyz/home> Dune.xyz</a>\n<br>\n\n[Installation](#installation) •\n[Getting started](#getting-started) •\n[Roadmap](#roadmap) •\n[Notes](#notes)\n\n</div>\n\n## Installation\n\n```sh\npip install dunebuggy\n```\n\n## Getting started\n\n### Retrieving a public query\n\nTo retrieve a query, all we\'ll need is the `query_id` for the public query we\'re interested in. In the below example we can take a look at the popular ["Custom NFT Floor Tracker" query by @smaroo](https://dune.xyz/queries/83579) (The `query_id` below can be found in the URL).\n\n```python\nfrom dunebuggy import Dune\n\ndune = Dune()\nquery = dune.fetch_query(83579)\n```\n\n`query` here is a `DuneQuery` object, we can get the `pandas` DataFrame for the query output bf calling `df` on the object\n\n```python\nprint(query.df.head())\n```\n\n<div>\n<table border="1" class="dataframe">\n  <thead>\n    <tr style="text-align: right;">\n      <th></th>\n      <th>Floor (Approx)</th>\n      <th>Time Interval</th>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <th>0</th>\n      <td>0.122649</td>\n      <td>2021-06-01T00:00:00+00:00</td>\n    </tr>\n    <tr>\n      <th>1</th>\n      <td>0.130000</td>\n      <td>2021-06-02T00:00:00+00:00</td>\n    </tr>\n    <tr>\n      <th>2</th>\n      <td>0.193455</td>\n      <td>2021-06-03T00:00:00+00:00</td>\n    </tr>\n    <tr>\n      <th>3</th>\n      <td>0.189000</td>\n      <td>2021-06-04T00:00:00+00:00</td>\n    </tr>\n    <tr>\n      <th>4</th>\n      <td>0.189930</td>\n      <td>2021-06-05T00:00:00+00:00</td>\n    </tr>\n  </tbody>\n</table>\n</div>\n\nWe can also take a look at some basic information about the returned query with `query.info`\n\n```python\nprint(query.info)\n```\n\n    {\'name\': \'Custom NFT Floor Tracker\',\n     \'author\': \'@smaroo\',\n     \'length\': 264,\n     \'query_id\': 83579,\n     \'result_id\': UUID(\'e5aef8a0-1453-44d1-a27b-f576ea2b3ba2\'),\n     \'job_id\': UUID(\'ec680fa9-217f-44c5-b223-56730cd07473\'),\n     \'columns\': [\'Time Interval\', \'Floor (Approx)\']}\n\nSome queries in Dune are "parameterized", meaning that the author exposes certain variables for the user to enter custom values. The example query (83579) happens to be parameterized, we can verify this by inspecting `query.parameters`\n\n```python\nprint(query.parameters)\n```\n\n    [QueryParameter(key=\'Enter NFT Contract Address\', type=\'text\', value=\'xc3f733ca98e0dad0386979eb96fb1722a1a05e69\', enumOptions=None),\n     QueryParameter(key=\'Floor Time Interval\', type=\'enum\', value=\'Day\', enumOptions=[\'Day\', \'Hour\']),\n     QueryParameter(key=\'Start Date\', type=\'datetime\', value=\'2021-06-01 00:00:00\', enumOptions=None)]\n\nIf you\'d like to run this query with your own custom parameters, all we\'ll need to do is take the parameters from from the initial query, change the values to what we want, and re-fetch the query.\n\nBelow we are replacing the old NFT contract address param with a new one ([the contract address for BAYC](https://etherscan.io/address/0xbc4ca0eda7647a8ab7c2061c2e118a18a936f13d))\n\n```python\nparams = query.parameters\n\n# Replacing with contract address for BAYC\nparams[0].value = \'xBC4CA0EdA7647A8aB7C2061c2E118A18a936f13D\'\ncustom_query = dune.fetch_query(83579, parameters=params)\n```\n\nNote: You can also create a fresh set of parameters by importing `QueryParameter` from `dunebuggy.models.query` and adding the values to the new object.\n\n```python\nfrom dunebuggy.models.query import QueryParameter\n\nparam_to_change = QueryParameter(\n  key=\'Enter NFT Contract Address\',\n  value=\'xBC4CA0EdA7647A8aB7C2061c2E118A18a936f13D\',\n)\nparams[0] = param_to_change\ncustom_query = dune.fetch_query(83579, parameters=params)\n```\n\n```python\nprint(custom_query.info)\n```\n\n    {\'name\': \'Custom NFT Floor Tracker\',\n     \'author\': \'@smaroo\',\n     \'length\': 265,\n     \'query_id\': 83579,\n     \'result_id\': UUID(\'42a3c13d-5fbd-42bd-86c0-acc9adcdc803\'),\n     \'job_id\': UUID(\'9051ebe7-862f-46d0-9999-b4645659ca56\'),\n     \'columns\': [\'Time Interval\', \'Floor (Approx)\']}\n\nNote that the `result_id` and `job_id` here are different, this is because we ran the query with our changed params\n\n```python\nprint(custom_query.parameters)\n```\n\n    [QueryParameter(key=\'Enter NFT Contract Address\', type=\'text\', value=\'xBC4CA0EdA7647A8aB7C2061c2E118A18a936f13D\', enumOptions=None),\n    QueryParameter(key=\'Floor Time Interval\', type=\'enum\', value=\'Day\', enumOptions=[\'Day\', \'Hour\']),\n    QueryParameter(key=\'Start Date\', type=\'datetime\', value=\'2021-06-01 00:00:00\', enumOptions=None)]\n\n```python\nprint(custom_query.df.head())\n```\n\n<div>\n<table border="1" class="dataframe">\n  <thead>\n    <tr style="text-align: right;">\n      <th></th>\n      <th>Floor (Approx)</th>\n      <th>Time Interval</th>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <th>0</th>\n      <td>0.8000</td>\n      <td>2021-06-01T00:00:00+00:00</td>\n    </tr>\n    <tr>\n      <th>1</th>\n      <td>0.8518</td>\n      <td>2021-06-02T00:00:00+00:00</td>\n    </tr>\n    <tr>\n      <th>2</th>\n      <td>0.8260</td>\n      <td>2021-06-03T00:00:00+00:00</td>\n    </tr>\n    <tr>\n      <th>3</th>\n      <td>0.7400</td>\n      <td>2021-06-04T00:00:00+00:00</td>\n    </tr>\n    <tr>\n      <th>4</th>\n      <td>0.8499</td>\n      <td>2021-06-05T00:00:00+00:00</td>\n    </tr>\n  </tbody>\n</table>\n</div>\n\n### Creating a new query\n\n`dunebuggy` also allows you to create a new using an existing Dune.xyz account.To login just need to pass in your username/password into the `Dune` object.\n\nYou can verify your login by inspecting your Dune `user_id`\n\n```python\nimport os\n\nusername = os.environ.get(\'DUNE_USERNAME\')\npassword = os.environ.get(\'DUNE_PASSWORD\')\n\ndune = Dune(username=username, password=password)\n# print(dune.user_id)\n```\n\nTo create a query now, all we need to do is pass in a `name`, `query_string` and `dataset_id`\n\nWe can construct the SQL query by using a raw sql string\n\n```python\nquery_string = "select * from ethereum.transactions\\nLIMIT 100\\n"\n```\n\nOr we could use a fancy ORM-style library like [pypika](https://github.com/kayak/pypika)\n\n```python\nfrom pypika import Database, Query\n\nethereum = Database(\'ethereum\')\nq = Query.from_(ethereum.transactions).select(\'*\').limit(100)\nquery_string = q.get_sql(quote_char=None)\nprint(query_string)\n```\n\n    \'SELECT * FROM ethereum.transactions LIMIT 100\'\n\nDune requires us to specify a `dataset_id` for each of their supported blockchain datasets upon query creation. The currently supported datasets are the following:\n\n| Blockchain Dataset | Id  |\n| ------------------ | --- |\n| ETHEREUM           | 4   |\n| XDAI               | 6   |\n| POLYGON            | 7   |\n| OPTIMISM_1         | 8   |\n| OPTIMISM_2         | 10  |\n| BINANCE            | 9   |\n| SOLANA             | 1   |\n\nWe can access these integer codes via the `DatasetId` enum\n\n```python\nfrom dunebuggy.models.constants import DatasetId\ncreated_query = dune.create_query("My Query\'s Name", query_string, DatasetId.ETHEREUM)\n```\n\nOur created query can be accessed like any other, you can also log into your Dune account as see it there as well!\n\n```python\nprint(created_query.df.head())\n```\n\n<div>\n<table border="1" class="dataframe">\n  <thead>\n    <tr style="text-align: right;">\n      <th></th>\n      <th>access_list</th>\n      <th>block_hash</th>\n      <th>block_number</th>\n      <th>block_time</th>\n      <th>data</th>\n      <th>from</th>\n      <th>gas_limit</th>\n      <th>gas_price</th>\n      <th>gas_used</th>\n      <th>hash</th>\n      <th>index</th>\n      <th>max_fee_per_gas</th>\n      <th>max_priority_fee_per_gas</th>\n      <th>nonce</th>\n      <th>priority_fee_per_gas</th>\n      <th>success</th>\n      <th>to</th>\n      <th>type</th>\n      <th>value</th>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <th>0</th>\n      <td>None</td>\n      <td>\\x887c665b0c52ccace092d817e984e2e828ef59079295...</td>\n      <td>47287</td>\n      <td>2015-08-07T08:50:01+00:00</td>\n      <td>None</td>\n      <td>\\xdb312d1d6a2ccc64dd94a3892928bac82b4e8c15</td>\n      <td>21000</td>\n      <td>100000000000</td>\n      <td>21000</td>\n      <td>\\xd3e6a2fc34066d20bb83020b1ee95b9dc7919fd242bd...</td>\n      <td>0</td>\n      <td>None</td>\n      <td>None</td>\n      <td>0</td>\n      <td>None</td>\n      <td>None</td>\n      <td>\\x34bb6978c5a1ad68777ad388c6787df53903430c</td>\n      <td>None</td>\n      <td>1000000000000000000</td>\n    </tr>\n    <tr>\n      <th>1</th>\n      <td>None</td>\n      <td>\\x4869e218b0a8f5784f16193ac66cbf35c4510ace0c9b...</td>\n      <td>48698</td>\n      <td>2015-08-07T15:29:53+00:00</td>\n      <td>None</td>\n      <td>\\x48040276e9c17ddbe5c8d2976245dcd0235efa43</td>\n      <td>90000</td>\n      <td>57550496008</td>\n      <td>21000</td>\n      <td>\\x8ba39f908731171fe96ee4e700e71d170ef8e651fac7...</td>\n      <td>0</td>\n      <td>None</td>\n      <td>None</td>\n      <td>0</td>\n      <td>None</td>\n      <td>None</td>\n      <td>\\xd8d0549637b65d58e7fb6cbdd11530b399d1ddac</td>\n      <td>None</td>\n      <td>100000000000000000000</td>\n    </tr>\n    <tr>\n      <th>2</th>\n      <td>None</td>\n      <td>\\xab9491b62b16bd928b281a83db82483584c22aeebc0d...</td>\n      <td>49051</td>\n      <td>2015-08-07T17:03:48+00:00</td>\n      <td>None</td>\n      <td>\\x8686578c4f7c75246f548299d6ffdac3b67b5cd1</td>\n      <td>90000</td>\n      <td>57178423039</td>\n      <td>21000</td>\n      <td>\\x57f8ba638903d6335e211eb470159587c73316788880...</td>\n      <td>0</td>\n      <td>None</td>\n      <td>None</td>\n      <td>0</td>\n      <td>None</td>\n      <td>None</td>\n      <td>\\x87abffa6b80f712c852a9558120ba6611f0b5e46</td>\n      <td>None</td>\n      <td>45150000000000000000</td>\n    </tr>\n    <tr>\n      <th>3</th>\n      <td>None</td>\n      <td>\\x1f9adc2190701ca3085b28252e4f1f467d980f763dad...</td>\n      <td>49174</td>\n      <td>2015-08-07T17:41:03+00:00</td>\n      <td>None</td>\n      <td>\\x18e4ce47483b53040adbab35172c01ef64506e0c</td>\n      <td>90000</td>\n      <td>58589751415</td>\n      <td>21000</td>\n      <td>\\xb8280da44f8d35011c3f431f7d1a82213477a4e742de...</td>\n      <td>2</td>\n      <td>None</td>\n      <td>None</td>\n      <td>0</td>\n      <td>None</td>\n      <td>None</td>\n      <td>\\xfb26ae2d3621829472555fbd11bb2a324b7a5c57</td>\n      <td>None</td>\n      <td>10000000000000000000</td>\n    </tr>\n    <tr>\n      <th>4</th>\n      <td>None</td>\n      <td>\\xf1f392fd197a149afe9f8843d7ba759d1a9f79d1ef62...</td>\n      <td>49938</td>\n      <td>2015-08-07T21:06:21+00:00</td>\n      <td>None</td>\n      <td>\\xc6bf5b6558f2ee21f2e43d9ff9b5408a0cb89413</td>\n      <td>90000</td>\n      <td>71214529679</td>\n      <td>21000</td>\n      <td>\\x538e1664c12c55287c98dc5dd248f60c642cbbbd7a18...</td>\n      <td>0</td>\n      <td>None</td>\n      <td>None</td>\n      <td>4</td>\n      <td>None</td>\n      <td>None</td>\n      <td>\\x33a3f479f6c3e7f91128348490d1f7e8d2a0fab5</td>\n      <td>None</td>\n      <td>5000000000000000000</td>\n    </tr>\n  </tbody>\n</table>\n</div>\n\n### Saving to CSV\n\nTo save a query to a CSV, we can take advantage of the `to_csv` method on our `df`\n\n```python\ncreated_query.df.to_csv(\'my_test_data.csv\')\n```\n\n## Roadmap\n\n- [ ] Cleanup punding TODO comments\n- [ ] Add support for embedding Dune graphs/ plotting w/ Dune style colors\n- [ ] Add tests (lol)\n- [ ] Add support for query updating\n- [ ] Investigate whether dashboard support makes sense?\n- [ ] Investigate whether there is a max row limit for data returned, if so, query in batches?\n- [ ] Better formatting for certain returned columns (links etc..)\n- [ ] Add Documentation (Sphinx or something else)\n\n## Notes\n\n_This project was inspired by the [itzemstar\'s duneanalytics repo](https://github.com/itzmestar/duneanalytics)_\n\n_README image is from the [IAN Symbols dataset](https://ian.umces.edu/media-library/symbols/)_\n',
    'author': 'rysuds',
    'author_email': 'ryan.sudhakaran@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/rysuds/dunebuggy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9',
}


setup(**setup_kwargs)
