"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const function_1 = require("./function");
const function_base_1 = require("./function-base");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
/**
 * A single newly-deployed version of a Lambda function.
 *
 * This object exists to--at deploy time--query the "then-current" version of
 * the Lambda function that it refers to. This Version object can then be
 * used in `Alias` to refer to a particular deployment of a Lambda.
 *
 * This means that for every new update you deploy to your Lambda (using the
 * CDK and Aliases), you must always create a new Version object. In
 * particular, it must have a different name, so that a new resource is
 * created.
 *
 * If you want to ensure that you're associating the right version with
 * the right deployment, specify the `codeSha256` property while
 * creating the `Version.
 */
class Version extends function_base_1.QualifiedFunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.canCreatePermissions = true;
        this.lambda = props.lambda;
        const version = new lambda_generated_1.CfnVersion(this, 'Resource', {
            codeSha256: props.codeSha256,
            description: props.description,
            functionName: props.lambda.functionName,
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        if (props.removalPolicy) {
            version.applyRemovalPolicy(props.removalPolicy, {
                default: core_1.RemovalPolicy.DESTROY,
            });
        }
        this.version = version.attrVersion;
        this.functionArn = version.ref;
        this.functionName = `${this.lambda.functionName}:${this.version}`;
        this.qualifier = version.attrVersion;
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
    }
    /**
     * Construct a Version object from a Version ARN.
     *
     * @param scope The cdk scope creating this resource
     * @param id The cdk id of this resource
     * @param versionArn The version ARN to create this version from
     */
    static fromVersionArn(scope, id, versionArn) {
        const version = extractQualifierFromArn(versionArn);
        const lambda = function_1.Function.fromFunctionArn(scope, `${id}Function`, versionArn);
        class Import extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.version = version;
                this.lambda = lambda;
                this.functionName = `${lambda.functionName}:${version}`;
                this.functionArn = versionArn;
                this.grantPrincipal = lambda.grantPrincipal;
                this.role = lambda.role;
                this.qualifier = version;
                this.canCreatePermissions = false;
            }
            addAlias(name, opts = {}) {
                return util_1.addAlias(this, this, name, opts);
            }
        }
        return new Import(scope, id);
    }
    static fromVersionAttributes(scope, id, attrs) {
        class Import extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.version = attrs.version;
                this.lambda = attrs.lambda;
                this.functionName = `${attrs.lambda.functionName}:${attrs.version}`;
                this.functionArn = `${attrs.lambda.functionArn}:${attrs.version}`;
                this.grantPrincipal = attrs.lambda.grantPrincipal;
                this.role = attrs.lambda.role;
                this.qualifier = attrs.version;
                this.canCreatePermissions = false;
            }
            addAlias(name, opts = {}) {
                return util_1.addAlias(this, this, name, opts);
            }
        }
        return new Import(scope, id);
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get role() {
        return this.lambda.role;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differes from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.lambda.functionName,
                // construct the ARN from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionArn}:${this.version}`,
            },
            ...props,
        });
    }
    /**
     * Defines an alias for this version.
     * @param aliasName The name of the alias (e.g. "live")
     * @param options Alias options
     */
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Version = Version;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the
 * qualifier (= version or alias) from the ARN.
 *
 * Version ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name:qualifier
 *
 * ..which means that in order to extract the `qualifier` component from the ARN, we can
 * split the ARN using ":" and select the component in index 7.
 *
 * @returns `FnSelect(7, FnSplit(':', arn))`
 */
function extractQualifierFromArn(arn) {
    return core_1.Fn.select(7, core_1.Fn.split(':', arn));
}
exports.extractQualifierFromArn = extractQualifierFromArn;
//# sourceMappingURL=data:application/json;base64,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