"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualRouter = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const route_1 = require("./route");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualRouterBase extends cdk.Resource {
    /**
     * (experimental) Add a single route to the router.
     *
     * @experimental
     */
    addRoute(id, props) {
        const route = new route_1.Route(this, id, {
            ...props,
            routeName: id,
            mesh: this.mesh,
            virtualRouter: this,
        });
        return route;
    }
}
/**
 * @experimental
 */
class VirtualRouter extends VirtualRouterBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualRouterName || cdk.Lazy.stringValue({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.listeners = new Array();
        this.mesh = props.mesh;
        this.addListener(props.listener || { portMapping: { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP } });
        const router = new appmesh_generated_1.CfnVirtualRouter(this, 'Resource', {
            virtualRouterName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners,
            },
        });
        this.virtualRouterName = this.getResourceNameAttribute(router.attrVirtualRouterName);
        this.virtualRouterArn = this.getResourceArnAttribute(router.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualRouter given an ARN.
     *
     * @experimental
     */
    static fromVirtualRouterArn(scope, id, virtualRouterArn) {
        return new ImportedVirtualRouter(scope, id, { virtualRouterArn });
    }
    /**
     * (experimental) Import an existing VirtualRouter given names.
     *
     * @experimental
     */
    static fromVirtualRouterName(scope, id, meshName, virtualRouterName) {
        return new ImportedVirtualRouter(scope, id, { meshName, virtualRouterName });
    }
    /**
     * (experimental) Import an existing virtual router given attributes.
     *
     * @experimental
     */
    static fromVirtualRouterAttributes(scope, id, attrs) {
        return new ImportedVirtualRouter(scope, id, attrs);
    }
    /**
     * Add port mappings to the router
     */
    addListener(listener) {
        this.listeners.push({
            portMapping: listener.portMapping,
        });
    }
}
exports.VirtualRouter = VirtualRouter;
/**
 * Used to import a VirtualRouter and perform actions or read properties from
 */
class ImportedVirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.mesh) {
            this._mesh = props.mesh;
        }
        if (props.meshName) {
            if (props.mesh) {
                throw new Error('Supply either \'mesh\' or \'meshName\', but not both');
            }
            this._mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', props.meshName);
        }
        if (props.virtualRouterArn) {
            this.virtualRouterArn = props.virtualRouterArn;
            this.virtualRouterName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualRouterArn).resourceName));
        }
        else if (props.virtualRouterName && props.meshName) {
            this.virtualRouterName = props.virtualRouterName;
            this.virtualRouterArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter`,
                resourceName: this.virtualRouterName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
    /**
     * The AppMesh mesh the VirtualRouter belongs to
     */
    get mesh() {
        if (!this._mesh) {
            throw new Error('Please supply either \'mesh\' or \'meshName\' when calling \'fromVirtualRouterAttributes\'');
        }
        return this._mesh;
    }
}
//# sourceMappingURL=data:application/json;base64,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