"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualNodeBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        /**
         * @experimental
         */
        this.backends = new Array();
        /**
         * @experimental
         */
        this.listeners = new Array();
    }
    /**
     * (experimental) Add a Virtual Services that this node is expected to send outbound traffic to.
     *
     * @experimental
     */
    addBackends(...props) {
        for (const s of props) {
            this.backends.push({
                virtualService: {
                    virtualServiceName: s.virtualServiceName,
                },
            });
        }
    }
    /**
     * (experimental) Utility method to add an inbound listener for this virtual node.
     *
     * @experimental
     */
    addListeners(...listeners) {
        if (this.listeners.length + listeners.length > 1) {
            throw new Error('VirtualNode may have at most one listener');
        }
        for (const listener of listeners) {
            const portMapping = listener.portMapping || { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP };
            this.listeners.push({
                portMapping,
                healthCheck: renderHealthCheck(listener.healthCheck, portMapping),
            });
        }
    }
}
function renderHealthCheck(hc, pm) {
    var _a;
    if (hc === undefined) {
        return undefined;
    }
    if (hc.protocol === shared_interfaces_1.Protocol.TCP && hc.path) {
        throw new Error('The path property cannot be set with Protocol.TCP');
    }
    if (hc.protocol === shared_interfaces_1.Protocol.GRPC && hc.path) {
        throw new Error('The path property cannot be set with Protocol.GRPC');
    }
    const protocol = (_a = hc.protocol) !== null && _a !== void 0 ? _a : pm.protocol;
    const healthCheck = {
        healthyThreshold: hc.healthyThreshold || 2,
        intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
        path: hc.path || (protocol === shared_interfaces_1.Protocol.HTTP ? '/' : undefined),
        port: hc.port || pm.port,
        protocol: hc.protocol || pm.protocol,
        timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
        unhealthyThreshold: hc.unhealthyThreshold || 2,
    };
    utils_1.validateHealthChecks(healthCheck);
    return healthCheck;
}
/**
 * (experimental) VirtualNode represents a newly defined App Mesh VirtualNode.
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 * @experimental
 */
class VirtualNode extends VirtualNodeBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.stringValue({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.mesh = props.mesh;
        this.addBackends(...props.backends || []);
        this.addListeners(...props.listener ? [props.listener] : []);
        const accessLogging = (_a = props.accessLog) === null || _a === void 0 ? void 0 : _a.bind(this);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                backends: cdk.Lazy.anyValue({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.anyValue({ produce: () => this.listeners }, { omitEmptyArray: true }),
                serviceDiscovery: {
                    dns: props.dnsHostName !== undefined ? { hostname: props.dnsHostName } : undefined,
                    awsCloudMap: props.cloudMapService !== undefined ? {
                        serviceName: props.cloudMapService.serviceName,
                        namespaceName: props.cloudMapService.namespace.namespaceName,
                        attributes: renderAttributes(props.cloudMapServiceInstanceAttributes),
                    } : undefined,
                },
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualNodeAccessLog,
                } : undefined,
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualNode given an ARN.
     *
     * @experimental
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new ImportedVirtualNode(scope, id, { virtualNodeArn });
    }
    /**
     * (experimental) Import an existing VirtualNode given its name.
     *
     * @experimental
     */
    static fromVirtualNodeName(scope, id, meshName, virtualNodeName) {
        return new ImportedVirtualNode(scope, id, {
            meshName,
            virtualNodeName,
        });
    }
}
exports.VirtualNode = VirtualNode;
function renderAttributes(attrs) {
    if (attrs === undefined) {
        return undefined;
    }
    return Object.entries(attrs).map(([key, value]) => ({ key, value }));
}
/**
 * Used to import a VirtualNode and read its properties
 */
class ImportedVirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualNodeArn) {
            this.virtualNodeArn = props.virtualNodeArn;
            this.virtualNodeName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualNodeArn).resourceName));
        }
        else if (props.virtualNodeName && props.meshName) {
            this.virtualNodeName = props.virtualNodeName;
            this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualNode`,
                resourceName: this.virtualNodeName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmlydHVhbC1ub2RlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidmlydHVhbC1ub2RlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHFDQUFxQztBQUVyQywyREFBcUQ7QUFFckQsMkNBQXVEO0FBQ3ZELDJEQUF5RztBQTBHekcsTUFBZSxlQUFnQixTQUFRLEdBQUcsQ0FBQyxRQUFRO0lBQW5EOzs7OztRQVdxQixhQUFRLEdBQUcsSUFBSSxLQUFLLEVBQWtDLENBQUM7Ozs7UUFDdkQsY0FBUyxHQUFHLElBQUksS0FBSyxFQUFtQyxDQUFDO0lBK0I5RSxDQUFDOzs7Ozs7SUExQlEsV0FBVyxDQUFDLEdBQUcsS0FBd0I7UUFDNUMsS0FBSyxNQUFNLENBQUMsSUFBSSxLQUFLLEVBQUU7WUFDckIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUM7Z0JBQ2pCLGNBQWMsRUFBRTtvQkFDZCxrQkFBa0IsRUFBRSxDQUFDLENBQUMsa0JBQWtCO2lCQUN6QzthQUNGLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQzs7Ozs7O0lBS00sWUFBWSxDQUFDLEdBQUcsU0FBZ0M7UUFDckQsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sR0FBRyxTQUFTLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNoRCxNQUFNLElBQUksS0FBSyxDQUFDLDJDQUEyQyxDQUFDLENBQUM7U0FDOUQ7UUFFRCxLQUFLLE1BQU0sUUFBUSxJQUFJLFNBQVMsRUFBRTtZQUNoQyxNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsV0FBVyxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsNEJBQVEsQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNwRixJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQztnQkFDbEIsV0FBVztnQkFDWCxXQUFXLEVBQUUsaUJBQWlCLENBQUMsUUFBUSxDQUFDLFdBQVcsRUFBRSxXQUFXLENBQUM7YUFDbEUsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDO0NBQ0Y7QUFFRCxTQUFTLGlCQUFpQixDQUFDLEVBQTJCLEVBQUUsRUFBZTs7SUFDckUsSUFBSSxFQUFFLEtBQUssU0FBUyxFQUFFO1FBQUUsT0FBTyxTQUFTLENBQUM7S0FBRTtJQUUzQyxJQUFJLEVBQUUsQ0FBQyxRQUFRLEtBQUssNEJBQVEsQ0FBQyxHQUFHLElBQUksRUFBRSxDQUFDLElBQUksRUFBRTtRQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLG1EQUFtRCxDQUFDLENBQUM7S0FDdEU7SUFFRCxJQUFJLEVBQUUsQ0FBQyxRQUFRLEtBQUssNEJBQVEsQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDLElBQUksRUFBRTtRQUM1QyxNQUFNLElBQUksS0FBSyxDQUFDLG9EQUFvRCxDQUFDLENBQUM7S0FDdkU7SUFFRCxNQUFNLFFBQVEsU0FBRyxFQUFFLENBQUMsUUFBUSxtQ0FBSSxFQUFFLENBQUMsUUFBUSxDQUFDO0lBRTVDLE1BQU0sV0FBVyxHQUF1QztRQUN0RCxnQkFBZ0IsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLElBQUksQ0FBQztRQUMxQyxjQUFjLEVBQUUsQ0FBQyxFQUFFLENBQUMsUUFBUSxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxFQUFFO1FBQ3pFLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxJQUFJLENBQUMsUUFBUSxLQUFLLDRCQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUMvRCxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksSUFBSSxFQUFFLENBQUMsSUFBSTtRQUN4QixRQUFRLEVBQUUsRUFBRSxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUMsUUFBUTtRQUNwQyxhQUFhLEVBQUUsQ0FBQyxFQUFFLENBQUMsT0FBTyxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxFQUFFO1FBQ3ZFLGtCQUFrQixFQUFFLEVBQUUsQ0FBQyxrQkFBa0IsSUFBSSxDQUFDO0tBQy9DLENBQUM7SUFFRiw0QkFBb0IsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUVsQyxPQUFPLFdBQVcsQ0FBQztBQUNyQixDQUFDOzs7Ozs7Ozs7OztBQVdELE1BQWEsV0FBWSxTQUFRLGVBQWU7Ozs7SUFpQzlDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7O1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxlQUFlLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztTQUN6RyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFFdkIsSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksRUFBRSxDQUFDLENBQUM7UUFDMUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUM3RCxNQUFNLGFBQWEsU0FBRyxLQUFLLENBQUMsU0FBUywwQ0FBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFbEQsTUFBTSxJQUFJLEdBQUcsSUFBSSxrQ0FBYyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDaEQsZUFBZSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQ2xDLFFBQVEsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVE7WUFDNUIsSUFBSSxFQUFFO2dCQUNKLFFBQVEsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUM7Z0JBQ3ZGLFNBQVMsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUM7Z0JBQ3pGLGdCQUFnQixFQUFFO29CQUNoQixHQUFHLEVBQUUsS0FBSyxDQUFDLFdBQVcsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsUUFBUSxFQUFFLEtBQUssQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztvQkFDbEYsV0FBVyxFQUFFLEtBQUssQ0FBQyxlQUFlLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQzt3QkFDakQsV0FBVyxFQUFFLEtBQUssQ0FBQyxlQUFlLENBQUMsV0FBVzt3QkFDOUMsYUFBYSxFQUFFLEtBQUssQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDLGFBQWE7d0JBQzVELFVBQVUsRUFBRSxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsaUNBQWlDLENBQUM7cUJBQ3RFLENBQUMsQ0FBQyxDQUFDLFNBQVM7aUJBQ2Q7Z0JBQ0QsT0FBTyxFQUFFLGFBQWEsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDO29CQUNyQyxTQUFTLEVBQUUsYUFBYSxDQUFDLG9CQUFvQjtpQkFDOUMsQ0FBQyxDQUFDLENBQUMsU0FBUzthQUNkO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDL0UsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUMzRCxPQUFPLEVBQUUsU0FBUztZQUNsQixRQUFRLEVBQUUsUUFBUSxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsY0FBYztZQUNuRCxZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDaEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7O0lBbEVNLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxjQUFzQjtRQUNuRixPQUFPLElBQUksbUJBQW1CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7SUFDaEUsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQWdCLEVBQUUsZUFBdUI7UUFDdkcsT0FBTyxJQUFJLG1CQUFtQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDeEMsUUFBUTtZQUNSLGVBQWU7U0FDaEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQXVERjtBQXZFRCxrQ0F1RUM7QUFFRCxTQUFTLGdCQUFnQixDQUFDLEtBQStCO0lBQ3ZELElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtRQUFFLE9BQU8sU0FBUyxDQUFDO0tBQUU7SUFDOUMsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQztBQUN2RSxDQUFDO0FBc0JEOztHQUVHO0FBQ0gsTUFBTSxtQkFBb0IsU0FBUSxlQUFlO0lBVy9DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBNEI7UUFDcEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLEtBQUssQ0FBQyxjQUFjLEVBQUU7WUFDeEIsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQzNDLElBQUksQ0FBQyxlQUFlLEdBQUcsR0FBRyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDLFlBQWEsQ0FBQyxDQUFDLENBQUM7U0FDOUg7YUFBTSxJQUFJLEtBQUssQ0FBQyxlQUFlLElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNsRCxJQUFJLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUM7WUFDN0MsSUFBSSxDQUFDLGNBQWMsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7Z0JBQ2pELE9BQU8sRUFBRSxTQUFTO2dCQUNsQixRQUFRLEVBQUUsUUFBUSxLQUFLLENBQUMsUUFBUSxjQUFjO2dCQUM5QyxZQUFZLEVBQUUsSUFBSSxDQUFDLGVBQWU7YUFDbkMsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsK0JBQStCLENBQUMsQ0FBQztTQUNsRDtJQUNILENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNsb3VkbWFwIGZyb20gJ0Bhd3MtY2RrL2F3cy1zZXJ2aWNlZGlzY292ZXJ5JztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuVmlydHVhbE5vZGUgfSBmcm9tICcuL2FwcG1lc2guZ2VuZXJhdGVkJztcbmltcG9ydCB7IElNZXNoIH0gZnJvbSAnLi9tZXNoJztcbmltcG9ydCB7IHZhbGlkYXRlSGVhbHRoQ2hlY2tzIH0gZnJvbSAnLi9wcml2YXRlL3V0aWxzJztcbmltcG9ydCB7IEFjY2Vzc0xvZywgSGVhbHRoQ2hlY2ssIFBvcnRNYXBwaW5nLCBQcm90b2NvbCwgVmlydHVhbE5vZGVMaXN0ZW5lciB9IGZyb20gJy4vc2hhcmVkLWludGVyZmFjZXMnO1xuaW1wb3J0IHsgSVZpcnR1YWxTZXJ2aWNlIH0gZnJvbSAnLi92aXJ0dWFsLXNlcnZpY2UnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJVmlydHVhbE5vZGUgZXh0ZW5kcyBjZGsuSVJlc291cmNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZpcnR1YWxOb2RlTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdmlydHVhbE5vZGVBcm46IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgYWRkQmFja2VuZHMoLi4ucHJvcHM6IElWaXJ0dWFsU2VydmljZVtdKTogdm9pZDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgYWRkTGlzdGVuZXJzKC4uLmxpc3RlbmVyczogVmlydHVhbE5vZGVMaXN0ZW5lcltdKTogdm9pZDtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgVmlydHVhbE5vZGVCYXNlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2aXJ0dWFsTm9kZU5hbWU/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRuc0hvc3ROYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNsb3VkTWFwU2VydmljZT86IGNsb3VkbWFwLklTZXJ2aWNlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNsb3VkTWFwU2VydmljZUluc3RhbmNlQXR0cmlidXRlcz86IHtba2V5OiBzdHJpbmddOiBzdHJpbmd9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGJhY2tlbmRzPzogSVZpcnR1YWxTZXJ2aWNlW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbGlzdGVuZXI/OiBWaXJ0dWFsTm9kZUxpc3RlbmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYWNjZXNzTG9nPzogQWNjZXNzTG9nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBWaXJ0dWFsTm9kZVByb3BzIGV4dGVuZHMgVmlydHVhbE5vZGVCYXNlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWVzaDogSU1lc2g7XG59XG5cbmFic3RyYWN0IGNsYXNzIFZpcnR1YWxOb2RlQmFzZSBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElWaXJ0dWFsTm9kZSB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgVmlydHVhbE5vZGVcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB2aXJ0dWFsTm9kZU5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFtYXpvbiBSZXNvdXJjZSBOYW1lIGJlbG9uZ2luZyB0byB0aGUgVmlydHVhbE5kb2VcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB2aXJ0dWFsTm9kZUFybjogc3RyaW5nO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSBiYWNrZW5kcyA9IG5ldyBBcnJheTxDZm5WaXJ0dWFsTm9kZS5CYWNrZW5kUHJvcGVydHk+KCk7XG4gIHByb3RlY3RlZCByZWFkb25seSBsaXN0ZW5lcnMgPSBuZXcgQXJyYXk8Q2ZuVmlydHVhbE5vZGUuTGlzdGVuZXJQcm9wZXJ0eT4oKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkQmFja2VuZHMoLi4ucHJvcHM6IElWaXJ0dWFsU2VydmljZVtdKSB7XG4gICAgZm9yIChjb25zdCBzIG9mIHByb3BzKSB7XG4gICAgICB0aGlzLmJhY2tlbmRzLnB1c2goe1xuICAgICAgICB2aXJ0dWFsU2VydmljZToge1xuICAgICAgICAgIHZpcnR1YWxTZXJ2aWNlTmFtZTogcy52aXJ0dWFsU2VydmljZU5hbWUsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZExpc3RlbmVycyguLi5saXN0ZW5lcnM6IFZpcnR1YWxOb2RlTGlzdGVuZXJbXSkge1xuICAgIGlmICh0aGlzLmxpc3RlbmVycy5sZW5ndGggKyBsaXN0ZW5lcnMubGVuZ3RoID4gMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdWaXJ0dWFsTm9kZSBtYXkgaGF2ZSBhdCBtb3N0IG9uZSBsaXN0ZW5lcicpO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgbGlzdGVuZXIgb2YgbGlzdGVuZXJzKSB7XG4gICAgICBjb25zdCBwb3J0TWFwcGluZyA9IGxpc3RlbmVyLnBvcnRNYXBwaW5nIHx8IHsgcG9ydDogODA4MCwgcHJvdG9jb2w6IFByb3RvY29sLkhUVFAgfTtcbiAgICAgIHRoaXMubGlzdGVuZXJzLnB1c2goe1xuICAgICAgICBwb3J0TWFwcGluZyxcbiAgICAgICAgaGVhbHRoQ2hlY2s6IHJlbmRlckhlYWx0aENoZWNrKGxpc3RlbmVyLmhlYWx0aENoZWNrLCBwb3J0TWFwcGluZyksXG4gICAgICB9KTtcbiAgICB9XG4gIH1cbn1cblxuZnVuY3Rpb24gcmVuZGVySGVhbHRoQ2hlY2soaGM6IEhlYWx0aENoZWNrIHwgdW5kZWZpbmVkLCBwbTogUG9ydE1hcHBpbmcpOiBDZm5WaXJ0dWFsTm9kZS5IZWFsdGhDaGVja1Byb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgaWYgKGhjID09PSB1bmRlZmluZWQpIHsgcmV0dXJuIHVuZGVmaW5lZDsgfVxuXG4gIGlmIChoYy5wcm90b2NvbCA9PT0gUHJvdG9jb2wuVENQICYmIGhjLnBhdGgpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoZSBwYXRoIHByb3BlcnR5IGNhbm5vdCBiZSBzZXQgd2l0aCBQcm90b2NvbC5UQ1AnKTtcbiAgfVxuXG4gIGlmIChoYy5wcm90b2NvbCA9PT0gUHJvdG9jb2wuR1JQQyAmJiBoYy5wYXRoKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgcGF0aCBwcm9wZXJ0eSBjYW5ub3QgYmUgc2V0IHdpdGggUHJvdG9jb2wuR1JQQycpO1xuICB9XG5cbiAgY29uc3QgcHJvdG9jb2wgPSBoYy5wcm90b2NvbCA/PyBwbS5wcm90b2NvbDtcblxuICBjb25zdCBoZWFsdGhDaGVjazogQ2ZuVmlydHVhbE5vZGUuSGVhbHRoQ2hlY2tQcm9wZXJ0eSA9IHtcbiAgICBoZWFsdGh5VGhyZXNob2xkOiBoYy5oZWFsdGh5VGhyZXNob2xkIHx8IDIsXG4gICAgaW50ZXJ2YWxNaWxsaXM6IChoYy5pbnRlcnZhbCB8fCBjZGsuRHVyYXRpb24uc2Vjb25kcyg1KSkudG9NaWxsaXNlY29uZHMoKSwgLy8gbWluXG4gICAgcGF0aDogaGMucGF0aCB8fCAocHJvdG9jb2wgPT09IFByb3RvY29sLkhUVFAgPyAnLycgOiB1bmRlZmluZWQpLFxuICAgIHBvcnQ6IGhjLnBvcnQgfHwgcG0ucG9ydCxcbiAgICBwcm90b2NvbDogaGMucHJvdG9jb2wgfHwgcG0ucHJvdG9jb2wsXG4gICAgdGltZW91dE1pbGxpczogKGhjLnRpbWVvdXQgfHwgY2RrLkR1cmF0aW9uLnNlY29uZHMoMikpLnRvTWlsbGlzZWNvbmRzKCksXG4gICAgdW5oZWFsdGh5VGhyZXNob2xkOiBoYy51bmhlYWx0aHlUaHJlc2hvbGQgfHwgMixcbiAgfTtcblxuICB2YWxpZGF0ZUhlYWx0aENoZWNrcyhoZWFsdGhDaGVjayk7XG5cbiAgcmV0dXJuIGhlYWx0aENoZWNrO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFZpcnR1YWxOb2RlIGV4dGVuZHMgVmlydHVhbE5vZGVCYXNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21WaXJ0dWFsTm9kZUFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCB2aXJ0dWFsTm9kZUFybjogc3RyaW5nKTogSVZpcnR1YWxOb2RlIHtcbiAgICByZXR1cm4gbmV3IEltcG9ydGVkVmlydHVhbE5vZGUoc2NvcGUsIGlkLCB7IHZpcnR1YWxOb2RlQXJuIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVZpcnR1YWxOb2RlTmFtZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBtZXNoTmFtZTogc3RyaW5nLCB2aXJ0dWFsTm9kZU5hbWU6IHN0cmluZyk6IElWaXJ0dWFsTm9kZSB7XG4gICAgcmV0dXJuIG5ldyBJbXBvcnRlZFZpcnR1YWxOb2RlKHNjb3BlLCBpZCwge1xuICAgICAgbWVzaE5hbWUsXG4gICAgICB2aXJ0dWFsTm9kZU5hbWUsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHZpcnR1YWxOb2RlTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB2aXJ0dWFsTm9kZUFybjogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IG1lc2g6IElNZXNoO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBWaXJ0dWFsTm9kZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLnZpcnR1YWxOb2RlTmFtZSB8fCBjZGsuTGF6eS5zdHJpbmdWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IGNkay5OYW1lcy51bmlxdWVJZCh0aGlzKSB9KSxcbiAgICB9KTtcblxuICAgIHRoaXMubWVzaCA9IHByb3BzLm1lc2g7XG5cbiAgICB0aGlzLmFkZEJhY2tlbmRzKC4uLnByb3BzLmJhY2tlbmRzIHx8IFtdKTtcbiAgICB0aGlzLmFkZExpc3RlbmVycyguLi5wcm9wcy5saXN0ZW5lciA/IFtwcm9wcy5saXN0ZW5lcl0gOiBbXSk7XG4gICAgY29uc3QgYWNjZXNzTG9nZ2luZyA9IHByb3BzLmFjY2Vzc0xvZz8uYmluZCh0aGlzKTtcblxuICAgIGNvbnN0IG5vZGUgPSBuZXcgQ2ZuVmlydHVhbE5vZGUodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgdmlydHVhbE5vZGVOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIG1lc2hOYW1lOiB0aGlzLm1lc2gubWVzaE5hbWUsXG4gICAgICBzcGVjOiB7XG4gICAgICAgIGJhY2tlbmRzOiBjZGsuTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuYmFja2VuZHMgfSwgeyBvbWl0RW1wdHlBcnJheTogdHJ1ZSB9KSxcbiAgICAgICAgbGlzdGVuZXJzOiBjZGsuTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMubGlzdGVuZXJzIH0sIHsgb21pdEVtcHR5QXJyYXk6IHRydWUgfSksXG4gICAgICAgIHNlcnZpY2VEaXNjb3Zlcnk6IHtcbiAgICAgICAgICBkbnM6IHByb3BzLmRuc0hvc3ROYW1lICE9PSB1bmRlZmluZWQgPyB7IGhvc3RuYW1lOiBwcm9wcy5kbnNIb3N0TmFtZSB9IDogdW5kZWZpbmVkLFxuICAgICAgICAgIGF3c0Nsb3VkTWFwOiBwcm9wcy5jbG91ZE1hcFNlcnZpY2UgIT09IHVuZGVmaW5lZCA/IHtcbiAgICAgICAgICAgIHNlcnZpY2VOYW1lOiBwcm9wcy5jbG91ZE1hcFNlcnZpY2Uuc2VydmljZU5hbWUsXG4gICAgICAgICAgICBuYW1lc3BhY2VOYW1lOiBwcm9wcy5jbG91ZE1hcFNlcnZpY2UubmFtZXNwYWNlLm5hbWVzcGFjZU5hbWUsXG4gICAgICAgICAgICBhdHRyaWJ1dGVzOiByZW5kZXJBdHRyaWJ1dGVzKHByb3BzLmNsb3VkTWFwU2VydmljZUluc3RhbmNlQXR0cmlidXRlcyksXG4gICAgICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgICAgfSxcbiAgICAgICAgbG9nZ2luZzogYWNjZXNzTG9nZ2luZyAhPT0gdW5kZWZpbmVkID8ge1xuICAgICAgICAgIGFjY2Vzc0xvZzogYWNjZXNzTG9nZ2luZy52aXJ0dWFsTm9kZUFjY2Vzc0xvZyxcbiAgICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0aGlzLnZpcnR1YWxOb2RlTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKG5vZGUuYXR0clZpcnR1YWxOb2RlTmFtZSk7XG4gICAgdGhpcy52aXJ0dWFsTm9kZUFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUobm9kZS5yZWYsIHtcbiAgICAgIHNlcnZpY2U6ICdhcHBtZXNoJyxcbiAgICAgIHJlc291cmNlOiBgbWVzaC8ke3Byb3BzLm1lc2gubWVzaE5hbWV9L3ZpcnR1YWxOb2RlYCxcbiAgICAgIHJlc291cmNlTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgfSk7XG4gIH1cbn1cblxuZnVuY3Rpb24gcmVuZGVyQXR0cmlidXRlcyhhdHRycz86IHtba2V5OiBzdHJpbmddOiBzdHJpbmd9KSB7XG4gIGlmIChhdHRycyA9PT0gdW5kZWZpbmVkKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cbiAgcmV0dXJuIE9iamVjdC5lbnRyaWVzKGF0dHJzKS5tYXAoKFtrZXksIHZhbHVlXSkgPT4gKHsga2V5LCB2YWx1ZSB9KSk7XG59XG5cbi8qKlxuICogSW50ZXJmYWNlIHdpdGggcHJvcGVydGllcyBuY2VjZXNzYXJ5IHRvIGltcG9ydCBhIHJldXNhYmxlIFZpcnR1YWxOb2RlXG4gKi9cbmludGVyZmFjZSBWaXJ0dWFsTm9kZUF0dHJpYnV0ZXMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIFZpcnR1YWxOb2RlXG4gICAqL1xuICByZWFkb25seSB2aXJ0dWFsTm9kZU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZSBiZWxvbmdpbmcgdG8gdGhlIFZpcnR1YWxOZG9lXG4gICAqL1xuICByZWFkb25seSB2aXJ0dWFsTm9kZUFybj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHNlcnZpY2UgbWVzaCB0aGF0IHRoZSB2aXJ0dWFsIG5vZGUgcmVzaWRlcyBpblxuICAgKi9cbiAgcmVhZG9ubHkgbWVzaE5hbWU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogVXNlZCB0byBpbXBvcnQgYSBWaXJ0dWFsTm9kZSBhbmQgcmVhZCBpdHMgcHJvcGVydGllc1xuICovXG5jbGFzcyBJbXBvcnRlZFZpcnR1YWxOb2RlIGV4dGVuZHMgVmlydHVhbE5vZGVCYXNlIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBWaXJ0dWFsTm9kZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZpcnR1YWxOb2RlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIFJlc291cmNlIE5hbWUgYmVsb25naW5nIHRvIHRoZSBWaXJ0dWFsTmRvZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZpcnR1YWxOb2RlQXJuOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFZpcnR1YWxOb2RlQXR0cmlidXRlcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMudmlydHVhbE5vZGVBcm4pIHtcbiAgICAgIHRoaXMudmlydHVhbE5vZGVBcm4gPSBwcm9wcy52aXJ0dWFsTm9kZUFybjtcbiAgICAgIHRoaXMudmlydHVhbE5vZGVOYW1lID0gY2RrLkZuLnNlbGVjdCgyLCBjZGsuRm4uc3BsaXQoJy8nLCBjZGsuU3RhY2sub2Yoc2NvcGUpLnBhcnNlQXJuKHByb3BzLnZpcnR1YWxOb2RlQXJuKS5yZXNvdXJjZU5hbWUhKSk7XG4gICAgfSBlbHNlIGlmIChwcm9wcy52aXJ0dWFsTm9kZU5hbWUgJiYgcHJvcHMubWVzaE5hbWUpIHtcbiAgICAgIHRoaXMudmlydHVhbE5vZGVOYW1lID0gcHJvcHMudmlydHVhbE5vZGVOYW1lO1xuICAgICAgdGhpcy52aXJ0dWFsTm9kZUFybiA9IGNkay5TdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICBzZXJ2aWNlOiAnYXBwbWVzaCcsXG4gICAgICAgIHJlc291cmNlOiBgbWVzaC8ke3Byb3BzLm1lc2hOYW1lfS92aXJ0dWFsTm9kZWAsXG4gICAgICAgIHJlc291cmNlTmFtZTogdGhpcy52aXJ0dWFsTm9kZU5hbWUsXG4gICAgICB9KTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdOZWVkIGVpdGhlciBhcm4gb3IgYm90aCBuYW1lcycpO1xuICAgIH1cbiAgfVxufVxuIl19