"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGateway = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const gateway_route_1 = require("./gateway-route");
const mesh_1 = require("./mesh");
const virtual_gateway_listener_1 = require("./virtual-gateway-listener");
class VirtualGatewayBase extends cdk.Resource {
    /**
     * (experimental) Utility method to add a new GatewayRoute to the VirtualGateway.
     *
     * @experimental
     */
    addGatewayRoute(id, props) {
        return new gateway_route_1.GatewayRoute(this, id, {
            ...props,
            virtualGateway: this,
        });
    }
}
/**
 * (experimental) VirtualGateway represents a newly defined App Mesh Virtual Gateway.
 *
 * A virtual gateway allows resources that are outside of your mesh to communicate to resources that
 * are inside of your mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
 * @experimental
 */
class VirtualGateway extends VirtualGatewayBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.virtualGatewayName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        /**
         * @experimental
         */
        this.listeners = new Array();
        this.mesh = props.mesh;
        if (!props.listeners) {
            // Use listener default of http listener port 8080 if no listener is defined
            this.listeners.push(virtual_gateway_listener_1.VirtualGatewayListener.httpGatewayListener().bind(this));
        }
        else {
            props.listeners.forEach(listener => this.listeners.push(listener.bind(this)));
        }
        const accessLogging = (_a = props.accessLog) === null || _a === void 0 ? void 0 : _a.bind(this);
        const node = new appmesh_generated_1.CfnVirtualGateway(this, 'Resource', {
            virtualGatewayName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners.map(listener => listener.listener),
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualGatewayAccessLog,
                } : undefined,
            },
        });
        this.virtualGatewayName = this.getResourceNameAttribute(node.attrVirtualGatewayName);
        this.virtualGatewayArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualGateway`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualGateway given an ARN.
     *
     * @experimental
     */
    static fromVirtualGatewayArn(scope, id, virtualGatewayArn) {
        return new ImportedVirtualGateway(scope, id, { virtualGatewayArn });
    }
}
exports.VirtualGateway = VirtualGateway;
/**
 * Used to import a VirtualGateway and read its properties
 */
class ImportedVirtualGateway extends VirtualGatewayBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualGatewayArn) {
            const meshName = cdk.Fn.select(0, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualGatewayArn).resourceName));
            this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', meshName);
            this.virtualGatewayArn = props.virtualGatewayArn;
            this.virtualGatewayName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualGatewayArn).resourceName));
        }
        else {
            throw new Error('Need virtualGatewayArn');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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